"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Between = exports.BinaryCondition = exports.BeginsWith = exports.BaseKeyCondition = void 0;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(', ');
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(', ');
    }
}
exports.BaseKeyCondition = BaseKeyCondition;
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BeginsWith = BeginsWith;
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BinaryCondition = BinaryCondition;
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
exports.Between = Between;
//# sourceMappingURL=data:application/json;base64,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