"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MappingTemplate = void 0;
const fs_1 = require("fs");
/**
 * MappingTemplates for AppSync resolvers
 */
class MappingTemplate {
    /**
     * Create a mapping template from the given string
     */
    static fromString(template) {
        return new StringMappingTemplate(template);
    }
    /**
     * Create a mapping template from the given file
     */
    static fromFile(fileName) {
        return new StringMappingTemplate(fs_1.readFileSync(fileName).toString('UTF-8'));
    }
    /**
     * Mapping template for a result list from DynamoDB
     */
    static dynamoDbResultList() {
        return this.fromString('$util.toJson($ctx.result.items)');
    }
    /**
     * Mapping template for a single result item from DynamoDB
     */
    static dynamoDbResultItem() {
        return this.fromString('$util.toJson($ctx.result)');
    }
    /**
     * Mapping template to scan a DynamoDB table to fetch all entries
     */
    static dynamoDbScanTable() {
        return this.fromString('{"version" : "2017-02-28", "operation" : "Scan"}');
    }
    /**
     * Mapping template to query a set of items from a DynamoDB table
     *
     * @param cond the key condition for the query
     */
    static dynamoDbQuery(cond) {
        return this.fromString(`{"version" : "2017-02-28", "operation" : "Query", ${cond.renderTemplate()}}`);
    }
    /**
     * Mapping template to get a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Query argument
     */
    static dynamoDbGetItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "GetItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to delete a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Mutation argument
     */
    static dynamoDbDeleteItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "DeleteItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to save a single item to a DynamoDB table
     *
     * @param key the assigment of Mutation values to the primary key
     * @param values the assignment of Mutation values to the table attributes
     */
    static dynamoDbPutItem(key, values) {
        return this.fromString(`
      ${values.renderVariables()}
      {
        "version": "2017-02-28",
        "operation": "PutItem",
        ${key.renderTemplate()},
        ${values.renderTemplate()}
      }`);
    }
    /**
     * Mapping template to invoke a Lambda function
     *
     * @param payload the VTL template snippet of the payload to send to the lambda.
     * If no payload is provided all available context fields are sent to the Lambda function
     */
    static lambdaRequest(payload = '$util.toJson($ctx)') {
        return this.fromString(`{"version": "2017-02-28", "operation": "Invoke", "payload": ${payload}}`);
    }
    /**
     * Mapping template to return the Lambda result to the caller
     */
    static lambdaResult() {
        return this.fromString('$util.toJson($ctx.result)');
    }
}
exports.MappingTemplate = MappingTemplate;
class StringMappingTemplate extends MappingTemplate {
    constructor(template) {
        super();
        this.template = template;
    }
    renderTemplate() {
        return this.template;
    }
}
//# sourceMappingURL=data:application/json;base64,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