import pytest
from mindee import Client, Response, Receipt, Passport


@pytest.fixture
def empty_client():
    return Client()


@pytest.fixture
def dummy_client():
    return Client(expense_receipt_token="dummy",
                  invoice_token="dummy",
                  passport_token="dummy",
                  license_plate_token="dummy")


@pytest.fixture
def dummy_client_dont_raise():
    return Client(expense_receipt_token="dummy",
                  invoice_token="dummy",
                  passport_token="dummy",
                  license_plate_token="dummy",
                  raise_on_error=False)


@pytest.fixture
def response():
    return Response.load("./tests/data/expense_receipts/v3/receipt.json")


def test_parse_receipt_without_token(empty_client):
    with pytest.raises(Exception):
        empty_client.parse_receipt('./tests/data/expense_receipts/receipt.jpg')


def test_parse_invoice_without_token(empty_client):
    with pytest.raises(Exception):
        empty_client.parse_invoice('./tests/data/expense_receipts/receipt.jpg')


def test_parse_financial_doc_without_token(empty_client):
    with pytest.raises(Exception):
        empty_client.parse_financial_document('./tests/data/expense_receipts/receipt.jpg')


def test_parse_passport_without_token(empty_client):
    with pytest.raises(Exception):
        empty_client.parse_passport('./tests/data/expense_receipts/receipt.jpg')


def test_parse_license_plate_without_token(empty_client):
    with pytest.raises(Exception):
        empty_client.parse_license_plate('./tests/data/license_plates/plate.png')


def test_parse_receipt_with_wrong_filetype(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_receipt('./tests/data/expense_receipts/receipt.jpga')


def test_parse_invoice_with_wrong_filetype(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_invoice('./tests/data/expense_receipts/receipt.jpga')


def test_parse_financial_doc_with_wrong_filetype(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_financial_document('./tests/data/expense_receipts/receipt.jpga')


def test_parse_passport_with_wrong_filetype(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_passport('./tests/data/expense_receipts/receipt.jpga')


def test_parse_plate_with_wrong_filetype(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_license_plate('./tests/data/expense_receipts/receipt.jpga')


def test_parse_receipt_with_wrong_token(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_receipt('./tests/data/expense_receipts/receipt.jpg')


def test_parse_receipt_with_wrong_version(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_receipt('./tests/data/expense_receipts/receipt.jpg', version="4000")


def test_parse_invoice_with_wrong_token(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_invoice('./tests/data/expense_receipts/receipt.jpg')


def test_parse_financial_doc_with_wrong_token_jpg(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_financial_document('./tests/data/expense_receipts/receipt.jpg')


def test_parse_financial_doc_with_wrong_token_pdf(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_financial_document('./tests/data/invoices/invoice.pdf')


def test_parse_passport_with_wrong_token(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_passport('./tests/data/expense_receipts/receipt.jpg')


def test_parse_license_plate_with_wrong_token(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_license_plate('./tests/data/license_plates/plate.png')


def test_response_dump(response):
    assert isinstance(response.receipt, Receipt)
    response.dump("./tests/data/response_dump.json")


def test_response_dump_failure(response):
    with pytest.raises(Exception):
        response.dump(open("./tests/pathDoesNotExist/aaa"))


def test_response_load_failure():
    with pytest.raises(Exception):
        Response.load("notAFile")


def test_response_with_passport_type():
    response = Response.load("./tests/data/passport/v1/passport.json")
    assert isinstance(response.passport, Passport)


def test_request_with_filepath(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_receipt('./tests/data/expense_receipts/receipt.jpg', input_type="path")


def test_request_with_file(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_receipt(open('./tests/data/expense_receipts/receipt.jpg'), input_type="file")


def test_request_with_base64(dummy_client):
    b64 = "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"
    with pytest.raises(Exception):
        dummy_client.parse_receipt(b64, input_type="base64")


def test_request_without_raise_on_error(dummy_client_dont_raise):
    result = dummy_client_dont_raise.parse_receipt('./tests/data/expense_receipts/receipt.jpg', input_type="path")
    assert result.receipt is None
    assert len(result.receipts) == 0


def test_request_without_raise_on_error_include_words(dummy_client_dont_raise):
    result = dummy_client_dont_raise.parse_receipt('./tests/data/expense_receipts/receipt.jpg', input_type="path", include_words=True)
    assert result.receipt is None
    assert len(result.receipts) == 0


def test_request_with_file_wrong_type(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_receipt(open('./tests/data/test.txt'), input_type="file")

    with pytest.raises(Exception):
        dummy_client.parse_receipt('./tests/data/test.txt', input_type="path")


def test_mpdf_reconstruct(dummy_client):
    with pytest.raises(Exception):
        dummy_client.parse_invoice('./tests/data/invoices/invoice_6p.pdf')