"""The :mod:`paddy.Default_Numerics` module contains functions for numeric
problems.

Routine listings
----------------
eval_numeric(object)

polynomial(object)

poly(x,seed)

trig_inter(x,seed)

mse_func(target, output)

gramacy_lee()

See Also
--------
:mod:`paddy.Paddy_Runner`

Notes
-----
The background information regarding how general functions and classes used to
initiate an instance of :class:`~paddy.Paddy_Runner.PFARunner` that are also
provided as builtins in the :mod:`~paddy.Default_Numerics` module is described
in the :mod:`~paddy.Paddy_Runner` module.

Examples
--------

"""

import math
import numpy as np
from paddy.Paddy_Parameter import PaddyParameter


def gramacy_lee():
    """Return Gramacy and Lee function coordinates.

    This function generates the *x-y* coordinates used to evaluate
    interpolation of the Gramacy and Lee function via a default or user
    defined intrerpolation function.

    See Also
    --------
    :func:`eval_numeric`

    References
    ----------
    .. [1] Gramacy, R. B., & Lee, H. K. (2012). Cases for the nugget in
       modeling computer experiments. Statistics and Computing, 22(3), 713-722
       .

    """
    x_dummy = -0.5
    y_list = []
    counter = 0
    x_list = []
    x_range = np.arange(-.5, 2.501, 0.001)
    while counter < len(x_range):
        x_list.append(round(x_range[counter], 3))
        counter = counter+1
    counter = 0
    while counter < len(x_range):
        y_list.append(((math.sin(10*math.pi*x_dummy))/float(2*x_dummy))+((x_dummy-1)**4))
        x_dummy = x_dummy + 0.001
        counter = counter+1
    return x_list, y_list


def mse_func(target, mse_input):
    """Return error of interpolation.

    This function returns eval_numeric()the mean squared error for two sequences of
    numeric values.

    """
    counter = 0
    error = []
    while counter < len(target):
        error.append((abs(target[counter]-mse_input[counter]))**2)
        counter = counter+1
    ave_error = sum(error)/float(len(target))
    return ave_error


def poly(x_list, seed):
    r"""Return cordinates of a polynomial.

    This function returns the 2-D cordinates of a polynomial, and is used
    for interpolation problems.

    Parameters
    ----------
    x_list : list
        A list of numerics that defines the x values evaluated over.

    seed : array-like, shape = (parameters, 2)
        A numpy array of parameters generated by paddy.  Refer to
        :mod:`~paddy.Paddy_runner` for details regarding generation of the
        array.

    Returns
    -------
    output : list of floats
        A list of float values representing the y values of the input x.

    See Also
    --------
    :func:`eval_numeric`

    Notes
    -----
    The polynomial is defined as:

    .. math:: \sum_{k=0}^{n}a_{k}x^k

    Where ``a`` represents the sequence of parameter values generated by
    paddy.  ``x`` is simply the `x` parameter of numeric values that the
    polynomial is evaluated over.

    """
    counter = 0
    output = []
    s_x = []
    while counter < len(x_list):
        k = 0
        xn_out = []
        while k < len(seed):
            ###sums at x1###
            temp_y = seed[k][0]*(x_list[counter]**k)
            xn_out.append(temp_y)
            k = k+1
        s_x.append(sum(xn_out))
        counter = counter+1
    output = s_x
    return output

def trig_inter(x_list, seed):
    r"""Return cordinates of a trigonometric polynomial.

    This function returns the 2-D cordinates of a trigonometric polynomial,
    and is used for interpolation problems.

    Parameters
    ----------
    x_list : list
        A list of numerics that defines the x values evaluated over.

    seed : array-like, shape = (parameters,2)
        A numpy array of parameters generated by paddy.  Refer to
        :mod:`~paddy.Paddy_runner` for details regarding generation of the
        array.

    See Also
    --------
    :func:`eval_numeric`

    Notes
    -----
    The trigonometric polynomial is defined as:

    .. math:: t(x)=a_{0}+\sum_{n=1}^{N}a_{n}\cos(nx)+\sum_{n=1}^{N}b_{n}\sin(
        nx)

    Where ``a`` represents the sequence of parameter values generated by
    paddy and are passed as the odd indices of `seed` and ``b`` being even.
    Notice that this would mean that a trigonometric polynomial of the 10th
    degree would be defined by a paddy space of 11 individual parameters.

    Refrences
    ---------
    .. [1] Rudin, Walter (1987), Real and complex analysis (3rd ed.), New York
        : McGraw-Hill, ISBN 978-0-07-054234-1, MR 0924157.

    """
    if len(seed) % 2 == 0:
        print("must use odd value for dim greater than 1!")
        return
    counter = 0
    s_x = []
    while counter < len(x_list):
        ###this evaluates over x
        number = 1
        xn_out = []
        while number <= ((len(seed)-1)/2.0):
            ###evaluates sums of cos and sin
            alpha = ((number*2)-1)
            beta = (number*2)
            temp_y = ((
                seed[alpha][0])*(math.cos((number*x_list[counter]))))+(
                    (seed[beta][0])*(math.sin((number*x_list[counter]))))
            xn_out.append(temp_y)
            number += 1
        xn_out.append(seed[0][0])
        s_x.append(sum(xn_out))
        counter += 1
    return s_x


class EvalNumeric(object):
    r"""Return error after evaluating a numeric function.

    This class can be used to evaluate an optimization problem where fiting
    to a numeric value is the goal.  This can range from interpolation to
    min-max optimization.

    Parameters
    ----------
    seed : array-like, shape = (parameters,2)
        A numpy array of parameters generated by paddy.  Refer to
        :mod:`~paddy.Paddy_runner` for details regarding generation of the
        array.

    error_func : function, optional (default : mse_func)
        An error function thats output is minimized by paddy.  If user defined
        , make sure to note the internal sign change used to maximize the
        negative error value returned by `eval_numeric`.

    t_func : function, optional (default : gramacy_lee)
        A target function that provides the input and target value(s) for the
        error and fitting functions.

    f_func : function, optional (default : trig_inter)
        A fitting function that recieves an input generated by the target
        function, and returns an output to compate with the desired answer
        being optimized twards.

    Returns
    -------
    error : float
        A numeric value that serves as the fitness value for the seed
        evaluated.  This is defined as the negative output of the error
        function.

    See Also
    --------
    :func:`mse_func`

    :func:`gramacy_lee`

    :func:`trig_inter`

    Notes
    -----
    The default functionalities of this function is evaluation of seed values
    as the coeficents of a trigonometric polynomial used to interpolate the
    Gramacy and Lee function, where the evaluation metric function is the
    mean squared error of the x-y coordinates of Gramacy and Lee and the
    generated polynomial where :math:`x\in[ -0.5,2.5 ]` with a resolution of
    0.001.  The default and user defined instances of this class use the
    method `eval(seed)` to evaluate and return results.

    """
    def __init__(self, error_func=mse_func,
                 t_func=gramacy_lee, f_func=trig_inter):
        self.error_func = error_func
        self.t_func = t_func
        self.f_func = f_func
        self.x_vals, self.answer = self.t_func()

    def eval(self, seed):
        """Method of `eval_numeric`.
        """
        y_val = self.f_func(self.x_vals, seed)
        #made negative for maximization problem
        error = -self.error_func(self.answer, y_val)
        return error

class Polynomial(object):
    """Generate paddy space that is apt for polynomial fitting.

    See Also
    --------
    :class:`eval_numeric`

    :class:`paddy.Paddy_Parameter.PaddyParameter`

    """
    def __init__(self, length, scope, gausian_type, normalization=True,
                 limits=True):
        if limits is False:
            limit_init = None
        else:
            limit_init = [-scope, scope]
        counter = 0
        while counter < length:
            vars(self)['polly{0}'.format(counter)] = (
                PaddyParameter(param_range=[-scope, scope, scope*.05],
                               param_type='continuous', limits=limit_init,
                               gaussian=gausian_type,
                               normalization=normalization)
            )
            counter += 1
            