#!/usr/bin/env python3

#/*******************************************************************************
# Copyright (c) 2012 IBM Corp.

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#/*******************************************************************************

'''
    Created on Nov 17, 2011

    Passive Object Operations Library

    @author: Marcio A. Silva, Michael R. Galaxy
'''
from os import makedirs, access, F_OK
from time import asctime, localtime, sleep, time, strftime
from redis import ConnectionError
from subprocess import Popen, PIPE
from xdrlib import Packer
from sys import path
from threading import Condition
import copy, re, socket, os
import json
import shutil
import textwrap
import threading
import traceback
from hashlib import sha1
from base64 import b64encode

cwd = (re.compile(".*\/").search(os.path.realpath(__file__)).group(0)) + "/../../"
path.append(cwd)

from lib.auxiliary.code_instrumentation import trace, cblog, cbdebug, cberr, cbwarn, cbinfo, cbcrit
from lib.auxiliary.value_generation import ValueGeneration
from lib.remote.process_management import ProcessManagement
from lib.auxiliary.data_ops import str2dic, dic2str, makeTimestamp
from lib.operations.base_operations import BaseObjectOperations
from lib.stores.common_datastore_adapter import MetricStoreMgdConnException
from scripts.common.cb_common import report_app_metrics

qemu_supported = False

try :
    import libvirt
except ImportError:
    cbwarn("Libvirt cannot be imported on this VM")
    pass    

try :
    # Provided by RHEL 6.2+
    from libvirt_qemu import qemuMonitorCommand
    qemu_supported = True
except ImportError:
    cbwarn("QEMU scraper cannot run without qemuMonitorCommand")
    pass

slope_str2int = {'zero':0,
                 'positive':1,
                 'negative':2,
                 'both':3,
                 'unspecified':4}

# could be autogenerated from previous but whatever
slope_int2str = {0: 'zero',
                 1: 'positive',
                 2: 'negative',
                 3: 'both',
                 4: 'unspecified'}

class Gmetric:
    """
    Class to send gmetric/gmond 2.X packets

    Thread safe
    """

    type = ('', 'string', 'uint16', 'int16', 'uint32', 'int32', 'float',
            'double', 'timestamp')
    protocol = ('udp', 'multicast')

    def __init__(self, host, port, protocol):
        if protocol not in self.protocol:
            raise ValueError("Protocol must be one of: " + str(self.protocol))

        self.socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        if protocol == 'multicast':
            self.socket.setsockopt(socket.IPPROTO_IP,
                                   socket.IP_MULTICAST_TTL, 20)
        self.hostport = (host, int(port))
        #self.socket.connect(self.hostport)

    def send(self, NAME, VAL, TYPE='', UNITS='', SLOPE='both',
             TMAX=60, DMAX=0, GROUP="", SPOOF=""):
        if SLOPE not in slope_str2int:
            raise ValueError("Slope must be one of: " + str(list(self.slope.keys())))
        if TYPE not in self.type:
            raise ValueError("Type must be one of: " + str(self.type))
        if len(NAME) == 0:
            raise ValueError("Name must be non-empty")

        ( meta_msg, data_msg )  = gmetric_write(NAME, VAL, TYPE, UNITS, SLOPE, TMAX, DMAX, GROUP, SPOOF)

        self.socket.sendto(meta_msg, self.hostport)
        self.socket.sendto(data_msg, self.hostport)

def gmetric_write(NAME, VAL, TYPE, UNITS, SLOPE, TMAX, DMAX, GROUP, SPOOF):
    """
    Arguments are in all upper-case to match XML
    """
    packer = Packer()
    HOSTNAME="test"
    if SPOOF == "":
        SPOOFENABLED=0
    else :
        SPOOFENABLED=1
    # Meta data about a metric
    packer.pack_int(128)
    if SPOOFENABLED == 1:
        packer.pack_string(SPOOF)
    else:
        packer.pack_string(HOSTNAME)
    packer.pack_string(NAME)
    packer.pack_int(SPOOFENABLED)
    packer.pack_string(TYPE)
    packer.pack_string(NAME)
    packer.pack_string(UNITS)
    packer.pack_int(slope_str2int[SLOPE]) # map slope string to int
    packer.pack_uint(int(TMAX))
    packer.pack_uint(int(DMAX))
    # Magic number. Indicates number of entries to follow. Put in 1 for GROUP
    if GROUP == "":
        packer.pack_int(0)
    else:
        packer.pack_int(1)
        packer.pack_string("GROUP")
        packer.pack_string(GROUP)

    # Actual data sent in a separate packet
    data = Packer()
    data.pack_int(128+5)
    if SPOOFENABLED == 1:
        data.pack_string(SPOOF)
    else:
        data.pack_string(HOSTNAME)
    data.pack_string(NAME)
    data.pack_int(SPOOFENABLED)
    data.pack_string("%s")
    data.pack_string(str(VAL))

    return ( packer.get_buffer() ,  data.get_buffer() )

class PassiveObjectOperations(BaseObjectOperations) :
    
    @trace
    def keyfunc(self, x):
        return x.split("|")[1]
    
    def namefunc(self, x):
        return x["name"]
    
    @trace
    def get_fields(self, obj_type, criteria = "attached"):
        if obj_type == "CLOUD" :
            _obj_inst = self.pid
            _fields = []
            _fields.append("|name                    ")
            _fields.append("|model                   ")
            _fields.append("|description             ")
        
        elif obj_type == "VMC" :
            _fields = []
            _fields.append("|name                    ")
            _fields.append("|host_count      ")
            _fields.append("|pool                    ")
            _fields.append("|cloud_hostname                  ")
            _fields.append("|cloud_ip         ")        

        elif obj_type == "HOST" :
            _fields = []
            _fields.append("|name                               ")
            _fields.append("|vmc_name            ")
            _fields.append("|function                                          ")
            _fields.append("|pool         ")
            _fields.append("|cloud_hostname              ")
            _fields.append("|cloud_ip        ")      

        elif obj_type == "VM" :
            _fields = []
            _fields.append("|name          ")
            _fields.append("|role                ")
            _fields.append("|size        ")
            _fields.append("|cloud_ip        ")
            _fields.append("|host_name                  ")
            _fields.append("|vmc_pool      ")
            _fields.append("|netname    ")            
            _fields.append("|ai      ")
            _fields.append("|aidrs      ")
            if criteria == "failed" :
                _fields.append("|abort                        ")            
            _fields.append("|uuid")
#            _fields.append("|uuid                                 ")
        elif obj_type == "AI" :
            _fields = []
            _fields.append("|name      ")
            _fields.append("|type           ")
#            _fields.append("|uuid                                 ")
            _fields.append("|sut                                               ")
            _fields.append("|cloud_ip        ")
            _fields.append("|arrival        ")
            _fields.append("|aidrs                                   ")
            _fields.append("|uuid")
        elif obj_type == "AIDRS" :
            _fields = []
            _fields.append("|name                ")
            _fields.append("|pattern                ")
#            _fields.append("|uuid                                 ")
            _fields.append("|type              ")
        elif obj_type == "VMCRS" :
            _fields = []
            _fields.append("|name                ")
            _fields.append("|scope                ")
#            _fields.append("|uuid                                 ")
        elif obj_type == "FIRS" :
            _fields = []
            _fields.append("|name                ")
            _fields.append("|scope                ")
#            _fields.append("|uuid                                 ")
        else :
            _msg = "Unknown object: " + obj_type
            raise self.ObjectOperationException(_msg, 28)
        
        return _fields
        
    @trace
    def get_display_value(self, obj_attrs, translation_cache, field, cloud_name) :
        '''
        TBD
        '''
        _af = field[1:].strip()
        if _af == "vmc" or \
        (_af == "ai" and "ai" in obj_attrs and obj_attrs[_af] != "none") or \
        (_af == "aidrs" and "aidrs" in obj_attrs and obj_attrs[_af] != "none") :
            _obj_name = self.fast_uuid_to_name(cloud_name, _af.split("_")[-1].upper(), \
                                               obj_attrs[_af], \
                                               translation_cache)
            
            #Screen is too small, just show names.
            #User can then later type 'ailist' or 'aslist' or 'vmclist'
            #to discover the UUID that they are interested in
            #_display_value = _obj_attrs[_af] + ' (' + _obj_name + ')'
            _display_value = _obj_name
        else :
            if _af in obj_attrs :
                _display_value = str(obj_attrs[_af])
            else :
                _display_value = "unavailable" 
            
        return _display_value

    @trace
    def list_objects(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        try :
            _print_state = "attached"
            _status = 100
            _result = []
            _fmsg = "An error has occurred, but no error message was captured"
            _translation_cache = {}
            _fmt_obj_list = "No objects available."
            _obj_type = command.split('-')[0].upper()
            _obj_list = []
            _total = 0

            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)
            
            if not _status :
                _fields = self.get_fields(_obj_type, obj_attr_list["state"])                
                _status, _fmsg = self.initialize_object(obj_attr_list, command)

                if not _status and (obj_attr_list["state"].lower() not in ["pending", "failed", "finished"]):
                    if "limit" in obj_attr_list and obj_attr_list["limit"] != "none" :
                        _limit = int(obj_attr_list["limit"])
                    else:
                        _limit = 0
    
                    _fmt_obj_list = ''.join(_fields) + '\n'
    
                    if _obj_type == "CLOUD" :
                        _obj_list = self.osci.get_object_list(obj_attr_list["cloud_name"], _obj_type)
                    else :
                        _obj_list = self.osci.query_by_view(obj_attr_list["cloud_name"], _obj_type, "BYUSERNAME", obj_attr_list["username"], "name", "all", False)
                        
                        if "regression" in obj_attr_list and obj_attr_list["regression"] == "true" :
                            _obj_list.sort(key=self.keyfunc)
    
                    if _obj_list :
                        for _obj in _obj_list :
                            if _obj.count('|') :
                                _obj_uuid, _obj_name = _obj.split('|')
                            else :
                                _obj_uuid = _obj
                            try :
                                _obj_attrs = self.osci.get_object(obj_attr_list["cloud_name"], _obj_type, False, _obj_uuid, False)
                            except self.osci.ObjectStoreMgdConnException as e :
                                if _obj_type == "CLOUD" :
                                    # We have stabilized the use of a shared Redis database. Unfortunately,
                                    # this allows multiple users to leave leftover information in the database,
                                    # which can happen for various reasons. In this case, if a cloud was leftover
                                    # it can cause the API service to fail to startup. If we find such information
                                    # just skip the invalid Cloud and move on with a warning.
                                    cbwarn("Warning: Cloud " + _obj_uuid + " does not exist.", True)
                                    continue
                                raise e
    
                            if _obj_type == "VM" or _obj_type == "AI" :
                                _state = self.osci.get_object_state(obj_attr_list["cloud_name"], _obj_type, _obj_uuid)
                                _obj_attrs['state'] = _state
                                if obj_attr_list["state"]  != "all" : 
                                    if _state != obj_attr_list["state"] :
                                        continue
                                
                            if _limit and _total == _limit :
                                break
                            
                            _total += 1
                            _result.append(_obj_attrs)
                                
                             
                            for _field in _fields :
                                _display_value = self.get_display_value(_obj_attrs, _translation_cache, _field, obj_attr_list["cloud_name"])
                                _fmt_obj_list += ('|' + _display_value).ljust(len(_field))
                            _fmt_obj_list += '\n'
    
                    else :
                        _fmt_obj_list = "No objects available."

                if obj_attr_list["state"] == "pending" :
                    objs = self.osci.get_list(obj_attr_list["cloud_name"], _obj_type, "PENDING", True)
                    if len(objs) :
                        _print_state = "pending"
                        
                    _fmt_obj_list = ''.join(_fields) + '\n'
                    for obj in objs :
                        _obj_uuid, _obj_name = obj[0].split("|")
                        _new_result = self.osci.pending_object_get(obj_attr_list["cloud_name"], _obj_type, _obj_uuid)
                        _new_result.update({"uuid" : _obj_uuid, "name" : _obj_name, "status" : "pending", "tracking" : _new_result["status"]})
                        _result.append(_new_result)
                        for _field in _fields :
                            _display_value = self.get_display_value(_new_result, _translation_cache, _field, obj_attr_list["cloud_name"])
                            _fmt_obj_list += ('|' + _display_value).ljust(len(_field))
                        _fmt_obj_list += '\n'
                        
                for state in ["failed", "finished" ] :
                    if obj_attr_list["state"] != state :
                        continue
                    
                    _print_state = state
                    objs = self.osci.get_object_list(obj_attr_list["cloud_name"], state.upper() + "TRACKING" + _obj_type, True)
                    if not objs :
                        continue
                     
                    _fmt_obj_list = ''.join(_fields) + '\n'
                    for obj in objs :
                        sub_attrs = self.osci.get_object(obj_attr_list["cloud_name"], state.upper() + "TRACKING" + _obj_type, False, obj, False)
                        sub_attrs_split = obj.split("-")
                        sub_attrs["operation"] = sub_attrs_split[5]
                        sub_attrs["order"] = sub_attrs_split[6]
                        _result.append(sub_attrs)
                        for _field in _fields :
                            _display_value = self.get_display_value(sub_attrs, _translation_cache, _field, obj_attr_list["cloud_name"])
                            _fmt_obj_list += ('|' + _display_value).ljust(len(_field))
                        _fmt_obj_list += '\n'
    
                _status = 0
            
            else :
                _fmt_obj_list = "No objects available."

        except self.ObjectOperationException as obj :
            for line in traceback.format_exc().splitlines() :
                cbwarn(line, True)
            _status = 8
            _fmsg = str(obj.msg)

        except self.osci.ObjectStoreMgdConnException as obj :
            for line in traceback.format_exc().splitlines() :
                cbwarn(line, True)
            _status = 2
            _fmsg = str(obj.msg)

        except Exception as e :
            for line in traceback.format_exc().splitlines() :
                print(line)
            _status = 24
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "Unable to get object list: " + _fmsg
                cberr(_msg)
            else : 
                _msg = "The following " + _print_state.upper() + " " + _obj_type + "s are in this "
                _msg += "experiment (Cloud "
                _msg += obj_attr_list["cloud_name"]  + ") :\n" + _fmt_obj_list
                cbdebug(_msg)
                
            return self.package(_status, _msg, _result)
        
    @trace
    def show_object(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        try :
            _status = 100
            _obj_type = command.split('-')[0].upper()
            _result = {}

            obj_attr_list["cloud_name"] = "undefined"
            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)

            if not _status :
                _status, _fmsg = self.initialize_object(obj_attr_list, command)

                if not _status :

                    _obj_select_attribs = obj_attr_list["specified_attributes"].split(',')
         
                    _smsg = "The " + _obj_type + " object " + obj_attr_list["name"]
                    _smsg += ", attached to this experiment, has the "
                    _smsg += "following attributes (Cloud "
                    _smsg += obj_attr_list["cloud_name"] + ") :\n"
            
                    _fmsg = "Unable to get the attributes for the " + _obj_type 
                    _fmsg += " object (Cloud " + obj_attr_list["cloud_name"] + "): "
        
                    if _obj_type == "CLOUD" :
                        _obj_type = "GLOBAL"
                        _fields = []            
                        _fields.append("|attribute (GLOBAL object)        ")
                        _fields.append("|\"sub-attribute\" (key)          ")
                        _fields.append("|value                                ")
        
                        if _obj_select_attribs[0] == "all" :
                            _obj_ids = sorted(obj_attr_list["all"].replace("command,",'').split(','))
                        else :
                            _obj_ids = _obj_select_attribs
                        
                        _fmt_obj_attr_list = ''.join(_fields) + '\n'
        
                        for _obj_id in _obj_ids :
                            try :
                                _obj_attribs = self.osci.get_object(obj_attr_list["cloud_name"], _obj_type, False, \
                                                                    _obj_id, False)
                            except self.osci.ObjectStoreMgdConnException as e :
                                if e.status == 23 :
                                    cbwarn("Could not find: " + _obj_id + ". Continuing.")
                                else :
                                    raise e
                            
                            _result[_obj_id] = {}
    
                            for _attrib, _value in iter(sorted(_obj_attribs.items())) :
                                _fmt_obj_attr_list += '|' + _obj_id.ljust(len(_fields[0]) - 1)
                                _fmt_obj_attr_list += '|' + _attrib.ljust(len(_fields[1]) - 1)
                                _fmt_obj_attr_list += '|' + _value.ljust(len(_fields[2]) - 1)
                                _fmt_obj_attr_list += '\n'
                                
                                '''
                                    This is a method of reverse-engineering the cloud attributes
                                    in a way that is displayable in group form on the GUI
                                    the same way it is ordered in the configuration file.
                                '''
                                if _obj_select_attribs[0] == "all" :
                                    components = _attrib.split("_", 1)
                                    if len(components) == 1 :
                                        prefix = _attrib
                                        suffix = None
                                    else :
                                        prefix, suffix = components 
                                    
                                    if prefix in _result[_obj_id] and isinstance(_result[_obj_id][prefix], list):
                                        _result[_obj_id][prefix].append((suffix, _value))
                                    elif suffix != None :
                                        _result[_obj_id][prefix] = [(suffix, _value)]
                                    else :
                                        _result[_obj_id][prefix] = _value
                                        
                            for prefix in _result[_obj_id] :
                                liste = _result[_obj_id][prefix]
                                if isinstance(liste, list) and len(liste) == 1 :
                                    _result[_obj_id][prefix + "_" + liste[0][0]] = liste[0][1]
                                    del _result[_obj_id][prefix]
                                
                            if _obj_select_attribs[0] != "all" :
                                _result = copy.deepcopy(_obj_attribs)
        
                    elif _obj_type == "VMC" or _obj_type == "VM" or _obj_type == "HOST" or \
                        _obj_type == "AI" or _obj_type == "AIDRS" or \
                        _obj_type == "VMCRS" or _obj_type == "FIRS" :
                        _fields = []
                        _fields.append("|attribute (" + _obj_type + " object key)               ")
                        _fields.append("|value                                ")
    
                        _fmt_obj_attr_list = ''.join(_fields) + '\n'
                        
                        if obj_attr_list["state"] != "pending" :
                            if len(obj_attr_list["name"]) == 36 and obj_attr_list["name"].count('-') == 4 :
                                _is_tag = False
                            else :
                                _is_tag = True
                                
                            _obj_attribs = self.osci.get_object(obj_attr_list["cloud_name"], _obj_type, _is_tag, \
                                                                obj_attr_list["name"], \
                                                                False)
                            _obj_attribs["state"] = self.osci.get_object_state(obj_attr_list["cloud_name"], _obj_type, _obj_attribs["uuid"])
                        else :
                            if len(obj_attr_list["name"]) == 36 and obj_attr_list["name"].count('-') == 4 :
                                True
                            else :
                                # This code path is expected to be rarely used, mostly for debugging
                                for _obj_uuid  in self.osci.get_list(obj_attr_list["cloud_name"], _obj_type, "PENDING", True) :
                                    if _obj_uuid[0].count(obj_attr_list["name"]) :
                                        obj_attr_list["name"] = _obj_uuid[0].replace('|' + obj_attr_list["name"],'')
                            
                            _obj_attribs = self.osci.pending_object_get(obj_attr_list["cloud_name"], _obj_type, obj_attr_list["name"], _obj_select_attribs[0])
                            
                            _obj_attribs["state"] = "pending"
    
                        for _attrib, _value in iter(sorted(_obj_attribs.items())) :
                            if _attrib in _obj_select_attribs or \
                            _obj_select_attribs[0] == "all" :
                                _fmt_obj_attr_list += '|' + \
                                _attrib.ljust(len(_fields[0]) - 1)
                                _fmt_obj_attr_list += '|' + \
                                _value.ljust(len(_fields[1]) - 1)
                                _fmt_obj_attr_list += '\n'
                                
                        _result = copy.deepcopy(_obj_attribs)
                    else :
                        _msg = "Unknown object: " + _obj_type
                        raise self.ObjectOperationException(_msg, 28)            
        
                    _status = 0

        except self.ObjectOperationException as obj :
            for line in traceback.format_exc().splitlines() :
                cbwarn(line, True)
            _status = 8
            _fmsg = str(obj.msg)

        except self.osci.ObjectStoreMgdConnException as obj :
            for line in traceback.format_exc().splitlines() :
                cbwarn(line, True)
            _status = 8
            _fmsg = str(obj.msg)

        except Exception as e :
            for line in traceback.format_exc().splitlines() :
                cbwarn(line, True)
            _status = 23
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "Unable to get the attributes for the " + _obj_type 
                _msg += " object (Cloud " + obj_attr_list["cloud_name"] + "): "
                _msg += _fmsg
                cberr(_msg)
            else :
                _msg = "The " + _obj_type + " object " + obj_attr_list["name"]
                _msg += ", attached to this experiment, has the "
                _msg += "following attributes (Cloud "
                _msg += obj_attr_list["cloud_name"] + ") :\n" + _fmt_obj_attr_list
                cbdebug(_msg)
            return self.package(_status, _msg, _result)

    @trace    
    def alter_object(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        _xfmsg = ""
        _fmsg = ""
        try :
            _status = 100
            _obj_type = command.split('-')[0].upper()
            _result = {}

            obj_attr_list["cloud_name"] = "undefined"
            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)

            if not _status :
                _status, _fmsg = self.initialize_object(obj_attr_list, command)

                if not _status :
                    _obj_attrib_kvs = []
                    
                    # Some values, themselves, can have commas, which should not be treated like additional
                    # keys, in which case we need to re-combine them into a single value.
                    # To do that, search for list entries that do not contain equal signs 
                    
                    for _suspected_kv in obj_attr_list["specified_kv_pairs"].split(',') :
                        if _suspected_kv.count("=") :
                            _obj_attrib_kvs.append(_suspected_kv)
                        else :
                            _obj_attrib_kvs[-1] += "," + _suspected_kv
    
                    _fields = [] 
    
                    if _obj_type == "CLOUD" :
                        _is_tag = False
                        obj_attr_list["state"] = "default"
                        _obj_type = "GLOBAL"
                        _obj_uuid = obj_attr_list["specified_attributes"]
    
                        _fields.append("|\"sub-attribute\" (key)                ")
                        _fields.append("|old value                          ")
                        _fields.append("|new value                          ")
    
                        _smsg = "The global object \"" + obj_attr_list["specified_attributes"]
                        _smsg += "\" on Cloud " + obj_attr_list["cloud_name"] 
                        _smsg += " was modified:\n"
    
                        _fmsg = "Unable to change the \"sub-attributes\" " 
                        _fmsg += ','.join(_obj_attrib_kvs) + " part of the attribute \""
                        _fmsg += obj_attr_list["specified_attributes"] + "\" on Cloud " + obj_attr_list["cloud_name"] + ": "
    
                    else :
                        if len(obj_attr_list["name"]) == 36 and obj_attr_list["name"].count('-') == 4 :
                            _is_tag = False
                        else :
                            _is_tag = True
                            
                        _obj_uuid = obj_attr_list["name"]
    
                        _fields.append("|attribute                              ")
                        _fields.append("|old value                          ")
                        _fields.append("|new value                          ")
    
                        _smsg = "The following attributes for the " + _obj_type
                        _smsg += " object were changed (Cloud " + obj_attr_list["cloud_name"] + "):\n"
    
                        _fmsg = "Unable to change the attributes " 
                        _fmsg += ','.join(_obj_attrib_kvs) + " for the " + _obj_type
                        _fmsg += " (Cloud " + obj_attr_list["cloud_name"] + ")."
    
                    _fmt_obj_chg_attr = ''.join(_fields) + '\n'
    
                    if obj_attr_list["state"] != "pending" :
                        _old_values = self.osci.get_object(obj_attr_list["cloud_name"], _obj_type, _is_tag, \
                                                           _obj_uuid, False)
                    else :
                        if len(obj_attr_list["name"]) == 36 and obj_attr_list["name"].count('-') == 4 :
                            True
                        else :
                            # This code path is expected to be rarely used, mostly for debugging
                            for _c_obj_uuid  in self.osci.get_list(obj_attr_list["cloud_name"], _obj_type, "PENDING", True) :
                                if _c_obj_uuid[0].count(obj_attr_list["name"]) :
                                    _obj_uuid = _c_obj_uuid[0].replace('|' + obj_attr_list["name"],'')
                        
                        _old_values = self.osci.pending_object_get(obj_attr_list["cloud_name"], _obj_type, _obj_uuid, "all")                    

                    for _kv in _obj_attrib_kvs :
                        # use split('=', 1) to allow for value to contain
                        # equal signs themselves as a part of the value
                        _key, _value = _kv.split('=', 1)
    
                        if not _key in _old_values :
                            _old_values[_key] = "non-existent"
    
                        if _key[0] == '-' and obj_attr_list["state"] != "pending" :
                            _key = _key[1:]
                            self.osci.remove_object_attribute(obj_attr_list["cloud_name"], _obj_type, _obj_uuid, \
                                                              _is_tag, _key)
                        else :
                            if _value.count("C+") and not _value.count(',') :
                                _counter = True
                                _value = _value.replace("C+",'+')
                            elif _value.count("C-") and not _value.count(',') :
                                _value = _value.replace("C-",'-')
                                _counter = True
                            else :
                                _counter = False

                            if obj_attr_list["state"] != "pending" :
                                self.osci.update_object_attribute(obj_attr_list["cloud_name"], _obj_type, _obj_uuid, \
                                                                  _is_tag, _key, _value, _counter)
                            else :
                                self.osci.pending_object_set(obj_attr_list["cloud_name"], _obj_type, _obj_uuid, _key, _value)
    
                        _fmt_obj_chg_attr += '|' + _key.ljust(len(_fields[0]) - 1)
                        _fmt_obj_chg_attr += '|' + _old_values[_key].ljust(len(_fields[1]) - 1)
                        _fmt_obj_chg_attr += '|' + _value.ljust(len(_fields[2]) - 1) + '\n'
                        _result["old_" + _key] = _old_values[_key]
                        _result[_key] = _value
    
                    _smsg += _fmt_obj_chg_attr
    
                    _status = 0

        except self.ObjectOperationException as obj :
            _status = 8
            _xfmsg = str(obj.msg)

        except self.osci.ObjectStoreMgdConnException as obj :
            _status = 8
            _xfmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _xfmsg = str(e)

        finally :
            if _status :
                _msg = _fmsg + _xfmsg
                cberr(_fmsg)
            else :                
                if obj_attr_list["specified_kv_pairs"].count("experiment_id") :
                    _msg = "The attribute \" experiment_id \" was changed."
                    _msg += " Checking if a Host OS performance data collection"
                    _msg += " daemon restart is needed."
                    cbdebug(_msg)
                    self.update_host_os_perfmon(obj_attr_list)
                    self.update_logstore(obj_attr_list)
                _msg = _smsg
                cbdebug(_smsg)

            return self.package(_status, _msg, _result)

    @trace
    def show_view(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''        
        try :
            _fmsg = ""
            _status = 100
            _obj_type = "undefined"

            _criterion = "byundefined"
            _expression = "undefined"
            _filter = "undefined"
            _sorting = "arrival"
            _result = []

            obj_attr_list["cloud_name"] = "undefined"

            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)

            if not _status :
                _status, _fmsg = self.initialize_object(obj_attr_list, command)

            if not _status :
                _obj_type =  obj_attr_list["object_type"]
                _criterion = "BY" + obj_attr_list["criterion"] 
                _expression = obj_attr_list["expression"]
                _sorting = obj_attr_list["sorting"]
                _filter = obj_attr_list["filter"]

                _fields = []
    
                _fields.append("|Object Type                 ")
                _fields.append("|Predicate                                                ")

                _expression_list = self.osci.query_by_view(obj_attr_list["cloud_name"], _obj_type, _criterion, _expression, _sorting, _filter, True)
                _fields.append("|Object UUID                           ")
                _fields.append("|Object Name            ")
                _fields.append("|" + _sorting.capitalize() + " Time                       ")
                _fmt_obj_list = ''.join(_fields) + '\n'
                for _item in _expression_list :
                    if _expression_list.index(_item) == 0 :
                        _fmt_obj_list += ('|' + _obj_type).ljust(len(_fields[0]))
                        _fmt_obj_list += ("|where " + _criterion.upper() + " = " +  _expression).ljust(len(_fields[1]))
                    else :
                        _fmt_obj_list += ('|').ljust(len(_fields[0]))
                        _fmt_obj_list += ('|').ljust(len(_fields[1]))
                    _uuid, _name = _item[0].split('|')
                    _result.append({"uuid" : _uuid, "name" : _name})
                    _fmt_obj_list += ('|' + _uuid).ljust(len(_fields[2]))
                    _fmt_obj_list += ('|' + _name).ljust(len(_fields[3]))
                    if _item[1] != "Empty" :
                        _hrtf = asctime(localtime(float(_item[1])))
                    else :
                        _hrtf = "Empty"
                    _fmt_obj_list += ('|' + _hrtf + " (" + str(_item[1]) + ')').ljust(len(_fields[4]))
                    _fmt_obj_list += '\n'

                _status = 0

        except self.ObjectOperationException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except self.osci.ObjectStoreMgdConnException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if not _expression :
                _expression = _filter
            if _status :
                _msg = "Unable to get the list of " + _obj_type + " objects "
                _msg += "available on the view \"" + _criterion + "\" where \""
                _msg += _criterion + " = " + _expression + "\" on this "
                _msg += "experiment (Cloud "
                _msg += obj_attr_list["cloud_name"]  + "), sorted by "
                _msg += _sorting + "): " + _fmsg
                cberr(_msg)
            else :
                _msg = "The following " + _obj_type + " objects are part of the"
                _msg += " view \"" + _criterion + "\" where \""
                _msg += _criterion + " = " + _expression + "\" on this "
                _msg += "experiment (Cloud "
                _msg += obj_attr_list["cloud_name"]  + "), sorted by " 
                _msg += _sorting + ") :\n" + _fmt_obj_list
                cbdebug(_msg)
                
            return self.package(_status, _msg, _result)

    @trace
    def stats(self, obj_attr_list, parameters, command, internal = False) :
        '''
        TBD
        '''
        try :
            _status = 100
            _stats = {}
            _msci = False
            
            obj_attr_list["cloud_name"] = "undefined"
            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)
            
            if not _status :
                _status, _fmsg = self.initialize_object(obj_attr_list, command)

                if not _status :
                    
                    _stats["object_store"] = {}
                    _stats["metric_store"] = {}                    
                    _stats["experiment_objects"] = {}
                    _stats["experiment_counters"] = {}
                    _stats["vmc-wide_counters"] = {}
                    _stats["aidrs-wide_counters"] = {}

                    _query_object = self.osci.get_object(obj_attr_list["cloud_name"], "GLOBAL", False, "query", False)

                    if obj_attr_list["type"] == "all" :
                        _obj_list = _query_object["object_type_list"].split(',')
                    else :
                        _obj_list = obj_attr_list["type"].upper().split(',')
                    _msci = self.get_msci(obj_attr_list["cloud_name"])
                    _fmt_obj_list = "\n"

                    if obj_attr_list["type"] == "all" :

                        _info = self.osci.get_info()
                        _fields = []

                        if obj_attr_list["output"] == "print" :  
                            _fmt_obj_list = "------------------------- OBJECT STORE -----------------------\n"
        
                            _fields.append("|Metric                                               ")
                            _fields.append("|Value                         ")
                            _fmt_obj_list += ''.join(_fields) + '\n'
                        
                        for _line in _info :
                            _stats["object_store"][_line[0]] = str(_line[1])
                            if obj_attr_list["output"] == "print" :                              
                                _fmt_obj_list += ('|' + _line[0]).ljust(len(_fields[0]))
                                _fmt_obj_list += ('|' + _line[1]).ljust(len(_fields[1]))
                                _fmt_obj_list += '\n'

                        _info = _msci.get_info()
                        
                        if obj_attr_list["output"] == "print" :  
                            _fmt_obj_list += "------------------------- METRIC STORE -----------------------\n"
                            
                            _fields = []
                            
                            _fields.append("|Metric                                               ")
                            _fields.append("|Value                         ")
                            _fmt_obj_list += ''.join(_fields) + '\n'
                            
                        for _line in _info :
                            _stats["metric_store"][_line[0]] = str(_line[1])

                        if obj_attr_list["output"] == "print" :                              
                            _fmt_obj_list += ('|' + _line[0]).ljust(len(_fields[0]))
                            _fmt_obj_list += ('|' + _line[1]).ljust(len(_fields[1]))
                            _fmt_obj_list += '\n'

                    if obj_attr_list["output"] == "print" :  
                        _fmt_obj_list += "--------------------- EXPERIMENT OBJECTS ---------------------\n" 
    
                        _fields = []    
                        _fields.append("|Object                                               ")
                        _fields.append("|Count                         ")
                        _fmt_obj_list += ''.join(_fields) + '\n'
    
                    for _obj_type in _obj_list :
                        _obj_count = str(self.get_object_count(obj_attr_list["cloud_name"], _obj_type))
                        _stats["experiment_objects"][_obj_type] = _obj_count
                        
                        if obj_attr_list["output"] == "print" :                          
                            _fmt_obj_list += ('|' + _obj_type + 's').ljust(len(_fields[0]))
                            _fmt_obj_list += ('|' + _obj_count ).ljust(len(_fields[1]))
                            _fmt_obj_list += '\n'

                    if obj_attr_list["output"] == "print" :                                              
                        _fmt_obj_list += "------------------ EXPERIMENT-WIDE COUNTERS ------------------\n" 
                        _fields = []
        
                        _fields.append("|Counter                                              ")
                        _fields.append("|Value                         ")
                        _fmt_obj_list += ''.join(_fields) + '\n'
    
                    for _obj_type in _obj_list :
                        if _obj_type not in _stats["experiment_counters"] :
                            _stats["experiment_counters"][_obj_type] = {}
                            
                        _obj_count = str(self.get_object_count(obj_attr_list["cloud_name"], _obj_type, "RESERVATIONS"))
                        _stats["experiment_counters"][_obj_type]["reservations"] = _obj_count

                        if obj_attr_list["output"] == "print" :                                
                            _fmt_obj_list += ('|' + _obj_type + " RESERVATIONS").ljust(len(_fields[0]))
                            _fmt_obj_list += ('|' + _obj_count).ljust(len(_fields[1]))
                            _fmt_obj_list += '\n'

                        if _obj_type == "VM" :
                            
                            if str(_query_object["get_vm_list"]).lower() == "true" or str(obj_attr_list["include_vmcount"]) == "true" :
                                _vm_defaults = self.osci.get_object(obj_attr_list["cloud_name"], "GLOBAL", False, "vm_defaults", False)                                
                                _vm_defaults["cloud_name"] = obj_attr_list["cloud_name"]                            
                                self.set_cloud_operations_instance(obj_attr_list["cloud_model"])      
                                _cld_conn = self.coi[obj_attr_list["cloud_model"]][self.pid + '-' + self.expid]
                                _stats["experiment_counters"][_obj_type]["reported"] = _cld_conn.vmcount(_vm_defaults)
                                _obj_count = str(_stats["experiment_counters"][_obj_type]["reported"])

                                if obj_attr_list["output"] == "print" :                                
                                    _fmt_obj_list += ('|' + _obj_type + "s REPORTED").ljust(len(_fields[0]))
                                    _fmt_obj_list += ('|' + _obj_count).ljust(len(_fields[1]))
                                    _fmt_obj_list += '\n'

                        _obj_count = str(self.get_object_count(obj_attr_list["cloud_name"], _obj_type, "COUNTER"))
                        _stats["experiment_counters"][_obj_type]["issued"] = _obj_count

                        if obj_attr_list["output"] == "print" :
                            _fmt_obj_list += ('|' + _obj_type + "s ISSUED").ljust(len(_fields[0]))
                            _fmt_obj_list += ('|' + _obj_count).ljust(len(_fields[1]))
                            _fmt_obj_list += '\n'
                                                    
                        _obj_count = str(self.get_object_count(obj_attr_list["cloud_name"], _obj_type, "ARRIVED"))
                        _stats["experiment_counters"][_obj_type]["arrived"] = _obj_count

                        if obj_attr_list["output"] == "print" :
                            _fmt_obj_list += ('|' + _obj_type + "s ARRIVED").ljust(len(_fields[0]))
                            _fmt_obj_list += ('|' + _obj_count).ljust(len(_fields[1]))
                            _fmt_obj_list += '\n'
                        
                        _obj_count = str(self.get_object_count(obj_attr_list["cloud_name"], _obj_type, "ARRIVING"))
                        _stats["experiment_counters"][_obj_type]["arriving"] = _obj_count

                        if obj_attr_list["output"] == "print" :                                                  
                            _fmt_obj_list += ('|' + _obj_type + "s ARRIVING").ljust(len(_fields[0]))
                            _fmt_obj_list += ('|' + _obj_count).ljust(len(_fields[1]))
                            _fmt_obj_list += '\n'

                        if _obj_type == "VM" or _obj_type == "AI" :                            
                            _obj_count = str(self.get_object_count(obj_attr_list["cloud_name"], _obj_type, "CAPTURING"))
                            _stats["experiment_counters"][_obj_type]["capturing"] = _obj_count
                            
                            if obj_attr_list["output"] == "print" :                                                      
                                _fmt_obj_list += ('|' + _obj_type + "s CAPTURING").ljust(len(_fields[0]))
                                _fmt_obj_list += ('|' + _obj_count).ljust(len(_fields[1]))
                                _fmt_obj_list += '\n'

                        _obj_count = str(self.get_object_count(obj_attr_list["cloud_name"], _obj_type, "DEPARTED"))
                        _stats["experiment_counters"][_obj_type]["departed"] = _obj_count
                        
                        if obj_attr_list["output"] == "print" :                                                  
                            _fmt_obj_list += ('|' + _obj_type + "s DEPARTED").ljust(len(_fields[0]))
                            _fmt_obj_list += ('|' + _obj_count).ljust(len(_fields[1]))
                            _fmt_obj_list += '\n'

                        _obj_count = str(self.get_object_count(obj_attr_list["cloud_name"], _obj_type, "DEPARTING"))
                        _stats["experiment_counters"][_obj_type]["departing"] = _obj_count
                        
                        if obj_attr_list["output"] == "print" :                                                  
                            _fmt_obj_list += ('|' + _obj_type + "s DEPARTING").ljust(len(_fields[0]))
                            _fmt_obj_list += ('|' + _obj_count).ljust(len(_fields[1]))
                            _fmt_obj_list += '\n'
                        
                        _obj_count = str(self.get_object_count(obj_attr_list["cloud_name"], _obj_type, "FAILED"))
                        _stats["experiment_counters"][_obj_type]["failed"] = _obj_count

                        if obj_attr_list["output"] == "print" :                                                  
                            _fmt_obj_list += ('|' + _obj_type + "s FAILED").ljust(len(_fields[0]))
                            _fmt_obj_list += ('|' + _obj_count).ljust(len(_fields[1]))
                            _fmt_obj_list += '\n'

                        if _obj_type == "VM" :
                            _obj_count = str(len(self.osci.query_by_view(obj_attr_list["cloud_name"], _obj_type, "BYSLA_PROVISIONING", "violated")))
                            _stats["experiment_counters"][_obj_type]["sla_provisioning_violated"] = _obj_count
    
                            if obj_attr_list["output"] == "print" :                                                  
                                _fmt_obj_list += ('|' + _obj_type + "s SLA PROVISIONING VIOLATED").ljust(len(_fields[0]))
                                _fmt_obj_list += ('|' + _obj_count).ljust(len(_fields[1]))
                                _fmt_obj_list += '\n'

                            _obj_count = str(len(self.osci.query_by_view(obj_attr_list["cloud_name"], _obj_type, "BYSLA_RUNTIME", "violated")))
                            _stats["experiment_counters"][_obj_type]["sla_runtime_violated"] = _obj_count
    
                            if obj_attr_list["output"] == "print" :                                                  
                                _fmt_obj_list += ('|' + _obj_type + "s SLA RUNTIME VIOLATED").ljust(len(_fields[0]))
                                _fmt_obj_list += ('|' + _obj_count).ljust(len(_fields[1]))
                                _fmt_obj_list += '\n'

                            _obj_count = str(len(self.osci.query_by_view(obj_attr_list["cloud_name"], _obj_type, "BYERRORS", "yes")))
                            _stats["experiment_counters"][_obj_type]["app_errors"] = _obj_count

                            if obj_attr_list["output"] == "print" :                                                  
                                _fmt_obj_list += ('|' + _obj_type + "s APPLICATION ERRORS").ljust(len(_fields[0]))
                                _fmt_obj_list += ('|' + _obj_count).ljust(len(_fields[1]))
                                _fmt_obj_list += '\n'
    
                    _obj_count = str(self.osci.count_object(obj_attr_list["cloud_name"], "GLOBAL", "experiment_counter"))
                    _stats["experiment_counters"]["Experiment Counter"] = _obj_count
                    
                    if obj_attr_list["output"] == "print" :                                              
                        _fmt_obj_list += "|EXPERIMENT COUNTER".ljust(len(_fields[0]))
                        _fmt_obj_list += ('|' + _obj_count).ljust(len(_fields[1]))
                        _fmt_obj_list += '\n'

                    if "VMC" in _obj_list :
                        _vmc_uuid_list = self.osci.get_object_list(obj_attr_list["cloud_name"], "VMC")                        

                        if _vmc_uuid_list :
                            if obj_attr_list["output"] == "print" :                                                      
                                _fmt_obj_list += "\n ---------------- VMC-WIDE COUNTERS ----------------\n"
    
                            for _vmc_uuid in _vmc_uuid_list :
                                if _vmc_uuid not in _stats["vmc-wide_counters"] :
                                    _stats["vmc-wide_counters"][_vmc_uuid] = {}
                                    _stats["vmc-wide_counters"][_vmc_uuid]["VM"] = {}
                                    
                                _vmc_attr_list = self.osci.get_object(obj_attr_list["cloud_name"], "VMC", False, _vmc_uuid, False)
                                _nr_vms = str(_vmc_attr_list["nr_vms"])
                                _stats["vmc-wide_counters"][_vmc_uuid]["VM"]["reservations"] =  _nr_vms

                                if obj_attr_list["output"] == "print" :                                                          
                                    _fmt_obj_list += ('|' + _vmc_uuid + " (" + _vmc_attr_list["name"] + ") VM RESERVATIONS").ljust(len(_fields[0]))
                                    _fmt_obj_list += ('|' + _nr_vms).ljust(len(_fields[1]))
                                    _fmt_obj_list += '\n'

                    if "AIDRS" in _obj_list :
                        _aidrs_uuid_list = self.osci.get_object_list(obj_attr_list["cloud_name"], "AIDRS")
                        
                        if _aidrs_uuid_list :
                            if obj_attr_list["output"] == "print" :                                                      
                                _fmt_obj_list += "\n ---------------- AIDRS-WIDE COUNTERS ----------------\n"
                            
                            for _aidrs_uuid in _aidrs_uuid_list :
                                
                                if _aidrs_uuid not in _stats["aidrs-wide_counters"] :
                                    _stats["aidrs-wide_counters"][_aidrs_uuid] = {}
                                    _stats["aidrs-wide_counters"][_aidrs_uuid]["AI"] = {}
    
                                _aidrs_attr_list = self.osci.get_object(obj_attr_list["cloud_name"], "AIDRS", False, _aidrs_uuid, False)
                                _nr_ais = str(_aidrs_attr_list["nr_ais"])
                                _stats["aidrs-wide_counters"][_aidrs_uuid]["AI"]["reservations"] = str(_aidrs_attr_list["nr_ais"])

                                if obj_attr_list["output"] == "print" :                                                          
                                    _fmt_obj_list += ('|' + _aidrs_uuid + " (" + _aidrs_attr_list["name"] + ") AI RESERVATIONS").ljust(len(_fields[0]))
                                    _fmt_obj_list += ('|' + _nr_ais).ljust(len(_fields[1]))
                                    _fmt_obj_list += '\n'

                    _status = 0

        except self.ObjectOperationException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except self.osci.ObjectStoreMgdConnException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except Exception as e :
            for line in traceback.format_exc().splitlines() :
                cbwarn(line, True)
            _status = 23
            _fmsg = str(e)

        finally :
             
            if _status :
                _msg = "Unable to get the values of the counters available on "
                _msg += "this experiment (Cloud "
                _msg += obj_attr_list["cloud_name"] + "): " + _fmsg
                cberr(_msg)
            else :
                if len(_fmt_obj_list) > 5 :
                    _msg = "The following statistics are available on this "
                    _msg += "experiment (Cloud " + obj_attr_list["cloud_name"]
                    _msg += ") :\n" + _fmt_obj_list
                    cbdebug(_msg)
                else :
                    _msg = "Success (data returned on a JSON object)"
                    cbdebug(_msg)

            if not internal :
                return self.package(_status, _msg, _stats)
            else :
                return _status, _msg, _stats

    @trace
    def show_state(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        try :
            _status = 100
            _obj_type = "undefined"
            _result = []

            obj_attr_list["cloud_name"] = "undefined"
            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)

            if "filter" in obj_attr_list :
                _filter = obj_attr_list["filter"]
                del obj_attr_list["filter"]
            else :
                _filter = False
                
            if not _status :
                _status, _fmsg = self.initialize_object(obj_attr_list, command)

                
            if not _status :
                _fmt_obj_header = "------------------ PER-OBJECT STATE (CLOUD "
                _fmt_obj_header += obj_attr_list["cloud_name"] + ")  -----------------\n" 
                _fields = []
                _fmt_obj_list = ""

                _fields.append("|Object Type    ")
                _fields.append("|Object Name    ")
                _fields.append("|Object UUID                             ")
                _fields.append("|State                     ")
                _fmt_obj_list += ''.join(_fields) + '\n'
                _query_object = self.osci.get_object(obj_attr_list["cloud_name"], "GLOBAL", False, "query", False)

                _count = 0
                
                for _obj_type in _query_object["object_type_list"].split(',') :
                    _obj_list = self.osci.get_object_list(obj_attr_list["cloud_name"], _obj_type)
                    if _obj_list :
                        for _obj_uuid in _obj_list :
                            _obj_state = self.osci.get_object_state(obj_attr_list["cloud_name"], _obj_type, _obj_uuid)
                            _obj_name = self.get_object_attribute(obj_attr_list["cloud_name"], \
                                                    _obj_type, _obj_uuid, "name")
                            if _filter and _obj_state != _filter :
                                continue
                            
                            _result.append({"type" : _obj_type, "name" : _obj_name, "uuid" : _obj_uuid, "state" : _obj_state})
                            
                        if "regression" in obj_attr_list and obj_attr_list["regression"] == "true" :
                            _result.sort(key=self.namefunc)
                            
                        for res in _result :
                            _fmt_obj_list += ('|' + res["type"]).ljust(len(_fields[0]))
                            _fmt_obj_list += ('|' + res["name"]).ljust(len(_fields[1]))
                            _fmt_obj_list += ('|' + res["uuid"]).ljust(len(_fields[2]))
                            _fmt_obj_list += ('|' + res["state"]).ljust(len(_fields[3]))
                            _fmt_obj_list += '\n'
                            _count += 1
                            
                if _count > 0 :
                    _fmt_obj_list = _fmt_obj_header + _fmt_obj_list
                else :
                    _fmt_obj_list = "No objects available.\n"

                _status = 0

        except self.ObjectOperationException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except self.osci.ObjectStoreMgdConnException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "Unable to get the values of the state for objects on "
                _msg += "this experiment (Cloud "
                _msg += obj_attr_list["cloud_name"] + "): " + _fmsg
                cberr(_msg)
            else :
                _msg = "The following state values are available for the objects"
                _msg +=" on this experiment (Cloud "
                _msg += obj_attr_list["cloud_name"] + ") :\n" + _fmt_obj_list
                cbdebug(_msg)
                
            return self.package(_status, _msg, _result)

    def reset_counters(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''

        '''
        TBD
        '''
        try :
            _status = 100
            _obj_type = "undefined"
            _result = []

            obj_attr_list["cloud_name"] = "undefined"
            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)
                
            if not _status :

                _status, _fmsg = self.initialize_object(obj_attr_list, command)

                _x, _y, _stats = self.stats(obj_attr_list, obj_attr_list["cloud_name"] + " all noprint false", "stats-get", True)

                _exp_counters = _stats["experiment_counters"]

                _aidrs = 0

                # Not sure what's going on here in simcloud. Needed a dirty fix.
                if str(_exp_counters["AIDRS"]["reservations"]) != "None" :
                    _aidrs += int(_exp_counters["AIDRS"]["reservations"])
                # Arrived doesn't mean that a submitter is present.
                #_aidrs += int(_exp_counters["AIDRS"]["arrived"])
                _aidrs += int(_exp_counters["AIDRS"]["arriving"]) 

                if _aidrs :
                    _fmsg = "Unable to reset counters. At least one VAppS (AIDRS)"
                    _fmsg += " is present on the experiment."
                    _status = 1972

                else :
                    
                    _status = 0
                    
                    for _obj in obj_attr_list["object_list"].split(',') :
                        _obj = _obj.upper()
                        
                        _obj_count = int(_exp_counters[_obj]["reservations"]) 
                        _obj_count += int(_exp_counters[_obj]["arriving"]) 

                        if _obj_count :
                            _fmsg = "Unable to reset counters. At least one " + _obj
                            _fmsg += " is present on the experiment."
                            _status = 1972
                    
                    if not _status :
                        self.osci.reset_counters(obj_attr_list["cloud_name"], {}, False, counter_list = obj_attr_list["object_list"].upper())
                        _x, _y, _stats = self.stats(obj_attr_list, obj_attr_list["cloud_name"] + " all noprint false", "stats-get", True)                    
                        _status = 0

        except self.ObjectOperationException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except self.osci.ObjectStoreMgdConnException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except Exception as e :
            for line in traceback.format_exc().splitlines() :
                cbwarn(line, True)
            _status = 23
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "Unable to reset the counters for objects \""  + obj_attr_list["object_list"]
                _msg += "\" on this experiment (Cloud "
                _msg += obj_attr_list["cloud_name"] + "): " + _fmsg
                cberr(_msg)
            else :
                _msg = "Successfully reset the counters for objects \""  + obj_attr_list["object_list"]
                _msg += "\" on this experiment (Cloud "
                _msg += obj_attr_list["cloud_name"] + ")."
                cbdebug(_msg)
                
            return self.package(_status, _msg, _stats)

    @trace
    def alter_state(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        try :
            _status = 100
            
            obj_attr_list["name"] = "undefined"
            obj_attr_list["cloud_name"] = "undefined"
            _old_state = "undefined"
            _result = None
            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)

            if not _status :
                _status, _fmsg = self.initialize_object(obj_attr_list, command)

            if not _status :
                _query_object = self.osci.get_object(obj_attr_list["cloud_name"], "GLOBAL", False, "query", False)

                for _obj_type in _query_object["object_type_list"].split(',') :
                    _obj_uuid = self.osci.object_exists(obj_attr_list["cloud_name"], _obj_type, obj_attr_list["name"], True)
                    if _obj_uuid :
                        _old_state = str(self.osci.get_object_state(obj_attr_list["cloud_name"], _obj_type, _obj_uuid))
                        str(self.osci.set_object_state(obj_attr_list["cloud_name"], _obj_type, _obj_uuid, obj_attr_list["specified_state"]))
                        break
                _result = obj_attr_list
                _status = 0

        except self.ObjectOperationException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except self.osci.ObjectStoreMgdConnException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "Unable to get the state value for the object \""
                _msg += obj_attr_list["name"] + "\" on this "
                _msg += "this experiment (Cloud "
                _msg += obj_attr_list["cloud_name"] + "): " + _fmsg
                cberr(_msg)
            else :
                _msg = "The " + _obj_type.upper() + " object " + obj_attr_list["name"]
                _msg += " had its state altered from \"" + _old_state + "\" to "
                _msg += "\"" + obj_attr_list["specified_state"] + "\""
                _msg +=" on this experiment (Cloud "
                _msg += obj_attr_list["cloud_name"] + ")\n"
                cbdebug(_msg)
            return self.package(_status, _msg, _result)

    @trace
    def wait_for(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        try :
            _status = 100
            _obj_type = command.split('-')[0].upper()

            obj_attr_list["cloud_name"] = "undefined"
            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)

            if not _status :
                _cloud_name = obj_attr_list["cloud_name"]
                _vg = ValueGeneration(self.pid)
                _time_to_wait = int(_vg.time2seconds(obj_attr_list["specified_time"]))   
                
                if obj_attr_list["interval"] == "default" or obj_attr_list["interval"] == "0" :
                    if _time_to_wait > 10 :
                        _update_interval = 10
                    else :
                        _update_interval = max(1,_time_to_wait/10)
                else :
                    _update_interval = _time_to_wait/int(obj_attr_list["interval"])

                _msg = "Going to unconditionally wait for "
                _msg += obj_attr_list["specified_time"] + " (" + str(_time_to_wait)
                _msg += " seconds). The command line interface will be blocked"
                _msg += " during the waiting."
                print(_msg)
                
                _start_time = int(time())
                _elapsed_time = 0
                while _elapsed_time < _time_to_wait :
                    _remaining_time = _time_to_wait - (int(time()) - _start_time)
                    if _remaining_time < _update_interval :
                        _update_interval = _remaining_time 
                    sleep(_update_interval)
                    _elapsed_time = int(time()) - _start_time
                    _msg = "Waited " + str(_elapsed_time) + " seconds... (" 
                    _msg += str(float(_elapsed_time)*100/float(_time_to_wait)) + "%)" 
                    print(_msg)

                _status = 0

        except ValueGeneration.ValueGenerationException as obj :
            _status = 28
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "Error while \"waiting for\": " + _fmsg
                cberr(_msg)
            else :
                _msg = "Waited for " + str(_time_to_wait) + " seconds."
                cbdebug(_msg)

                _x, _y, _stats = self.stats(obj_attr_list, _cloud_name + " all noprint false", "stats-get", True)
                    
            return self.package(_status, _msg, _stats)
        
    @trace
    def wait_until(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        try :
            _status = 100
            _start_time = int(time())
            _obj_type = "undefined"

            obj_attr_list["cloud_name"] = "undefined"

            _time_elapsed = 0
                
            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)

            _cloud_name = False
            _stats = {}
                        
            if not _status :

                _cloud_name = obj_attr_list["cloud_name"]                                 
                _obj_type = obj_attr_list["type"].upper()

                if len(obj_attr_list["counter"]) :
                    _counter_type = obj_attr_list["counter"]
                    _counter_name = obj_attr_list["counter"]
                else :
                    _counter_type = False
                    _counter_name = "(Objects created on the Object Store)"
                
                if obj_attr_list["direction"] == "increasing" :
                    _direction = "increasing"
                elif obj_attr_list["direction"] == "decreasing" :
                    _direction = "decreasing"
                else :
                    _msg = "Unknown direction for counter: " + str(obj_attr_list["direction"])
                    _status = 716
                    cberr(_msg)
                    raise self.ObjectOperationException(_msg, _status)            

                _counter_value = self.get_object_count(_cloud_name, _obj_type, _counter_type)                
                if _counter_value != "-1" :
                    True
                else :
                    _msg = "Warning: The specified counter (" + _counter_name + ')'
                    _msg += " does not exist. Will keep polling and checking anyway."
                    cbdebug(_msg)

                _time_limit = float(obj_attr_list["time_limit"])
                _check_interval = float(obj_attr_list["interval"])
                _start_time = float(time())

                if _counter_value :
                    _msg = "Going to wait until the value on counter \"" + _obj_type
                    _msg += ' ' + _counter_name + "\" is equal to " + str(obj_attr_list["value"])
                    _msg += " (currently it is equal to " + str(_counter_value) + ") "
                    _msg += "waiting " + str(_check_interval) + " seconds between "
                    _msg += "samples, for a maximum of " + str(_time_limit) + " seconds."
                    _msg += " The counter is assumed to be " + _direction + '.'
                    print(_msg)
                
                    while _time_elapsed < _time_limit :
                        sleep(_check_interval)
                        _current_time = int(time())

                        _time_elapsed = _current_time - _start_time
                        _counter_value = self.get_object_count(obj_attr_list["cloud_name"], _obj_type, _counter_type)

                        _msg = "Counter \"" + _obj_type + ' ' + _counter_name
                        _msg += "\" equals " + str(_counter_value) + " after "
                        _msg += str(_current_time - _start_time) + " seconds ("
                        _msg += "the counter is assumed to be " + _direction + ")."
                        cbdebug(_msg, True)

                        if _direction == "increasing" and int(_counter_value) >= int(obj_attr_list["value"]) :
                            break
                        elif _direction == "decreasing" and int(_counter_value) <= int(obj_attr_list["value"]) :
                            break

                    _status = 0

                else:
                    _fmsg = "Counter \"" + _obj_type + ' ' + _counter_name + "\""
                    _fmsg += " does not exist."
                    _status = 34

        except self.osci.ObjectStoreMgdConnException as obj :
            _status = obj.status
            _fmsg = str(obj.msg)

        except self.ObjectOperationException as obj :
            _status = obj.status
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            _total_time = int(time()) - _start_time

            if _status :
                _msg = "Error while \"waiting until\": " + _fmsg
                cberr(_msg)
            else :
                if _time_elapsed >= _time_limit :
                    _total_time = - _total_time
                    _msg = "After " + str(obj_attr_list["time_limit"]) + " seconds"
                    _msg += " the \"" + _obj_type + "s " + _counter_name
                    _msg += "\" was still not equal to " + str(obj_attr_list["value"]) + '.'
                    cbdebug(_msg)                    
                else :
                    _msg = "Waited " + str(_total_time) + " seconds"
                    _msg += " until \"" + _obj_type + "s " + _counter_name
                    _msg += "\" was equal to " + str(obj_attr_list["value"]) + '.'
                    cbdebug(_msg)

            if _cloud_name :                    
                _x, _y, _stats = self.stats(obj_attr_list, _cloud_name + " all noprint false", "stats-get", True)
    
                _stats["total_time"] = _total_time
                        
            return self.package(_status, _msg, _stats)

    @trace
    def wait_on(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        try :
            _status = 100
            _obj_type = "undefined"

            obj_attr_list["cloud_name"] = "undefined"
            obj_attr_list["channel"] = "undefined"
            _start = time()
            
            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)
            
            if not _status :

                _obj_type = obj_attr_list["type"].upper()                

                if obj_attr_list["keyword"] != "getsubscription" :

                    _sub_channel = self.osci.subscribe(obj_attr_list["cloud_name"], \
                                                       _obj_type, \
                                                       obj_attr_list["channel"], \
                                                       int(obj_attr_list["timeout"]))
    
                    _msg = "Subscribed to channel \"" + obj_attr_list["channel"] + "\""
                    _msg += " with a timeout of " + str(obj_attr_list["timeout"]) + " seconds "
                    _msg += "(object \"" + _obj_type + "\" listening for messages with"
                    _msg += " the keyword \"" + obj_attr_list["keyword"] + "\")"
                    print(_msg)
    
                    for _message in _sub_channel.listen() :
                        if _message["type"] == "message" :
                            _msg = "Message received (" + _message["data"] 
                            _msg += "). Proceeding to parse it"
                            cbdebug(_msg)
    
                            if _message["data"].count(obj_attr_list["keyword"]) :
                                _msg = "Message \"" + _message["data"] + "\" received"
                                _msg += " on channel \"" + obj_attr_list["channel"]
                                _msg += "\"."
                                cbdebug(_msg)
                                _sub_channel.unsubscribe()
                                break

                _status = 0
                
        except self.osci.ObjectStoreMgdConnException as obj :
            _status = obj.status
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            
            _end = time() - _start

            _result = {"subscription" : self.pid + ':' + obj_attr_list["cloud_name"] + ':' + _obj_type + ':' + obj_attr_list["channel"], \
                       "total_time" : _end, \
                       "host" : self.osci.host, \
                       "dbid" : self.osci.dbid }
                        
            if _status :
                _msg = "Error while \"waiting on channel\" after " + str(_end) + " seconds : " + _fmsg
                cberr(_msg)
            else :
                _msg = "Waited " + str(_end) + " seconds until a message containing the keyword \"" + obj_attr_list["keyword"]
                _msg += "\" was received on the channel \"" + obj_attr_list["channel"]
                _msg += "\" (" + _obj_type + ")."
                cbdebug(_msg)

            return self.package(_status, _msg, _result)

    @trace
    def msgpub(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        try :
            _status = 100
            _obj_type = "undefined"

            obj_attr_list["cloud_name"] = "undefined"
            obj_attr_list["channel"] = "undefined"

            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)
            
            if not _status :

                _obj_type = obj_attr_list["type"].upper()                
                self.osci.publish_message(obj_attr_list["cloud_name"], _obj_type, obj_attr_list["channel"], obj_attr_list["message"], 1, 3600)

                _status = 0

        except self.osci.ObjectStoreMgdConnException as obj :
            _status = obj.status
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "Error while publishing message: " + _fmsg
                cberr(_msg)
            else :
                _msg = "Message \"" + obj_attr_list["message"] + "\""
                _msg += " published on channel \"" + obj_attr_list["channel"] 
                _msg += "\" (object \"" + _obj_type + "\")."
                cbdebug(_msg)

            return self.package(_status, _msg, None)

    @trace
    def debug_startup(self, obj_attr_list, params, cmd) :
        '''
        TBD
        '''
        try :
            _status = 100
            
            _fmsg = "An error has occurred, but no error message was captured"
            _status, _fmsg = self.parse_cli(obj_attr_list, params, cmd)
            _smsg = ""
    
            if not _status :
                _status, _fmsg = self.initialize_object(obj_attr_list, cmd)
    
            if not _status :
                if "qemu_debug" in obj_attr_list and obj_attr_list["qemu_debug"].lower() == "true" :
                    port = obj_attr_list["qemu_debug_port"]
                else :
                    return (443, "Debugging not enabled for this object.", None)
                
                _cmd = "xterm -fg white -bg black -e gdb "
                _cmd += obj_attr_list["qemu_binary"] + " "
                _cmd += " -ex \"target remote " + obj_attr_list["vmc_name"] + ":" + port + "\""
                _cmd += " -ex \"handle SIGUSR2 noprint\""
                for breakpoint in obj_attr_list["breakpoints"] :
                    _cmd += " -ex \"b " + breakpoint + "\""
                _cmd += " -ex \"continue\""
                _proc_h = Popen(_cmd, shell=True, stdout=PIPE, stderr=PIPE)
        
                if _proc_h.pid :
                    _xmsg = "Console command \"" + _cmd + "\" "
                    _xmsg += " was successfully started. "
                    _smsg = _xmsg + " The process id is " + str(_proc_h.pid) + "."
                    
                    _cld_ops_class = self.get_cloud_class(obj_attr_list["model"])
                    _cld_conn = _cld_ops_class(self.pid, self.osci)
                    sleep(2)
                    unused, unused2 = _cld_conn.vm_fixpause(obj_attr_list)
                    _status = 0
                        
        except self.osci.ObjectStoreMgdConnException as obj :
            _status = 40
            _fmsg = str(obj.msg)
            
        finally :
            if _status :
                _msg ="Debug attachment failure: " + _fmsg 
                cberr(_msg)
            else :
                _msg = "Debug attachment success (" + _smsg + ")."
                cbdebug(_msg)

        return 0, _msg, None

    @trace
    def monitoring_extractall(self, parameters, command) :
        '''
        TBD
        '''
        try : 
            _status = 100
            _fmsg = "An error has occurred, but no error message was captured"
            _smsg = ''

            _expid = None
            
            _obj_attr_list = {}
            _status, _fmsg = self.parse_cli(_obj_attr_list, parameters, command)
            _msg = ""

            if not _status :
                _status, _fmsg = self.initialize_object(_obj_attr_list, command)

                if not _status :
                    _space_attr_list = self.osci.get_object(_obj_attr_list["cloud_name"], "GLOBAL", False, "space", False)
                    _time_attr_list = self.osci.get_object(_obj_attr_list["cloud_name"], "GLOBAL", False, "time", False)
        
                    if _obj_attr_list["expid"] == "current" :
                        _expid = _obj_attr_list["current_experiment_id"]
                    else :
                        _expid = _obj_attr_list["expid"]
        
                    _filestor_attr_list = self.osci.get_object(_obj_attr_list["cloud_name"], \
                                                               "GLOBAL", False, "filestore", \
                                                               False)

                    _logstor_attr_list = self.osci.get_object(_obj_attr_list["cloud_name"], \
                                                               "GLOBAL", False, "logstore", \
                                                               False)
        
                    _fshn = _filestor_attr_list["hostname"]
                    _fspn = _filestor_attr_list["port"]
                    _fsun = _filestor_attr_list["username"]
                    _url =  "rsync://" + _fshn + ':' + _fspn + '/' + _fsun + "_cb/data/" + _expid
                        
                    self.monitoring_extract(_obj_attr_list["cloud_name"] + " HOST os " + _expid, "mon-extract")
                    self.monitoring_extract(_obj_attr_list["cloud_name"] + " VM os "  + _expid, "mon-extract")
                    self.monitoring_extract(_obj_attr_list["cloud_name"] + " VM app " + _expid, "mon-extract")
                    self.monitoring_extract(_obj_attr_list["cloud_name"] + " VM management " + _expid, "mon-extract")
        
                    _destination = _space_attr_list["data_working_dir"] + '/' + _expid
                    
                    if _space_attr_list["tracefile"].lower() != "none" :
                        _msg = "This experiment was created with a trace file, which will"
                        _msg += " also be included alongside the extracted metrics."
                        cbdebug(_msg, True)
                        if _space_attr_list["tracefile"][0] != "/" :
                            _source = _space_attr_list["base_dir"] + '/' + _space_attr_list["tracefile"]
                            shutil.copy2(_source, _destination)

                    if str(_logstor_attr_list["just_restarted"]).lower() == "true" :
                        _msg = "This experiment was run right after a flushing of the Log Store."
                        _msg += " Will also include all logs files (from the Log Store) "
                        _msg += "alongside the extracted metrics."
                        cbdebug(_msg, True)
                        
                        _file_list = []
                        _file_list.append("operations.log")
                        _file_list.append("report.log")
                        _file_list.append("submmiter.log")
                        _file_list.append("loadmanager.log")
                        _file_list.append("gui.log")
                        _file_list.append("remotescripts.log")
                        _file_list.append("monitor.log")
                        _file_list.append("subscribe.log")
        
                        for _fn in  _file_list :
                            _source = _space_attr_list["log_dir"] + '/' + _space_attr_list["username"] + '_' + _fn
                            if access(_source, F_OK) :
                                shutil.copy2(_source, _destination)

                        self.osci.update_object_attribute(_obj_attr_list["cloud_name"], "GLOBAL", "logstore", False, "just_restarted", "false")

                    else :
                        _msg = "Bypassing the copy of all logs files"
                        cbdebug(_msg, True)

                    _status = 0
                    
        except self.ObjectOperationException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except MetricStoreMgdConnException as obj :
            _status = 40
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status and _status != 1111:
                _msg = "Monitor extraction failure: " + _fmsg
                cberr(_msg)
            else :
                _msg = "Monitor extraction success. All metrics written to csv"
                _msg += " files on the directory \""  + _space_attr_list["data_working_dir"] + "\".\n"
                _msg += "Data is available at the url \"" + _url + "\"." 
                cbdebug(_msg)

            return self.package(_status, _msg, _url)

    @trace
    def monitoring_purge(self, parameters, command) :
        '''
        TBD
        '''
        try : 
            _status = 100
            _fmsg = "An error has occurred, but no error message was captured"
            _smsg = ''

            _expid = None
            
            _obj_attr_list = {}
            _status, _fmsg = self.parse_cli(_obj_attr_list, parameters, command)
            _msg = ""

            if not _status :
                _status, _fmsg = self.initialize_object(_obj_attr_list, command)

                if not _status :
                    _space_attr_list = self.osci.get_object(_obj_attr_list["cloud_name"], "GLOBAL", False, "space", False)

                    _username = _space_attr_list["username"]

                    _expid = _obj_attr_list["expid"]

                    _msg = "Purging all data collected during experiment \""
                    _msg += _expid + "\" from the Metric Store..."
                    cbdebug(_msg, True)

                    if _expid.count('*') :
                        _expid = {'$regex':_expid}
                                            
                    self.get_msci(obj_attr_list["cloud_name"]).flush_metric_store(_username, True, {"expid" : _expid })

                    _status = 0
            
        except self.ObjectOperationException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except MetricStoreMgdConnException as obj :
            _status = 40
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status and _status != 1111:
                _msg = "Monitor data purge failure: " + _fmsg
                cberr(_msg)
            else :
                _msg = "Monitor purge success." 
                cbdebug(_msg)

            return self.package(_status, _msg, {})
        
    @trace
    def monitoring_extract(self, parameters, command) :
        '''
        TBD
        '''
        try : 
            _status = 100
            _fmsg = "An error has occurred, but no error message was captured"

            _obj_attr_list = {}
            _status, _fmsg = self.parse_cli(_obj_attr_list, parameters, command)
            _msg = ""

            _url = False
            
            if not _status :
                _status, _fmsg = self.initialize_object(_obj_attr_list, command)

                if not _status :

                    _obj_type = _obj_attr_list["type"].lower()
    
                    _metric_type = _obj_attr_list["metric_type"].lower()
                    _msci = self.get_msci(obj_attr_list["cloud_name"])

                    if _obj_attr_list["unchanged_string"] == "None" :
                        _obj_attr_list["unchanged_string"] = ''

                    if _metric_type == "m" or _metric_type == "mgt" or _metric_type == "man" or _metric_type == "management" :
                        _metric_type = "management"
                    elif _metric_type == "a" or _metric_type == "app" or _metric_type == "application" and _obj_type == "VM" :
                        _metric_type = "runtime_app"
                    elif _metric_type == "os" or _metric_type == "system" or _metric_type == "operatingsystem" :
                        _metric_type = "runtime_os"
                    else :
                        _msg = "Metrics of type \"" + _metric_type + "\" are not available for \"" + _obj_type + "\" objects." 
                        _status = 1028
                        raise self.ObjectOperationException(_msg, _status)

                    if _obj_attr_list["expid"] == "current" :
                        _criteria = { "expid" : _obj_attr_list["current_experiment_id"] }
                    else :
                        _criteria = { "expid" : _obj_attr_list["expid"] }

                    _space_attr_list = self.osci.get_object(_obj_attr_list["cloud_name"], "GLOBAL", False, "space", False)

                    _time_attr_list = self.osci.get_object(_obj_attr_list["cloud_name"], "GLOBAL", False, "time", False)
                    
                    _obj_attr_list["data_working_dir"] = _space_attr_list["data_working_dir"]                   
                    _obj_attr_list["data_file_location"] = _obj_attr_list["data_working_dir"] + '/' + _criteria["expid"]
                    _obj_attr_list["base_dir"] = _space_attr_list["base_dir"]
                    
                    _filestor_attr_list = self.osci.get_object(_obj_attr_list["cloud_name"], \
                                                               "GLOBAL", False, "filestore", \
                                                               False)

                    _fshn = _filestor_attr_list["hostname"]
                    _fspn = _filestor_attr_list["port"]
                    _fsun = _filestor_attr_list["username"]
                    _url =  "rsync://" + _fshn + ':' + _fspn + '/' + _fsun + "_cb/data/" + _criteria["expid"]

                    if not access(_obj_attr_list["data_file_location"], F_OK) :
                        makedirs(_obj_attr_list["data_file_location"])

                    _csv_contents_header = _obj_attr_list[_obj_type + '_' + _metric_type + "_metrics_header"]
                    
                    if _metric_type == "management" :
                        if "time_breakdown_keys" in _obj_attr_list :
                            _extra_csv_contents_header = _obj_attr_list["time_breakdown_keys"][0:-1].split(',')
                            _extra_csv_contents_header.sort()
                            _csv_contents_header += ',' + ','.join(_extra_csv_contents_header)
                            
                    _fn = _obj_attr_list["data_file_location"] + '/' 
                    _fn += _obj_type.upper() + '_' + _metric_type + '_'
                    _fn += _criteria["expid"] + ".csv"
                    _fd = open(_fn, 'w', 0)

                    _fd.write("#field:column #\n")
                    for _index, _item in enumerate(_csv_contents_header.split(',')) :
                        _fd.write('#' + _item + ':' + str(_index + 1) + '\n')
                    _fd.write("\n")

                    _msg = "Preparing to extract " + _metric_type + " metrics for all "
                    _msg += _obj_type.upper() + " objects with experiment id \"" 
                    _msg += str(_obj_attr_list["expid"]) + "\""
                    cbdebug(_msg)
    
                    _fd.write(_csv_contents_header + '\n')
                    
                    _msg = "Populating in-memory cache with an \"UUID to attribute\""
                    _msg += " cache for all " + _obj_type.upper() + " objects."
                    cbdebug(_msg)
                
                    _desired_keys = _csv_contents_header.split(',')

                    # We use the "trace" collection to determine the actual start
                    # of the experiment. It is *very* important to have VMs and 
                    # the main CloudbBench host synchronized through NTP
                    _collection_name = "trace_" + _obj_attr_list["username"]                         
                    _trace = _msci.find_document(_collection_name, \
                                                          _criteria, \
                                                          False, \
                                                          [("command_originated", 1)])

                    if "experiment_start_time" in _time_attr_list :
                        _experiment_start_time = _time_attr_list
                    else :
                        _experiment_start_time = int(_trace["command_originated"])

                    # Management metrics collection is always extracted, because it 
                    # contains all the information that maps UUIDs to all other attributes.
                    # Given that a user can change the "experiment id" after a given
                    # object was attached, we get all management metrics for the 
                    # UUID to "all other attributes" map, but only print the metrics pertaining
                    # to the current "experiment id". This needs to be optimized 
                    # later.
                    
                    _mgmt_collection_name = "management_" + _obj_type.upper() + '_' + _obj_attr_list["username"]

                    _management_metrics_list = _msci.find_document(_mgmt_collection_name, {'expid' : _criteria['expid']}, True)
                    
                    for _metric in _management_metrics_list :
                        _csv_contents_line = ''
                        for _key in _desired_keys  :
                            if _metric["expid"] == _criteria["expid"] :
                                if _key in _metric : 
                                    if _key == "mgt_001_provisioning_request_originated" :
                                        _csv_contents_line += str(int(_metric[_key]) - _experiment_start_time) + ','
                                    elif _key == "mgt_101_capture_request_originated" :
                                        _csv_contents_line += str(int(_metric[_key]) - _experiment_start_time) + ','                                        
                                    elif _key == "mgt_201_runstate_request_originated" :
                                        _csv_contents_line += str(int(_metric[_key]) - _experiment_start_time) + ','
                                    elif _key == "mgt_901_deprovisioning_request_originated" :
                                        _csv_contents_line += str(int(_metric[_key]) - _experiment_start_time) + ','                                           
                                    else :
                                        _csv_contents_line += str(_metric[_key]) + ','
                                else :
                                    if _key == "mgt_001_provisioning_request_originated_abs" :
                                        _key = "mgt_001_provisioning_request_originated"
                                        _csv_contents_line += str(int(_metric[_key])) + ','                                       
                                    else:
                                        _csv_contents_line += _obj_attr_list["filler_string"] + ','

                        if _metric_type == "management" :
                            _csv_contents_line = _csv_contents_line[:-1] + '\n'
                            _fd.write(_csv_contents_line)
    
                    _msg = "Done populating cache"
                    cbdebug(_msg)
    
                    if _metric_type == "management" :
                        _empty = False
                        _trace_csv_contents_header = _obj_attr_list["trace_header"]
        
                        _trace_desired_keys = _trace_csv_contents_header.split(',')
    
                        _trace_fn = _obj_attr_list["data_file_location"] + '/' 
                        _trace_fn += "trace_" 
                        _trace_fn += _criteria["expid"] + ".csv"
                        _trace_fd = open(_trace_fn, 'w', 0)
    
                        _trace_fd.write("#field:column #\n")
                        for _index, _item in enumerate(_trace_csv_contents_header.split(',')) :
                            _trace_fd.write('#' + _item + ':' + str(_index) + '\n')
                        _trace_fd.write("\n")
    
                        _trace_fd.write(_trace_csv_contents_header + '\n')

                        _collection_name = "trace_" + _obj_attr_list["username"]                         
                        _trace_list = _msci.find_document(_collection_name, \
                                                              _criteria, \
                                                              True, \
                                                              [("command_originated", 1)])

                        for _trace_item in _trace_list :
                            _trace_csv_contents_header = ''
                            for _key in _trace_desired_keys  :
                                if _key in _trace_item :
                                    _trace_csv_contents_header += str(_trace_item[_key]).replace(',',"replacewithcomma") + ','
                                else :
                                    _trace_csv_contents_header += _obj_attr_list["filler_string"] + ','
    
                            _trace_csv_contents_header = _trace_csv_contents_header[:-1] + '\n'
                            _trace_fd.write(_trace_csv_contents_header)
    
                        _trace_fd.close()
                    
                    if _metric_type == "runtime_os" or _metric_type == "runtime_app" :

                        _last_unchanged_metric = {}

                        _collection_name = _metric_type + '_' + _obj_type.upper() + '_' + _obj_attr_list["username"]
                        filters = []
                        if _metric_type != "runtime_os" :
                            filters.append(("time", 1))
                        _runtime_metric_list = _msci.find_document(_collection_name, _criteria, True, filters)

                        _empty = True

                        for _metric in _runtime_metric_list :
                            _empty = False
                            _current_uuid = _metric["uuid"]
                            _csv_contents_line = ''
    
                            if not _current_uuid in _last_unchanged_metric :
                                # This is another dictionary, used to 
                                # "carry over" runtime metrics that are not reported
                                # frequently, because they seldom change. For
                                # instance "cpu_freq" is reported with a very low
                                # frequency. To save memory and storage, the gmetad
                                # DOES NOT write "old" values to the Metric Store.
                                # However, we do need these values for the building
                                # of the csv, and thus we just "borrow" the previous
                                # value when a metric is not found on the Metric Store.
                                _last_unchanged_metric[_current_uuid] = {}
    
                            _tmp_metric = _msci.find_document(_mgmt_collection_name, {"_id" : _metric["uuid"]})
                            for _key in _desired_keys :
                                if _key in _metric and _key != "uuid" and _key != "time" and _key != "time_h" and _key != "time_cbtool" and _key != "time_cbtool_h" :
                                    _val = str(_metric[_key]["val"])
                                    # Every time we find a metric, we add it to an
                                    # in-memory cache (dictionary)
                                    _last_unchanged_metric[_current_uuid][_key] = str(_val)
    
                                elif _key == "uuid" or _key == "time_h" or _key == "time_cbtool_h" :
                                    _val = str(_metric[_key])
    
                                elif _key == "time" or _key == "time_cbtool" :
                                    _val = str(int(_metric[_key]) - _experiment_start_time)
    
                                else :
                                    if _tmp_metric and _key in _tmp_metric :
                                        _val = str(_tmp_metric[_key])
                                    else :
                                        if _key in _last_unchanged_metric[_current_uuid] :
                                            # Every time a metric is not found, we just
                                            # replay it from the in-memory cache (dictionary)
                                            _val = str(_last_unchanged_metric[_current_uuid][_key]) + ' ' + _obj_attr_list["unchanged_string"]
                                        else :
                                            _val = _obj_attr_list["filler_string"]
                                _csv_contents_line += _val + ','
                                
                            _fd.write(_csv_contents_line[:-1] + '\n')

                    _fd.close()

                    _dtb1 = ''
                    _dtb2 = ''
                    if "time_breakdown_keys" in _obj_attr_list :
                        _dtbc = _obj_attr_list["time_breakdown_keys"][0:-1].split(',')
                        _dtbc.sort()
                        _dtb1 = " --breakdown " + ','.join(_dtbc)        
                        _dtb2 = " --breakdown " + ','.join(reversed(_dtbc))

                    _plot_fn = _obj_attr_list["data_file_location"] + '/' 
                    _plot_fn += "plot.sh" 
                    _plot_fd = open(_plot_fn, 'w', 0)
                                
                    _plot_fd.write("#!/bin/bash\n")
                    _plot_fd.write("if [ $0 != \"-bash\" ] ; then\n")
                    _plot_fd.write("    pushd `dirname \"$0\"` 2>&1 > /dev/null\n")
                    _plot_fd.write("fi\n")
                    _plot_fd.write("CB_BASE_DIR=$(pwd)\n")
                    _plot_fd.write("if [ $0 != \"-bash\" ] ; then\n")
                    _plot_fd.write("    popd 2>&1 > /dev/null\n")
                    _plot_fd.write("fi\n")
                    _plot_fd.write("CB_EXPID=$(echo $CB_BASE_DIR | rev | cut -d '/' -f 1 | rev)\n")
                    _plot_fd.write("CB_BASE_DIR=$(readlink -f $CB_BASE_DIR/../../)\n")
                    _plot_fd.write("$CB_BASE_DIR/util/plot/cbplotgen.R --directory $CB_BASE_DIR/data  --expid " + _criteria["expid"] + " --cleanup --provisionmetrics --runtimemetrics " + _dtb1 + '\n')
                    _plot_fd.write("#$CB_BASE_DIR/util/plot/cbplotgen.R --directory $CB_BASE_DIR/data  --expid " + _criteria["expid"] + " --cleanup --provisionmetrics --runtimemetrics " + _dtb2 + '\n')                    
                    _plot_fd.close()
                    os.chmod(_plot_fn, 0o755)

                    if _empty :
                        _msg = "No samples of " + _metric_type + " metrics for "
                        _msg += "all " + _obj_type.upper() + "s were found (the"
                        _msg += " file " + _fn + " will be empty)."
                        cbdebug(_msg, True)
                    else :
                        _msg = "Multiple samples of " + _metric_type + " metrics for "
                        _msg += "all " + _obj_type.upper() + "s were found (the"
                        _msg += " file " + _fn + " will *NOT* be empty)."
                        cbdebug(_msg, True)

                    _msg = _metric_type + " metrics for all " + _obj_type.upper()
                    _msg += " objects were successfully extracted. File name is "
                    _msg += _fn + ' .'
                    cbdebug(_msg)

                    _status = 0

        except self.ObjectOperationException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except MetricStoreMgdConnException as obj :
            _status = 40
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status and _status != 1111:
                _msg = "Monitor extraction failure: " + _fmsg
                cberr(_msg)
            else :
                _msg = "Monitor extraction success. " + _obj_type.upper()
                _msg += ' ' + _metric_type.replace("_app", " application").replace("_os", " OS")
                _msg += " performance data samples were written to the file " + _fn + ".\n"
                if _url :
                    _msg += "Data is available at the url \"" + _url + "\"." 
                cbdebug(_msg)

            return self.package(_status, _msg, _url)

    @trace
    def monitoring_list(self, parameters, command) :
        '''
        TBD
        '''
        try : 
            _status = 100
            _fmsg = "An error has occurred, but no error message was captured"
            _result = {"management" : [], "runtime" : []} 
            _obj_attr_list = {}
            _status, _fmsg = self.parse_cli(_obj_attr_list, parameters, command)
            _msg = ""

            if not _status :
                _status, _fmsg = self.initialize_object(_obj_attr_list, command)

                if not _status :
                    _curr_time = int(time())
                    _obj_type = _obj_attr_list["type"]

                    _cloud_name = _obj_attr_list["cloud_name"]
                    _msci = self.get_msci(_cloud_name)

                    _msg = "The following " + _obj_type + "s reported management metrics:\n"
                    _field1 = "Name                        "
                    _field2 = "Age (seconds)     "
                    _field3 = "Experiment id                  "
                    _field4 = "Number of samples"
                    _msg += _field1 + '|' + _field2 + '|' + _field3 + '|' + _field4 + '\n'

                    #_uuid_to_name_dict = {}
                    
                    _coll_name = "latest_management_" + _obj_type + '_' + _obj_attr_list["username"]
                    _metrics_list = _msci.find_document(_coll_name, {}, True)

                    for _metric in _metrics_list :
                        _msg += _metric["name"].ljust(len(_field1))
                        _msg += '|' + str( _curr_time- int(_metric["mgt_001_provisioning_request_originated"])).ljust(len(_field2))
                        _msg += '|' + _metric["expid"].ljust(len(_field3))
                        _nr_samples = _msci.count_document(_coll_name.replace("latest_",''), {"expid": _metric["expid"], "uuid": _metric["uuid"]})
                        _msg += '|' + str(_nr_samples).ljust(len(_field4)) + '\n'

                        #_uuid_to_name_dict[_metric["_id"]] = _metric["name"]

                    _msg += "\nThe following " + _obj_type  + "s reported runtime (OS) metrics:\n"
                    _msg += _field1 + '|' + _field2 + '|' + _field3 + '|' + _field4 + '\n'

                    _coll_name = "latest_runtime_os_" + _obj_type + '_' + _obj_attr_list["username"]
                    _metrics_list = _msci.find_document(_coll_name, {}, True)

                    for _metric in _metrics_list :
                        if self.osci.object_exists(_cloud_name, _obj_type, _metric["_id"], False) :
                            _obj_attr_list = self.osci.get_object(_cloud_name, _obj_type, False, _metric["_id"], False)
                            _result["runtime"].append([_obj_attr_list["name"], _metric["time"]])
                            _msg += _obj_attr_list["name"].ljust(len(_field1))
                            _msg += '|' +  str( _curr_time- int(_metric["time"])).ljust(len(_field2))
                            _msg += '|' + _metric["expid"].ljust(len(_field3))
                            _nr_samples = _msci.count_document(_coll_name.replace("latest_",''), {"expid": _metric["expid"], "uuid": _metric["uuid"]})
                            _msg += '|' + str(_nr_samples).ljust(len(_field4)) + '\n'

                    if _obj_type == "VM" :
                        _msg += "\nThe following " + _obj_type  + "s reported runtime (Application) metrics:\n"
                        _msg += _field1 + '|' + _field2 + '|' + _field3 + '|' + _field4 + '\n'

                        _coll_name = "latest_runtime_app_" + _obj_type + '_' + _obj_attr_list["username"]
                        _metrics_list = _msci.find_document(_coll_name, {}, True)

                        for _metric in _metrics_list :
                            if self.osci.object_exists(_cloud_name, _obj_type, _metric["_id"], False) :
                                _obj_attr_list = self.osci.get_object(_cloud_name, _obj_type, False, _metric["_id"], False)
                                _result["management"].append([_obj_attr_list["name"], _metric["time"]])
                                _obj_attr_list = self.osci.get_object(_obj_attr_list["cloud_name"], _obj_type, False, _metric["_id"], False)
                                _msg += _obj_attr_list["name"].ljust(len(_field1))
                                _msg += '|' +  str( _curr_time- int(_metric["time"])).ljust(len(_field2))                                              
                                _msg += '|' + _metric["expid"].ljust(len(_field3))
                                _nr_samples = _msci.count_document(_coll_name.replace("latest_",''), {"expid": _metric["expid"], "uuid": _metric["uuid"]})
                                _msg += '|' + str(_nr_samples).ljust(len(_field4)) + '\n'

                    _status = 0

        except self.ObjectOperationException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except MetricStoreMgdConnException as obj :
            _status = 40
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status and _status != 1111:
                _msg = "Monitor extraction failure: " + _fmsg
                cberr(_msg)
            else :
                cbdebug(_msg)
            return self.package(_status, _msg, _result)
            
    @trace
    def run_api_service(self, passive, active, debug, port, hostnames) :
        '''
        TBD
        '''
        _status = 100
        _fmsg = "An error has occurred, but no error message was captured"
        apiservices = [] 
        
        try :
            for hostname in hostnames  :
                self.wait_for_port_ready(hostname, port)
                from lib.api.api_service import APIService, append_service, remove_service
                apiservice = APIService(self.pid, \
                                        passive, \
                                        active, \
                                        debug, \
                                        port, \
                                        hostname)
                apiservices.append(apiservice)
                apiservice.daemon = True
                
            for apiservice in apiservices :
                if debug and not len(apiservices) > 1:
                    apiservice.run()
                    # only debug the first hostname
                    break
                else :
                    append_service(hostname, apiservice)
                        
            if not debug or len(apiservices) > 1 :
                while True :
                    sleep(10)
                    
                for apiservice in apiservices :
                    remove_service(hostname)
                
            _status = 0

        except Exception as e :
            for line in traceback.format_exc().splitlines() :
                cbwarn(line, True)
            _status = 23
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "API Service startup failure: " + _fmsg
                cberr(_msg)
                raise self.ObjectOperationException(_msg, _status)
            else :
                _msg = "API Service startup success."
                cbdebug(_msg)
                return _status, _msg

    @trace
    def cloud_watch(self, cloud_name, uuid) :
        '''
        TBD
        '''
        try :
            _status = 100
            _fmsg = "An error has occurred, but no error message was captured"

            _cloud_parameters = self.get_cloud_parameters(cloud_name)

            _cld_ops_class = self.get_cloud_class(_cloud_parameters["model"])
                
            _cld_conn = _cld_ops_class(self.pid, self.osci)

            for _vmc_uuid in self.osci.get_object_list(cloud_name, "VMC") : 
                _obj_attr_list = self.osci.get_object(cloud_name, "VMC", False, _vmc_uuid, False)
                _obj_attr_list["vmc_name"] = _obj_attr_list["name"]
                _vm_list = _cld_conn.get_vm_instances(_obj_attr_list) 
                if _vm_list :
                    for _vm in _vm_list :
                        print(_vm.name, _vm.id, _vm.status)
                        sleep(float(_obj_attr_list["update_frequency"]))
            _status = 0

        except self.osci.ObjectStoreMgdConnException as obj :
            _status = 40
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "Cloud Watch Service startup failure: " + _fmsg
                cberr(_msg)
                raise self.ObjectOperationException(_msg, _status)
            else :
                _msg = "Cloud Watch Service startup success."
                cbdebug(_msg)
                return _status, _msg

    @trace
    def performance_metrics_emitter(self, cloud_name, object_uuid) :
        '''
        TBD
        '''
        _status = 100
        _fmsg = "An error has occurred, but no error message was captured"

        _ai_state = True
        _error = False
        _prev_load_level = 0
        _prev_load_duration = 0
        _prev_load_id =  0
        
        _initial_ai_attr_list = self.osci.get_object(cloud_name, "AI", False, object_uuid, False)

        _mode = _initial_ai_attr_list["mode"]
        
        _check_frequency = float(_initial_ai_attr_list["update_frequency"])
                        
        while _ai_state and not _error :        

            try :

                if _mode == "controllable" :
                    _ai_state = self.osci.get_object_state(cloud_name, "AI", object_uuid)
                    _ai_attr_list = self.osci.get_object(cloud_name, "AI", False, object_uuid, False)
                    _mode = _ai_attr_list["mode"]
                    _check_frequency = float(_ai_attr_list["update_frequency"])
                else :
                    _ai_state = "attached"
                    _ai_attr_list = _initial_ai_attr_list
    
                _username = _ai_attr_list["username"]

                if _ai_state and _ai_state == "attached" :
                    
                    _load = self.get_load(cloud_name, _ai_attr_list, False, \
                                          _prev_load_level, _prev_load_duration, \
                                          _prev_load_id)

                    if _load :
                        _prev_load_level = _ai_attr_list["current_load_level"]
                        _prev_load_duration = _ai_attr_list["current_load_duration"]
                        _prev_load_id = _ai_attr_list["current_load_id"]

                    if _mode == "controllable" :

                        self.update_object_attribute(cloud_name, \
                                                     "AI", \
                                                     object_uuid, \
                                                     "current_load_level", \
                                                     _ai_attr_list["current_load_level"]) 
                            
                        self.update_object_attribute(cloud_name, \
                                                     "AI", \
                                                     object_uuid, \
                                                     "current_load_duration", \
                                                     _ai_attr_list["current_load_duration"])
        
                        self.update_object_attribute(cloud_name, \
                                                     "AI", \
                                                     object_uuid, \
                                                     "current_load_id", \
                                                     _ai_attr_list["current_load_id"])


                    _ai_attr_list["current_load_profile"] = _ai_attr_list["load_profile"]

                    if "reported_metrics" in _ai_attr_list :

                        _vg = ValueGeneration(self.pid)

                        _metrics_list = ''
                        _sla_target_list = ''
                        
                        for _item in [ "id-seqnum", "level-load", "profile-name", "duration-sec"] :
                            _metric, _unit = _item.split('-')
                            _metrics_list += "load_" + _metric + ':' + str(_ai_attr_list["current_load_" + _metric]) + ':' + _unit + ' '
                        
                        for _metric in _ai_attr_list["reported_metrics"].split(',') :

                            if _metric + "_value" in _ai_attr_list :
                                _val = _vg.get_value(_ai_attr_list[_metric + "_value"])
                            else :
                                _val = _vg.get_value("uniformIXIXI1I1000")

                            if _metric.count("latency") :
                                _unit = "us"
                            elif _metric.count("bandwidth") :
                                _unit = "MBps"
                            elif _metric.count("time") :
                                _unit = "seconds"
                            elif _metric.count("size") :
                                _unit = "size"
                            elif _metric.count("throughput") :
                                _unit = "tps"
                            elif _metric.count("errors") :
                                _unit = "num"                               
                            else :
                                _unit = "NA"

                            _metrics_list += _metric + ':' + str(_val) + ':' + _unit + ' '

                            if "sla_runtime_target_" + _metric in _ai_attr_list :
                                _sla_target_list += "sla_runtime_target_" + _metric + ':' + _ai_attr_list["sla_runtime_target_" + _metric] + ' '

                    _msg = "Preparing to execute AI load generation"
                    cbdebug(_msg)

                    _msg = "Sleeping for " + str(_ai_attr_list["current_load_duration"]) + " seconds"
                    cbdebug(_msg)
                    
                    sleep(int(_ai_attr_list["current_load_duration"]))

                    _vm_uuid = _ai_attr_list["load_generator_vm"]
                    _expid = _ai_attr_list["experiment_id"]
                    
                    report_app_metrics(_metrics_list, \
                                       _sla_target_list, \
                                       [_msci, _vm_uuid, _expid,  _username], \
                                       [self.osci, _vm_uuid, cloud_name], \
                                       False)
                    
                else :
                    # Only reset individual applications on the AI. Don't send
                    # any load.
                    _msg = "AI object " + object_uuid 
                    _msg += " state was set to \"" + _ai_state + "\". No load will "
                    _msg += "be applied until the state is changed. The "
                    _msg += "current load will be allowed to finish its course."
                    cbdebug(_msg)

            except MetricStoreMgdConnException as obj :
                _error = True
                _status = 44
                _fmsg = str(obj.msg)

            except self.osci.ObjectStoreMgdConnException as obj :
                _error = True                
                _status = 40
                _fmsg = str(obj.msg)
    
            except Exception as e :
                _error = True                
                _status = 23
                _fmsg = str(e)

        if _status :
            _msg = "Application Performance Metrics emitter failure: " + _fmsg
            cberr(_msg)
            return _status, _msg
        else :
            _msg = "Application Performance Metrics emitter completed successfully"
            cbdebug(_msg)
            return _status, _msg

    @trace
    def execute_shell(self, parameters, command) :
        '''
        TBD
        '''
        result_dict = {}
        try : 
            _status = 100
            _fmsg = "An error has occurred, but no error message was captured"
            _obj_attr_list = {}
            _status, _fmsg = self.parse_cli(_obj_attr_list, parameters, command)
            _msg = ""

            _vm = None
            _actual_cmd = ''
            
            if not _status :

                _status, _fmsg = self.initialize_object(_obj_attr_list, command)

                if not _status :
                    _cmd = _obj_attr_list["cmdexec"]

                    for _word in _cmd.split() :

                        if _word.count("vm_") :
                            _vm = _word
                            
                            _vm_attr_list = \
                            self.osci.get_object(_obj_attr_list["cloud_name"], \
                                                 "VM", \
                                                 True, \
                                                 _vm, \
                                                 False)
                        else :
                            _actual_cmd += _word + ' ' 

                    if not _vm :
                        _proc_man =  ProcessManagement()                                                 

                    else :
                        _proc_man =  ProcessManagement(hostname = _vm_attr_list["prov_cloud_ip"], \
                                                       port = _vm_attr_list["prov_cloud_port"], \
                                                       username = _vm_attr_list["login"], \
                                                       cloud_name = _vm_attr_list["cloud_name"], \
                                                       priv_key = _vm_attr_list["identity"])
                        
#                        _ssh_cmd = "ssh -i " + _vm_attr_list["identity"]

#                        if "ssh_config_file" in _vm_attr_list :
#                            _ssh_cmd += " -F " + _vm_attr_list["ssh_config_file"]

#                        _ssh_cmd += " -o StrictHostKeyChecking=no"
#                        _ssh_cmd += " -o UserKnownHostsFile=/dev/null" 
#                        _ssh_cmd += " -l " + _vm_attr_list["login"] + ' '

#                        _cmd = _ssh_cmd + ' ' + _vm_attr_list["prov_cloud_ip"] + " \"" + _cmd.replace(_vm,'') + "\""

#                    _proc_man =  ProcessManagement()                                                 
                    print("running shell command: \"" + _actual_cmd + "\"....")
                    _status, _result_stdout, _result_stderr = _proc_man.run_os_command(_actual_cmd)
                    result_dict = {"stdout": _result_stdout, "stderr": _result_stderr}
 
                    if not _status :
                        print("stdout:\n " + _result_stdout)

                        if len(_result_stderr) :
                            print("stderr:\n " + _result_stderr)
                    else :
                        _fmsg = _result_stderr

        except self.ObjectOperationException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except MetricStoreMgdConnException as obj :
            _status = 40
            _fmsg = str(obj.msg)

        except ProcessManagement.ProcessManagementException as obj :
            _status = str(obj.status)
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "Shell command execution failure: " + _fmsg
                cberr(_msg)
            else :
                cbdebug(_msg)
            return self.package(_status, _msg, result_dict)

    @trace
    def globallist(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        try :
            _status = 100
            _fmsg = "An error has occurred, but no error message was captured"

            _result = []

            _msg_string = {}
            _msg_string["vm_roles"] = " can be attached to "
            _msg_string["ai_types"] = " can be attached to "
            _msg_string["aidrs_patterns"] = " can be attached to "
            _msg_string["vmc_pools"] = " are attached to "
            _msg_string["view_criteria"] = " can be used on"
            _msg_string["fi_situations"] = " can be executed on "

            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)

            if not _status :
                _status, _fmsg = self.initialize_object(obj_attr_list, command)

                if not _status :
                    # IMPORTANT: Rememeber that the list is created at cloud
                    # attachment time by the "initialize_object_store" method
                    # in the datastore adapter
                    _list_name = obj_attr_list["object_type"][:-1].lower() + '_' + obj_attr_list["object_attribute"]
                    _result = list(self.osci.get_list(obj_attr_list["cloud_name"], "GLOBAL", _list_name))

                    if _list_name == "ai_types" :
                        _tmp_dict = {}
                        for _ai_item in _result :
                            if _ai_item.count('/') :
                                _category, _ai_type = _ai_item.split('/')
                                if _category not in _tmp_dict :
                                    _tmp_dict[_category] = []
                                _tmp_dict[_category].append(_ai_type)
                            
                        _list = ''
                        for _category in list(_tmp_dict.keys()) :
                            _list += '\n' + _category + ":\n"
                            for _ai_type in sorted(_tmp_dict[_category]) :
                                _list += "  " + _ai_type + '\n'
                    else :
                        _list = "\n".join(sorted(list(_result)))
                        
                    _vmc_list = self.osci.get_object_list(obj_attr_list["cloud_name"], "VMC")

                    if not _vmc_list :
                        _vmc_list = []

                    if _list_name == "view_criteria" :
                        _view_dict = {
                                      "vmc" : { 
                                               "pool": [ 0, { "label" : "VMC Pool", "criterion" : "pool", "expression" : sorted(list(self.osci.get_list(obj_attr_list["cloud_name"], "GLOBAL", "vmc_pools"))) } ],
                                               "arrival": [ 1, { "label" : "Arrival Time", "criterion" : "username", "expression" : self.username } ],
                                               },

                                      "host" : { 
                                               "vmc": [ 0, { "label" : "VMC", "criterion" : "vmc", "expression" : sorted(list(_vmc_list)) } ],
                                               "arrival": [ 1, { "label" : "Arrival Time", "criterion" : "username", "expression" : self.username} ],
                                               },
                                      
                                      "vm" : { 
                                              "role": [ 0, { "label" : "Role", "criterion" : "role", "expression" : sorted(list(self.osci.get_list(obj_attr_list["cloud_name"], "GLOBAL", "vm_roles")))} ],
                                              "type": [ 1, { "label" : "Type", "criterion" : "type", "expression" : sorted(list(self.osci.get_list(obj_attr_list["cloud_name"], "GLOBAL", "ai_types")))} ],
                                              "arrival": [ 2, { "label" : "Arrival", "criterion" : "arrival", "expression" : self.username} ],
                                              },
        
                                      "app" : {
                                               "type": [ 0, { "label" : "Type", "criterion" : "type", "expression" : sorted(list(self.osci.get_list(obj_attr_list["cloud_name"], "GLOBAL", "ai_types")))} ],
                                               "arrival": [ 1, { "label" : "Arrival", "criterion" : "arrival", "expression" : "all"} ],
                                               },
                                      "aidrs" : {
                                                 "pattern": [ 0, { "label" : "Pattern", "criterion" : "pattern", "expression" : sorted(list(self.osci.get_list(obj_attr_list["cloud_name"], "GLOBAL", "aidrs_patterns")))} ],
                                                 "arrival": [ 1, { "label" : "Arrival", "criterion" : "arrival", "expression" : "all"} ],
                                               },
                                      
                                      }
                    else :
                        _view_dict = _result
                                
                    _status = 0

        except self.osci.ObjectStoreMgdConnException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "Unable to get GLOBAL object list: " + _fmsg
                cberr(_msg)
                _view_dict = False

            else :
                _msg = obj_attr_list["object_type"] + " with the following "
                _msg += obj_attr_list["object_attribute"] + _msg_string[_list_name]
                _msg += "to this experiment (Cloud "
                _msg += obj_attr_list["cloud_name"]  + ") :\n" + _list
                cbdebug(_msg)

                if isinstance(_view_dict, set) :
                    _view_dict = sorted(list(_view_dict))

                if isinstance(_view_dict, list) :
                    _view_dict = sorted(_view_dict)
                    
            return self.package(_status, _msg, _view_dict)

    @trace
    def globalshow(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        try :
            _status = 100
            _fmsg = "An error has occurred, but no error message was captured"

            _result = []
            
            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)

            if not _status :
                _status, _fmsg = self.initialize_object(obj_attr_list, command)
 
                if not _status :
                    _list_name = obj_attr_list["object_type"].lower() + '_' + obj_attr_list["object_attribute"] + 's'

                    if obj_attr_list["attribute_name"] in list(self.osci.get_list(obj_attr_list["cloud_name"], "GLOBAL", _list_name)) :

                        _formatted_result = []
                        _object_contents = self.osci.get_object(obj_attr_list["cloud_name"], "GLOBAL", False, obj_attr_list["global_object"], False)

                        if obj_attr_list["global_object"] == "vm_templates" :
                            if obj_attr_list["attribute_name"] in _object_contents :
                                _result = str2dic(_object_contents[obj_attr_list["attribute_name"]])
                                for _key,_value in list(_result.items()) :
                                    _formatted_result.append(_key + ": " + _value)
    
                        elif obj_attr_list["global_object"] == "ai_templates" or \
                        obj_attr_list["global_object"] == "aidrs_templates" or \
                        obj_attr_list["global_object"] == "fi_templates" :

                            _result = {}
                            _profiles = ""
                            for _key, _value in list(_object_contents.items()) :

                                if _key.count(obj_attr_list["attribute_name"]) :

                                    if not (_key.count("_pref_host") or \
                                            _key.count("_pref_pool") or \
                                            _key.count("_meta_tag") or \
                                            _key.count("_size") or \
                                            _key.count("_netname")) :
                                        _key = _key.replace(obj_attr_list["attribute_name"] + '_', '', 1)
                                    else :
                                        _key = _key.replace(obj_attr_list["attribute_name"] + '_', '', 1)
                                    _result[_key] = _value
                                    
                                    # A trick to display the AI definition
                                    # in a specific order. It is ugly and not
                                    # efficient, but it will do for now.
                                    _load_balancer_supported = False
                                    
                                    if _key == "description" :
                                        _key = _key.replace(_key, "00___" + _key)
                                        _ww_value = ''
                                        for _line in _value.split('\\n') :
                                            _line = textwrap.fill(_line, 80) + "\n"
                                            _ww_value += _line
                                        _value = _ww_value
                                        
                                    elif _key == "sut" :
                                        _key = _key.replace(_key, "01___" + _key)
                                    elif _key == "load_balancer_supported" :
                                        _key = _key.replace(_key, "02___" + _key)
                                        if _value.lower() == "true" :
                                            _load_balancer_supported = True
                                    elif _key == "resize_supported" :
                                        _key = _key.replace(_key, "03___" + _key)
                                    elif _key == "regenerate_data" :
                                        _key = _key.replace(_key, "04___" + _key)
                                    elif _key == "role_list" :
                                        _key = _key.replace(_key, "05___" + _key)                                        
                                    elif _key == "load_generator_role" :
                                        _key = _key.replace(_key, "06___" + _key)
                                    elif _key == "load_manager_role" :
                                        _key = _key.replace(_key, "07___" + _key)         
                                    elif _key == "metric_aggregator_role" :
                                        _key = _key.replace(_key, "08___" + _key)
                                    elif _key == "capture_role" :
                                        _key = _key.replace(_key, "09___" + _key)
                                    elif _key == "load_balancer" :
                                        _key = _key.replace(_key, "10___" + _key)                                                                                                                        
                                    elif _key == "load_profile" :
                                        _key = _key.replace(_key, "11___" + _key)
                                    elif _key == "load_level" :
                                        _key = _key.replace(_key, "12___" + _key)
                                    elif _key == "load_duration" :
                                        _key = _key.replace(_key, "13___" + _key)                                                                                                                        
                                    elif _key == "reported_metrics" :
                                        _key = _key.replace(_key, "14___" + _key)           
                                    elif _key == "category" :
                                        _key = _key.replace(_key, "15___" + _key) 
                                    elif _key == "profiles" :
                                        _key = _key.replace(_key, "16___" + _key)           
                                        _profiles = _value
                                    elif _key == "reference" :
                                        _key = _key.replace(_key, "17___" + _key)
                                    elif _key == "license" :
                                        _key = _key.replace(_key, "18___" + _key)                                                                                
                                        
                                    elif _key.count("setup") :
                                        _key = _key.replace(_key, "19___" + _key)
                                    elif _key.count("reset") :
                                        _key = _key.replace(_key, "20___" + _key)
                                    elif _key.count("resize") :
                                        _key = _key.replace(_key, "21___" + _key)                                        
                                    elif _key.count("start") :
                                        _key = _key.replace(_key, "22___" + _key)

                                    elif _key == "type" :
                                        _key = _key.replace(_key, "00___" + _key)
                                    elif _key == "max_ais" :
                                        _key = _key.replace(_key, "01___" + _key)
                                    elif _key == "iait" :
                                        _key = _key.replace(_key, "02___" + _key)
                                    elif _key == "lifetime" :
                                        _key = _key.replace(_key, "09___" + _key)

                                    _formatted_result.append(_key + ": " + _value)

                        _formatted_result.sort()

                        if obj_attr_list["global_object"] == "ai_templates" or obj_attr_list["global_object"] == "aidrs_templates" :
                            _sh = 'Z'
                            for _line_number in range(0, len(_formatted_result)) :

                                if _formatted_result[_line_number].count("_PROFILES_") :
                                    _formatted_result[_line_number] = _formatted_result[_line_number].replace("_PROFILES_",_profiles)
                                    
                                if _formatted_result[_line_number].count("___") :
                                    if _formatted_result[_line_number].count("sut") :
                                        _formatted_result[_line_number] = "# Attributes MANDATORY for all Virtual Applications: \n\n" + _formatted_result[_line_number][5:]
                                    elif _formatted_result[_line_number].count("reported_metrics") :
                                        _formatted_result[_line_number] = _formatted_result[_line_number][5:]
                                        _formatted_result[_line_number] += "\n\n# Virtual Application-specific MANDATORY attributes: \n"
                                    else :
                                        _formatted_result[_line_number] = _formatted_result[_line_number][5:]
                                        if _formatted_result[_line_number].count("description:") :
                                            _formatted_result[_line_number] = '\n' + _formatted_result[_line_number]                                        
                                        if _formatted_result[_line_number].count("license") :
                                            _formatted_result[_line_number] = _formatted_result[_line_number].replace('_',' ') + '\n'

                                else :
                                    if len(_sh) == 1 :
                                        _sh = "\n# Virtual Application-specific OPTIONAL attributes: \n\n"
                                        
                                    _formatted_result[_line_number] = _sh + _formatted_result[_line_number].replace('\\n','\n')
                                    _sh = ''
                                    
                        _status = 0
                    else :
                        _status = 179
                        _fmsg = "Unknown " + obj_attr_list["object_type"] + ' '
                        _fmsg += obj_attr_list["object_attribute"] 
                        _fmsg += " (" + obj_attr_list["attribute_name"] + ")"
                        
        except self.osci.ObjectStoreMgdConnException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "Unable to get object: " + _fmsg
                cberr(_msg)
                _view_dict = False

            else :
                _msg = "The " + obj_attr_list["object_type"] + " with the "
                _msg += obj_attr_list["object_attribute"] + ' ' + obj_attr_list["attribute_name"]
                _msg += " has the following configuration on experiment (Cloud "
                _msg += obj_attr_list["cloud_name"]  + ") :\n" + '\n'.join(_formatted_result)
                cbdebug(_msg)

            return self.package(_status, _msg, _result)

    @trace
    def globalalter(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        try :
            _status = 100
            _fmsg = "An error has occurred, but no error message was captured"
            _xfmsg = ''
            _result = {}

            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)

            if not _status :
                _status, _fmsg = self.initialize_object(obj_attr_list, command)

                if not _status :
                    _obj_attrib, _obj_value = obj_attr_list["specified_kv_pair"].split('=')

                    _fields = [] 
                    _fields.append("|attribute                              ")
                    _fields.append("|old value                          ")
                    _fields.append("|new value                          ")

                    _p_obj_attrib = _obj_attrib.replace(obj_attr_list["specified_attribute"] + '_', '')

                    _smsg = "The attribute \"" + _p_obj_attrib + "\" on "
                    _smsg += obj_attr_list["specified_attribute"] + " "
                    _smsg += obj_attr_list["object_type"] 
                    _smsg += " was modified:\n"

                    _fmsg = "The attribute \"" + _p_obj_attrib + "\" on "
                    _fmsg += obj_attr_list["specified_attribute"] + " "
                    _fmsg += obj_attr_list["object_type"] 
                    _fmsg += " could not be modified modified:\n"

                    _fmt_obj_chg_attr = ''.join(_fields) + '\n'

                    _old_values = self.osci.get_object(obj_attr_list["cloud_name"], "GLOBAL", False, \
                                                       obj_attr_list["global_object"], \
                                                       False)

                    # This is just an ugly hack. The global object "vm_templates"
                    # is stored in a manner very different from any other object
                    if obj_attr_list["global_object"] == "vm_templates" :

                        _obj_attrib = _p_obj_attrib
                        if obj_attr_list["specified_attribute"] not in _old_values :
                            _old_values[obj_attr_list["specified_attribute"]] = _obj_attrib + ":non-existent"
                            self.osci.add_to_list(obj_attr_list["cloud_name"], "GLOBAL", "vm_roles", obj_attr_list["specified_attribute"])

                        _old_values = _old_values[obj_attr_list["specified_attribute"]]
                        _old_values = str2dic(_old_values)

                        _current_values = copy.deepcopy(_old_values)

                    if not _obj_attrib in _old_values :
                        _old_values[_obj_attrib] = "non-existent"

                    # Ugly hack continues here.
                    if obj_attr_list["global_object"] == "vm_templates" :

                        _current_values[_obj_attrib] = _obj_value
                        _current_values = dic2str(_current_values)

                        self.osci.update_object_attribute(obj_attr_list["cloud_name"], \
                                                          "GLOBAL", \
                                                          obj_attr_list["global_object"], \
                                                          False, \
                                                          obj_attr_list["specified_attribute"], \
                                                          _current_values)
                    else :
                        self.osci.update_object_attribute(obj_attr_list["cloud_name"], \
                                                          "GLOBAL", \
                                                          obj_attr_list["global_object"], \
                                                          False, \
                                                          _obj_attrib, \
                                                          _obj_value)

                    _fmt_obj_chg_attr += '|' + _obj_attrib.ljust(len(_fields[0]) - 1)
                    _fmt_obj_chg_attr += '|' + _old_values[_obj_attrib].ljust(len(_fields[1]) - 1)
                    _fmt_obj_chg_attr += '|' + _obj_value.ljust(len(_fields[2]) - 1)
                    _result["old_" + _obj_attrib] = _old_values[_obj_attrib]

                    _result[_obj_attrib] = _obj_value  
                    _smsg += _fmt_obj_chg_attr
    
                    _status = 0

        except self.ObjectOperationException as obj :
            _status = 8
            _xfmsg = str(obj.msg)

        except self.osci.ObjectStoreMgdConnException as obj :
            _status = 8
            _xfmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _xfmsg = str(e)

        finally :
            if _status :
                _msg = _fmsg + _xfmsg
                cberr(_msg)
            else :
                _msg = _smsg
                cbdebug(_msg)

            return self.package(_status, _msg, _result)

    @trace
    def expidmanage(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        try :
            _status = 100
            _fmsg = "An error has occurred, but no error message was captured"
            
            _result = None 
            _new_expid = None
            
            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)
            
            if not _status :
                
                _status, _fmsg = self.initialize_object(obj_attr_list, command)

                if not _status :
                    
                    _time_obj_attr_list = self.osci.get_object(obj_attr_list["cloud_name"], \
                                                               "GLOBAL", False, \
                                                               "time", False)

                    _username = _time_obj_attr_list["username"]
                    
                    _curr_expid = _time_obj_attr_list["experiment_id"]

                    self.expid = _curr_expid
                    
                    _parameters = obj_attr_list["command"].split()

                    _result = {}
                                        
                    if len(_parameters) == 2 :
                                    
                        _result["current"] = _time_obj_attr_list["experiment_id"]
            
                    else :
                        
                        _new_expid = _parameters[2]
                        
                        if _new_expid.count("$CLOUD_NAME") :
                            _new_expid = _new_expid.replace("$CLOUD_NAME",obj_attr_list["cloud_name"])
                            
                        self.osci.update_object_attribute(obj_attr_list["cloud_name"], \
                                                          "GLOBAL", "time", False,\
                                                           "experiment_id", \
                                                           _new_expid, False)

                        self.expid = _new_expid
                        self.initialize_metric_name_list(obj_attr_list)

                        if not _status :
                            _result["current"] = _new_expid

                    _result["experiment_list"] = self.get_msci(obj_attr_list["cloud_name"]).get_experiment_list("reported_management_VM_metric_names_" + _username)
                    
                    _status = 0

        except self.osci.ObjectStoreMgdConnException as obj :
            _status = 8
            _fmsg = str(obj.msg)

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "Unable to get EXPID object list: " + _fmsg
                cberr(_msg)
                _view_dict = False

            else :
                if _new_expid :
                    _msg = "The attribute \"experiment_id \" was changed from \""
                    _msg += _curr_expid + "\" to \"" + _result["current"] + "\"."
                    cbdebug(_msg)
                    self.update_host_os_perfmon(obj_attr_list)
                    self.update_logstore(obj_attr_list)
                else :
                    _msg = "Current experiment identifier is \"" + _result["current"] + "\".\n\n"
                    _msg += "Also, data for the following experiments was found on "
                    _msg += "the metric store:\n"
                    _msg += '\n'.join(_result["experiment_list"])
                    cbdebug(_msg)
                    
            return self.package(_status, _msg, _result)

    @trace
    def getrandnr(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        try :
            _status = 100
            _fmsg = "An error has occurred, but no error message was captured"

            _result = None

            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)

            if not _status :

                _status, _fmsg = self.initialize_object(obj_attr_list, command)

                if not _status :

                    _dist = obj_attr_list["distribution"]
                    _vg = ValueGeneration(self.pid)

                    _start = time()
                    _result = _vg.get_value(_dist)
                    _end = time() - _start
                    
                    _status = 0

        except Exception as e :
            _status = 23
            _fmsg = str(e)

        finally :
            if _status :
                _msg = "Unable to get a random number using distribution  \"" + _dist + "\""
                cberr(_msg)

            else :
                _msg = "Got the value \"" + str(_result) + "\" after " + str(_end) + " seconds"
                cbdebug(_msg)
                                    
            return self.package(_status, _msg, _result)

    def list_domains(self, cloud_name, lvirt_conns,  uuid) :
        '''
        TBD
        '''
        domains = {}
        ips = {}
        status = True
        attrs = {}
    
        app = self.osci.get_object(cloud_name, "AI", False, uuid, False)
        metric_vm = self.osci.get_object(cloud_name, "VM", False, app["metric_aggregator_vm"], False)
        metric_defaults = self.osci.get_object(cloud_name, "GLOBAL", False, "mon_defaults", False)
        metric_port = metric_defaults["collector_vm_port"]
    
        for vm_parts in app["vms"].split(",") :
            parts = vm_parts.split("|")
            vm_uuid = parts[0]
            vm = self.osci.get_object(cloud_name, "VM", False, vm_uuid, False)
            if "host_cloud_ip" not in vm :
                continue
    
            host_ip = vm["host_cloud_ip"]
            if vm_uuid in lvirt_conns :
                lvirt_conns[vm_uuid].close()
    
            lvirt_conns[vm_uuid] = libvirt.open("qemu+tcp://" + host_ip + "/system")
            domains[vm_uuid] = lvirt_conns[vm_uuid].lookupByName(vm["cloud_vm_uuid"])
            ips[vm_uuid] = vm["cloud_ip"]
            attrs[vm_uuid] = vm
    
        return status, domains, ips, metric_vm["cloud_ip"], metric_port, attrs

    def qemu_send(self, g, key, val, typ, unit, direction, lifetime, category, spoof, attrs) :
        '''
        TBD
        '''
        if key.count("mbps") :
            unit = "mbps"
        elif key.count("bytes") :
            unit = "bytes"

        g.send(key, val, typ, unit, direction, self.freq_str, lifetime, category, spoof)
        attrs["mgt_505_" + key] = val
        
        
    def deliver(self, g, key, value, spoof, attrs) :
        '''
        TBD
        '''
        unit = "n/a"
        num = False
        typ = "n/a"
        try :
            num = int(value)
            unit = "ms"
            typ = "int32"
        except ValueError :
            try :
                num = float(value)
                unit = "mbps"
                typ = "float"
                self.qemu_send(g, key, str(value), "float", "mbps", "both", "0", "mc", spoof, attrs)
            except ValueError :
                pass
    
        if num is not False :
            self.qemu_send(g, "qemu_" + key.replace("-", "_"), str(num), typ, unit, "both", "0", "mc", spoof, attrs)

    def get_stats_all_domains(self, cloud_name, g, ips, domains, agg_ip, vms) :
        '''
        TBD
        '''
        for name in domains :
            dom = domains[name]
            stats = dom.jobStats()
            
            # If your libvirt is too old (< 0.10), then memoryStats() will not return anything useful
            mem = dom.memoryStats()
            ip = ips[name]
            attrs = vms[name]
            if not ip :
                continue
    
            spoof = ip + ":" + ip
            #cbdebug("domain: " + str(name) + " ip " + str(ip) + " uuid: " + \
            #        str(dom.UUIDString()) + " stats: " + str(stats) + " qemu: " + str(mem) + "\n")
    
            if "actual" in mem :
                g.send("qemu_max", str(mem["actual"]), "int32", "KB", "both", self.freq_str, "0", "memory", spoof)
            else :
                cbwarn("QEMU result is missing the 'actual' key. strange.")
            
            if "rss" in mem :
                g.send("qemu_rss", str(mem["rss"]), "int32", "KB", "both", self.freq_str, "0", "memory", spoof)
            else :
                cbwarn("QEMU result is missing the 'rss' key. strange.")
    
            for key in stats :
                value = stats[key]
                
                if isinstance(value, str) :
                    self.deliver(g, key, value, spoof, attrs)
                elif isinstance(value, int) or isinstance(value, int) :
                    self.deliver(g, key, str(value), spoof, attrs)
                elif isinstance(value, float) :
                    self.deliver(g, key, str(value), spoof, attrs)
                elif isinstance(value, dict) :
                    for subkey in stats[key] :
                        value = stats[key][subkey]
                        self.deliver(g, subkey, value, spoof, attrs)
            else :
                cbwarn("skipping key! " + str(key) + " " + str(value) + " type : " + str(type(value)))
                    
            self.record_management_metrics(cloud_name, "VM", attrs, "runstate")
    
    def migration_checker(self, my_uuid, cv, cloud_name) :
        '''
        TBD
        '''
        cbdebug("Migration checker started.")
        
        while self.check_for_migrate :
            try :
                sub_channel = self.osci.subscribe(cloud_name, "AI", "migrate_" + my_uuid)
                
                cbdebug("Migration subsribe channel created")
                
                while self.check_for_migrate :
                    for message in sub_channel.listen() :
                        val = str(message["data"])
                        args = val.split(";")
                        if len(args) != 3 :
                            cbdebug("Not for me: " + val)
                            continue
        
                        uuid, action, info = args
                        if action == "error" :
                            cberr("Migration checker breaking on error: " + info)
                            break
                        
                        if action == "start" :
                            self.qemu_check_freq = float(info)
                            self.freq_str = str(self.qemu_check_freq)
                            cbdebug("waking up scraper for: " + val + " " + self.freq_str + " intervals, VM: " + uuid)
                        elif action == "stop" :
                            self.qemu_check_freq = self.qemu_check_freq_default 
                            self.freq_str = str(self.qemu_check_freq)
                            cbdebug("migration complete: Setting qemu check frequency back to normal: " + \
                                    self.freq_str + " secs.")
                            
                        cv.acquire()
                        cv.notify()
                        cv.release()
                    
            except self.osci.ObjectStoreMgdConnException as msg :
                cberr("Migration checker subscription channel broken: " + str(msg) + ", trying again in 5 seconds...")
                sleep(5)
            except ConnectionError as msg :
                cberr("Migration checker subscription channel broken: " + str(msg) + ", trying again in 5 seconds...")
                sleep(5)
                    
        sub_channel.unsubscribe()
        cbdebug("Migration checker exiting.")

    def qemu_scraper(self, cloud_name, uuid) :
        '''
        TBD
        '''
        if not qemu_supported :
            cbwarn("Not starting scraper because associated libvirt bindings are not available.")
            return
        

        last_refresh = str(time())
    
        g = None
        self.qemu_check_freq_default = 15
        self.qemu_check_freq = self.qemu_check_freq_default 
        self.freq_str = str(self.qemu_check_freq)
        count = 0
        lvirt_conns = {}
        
        cv = Condition()
        cv.acquire()
           
        self.check_for_migrate = True
        t = threading.Thread(target=self.migration_checker, args = [uuid, cv, cloud_name])
        t.daemon = True
        t.start()
        
        while True :
            try :
                if count == 0 or ((self.qemu_check_freq == self.qemu_check_freq_default) and \
                                    self.compare_refresh(cloud_name, last_refresh)) :
                    last_refresh = str(time())
                    count = 10 
                    (status, domains, ips, agg_ip, agg_port, vms) = self.list_domains(cloud_name, lvirt_conns, uuid)
                    g = Gmetric(agg_ip, str(agg_port), "udp")
                    cbdebug("Found " + str(len(domains)) + " domains")
    
                if not status :
                    cbwarn("Listing domains failed! darn.")
                else :
                    self.get_stats_all_domains(cloud_name, g, ips, domains, agg_ip, vms)
    
            except Exception as msg:
                cberr("Failure: " + str(msg))
                try :
                    count = 1
                except Exception as msg:
                    cberr("Failed to re-list during failure: " + str(msg))
                    pass
                
                cberr("Failed to deliver metrics: " + str(msg))
    
            cv.wait(self.qemu_check_freq)
            self.freq_str = str(self.qemu_check_freq)
            count -= 1
        
        self.check_for_migrate = False
            
    @trace
    def should_refresh(self, obj_attr_list, parameters, command) :
        '''
        TBD
        '''
        _result = False
        
        try :
            _status = 100
            _fmsg = "An error has occurred, but no error message was captured"
            _obj_type = command.split('-')[0].upper()
            _status, _fmsg = self.parse_cli(obj_attr_list, parameters, command)

            if not _status :
                _result = self.compare_refresh(obj_attr_list["cloud_name"], float(obj_attr_list["time"]))
                    
            _status = 0
            
        except BaseObjectOperations.ObjectOperationException as msg :
            _status = 30
            _fmsg = str(msg)
            
        finally :
            if _status :
                _msg = "Refresh check failure: " + _fmsg
                cberr(_msg)
            else :
                _msg = "Refresh check: " + str(_result)
                cbdebug(_msg)
                
        return self.package(_status, _msg, _result)
