from unittest.mock import patch

import pytest

from src.tasks.controllers.regress_controller import regress_controller
from src.tasks.models.task import Task


@pytest.fixture
def mocked_view_all_controller():
    with patch(
        "src.tasks.controllers.regress_controller.view_all_controller"
    ) as m:
        yield m


def test_can_regress_a_task(tmp_db, mocked_view_all_controller):
    task1 = Task.create(title="buy milk", status=1)
    assert Task.get(Task.id == task1.id).status == 1, (
        "Sanity check: task being at level 1"
    )

    regress_controller(task_ids=[task1.id])

    assert Task.get(Task.id == task1.id).status == 0, "Task status moved down"
    mocked_view_all_controller.asssert_called_once()


def test_can_regress_multiple_tasks(tmp_db, mocked_view_all_controller):
    task1 = Task.create(title="task 1", status=1)
    task2 = Task.create(title="task 2", status=2)
    task3 = Task.create(title="task 3", status=3)

    regress_controller(task_ids=[task1.id, task2.id])

    mocked_view_all_controller.asssert_called_once()
    assert Task.get(Task.id == task1.id).status == 0, (
        f"Task {task1.id} status moved down"
    )
    assert Task.get(Task.id == task2.id).status == 1, (
        f"Task {task2.id} status moved down"
    )
    assert Task.get(Task.id == task3.id).status == 3, (
        f"Task {task3.id} status was unchanged"
    )


def test_cannot_regress_beyond_highest_status(
    tmp_db, mocked_view_all_controller
):
    first_status = 0
    task1 = Task.create(title="task 1", status=first_status)

    regress_controller(task_ids=[task1.id])

    mocked_view_all_controller.asssert_called_once()
    assert Task.get(Task.id == task1.id).status == first_status, (
        "Cannot move beyond first status"
    )


def test_non_existing_tasks_are_silently_ignored(
    tmp_db, mocked_view_all_controller
):
    task1 = Task.create(title="task 1", status=1)

    # 2 and 3 are ids of non-existing todos
    regress_controller(task_ids=[task1.id, 2, 3])

    # It should succeed with no issues
    assert Task.get(Task.id == task1.id).status == 0, (
        f"Task {task1.id} status moved down"
    )
    mocked_view_all_controller.asssert_called_once()
