"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Provider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const eks = require("@aws-cdk/aws-eks");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const runner = require("./runner");
const gitLabClusterRoleBinding = {
    apiVersion: 'rbac.authorization.k8s.io/v1beta1',
    kind: 'ClusterRoleBinding',
    metadata: { name: 'gitlab-admin' },
    roleRef: {
        apiGroup: 'rbac.authorization.k8s.io',
        kind: 'ClusterRole',
        name: 'cluster-admin',
    },
    subjects: [
        {
            kind: 'ServiceAccount',
            name: 'gitlab',
            namespace: 'kube-system',
        },
    ],
};
/**
 * The Provider to create GitLab Integrations with AWS.
 *
 * @stability stable
 */
class Provider extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b;
        super(scope, id);
        const stack = cdk.Stack.of(this);
        this.vpc = (_b = props.vpc) !== null && _b !== void 0 ? _b : new ec2.Vpc(this, 'Vpc', { natGateways: 1 });
        new cdk.CfnOutput(this, 'Region', { value: stack.region });
    }
    /**
     * @stability stable
     */
    createFargateRunner(executor) {
        const token = this.node.tryGetContext('GITLAB_REGISTRATION_TOKEN') || process.env.GITLAB_REGISTRATION_TOKEN;
        if (!token) {
            throw new Error('missing GITLAB_REGISTRATION_TOKEN in the context variable');
        }
        new runner.FargateRunner(this, 'FargateRunner', {
            vpc: this.vpc,
            registrationToken: this.node.tryGetContext('GITLAB_REGISTRATION_TOKEN'),
            executor,
        });
    }
    /**
     * @stability stable
     */
    createEksCluster(scope, id, props) {
        const cluster = new eks.Cluster(scope, id, props.clusterOptions);
        if (props.rbac != false) {
            cluster.addServiceAccount('gitlab');
            cluster.addManifest('ClusterRoleBinding', gitLabClusterRoleBinding);
        }
        return cluster;
    }
    /**
     * @stability stable
     */
    createFargateEksCluster(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l;
        const cluster = new eks.FargateCluster(scope, id, props.clusterOptions);
        if (props.rbac != false) {
            cluster.addServiceAccount('gitlab');
            cluster.addManifest('ClusterRoleBinding', gitLabClusterRoleBinding);
        }
        const registrationToken = (_c = (_b = props.helmRunnerOptions) === null || _b === void 0 ? void 0 : _b.registrationToken) !== null && _c !== void 0 ? _c : (this.node.tryGetContext('GITLAB_REGISTRATION_TOKEN') || process.env.GITLAB_REGISTRATION_TOKEN);
        if (!registrationToken) {
            throw new Error('missing GITLAB_REGISTRATION_TOKEN in the context variable');
        }
        cluster.addHelmChart('helmrunner', {
            chart: 'gitlab-runner',
            repository: 'https://charts.gitlab.io',
            namespace: 'default',
            values: {
                // use fargate run job ,always need to pull.
                imagePullPolicy: 'Always',
                terminationGracePeriodSeconds: 3600,
                concurrent: (_e = (_d = props.helmRunnerOptions) === null || _d === void 0 ? void 0 : _d.concurrent) !== null && _e !== void 0 ? _e : 10,
                checkInterval: 30,
                gitlabUrl: (_g = (_f = props.helmRunnerOptions) === null || _f === void 0 ? void 0 : _f.gitlabURL) !== null && _g !== void 0 ? _g : 'https://gitlab.com/',
                runnerRegistrationToken: registrationToken,
                unregisterRunners: true,
                rbac: {
                    create: true,
                    clusterWideAccess: false,
                },
                metrics: {
                    enabled: true,
                },
                securityContext: {
                    fsGroup: 65533,
                    runAsUser: 100,
                },
                // runners default image when job start not set "image" in gitlab-ci.yaml.
                runners: {
                    image: (_j = (_h = props.helmRunnerOptions) === null || _h === void 0 ? void 0 : _h.jobDefaultImage) !== null && _j !== void 0 ? _j : 'alpine:3.11',
                    tags: this.synthesizeTags((_l = (_k = props.helmRunnerOptions) === null || _k === void 0 ? void 0 : _k.tags) !== null && _l !== void 0 ? _l : ['eks', 'fargate', 'runner']),
                    privileged: false,
                },
                envVars: [
                    {
                        name: 'RUNNER_EXECUTOR',
                        value: 'kubernetes',
                    },
                ],
            },
        });
        return cluster;
    }
    /**
     * @stability stable
     */
    createSecurityGroup() {
        const sg = new ec2.SecurityGroup(this, 'GitlabEksSecurityGroup', {
            vpc: this.vpc,
        });
        new cdk.CfnOutput(this, 'SecurityGroup', { value: sg.securityGroupName });
        return sg;
    }
    /**
     * @stability stable
     */
    createEksServiceRole() {
        return new iam.Role(this, 'GitlabEksServiceRole', {
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy')],
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
        });
    }
    // this.gitlabEksRole = new iam.Role(this, 'GitlabManagedEksRole', {
    //   assumedBy: new iam.AccountPrincipal(props.accountId),
    //   externalIds: [props.externalId],
    //   inlinePolicies: {
    //     default: policy,
    //   },
    // });
    //   new cdk.CfnOutput(this, 'RoleArn', { value: this.gitlabEksRole.roleArn });
    // }
    /**
     * @stability stable
     */
    createGitlabManagedEksRole(props) {
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'autoscaling:CreateAutoScalingGroup',
                        'autoscaling:DescribeAutoScalingGroups',
                        'autoscaling:DescribeScalingActivities',
                        'autoscaling:UpdateAutoScalingGroup',
                        'autoscaling:CreateLaunchConfiguration',
                        'autoscaling:DescribeLaunchConfigurations',
                        'cloudformation:CreateStack',
                        'cloudformation:DescribeStacks',
                        'ec2:AuthorizeSecurityGroupEgress',
                        'ec2:AuthorizeSecurityGroupIngress',
                        'ec2:RevokeSecurityGroupEgress',
                        'ec2:RevokeSecurityGroupIngress',
                        'ec2:CreateSecurityGroup',
                        'ec2:createTags',
                        'ec2:DescribeImages',
                        'ec2:DescribeKeyPairs',
                        'ec2:DescribeRegions',
                        'ec2:DescribeSecurityGroups',
                        'ec2:DescribeSubnets',
                        'ec2:DescribeVpcs',
                        'eks:CreateCluster',
                        'eks:DescribeCluster',
                        'iam:AddRoleToInstanceProfile',
                        'iam:AttachRolePolicy',
                        'iam:CreateRole',
                        'iam:CreateInstanceProfile',
                        'iam:CreateServiceLinkedRole',
                        'iam:GetRole',
                        'iam:ListRoles',
                        'iam:PassRole',
                        'ssm:GetParameters',
                    ],
                    resources: ['*'],
                }),
            ],
        });
        this.gitlabEksRole = new iam.Role(this, 'GitlabManagedEksRole', {
            assumedBy: new iam.AccountPrincipal(props.accountId),
            externalIds: [props.externalId],
            inlinePolicies: {
                default: policy,
            },
        });
        const eksAdminRole = this.createEksServiceRole();
        this.createSecurityGroup();
        const subnetIds = this.vpc.selectSubnets({
            subnetType: ec2.SubnetType.PRIVATE,
        }).subnetIds;
        new cdk.CfnOutput(this, 'EksClusterSubnetIds', {
            value: subnetIds.join(','),
        });
        new cdk.CfnOutput(this, 'RoleArn', { value: this.gitlabEksRole.roleArn });
        new cdk.CfnOutput(this, 'EksAdminRole', { value: eksAdminRole.roleName });
    }
    synthesizeTags(tags) {
        return tags.join(',');
    }
}
exports.Provider = Provider;
_a = JSII_RTTI_SYMBOL_1;
Provider[_a] = { fqn: "cdk-gitlab.Provider", version: "0.1.229" };
//# sourceMappingURL=data:application/json;base64,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