# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['csv_reconcile', 'csv_reconcile_dice']

package_data = \
{'': ['*']}

install_requires = \
['cython>=0.29.21,<0.30.0',
 'flask-cors>=3.0.10,<4.0.0',
 'flask>=1.1.2,<2.0.0',
 'normality>=2.1.1,<3.0.0']

extras_require = \
{':python_version >= "3.7" and python_version < "3.8"': ['importlib_metadata>=3.7.3,<4.0.0']}

entry_points = \
{'console_scripts': ['csv-reconcile = csv_reconcile:main'],
 'csv_reconcile.scorers': ['dice = csv_reconcile_dice']}

setup_kwargs = {
    'name': 'csv-reconcile',
    'version': '0.2.3',
    'description': 'OpenRefine reconciliation service backed by csv resource',
    'long_description': '\n# Table of Contents\n\n1.  [CSV Reconcile](#org711da59)\n    1.  [Quick start](#org6bdaac8)\n    2.  [Poetry](#org888d7e4)\n        1.  [Building](#org22f1c7d)\n    3.  [Description](#org8f09222)\n    4.  [Usage](#org77b4857)\n    5.  [Common configuration](#orgc41ff08)\n    6.  [Scoring plugins](#org886e0f7)\n        1.  [Implementing](#org5d17c25)\n        2.  [Installing](#org33a48ba)\n        3.  [Using](#orgcc9e8bc)\n    7.  [Future enhancements](#org3b13faa)\n\n\n<a id="org711da59"></a>\n\n# CSV Reconcile\n\nA [reconciliation service](https://github.com/reconciliation-api/specs) for [OpenRefine](https://openrefine.org/) based on a CSV file similar to [reconcile-csv](http://okfnlabs.org/reconcile-csv/).  This one is written in Python and has some more configurability.\n\n\n<a id="org6bdaac8"></a>\n\n## Quick start\n\n-   Clone this repository\n-   Run the service\n    \n        $ python -m venv venv                                             # create virtualenv\n        $ venv/bin/pip install csv-reconcile                              # install package\n        $ source venv/bin/activate                                        # activate virtual environment\n        (venv) $ csv-reconcile --init-db sample/reps.tsv item itemLabel   # start the service\n        (venv) $ deactivate                                               # remove virtual environment\n\nThe service is run at <http://127.0.0.1:5000/reconcile>.  You can point at a different host:port by\nadding [SERVER\\_NAME](https://flask.palletsprojects.com/en/0.12.x/config/) to the sample.cfg.  Since this is running from a virtualenv, you can simply\ndelete the whole lot to clean up.\n\nIf you have a C compiler installed you may prefer to install the sdist\n`dist/csv-reconcile-0.1.0.tar.gz` which will build a [Cython](https://cython.readthedocs.io/en/latest/) version of the computationally\nintensive fuzzy match routine for speed.  With `pip` add the option `--no-binary csv-reconcile`.\n\n\n<a id="org888d7e4"></a>\n\n## Poetry\n\nThis is packaged with [poetry](https://python-poetry.org/docs/), so you can use those commands if you have it installed.\n\n    $ poetry run poe install\n    $ poetry run csv-reconcile --init-db sample/reps.tsv item itemLabel\n\n\n<a id="org22f1c7d"></a>\n\n### Building\n\nBecause this package uses a `README.org` file and `pip` requires a `README.md`, there are extra\nbuild steps beyond what `poetry` supplies.  These are managed using [poethepoet](https://pypi.org/project/poethepoet/0.0.3/).  Thus building is\ndone as follows:\n\n    $ poetry run poe build\n\nIf you want to build a platform agnostic wheel, you\'ll have to comment out the `build =\n    "build.py"` line from `pyproject.toml` until `poetry` supports [selecting build platform](https://github.com/python-poetry/poetry/issues/3594).\n\n\n<a id="org8f09222"></a>\n\n## Description\n\nThis reconciliation service uses [Dice coefficient scoring](https://en.wikipedia.org/wiki/S%C3%B8rensen%E2%80%93Dice_coefficient) to reconcile values against a given column\nin a [CSV](https://en.wikipedia.org/wiki/Comma-separated_values) file.  The CSV file must contain a column containing distinct values to reconcile to.\nWe\'ll call this the *id column*.  We\'ll call the column being reconciled against the *name column*.\n\nFor performance reasons, the *name column* is preprocessed to normalized values which are stored in\nan [sqlite](https://www.sqlite.org/index.html) database.  This database must be initialized at least once by passing the `--init-db` on\nthe command line.  Once initialized this option can be removed from subsequent runs.\n\nNote that the service supplies all its data with a dummy *type* so there is no reason to reconcile\nagainst any particular *type*.\n\nIn addition to reconciling against the *name column*, the service also functions as a [data extension\nservice](https://reconciliation-api.github.io/specs/latest/#data-extension-service), which offers any of the other columns of the CSV file.\n\nNote that Dice coefficient scoring is agnostic to word ordering.\n\n\n<a id="org77b4857"></a>\n\n## Usage\n\nBasic usage requires passing the name of the CSV file, the *id column* and the *name column*.\n\n    $ poetry run csv-reconcile --help\n    Usage: csv-reconcile [OPTIONS] CSVFILE IDCOL NAMECOL\n    \n    Options:\n      --config TEXT  config file\n      --scorer TEXT  scoring plugin to use\n      --init-db      initialize the db\n      --help         Show this message and exit.\n    $\n\nIn addition to the `--init-db` switch mentioned above you may use the `--config` option to point to\na configuration file.  The file is a [Flask configuration](https://flask.palletsprojects.com/en/1.1.x/config/) and hence is Python code though most\nconfiguration is simply setting variables to constant values.\n\n\n<a id="orgc41ff08"></a>\n\n## Common configuration\n\n-   `SERVER_NAME`  - The host and port the service is bound to.\n    e.g. `SERVER_NAME=localhost:5555`.  ( Default localhost:5000 )\n-   `CSVKWARGS`  - Arguments to pass to [csv.reader](https://docs.python.org/3/library/csv.html).\n    e.g. `CSVKWARGS={\'delimiter\': \',\', \'quotechar\': \'"\'}` for comma delimited files using `"` as quote character.\n-   `CSVENCODING` - Encoding of the CSV file.\n    e.g. `CSVECODING=\'utf-8-sig\'` is the encoding used for data downloaded from [GNIS](https://www.usgs.gov/core-science-systems/ngp/board-on-geographic-names/download-gnis-data).\n-   `SCOREOPTIONS`  - Options passed to scoring plugin during normalization.\n    e.g. `SCOREOPTIONS={\'stopwords\':[\'lake\',\'reservoir\']}`\n-   `LIMIT`      - The maximum number of reonciliation candidates returned per entry.  ( Default 10 )\n    e.g. `LIMIT=10`\n-   `THRESHOLD`  - The minimum score for returned reconciliation candidates.  ( Default 30.0 )\n    e.g. `THRESHOLD=80.5`\n-   `DATABASE`   - The name of the generated sqlite database containing pre-processed values.  (Default `csvreconcile.db`)\n    e.g. `DATABASE=\'lakes.db\'`  You may want to change the name of the database if you regularly switch between databases being used.\n-   `MANIFEST`   - Overrides for the service manifest.\n    e.g. `MANIFEST={"name": "My service"}` sets the name of the service to "My service".\n\nThis last is most interesting.  If your data is coming from [Wikidata](https://www.wikidata.org) and your *id column*\ncontains [Q values](https://www.wikidata.org/wiki/Help:Items), then a manifest like the following will allow your links to be clickable inside OpenRefine.\n\n    MANIFEST = {\n      "identifierSpace": "http://www.wikidata.org/entity/",\n      "schemaSpace": "http://www.wikidata.org/prop/direct/",\n      "view": {"url":"https://www.wikidata.org/wiki/{{id}}"},\n      "name": "My reconciliation service"\n    }\n\nIf your CSV is made up of data taken from another [reconciliation service](https://reconciliation-api.github.io/testbench/), you may similiarly copy\nparts of their manifest to make use of their features, such as the [preview service](https://reconciliation-api.github.io/specs/latest/#preview-service).  See the\nreconciliation spec for details.\n\n\n<a id="org886e0f7"></a>\n\n## Scoring plugins\n\nAs mentioned above the default scoring method is to use [Dice coefficient scoring](https://en.wikipedia.org/wiki/S%C3%B8rensen%E2%80%93Dice_coefficient), but this method\ncan be overridden by implementing a `cvs_reconcile.scorers` plugin.\n\n\n<a id="org5d17c25"></a>\n\n### Implementing\n\nA plugin module may override any of the methods in the `csv_reconcile.scorers` module by simply\nimplementing a method of the same name with the decorator `@cvs_reconcile.scorer.register`.\n\nSee `csv_reconcile_dice` for how Dice coefficient scoring is implemented.\n\nThe basic hooks are as follows:\n\n-   `normalizedWord(word, **scoreOptions)` preprocesses values to be reconciled to produce a tuple\n    used in fuzzy match scoring.  The value of `SCOREOPTIONS` in the configuration will be passed\n    in to allow configuration of this preprocessing.  This hook is required.\n-   `normalizedRow(word, row, **scoreOptions)` preprocesses values to be reconciled against to\n    produce a tuple used in fuzzy match scoring.  Note that both the reconciled column and the\n    entire row is available for calculating the normalized value and that the column reconciled\n    against is required even when not used.  The value of `SCOREOPTIONS` in the configuration will\n    be passed in to allow configuration of this preprocessing.  This defaults to calling\n    normalizeWord(word,\\*\\*scoreOptions).\n-   `getNormalizedFields()` returns a tuple of names for the columns produced by `normalizeWord()`.\n    The length of the return value from both functions must match.  This defaults to calling\n    normalizeWord(word,\\*\\*scoreOptions).  This hook is required.\n-   `processScoreOptions(options)` is passed the value of `SCOREOPTIONS` to allow it to be adjusted\n    prior to being used.  This can be used for adding defaults and/or validating the configuration.\n    This hook is optional\n-   `scoreMatch(left, right, **scoreOptions)` gets passed two tuples as returned by\n    `normalizedWord()`.  The `left` value is the value being reconciled and the `right` value is\n    the value being reconciled against.  The value of `SCOREOPTIONS` in the configuration will be\n    passed in to allow configuration of this preprocessing.  Returning a score of `None` will not\n    add tested value as a candidate. This hook is required.\n-   `valid(normalizedFields)` is passed the normalized tuple prior to being scored to make sure\n    it\'s appropriate for the calculation.  This hook is optional.\n\n\n<a id="org33a48ba"></a>\n\n### Installing\n\nHooks are automatically discovered as long as they provide a `csv_reconcile.scorers` [setuptools\nentry point](https://setuptools.readthedocs.io/en/latest/userguide/entry_point.html).  Poetry supplies a [plugins](https://python-poetry.org/docs/pyproject/#plugins) configuration which wraps the setuptools funtionality.\n\nThe default Dice coefficent scoring is supplied via the following snippet from `pyproject.toml`\nfile.\n\n    [tool.poetry.plugins."csv_reconcile.scorers"]\n    "dice" = "csv_reconcile_dice"\n\nHere `dice` becomes the name of the scoring option and `csv_reconcile_dice` is the package\nimplementing the plugin.\n\n\n<a id="orgcc9e8bc"></a>\n\n### Using\n\nIf there is only one scoring plugin available, that plugin is used.  If there are more than one\navailable, you will be prompted to pass the `--scorer` option to select among the scoring options.\n\n\n<a id="org3b13faa"></a>\n\n## Future enhancements\n\nIt would be nice to add support for using [properties](https://reconciliation-api.github.io/specs/latest/#structure-of-a-reconciliation-query) as part of the scoring, so that more than\none column of the csv could be taken into consideration.\n\n',
    'author': 'Douglas Mennella',
    'author_email': 'trx2358-pypi@yahoo.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/gitonthescene/csv-reconcile',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}
from build import *
build(setup_kwargs)

setup(**setup_kwargs)
