import ast
import filecmp
import json
import time

import pytest
from click.testing import CliRunner

from commands import report
from commands.logout import cli as logout
from commands.workspaces import cli as workspaces
from tests.helpers.test_utils import assert_success
from neoload_cli_lib import rest_crud
from neoload_cli_lib.user_data import UserData


@pytest.mark.report
@pytest.mark.usefixtures("neoload_login")  # it's like @Before on the neoload_login function
class TestReportJsonOutput:

    @pytest.mark.datafiles('tests/resources/report/expected_report.json')
    def test_parse_source_data_spec(self, monkeypatch, datafiles):
        runner = CliRunner()
        result_ws = runner.invoke(workspaces, ['use', '5f689c3f0860270001606902'])
        assert_success(result_ws)

        monkeypatch.setattr(rest_crud, 'get', lambda actual_endpoint: ast.literal_eval(self.__return_json(actual_endpoint)))
        monkeypatch.setattr(time, 'localtime', lambda timestamp_utc: time.gmtime(timestamp_utc))
        model = report.initialize_model("", "")
        json_data = report.parse_source_data_spec(None, model, "single", "c6ae22a9-9868-4966-8b1e-8439b985e792")

        result_logout = runner.invoke(logout)
        assert_success(result_logout)

        with open('tests/resources/report/actual_report.json', 'w', newline='\n') as f:
            f.write(json.dumps(json_data, indent=2))
        assert filecmp.cmp(datafiles.listdir()[0], 'tests/resources/report/actual_report.json') is True, "Json output for the report (file tests/resources/report/actual_report.json) is not the one expected (file tests/resources/report/expected_report.json)"

    @pytest.mark.datafiles('tests/resources/report/expected_report_filtered.json')
    @pytest.mark.skip(reason="This unit test fails since modifications on report command after 1.3.0")
    def test_parse_source_data_spec_with_filter(self, monkeypatch, datafiles):
        runner = CliRunner()
        result_ws = runner.invoke(workspaces, ['use', '5f689c3f0860270001606902'])
        assert_success(result_ws)

        monkeypatch.setattr(rest_crud, 'get', lambda actual_endpoint: ast.literal_eval(self.__return_json(actual_endpoint)))
        monkeypatch.setattr(time, 'localtime', lambda timestamp_utc: time.gmtime(timestamp_utc))
        model = report.initialize_model("timespan=50%-70%;element=passed", "")
        json_data = report.parse_source_data_spec(None, model, "single", "c6ae22a9-9868-4966-8b1e-8439b985e792")

        result_logout = runner.invoke(logout)
        assert_success(result_logout)

        with open('tests/resources/report/actual_report_filtered.json', 'w', newline='\n') as f:
            f.write(json.dumps(json_data, indent=2))
        assert filecmp.cmp(datafiles.listdir()[0], 'tests/resources/report/actual_report_filtered.json') is True, "Json output for the report (file tests/resources/report/actual_report_filtered.json) is not the one expected (file tests/resources/report/expected_report_filtered.json)"

    def __return_json(self, endpoint):
        if endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792':
            return '{"id": "c6ae22a9-9868-4966-8b1e-8439b985e792", "name": "#3", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "rest_api", "scenario": "very_long_2min", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1612539609521, "endDate": 1612539729665, "duration": 120144, "testId": "432e95c8-aeb3-46c8-a026-443dfb18874a"}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/slas/statistics':
            return '[]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/slas/per-test':
            return '[{"kpi": "errors-count", "status": "FAILED", "value": 3550.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "errors-count", "status": "FAILED", "value": 0.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7bf42114-d630-494f-bc13-fea402ce085d", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "errors-count", "status": "FAILED", "value": 0.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d15f2c71-0e0b-454f-9802-2d9a5cc9baac", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "errors-count", "status": "FAILED", "value": 3550.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "6a0ef850-8644-4236-80e5-53748fbd2d01", "name": "transaction_sla_failed", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "errors-count", "status": "WARNING", "value": 3550.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a", "name": "transaction_sla_warning", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "errors-count", "status": "WARNING", "value": 0.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d319f4e8-d31e-4a1b-9187-b8238bc57007", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "errors-count", "status": "WARNING", "value": 3550.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "1034d624-451c-4523-b579-6f003ebd93f8", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "errors-count", "status": "WARNING", "value": 0.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "b24becfd-0be5-4e27-93c9-c7ce4a7db5e3", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "avg-page-resp-time", "status": "PASSED", "value": 0.03813545, "warningThreshold": {"operator": ">=", "values": [3600.0]}, "failedThreshold": {"operator": ">=", "values": [36000.0]}, "element": {"elementId": "1ea4878e-6df1-4c5b-bd5d-3c1f0566c91d", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_passed"}}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/slas/per-interval':
            return '[{"kpi": "avg-throughput-per-sec", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d15f2c71-0e0b-454f-9802-2d9a5cc9baac", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "avg-throughput-per-sec", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7bf42114-d630-494f-bc13-fea402ce085d", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "error-rate", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "errors-per-sec", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d15f2c71-0e0b-454f-9802-2d9a5cc9baac", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "error-rate", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "6a0ef850-8644-4236-80e5-53748fbd2d01", "name": "transaction_sla_failed", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "avg-elt-per-sec", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "6a0ef850-8644-4236-80e5-53748fbd2d01", "name": "transaction_sla_failed", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "avg-resp-time", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "errors-per-sec", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7bf42114-d630-494f-bc13-fea402ce085d", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "avg-elt-per-sec", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7bf42114-d630-494f-bc13-fea402ce085d", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "avg-resp-time", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7bf42114-d630-494f-bc13-fea402ce085d", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "avg-elt-per-sec", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d15f2c71-0e0b-454f-9802-2d9a5cc9baac", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "errors-per-sec", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "avg-resp-time", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d15f2c71-0e0b-454f-9802-2d9a5cc9baac", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "errors-per-sec", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "6a0ef850-8644-4236-80e5-53748fbd2d01", "name": "transaction_sla_failed", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "error-rate", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d15f2c71-0e0b-454f-9802-2d9a5cc9baac", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "avg-throughput-per-sec", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "error-rate", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7bf42114-d630-494f-bc13-fea402ce085d", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "avg-resp-time", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "6a0ef850-8644-4236-80e5-53748fbd2d01", "name": "transaction_sla_failed", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "avg-elt-per-sec", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "avg-throughput-per-sec", "status": "FAILED", "failed": 100.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "6a0ef850-8644-4236-80e5-53748fbd2d01", "name": "transaction_sla_failed", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "error-rate", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a", "name": "transaction_sla_warning", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "avg-resp-time", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "b24becfd-0be5-4e27-93c9-c7ce4a7db5e3", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "error-rate", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d319f4e8-d31e-4a1b-9187-b8238bc57007", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "error-rate", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "1034d624-451c-4523-b579-6f003ebd93f8", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "avg-resp-time", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a", "name": "transaction_sla_warning", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "avg-throughput-per-sec", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "b24becfd-0be5-4e27-93c9-c7ce4a7db5e3", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "avg-elt-per-sec", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "1034d624-451c-4523-b579-6f003ebd93f8", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "errors-per-sec", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "1034d624-451c-4523-b579-6f003ebd93f8", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "avg-elt-per-sec", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "b24becfd-0be5-4e27-93c9-c7ce4a7db5e3", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "errors-per-sec", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "b24becfd-0be5-4e27-93c9-c7ce4a7db5e3", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "avg-elt-per-sec", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d319f4e8-d31e-4a1b-9187-b8238bc57007", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "avg-throughput-per-sec", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a", "name": "transaction_sla_warning", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "errors-per-sec", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d319f4e8-d31e-4a1b-9187-b8238bc57007", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "avg-resp-time", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d319f4e8-d31e-4a1b-9187-b8238bc57007", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "error-rate", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "b24becfd-0be5-4e27-93c9-c7ce4a7db5e3", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "avg-throughput-per-sec", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "1034d624-451c-4523-b579-6f003ebd93f8", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "avg-elt-per-sec", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a", "name": "transaction_sla_warning", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "errors-per-sec", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a", "name": "transaction_sla_warning", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "avg-throughput-per-sec", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d319f4e8-d31e-4a1b-9187-b8238bc57007", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "avg-resp-time", "status": "WARNING", "warning": 100.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "1034d624-451c-4523-b579-6f003ebd93f8", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "avg-elt-per-sec", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [0.0]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [0.0]}, "element": {"elementId": "1ea4878e-6df1-4c5b-bd5d-3c1f0566c91d", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_passed"}}, {"kpi": "errors-per-sec", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [-0.1]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [-0.1]}, "element": {"elementId": "1df02e91-0b8c-4032-9d14-0853e9724294", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "avg-throughput-per-sec", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [0.0]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [0.0]}, "element": {"elementId": "1ea4878e-6df1-4c5b-bd5d-3c1f0566c91d", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_passed"}}, {"kpi": "avg-resp-time", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": ">=", "values": [3600.0]}, "failed": 0.0, "failedThreshold": {"operator": ">=", "values": [3601.0]}, "element": {"elementId": "b29523e3-6f58-40fc-98a4-8e24823d9390", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_passed"}}, {"kpi": "errors-per-sec", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [-0.1]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [-0.1]}, "element": {"elementId": "818d2419-daec-43d6-b004-b1195799df57", "name": "transaction_sla_passed", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "avg-resp-time", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": ">=", "values": [3600.0]}, "failed": 0.0, "failedThreshold": {"operator": ">=", "values": [3601.0]}, "element": {"elementId": "1df02e91-0b8c-4032-9d14-0853e9724294", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "error-rate", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [-0.1]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [-0.1]}, "element": {"elementId": "1df02e91-0b8c-4032-9d14-0853e9724294", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "errors-per-sec", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [-0.1]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [-0.1]}, "element": {"elementId": "b29523e3-6f58-40fc-98a4-8e24823d9390", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_passed"}}, {"kpi": "avg-throughput-per-sec", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [0.0]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [0.0]}, "element": {"elementId": "1df02e91-0b8c-4032-9d14-0853e9724294", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "avg-elt-per-sec", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [0.0]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [0.0]}, "element": {"elementId": "1df02e91-0b8c-4032-9d14-0853e9724294", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "errors-per-sec", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [-0.1]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [-0.1]}, "element": {"elementId": "1ea4878e-6df1-4c5b-bd5d-3c1f0566c91d", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_passed"}}, {"kpi": "avg-throughput-per-sec", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [0.0]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [0.0]}, "element": {"elementId": "818d2419-daec-43d6-b004-b1195799df57", "name": "transaction_sla_passed", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "avg-elt-per-sec", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [0.0]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [0.0]}, "element": {"elementId": "b29523e3-6f58-40fc-98a4-8e24823d9390", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_passed"}}, {"kpi": "avg-elt-per-sec", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [0.0]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [0.0]}, "element": {"elementId": "818d2419-daec-43d6-b004-b1195799df57", "name": "transaction_sla_passed", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "avg-throughput-per-sec", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [0.0]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [0.0]}, "element": {"elementId": "b29523e3-6f58-40fc-98a4-8e24823d9390", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_passed"}}, {"kpi": "avg-resp-time", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": ">=", "values": [3600.0]}, "failed": 0.0, "failedThreshold": {"operator": ">=", "values": [3601.0]}, "element": {"elementId": "1ea4878e-6df1-4c5b-bd5d-3c1f0566c91d", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_passed"}}, {"kpi": "error-rate", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [-0.1]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [-0.1]}, "element": {"elementId": "b29523e3-6f58-40fc-98a4-8e24823d9390", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_passed"}}, {"kpi": "error-rate", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [-0.1]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [-0.1]}, "element": {"elementId": "818d2419-daec-43d6-b004-b1195799df57", "name": "transaction_sla_passed", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "error-rate", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": "<=", "values": [-0.1]}, "failed": 0.0, "failedThreshold": {"operator": "<=", "values": [-0.1]}, "element": {"elementId": "1ea4878e-6df1-4c5b-bd5d-3c1f0566c91d", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_passed"}}, {"kpi": "avg-resp-time", "status": "PASSED", "warning": 0.0, "warningThreshold": {"operator": ">=", "values": [3600.0]}, "failed": 0.0, "failedThreshold": {"operator": ">=", "values": [3601.0]}, "element": {"elementId": "818d2419-daec-43d6-b004-b1195799df57", "name": "transaction_sla_passed", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/statistics':
            return '{"totalRequestCountSuccess": 14524, "totalRequestCountFailure": 14205, "totalRequestDurationAverage": 41.52066, "totalRequestCountPerSecond": 239.12138, "totalTransactionCountSuccess": 152, "totalTransactionCountFailure": 14205, "totalTransactionDurationAverage": 82.98767, "totalTransactionCountPerSecond": 119.49827, "totalIterationCountSuccess": 19, "totalIterationCountFailure": 3567, "totalGlobalDownloadedBytes": 41992679, "totalGlobalDownloadedBytesPerSecond": 349519.56, "totalGlobalCountFailure": 14205}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/events':
            return '[{"id": "14205", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119729, "type": "ERROR", "duration": 0, "source": "kamoulox#4 - Actions#357"}, {"id": "14204", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119726, "type": "ERROR", "duration": 0, "source": "kamoulox#3 - Actions#359"}, {"id": "14203", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119712, "type": "ERROR", "duration": 0, "source": "kamoulox#5 - Actions#358"}, {"id": "14202", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119711, "type": "ERROR", "duration": 0, "source": "kamoulox#9 - Actions#358"}, {"id": "14201", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119710, "type": "ERROR", "duration": 0, "source": "kamoulox#1 - Actions#361"}, {"id": "14200", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119682, "type": "ERROR", "duration": 0, "source": "kamoulox#0 - Actions#361"}, {"id": "14197", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119657, "type": "ERROR", "duration": 0, "source": "kamoulox#2 - Actions#358"}, {"id": "14198", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119657, "type": "ERROR", "duration": 0, "source": "kamoulox#7 - Actions#360"}, {"id": "14199", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119657, "type": "ERROR", "duration": 0, "source": "kamoulox#6 - Actions#359"}, {"id": "14196", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119633, "type": "ERROR", "duration": 0, "source": "kamoulox#8 - Actions#357"}, {"id": "14195", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119602, "type": "ERROR", "duration": 0, "source": "kamoulox#4 - Actions#357"}, {"id": "14194", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119589, "type": "ERROR", "duration": 0, "source": "kamoulox#3 - Actions#359"}, {"id": "14193", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119582, "type": "ERROR", "duration": 0, "source": "kamoulox#5 - Actions#358"}, {"id": "14192", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119568, "type": "ERROR", "duration": 0, "source": "kamoulox#1 - Actions#361"}, {"id": "14190", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119542, "type": "ERROR", "duration": 0, "source": "kamoulox#0 - Actions#361"}, {"id": "14191", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119542, "type": "ERROR", "duration": 0, "source": "kamoulox#6 - Actions#359"}, {"id": "14189", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119541, "type": "ERROR", "duration": 0, "source": "kamoulox#7 - Actions#360"}, {"id": "14188", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119537, "type": "ERROR", "duration": 0, "source": "kamoulox#9 - Actions#357"}, {"id": "14187", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119480, "type": "ERROR", "duration": 0, "source": "kamoulox#4 - Actions#357"}, {"id": "14186", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119455, "type": "ERROR", "duration": 0, "source": "kamoulox#2 - Actions#357"}, {"id": "14185", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119445, "type": "ERROR", "duration": 0, "source": "kamoulox#5 - Actions#358"}, {"id": "14184", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119435, "type": "ERROR", "duration": 0, "source": "kamoulox#6 - Actions#359"}, {"id": "14183", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119433, "type": "ERROR", "duration": 0, "source": "kamoulox#0 - Actions#361"}, {"id": "14182", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119424, "type": "ERROR", "duration": 0, "source": "kamoulox#7 - Actions#360"}, {"id": "14181", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119422, "type": "ERROR", "duration": 0, "source": "kamoulox#3 - Actions#358"}, {"id": "14180", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119413, "type": "ERROR", "duration": 0, "source": "kamoulox#9 - Actions#357"}, {"id": "14179", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119391, "type": "ERROR", "duration": 0, "source": "kamoulox#8 - Actions#357"}, {"id": "14178", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119376, "type": "ERROR", "duration": 0, "source": "kamoulox#1 - Actions#360"}, {"id": "14177", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119366, "type": "ERROR", "duration": 0, "source": "kamoulox#4 - Actions#357"}, {"id": "14176", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119329, "type": "ERROR", "duration": 0, "source": "kamoulox#2 - Actions#357"}, {"id": "14175", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119318, "type": "ERROR", "duration": 0, "source": "kamoulox#5 - Actions#358"}, {"id": "14174", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119314, "type": "ERROR", "duration": 0, "source": "kamoulox#6 - Actions#359"}, {"id": "14172", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119312, "type": "ERROR", "duration": 0, "source": "kamoulox#7 - Actions#360"}, {"id": "14173", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119312, "type": "ERROR", "duration": 0, "source": "kamoulox#3 - Actions#358"}, {"id": "14171", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119280, "type": "ERROR", "duration": 0, "source": "kamoulox#9 - Actions#357"}, {"id": "14170", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119253, "type": "ERROR", "duration": 0, "source": "kamoulox#1 - Actions#360"}, {"id": "14169", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119248, "type": "ERROR", "duration": 0, "source": "kamoulox#0 - Actions#360"}, {"id": "14168", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119184, "type": "ERROR", "duration": 0, "source": "kamoulox#2 - Actions#357"}, {"id": "14167", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119157, "type": "ERROR", "duration": 0, "source": "kamoulox#3 - Actions#358"}, {"id": "14166", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119155, "type": "ERROR", "duration": 0, "source": "kamoulox#8 - Actions#356"}, {"id": "14165", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119144, "type": "ERROR", "duration": 0, "source": "kamoulox#4 - Actions#356"}, {"id": "14164", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119122, "type": "ERROR", "duration": 0, "source": "kamoulox#1 - Actions#360"}, {"id": "14163", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119121, "type": "ERROR", "duration": 0, "source": "kamoulox#5 - Actions#357"}, {"id": "14162", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119119, "type": "ERROR", "duration": 0, "source": "kamoulox#9 - Actions#357"}, {"id": "14161", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119101, "type": "ERROR", "duration": 0, "source": "kamoulox#0 - Actions#360"}, {"id": "14160", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119088, "type": "ERROR", "duration": 0, "source": "kamoulox#7 - Actions#359"}, {"id": "14159", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119044, "type": "ERROR", "duration": 0, "source": "kamoulox#2 - Actions#357"}, {"id": "14157", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119034, "type": "ERROR", "duration": 0, "source": "kamoulox#8 - Actions#356"}, {"id": "14158", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119034, "type": "ERROR", "duration": 0, "source": "kamoulox#3 - Actions#358"}, {"id": "14155", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 119001, "type": "ERROR", "duration": 0, "source": "kamoulox#1 - Actions#360"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements?category=REQUEST':
            return '[{"id": "all-requests", "name": "<all requests>", "type": "ALL_REQUESTS"}, {"id": "54d47517-89bb-4e58-b22c-4fb9ac78000b", "name": "/", "path": ["kamoulox", "Actions", "transaction_no_sla", "page", "/"], "type": "REQUEST"}, {"id": "1a080b9f-05dc-4424-b575-866dedb2acb0", "name": "/api/webhooks/last", "path": ["kamoulox", "Actions", "transaction_no_sla", "/api/webhooks/last"], "type": "REQUEST"}, {"id": "d319f4e8-d31e-4a1b-9187-b8238bc57007", "name": "/", "path": ["kamoulox", "Actions", "transaction_sla_warning", "page", "/"], "type": "REQUEST"}, {"id": "1034d624-451c-4523-b579-6f003ebd93f8", "name": "/api/webhooks/last", "path": ["kamoulox", "Actions", "transaction_sla_warning", "/api/webhooks/last"], "type": "REQUEST"}, {"id": "7bf42114-d630-494f-bc13-fea402ce085d", "name": "/", "path": ["kamoulox", "Actions", "transaction_sla_failed", "page", "/"], "type": "REQUEST"}, {"id": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "name": "/api/webhooks/last", "path": ["kamoulox", "Actions", "transaction_sla_failed", "/api/webhooks/last"], "type": "REQUEST"}, {"id": "1df02e91-0b8c-4032-9d14-0853e9724294", "name": "/", "path": ["kamoulox", "Actions", "transaction_sla_passed", "page", "/"], "type": "REQUEST"}, {"id": "b29523e3-6f58-40fc-98a4-8e24823d9390", "name": "/api/webhooks/last", "path": ["kamoulox", "Actions", "transaction_sla_passed", "/api/webhooks/last"], "type": "REQUEST"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements/all-requests/values':
            return '{"count": 28729, "elementPerSecond": 239.12138, "minDuration": 28, "maxDuration": 384, "sumDuration": 1192847, "avgDuration": 41.52066, "minTTFB": 27, "maxTTFB": 384, "sumTTFB": 1162004, "avgTTFB": 40.447075, "sumDownloadedBytes": 41992679, "downloadedBytesPerSecond": 349519.56, "successCount": 14524, "successPerSecond": 120.88827, "successRate": 50.555187, "failureCount": 14205, "failurePerSecond": 118.23312, "failureRate": 49.444813}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements/all-requests/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 1000, "AVG_TTFB": 49.123077, "ERRORS": 0, "AVG_DURATION": 50.00769, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 30, "ERROR_RATE": 0.0, "MIN_TTFB": 29, "MAX_TTFB": 162, "MAX_DURATION": 163, "ELEMENTS_PER_SECOND": 130.0, "THROUGHPUT": 103242.0, "COUNT": 130}, {"from": 1000, "to": 2000, "AVG_TTFB": 37.57143, "ERRORS": 121, "AVG_DURATION": 38.865078, "ERRORS_PER_SECOND": 121.0, "MIN_DURATION": 29, "ERROR_RATE": 48.015873, "MIN_TTFB": 28, "MAX_TTFB": 70, "MAX_DURATION": 72, "ELEMENTS_PER_SECOND": 252.0, "THROUGHPUT": 361723.0, "COUNT": 252}, {"from": 2000, "to": 3000, "AVG_TTFB": 37.992, "ERRORS": 124, "AVG_DURATION": 39.272, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 29, "ERROR_RATE": 49.6, "MIN_TTFB": 28, "MAX_TTFB": 87, "MAX_DURATION": 88, "ELEMENTS_PER_SECOND": 250.0, "THROUGHPUT": 366835.0, "COUNT": 250}, {"from": 3000, "to": 4000, "AVG_TTFB": 37.583332, "ERRORS": 127, "AVG_DURATION": 38.79365, "ERRORS_PER_SECOND": 127.0, "MIN_DURATION": 30, "ERROR_RATE": 50.396824, "MIN_TTFB": 28, "MAX_TTFB": 74, "MAX_DURATION": 75, "ELEMENTS_PER_SECOND": 252.0, "THROUGHPUT": 370893.0, "COUNT": 252}, {"from": 4000, "to": 5000, "AVG_TTFB": 36.534092, "ERRORS": 133, "AVG_DURATION": 37.651516, "ERRORS_PER_SECOND": 133.0, "MIN_DURATION": 29, "ERROR_RATE": 50.378788, "MIN_TTFB": 29, "MAX_TTFB": 71, "MAX_DURATION": 71, "ELEMENTS_PER_SECOND": 264.0, "THROUGHPUT": 388507.0, "COUNT": 264}, {"from": 5000, "to": 6000, "AVG_TTFB": 39.210526, "ERRORS": 122, "AVG_DURATION": 40.255062, "ERRORS_PER_SECOND": 122.0, "MIN_DURATION": 29, "ERROR_RATE": 49.39271, "MIN_TTFB": 28, "MAX_TTFB": 96, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 247.0, "THROUGHPUT": 362046.0, "COUNT": 247}, {"from": 6000, "to": 7000, "AVG_TTFB": 36.844105, "ERRORS": 132, "AVG_DURATION": 37.76426, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 29, "ERROR_RATE": 50.190113, "MIN_TTFB": 28, "MAX_TTFB": 76, "MAX_DURATION": 76, "ELEMENTS_PER_SECOND": 263.0, "THROUGHPUT": 386815.0, "COUNT": 263}, {"from": 7000, "to": 8000, "AVG_TTFB": 36.877396, "ERRORS": 131, "AVG_DURATION": 37.927204, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 29, "ERROR_RATE": 50.19157, "MIN_TTFB": 28, "MAX_TTFB": 69, "MAX_DURATION": 69, "ELEMENTS_PER_SECOND": 261.0, "THROUGHPUT": 383771.0, "COUNT": 261}, {"from": 8000, "to": 9000, "AVG_TTFB": 36.825756, "ERRORS": 133, "AVG_DURATION": 37.81818, "ERRORS_PER_SECOND": 133.0, "MIN_DURATION": 29, "ERROR_RATE": 50.378788, "MIN_TTFB": 28, "MAX_TTFB": 74, "MAX_DURATION": 74, "ELEMENTS_PER_SECOND": 264.0, "THROUGHPUT": 388488.0, "COUNT": 264}, {"from": 9000, "to": 10000, "AVG_TTFB": 36.575756, "ERRORS": 130, "AVG_DURATION": 37.617424, "ERRORS_PER_SECOND": 130.0, "MIN_DURATION": 28, "ERROR_RATE": 49.242424, "MIN_TTFB": 28, "MAX_TTFB": 83, "MAX_DURATION": 83, "ELEMENTS_PER_SECOND": 264.0, "THROUGHPUT": 386723.0, "COUNT": 264}, {"from": 10000, "to": 11000, "AVG_TTFB": 36.711025, "ERRORS": 133, "AVG_DURATION": 37.593155, "ERRORS_PER_SECOND": 133.0, "MIN_DURATION": 29, "ERROR_RATE": 50.570343, "MIN_TTFB": 28, "MAX_TTFB": 82, "MAX_DURATION": 82, "ELEMENTS_PER_SECOND": 263.0, "THROUGHPUT": 387398.0, "COUNT": 263}, {"from": 11000, "to": 12000, "AVG_TTFB": 38.343872, "ERRORS": 126, "AVG_DURATION": 39.44664, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 29, "ERROR_RATE": 49.802372, "MIN_TTFB": 28, "MAX_TTFB": 101, "MAX_DURATION": 101, "ELEMENTS_PER_SECOND": 253.0, "THROUGHPUT": 371475.0, "COUNT": 253}, {"from": 12000, "to": 13000, "AVG_TTFB": 38.458168, "ERRORS": 124, "AVG_DURATION": 39.458168, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 29, "ERROR_RATE": 49.40239, "MIN_TTFB": 28, "MAX_TTFB": 113, "MAX_DURATION": 114, "ELEMENTS_PER_SECOND": 251.0, "THROUGHPUT": 367923.0, "COUNT": 251}, {"from": 13000, "to": 14000, "AVG_TTFB": 36.988506, "ERRORS": 132, "AVG_DURATION": 38.057472, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 29, "ERROR_RATE": 50.57471, "MIN_TTFB": 28, "MAX_TTFB": 88, "MAX_DURATION": 88, "ELEMENTS_PER_SECOND": 261.0, "THROUGHPUT": 384360.0, "COUNT": 261}, {"from": 14000, "to": 15000, "AVG_TTFB": 36.46212, "ERRORS": 132, "AVG_DURATION": 37.526516, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 28, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 78, "MAX_DURATION": 78, "ELEMENTS_PER_SECOND": 264.0, "THROUGHPUT": 387936.0, "COUNT": 264}, {"from": 15000, "to": 16000, "AVG_TTFB": 36.18352, "ERRORS": 134, "AVG_DURATION": 37.0412, "ERRORS_PER_SECOND": 134.0, "MIN_DURATION": 28, "ERROR_RATE": 50.187267, "MIN_TTFB": 27, "MAX_TTFB": 87, "MAX_DURATION": 88, "ELEMENTS_PER_SECOND": 267.0, "THROUGHPUT": 392689.0, "COUNT": 267}, {"from": 16000, "to": 17000, "AVG_TTFB": 61.607594, "ERRORS": 78, "AVG_DURATION": 63.77848, "ERRORS_PER_SECOND": 78.0, "MIN_DURATION": 29, "ERROR_RATE": 49.36709, "MIN_TTFB": 28, "MAX_TTFB": 305, "MAX_DURATION": 306, "ELEMENTS_PER_SECOND": 158.0, "THROUGHPUT": 231569.0, "COUNT": 158}, {"from": 17000, "to": 18000, "AVG_TTFB": 38.641434, "ERRORS": 124, "AVG_DURATION": 39.581673, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 29, "ERROR_RATE": 49.40239, "MIN_TTFB": 28, "MAX_TTFB": 95, "MAX_DURATION": 95, "ELEMENTS_PER_SECOND": 251.0, "THROUGHPUT": 367954.0, "COUNT": 251}, {"from": 18000, "to": 19000, "AVG_TTFB": 37.213177, "ERRORS": 130, "AVG_DURATION": 38.2093, "ERRORS_PER_SECOND": 130.0, "MIN_DURATION": 28, "ERROR_RATE": 50.387596, "MIN_TTFB": 27, "MAX_TTFB": 77, "MAX_DURATION": 78, "ELEMENTS_PER_SECOND": 258.0, "THROUGHPUT": 379728.0, "COUNT": 258}, {"from": 19000, "to": 20000, "AVG_TTFB": 38.812, "ERRORS": 125, "AVG_DURATION": 39.776, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 28, "ERROR_RATE": 50.0, "MIN_TTFB": 27, "MAX_TTFB": 132, "MAX_DURATION": 132, "ELEMENTS_PER_SECOND": 250.0, "THROUGHPUT": 367338.0, "COUNT": 250}, {"from": 20000, "to": 21000, "AVG_TTFB": 39.04918, "ERRORS": 122, "AVG_DURATION": 40.045082, "ERRORS_PER_SECOND": 122.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 106, "MAX_DURATION": 106, "ELEMENTS_PER_SECOND": 244.0, "THROUGHPUT": 358563.0, "COUNT": 244}, {"from": 21000, "to": 22000, "AVG_TTFB": 46.050926, "ERRORS": 108, "AVG_DURATION": 46.976852, "ERRORS_PER_SECOND": 108.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 268, "MAX_DURATION": 268, "ELEMENTS_PER_SECOND": 216.0, "THROUGHPUT": 317411.0, "COUNT": 216}, {"from": 22000, "to": 23000, "AVG_TTFB": 36.825756, "ERRORS": 133, "AVG_DURATION": 37.825756, "ERRORS_PER_SECOND": 133.0, "MIN_DURATION": 29, "ERROR_RATE": 50.378788, "MIN_TTFB": 28, "MAX_TTFB": 81, "MAX_DURATION": 81, "ELEMENTS_PER_SECOND": 264.0, "THROUGHPUT": 388468.0, "COUNT": 264}, {"from": 23000, "to": 24000, "AVG_TTFB": 37.49807, "ERRORS": 129, "AVG_DURATION": 38.57529, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 28, "ERROR_RATE": 49.80695, "MIN_TTFB": 27, "MAX_TTFB": 93, "MAX_DURATION": 93, "ELEMENTS_PER_SECOND": 259.0, "THROUGHPUT": 380321.0, "COUNT": 259}, {"from": 24000, "to": 25000, "AVG_TTFB": 36.320755, "ERRORS": 132, "AVG_DURATION": 37.38113, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 28, "ERROR_RATE": 49.81132, "MIN_TTFB": 28, "MAX_TTFB": 82, "MAX_DURATION": 82, "ELEMENTS_PER_SECOND": 265.0, "THROUGHPUT": 389180.0, "COUNT": 265}, {"from": 25000, "to": 26000, "AVG_TTFB": 36.773075, "ERRORS": 129, "AVG_DURATION": 37.673077, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 28, "ERROR_RATE": 49.615383, "MIN_TTFB": 28, "MAX_TTFB": 113, "MAX_DURATION": 113, "ELEMENTS_PER_SECOND": 260.0, "THROUGHPUT": 381419.0, "COUNT": 260}, {"from": 26000, "to": 27000, "AVG_TTFB": 40.48117, "ERRORS": 120, "AVG_DURATION": 41.627617, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 29, "ERROR_RATE": 50.209206, "MIN_TTFB": 28, "MAX_TTFB": 132, "MAX_DURATION": 132, "ELEMENTS_PER_SECOND": 239.0, "THROUGHPUT": 351438.0, "COUNT": 239}, {"from": 27000, "to": 28000, "AVG_TTFB": 38.265385, "ERRORS": 130, "AVG_DURATION": 39.16923, "ERRORS_PER_SECOND": 130.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 106, "MAX_DURATION": 106, "ELEMENTS_PER_SECOND": 260.0, "THROUGHPUT": 381974.0, "COUNT": 260}, {"from": 28000, "to": 29000, "AVG_TTFB": 37.558594, "ERRORS": 129, "AVG_DURATION": 38.710938, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 29, "ERROR_RATE": 50.390625, "MIN_TTFB": 27, "MAX_TTFB": 96, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 256.0, "THROUGHPUT": 376787.0, "COUNT": 256}, {"from": 29000, "to": 30000, "AVG_TTFB": 48.039024, "ERRORS": 102, "AVG_DURATION": 49.039024, "ERRORS_PER_SECOND": 102.0, "MIN_DURATION": 29, "ERROR_RATE": 49.756096, "MIN_TTFB": 28, "MAX_TTFB": 252, "MAX_DURATION": 253, "ELEMENTS_PER_SECOND": 205.0, "THROUGHPUT": 300949.0, "COUNT": 205}, {"from": 30000, "to": 31000, "AVG_TTFB": 40.331932, "ERRORS": 120, "AVG_DURATION": 41.361343, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 29, "ERROR_RATE": 50.42017, "MIN_TTFB": 28, "MAX_TTFB": 120, "MAX_DURATION": 120, "ELEMENTS_PER_SECOND": 238.0, "THROUGHPUT": 350359.0, "COUNT": 238}, {"from": 31000, "to": 32000, "AVG_TTFB": 36.966164, "ERRORS": 132, "AVG_DURATION": 37.93233, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 29, "ERROR_RATE": 49.62406, "MIN_TTFB": 28, "MAX_TTFB": 87, "MAX_DURATION": 87, "ELEMENTS_PER_SECOND": 266.0, "THROUGHPUT": 390297.0, "COUNT": 266}, {"from": 32000, "to": 33000, "AVG_TTFB": 37.233715, "ERRORS": 132, "AVG_DURATION": 38.24138, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 29, "ERROR_RATE": 50.57471, "MIN_TTFB": 28, "MAX_TTFB": 109, "MAX_DURATION": 110, "ELEMENTS_PER_SECOND": 261.0, "THROUGHPUT": 384401.0, "COUNT": 261}, {"from": 33000, "to": 34000, "AVG_TTFB": 36.61597, "ERRORS": 131, "AVG_DURATION": 37.692017, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 29, "ERROR_RATE": 49.809887, "MIN_TTFB": 29, "MAX_TTFB": 72, "MAX_DURATION": 73, "ELEMENTS_PER_SECOND": 263.0, "THROUGHPUT": 386211.0, "COUNT": 263}, {"from": 34000, "to": 35000, "AVG_TTFB": 38.88, "ERRORS": 124, "AVG_DURATION": 39.856, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 29, "ERROR_RATE": 49.6, "MIN_TTFB": 28, "MAX_TTFB": 98, "MAX_DURATION": 98, "ELEMENTS_PER_SECOND": 250.0, "THROUGHPUT": 366851.0, "COUNT": 250}, {"from": 35000, "to": 36000, "AVG_TTFB": 40.728035, "ERRORS": 121, "AVG_DURATION": 41.761505, "ERRORS_PER_SECOND": 121.0, "MIN_DURATION": 29, "ERROR_RATE": 50.627617, "MIN_TTFB": 28, "MAX_TTFB": 136, "MAX_DURATION": 136, "ELEMENTS_PER_SECOND": 239.0, "THROUGHPUT": 352155.0, "COUNT": 239}, {"from": 36000, "to": 37000, "AVG_TTFB": 37.949417, "ERRORS": 127, "AVG_DURATION": 38.984436, "ERRORS_PER_SECOND": 127.0, "MIN_DURATION": 29, "ERROR_RATE": 49.416344, "MIN_TTFB": 28, "MAX_TTFB": 102, "MAX_DURATION": 102, "ELEMENTS_PER_SECOND": 257.0, "THROUGHPUT": 376744.0, "COUNT": 257}, {"from": 37000, "to": 38000, "AVG_TTFB": 36.643677, "ERRORS": 132, "AVG_DURATION": 37.86207, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 29, "ERROR_RATE": 50.57471, "MIN_TTFB": 28, "MAX_TTFB": 80, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 261.0, "THROUGHPUT": 384439.0, "COUNT": 261}, {"from": 38000, "to": 39000, "AVG_TTFB": 37.184616, "ERRORS": 129, "AVG_DURATION": 38.134617, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 28, "ERROR_RATE": 49.615383, "MIN_TTFB": 28, "MAX_TTFB": 85, "MAX_DURATION": 85, "ELEMENTS_PER_SECOND": 260.0, "THROUGHPUT": 381376.0, "COUNT": 260}, {"from": 39000, "to": 40000, "AVG_TTFB": 36.890152, "ERRORS": 131, "AVG_DURATION": 37.810608, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 28, "ERROR_RATE": 49.621212, "MIN_TTFB": 28, "MAX_TTFB": 91, "MAX_DURATION": 91, "ELEMENTS_PER_SECOND": 264.0, "THROUGHPUT": 387392.0, "COUNT": 264}, {"from": 40000, "to": 41000, "AVG_TTFB": 39.262096, "ERRORS": 125, "AVG_DURATION": 40.504032, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 29, "ERROR_RATE": 50.403225, "MIN_TTFB": 28, "MAX_TTFB": 115, "MAX_DURATION": 115, "ELEMENTS_PER_SECOND": 248.0, "THROUGHPUT": 365005.0, "COUNT": 248}, {"from": 41000, "to": 42000, "AVG_TTFB": 36.825096, "ERRORS": 133, "AVG_DURATION": 37.760456, "ERRORS_PER_SECOND": 133.0, "MIN_DURATION": 28, "ERROR_RATE": 50.570343, "MIN_TTFB": 28, "MAX_TTFB": 114, "MAX_DURATION": 114, "ELEMENTS_PER_SECOND": 263.0, "THROUGHPUT": 387365.0, "COUNT": 263}, {"from": 42000, "to": 43000, "AVG_TTFB": 36.829544, "ERRORS": 130, "AVG_DURATION": 37.803032, "ERRORS_PER_SECOND": 130.0, "MIN_DURATION": 28, "ERROR_RATE": 49.242424, "MIN_TTFB": 28, "MAX_TTFB": 119, "MAX_DURATION": 119, "ELEMENTS_PER_SECOND": 264.0, "THROUGHPUT": 386662.0, "COUNT": 264}, {"from": 43000, "to": 44000, "AVG_TTFB": 37.73541, "ERRORS": 130, "AVG_DURATION": 38.7821, "ERRORS_PER_SECOND": 130.0, "MIN_DURATION": 29, "ERROR_RATE": 50.583656, "MIN_TTFB": 28, "MAX_TTFB": 113, "MAX_DURATION": 113, "ELEMENTS_PER_SECOND": 257.0, "THROUGHPUT": 378524.0, "COUNT": 257}, {"from": 44000, "to": 45000, "AVG_TTFB": 38.575397, "ERRORS": 125, "AVG_DURATION": 39.511906, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 29, "ERROR_RATE": 49.603176, "MIN_TTFB": 28, "MAX_TTFB": 109, "MAX_DURATION": 109, "ELEMENTS_PER_SECOND": 252.0, "THROUGHPUT": 369700.0, "COUNT": 252}, {"from": 45000, "to": 46000, "AVG_TTFB": 37.02281, "ERRORS": 132, "AVG_DURATION": 38.0, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 29, "ERROR_RATE": 50.190113, "MIN_TTFB": 28, "MAX_TTFB": 115, "MAX_DURATION": 115, "ELEMENTS_PER_SECOND": 263.0, "THROUGHPUT": 386824.0, "COUNT": 263}, {"from": 46000, "to": 47000, "AVG_TTFB": 36.805344, "ERRORS": 131, "AVG_DURATION": 37.74046, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 112, "MAX_DURATION": 112, "ELEMENTS_PER_SECOND": 262.0, "THROUGHPUT": 384984.0, "COUNT": 262}, {"from": 47000, "to": 48000, "AVG_TTFB": 39.32922, "ERRORS": 121, "AVG_DURATION": 40.452675, "ERRORS_PER_SECOND": 121.0, "MIN_DURATION": 30, "ERROR_RATE": 49.79424, "MIN_TTFB": 29, "MAX_TTFB": 85, "MAX_DURATION": 85, "ELEMENTS_PER_SECOND": 243.0, "THROUGHPUT": 356790.0, "COUNT": 243}, {"from": 48000, "to": 49000, "AVG_TTFB": 38.22353, "ERRORS": 128, "AVG_DURATION": 39.388237, "ERRORS_PER_SECOND": 128.0, "MIN_DURATION": 29, "ERROR_RATE": 50.19608, "MIN_TTFB": 28, "MAX_TTFB": 99, "MAX_DURATION": 99, "ELEMENTS_PER_SECOND": 255.0, "THROUGHPUT": 375072.0, "COUNT": 255}, {"from": 49000, "to": 50000, "AVG_TTFB": 38.19922, "ERRORS": 128, "AVG_DURATION": 39.257812, "ERRORS_PER_SECOND": 128.0, "MIN_DURATION": 28, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 103, "MAX_DURATION": 103, "ELEMENTS_PER_SECOND": 256.0, "THROUGHPUT": 376180.0, "COUNT": 256}, {"from": 50000, "to": 51000, "AVG_TTFB": 41.008404, "ERRORS": 119, "AVG_DURATION": 41.92857, "ERRORS_PER_SECOND": 119.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 110, "MAX_DURATION": 110, "ELEMENTS_PER_SECOND": 238.0, "THROUGHPUT": 349673.0, "COUNT": 238}, {"from": 51000, "to": 52000, "AVG_TTFB": 38.503967, "ERRORS": 126, "AVG_DURATION": 39.535713, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 127, "MAX_DURATION": 127, "ELEMENTS_PER_SECOND": 252.0, "THROUGHPUT": 370253.0, "COUNT": 252}, {"from": 52000, "to": 53000, "AVG_TTFB": 44.678898, "ERRORS": 108, "AVG_DURATION": 45.766056, "ERRORS_PER_SECOND": 108.0, "MIN_DURATION": 30, "ERROR_RATE": 49.541283, "MIN_TTFB": 28, "MAX_TTFB": 208, "MAX_DURATION": 208, "ELEMENTS_PER_SECOND": 218.0, "THROUGHPUT": 319614.0, "COUNT": 218}, {"from": 53000, "to": 54000, "AVG_TTFB": 38.93651, "ERRORS": 127, "AVG_DURATION": 39.896824, "ERRORS_PER_SECOND": 127.0, "MIN_DURATION": 29, "ERROR_RATE": 50.396824, "MIN_TTFB": 28, "MAX_TTFB": 128, "MAX_DURATION": 128, "ELEMENTS_PER_SECOND": 252.0, "THROUGHPUT": 370770.0, "COUNT": 252}, {"from": 54000, "to": 55000, "AVG_TTFB": 38.50199, "ERRORS": 124, "AVG_DURATION": 39.55777, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 30, "ERROR_RATE": 49.40239, "MIN_TTFB": 29, "MAX_TTFB": 93, "MAX_DURATION": 93, "ELEMENTS_PER_SECOND": 251.0, "THROUGHPUT": 367767.0, "COUNT": 251}, {"from": 55000, "to": 56000, "AVG_TTFB": 37.960785, "ERRORS": 128, "AVG_DURATION": 38.960785, "ERRORS_PER_SECOND": 128.0, "MIN_DURATION": 29, "ERROR_RATE": 50.19608, "MIN_TTFB": 29, "MAX_TTFB": 126, "MAX_DURATION": 126, "ELEMENTS_PER_SECOND": 255.0, "THROUGHPUT": 374814.0, "COUNT": 255}, {"from": 56000, "to": 57000, "AVG_TTFB": 38.279526, "ERRORS": 128, "AVG_DURATION": 39.212597, "ERRORS_PER_SECOND": 128.0, "MIN_DURATION": 29, "ERROR_RATE": 50.3937, "MIN_TTFB": 28, "MAX_TTFB": 119, "MAX_DURATION": 119, "ELEMENTS_PER_SECOND": 254.0, "THROUGHPUT": 373711.0, "COUNT": 254}, {"from": 57000, "to": 58000, "AVG_TTFB": 38.868526, "ERRORS": 124, "AVG_DURATION": 39.976097, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 29, "ERROR_RATE": 49.40239, "MIN_TTFB": 28, "MAX_TTFB": 81, "MAX_DURATION": 81, "ELEMENTS_PER_SECOND": 251.0, "THROUGHPUT": 367820.0, "COUNT": 251}, {"from": 58000, "to": 59000, "AVG_TTFB": 40.949154, "ERRORS": 117, "AVG_DURATION": 42.20339, "ERRORS_PER_SECOND": 117.0, "MIN_DURATION": 29, "ERROR_RATE": 49.57627, "MIN_TTFB": 29, "MAX_TTFB": 127, "MAX_DURATION": 127, "ELEMENTS_PER_SECOND": 236.0, "THROUGHPUT": 346027.0, "COUNT": 236}, {"from": 59000, "to": 60000, "AVG_TTFB": 42.048035, "ERRORS": 89, "AVG_DURATION": 42.960697, "ERRORS_PER_SECOND": 89.0, "MIN_DURATION": 30, "ERROR_RATE": 38.864628, "MIN_TTFB": 28, "MAX_TTFB": 111, "MAX_DURATION": 111, "ELEMENTS_PER_SECOND": 229.0, "THROUGHPUT": 299468.0, "COUNT": 229}, {"from": 60000, "to": 61000, "AVG_TTFB": 39.91129, "ERRORS": 85, "AVG_DURATION": 40.572582, "ERRORS_PER_SECOND": 85.0, "MIN_DURATION": 30, "ERROR_RATE": 34.274193, "MIN_TTFB": 29, "MAX_TTFB": 111, "MAX_DURATION": 111, "ELEMENTS_PER_SECOND": 248.0, "THROUGHPUT": 309823.0, "COUNT": 248}, {"from": 61000, "to": 62000, "AVG_TTFB": 38.258965, "ERRORS": 126, "AVG_DURATION": 39.474102, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 30, "ERROR_RATE": 50.199203, "MIN_TTFB": 29, "MAX_TTFB": 126, "MAX_DURATION": 126, "ELEMENTS_PER_SECOND": 251.0, "THROUGHPUT": 369152.0, "COUNT": 251}, {"from": 62000, "to": 63000, "AVG_TTFB": 39.57377, "ERRORS": 121, "AVG_DURATION": 40.692623, "ERRORS_PER_SECOND": 121.0, "MIN_DURATION": 30, "ERROR_RATE": 49.590164, "MIN_TTFB": 28, "MAX_TTFB": 135, "MAX_DURATION": 135, "ELEMENTS_PER_SECOND": 244.0, "THROUGHPUT": 357936.0, "COUNT": 244}, {"from": 63000, "to": 64000, "AVG_TTFB": 37.15649, "ERRORS": 131, "AVG_DURATION": 38.160305, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 29, "MAX_TTFB": 95, "MAX_DURATION": 95, "ELEMENTS_PER_SECOND": 262.0, "THROUGHPUT": 385031.0, "COUNT": 262}, {"from": 64000, "to": 65000, "AVG_TTFB": 37.303844, "ERRORS": 132, "AVG_DURATION": 38.326923, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 28, "ERROR_RATE": 50.76923, "MIN_TTFB": 28, "MAX_TTFB": 136, "MAX_DURATION": 136, "ELEMENTS_PER_SECOND": 260.0, "THROUGHPUT": 383332.0, "COUNT": 260}, {"from": 65000, "to": 66000, "AVG_TTFB": 41.05485, "ERRORS": 117, "AVG_DURATION": 42.13502, "ERRORS_PER_SECOND": 117.0, "MIN_DURATION": 29, "ERROR_RATE": 49.36709, "MIN_TTFB": 28, "MAX_TTFB": 140, "MAX_DURATION": 140, "ELEMENTS_PER_SECOND": 237.0, "THROUGHPUT": 347374.0, "COUNT": 237}, {"from": 66000, "to": 67000, "AVG_TTFB": 39.133064, "ERRORS": 125, "AVG_DURATION": 40.258064, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 29, "ERROR_RATE": 50.403225, "MIN_TTFB": 29, "MAX_TTFB": 114, "MAX_DURATION": 115, "ELEMENTS_PER_SECOND": 248.0, "THROUGHPUT": 365021.0, "COUNT": 248}, {"from": 67000, "to": 68000, "AVG_TTFB": 39.13008, "ERRORS": 122, "AVG_DURATION": 40.1626, "ERRORS_PER_SECOND": 122.0, "MIN_DURATION": 30, "ERROR_RATE": 49.593494, "MIN_TTFB": 29, "MAX_TTFB": 117, "MAX_DURATION": 118, "ELEMENTS_PER_SECOND": 246.0, "THROUGHPUT": 360822.0, "COUNT": 246}, {"from": 68000, "to": 69000, "AVG_TTFB": 39.107143, "ERRORS": 126, "AVG_DURATION": 39.98016, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 94, "MAX_DURATION": 94, "ELEMENTS_PER_SECOND": 252.0, "THROUGHPUT": 370279.0, "COUNT": 252}, {"from": 69000, "to": 70000, "AVG_TTFB": 39.737705, "ERRORS": 123, "AVG_DURATION": 40.840164, "ERRORS_PER_SECOND": 123.0, "MIN_DURATION": 29, "ERROR_RATE": 50.409836, "MIN_TTFB": 28, "MAX_TTFB": 147, "MAX_DURATION": 147, "ELEMENTS_PER_SECOND": 244.0, "THROUGHPUT": 359140.0, "COUNT": 244}, {"from": 70000, "to": 71000, "AVG_TTFB": 39.958847, "ERRORS": 120, "AVG_DURATION": 40.90535, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 29, "ERROR_RATE": 49.382717, "MIN_TTFB": 28, "MAX_TTFB": 104, "MAX_DURATION": 104, "ELEMENTS_PER_SECOND": 243.0, "THROUGHPUT": 356138.0, "COUNT": 243}, {"from": 71000, "to": 72000, "AVG_TTFB": 38.304688, "ERRORS": 128, "AVG_DURATION": 39.32422, "ERRORS_PER_SECOND": 128.0, "MIN_DURATION": 30, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 133, "MAX_DURATION": 134, "ELEMENTS_PER_SECOND": 256.0, "THROUGHPUT": 376094.0, "COUNT": 256}, {"from": 72000, "to": 73000, "AVG_TTFB": 40.92827, "ERRORS": 120, "AVG_DURATION": 41.881855, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 30, "ERROR_RATE": 50.63291, "MIN_TTFB": 28, "MAX_TTFB": 147, "MAX_DURATION": 147, "ELEMENTS_PER_SECOND": 237.0, "THROUGHPUT": 349229.0, "COUNT": 237}, {"from": 73000, "to": 74000, "AVG_TTFB": 39.385246, "ERRORS": 121, "AVG_DURATION": 40.331966, "ERRORS_PER_SECOND": 121.0, "MIN_DURATION": 30, "ERROR_RATE": 49.590164, "MIN_TTFB": 29, "MAX_TTFB": 134, "MAX_DURATION": 134, "ELEMENTS_PER_SECOND": 244.0, "THROUGHPUT": 357980.0, "COUNT": 244}, {"from": 74000, "to": 75000, "AVG_TTFB": 37.389313, "ERRORS": 131, "AVG_DURATION": 38.408398, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 122, "MAX_DURATION": 122, "ELEMENTS_PER_SECOND": 262.0, "THROUGHPUT": 385052.0, "COUNT": 262}, {"from": 75000, "to": 76000, "AVG_TTFB": 36.91571, "ERRORS": 131, "AVG_DURATION": 37.89272, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 29, "ERROR_RATE": 50.19157, "MIN_TTFB": 28, "MAX_TTFB": 117, "MAX_DURATION": 117, "ELEMENTS_PER_SECOND": 261.0, "THROUGHPUT": 383894.0, "COUNT": 261}, {"from": 76000, "to": 77000, "AVG_TTFB": 38.888, "ERRORS": 125, "AVG_DURATION": 40.084, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 29, "MAX_TTFB": 148, "MAX_DURATION": 148, "ELEMENTS_PER_SECOND": 250.0, "THROUGHPUT": 367346.0, "COUNT": 250}, {"from": 77000, "to": 78000, "AVG_TTFB": 38.374016, "ERRORS": 125, "AVG_DURATION": 39.377953, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 29, "ERROR_RATE": 49.212597, "MIN_TTFB": 28, "MAX_TTFB": 129, "MAX_DURATION": 129, "ELEMENTS_PER_SECOND": 254.0, "THROUGHPUT": 371993.0, "COUNT": 254}, {"from": 78000, "to": 79000, "AVG_TTFB": 40.228813, "ERRORS": 120, "AVG_DURATION": 41.584747, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 29, "ERROR_RATE": 50.847458, "MIN_TTFB": 28, "MAX_TTFB": 120, "MAX_DURATION": 121, "ELEMENTS_PER_SECOND": 236.0, "THROUGHPUT": 347894.0, "COUNT": 236}, {"from": 79000, "to": 80000, "AVG_TTFB": 37.841698, "ERRORS": 129, "AVG_DURATION": 38.934364, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 28, "ERROR_RATE": 49.80695, "MIN_TTFB": 28, "MAX_TTFB": 107, "MAX_DURATION": 107, "ELEMENTS_PER_SECOND": 259.0, "THROUGHPUT": 380260.0, "COUNT": 259}, {"from": 80000, "to": 81000, "AVG_TTFB": 39.76543, "ERRORS": 120, "AVG_DURATION": 40.806583, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 29, "ERROR_RATE": 49.382717, "MIN_TTFB": 28, "MAX_TTFB": 161, "MAX_DURATION": 161, "ELEMENTS_PER_SECOND": 243.0, "THROUGHPUT": 356206.0, "COUNT": 243}, {"from": 81000, "to": 82000, "AVG_TTFB": 36.6692, "ERRORS": 132, "AVG_DURATION": 37.688213, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 29, "ERROR_RATE": 50.190113, "MIN_TTFB": 28, "MAX_TTFB": 75, "MAX_DURATION": 75, "ELEMENTS_PER_SECOND": 263.0, "THROUGHPUT": 386744.0, "COUNT": 263}, {"from": 82000, "to": 83000, "AVG_TTFB": 40.866665, "ERRORS": 122, "AVG_DURATION": 41.945835, "ERRORS_PER_SECOND": 122.0, "MIN_DURATION": 29, "ERROR_RATE": 50.833332, "MIN_TTFB": 28, "MAX_TTFB": 150, "MAX_DURATION": 150, "ELEMENTS_PER_SECOND": 240.0, "THROUGHPUT": 353891.0, "COUNT": 240}, {"from": 83000, "to": 84000, "AVG_TTFB": 38.76829, "ERRORS": 123, "AVG_DURATION": 39.853657, "ERRORS_PER_SECOND": 123.0, "MIN_DURATION": 30, "ERROR_RATE": 50.0, "MIN_TTFB": 29, "MAX_TTFB": 111, "MAX_DURATION": 112, "ELEMENTS_PER_SECOND": 246.0, "THROUGHPUT": 361542.0, "COUNT": 246}, {"from": 84000, "to": 85000, "AVG_TTFB": 38.916668, "ERRORS": 125, "AVG_DURATION": 39.924603, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 29, "ERROR_RATE": 49.603176, "MIN_TTFB": 28, "MAX_TTFB": 145, "MAX_DURATION": 145, "ELEMENTS_PER_SECOND": 252.0, "THROUGHPUT": 369647.0, "COUNT": 252}, {"from": 85000, "to": 86000, "AVG_TTFB": 38.192913, "ERRORS": 126, "AVG_DURATION": 39.374016, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 29, "ERROR_RATE": 49.6063, "MIN_TTFB": 28, "MAX_TTFB": 113, "MAX_DURATION": 114, "ELEMENTS_PER_SECOND": 254.0, "THROUGHPUT": 372634.0, "COUNT": 254}, {"from": 86000, "to": 87000, "AVG_TTFB": 36.885498, "ERRORS": 133, "AVG_DURATION": 37.950382, "ERRORS_PER_SECOND": 133.0, "MIN_DURATION": 29, "ERROR_RATE": 50.76336, "MIN_TTFB": 29, "MAX_TTFB": 72, "MAX_DURATION": 72, "ELEMENTS_PER_SECOND": 262.0, "THROUGHPUT": 386105.0, "COUNT": 262}, {"from": 87000, "to": 88000, "AVG_TTFB": 37.65116, "ERRORS": 129, "AVG_DURATION": 38.666668, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 180, "MAX_DURATION": 180, "ELEMENTS_PER_SECOND": 258.0, "THROUGHPUT": 379101.0, "COUNT": 258}, {"from": 88000, "to": 89000, "AVG_TTFB": 36.505882, "ERRORS": 127, "AVG_DURATION": 37.513725, "ERRORS_PER_SECOND": 127.0, "MIN_DURATION": 29, "ERROR_RATE": 49.80392, "MIN_TTFB": 28, "MAX_TTFB": 116, "MAX_DURATION": 116, "ELEMENTS_PER_SECOND": 255.0, "THROUGHPUT": 374386.0, "COUNT": 255}, {"from": 89000, "to": 90000, "AVG_TTFB": 50.465347, "ERRORS": 101, "AVG_DURATION": 51.50495, "ERRORS_PER_SECOND": 101.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 384, "MAX_DURATION": 384, "ELEMENTS_PER_SECOND": 202.0, "THROUGHPUT": 296781.0, "COUNT": 202}, {"from": 90000, "to": 91000, "AVG_TTFB": 37.313953, "ERRORS": 129, "AVG_DURATION": 38.368217, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 114, "MAX_DURATION": 115, "ELEMENTS_PER_SECOND": 258.0, "THROUGHPUT": 379067.0, "COUNT": 258}, {"from": 91000, "to": 92000, "AVG_TTFB": 36.826416, "ERRORS": 131, "AVG_DURATION": 37.856606, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 29, "ERROR_RATE": 49.433964, "MIN_TTFB": 29, "MAX_TTFB": 149, "MAX_DURATION": 149, "ELEMENTS_PER_SECOND": 265.0, "THROUGHPUT": 388537.0, "COUNT": 265}, {"from": 92000, "to": 93000, "AVG_TTFB": 37.107693, "ERRORS": 131, "AVG_DURATION": 38.01154, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 28, "ERROR_RATE": 50.384617, "MIN_TTFB": 28, "MAX_TTFB": 117, "MAX_DURATION": 117, "ELEMENTS_PER_SECOND": 260.0, "THROUGHPUT": 382659.0, "COUNT": 260}, {"from": 93000, "to": 94000, "AVG_TTFB": 37.147285, "ERRORS": 129, "AVG_DURATION": 38.0969, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 28, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 155, "MAX_DURATION": 156, "ELEMENTS_PER_SECOND": 258.0, "THROUGHPUT": 379120.0, "COUNT": 258}, {"from": 94000, "to": 95000, "AVG_TTFB": 42.86087, "ERRORS": 117, "AVG_DURATION": 43.852173, "ERRORS_PER_SECOND": 117.0, "MIN_DURATION": 28, "ERROR_RATE": 50.869564, "MIN_TTFB": 27, "MAX_TTFB": 264, "MAX_DURATION": 264, "ELEMENTS_PER_SECOND": 230.0, "THROUGHPUT": 339156.0, "COUNT": 230}, {"from": 95000, "to": 96000, "AVG_TTFB": 37.33712, "ERRORS": 130, "AVG_DURATION": 38.333332, "ERRORS_PER_SECOND": 130.0, "MIN_DURATION": 28, "ERROR_RATE": 49.242424, "MIN_TTFB": 28, "MAX_TTFB": 155, "MAX_DURATION": 155, "ELEMENTS_PER_SECOND": 264.0, "THROUGHPUT": 386640.0, "COUNT": 264}, {"from": 96000, "to": 97000, "AVG_TTFB": 38.914978, "ERRORS": 123, "AVG_DURATION": 40.016193, "ERRORS_PER_SECOND": 123.0, "MIN_DURATION": 28, "ERROR_RATE": 49.79757, "MIN_TTFB": 28, "MAX_TTFB": 183, "MAX_DURATION": 183, "ELEMENTS_PER_SECOND": 247.0, "THROUGHPUT": 362701.0, "COUNT": 247}, {"from": 97000, "to": 98000, "AVG_TTFB": 37.665386, "ERRORS": 131, "AVG_DURATION": 38.623077, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 29, "ERROR_RATE": 50.384617, "MIN_TTFB": 28, "MAX_TTFB": 151, "MAX_DURATION": 151, "ELEMENTS_PER_SECOND": 260.0, "THROUGHPUT": 382692.0, "COUNT": 260}, {"from": 98000, "to": 99000, "AVG_TTFB": 38.69672, "ERRORS": 122, "AVG_DURATION": 39.68033, "ERRORS_PER_SECOND": 122.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 164, "MAX_DURATION": 164, "ELEMENTS_PER_SECOND": 244.0, "THROUGHPUT": 358589.0, "COUNT": 244}, {"from": 99000, "to": 100000, "AVG_TTFB": 41.746834, "ERRORS": 119, "AVG_DURATION": 42.751057, "ERRORS_PER_SECOND": 119.0, "MIN_DURATION": 29, "ERROR_RATE": 50.21097, "MIN_TTFB": 28, "MAX_TTFB": 163, "MAX_DURATION": 163, "ELEMENTS_PER_SECOND": 237.0, "THROUGHPUT": 348574.0, "COUNT": 237}, {"from": 100000, "to": 101000, "AVG_TTFB": 41.212765, "ERRORS": 118, "AVG_DURATION": 42.582977, "ERRORS_PER_SECOND": 118.0, "MIN_DURATION": 29, "ERROR_RATE": 50.212765, "MIN_TTFB": 28, "MAX_TTFB": 148, "MAX_DURATION": 148, "ELEMENTS_PER_SECOND": 235.0, "THROUGHPUT": 345668.0, "COUNT": 235}, {"from": 101000, "to": 102000, "AVG_TTFB": 56.32, "ERRORS": 87, "AVG_DURATION": 57.451427, "ERRORS_PER_SECOND": 87.0, "MIN_DURATION": 30, "ERROR_RATE": 49.714287, "MIN_TTFB": 30, "MAX_TTFB": 314, "MAX_DURATION": 314, "ELEMENTS_PER_SECOND": 175.0, "THROUGHPUT": 256853.0, "COUNT": 175}, {"from": 102000, "to": 103000, "AVG_TTFB": 58.939026, "ERRORS": 81, "AVG_DURATION": 60.30488, "ERRORS_PER_SECOND": 81.0, "MIN_DURATION": 34, "ERROR_RATE": 49.390244, "MIN_TTFB": 31, "MAX_TTFB": 142, "MAX_DURATION": 142, "ELEMENTS_PER_SECOND": 164.0, "THROUGHPUT": 240398.0, "COUNT": 164}, {"from": 103000, "to": 104000, "AVG_TTFB": 54.847458, "ERRORS": 89, "AVG_DURATION": 56.293785, "ERRORS_PER_SECOND": 89.0, "MIN_DURATION": 31, "ERROR_RATE": 50.282486, "MIN_TTFB": 31, "MAX_TTFB": 124, "MAX_DURATION": 124, "ELEMENTS_PER_SECOND": 177.0, "THROUGHPUT": 260440.0, "COUNT": 177}, {"from": 104000, "to": 105000, "AVG_TTFB": 69.97122, "ERRORS": 70, "AVG_DURATION": 71.60432, "ERRORS_PER_SECOND": 70.0, "MIN_DURATION": 46, "ERROR_RATE": 50.35971, "MIN_TTFB": 44, "MAX_TTFB": 222, "MAX_DURATION": 222, "ELEMENTS_PER_SECOND": 139.0, "THROUGHPUT": 204520.0, "COUNT": 139}, {"from": 105000, "to": 106000, "AVG_TTFB": 64.76974, "ERRORS": 76, "AVG_DURATION": 66.52631, "ERRORS_PER_SECOND": 76.0, "MIN_DURATION": 31, "ERROR_RATE": 50.0, "MIN_TTFB": 30, "MAX_TTFB": 225, "MAX_DURATION": 225, "ELEMENTS_PER_SECOND": 152.0, "THROUGHPUT": 223388.0, "COUNT": 152}, {"from": 106000, "to": 107000, "AVG_TTFB": 42.170307, "ERRORS": 113, "AVG_DURATION": 43.432316, "ERRORS_PER_SECOND": 113.0, "MIN_DURATION": 29, "ERROR_RATE": 49.34498, "MIN_TTFB": 28, "MAX_TTFB": 152, "MAX_DURATION": 152, "ELEMENTS_PER_SECOND": 229.0, "THROUGHPUT": 335689.0, "COUNT": 229}, {"from": 107000, "to": 108000, "AVG_TTFB": 41.97826, "ERRORS": 115, "AVG_DURATION": 43.113045, "ERRORS_PER_SECOND": 115.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 28, "MAX_TTFB": 170, "MAX_DURATION": 170, "ELEMENTS_PER_SECOND": 230.0, "THROUGHPUT": 338043.0, "COUNT": 230}, {"from": 108000, "to": 109000, "AVG_TTFB": 44.366974, "ERRORS": 110, "AVG_DURATION": 45.380733, "ERRORS_PER_SECOND": 110.0, "MIN_DURATION": 29, "ERROR_RATE": 50.458717, "MIN_TTFB": 28, "MAX_TTFB": 136, "MAX_DURATION": 136, "ELEMENTS_PER_SECOND": 218.0, "THROUGHPUT": 320990.0, "COUNT": 218}, {"from": 109000, "to": 110000, "AVG_TTFB": 46.736843, "ERRORS": 104, "AVG_DURATION": 48.062202, "ERRORS_PER_SECOND": 104.0, "MIN_DURATION": 31, "ERROR_RATE": 49.760765, "MIN_TTFB": 29, "MAX_TTFB": 107, "MAX_DURATION": 107, "ELEMENTS_PER_SECOND": 209.0, "THROUGHPUT": 306786.0, "COUNT": 209}, {"from": 110000, "to": 111000, "AVG_TTFB": 50.07772, "ERRORS": 95, "AVG_DURATION": 51.50259, "ERRORS_PER_SECOND": 95.0, "MIN_DURATION": 30, "ERROR_RATE": 49.222797, "MIN_TTFB": 29, "MAX_TTFB": 138, "MAX_DURATION": 138, "ELEMENTS_PER_SECOND": 193.0, "THROUGHPUT": 282658.0, "COUNT": 193}, {"from": 111000, "to": 112000, "AVG_TTFB": 39.67347, "ERRORS": 124, "AVG_DURATION": 40.812244, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 29, "ERROR_RATE": 50.612244, "MIN_TTFB": 29, "MAX_TTFB": 106, "MAX_DURATION": 106, "ELEMENTS_PER_SECOND": 245.0, "THROUGHPUT": 360847.0, "COUNT": 245}, {"from": 112000, "to": 113000, "AVG_TTFB": 42.606987, "ERRORS": 114, "AVG_DURATION": 43.855896, "ERRORS_PER_SECOND": 114.0, "MIN_DURATION": 29, "ERROR_RATE": 49.78166, "MIN_TTFB": 28, "MAX_TTFB": 173, "MAX_DURATION": 173, "ELEMENTS_PER_SECOND": 229.0, "THROUGHPUT": 336064.0, "COUNT": 229}, {"from": 113000, "to": 114000, "AVG_TTFB": 42.154186, "ERRORS": 114, "AVG_DURATION": 43.356827, "ERRORS_PER_SECOND": 114.0, "MIN_DURATION": 30, "ERROR_RATE": 50.220264, "MIN_TTFB": 28, "MAX_TTFB": 129, "MAX_DURATION": 129, "ELEMENTS_PER_SECOND": 227.0, "THROUGHPUT": 333756.0, "COUNT": 227}, {"from": 114000, "to": 115000, "AVG_TTFB": 49.88601, "ERRORS": 96, "AVG_DURATION": 51.165802, "ERRORS_PER_SECOND": 96.0, "MIN_DURATION": 30, "ERROR_RATE": 49.740932, "MIN_TTFB": 29, "MAX_TTFB": 126, "MAX_DURATION": 126, "ELEMENTS_PER_SECOND": 193.0, "THROUGHPUT": 283228.0, "COUNT": 193}, {"from": 115000, "to": 116000, "AVG_TTFB": 53.83978, "ERRORS": 92, "AVG_DURATION": 55.292816, "ERRORS_PER_SECOND": 92.0, "MIN_DURATION": 35, "ERROR_RATE": 50.828728, "MIN_TTFB": 35, "MAX_TTFB": 161, "MAX_DURATION": 161, "ELEMENTS_PER_SECOND": 181.0, "THROUGHPUT": 266757.0, "COUNT": 181}, {"from": 116000, "to": 117000, "AVG_TTFB": 64.80822, "ERRORS": 72, "AVG_DURATION": 66.23972, "ERRORS_PER_SECOND": 72.0, "MIN_DURATION": 39, "ERROR_RATE": 49.315067, "MIN_TTFB": 38, "MAX_TTFB": 158, "MAX_DURATION": 158, "ELEMENTS_PER_SECOND": 146.0, "THROUGHPUT": 213856.0, "COUNT": 146}, {"from": 117000, "to": 118000, "AVG_TTFB": 67.89262, "ERRORS": 75, "AVG_DURATION": 69.19463, "ERRORS_PER_SECOND": 75.0, "MIN_DURATION": 43, "ERROR_RATE": 50.33557, "MIN_TTFB": 42, "MAX_TTFB": 219, "MAX_DURATION": 219, "ELEMENTS_PER_SECOND": 149.0, "THROUGHPUT": 219176.0, "COUNT": 149}, {"from": 118000, "to": 119000, "AVG_TTFB": 65.12752, "ERRORS": 74, "AVG_DURATION": 66.7047, "ERRORS_PER_SECOND": 74.0, "MIN_DURATION": 48, "ERROR_RATE": 49.66443, "MIN_TTFB": 46, "MAX_TTFB": 139, "MAX_DURATION": 139, "ELEMENTS_PER_SECOND": 149.0, "THROUGHPUT": 218564.0, "COUNT": 149}, {"from": 119000, "to": 120000, "AVG_TTFB": 74.30075, "ERRORS": 51, "AVG_DURATION": 75.4812, "ERRORS_PER_SECOND": 51.0, "MIN_DURATION": 53, "ERROR_RATE": 38.345863, "MIN_TTFB": 51, "MAX_TTFB": 240, "MAX_DURATION": 240, "ELEMENTS_PER_SECOND": 133.0, "THROUGHPUT": 172527.0, "COUNT": 133}, {"from": 120000, "to": 121000, "AVG_TTFB": 89.86667, "ERRORS": 0, "AVG_DURATION": 90.066666, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 125, "MAX_DURATION": 125, "ELEMENTS_PER_SECOND": 15.0, "THROUGHPUT": 12738.0, "COUNT": 15}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements?category=TRANSACTION':
            return '[{"id": "all-transactions", "name": "<all transactions>", "type": "ALL_TRANSACTIONS"}, {"id": "e66cb9ec-3c72-49b4-b3a2-5fa91e2fd911", "name": "transaction_no_sla", "path": ["kamoulox", "Actions", "transaction_no_sla"], "type": "TRANSACTION"}, {"id": "7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a", "name": "transaction_sla_warning", "path": ["kamoulox", "Actions", "transaction_sla_warning"], "type": "TRANSACTION"}, {"id": "6a0ef850-8644-4236-80e5-53748fbd2d01", "name": "transaction_sla_failed", "path": ["kamoulox", "Actions", "transaction_sla_failed"], "type": "TRANSACTION"}, {"id": "818d2419-daec-43d6-b004-b1195799df57", "name": "transaction_sla_passed", "path": ["kamoulox", "Actions", "transaction_sla_passed"], "type": "TRANSACTION"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements/all-transactions/values':
            return '{"count": 14357, "elementPerSecond": 119.49827, "minDuration": 58, "maxDuration": 436, "sumDuration": 1191454, "avgDuration": 82.98767, "minTTFB": 27, "maxTTFB": 384, "sumTTFB": 654301, "avgTTFB": 45.573658, "sumDownloadedBytes": 41979146, "downloadedBytesPerSecond": 349406.94, "successCount": 152, "successPerSecond": 1.2651485, "successRate": 1.058717, "failureCount": 14205, "failurePerSecond": 118.23312, "failureRate": 98.941284, "percentile50": 70.0, "percentile90": 90.0, "percentile95": 113.0, "percentile99": 150.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements/all-transactions/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 1000, "AVG_TTFB": 58.836067, "ERRORS": 0, "AVG_DURATION": 101.78689, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 67, "ERROR_RATE": 0.0, "MIN_TTFB": 29, "MAX_TTFB": 162, "MAX_DURATION": 207, "ELEMENTS_PER_SECOND": 61.0, "THROUGHPUT": 93880.0, "COUNT": 61}, {"from": 1000, "to": 2000, "AVG_TTFB": 41.421875, "ERRORS": 121, "AVG_DURATION": 77.484375, "ERRORS_PER_SECOND": 121.0, "MIN_DURATION": 60, "ERROR_RATE": 94.53125, "MIN_TTFB": 29, "MAX_TTFB": 70, "MAX_DURATION": 109, "ELEMENTS_PER_SECOND": 128.0, "THROUGHPUT": 366417.0, "COUNT": 128}, {"from": 2000, "to": 3000, "AVG_TTFB": 42.95968, "ERRORS": 124, "AVG_DURATION": 78.814514, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 87, "MAX_DURATION": 125, "ELEMENTS_PER_SECOND": 124.0, "THROUGHPUT": 364491.0, "COUNT": 124}, {"from": 3000, "to": 4000, "AVG_TTFB": 41.732285, "ERRORS": 127, "AVG_DURATION": 76.71654, "ERRORS_PER_SECOND": 127.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 70, "MAX_DURATION": 108, "ELEMENTS_PER_SECOND": 127.0, "THROUGHPUT": 373243.0, "COUNT": 127}, {"from": 4000, "to": 5000, "AVG_TTFB": 41.353382, "ERRORS": 133, "AVG_DURATION": 75.93233, "ERRORS_PER_SECOND": 133.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 74, "MAX_DURATION": 114, "ELEMENTS_PER_SECOND": 133.0, "THROUGHPUT": 390829.0, "COUNT": 133}, {"from": 5000, "to": 6000, "AVG_TTFB": 44.663933, "ERRORS": 122, "AVG_DURATION": 80.77049, "ERRORS_PER_SECOND": 122.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 96, "MAX_DURATION": 144, "ELEMENTS_PER_SECOND": 122.0, "THROUGHPUT": 358552.0, "COUNT": 122}, {"from": 6000, "to": 7000, "AVG_TTFB": 41.098484, "ERRORS": 132, "AVG_DURATION": 75.61364, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 76, "MAX_DURATION": 113, "ELEMENTS_PER_SECOND": 132.0, "THROUGHPUT": 387967.0, "COUNT": 132}, {"from": 7000, "to": 8000, "AVG_TTFB": 40.603054, "ERRORS": 131, "AVG_DURATION": 75.03817, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 69, "MAX_DURATION": 106, "ELEMENTS_PER_SECOND": 131.0, "THROUGHPUT": 384945.0, "COUNT": 131}, {"from": 8000, "to": 9000, "AVG_TTFB": 41.99248, "ERRORS": 133, "AVG_DURATION": 76.30075, "ERRORS_PER_SECOND": 133.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 74, "MAX_DURATION": 107, "ELEMENTS_PER_SECOND": 133.0, "THROUGHPUT": 390826.0, "COUNT": 133}, {"from": 9000, "to": 10000, "AVG_TTFB": 41.015385, "ERRORS": 130, "AVG_DURATION": 75.292305, "ERRORS_PER_SECOND": 130.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 83, "MAX_DURATION": 116, "ELEMENTS_PER_SECOND": 130.0, "THROUGHPUT": 382045.0, "COUNT": 130}, {"from": 10000, "to": 11000, "AVG_TTFB": 40.25564, "ERRORS": 133, "AVG_DURATION": 74.947365, "ERRORS_PER_SECOND": 133.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 82, "MAX_DURATION": 116, "ELEMENTS_PER_SECOND": 133.0, "THROUGHPUT": 390916.0, "COUNT": 133}, {"from": 11000, "to": 12000, "AVG_TTFB": 42.45238, "ERRORS": 126, "AVG_DURATION": 78.92857, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 101, "MAX_DURATION": 151, "ELEMENTS_PER_SECOND": 126.0, "THROUGHPUT": 370301.0, "COUNT": 126}, {"from": 12000, "to": 13000, "AVG_TTFB": 43.23387, "ERRORS": 124, "AVG_DURATION": 78.935486, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 113, "MAX_DURATION": 150, "ELEMENTS_PER_SECOND": 124.0, "THROUGHPUT": 364425.0, "COUNT": 124}, {"from": 13000, "to": 14000, "AVG_TTFB": 41.765152, "ERRORS": 132, "AVG_DURATION": 76.356064, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 88, "MAX_DURATION": 123, "ELEMENTS_PER_SECOND": 132.0, "THROUGHPUT": 387854.0, "COUNT": 132}, {"from": 14000, "to": 15000, "AVG_TTFB": 40.772728, "ERRORS": 132, "AVG_DURATION": 74.795456, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 78, "MAX_DURATION": 112, "ELEMENTS_PER_SECOND": 132.0, "THROUGHPUT": 387930.0, "COUNT": 132}, {"from": 15000, "to": 16000, "AVG_TTFB": 40.686565, "ERRORS": 134, "AVG_DURATION": 74.29851, "ERRORS_PER_SECOND": 134.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 87, "MAX_DURATION": 125, "ELEMENTS_PER_SECOND": 134.0, "THROUGHPUT": 393867.0, "COUNT": 134}, {"from": 16000, "to": 17000, "AVG_TTFB": 67.551285, "ERRORS": 78, "AVG_DURATION": 127.88461, "ERRORS_PER_SECOND": 78.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 305, "MAX_DURATION": 436, "ELEMENTS_PER_SECOND": 78.0, "THROUGHPUT": 229229.0, "COUNT": 78}, {"from": 17000, "to": 18000, "AVG_TTFB": 42.85484, "ERRORS": 124, "AVG_DURATION": 78.09677, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 92, "MAX_DURATION": 135, "ELEMENTS_PER_SECOND": 124.0, "THROUGHPUT": 364456.0, "COUNT": 124}, {"from": 18000, "to": 19000, "AVG_TTFB": 42.946156, "ERRORS": 130, "AVG_DURATION": 77.815384, "ERRORS_PER_SECOND": 130.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 95, "MAX_DURATION": 133, "ELEMENTS_PER_SECOND": 130.0, "THROUGHPUT": 382070.0, "COUNT": 130}, {"from": 19000, "to": 20000, "AVG_TTFB": 43.968, "ERRORS": 125, "AVG_DURATION": 79.16, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 27, "MAX_TTFB": 132, "MAX_DURATION": 166, "ELEMENTS_PER_SECOND": 125.0, "THROUGHPUT": 367338.0, "COUNT": 125}, {"from": 20000, "to": 21000, "AVG_TTFB": 43.13115, "ERRORS": 122, "AVG_DURATION": 79.72951, "ERRORS_PER_SECOND": 122.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 106, "MAX_DURATION": 145, "ELEMENTS_PER_SECOND": 122.0, "THROUGHPUT": 358547.0, "COUNT": 122}, {"from": 21000, "to": 22000, "AVG_TTFB": 51.175926, "ERRORS": 108, "AVG_DURATION": 94.32407, "ERRORS_PER_SECOND": 108.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 268, "MAX_DURATION": 305, "ELEMENTS_PER_SECOND": 108.0, "THROUGHPUT": 317421.0, "COUNT": 108}, {"from": 22000, "to": 23000, "AVG_TTFB": 41.413532, "ERRORS": 133, "AVG_DURATION": 75.73684, "ERRORS_PER_SECOND": 133.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 81, "MAX_DURATION": 124, "ELEMENTS_PER_SECOND": 133.0, "THROUGHPUT": 390798.0, "COUNT": 133}, {"from": 23000, "to": 24000, "AVG_TTFB": 43.015503, "ERRORS": 129, "AVG_DURATION": 77.20155, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 93, "MAX_DURATION": 127, "ELEMENTS_PER_SECOND": 129.0, "THROUGHPUT": 379161.0, "COUNT": 129}, {"from": 24000, "to": 25000, "AVG_TTFB": 40.75, "ERRORS": 132, "AVG_DURATION": 75.01515, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 82, "MAX_DURATION": 115, "ELEMENTS_PER_SECOND": 132.0, "THROUGHPUT": 388010.0, "COUNT": 132}, {"from": 25000, "to": 26000, "AVG_TTFB": 41.0, "ERRORS": 129, "AVG_DURATION": 74.93799, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 113, "MAX_DURATION": 147, "ELEMENTS_PER_SECOND": 129.0, "THROUGHPUT": 379073.0, "COUNT": 129}, {"from": 26000, "to": 27000, "AVG_TTFB": 43.975, "ERRORS": 120, "AVG_DURATION": 82.666664, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 132, "MAX_DURATION": 170, "ELEMENTS_PER_SECOND": 120.0, "THROUGHPUT": 352620.0, "COUNT": 120}, {"from": 27000, "to": 28000, "AVG_TTFB": 44.06923, "ERRORS": 130, "AVG_DURATION": 78.92308, "ERRORS_PER_SECOND": 130.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 106, "MAX_DURATION": 169, "ELEMENTS_PER_SECOND": 130.0, "THROUGHPUT": 381972.0, "COUNT": 130}, {"from": 28000, "to": 29000, "AVG_TTFB": 42.170544, "ERRORS": 129, "AVG_DURATION": 77.387596, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 96, "MAX_DURATION": 132, "ELEMENTS_PER_SECOND": 129.0, "THROUGHPUT": 379127.0, "COUNT": 129}, {"from": 29000, "to": 30000, "AVG_TTFB": 52.990196, "ERRORS": 102, "AVG_DURATION": 97.90196, "ERRORS_PER_SECOND": 102.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 242, "MAX_DURATION": 287, "ELEMENTS_PER_SECOND": 102.0, "THROUGHPUT": 299763.0, "COUNT": 102}, {"from": 30000, "to": 31000, "AVG_TTFB": 44.366665, "ERRORS": 120, "AVG_DURATION": 82.26667, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 120, "MAX_DURATION": 151, "ELEMENTS_PER_SECOND": 120.0, "THROUGHPUT": 352701.0, "COUNT": 120}, {"from": 31000, "to": 32000, "AVG_TTFB": 42.507576, "ERRORS": 132, "AVG_DURATION": 76.51515, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 87, "MAX_DURATION": 118, "ELEMENTS_PER_SECOND": 132.0, "THROUGHPUT": 387967.0, "COUNT": 132}, {"from": 32000, "to": 33000, "AVG_TTFB": 41.696968, "ERRORS": 132, "AVG_DURATION": 76.606064, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 109, "MAX_DURATION": 143, "ELEMENTS_PER_SECOND": 132.0, "THROUGHPUT": 387901.0, "COUNT": 132}, {"from": 33000, "to": 34000, "AVG_TTFB": 40.717556, "ERRORS": 131, "AVG_DURATION": 75.12214, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 72, "MAX_DURATION": 109, "ELEMENTS_PER_SECOND": 131.0, "THROUGHPUT": 385039.0, "COUNT": 131}, {"from": 34000, "to": 35000, "AVG_TTFB": 43.98387, "ERRORS": 124, "AVG_DURATION": 79.95968, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 98, "MAX_DURATION": 139, "ELEMENTS_PER_SECOND": 124.0, "THROUGHPUT": 364503.0, "COUNT": 124}, {"from": 35000, "to": 36000, "AVG_TTFB": 45.867767, "ERRORS": 121, "AVG_DURATION": 83.32231, "ERRORS_PER_SECOND": 121.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 136, "MAX_DURATION": 173, "ELEMENTS_PER_SECOND": 121.0, "THROUGHPUT": 355679.0, "COUNT": 121}, {"from": 36000, "to": 37000, "AVG_TTFB": 43.102364, "ERRORS": 127, "AVG_DURATION": 78.29921, "ERRORS_PER_SECOND": 127.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 102, "MAX_DURATION": 133, "ELEMENTS_PER_SECOND": 127.0, "THROUGHPUT": 373252.0, "COUNT": 127}, {"from": 37000, "to": 38000, "AVG_TTFB": 40.833332, "ERRORS": 132, "AVG_DURATION": 75.356064, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 80, "MAX_DURATION": 115, "ELEMENTS_PER_SECOND": 132.0, "THROUGHPUT": 387937.0, "COUNT": 132}, {"from": 38000, "to": 39000, "AVG_TTFB": 42.44961, "ERRORS": 129, "AVG_DURATION": 76.48837, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 85, "MAX_DURATION": 118, "ELEMENTS_PER_SECOND": 129.0, "THROUGHPUT": 379034.0, "COUNT": 129}, {"from": 39000, "to": 40000, "AVG_TTFB": 41.778625, "ERRORS": 131, "AVG_DURATION": 75.503815, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 91, "MAX_DURATION": 127, "ELEMENTS_PER_SECOND": 131.0, "THROUGHPUT": 385046.0, "COUNT": 131}, {"from": 40000, "to": 41000, "AVG_TTFB": 45.096, "ERRORS": 125, "AVG_DURATION": 81.208, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 115, "MAX_DURATION": 150, "ELEMENTS_PER_SECOND": 125.0, "THROUGHPUT": 367353.0, "COUNT": 125}, {"from": 41000, "to": 42000, "AVG_TTFB": 41.142857, "ERRORS": 133, "AVG_DURATION": 75.19549, "ERRORS_PER_SECOND": 133.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 114, "MAX_DURATION": 143, "ELEMENTS_PER_SECOND": 133.0, "THROUGHPUT": 390865.0, "COUNT": 133}, {"from": 42000, "to": 43000, "AVG_TTFB": 41.776924, "ERRORS": 130, "AVG_DURATION": 76.04615, "ERRORS_PER_SECOND": 130.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 119, "MAX_DURATION": 155, "ELEMENTS_PER_SECOND": 130.0, "THROUGHPUT": 381980.0, "COUNT": 130}, {"from": 43000, "to": 44000, "AVG_TTFB": 42.476925, "ERRORS": 130, "AVG_DURATION": 77.37692, "ERRORS_PER_SECOND": 130.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 113, "MAX_DURATION": 149, "ELEMENTS_PER_SECOND": 130.0, "THROUGHPUT": 382038.0, "COUNT": 130}, {"from": 44000, "to": 45000, "AVG_TTFB": 44.0, "ERRORS": 125, "AVG_DURATION": 79.136, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 109, "MAX_DURATION": 144, "ELEMENTS_PER_SECOND": 125.0, "THROUGHPUT": 367354.0, "COUNT": 125}, {"from": 45000, "to": 46000, "AVG_TTFB": 41.401516, "ERRORS": 132, "AVG_DURATION": 75.492424, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 115, "MAX_DURATION": 146, "ELEMENTS_PER_SECOND": 132.0, "THROUGHPUT": 388004.0, "COUNT": 132}, {"from": 46000, "to": 47000, "AVG_TTFB": 41.52672, "ERRORS": 131, "AVG_DURATION": 75.65649, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 112, "MAX_DURATION": 148, "ELEMENTS_PER_SECOND": 131.0, "THROUGHPUT": 384980.0, "COUNT": 131}, {"from": 47000, "to": 48000, "AVG_TTFB": 44.404957, "ERRORS": 121, "AVG_DURATION": 81.17355, "ERRORS_PER_SECOND": 121.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 85, "MAX_DURATION": 121, "ELEMENTS_PER_SECOND": 121.0, "THROUGHPUT": 355616.0, "COUNT": 121}, {"from": 48000, "to": 49000, "AVG_TTFB": 42.867188, "ERRORS": 128, "AVG_DURATION": 78.78125, "ERRORS_PER_SECOND": 128.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 99, "MAX_DURATION": 132, "ELEMENTS_PER_SECOND": 128.0, "THROUGHPUT": 376252.0, "COUNT": 128}, {"from": 49000, "to": 50000, "AVG_TTFB": 43.046875, "ERRORS": 128, "AVG_DURATION": 77.97656, "ERRORS_PER_SECOND": 128.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 94, "MAX_DURATION": 130, "ELEMENTS_PER_SECOND": 128.0, "THROUGHPUT": 376180.0, "COUNT": 128}, {"from": 50000, "to": 51000, "AVG_TTFB": 45.983192, "ERRORS": 119, "AVG_DURATION": 84.47899, "ERRORS_PER_SECOND": 119.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 110, "MAX_DURATION": 153, "ELEMENTS_PER_SECOND": 119.0, "THROUGHPUT": 349669.0, "COUNT": 119}, {"from": 51000, "to": 52000, "AVG_TTFB": 43.333332, "ERRORS": 126, "AVG_DURATION": 79.06349, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 127, "MAX_DURATION": 159, "ELEMENTS_PER_SECOND": 126.0, "THROUGHPUT": 370257.0, "COUNT": 126}, {"from": 52000, "to": 53000, "AVG_TTFB": 50.333332, "ERRORS": 108, "AVG_DURATION": 91.046295, "ERRORS_PER_SECOND": 108.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 208, "MAX_DURATION": 242, "ELEMENTS_PER_SECOND": 108.0, "THROUGHPUT": 317288.0, "COUNT": 108}, {"from": 53000, "to": 54000, "AVG_TTFB": 44.244095, "ERRORS": 127, "AVG_DURATION": 80.125984, "ERRORS_PER_SECOND": 127.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 128, "MAX_DURATION": 160, "ELEMENTS_PER_SECOND": 127.0, "THROUGHPUT": 373090.0, "COUNT": 127}, {"from": 54000, "to": 55000, "AVG_TTFB": 43.403225, "ERRORS": 124, "AVG_DURATION": 79.48387, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 93, "MAX_DURATION": 126, "ELEMENTS_PER_SECOND": 124.0, "THROUGHPUT": 364265.0, "COUNT": 124}, {"from": 55000, "to": 56000, "AVG_TTFB": 42.28125, "ERRORS": 128, "AVG_DURATION": 77.07031, "ERRORS_PER_SECOND": 128.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 126, "MAX_DURATION": 164, "ELEMENTS_PER_SECOND": 128.0, "THROUGHPUT": 375990.0, "COUNT": 128}, {"from": 56000, "to": 57000, "AVG_TTFB": 43.96875, "ERRORS": 128, "AVG_DURATION": 79.03906, "ERRORS_PER_SECOND": 128.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 119, "MAX_DURATION": 151, "ELEMENTS_PER_SECOND": 128.0, "THROUGHPUT": 376037.0, "COUNT": 128}, {"from": 57000, "to": 58000, "AVG_TTFB": 43.33871, "ERRORS": 124, "AVG_DURATION": 79.66936, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 81, "MAX_DURATION": 119, "ELEMENTS_PER_SECOND": 124.0, "THROUGHPUT": 364312.0, "COUNT": 124}, {"from": 58000, "to": 59000, "AVG_TTFB": 46.145298, "ERRORS": 117, "AVG_DURATION": 83.91453, "ERRORS_PER_SECOND": 117.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 113, "MAX_DURATION": 153, "ELEMENTS_PER_SECOND": 117.0, "THROUGHPUT": 343699.0, "COUNT": 117}, {"from": 59000, "to": 60000, "AVG_TTFB": 48.181034, "ERRORS": 89, "AVG_DURATION": 86.56035, "ERRORS_PER_SECOND": 89.0, "MIN_DURATION": 61, "ERROR_RATE": 76.72414, "MIN_TTFB": 30, "MAX_TTFB": 127, "MAX_DURATION": 165, "ELEMENTS_PER_SECOND": 116.0, "THROUGHPUT": 302960.0, "COUNT": 116}, {"from": 60000, "to": 61000, "AVG_TTFB": 42.91129, "ERRORS": 85, "AVG_DURATION": 80.951614, "ERRORS_PER_SECOND": 85.0, "MIN_DURATION": 62, "ERROR_RATE": 68.548386, "MIN_TTFB": 30, "MAX_TTFB": 111, "MAX_DURATION": 152, "ELEMENTS_PER_SECOND": 124.0, "THROUGHPUT": 309819.0, "COUNT": 124}, {"from": 61000, "to": 62000, "AVG_TTFB": 42.968254, "ERRORS": 126, "AVG_DURATION": 78.7619, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 126, "MAX_DURATION": 156, "ELEMENTS_PER_SECOND": 126.0, "THROUGHPUT": 370330.0, "COUNT": 126}, {"from": 62000, "to": 63000, "AVG_TTFB": 45.18182, "ERRORS": 121, "AVG_DURATION": 81.380165, "ERRORS_PER_SECOND": 121.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 135, "MAX_DURATION": 171, "ELEMENTS_PER_SECOND": 121.0, "THROUGHPUT": 355612.0, "COUNT": 121}, {"from": 63000, "to": 64000, "AVG_TTFB": 41.580154, "ERRORS": 131, "AVG_DURATION": 76.72519, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 95, "MAX_DURATION": 126, "ELEMENTS_PER_SECOND": 131.0, "THROUGHPUT": 385015.0, "COUNT": 131}, {"from": 64000, "to": 65000, "AVG_TTFB": 42.265152, "ERRORS": 132, "AVG_DURATION": 76.583336, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 136, "MAX_DURATION": 168, "ELEMENTS_PER_SECOND": 132.0, "THROUGHPUT": 388004.0, "COUNT": 132}, {"from": 65000, "to": 66000, "AVG_TTFB": 47.247864, "ERRORS": 117, "AVG_DURATION": 84.42735, "ERRORS_PER_SECOND": 117.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 140, "MAX_DURATION": 174, "ELEMENTS_PER_SECOND": 117.0, "THROUGHPUT": 343882.0, "COUNT": 117}, {"from": 66000, "to": 67000, "AVG_TTFB": 43.928, "ERRORS": 125, "AVG_DURATION": 79.912, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 114, "MAX_DURATION": 150, "ELEMENTS_PER_SECOND": 125.0, "THROUGHPUT": 367355.0, "COUNT": 125}, {"from": 67000, "to": 68000, "AVG_TTFB": 44.295082, "ERRORS": 122, "AVG_DURATION": 80.92623, "ERRORS_PER_SECOND": 122.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 117, "MAX_DURATION": 149, "ELEMENTS_PER_SECOND": 122.0, "THROUGHPUT": 358482.0, "COUNT": 122}, {"from": 68000, "to": 69000, "AVG_TTFB": 43.61905, "ERRORS": 126, "AVG_DURATION": 80.007935, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 94, "MAX_DURATION": 125, "ELEMENTS_PER_SECOND": 126.0, "THROUGHPUT": 370285.0, "COUNT": 126}, {"from": 69000, "to": 70000, "AVG_TTFB": 44.82927, "ERRORS": 123, "AVG_DURATION": 81.544716, "ERRORS_PER_SECOND": 123.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 147, "MAX_DURATION": 178, "ELEMENTS_PER_SECOND": 123.0, "THROUGHPUT": 361474.0, "COUNT": 123}, {"from": 70000, "to": 71000, "AVG_TTFB": 45.3, "ERRORS": 120, "AVG_DURATION": 82.316666, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 104, "MAX_DURATION": 146, "ELEMENTS_PER_SECOND": 120.0, "THROUGHPUT": 352654.0, "COUNT": 120}, {"from": 71000, "to": 72000, "AVG_TTFB": 42.140625, "ERRORS": 128, "AVG_DURATION": 78.390625, "ERRORS_PER_SECOND": 128.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 133, "MAX_DURATION": 164, "ELEMENTS_PER_SECOND": 128.0, "THROUGHPUT": 376074.0, "COUNT": 128}, {"from": 72000, "to": 73000, "AVG_TTFB": 46.233334, "ERRORS": 120, "AVG_DURATION": 83.71667, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 147, "MAX_DURATION": 184, "ELEMENTS_PER_SECOND": 120.0, "THROUGHPUT": 352719.0, "COUNT": 120}, {"from": 73000, "to": 74000, "AVG_TTFB": 45.29752, "ERRORS": 121, "AVG_DURATION": 80.64463, "ERRORS_PER_SECOND": 121.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 134, "MAX_DURATION": 167, "ELEMENTS_PER_SECOND": 121.0, "THROUGHPUT": 355652.0, "COUNT": 121}, {"from": 74000, "to": 75000, "AVG_TTFB": 42.091602, "ERRORS": 131, "AVG_DURATION": 76.74809, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 122, "MAX_DURATION": 164, "ELEMENTS_PER_SECOND": 131.0, "THROUGHPUT": 385042.0, "COUNT": 131}, {"from": 75000, "to": 76000, "AVG_TTFB": 41.351147, "ERRORS": 131, "AVG_DURATION": 75.78626, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 117, "MAX_DURATION": 152, "ELEMENTS_PER_SECOND": 131.0, "THROUGHPUT": 385076.0, "COUNT": 131}, {"from": 76000, "to": 77000, "AVG_TTFB": 43.816, "ERRORS": 125, "AVG_DURATION": 79.816, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 148, "MAX_DURATION": 183, "ELEMENTS_PER_SECOND": 125.0, "THROUGHPUT": 367348.0, "COUNT": 125}, {"from": 77000, "to": 78000, "AVG_TTFB": 43.176, "ERRORS": 125, "AVG_DURATION": 78.408, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 129, "MAX_DURATION": 168, "ELEMENTS_PER_SECOND": 125.0, "THROUGHPUT": 367317.0, "COUNT": 125}, {"from": 78000, "to": 79000, "AVG_TTFB": 45.841667, "ERRORS": 120, "AVG_DURATION": 83.3, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 127, "MAX_DURATION": 169, "ELEMENTS_PER_SECOND": 120.0, "THROUGHPUT": 352562.0, "COUNT": 120}, {"from": 79000, "to": 80000, "AVG_TTFB": 42.968994, "ERRORS": 129, "AVG_DURATION": 77.8062, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 107, "MAX_DURATION": 144, "ELEMENTS_PER_SECOND": 129.0, "THROUGHPUT": 379098.0, "COUNT": 129}, {"from": 80000, "to": 81000, "AVG_TTFB": 45.491665, "ERRORS": 120, "AVG_DURATION": 82.15, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 161, "MAX_DURATION": 202, "ELEMENTS_PER_SECOND": 120.0, "THROUGHPUT": 352690.0, "COUNT": 120}, {"from": 81000, "to": 82000, "AVG_TTFB": 40.734848, "ERRORS": 132, "AVG_DURATION": 75.106064, "ERRORS_PER_SECOND": 132.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 75, "MAX_DURATION": 109, "ELEMENTS_PER_SECOND": 132.0, "THROUGHPUT": 387914.0, "COUNT": 132}, {"from": 82000, "to": 83000, "AVG_TTFB": 47.15574, "ERRORS": 122, "AVG_DURATION": 84.090164, "ERRORS_PER_SECOND": 122.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 150, "MAX_DURATION": 193, "ELEMENTS_PER_SECOND": 122.0, "THROUGHPUT": 358571.0, "COUNT": 122}, {"from": 83000, "to": 84000, "AVG_TTFB": 43.284554, "ERRORS": 123, "AVG_DURATION": 79.861786, "ERRORS_PER_SECOND": 123.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 111, "MAX_DURATION": 154, "ELEMENTS_PER_SECOND": 123.0, "THROUGHPUT": 361550.0, "COUNT": 123}, {"from": 84000, "to": 85000, "AVG_TTFB": 43.752, "ERRORS": 125, "AVG_DURATION": 79.272, "ERRORS_PER_SECOND": 125.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 145, "MAX_DURATION": 180, "ELEMENTS_PER_SECOND": 125.0, "THROUGHPUT": 367317.0, "COUNT": 125}, {"from": 85000, "to": 86000, "AVG_TTFB": 42.896824, "ERRORS": 126, "AVG_DURATION": 79.38095, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 113, "MAX_DURATION": 154, "ELEMENTS_PER_SECOND": 126.0, "THROUGHPUT": 370296.0, "COUNT": 126}, {"from": 86000, "to": 87000, "AVG_TTFB": 40.661655, "ERRORS": 133, "AVG_DURATION": 75.52631, "ERRORS_PER_SECOND": 133.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 72, "MAX_DURATION": 107, "ELEMENTS_PER_SECOND": 133.0, "THROUGHPUT": 390777.0, "COUNT": 133}, {"from": 87000, "to": 88000, "AVG_TTFB": 43.147285, "ERRORS": 129, "AVG_DURATION": 77.53488, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 180, "MAX_DURATION": 214, "ELEMENTS_PER_SECOND": 129.0, "THROUGHPUT": 379083.0, "COUNT": 129}, {"from": 88000, "to": 89000, "AVG_TTFB": 41.27559, "ERRORS": 127, "AVG_DURATION": 75.37795, "ERRORS_PER_SECOND": 127.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 116, "MAX_DURATION": 151, "ELEMENTS_PER_SECOND": 127.0, "THROUGHPUT": 373210.0, "COUNT": 127}, {"from": 89000, "to": 90000, "AVG_TTFB": 58.71287, "ERRORS": 101, "AVG_DURATION": 103.08911, "ERRORS_PER_SECOND": 101.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 384, "MAX_DURATION": 415, "ELEMENTS_PER_SECOND": 101.0, "THROUGHPUT": 296789.0, "COUNT": 101}, {"from": 90000, "to": 91000, "AVG_TTFB": 42.217052, "ERRORS": 129, "AVG_DURATION": 76.689926, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 114, "MAX_DURATION": 155, "ELEMENTS_PER_SECOND": 129.0, "THROUGHPUT": 379075.0, "COUNT": 129}, {"from": 91000, "to": 92000, "AVG_TTFB": 40.85496, "ERRORS": 131, "AVG_DURATION": 75.37405, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 149, "MAX_DURATION": 188, "ELEMENTS_PER_SECOND": 131.0, "THROUGHPUT": 385039.0, "COUNT": 131}, {"from": 92000, "to": 93000, "AVG_TTFB": 42.17557, "ERRORS": 131, "AVG_DURATION": 76.24428, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 117, "MAX_DURATION": 156, "ELEMENTS_PER_SECOND": 131.0, "THROUGHPUT": 384981.0, "COUNT": 131}, {"from": 93000, "to": 94000, "AVG_TTFB": 41.76744, "ERRORS": 129, "AVG_DURATION": 76.0, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 155, "MAX_DURATION": 188, "ELEMENTS_PER_SECOND": 129.0, "THROUGHPUT": 379136.0, "COUNT": 129}, {"from": 94000, "to": 95000, "AVG_TTFB": 47.478634, "ERRORS": 117, "AVG_DURATION": 87.45299, "ERRORS_PER_SECOND": 117.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 264, "MAX_DURATION": 302, "ELEMENTS_PER_SECOND": 117.0, "THROUGHPUT": 343818.0, "COUNT": 117}, {"from": 95000, "to": 96000, "AVG_TTFB": 42.153847, "ERRORS": 130, "AVG_DURATION": 76.57692, "ERRORS_PER_SECOND": 130.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 155, "MAX_DURATION": 189, "ELEMENTS_PER_SECOND": 130.0, "THROUGHPUT": 381970.0, "COUNT": 130}, {"from": 96000, "to": 97000, "AVG_TTFB": 43.772358, "ERRORS": 123, "AVG_DURATION": 79.138214, "ERRORS_PER_SECOND": 123.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 122, "MAX_DURATION": 153, "ELEMENTS_PER_SECOND": 123.0, "THROUGHPUT": 361545.0, "COUNT": 123}, {"from": 97000, "to": 98000, "AVG_TTFB": 43.67939, "ERRORS": 131, "AVG_DURATION": 77.900764, "ERRORS_PER_SECOND": 131.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 183, "MAX_DURATION": 218, "ELEMENTS_PER_SECOND": 131.0, "THROUGHPUT": 385020.0, "COUNT": 131}, {"from": 98000, "to": 99000, "AVG_TTFB": 44.27869, "ERRORS": 122, "AVG_DURATION": 79.62295, "ERRORS_PER_SECOND": 122.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 164, "MAX_DURATION": 203, "ELEMENTS_PER_SECOND": 122.0, "THROUGHPUT": 358587.0, "COUNT": 122}, {"from": 99000, "to": 100000, "AVG_TTFB": 48.991596, "ERRORS": 119, "AVG_DURATION": 85.420166, "ERRORS_PER_SECOND": 119.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 163, "MAX_DURATION": 198, "ELEMENTS_PER_SECOND": 119.0, "THROUGHPUT": 349740.0, "COUNT": 119}, {"from": 100000, "to": 101000, "AVG_TTFB": 46.51695, "ERRORS": 118, "AVG_DURATION": 85.52542, "ERRORS_PER_SECOND": 118.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 148, "MAX_DURATION": 199, "ELEMENTS_PER_SECOND": 118.0, "THROUGHPUT": 346834.0, "COUNT": 118}, {"from": 101000, "to": 102000, "AVG_TTFB": 63.770115, "ERRORS": 87, "AVG_DURATION": 114.78161, "ERRORS_PER_SECOND": 87.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 314, "MAX_DURATION": 356, "ELEMENTS_PER_SECOND": 87.0, "THROUGHPUT": 255697.0, "COUNT": 87}, {"from": 102000, "to": 103000, "AVG_TTFB": 68.111115, "ERRORS": 81, "AVG_DURATION": 120.85185, "ERRORS_PER_SECOND": 81.0, "MIN_DURATION": 73, "ERROR_RATE": 100.0, "MIN_TTFB": 39, "MAX_TTFB": 142, "MAX_DURATION": 196, "ELEMENTS_PER_SECOND": 81.0, "THROUGHPUT": 238050.0, "COUNT": 81}, {"from": 103000, "to": 104000, "AVG_TTFB": 61.101124, "ERRORS": 89, "AVG_DURATION": 111.70786, "ERRORS_PER_SECOND": 89.0, "MIN_DURATION": 74, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 124, "MAX_DURATION": 189, "ELEMENTS_PER_SECOND": 89.0, "THROUGHPUT": 261616.0, "COUNT": 89}, {"from": 104000, "to": 105000, "AVG_TTFB": 79.22857, "ERRORS": 70, "AVG_DURATION": 143.07143, "ERRORS_PER_SECOND": 70.0, "MIN_DURATION": 100, "ERROR_RATE": 100.0, "MIN_TTFB": 46, "MAX_TTFB": 222, "MAX_DURATION": 308, "ELEMENTS_PER_SECOND": 70.0, "THROUGHPUT": 205684.0, "COUNT": 70}, {"from": 105000, "to": 106000, "AVG_TTFB": 76.25, "ERRORS": 76, "AVG_DURATION": 133.85527, "ERRORS_PER_SECOND": 76.0, "MIN_DURATION": 75, "ERROR_RATE": 100.0, "MIN_TTFB": 32, "MAX_TTFB": 225, "MAX_DURATION": 275, "ELEMENTS_PER_SECOND": 76.0, "THROUGHPUT": 223396.0, "COUNT": 76}, {"from": 106000, "to": 107000, "AVG_TTFB": 47.123894, "ERRORS": 113, "AVG_DURATION": 86.76106, "ERRORS_PER_SECOND": 113.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 152, "MAX_DURATION": 186, "ELEMENTS_PER_SECOND": 113.0, "THROUGHPUT": 332165.0, "COUNT": 113}, {"from": 107000, "to": 108000, "AVG_TTFB": 47.034782, "ERRORS": 115, "AVG_DURATION": 85.58261, "ERRORS_PER_SECOND": 115.0, "MIN_DURATION": 64, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 170, "MAX_DURATION": 213, "ELEMENTS_PER_SECOND": 115.0, "THROUGHPUT": 338021.0, "COUNT": 115}, {"from": 108000, "to": 109000, "AVG_TTFB": 49.545456, "ERRORS": 110, "AVG_DURATION": 90.39091, "ERRORS_PER_SECOND": 110.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 136, "MAX_DURATION": 185, "ELEMENTS_PER_SECOND": 110.0, "THROUGHPUT": 323362.0, "COUNT": 110}, {"from": 109000, "to": 110000, "AVG_TTFB": 54.798077, "ERRORS": 104, "AVG_DURATION": 97.70192, "ERRORS_PER_SECOND": 104.0, "MIN_DURATION": 72, "ERROR_RATE": 100.0, "MIN_TTFB": 32, "MAX_TTFB": 133, "MAX_DURATION": 168, "ELEMENTS_PER_SECOND": 104.0, "THROUGHPUT": 305624.0, "COUNT": 104}, {"from": 110000, "to": 111000, "AVG_TTFB": 54.705265, "ERRORS": 95, "AVG_DURATION": 101.642105, "ERRORS_PER_SECOND": 95.0, "MIN_DURATION": 67, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 138, "MAX_DURATION": 181, "ELEMENTS_PER_SECOND": 95.0, "THROUGHPUT": 279156.0, "COUNT": 95}, {"from": 111000, "to": 112000, "AVG_TTFB": 45.419353, "ERRORS": 124, "AVG_DURATION": 82.53226, "ERRORS_PER_SECOND": 124.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 106, "MAX_DURATION": 156, "ELEMENTS_PER_SECOND": 124.0, "THROUGHPUT": 364335.0, "COUNT": 124}, {"from": 112000, "to": 113000, "AVG_TTFB": 49.666668, "ERRORS": 114, "AVG_DURATION": 87.81579, "ERRORS_PER_SECOND": 114.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 173, "MAX_DURATION": 206, "ELEMENTS_PER_SECOND": 114.0, "THROUGHPUT": 334910.0, "COUNT": 114}, {"from": 113000, "to": 114000, "AVG_TTFB": 47.69298, "ERRORS": 114, "AVG_DURATION": 86.90351, "ERRORS_PER_SECOND": 114.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 129, "MAX_DURATION": 176, "ELEMENTS_PER_SECOND": 114.0, "THROUGHPUT": 334898.0, "COUNT": 114}, {"from": 114000, "to": 115000, "AVG_TTFB": 54.03125, "ERRORS": 96, "AVG_DURATION": 101.510414, "ERRORS_PER_SECOND": 96.0, "MIN_DURATION": 65, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 126, "MAX_DURATION": 178, "ELEMENTS_PER_SECOND": 96.0, "THROUGHPUT": 282090.0, "COUNT": 96}, {"from": 115000, "to": 116000, "AVG_TTFB": 59.065216, "ERRORS": 92, "AVG_DURATION": 109.565216, "ERRORS_PER_SECOND": 92.0, "MIN_DURATION": 80, "ERROR_RATE": 100.0, "MIN_TTFB": 35, "MAX_TTFB": 161, "MAX_DURATION": 211, "ELEMENTS_PER_SECOND": 92.0, "THROUGHPUT": 270243.0, "COUNT": 92}, {"from": 116000, "to": 117000, "AVG_TTFB": 71.638885, "ERRORS": 72, "AVG_DURATION": 131.66667, "ERRORS_PER_SECOND": 72.0, "MIN_DURATION": 80, "ERROR_RATE": 100.0, "MIN_TTFB": 40, "MAX_TTFB": 158, "MAX_DURATION": 208, "ELEMENTS_PER_SECOND": 72.0, "THROUGHPUT": 211534.0, "COUNT": 72}, {"from": 117000, "to": 118000, "AVG_TTFB": 80.62666, "ERRORS": 75, "AVG_DURATION": 139.78667, "ERRORS_PER_SECOND": 75.0, "MIN_DURATION": 95, "ERROR_RATE": 100.0, "MIN_TTFB": 44, "MAX_TTFB": 219, "MAX_DURATION": 294, "ELEMENTS_PER_SECOND": 75.0, "THROUGHPUT": 220336.0, "COUNT": 75}, {"from": 118000, "to": 119000, "AVG_TTFB": 72.16216, "ERRORS": 74, "AVG_DURATION": 131.06757, "ERRORS_PER_SECOND": 74.0, "MIN_DURATION": 102, "ERROR_RATE": 100.0, "MIN_TTFB": 47, "MAX_TTFB": 139, "MAX_DURATION": 193, "ELEMENTS_PER_SECOND": 74.0, "THROUGHPUT": 217382.0, "COUNT": 74}, {"from": 119000, "to": 120000, "AVG_TTFB": 84.80882, "ERRORS": 51, "AVG_DURATION": 151.79411, "ERRORS_PER_SECOND": 51.0, "MIN_DURATION": 107, "ERROR_RATE": 75.0, "MIN_TTFB": 53, "MAX_TTFB": 240, "MAX_DURATION": 331, "ELEMENTS_PER_SECOND": 68.0, "THROUGHPUT": 176049.0, "COUNT": 68}, {"from": 120000, "to": 121000, "AVG_TTFB": 74.0, "ERRORS": 0, "AVG_DURATION": 190.0, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 190, "ERROR_RATE": 0.0, "MIN_TTFB": 74, "MAX_TTFB": 74, "MAX_DURATION": 190, "ELEMENTS_PER_SECOND": 1.0, "THROUGHPUT": 1535.0, "COUNT": 1}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements/6a0ef850-8644-4236-80e5-53748fbd2d01/values':
            return '{"count": 3588, "elementPerSecond": 29.864162, "minDuration": 58, "maxDuration": 436, "sumDuration": 269602, "avgDuration": 75.13991, "minTTFB": 28, "maxTTFB": 384, "sumTTFB": 135410, "avgTTFB": 37.73969, "sumDownloadedBytes": 10490844, "downloadedBytesPerSecond": 87318.914, "successCount": 38, "successPerSecond": 0.31628713, "successRate": 1.0590858, "failureCount": 3550, "failurePerSecond": 29.547876, "failureRate": 98.94092, "percentile50": 70.0, "percentile90": 90.0, "percentile95": 113.0, "percentile99": 150.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements/818d2419-daec-43d6-b004-b1195799df57/values':
            return '{"count": 3586, "elementPerSecond": 29.847517, "minDuration": 58, "maxDuration": 335, "sumDuration": 270440, "avgDuration": 75.415504, "minTTFB": 27, "maxTTFB": 301, "sumTTFB": 136065, "avgTTFB": 37.94339, "sumDownloadedBytes": 10493621, "downloadedBytesPerSecond": 87342.03, "successCount": 32, "successPerSecond": 0.26634705, "successRate": 0.8923592, "failureCount": 3554, "failurePerSecond": 29.58117, "failureRate": 99.10764, "percentile50": 70.0, "percentile90": 91.0, "percentile95": 116.0, "percentile99": 161.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements/e66cb9ec-3c72-49b4-b3a2-5fa91e2fd911/values':
            return '{"count": 3592, "elementPerSecond": 29.897457, "minDuration": 83, "maxDuration": 398, "sumDuration": 381736, "avgDuration": 106.27394, "minTTFB": 52, "maxTTFB": 324, "sumTTFB": 247696, "avgTTFB": 68.95769, "sumDownloadedBytes": 10498963, "downloadedBytesPerSecond": 87386.49, "successCount": 41, "successPerSecond": 0.34125715, "successRate": 1.1414254, "failureCount": 3551, "failurePerSecond": 29.5562, "failureRate": 98.858574, "percentile50": 99.0, "percentile90": 130.0, "percentile95": 160.0, "percentile99": 206.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements/7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a/values':
            return '{"count": 3591, "elementPerSecond": 29.889133, "minDuration": 58, "maxDuration": 401, "sumDuration": 269676, "avgDuration": 75.09775, "minTTFB": 28, "maxTTFB": 313, "sumTTFB": 135130, "avgTTFB": 37.630188, "sumDownloadedBytes": 10495718, "downloadedBytesPerSecond": 87359.484, "successCount": 41, "successPerSecond": 0.34125715, "successRate": 1.1417433, "failureCount": 3550, "failurePerSecond": 29.547876, "failureRate": 98.85825, "percentile50": 70.0, "percentile90": 90.0, "percentile95": 111.0, "percentile99": 156.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements/6a0ef850-8644-4236-80e5-53748fbd2d01/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 1000, "AVG_TTFB": 38.454544, "ERRORS": 0, "AVG_DURATION": 77.181816, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 72, "ERROR_RATE": 0.0, "MIN_TTFB": 32, "MAX_TTFB": 47, "MAX_DURATION": 86, "ELEMENTS_PER_SECOND": 11.0, "THROUGHPUT": 16916.0, "COUNT": 11}, {"from": 1000, "to": 2000, "AVG_TTFB": 34.31579, "ERRORS": 31, "AVG_DURATION": 72.07895, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 60, "ERROR_RATE": 81.57895, "MIN_TTFB": 29, "MAX_TTFB": 42, "MAX_DURATION": 93, "ELEMENTS_PER_SECOND": 38.0, "THROUGHPUT": 101889.0, "COUNT": 38}, {"from": 2000, "to": 3000, "AVG_TTFB": 33.870968, "ERRORS": 31, "AVG_DURATION": 70.32258, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 57, "MAX_DURATION": 101, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91120.0, "COUNT": 31}, {"from": 3000, "to": 4000, "AVG_TTFB": 36.866665, "ERRORS": 30, "AVG_DURATION": 72.066666, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 70, "MAX_DURATION": 103, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88173.0, "COUNT": 30}, {"from": 4000, "to": 5000, "AVG_TTFB": 32.433334, "ERRORS": 30, "AVG_DURATION": 66.166664, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 39, "MAX_DURATION": 74, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88146.0, "COUNT": 30}, {"from": 5000, "to": 6000, "AVG_TTFB": 36.233334, "ERRORS": 30, "AVG_DURATION": 71.53333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 60, "MAX_DURATION": 94, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88146.0, "COUNT": 30}, {"from": 6000, "to": 7000, "AVG_TTFB": 34.243244, "ERRORS": 37, "AVG_DURATION": 68.75676, "ERRORS_PER_SECOND": 37.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 52, "MAX_DURATION": 83, "ELEMENTS_PER_SECOND": 37.0, "THROUGHPUT": 108742.0, "COUNT": 37}, {"from": 7000, "to": 8000, "AVG_TTFB": 34.0, "ERRORS": 33, "AVG_DURATION": 69.030304, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 45, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96982.0, "COUNT": 33}, {"from": 8000, "to": 9000, "AVG_TTFB": 34.433334, "ERRORS": 30, "AVG_DURATION": 67.933334, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 60, "MAX_DURATION": 90, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88143.0, "COUNT": 30}, {"from": 9000, "to": 10000, "AVG_TTFB": 34.272728, "ERRORS": 33, "AVG_DURATION": 67.72727, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 59, "MAX_DURATION": 89, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96993.0, "COUNT": 33}, {"from": 10000, "to": 11000, "AVG_TTFB": 33.942856, "ERRORS": 35, "AVG_DURATION": 69.65714, "ERRORS_PER_SECOND": 35.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 60, "MAX_DURATION": 95, "ELEMENTS_PER_SECOND": 35.0, "THROUGHPUT": 102869.0, "COUNT": 35}, {"from": 11000, "to": 12000, "AVG_TTFB": 35.625, "ERRORS": 32, "AVG_DURATION": 71.875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 62, "MAX_DURATION": 109, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94031.0, "COUNT": 32}, {"from": 12000, "to": 13000, "AVG_TTFB": 36.2, "ERRORS": 30, "AVG_DURATION": 71.066666, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 87, "MAX_DURATION": 121, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88154.0, "COUNT": 30}, {"from": 13000, "to": 14000, "AVG_TTFB": 33.909092, "ERRORS": 33, "AVG_DURATION": 67.757576, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 61, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96957.0, "COUNT": 33}, {"from": 14000, "to": 15000, "AVG_TTFB": 34.303032, "ERRORS": 33, "AVG_DURATION": 68.0, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 78, "MAX_DURATION": 109, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96991.0, "COUNT": 33}, {"from": 15000, "to": 16000, "AVG_TTFB": 33.333332, "ERRORS": 33, "AVG_DURATION": 66.27273, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 42, "MAX_DURATION": 75, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 97011.0, "COUNT": 33}, {"from": 16000, "to": 17000, "AVG_TTFB": 72.65, "ERRORS": 20, "AVG_DURATION": 142.45, "ERRORS_PER_SECOND": 20.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 305, "MAX_DURATION": 436, "ELEMENTS_PER_SECOND": 20.0, "THROUGHPUT": 58771.0, "COUNT": 20}, {"from": 17000, "to": 18000, "AVG_TTFB": 35.19355, "ERRORS": 31, "AVG_DURATION": 70.3871, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 45, "MAX_DURATION": 82, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91058.0, "COUNT": 31}, {"from": 18000, "to": 19000, "AVG_TTFB": 34.612904, "ERRORS": 31, "AVG_DURATION": 70.258064, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 42, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91111.0, "COUNT": 31}, {"from": 19000, "to": 20000, "AVG_TTFB": 36.3125, "ERRORS": 32, "AVG_DURATION": 70.0, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 94, "MAX_DURATION": 127, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94007.0, "COUNT": 32}, {"from": 20000, "to": 21000, "AVG_TTFB": 37.096775, "ERRORS": 31, "AVG_DURATION": 74.258064, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 75, "MAX_DURATION": 144, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91100.0, "COUNT": 31}, {"from": 21000, "to": 22000, "AVG_TTFB": 50.851852, "ERRORS": 27, "AVG_DURATION": 101.55556, "ERRORS_PER_SECOND": 27.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 268, "MAX_DURATION": 305, "ELEMENTS_PER_SECOND": 27.0, "THROUGHPUT": 79347.0, "COUNT": 27}, {"from": 22000, "to": 23000, "AVG_TTFB": 34.705883, "ERRORS": 34, "AVG_DURATION": 70.5, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 65, "MAX_DURATION": 100, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99923.0, "COUNT": 34}, {"from": 23000, "to": 24000, "AVG_TTFB": 36.129032, "ERRORS": 31, "AVG_DURATION": 70.3871, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 68, "MAX_DURATION": 101, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91104.0, "COUNT": 31}, {"from": 24000, "to": 25000, "AVG_TTFB": 33.14706, "ERRORS": 34, "AVG_DURATION": 67.05882, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 41, "MAX_DURATION": 75, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99934.0, "COUNT": 34}, {"from": 25000, "to": 26000, "AVG_TTFB": 34.19355, "ERRORS": 31, "AVG_DURATION": 68.80645, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 66, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91082.0, "COUNT": 31}, {"from": 26000, "to": 27000, "AVG_TTFB": 34.733334, "ERRORS": 30, "AVG_DURATION": 77.76667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 47, "MAX_DURATION": 145, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88174.0, "COUNT": 30}, {"from": 27000, "to": 28000, "AVG_TTFB": 36.117645, "ERRORS": 34, "AVG_DURATION": 71.5, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 78, "MAX_DURATION": 113, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99928.0, "COUNT": 34}, {"from": 28000, "to": 29000, "AVG_TTFB": 34.451614, "ERRORS": 31, "AVG_DURATION": 69.41936, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 44, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91078.0, "COUNT": 31}, {"from": 29000, "to": 30000, "AVG_TTFB": 44.23077, "ERRORS": 26, "AVG_DURATION": 96.46154, "ERRORS_PER_SECOND": 26.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 238, "MAX_DURATION": 287, "ELEMENTS_PER_SECOND": 26.0, "THROUGHPUT": 76449.0, "COUNT": 26}, {"from": 30000, "to": 31000, "AVG_TTFB": 36.8, "ERRORS": 30, "AVG_DURATION": 73.26667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 57, "MAX_DURATION": 103, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88159.0, "COUNT": 30}, {"from": 31000, "to": 32000, "AVG_TTFB": 34.8125, "ERRORS": 32, "AVG_DURATION": 67.375, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 74, "MAX_DURATION": 105, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94016.0, "COUNT": 32}, {"from": 32000, "to": 33000, "AVG_TTFB": 35.114285, "ERRORS": 35, "AVG_DURATION": 70.057144, "ERRORS_PER_SECOND": 35.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 69, "MAX_DURATION": 99, "ELEMENTS_PER_SECOND": 35.0, "THROUGHPUT": 102879.0, "COUNT": 35}, {"from": 33000, "to": 34000, "AVG_TTFB": 33.78788, "ERRORS": 33, "AVG_DURATION": 68.545456, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 40, "MAX_DURATION": 75, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 97007.0, "COUNT": 33}, {"from": 34000, "to": 35000, "AVG_TTFB": 35.1, "ERRORS": 30, "AVG_DURATION": 70.7, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 46, "MAX_DURATION": 88, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88196.0, "COUNT": 30}, {"from": 35000, "to": 36000, "AVG_TTFB": 39.3, "ERRORS": 30, "AVG_DURATION": 75.666664, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 112, "MAX_DURATION": 144, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88180.0, "COUNT": 30}, {"from": 36000, "to": 37000, "AVG_TTFB": 36.51613, "ERRORS": 31, "AVG_DURATION": 70.80645, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 81, "MAX_DURATION": 114, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91118.0, "COUNT": 31}, {"from": 37000, "to": 38000, "AVG_TTFB": 33.37143, "ERRORS": 35, "AVG_DURATION": 67.0, "ERRORS_PER_SECOND": 35.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 39, "MAX_DURATION": 79, "ELEMENTS_PER_SECOND": 35.0, "THROUGHPUT": 102850.0, "COUNT": 35}, {"from": 38000, "to": 39000, "AVG_TTFB": 34.5, "ERRORS": 32, "AVG_DURATION": 68.78125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 39, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94010.0, "COUNT": 32}, {"from": 39000, "to": 40000, "AVG_TTFB": 34.5625, "ERRORS": 32, "AVG_DURATION": 67.46875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 79, "MAX_DURATION": 112, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94058.0, "COUNT": 32}, {"from": 40000, "to": 41000, "AVG_TTFB": 37.15625, "ERRORS": 32, "AVG_DURATION": 70.90625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 95, "MAX_DURATION": 131, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94028.0, "COUNT": 32}, {"from": 41000, "to": 42000, "AVG_TTFB": 33.0625, "ERRORS": 32, "AVG_DURATION": 66.53125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 39, "MAX_DURATION": 76, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94046.0, "COUNT": 32}, {"from": 42000, "to": 43000, "AVG_TTFB": 34.0, "ERRORS": 32, "AVG_DURATION": 68.4375, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 41, "MAX_DURATION": 79, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94013.0, "COUNT": 32}, {"from": 43000, "to": 44000, "AVG_TTFB": 34.57143, "ERRORS": 35, "AVG_DURATION": 69.171425, "ERRORS_PER_SECOND": 35.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 43, "MAX_DURATION": 85, "ELEMENTS_PER_SECOND": 35.0, "THROUGHPUT": 102907.0, "COUNT": 35}, {"from": 44000, "to": 45000, "AVG_TTFB": 33.580647, "ERRORS": 31, "AVG_DURATION": 68.70968, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 40, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91110.0, "COUNT": 31}, {"from": 45000, "to": 46000, "AVG_TTFB": 33.419353, "ERRORS": 31, "AVG_DURATION": 67.90323, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 38, "MAX_DURATION": 76, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91096.0, "COUNT": 31}, {"from": 46000, "to": 47000, "AVG_TTFB": 35.0, "ERRORS": 34, "AVG_DURATION": 69.35294, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 86, "MAX_DURATION": 118, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99907.0, "COUNT": 34}, {"from": 47000, "to": 48000, "AVG_TTFB": 37.166668, "ERRORS": 30, "AVG_DURATION": 73.2, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 48, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88205.0, "COUNT": 30}, {"from": 48000, "to": 49000, "AVG_TTFB": 34.90625, "ERRORS": 32, "AVG_DURATION": 71.0, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 51, "MAX_DURATION": 89, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94045.0, "COUNT": 32}, {"from": 49000, "to": 50000, "AVG_TTFB": 39.21875, "ERRORS": 32, "AVG_DURATION": 74.5, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 94, "MAX_DURATION": 130, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94036.0, "COUNT": 32}, {"from": 50000, "to": 51000, "AVG_TTFB": 40.551723, "ERRORS": 29, "AVG_DURATION": 79.10345, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 110, "MAX_DURATION": 143, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85184.0, "COUNT": 29}, {"from": 51000, "to": 52000, "AVG_TTFB": 36.875, "ERRORS": 32, "AVG_DURATION": 72.25, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 101, "MAX_DURATION": 139, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94033.0, "COUNT": 32}, {"from": 52000, "to": 53000, "AVG_TTFB": 38.0, "ERRORS": 26, "AVG_DURATION": 79.38461, "ERRORS_PER_SECOND": 26.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 102, "MAX_DURATION": 202, "ELEMENTS_PER_SECOND": 26.0, "THROUGHPUT": 76394.0, "COUNT": 26}, {"from": 53000, "to": 54000, "AVG_TTFB": 37.303032, "ERRORS": 33, "AVG_DURATION": 73.51515, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 108, "MAX_DURATION": 140, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96938.0, "COUNT": 33}, {"from": 54000, "to": 55000, "AVG_TTFB": 37.466667, "ERRORS": 30, "AVG_DURATION": 72.23333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 93, "MAX_DURATION": 126, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88112.0, "COUNT": 30}, {"from": 55000, "to": 56000, "AVG_TTFB": 36.0625, "ERRORS": 32, "AVG_DURATION": 70.21875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 89, "MAX_DURATION": 121, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94008.0, "COUNT": 32}, {"from": 56000, "to": 57000, "AVG_TTFB": 38.46875, "ERRORS": 32, "AVG_DURATION": 73.46875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 111, "MAX_DURATION": 149, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94008.0, "COUNT": 32}, {"from": 57000, "to": 58000, "AVG_TTFB": 36.096775, "ERRORS": 31, "AVG_DURATION": 72.19355, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 46, "MAX_DURATION": 89, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91099.0, "COUNT": 31}, {"from": 58000, "to": 59000, "AVG_TTFB": 39.366665, "ERRORS": 30, "AVG_DURATION": 75.36667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 64, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 89, "MAX_DURATION": 124, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88101.0, "COUNT": 30}, {"from": 59000, "to": 60000, "AVG_TTFB": 39.142857, "ERRORS": 23, "AVG_DURATION": 76.35714, "ERRORS_PER_SECOND": 23.0, "MIN_DURATION": 65, "ERROR_RATE": 82.14286, "MIN_TTFB": 30, "MAX_TTFB": 93, "MAX_DURATION": 130, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 75235.0, "COUNT": 28}, {"from": 60000, "to": 61000, "AVG_TTFB": 35.451614, "ERRORS": 21, "AVG_DURATION": 74.3871, "ERRORS_PER_SECOND": 21.0, "MIN_DURATION": 62, "ERROR_RATE": 67.741936, "MIN_TTFB": 30, "MAX_TTFB": 64, "MAX_DURATION": 107, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 77079.0, "COUNT": 31}, {"from": 61000, "to": 62000, "AVG_TTFB": 35.78125, "ERRORS": 32, "AVG_DURATION": 72.0, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 46, "MAX_DURATION": 85, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94086.0, "COUNT": 32}, {"from": 62000, "to": 63000, "AVG_TTFB": 37.29032, "ERRORS": 31, "AVG_DURATION": 73.67742, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 104, "MAX_DURATION": 137, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91102.0, "COUNT": 31}, {"from": 63000, "to": 64000, "AVG_TTFB": 35.21875, "ERRORS": 32, "AVG_DURATION": 69.9375, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 95, "MAX_DURATION": 126, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94086.0, "COUNT": 32}, {"from": 64000, "to": 65000, "AVG_TTFB": 37.264706, "ERRORS": 34, "AVG_DURATION": 70.73529, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 101, "MAX_DURATION": 133, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99942.0, "COUNT": 34}, {"from": 65000, "to": 66000, "AVG_TTFB": 39.57143, "ERRORS": 28, "AVG_DURATION": 77.92857, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 125, "MAX_DURATION": 159, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82281.0, "COUNT": 28}, {"from": 66000, "to": 67000, "AVG_TTFB": 37.0625, "ERRORS": 32, "AVG_DURATION": 73.25, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 65, "MAX_DURATION": 100, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94060.0, "COUNT": 32}, {"from": 67000, "to": 68000, "AVG_TTFB": 36.548386, "ERRORS": 31, "AVG_DURATION": 71.41936, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 100, "MAX_DURATION": 132, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91099.0, "COUNT": 31}, {"from": 68000, "to": 69000, "AVG_TTFB": 36.233334, "ERRORS": 30, "AVG_DURATION": 71.86667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 58, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88153.0, "COUNT": 30}, {"from": 69000, "to": 70000, "AVG_TTFB": 36.40625, "ERRORS": 32, "AVG_DURATION": 73.375, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 48, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94038.0, "COUNT": 32}, {"from": 70000, "to": 71000, "AVG_TTFB": 35.068966, "ERRORS": 29, "AVG_DURATION": 72.75862, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 64, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 40, "MAX_DURATION": 101, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85199.0, "COUNT": 29}, {"from": 71000, "to": 72000, "AVG_TTFB": 35.424244, "ERRORS": 33, "AVG_DURATION": 71.969696, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 48, "MAX_DURATION": 85, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96957.0, "COUNT": 33}, {"from": 72000, "to": 73000, "AVG_TTFB": 35.4, "ERRORS": 30, "AVG_DURATION": 72.166664, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 50, "MAX_DURATION": 119, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88164.0, "COUNT": 30}, {"from": 73000, "to": 74000, "AVG_TTFB": 35.068966, "ERRORS": 29, "AVG_DURATION": 71.68965, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 41, "MAX_DURATION": 82, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85221.0, "COUNT": 29}, {"from": 74000, "to": 75000, "AVG_TTFB": 38.454544, "ERRORS": 33, "AVG_DURATION": 73.878784, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 122, "MAX_DURATION": 164, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 97002.0, "COUNT": 33}, {"from": 75000, "to": 76000, "AVG_TTFB": 33.545456, "ERRORS": 33, "AVG_DURATION": 67.21212, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 40, "MAX_DURATION": 77, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96987.0, "COUNT": 33}, {"from": 76000, "to": 77000, "AVG_TTFB": 41.51613, "ERRORS": 31, "AVG_DURATION": 77.3871, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 148, "MAX_DURATION": 183, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91109.0, "COUNT": 31}, {"from": 77000, "to": 78000, "AVG_TTFB": 34.8125, "ERRORS": 32, "AVG_DURATION": 69.625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 55, "MAX_DURATION": 97, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94038.0, "COUNT": 32}, {"from": 78000, "to": 79000, "AVG_TTFB": 40.310345, "ERRORS": 29, "AVG_DURATION": 78.0, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 120, "MAX_DURATION": 160, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85183.0, "COUNT": 29}, {"from": 79000, "to": 80000, "AVG_TTFB": 36.333332, "ERRORS": 33, "AVG_DURATION": 71.030304, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 103, "MAX_DURATION": 133, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96974.0, "COUNT": 33}, {"from": 80000, "to": 81000, "AVG_TTFB": 38.931034, "ERRORS": 29, "AVG_DURATION": 75.72414, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 91, "MAX_DURATION": 130, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85244.0, "COUNT": 29}, {"from": 81000, "to": 82000, "AVG_TTFB": 34.085712, "ERRORS": 35, "AVG_DURATION": 69.22857, "ERRORS_PER_SECOND": 35.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 44, "MAX_DURATION": 83, "ELEMENTS_PER_SECOND": 35.0, "THROUGHPUT": 102866.0, "COUNT": 35}, {"from": 82000, "to": 83000, "AVG_TTFB": 38.5, "ERRORS": 30, "AVG_DURATION": 73.86667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 70, "MAX_DURATION": 108, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88143.0, "COUNT": 30}, {"from": 83000, "to": 84000, "AVG_TTFB": 34.83871, "ERRORS": 31, "AVG_DURATION": 71.22581, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 41, "MAX_DURATION": 86, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91110.0, "COUNT": 31}, {"from": 84000, "to": 85000, "AVG_TTFB": 35.774193, "ERRORS": 31, "AVG_DURATION": 71.67742, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 53, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91123.0, "COUNT": 31}, {"from": 85000, "to": 86000, "AVG_TTFB": 37.741936, "ERRORS": 31, "AVG_DURATION": 73.90323, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 64, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 113, "MAX_DURATION": 154, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91109.0, "COUNT": 31}, {"from": 86000, "to": 87000, "AVG_TTFB": 33.82353, "ERRORS": 34, "AVG_DURATION": 67.882355, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 40, "MAX_DURATION": 81, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99894.0, "COUNT": 34}, {"from": 87000, "to": 88000, "AVG_TTFB": 33.129032, "ERRORS": 31, "AVG_DURATION": 67.29032, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 39, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91085.0, "COUNT": 31}, {"from": 88000, "to": 89000, "AVG_TTFB": 33.59375, "ERRORS": 32, "AVG_DURATION": 67.09375, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 46, "MAX_DURATION": 79, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94016.0, "COUNT": 32}, {"from": 89000, "to": 90000, "AVG_TTFB": 47.84, "ERRORS": 25, "AVG_DURATION": 92.84, "ERRORS_PER_SECOND": 25.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 384, "MAX_DURATION": 415, "ELEMENTS_PER_SECOND": 25.0, "THROUGHPUT": 73473.0, "COUNT": 25}, {"from": 90000, "to": 91000, "AVG_TTFB": 36.545456, "ERRORS": 33, "AVG_DURATION": 70.969696, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 113, "MAX_DURATION": 150, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96963.0, "COUNT": 33}, {"from": 91000, "to": 92000, "AVG_TTFB": 33.18182, "ERRORS": 33, "AVG_DURATION": 67.969696, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 38, "MAX_DURATION": 77, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96975.0, "COUNT": 33}, {"from": 92000, "to": 93000, "AVG_TTFB": 34.25, "ERRORS": 32, "AVG_DURATION": 67.75, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 40, "MAX_DURATION": 76, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94009.0, "COUNT": 32}, {"from": 93000, "to": 94000, "AVG_TTFB": 37.090908, "ERRORS": 33, "AVG_DURATION": 71.09091, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 120, "MAX_DURATION": 154, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96988.0, "COUNT": 33}, {"from": 94000, "to": 95000, "AVG_TTFB": 39.633335, "ERRORS": 30, "AVG_DURATION": 77.666664, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 120, "MAX_DURATION": 192, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88130.0, "COUNT": 30}, {"from": 95000, "to": 96000, "AVG_TTFB": 33.70968, "ERRORS": 31, "AVG_DURATION": 68.19355, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 42, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91102.0, "COUNT": 31}, {"from": 96000, "to": 97000, "AVG_TTFB": 36.19355, "ERRORS": 31, "AVG_DURATION": 72.064514, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 48, "MAX_DURATION": 91, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91157.0, "COUNT": 31}, {"from": 97000, "to": 98000, "AVG_TTFB": 37.060608, "ERRORS": 33, "AVG_DURATION": 71.15151, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 127, "MAX_DURATION": 160, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96986.0, "COUNT": 33}, {"from": 98000, "to": 99000, "AVG_TTFB": 34.3, "ERRORS": 30, "AVG_DURATION": 69.13333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 40, "MAX_DURATION": 78, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88192.0, "COUNT": 30}, {"from": 99000, "to": 100000, "AVG_TTFB": 41.566666, "ERRORS": 30, "AVG_DURATION": 79.1, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 138, "MAX_DURATION": 176, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88159.0, "COUNT": 30}, {"from": 100000, "to": 101000, "AVG_TTFB": 38.827587, "ERRORS": 29, "AVG_DURATION": 77.034485, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 53, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85257.0, "COUNT": 29}, {"from": 101000, "to": 102000, "AVG_TTFB": 45.47619, "ERRORS": 21, "AVG_DURATION": 93.14286, "ERRORS_PER_SECOND": 21.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 32, "MAX_TTFB": 127, "MAX_DURATION": 167, "ELEMENTS_PER_SECOND": 21.0, "THROUGHPUT": 61688.0, "COUNT": 21}, {"from": 102000, "to": 103000, "AVG_TTFB": 57.45, "ERRORS": 20, "AVG_DURATION": 110.5, "ERRORS_PER_SECOND": 20.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 43, "MAX_TTFB": 139, "MAX_DURATION": 185, "ELEMENTS_PER_SECOND": 20.0, "THROUGHPUT": 58775.0, "COUNT": 20}, {"from": 103000, "to": 104000, "AVG_TTFB": 46.916668, "ERRORS": 24, "AVG_DURATION": 96.208336, "ERRORS_PER_SECOND": 24.0, "MIN_DURATION": 74, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 65, "MAX_DURATION": 127, "ELEMENTS_PER_SECOND": 24.0, "THROUGHPUT": 70525.0, "COUNT": 24}, {"from": 104000, "to": 105000, "AVG_TTFB": 58.625, "ERRORS": 16, "AVG_DURATION": 121.875, "ERRORS_PER_SECOND": 16.0, "MIN_DURATION": 104, "ERROR_RATE": 100.0, "MIN_TTFB": 50, "MAX_TTFB": 73, "MAX_DURATION": 143, "ELEMENTS_PER_SECOND": 16.0, "THROUGHPUT": 47010.0, "COUNT": 16}, {"from": 105000, "to": 106000, "AVG_TTFB": 59.55, "ERRORS": 20, "AVG_DURATION": 118.65, "ERRORS_PER_SECOND": 20.0, "MIN_DURATION": 75, "ERROR_RATE": 100.0, "MIN_TTFB": 32, "MAX_TTFB": 73, "MAX_DURATION": 136, "ELEMENTS_PER_SECOND": 20.0, "THROUGHPUT": 58838.0, "COUNT": 20}, {"from": 106000, "to": 107000, "AVG_TTFB": 37.892857, "ERRORS": 28, "AVG_DURATION": 78.07143, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 66, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 56, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82296.0, "COUNT": 28}, {"from": 107000, "to": 108000, "AVG_TTFB": 38.827587, "ERRORS": 29, "AVG_DURATION": 77.0, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 64, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 127, "MAX_DURATION": 161, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85228.0, "COUNT": 29}, {"from": 108000, "to": 109000, "AVG_TTFB": 40.566666, "ERRORS": 30, "AVG_DURATION": 80.2, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 53, "MAX_DURATION": 99, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88190.0, "COUNT": 30}, {"from": 109000, "to": 110000, "AVG_TTFB": 43.652172, "ERRORS": 23, "AVG_DURATION": 87.347824, "ERRORS_PER_SECOND": 23.0, "MIN_DURATION": 72, "ERROR_RATE": 100.0, "MIN_TTFB": 32, "MAX_TTFB": 56, "MAX_DURATION": 109, "ELEMENTS_PER_SECOND": 23.0, "THROUGHPUT": 67615.0, "COUNT": 23}, {"from": 110000, "to": 111000, "AVG_TTFB": 44.703705, "ERRORS": 27, "AVG_DURATION": 90.333336, "ERRORS_PER_SECOND": 27.0, "MIN_DURATION": 70, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 65, "MAX_DURATION": 107, "ELEMENTS_PER_SECOND": 27.0, "THROUGHPUT": 79332.0, "COUNT": 27}, {"from": 111000, "to": 112000, "AVG_TTFB": 35.433334, "ERRORS": 30, "AVG_DURATION": 70.86667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 54, "MAX_DURATION": 98, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88152.0, "COUNT": 30}, {"from": 112000, "to": 113000, "AVG_TTFB": 38.75, "ERRORS": 28, "AVG_DURATION": 76.35714, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 66, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 46, "MAX_DURATION": 88, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82262.0, "COUNT": 28}, {"from": 113000, "to": 114000, "AVG_TTFB": 41.32143, "ERRORS": 28, "AVG_DURATION": 80.60714, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 129, "MAX_DURATION": 176, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82282.0, "COUNT": 28}, {"from": 114000, "to": 115000, "AVG_TTFB": 45.291668, "ERRORS": 24, "AVG_DURATION": 93.5, "ERRORS_PER_SECOND": 24.0, "MIN_DURATION": 66, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 63, "MAX_DURATION": 140, "ELEMENTS_PER_SECOND": 24.0, "THROUGHPUT": 70518.0, "COUNT": 24}, {"from": 115000, "to": 116000, "AVG_TTFB": 51.739132, "ERRORS": 23, "AVG_DURATION": 102.04348, "ERRORS_PER_SECOND": 23.0, "MIN_DURATION": 85, "ERROR_RATE": 100.0, "MIN_TTFB": 35, "MAX_TTFB": 161, "MAX_DURATION": 211, "ELEMENTS_PER_SECOND": 23.0, "THROUGHPUT": 67536.0, "COUNT": 23}, {"from": 116000, "to": 117000, "AVG_TTFB": 58.235294, "ERRORS": 17, "AVG_DURATION": 117.17647, "ERRORS_PER_SECOND": 17.0, "MIN_DURATION": 83, "ERROR_RATE": 100.0, "MIN_TTFB": 41, "MAX_TTFB": 77, "MAX_DURATION": 149, "ELEMENTS_PER_SECOND": 17.0, "THROUGHPUT": 49958.0, "COUNT": 17}, {"from": 117000, "to": 118000, "AVG_TTFB": 64.45, "ERRORS": 20, "AVG_DURATION": 124.4, "ERRORS_PER_SECOND": 20.0, "MIN_DURATION": 95, "ERROR_RATE": 100.0, "MIN_TTFB": 49, "MAX_TTFB": 81, "MAX_DURATION": 172, "ELEMENTS_PER_SECOND": 20.0, "THROUGHPUT": 58764.0, "COUNT": 20}, {"from": 118000, "to": 119000, "AVG_TTFB": 59.666668, "ERRORS": 18, "AVG_DURATION": 119.27778, "ERRORS_PER_SECOND": 18.0, "MIN_DURATION": 102, "ERROR_RATE": 100.0, "MIN_TTFB": 47, "MAX_TTFB": 74, "MAX_DURATION": 142, "ELEMENTS_PER_SECOND": 18.0, "THROUGHPUT": 52866.0, "COUNT": 18}, {"from": 119000, "to": 120000, "AVG_TTFB": 65.35294, "ERRORS": 12, "AVG_DURATION": 132.70589, "ERRORS_PER_SECOND": 12.0, "MIN_DURATION": 107, "ERROR_RATE": 70.588234, "MIN_TTFB": 54, "MAX_TTFB": 80, "MAX_DURATION": 154, "ELEMENTS_PER_SECOND": 17.0, "THROUGHPUT": 42961.0, "COUNT": 17}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements/818d2419-daec-43d6-b004-b1195799df57/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 1000, "AVG_TTFB": 36.8, "ERRORS": 0, "AVG_DURATION": 82.1, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 67, "ERROR_RATE": 0.0, "MIN_TTFB": 31, "MAX_TTFB": 55, "MAX_DURATION": 98, "ELEMENTS_PER_SECOND": 10.0, "THROUGHPUT": 15376.0, "COUNT": 10}, {"from": 1000, "to": 2000, "AVG_TTFB": 33.433334, "ERRORS": 30, "AVG_DURATION": 69.566666, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 43, "MAX_DURATION": 87, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88176.0, "COUNT": 30}, {"from": 2000, "to": 3000, "AVG_TTFB": 34.757576, "ERRORS": 33, "AVG_DURATION": 70.15151, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 54, "MAX_DURATION": 87, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96989.0, "COUNT": 33}, {"from": 3000, "to": 4000, "AVG_TTFB": 35.27027, "ERRORS": 37, "AVG_DURATION": 70.297295, "ERRORS_PER_SECOND": 37.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 48, "MAX_DURATION": 83, "ELEMENTS_PER_SECOND": 37.0, "THROUGHPUT": 108739.0, "COUNT": 37}, {"from": 4000, "to": 5000, "AVG_TTFB": 33.266666, "ERRORS": 30, "AVG_DURATION": 67.566666, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 56, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88174.0, "COUNT": 30}, {"from": 5000, "to": 6000, "AVG_TTFB": 36.733334, "ERRORS": 30, "AVG_DURATION": 73.333336, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 62, "MAX_DURATION": 106, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88155.0, "COUNT": 30}, {"from": 6000, "to": 7000, "AVG_TTFB": 33.866665, "ERRORS": 30, "AVG_DURATION": 67.9, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 42, "MAX_DURATION": 76, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88169.0, "COUNT": 30}, {"from": 7000, "to": 8000, "AVG_TTFB": 33.92105, "ERRORS": 38, "AVG_DURATION": 68.42105, "ERRORS_PER_SECOND": 38.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 41, "MAX_DURATION": 81, "ELEMENTS_PER_SECOND": 38.0, "THROUGHPUT": 111684.0, "COUNT": 38}, {"from": 8000, "to": 9000, "AVG_TTFB": 34.59375, "ERRORS": 32, "AVG_DURATION": 67.90625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 57, "MAX_DURATION": 88, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 93992.0, "COUNT": 32}, {"from": 9000, "to": 10000, "AVG_TTFB": 34.133335, "ERRORS": 30, "AVG_DURATION": 68.5, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 42, "MAX_DURATION": 79, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88170.0, "COUNT": 30}, {"from": 10000, "to": 11000, "AVG_TTFB": 34.342857, "ERRORS": 35, "AVG_DURATION": 68.48572, "ERRORS_PER_SECOND": 35.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 46, "MAX_DURATION": 89, "ELEMENTS_PER_SECOND": 35.0, "THROUGHPUT": 102927.0, "COUNT": 35}, {"from": 11000, "to": 12000, "AVG_TTFB": 35.9375, "ERRORS": 32, "AVG_DURATION": 73.3125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 66, "MAX_DURATION": 123, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94050.0, "COUNT": 32}, {"from": 12000, "to": 13000, "AVG_TTFB": 33.35484, "ERRORS": 31, "AVG_DURATION": 69.83871, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 42, "MAX_DURATION": 126, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91099.0, "COUNT": 31}, {"from": 13000, "to": 14000, "AVG_TTFB": 34.28125, "ERRORS": 32, "AVG_DURATION": 70.03125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 54, "MAX_DURATION": 87, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94040.0, "COUNT": 32}, {"from": 14000, "to": 15000, "AVG_TTFB": 32.705883, "ERRORS": 34, "AVG_DURATION": 66.79412, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 46, "MAX_DURATION": 88, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99921.0, "COUNT": 34}, {"from": 15000, "to": 16000, "AVG_TTFB": 33.088234, "ERRORS": 34, "AVG_DURATION": 66.882355, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 41, "MAX_DURATION": 79, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99955.0, "COUNT": 34}, {"from": 16000, "to": 17000, "AVG_TTFB": 59.789474, "ERRORS": 19, "AVG_DURATION": 123.57895, "ERRORS_PER_SECOND": 19.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 135, "MAX_DURATION": 333, "ELEMENTS_PER_SECOND": 19.0, "THROUGHPUT": 55831.0, "COUNT": 19}, {"from": 17000, "to": 18000, "AVG_TTFB": 35.903225, "ERRORS": 31, "AVG_DURATION": 69.70968, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 76, "MAX_DURATION": 107, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91138.0, "COUNT": 31}, {"from": 18000, "to": 19000, "AVG_TTFB": 36.03125, "ERRORS": 32, "AVG_DURATION": 70.21875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 61, "MAX_DURATION": 94, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94018.0, "COUNT": 32}, {"from": 19000, "to": 20000, "AVG_TTFB": 36.387096, "ERRORS": 31, "AVG_DURATION": 72.51613, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 27, "MAX_TTFB": 104, "MAX_DURATION": 147, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91105.0, "COUNT": 31}, {"from": 20000, "to": 21000, "AVG_TTFB": 35.333332, "ERRORS": 30, "AVG_DURATION": 71.63333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 64, "MAX_DURATION": 101, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88165.0, "COUNT": 30}, {"from": 21000, "to": 22000, "AVG_TTFB": 47.655174, "ERRORS": 29, "AVG_DURATION": 89.17242, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 237, "MAX_DURATION": 267, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85222.0, "COUNT": 29}, {"from": 22000, "to": 23000, "AVG_TTFB": 33.741936, "ERRORS": 31, "AVG_DURATION": 67.41936, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 44, "MAX_DURATION": 78, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91079.0, "COUNT": 31}, {"from": 23000, "to": 24000, "AVG_TTFB": 34.794117, "ERRORS": 34, "AVG_DURATION": 67.76471, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 46, "MAX_DURATION": 77, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99912.0, "COUNT": 34}, {"from": 24000, "to": 25000, "AVG_TTFB": 32.9375, "ERRORS": 32, "AVG_DURATION": 67.15625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 41, "MAX_DURATION": 76, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94047.0, "COUNT": 32}, {"from": 25000, "to": 26000, "AVG_TTFB": 32.969696, "ERRORS": 33, "AVG_DURATION": 65.606064, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 40, "MAX_DURATION": 78, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96979.0, "COUNT": 33}, {"from": 26000, "to": 27000, "AVG_TTFB": 35.451614, "ERRORS": 31, "AVG_DURATION": 72.870964, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 83, "MAX_DURATION": 138, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91082.0, "COUNT": 31}, {"from": 27000, "to": 28000, "AVG_TTFB": 35.875, "ERRORS": 32, "AVG_DURATION": 70.0, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 75, "MAX_DURATION": 106, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94009.0, "COUNT": 32}, {"from": 28000, "to": 29000, "AVG_TTFB": 36.71875, "ERRORS": 32, "AVG_DURATION": 71.5625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 75, "MAX_DURATION": 107, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94025.0, "COUNT": 32}, {"from": 29000, "to": 30000, "AVG_TTFB": 61.076923, "ERRORS": 26, "AVG_DURATION": 102.0, "ERRORS_PER_SECOND": 26.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 242, "MAX_DURATION": 273, "ELEMENTS_PER_SECOND": 26.0, "THROUGHPUT": 76368.0, "COUNT": 26}, {"from": 30000, "to": 31000, "AVG_TTFB": 37.033333, "ERRORS": 30, "AVG_DURATION": 74.6, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 73, "MAX_DURATION": 104, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88166.0, "COUNT": 30}, {"from": 31000, "to": 32000, "AVG_TTFB": 34.935482, "ERRORS": 31, "AVG_DURATION": 69.064514, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 69, "MAX_DURATION": 107, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91104.0, "COUNT": 31}, {"from": 32000, "to": 33000, "AVG_TTFB": 34.545456, "ERRORS": 33, "AVG_DURATION": 69.51515, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 42, "MAX_DURATION": 82, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96964.0, "COUNT": 33}, {"from": 33000, "to": 34000, "AVG_TTFB": 32.67647, "ERRORS": 34, "AVG_DURATION": 65.94118, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 39, "MAX_DURATION": 77, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99919.0, "COUNT": 34}, {"from": 34000, "to": 35000, "AVG_TTFB": 36.419353, "ERRORS": 31, "AVG_DURATION": 72.870964, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 71, "MAX_DURATION": 107, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91131.0, "COUNT": 31}, {"from": 35000, "to": 36000, "AVG_TTFB": 40.333332, "ERRORS": 30, "AVG_DURATION": 78.7, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 105, "MAX_DURATION": 142, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88194.0, "COUNT": 30}, {"from": 36000, "to": 37000, "AVG_TTFB": 35.633335, "ERRORS": 30, "AVG_DURATION": 71.13333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 78, "MAX_DURATION": 123, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88192.0, "COUNT": 30}, {"from": 37000, "to": 38000, "AVG_TTFB": 34.764706, "ERRORS": 34, "AVG_DURATION": 69.617645, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 74, "MAX_DURATION": 106, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99911.0, "COUNT": 34}, {"from": 38000, "to": 39000, "AVG_TTFB": 36.303032, "ERRORS": 33, "AVG_DURATION": 70.42424, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 85, "MAX_DURATION": 118, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96987.0, "COUNT": 33}, {"from": 39000, "to": 40000, "AVG_TTFB": 36.515152, "ERRORS": 33, "AVG_DURATION": 71.0, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 91, "MAX_DURATION": 127, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96992.0, "COUNT": 33}, {"from": 40000, "to": 41000, "AVG_TTFB": 38.78125, "ERRORS": 32, "AVG_DURATION": 75.5625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 91, "MAX_DURATION": 132, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94039.0, "COUNT": 32}, {"from": 41000, "to": 42000, "AVG_TTFB": 34.09375, "ERRORS": 32, "AVG_DURATION": 68.34375, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 78, "MAX_DURATION": 114, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94063.0, "COUNT": 32}, {"from": 42000, "to": 43000, "AVG_TTFB": 33.6875, "ERRORS": 32, "AVG_DURATION": 67.84375, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 43, "MAX_DURATION": 77, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94054.0, "COUNT": 32}, {"from": 43000, "to": 44000, "AVG_TTFB": 37.580647, "ERRORS": 31, "AVG_DURATION": 72.935486, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 86, "MAX_DURATION": 119, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91054.0, "COUNT": 31}, {"from": 44000, "to": 45000, "AVG_TTFB": 40.794117, "ERRORS": 34, "AVG_DURATION": 79.02941, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 99, "MAX_DURATION": 134, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99913.0, "COUNT": 34}, {"from": 45000, "to": 46000, "AVG_TTFB": 33.121212, "ERRORS": 33, "AVG_DURATION": 67.09091, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 41, "MAX_DURATION": 79, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 97015.0, "COUNT": 33}, {"from": 46000, "to": 47000, "AVG_TTFB": 33.870968, "ERRORS": 31, "AVG_DURATION": 67.83871, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 41, "MAX_DURATION": 74, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91123.0, "COUNT": 31}, {"from": 47000, "to": 48000, "AVG_TTFB": 37.5, "ERRORS": 30, "AVG_DURATION": 74.36667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 55, "MAX_DURATION": 106, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88170.0, "COUNT": 30}, {"from": 48000, "to": 49000, "AVG_TTFB": 35.64706, "ERRORS": 34, "AVG_DURATION": 72.23529, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 95, "MAX_DURATION": 128, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 100009.0, "COUNT": 34}, {"from": 49000, "to": 50000, "AVG_TTFB": 35.16129, "ERRORS": 31, "AVG_DURATION": 69.32258, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 43, "MAX_DURATION": 77, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91105.0, "COUNT": 31}, {"from": 50000, "to": 51000, "AVG_TTFB": 39.4, "ERRORS": 30, "AVG_DURATION": 77.8, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 103, "MAX_DURATION": 134, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88155.0, "COUNT": 30}, {"from": 51000, "to": 52000, "AVG_TTFB": 35.064518, "ERRORS": 31, "AVG_DURATION": 71.548386, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 46, "MAX_DURATION": 83, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91097.0, "COUNT": 31}, {"from": 52000, "to": 53000, "AVG_TTFB": 43.25, "ERRORS": 28, "AVG_DURATION": 86.17857, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 116, "MAX_DURATION": 205, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82222.0, "COUNT": 28}, {"from": 53000, "to": 54000, "AVG_TTFB": 36.666668, "ERRORS": 30, "AVG_DURATION": 72.4, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 64, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 56, "MAX_DURATION": 90, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88153.0, "COUNT": 30}, {"from": 54000, "to": 55000, "AVG_TTFB": 35.90625, "ERRORS": 32, "AVG_DURATION": 72.84375, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 55, "MAX_DURATION": 92, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94024.0, "COUNT": 32}, {"from": 55000, "to": 56000, "AVG_TTFB": 34.121212, "ERRORS": 33, "AVG_DURATION": 69.30303, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 42, "MAX_DURATION": 78, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96922.0, "COUNT": 33}, {"from": 56000, "to": 57000, "AVG_TTFB": 35.5625, "ERRORS": 32, "AVG_DURATION": 70.65625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 95, "MAX_DURATION": 129, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94000.0, "COUNT": 32}, {"from": 57000, "to": 58000, "AVG_TTFB": 34.566666, "ERRORS": 30, "AVG_DURATION": 70.4, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 45, "MAX_DURATION": 81, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88124.0, "COUNT": 30}, {"from": 58000, "to": 59000, "AVG_TTFB": 37.655174, "ERRORS": 29, "AVG_DURATION": 75.586205, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 79, "MAX_DURATION": 113, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85176.0, "COUNT": 29}, {"from": 59000, "to": 60000, "AVG_TTFB": 43.266666, "ERRORS": 21, "AVG_DURATION": 82.0, "ERRORS_PER_SECOND": 21.0, "MIN_DURATION": 61, "ERROR_RATE": 70.0, "MIN_TTFB": 30, "MAX_TTFB": 127, "MAX_DURATION": 165, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 75512.0, "COUNT": 30}, {"from": 60000, "to": 61000, "AVG_TTFB": 34.677418, "ERRORS": 22, "AVG_DURATION": 71.58064, "ERRORS_PER_SECOND": 22.0, "MIN_DURATION": 63, "ERROR_RATE": 70.96774, "MIN_TTFB": 30, "MAX_TTFB": 45, "MAX_DURATION": 87, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 78518.0, "COUNT": 31}, {"from": 61000, "to": 62000, "AVG_TTFB": 34.29032, "ERRORS": 31, "AVG_DURATION": 69.03226, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 40, "MAX_DURATION": 77, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91108.0, "COUNT": 31}, {"from": 62000, "to": 63000, "AVG_TTFB": 41.419353, "ERRORS": 31, "AVG_DURATION": 77.0, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 112, "MAX_DURATION": 147, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91109.0, "COUNT": 31}, {"from": 63000, "to": 64000, "AVG_TTFB": 33.1875, "ERRORS": 32, "AVG_DURATION": 67.90625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 42, "MAX_DURATION": 77, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94044.0, "COUNT": 32}, {"from": 64000, "to": 65000, "AVG_TTFB": 33.625, "ERRORS": 32, "AVG_DURATION": 68.78125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 43, "MAX_DURATION": 77, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94040.0, "COUNT": 32}, {"from": 65000, "to": 66000, "AVG_TTFB": 38.83871, "ERRORS": 31, "AVG_DURATION": 74.32258, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 112, "MAX_DURATION": 145, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91098.0, "COUNT": 31}, {"from": 66000, "to": 67000, "AVG_TTFB": 37.774193, "ERRORS": 31, "AVG_DURATION": 73.58064, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 114, "MAX_DURATION": 150, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91128.0, "COUNT": 31}, {"from": 67000, "to": 68000, "AVG_TTFB": 37.4, "ERRORS": 30, "AVG_DURATION": 75.066666, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 101, "MAX_DURATION": 133, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88128.0, "COUNT": 30}, {"from": 68000, "to": 69000, "AVG_TTFB": 36.129032, "ERRORS": 31, "AVG_DURATION": 73.645164, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 57, "MAX_DURATION": 94, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91109.0, "COUNT": 31}, {"from": 69000, "to": 70000, "AVG_TTFB": 38.166668, "ERRORS": 30, "AVG_DURATION": 74.066666, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 96, "MAX_DURATION": 128, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88163.0, "COUNT": 30}, {"from": 70000, "to": 71000, "AVG_TTFB": 39.833332, "ERRORS": 30, "AVG_DURATION": 78.3, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 64, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 101, "MAX_DURATION": 141, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88145.0, "COUNT": 30}, {"from": 71000, "to": 72000, "AVG_TTFB": 34.96875, "ERRORS": 32, "AVG_DURATION": 71.59375, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 64, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 41, "MAX_DURATION": 79, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94020.0, "COUNT": 32}, {"from": 72000, "to": 73000, "AVG_TTFB": 38.258064, "ERRORS": 31, "AVG_DURATION": 75.90323, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 110, "MAX_DURATION": 148, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91112.0, "COUNT": 31}, {"from": 73000, "to": 74000, "AVG_TTFB": 34.366665, "ERRORS": 30, "AVG_DURATION": 69.53333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 43, "MAX_DURATION": 78, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88193.0, "COUNT": 30}, {"from": 74000, "to": 75000, "AVG_TTFB": 34.114285, "ERRORS": 35, "AVG_DURATION": 68.02857, "ERRORS_PER_SECOND": 35.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 42, "MAX_DURATION": 77, "ELEMENTS_PER_SECOND": 35.0, "THROUGHPUT": 102879.0, "COUNT": 35}, {"from": 75000, "to": 76000, "AVG_TTFB": 32.533333, "ERRORS": 30, "AVG_DURATION": 66.833336, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 41, "MAX_DURATION": 77, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88205.0, "COUNT": 30}, {"from": 76000, "to": 77000, "AVG_TTFB": 35.53125, "ERRORS": 32, "AVG_DURATION": 72.96875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 51, "MAX_DURATION": 97, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94053.0, "COUNT": 32}, {"from": 77000, "to": 78000, "AVG_TTFB": 40.19355, "ERRORS": 31, "AVG_DURATION": 74.870964, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 129, "MAX_DURATION": 168, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91089.0, "COUNT": 31}, {"from": 78000, "to": 79000, "AVG_TTFB": 38.29032, "ERRORS": 31, "AVG_DURATION": 75.83871, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 102, "MAX_DURATION": 169, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91059.0, "COUNT": 31}, {"from": 79000, "to": 80000, "AVG_TTFB": 36.70968, "ERRORS": 31, "AVG_DURATION": 71.645164, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 107, "MAX_DURATION": 144, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91123.0, "COUNT": 31}, {"from": 80000, "to": 81000, "AVG_TTFB": 39.533333, "ERRORS": 30, "AVG_DURATION": 77.63333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 161, "MAX_DURATION": 202, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88178.0, "COUNT": 30}, {"from": 81000, "to": 82000, "AVG_TTFB": 33.0, "ERRORS": 34, "AVG_DURATION": 67.32353, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 41, "MAX_DURATION": 78, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99908.0, "COUNT": 34}, {"from": 82000, "to": 83000, "AVG_TTFB": 41.3, "ERRORS": 30, "AVG_DURATION": 79.23333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 115, "MAX_DURATION": 152, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88216.0, "COUNT": 30}, {"from": 83000, "to": 84000, "AVG_TTFB": 37.5625, "ERRORS": 32, "AVG_DURATION": 75.28125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 111, "MAX_DURATION": 154, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94066.0, "COUNT": 32}, {"from": 84000, "to": 85000, "AVG_TTFB": 35.51613, "ERRORS": 31, "AVG_DURATION": 70.645164, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 50, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91061.0, "COUNT": 31}, {"from": 85000, "to": 86000, "AVG_TTFB": 34.096775, "ERRORS": 31, "AVG_DURATION": 69.935486, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 57, "MAX_DURATION": 94, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91099.0, "COUNT": 31}, {"from": 86000, "to": 87000, "AVG_TTFB": 34.8125, "ERRORS": 32, "AVG_DURATION": 69.40625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 41, "MAX_DURATION": 76, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94058.0, "COUNT": 32}, {"from": 87000, "to": 88000, "AVG_TTFB": 33.382355, "ERRORS": 34, "AVG_DURATION": 68.55882, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 48, "MAX_DURATION": 83, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99912.0, "COUNT": 34}, {"from": 88000, "to": 89000, "AVG_TTFB": 34.83871, "ERRORS": 31, "AVG_DURATION": 69.0, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 116, "MAX_DURATION": 151, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91132.0, "COUNT": 31}, {"from": 89000, "to": 90000, "AVG_TTFB": 43.64, "ERRORS": 25, "AVG_DURATION": 86.0, "ERRORS_PER_SECOND": 25.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 301, "MAX_DURATION": 335, "ELEMENTS_PER_SECOND": 25.0, "THROUGHPUT": 73457.0, "COUNT": 25}, {"from": 90000, "to": 91000, "AVG_TTFB": 33.25, "ERRORS": 32, "AVG_DURATION": 67.78125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 46, "MAX_DURATION": 85, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94080.0, "COUNT": 32}, {"from": 91000, "to": 92000, "AVG_TTFB": 33.909092, "ERRORS": 33, "AVG_DURATION": 68.78788, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 39, "MAX_DURATION": 78, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96997.0, "COUNT": 33}, {"from": 92000, "to": 93000, "AVG_TTFB": 33.617645, "ERRORS": 34, "AVG_DURATION": 68.17647, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 40, "MAX_DURATION": 81, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99921.0, "COUNT": 34}, {"from": 93000, "to": 94000, "AVG_TTFB": 34.125, "ERRORS": 32, "AVG_DURATION": 68.03125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 51, "MAX_DURATION": 83, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94071.0, "COUNT": 32}, {"from": 94000, "to": 95000, "AVG_TTFB": 39.931034, "ERRORS": 29, "AVG_DURATION": 75.27586, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 137, "MAX_DURATION": 193, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85191.0, "COUNT": 29}, {"from": 95000, "to": 96000, "AVG_TTFB": 34.18182, "ERRORS": 33, "AVG_DURATION": 68.333336, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 40, "MAX_DURATION": 77, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96956.0, "COUNT": 33}, {"from": 96000, "to": 97000, "AVG_TTFB": 39.033333, "ERRORS": 30, "AVG_DURATION": 73.8, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 122, "MAX_DURATION": 153, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88196.0, "COUNT": 30}, {"from": 97000, "to": 98000, "AVG_TTFB": 33.242424, "ERRORS": 33, "AVG_DURATION": 67.30303, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 37, "MAX_DURATION": 74, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 97013.0, "COUNT": 33}, {"from": 98000, "to": 99000, "AVG_TTFB": 35.0, "ERRORS": 31, "AVG_DURATION": 71.129036, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 41, "MAX_DURATION": 81, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91136.0, "COUNT": 31}, {"from": 99000, "to": 100000, "AVG_TTFB": 42.13793, "ERRORS": 29, "AVG_DURATION": 77.93104, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 124, "MAX_DURATION": 160, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85205.0, "COUNT": 29}, {"from": 100000, "to": 101000, "AVG_TTFB": 37.333332, "ERRORS": 30, "AVG_DURATION": 75.86667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 64, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 50, "MAX_DURATION": 98, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88152.0, "COUNT": 30}, {"from": 101000, "to": 102000, "AVG_TTFB": 60.0, "ERRORS": 20, "AVG_DURATION": 121.9, "ERRORS_PER_SECOND": 20.0, "MIN_DURATION": 79, "ERROR_RATE": 100.0, "MIN_TTFB": 33, "MAX_TTFB": 167, "MAX_DURATION": 247, "ELEMENTS_PER_SECOND": 20.0, "THROUGHPUT": 58758.0, "COUNT": 20}, {"from": 102000, "to": 103000, "AVG_TTFB": 62.333332, "ERRORS": 21, "AVG_DURATION": 111.809525, "ERRORS_PER_SECOND": 21.0, "MIN_DURATION": 85, "ERROR_RATE": 100.0, "MIN_TTFB": 44, "MAX_TTFB": 142, "MAX_DURATION": 196, "ELEMENTS_PER_SECOND": 21.0, "THROUGHPUT": 61723.0, "COUNT": 21}, {"from": 103000, "to": 104000, "AVG_TTFB": 50.454544, "ERRORS": 22, "AVG_DURATION": 101.63636, "ERRORS_PER_SECOND": 22.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 39, "MAX_TTFB": 69, "MAX_DURATION": 123, "ELEMENTS_PER_SECOND": 22.0, "THROUGHPUT": 64700.0, "COUNT": 22}, {"from": 104000, "to": 105000, "AVG_TTFB": 65.44444, "ERRORS": 18, "AVG_DURATION": 131.72223, "ERRORS_PER_SECOND": 18.0, "MIN_DURATION": 109, "ERROR_RATE": 100.0, "MIN_TTFB": 48, "MAX_TTFB": 83, "MAX_DURATION": 159, "ELEMENTS_PER_SECOND": 18.0, "THROUGHPUT": 52910.0, "COUNT": 18}, {"from": 105000, "to": 106000, "AVG_TTFB": 63.0, "ERRORS": 19, "AVG_DURATION": 124.10526, "ERRORS_PER_SECOND": 19.0, "MIN_DURATION": 83, "ERROR_RATE": 100.0, "MIN_TTFB": 44, "MAX_TTFB": 83, "MAX_DURATION": 161, "ELEMENTS_PER_SECOND": 19.0, "THROUGHPUT": 55845.0, "COUNT": 19}, {"from": 106000, "to": 107000, "AVG_TTFB": 44.689655, "ERRORS": 29, "AVG_DURATION": 84.965515, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 119, "MAX_DURATION": 160, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85275.0, "COUNT": 29}, {"from": 107000, "to": 108000, "AVG_TTFB": 38.1, "ERRORS": 30, "AVG_DURATION": 77.4, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 66, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 48, "MAX_DURATION": 98, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88168.0, "COUNT": 30}, {"from": 108000, "to": 109000, "AVG_TTFB": 40.4, "ERRORS": 25, "AVG_DURATION": 79.72, "ERRORS_PER_SECOND": 25.0, "MIN_DURATION": 67, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 48, "MAX_DURATION": 94, "ELEMENTS_PER_SECOND": 25.0, "THROUGHPUT": 73455.0, "COUNT": 25}, {"from": 109000, "to": 110000, "AVG_TTFB": 46.692307, "ERRORS": 26, "AVG_DURATION": 89.11539, "ERRORS_PER_SECOND": 26.0, "MIN_DURATION": 72, "ERROR_RATE": 100.0, "MIN_TTFB": 33, "MAX_TTFB": 133, "MAX_DURATION": 168, "ELEMENTS_PER_SECOND": 26.0, "THROUGHPUT": 76390.0, "COUNT": 26}, {"from": 110000, "to": 111000, "AVG_TTFB": 46.695652, "ERRORS": 23, "AVG_DURATION": 94.304344, "ERRORS_PER_SECOND": 23.0, "MIN_DURATION": 79, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 138, "MAX_DURATION": 181, "ELEMENTS_PER_SECOND": 23.0, "THROUGHPUT": 67625.0, "COUNT": 23}, {"from": 111000, "to": 112000, "AVG_TTFB": 36.515152, "ERRORS": 33, "AVG_DURATION": 75.51515, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 51, "MAX_DURATION": 111, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96919.0, "COUNT": 33}, {"from": 112000, "to": 113000, "AVG_TTFB": 39.535713, "ERRORS": 28, "AVG_DURATION": 77.14286, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 131, "MAX_DURATION": 178, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82262.0, "COUNT": 28}, {"from": 113000, "to": 114000, "AVG_TTFB": 44.392857, "ERRORS": 28, "AVG_DURATION": 83.82143, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 128, "MAX_DURATION": 170, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82221.0, "COUNT": 28}, {"from": 114000, "to": 115000, "AVG_TTFB": 41.791668, "ERRORS": 24, "AVG_DURATION": 89.666664, "ERRORS_PER_SECOND": 24.0, "MIN_DURATION": 66, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 65, "MAX_DURATION": 140, "ELEMENTS_PER_SECOND": 24.0, "THROUGHPUT": 70510.0, "COUNT": 24}, {"from": 115000, "to": 116000, "AVG_TTFB": 46.913044, "ERRORS": 23, "AVG_DURATION": 97.82609, "ERRORS_PER_SECOND": 23.0, "MIN_DURATION": 84, "ERROR_RATE": 100.0, "MIN_TTFB": 36, "MAX_TTFB": 66, "MAX_DURATION": 119, "ELEMENTS_PER_SECOND": 23.0, "THROUGHPUT": 67570.0, "COUNT": 23}, {"from": 116000, "to": 117000, "AVG_TTFB": 56.157894, "ERRORS": 19, "AVG_DURATION": 116.0, "ERRORS_PER_SECOND": 19.0, "MIN_DURATION": 80, "ERROR_RATE": 100.0, "MIN_TTFB": 40, "MAX_TTFB": 89, "MAX_DURATION": 176, "ELEMENTS_PER_SECOND": 19.0, "THROUGHPUT": 55812.0, "COUNT": 19}, {"from": 117000, "to": 118000, "AVG_TTFB": 60.22222, "ERRORS": 18, "AVG_DURATION": 122.22222, "ERRORS_PER_SECOND": 18.0, "MIN_DURATION": 96, "ERROR_RATE": 100.0, "MIN_TTFB": 44, "MAX_TTFB": 83, "MAX_DURATION": 157, "ELEMENTS_PER_SECOND": 18.0, "THROUGHPUT": 52880.0, "COUNT": 18}, {"from": 118000, "to": 119000, "AVG_TTFB": 60.842106, "ERRORS": 19, "AVG_DURATION": 118.947365, "ERRORS_PER_SECOND": 19.0, "MIN_DURATION": 102, "ERROR_RATE": 100.0, "MIN_TTFB": 48, "MAX_TTFB": 79, "MAX_DURATION": 137, "ELEMENTS_PER_SECOND": 19.0, "THROUGHPUT": 55832.0, "COUNT": 19}, {"from": 119000, "to": 120000, "AVG_TTFB": 65.35294, "ERRORS": 13, "AVG_DURATION": 129.11765, "ERRORS_PER_SECOND": 13.0, "MIN_DURATION": 110, "ERROR_RATE": 76.47059, "MIN_TTFB": 55, "MAX_TTFB": 92, "MAX_DURATION": 157, "ELEMENTS_PER_SECOND": 17.0, "THROUGHPUT": 44352.0, "COUNT": 17}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements/7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 1000, "AVG_TTFB": 35.7, "ERRORS": 0, "AVG_DURATION": 79.15, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 70, "ERROR_RATE": 0.0, "MIN_TTFB": 29, "MAX_TTFB": 45, "MAX_DURATION": 98, "ELEMENTS_PER_SECOND": 20.0, "THROUGHPUT": 30802.0, "COUNT": 20}, {"from": 1000, "to": 2000, "AVG_TTFB": 35.966667, "ERRORS": 30, "AVG_DURATION": 71.03333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 53, "MAX_DURATION": 88, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88158.0, "COUNT": 30}, {"from": 2000, "to": 3000, "AVG_TTFB": 36.533333, "ERRORS": 30, "AVG_DURATION": 71.73333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 49, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88226.0, "COUNT": 30}, {"from": 3000, "to": 4000, "AVG_TTFB": 33.0, "ERRORS": 30, "AVG_DURATION": 68.833336, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 41, "MAX_DURATION": 85, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88143.0, "COUNT": 30}, {"from": 4000, "to": 5000, "AVG_TTFB": 34.294117, "ERRORS": 34, "AVG_DURATION": 69.617645, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 57, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99896.0, "COUNT": 34}, {"from": 5000, "to": 6000, "AVG_TTFB": 38.225807, "ERRORS": 31, "AVG_DURATION": 73.58064, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 61, "MAX_DURATION": 103, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91131.0, "COUNT": 31}, {"from": 6000, "to": 7000, "AVG_TTFB": 34.942856, "ERRORS": 35, "AVG_DURATION": 70.057144, "ERRORS_PER_SECOND": 35.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 46, "MAX_DURATION": 79, "ELEMENTS_PER_SECOND": 35.0, "THROUGHPUT": 102853.0, "COUNT": 35}, {"from": 7000, "to": 8000, "AVG_TTFB": 34.4, "ERRORS": 30, "AVG_DURATION": 68.4, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 40, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88109.0, "COUNT": 30}, {"from": 8000, "to": 9000, "AVG_TTFB": 33.454544, "ERRORS": 33, "AVG_DURATION": 67.757576, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 40, "MAX_DURATION": 74, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96977.0, "COUNT": 33}, {"from": 9000, "to": 10000, "AVG_TTFB": 34.0, "ERRORS": 35, "AVG_DURATION": 68.6, "ERRORS_PER_SECOND": 35.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 62, "MAX_DURATION": 99, "ELEMENTS_PER_SECOND": 35.0, "THROUGHPUT": 102881.0, "COUNT": 35}, {"from": 10000, "to": 11000, "AVG_TTFB": 33.5, "ERRORS": 32, "AVG_DURATION": 68.40625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 50, "MAX_DURATION": 93, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94028.0, "COUNT": 32}, {"from": 11000, "to": 12000, "AVG_TTFB": 33.566666, "ERRORS": 30, "AVG_DURATION": 68.36667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 53, "MAX_DURATION": 129, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88177.0, "COUNT": 30}, {"from": 12000, "to": 13000, "AVG_TTFB": 37.5, "ERRORS": 32, "AVG_DURATION": 74.0, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 96, "MAX_DURATION": 137, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94073.0, "COUNT": 32}, {"from": 13000, "to": 14000, "AVG_TTFB": 34.575756, "ERRORS": 33, "AVG_DURATION": 69.21212, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 53, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96953.0, "COUNT": 33}, {"from": 14000, "to": 15000, "AVG_TTFB": 34.81818, "ERRORS": 33, "AVG_DURATION": 68.818184, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 66, "MAX_DURATION": 101, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96989.0, "COUNT": 33}, {"from": 15000, "to": 16000, "AVG_TTFB": 33.727272, "ERRORS": 33, "AVG_DURATION": 67.333336, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 62, "MAX_DURATION": 93, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96985.0, "COUNT": 33}, {"from": 16000, "to": 17000, "AVG_TTFB": 54.35, "ERRORS": 20, "AVG_DURATION": 106.45, "ERRORS_PER_SECOND": 20.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 184, "MAX_DURATION": 349, "ELEMENTS_PER_SECOND": 20.0, "THROUGHPUT": 58780.0, "COUNT": 20}, {"from": 17000, "to": 18000, "AVG_TTFB": 34.580647, "ERRORS": 31, "AVG_DURATION": 69.77419, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 59, "MAX_DURATION": 92, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91139.0, "COUNT": 31}, {"from": 18000, "to": 19000, "AVG_TTFB": 35.333332, "ERRORS": 33, "AVG_DURATION": 69.51515, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 62, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96993.0, "COUNT": 33}, {"from": 19000, "to": 20000, "AVG_TTFB": 36.1875, "ERRORS": 32, "AVG_DURATION": 69.5625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 78, "MAX_DURATION": 117, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94045.0, "COUNT": 32}, {"from": 20000, "to": 21000, "AVG_TTFB": 35.548386, "ERRORS": 31, "AVG_DURATION": 70.741936, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 70, "MAX_DURATION": 110, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91129.0, "COUNT": 31}, {"from": 21000, "to": 22000, "AVG_TTFB": 34.208332, "ERRORS": 24, "AVG_DURATION": 68.25, "ERRORS_PER_SECOND": 24.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 52, "MAX_DURATION": 86, "ELEMENTS_PER_SECOND": 24.0, "THROUGHPUT": 70506.0, "COUNT": 24}, {"from": 22000, "to": 23000, "AVG_TTFB": 34.4, "ERRORS": 35, "AVG_DURATION": 68.457146, "ERRORS_PER_SECOND": 35.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 54, "MAX_DURATION": 95, "ELEMENTS_PER_SECOND": 35.0, "THROUGHPUT": 102816.0, "COUNT": 35}, {"from": 23000, "to": 24000, "AVG_TTFB": 35.59375, "ERRORS": 32, "AVG_DURATION": 69.78125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 58, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 85, "MAX_DURATION": 125, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94037.0, "COUNT": 32}, {"from": 24000, "to": 25000, "AVG_TTFB": 32.5625, "ERRORS": 32, "AVG_DURATION": 67.15625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 40, "MAX_DURATION": 78, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94092.0, "COUNT": 32}, {"from": 25000, "to": 26000, "AVG_TTFB": 33.264706, "ERRORS": 34, "AVG_DURATION": 69.32353, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 60, "MAX_DURATION": 112, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99901.0, "COUNT": 34}, {"from": 26000, "to": 27000, "AVG_TTFB": 39.333332, "ERRORS": 30, "AVG_DURATION": 75.13333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 99, "MAX_DURATION": 132, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88141.0, "COUNT": 30}, {"from": 27000, "to": 28000, "AVG_TTFB": 34.0, "ERRORS": 30, "AVG_DURATION": 67.4, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 61, "MAX_DURATION": 94, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88133.0, "COUNT": 30}, {"from": 28000, "to": 29000, "AVG_TTFB": 34.588234, "ERRORS": 34, "AVG_DURATION": 70.73529, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 46, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99960.0, "COUNT": 34}, {"from": 29000, "to": 30000, "AVG_TTFB": 42.5, "ERRORS": 26, "AVG_DURATION": 92.46154, "ERRORS_PER_SECOND": 26.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 234, "MAX_DURATION": 276, "ELEMENTS_PER_SECOND": 26.0, "THROUGHPUT": 76404.0, "COUNT": 26}, {"from": 30000, "to": 31000, "AVG_TTFB": 34.266666, "ERRORS": 30, "AVG_DURATION": 73.53333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 47, "MAX_DURATION": 100, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88174.0, "COUNT": 30}, {"from": 31000, "to": 32000, "AVG_TTFB": 33.969696, "ERRORS": 33, "AVG_DURATION": 68.72727, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 43, "MAX_DURATION": 77, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 97039.0, "COUNT": 33}, {"from": 32000, "to": 33000, "AVG_TTFB": 35.424244, "ERRORS": 33, "AVG_DURATION": 70.48485, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 70, "MAX_DURATION": 101, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96960.0, "COUNT": 33}, {"from": 33000, "to": 34000, "AVG_TTFB": 34.741936, "ERRORS": 31, "AVG_DURATION": 69.48387, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 41, "MAX_DURATION": 79, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91084.0, "COUNT": 31}, {"from": 34000, "to": 35000, "AVG_TTFB": 36.774193, "ERRORS": 31, "AVG_DURATION": 71.548386, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 69, "MAX_DURATION": 101, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91101.0, "COUNT": 31}, {"from": 35000, "to": 36000, "AVG_TTFB": 33.83871, "ERRORS": 31, "AVG_DURATION": 68.51613, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 48, "MAX_DURATION": 82, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91108.0, "COUNT": 31}, {"from": 36000, "to": 37000, "AVG_TTFB": 34.515152, "ERRORS": 33, "AVG_DURATION": 69.30303, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 44, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96978.0, "COUNT": 33}, {"from": 37000, "to": 38000, "AVG_TTFB": 34.484848, "ERRORS": 33, "AVG_DURATION": 69.121216, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 53, "MAX_DURATION": 85, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96978.0, "COUNT": 33}, {"from": 38000, "to": 39000, "AVG_TTFB": 35.266666, "ERRORS": 30, "AVG_DURATION": 69.26667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 82, "MAX_DURATION": 118, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88134.0, "COUNT": 30}, {"from": 39000, "to": 40000, "AVG_TTFB": 35.058823, "ERRORS": 34, "AVG_DURATION": 68.617645, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 75, "MAX_DURATION": 108, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99932.0, "COUNT": 34}, {"from": 40000, "to": 41000, "AVG_TTFB": 39.633335, "ERRORS": 30, "AVG_DURATION": 77.03333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 86, "MAX_DURATION": 130, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88184.0, "COUNT": 30}, {"from": 41000, "to": 42000, "AVG_TTFB": 32.794117, "ERRORS": 34, "AVG_DURATION": 67.85294, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 39, "MAX_DURATION": 76, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99905.0, "COUNT": 34}, {"from": 42000, "to": 43000, "AVG_TTFB": 35.617645, "ERRORS": 34, "AVG_DURATION": 69.588234, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 82, "MAX_DURATION": 115, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99871.0, "COUNT": 34}, {"from": 43000, "to": 44000, "AVG_TTFB": 33.875, "ERRORS": 32, "AVG_DURATION": 68.875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 41, "MAX_DURATION": 75, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94041.0, "COUNT": 32}, {"from": 44000, "to": 45000, "AVG_TTFB": 38.2, "ERRORS": 30, "AVG_DURATION": 71.1, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 90, "MAX_DURATION": 129, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88162.0, "COUNT": 30}, {"from": 45000, "to": 46000, "AVG_TTFB": 33.424244, "ERRORS": 33, "AVG_DURATION": 67.181816, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 41, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96998.0, "COUNT": 33}, {"from": 46000, "to": 47000, "AVG_TTFB": 34.882355, "ERRORS": 34, "AVG_DURATION": 68.29412, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 87, "MAX_DURATION": 122, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99902.0, "COUNT": 34}, {"from": 47000, "to": 48000, "AVG_TTFB": 35.9, "ERRORS": 30, "AVG_DURATION": 72.5, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 51, "MAX_DURATION": 95, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88168.0, "COUNT": 30}, {"from": 48000, "to": 49000, "AVG_TTFB": 39.96875, "ERRORS": 32, "AVG_DURATION": 75.1875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 99, "MAX_DURATION": 132, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94064.0, "COUNT": 32}, {"from": 49000, "to": 50000, "AVG_TTFB": 33.580647, "ERRORS": 31, "AVG_DURATION": 68.96774, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 40, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91117.0, "COUNT": 31}, {"from": 50000, "to": 51000, "AVG_TTFB": 39.741936, "ERRORS": 31, "AVG_DURATION": 78.16129, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 109, "MAX_DURATION": 142, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91113.0, "COUNT": 31}, {"from": 51000, "to": 52000, "AVG_TTFB": 34.741936, "ERRORS": 31, "AVG_DURATION": 70.32258, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 40, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91113.0, "COUNT": 31}, {"from": 52000, "to": 53000, "AVG_TTFB": 45.107143, "ERRORS": 28, "AVG_DURATION": 86.42857, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 164, "MAX_DURATION": 199, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82240.0, "COUNT": 28}, {"from": 53000, "to": 54000, "AVG_TTFB": 35.741936, "ERRORS": 31, "AVG_DURATION": 71.129036, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 47, "MAX_DURATION": 90, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91051.0, "COUNT": 31}, {"from": 54000, "to": 55000, "AVG_TTFB": 36.677418, "ERRORS": 31, "AVG_DURATION": 73.03226, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 55, "MAX_DURATION": 94, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91053.0, "COUNT": 31}, {"from": 55000, "to": 56000, "AVG_TTFB": 36.0625, "ERRORS": 32, "AVG_DURATION": 70.53125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 103, "MAX_DURATION": 139, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 93991.0, "COUNT": 32}, {"from": 56000, "to": 57000, "AVG_TTFB": 38.612904, "ERRORS": 31, "AVG_DURATION": 73.41936, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 119, "MAX_DURATION": 151, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91061.0, "COUNT": 31}, {"from": 57000, "to": 58000, "AVG_TTFB": 36.78788, "ERRORS": 33, "AVG_DURATION": 73.57576, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 55, "MAX_DURATION": 92, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96948.0, "COUNT": 33}, {"from": 58000, "to": 59000, "AVG_TTFB": 36.392857, "ERRORS": 28, "AVG_DURATION": 75.75, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 54, "MAX_DURATION": 129, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82258.0, "COUNT": 28}, {"from": 59000, "to": 60000, "AVG_TTFB": 41.75862, "ERRORS": 23, "AVG_DURATION": 80.75862, "ERRORS_PER_SECOND": 23.0, "MIN_DURATION": 67, "ERROR_RATE": 79.31035, "MIN_TTFB": 30, "MAX_TTFB": 111, "MAX_DURATION": 152, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 76797.0, "COUNT": 29}, {"from": 60000, "to": 61000, "AVG_TTFB": 37.34375, "ERRORS": 22, "AVG_DURATION": 75.1875, "ERRORS_PER_SECOND": 22.0, "MIN_DURATION": 64, "ERROR_RATE": 68.75, "MIN_TTFB": 30, "MAX_TTFB": 111, "MAX_DURATION": 152, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 80063.0, "COUNT": 32}, {"from": 61000, "to": 62000, "AVG_TTFB": 35.0, "ERRORS": 31, "AVG_DURATION": 71.80645, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 41, "MAX_DURATION": 81, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91080.0, "COUNT": 31}, {"from": 62000, "to": 63000, "AVG_TTFB": 37.6, "ERRORS": 30, "AVG_DURATION": 73.566666, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 104, "MAX_DURATION": 148, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88172.0, "COUNT": 30}, {"from": 63000, "to": 64000, "AVG_TTFB": 33.875, "ERRORS": 32, "AVG_DURATION": 68.75, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 46, "MAX_DURATION": 81, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94019.0, "COUNT": 32}, {"from": 64000, "to": 65000, "AVG_TTFB": 35.47059, "ERRORS": 34, "AVG_DURATION": 70.35294, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 99, "MAX_DURATION": 139, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99953.0, "COUNT": 34}, {"from": 65000, "to": 66000, "AVG_TTFB": 37.482758, "ERRORS": 29, "AVG_DURATION": 74.793106, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 85, "MAX_DURATION": 120, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85260.0, "COUNT": 29}, {"from": 66000, "to": 67000, "AVG_TTFB": 34.6, "ERRORS": 30, "AVG_DURATION": 69.86667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 46, "MAX_DURATION": 85, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88154.0, "COUNT": 30}, {"from": 67000, "to": 68000, "AVG_TTFB": 36.866665, "ERRORS": 30, "AVG_DURATION": 74.46667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 61, "MAX_DURATION": 102, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88156.0, "COUNT": 30}, {"from": 68000, "to": 69000, "AVG_TTFB": 37.058823, "ERRORS": 34, "AVG_DURATION": 74.117645, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 94, "MAX_DURATION": 125, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99932.0, "COUNT": 34}, {"from": 69000, "to": 70000, "AVG_TTFB": 37.451614, "ERRORS": 31, "AVG_DURATION": 75.41936, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 110, "MAX_DURATION": 142, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91110.0, "COUNT": 31}, {"from": 70000, "to": 71000, "AVG_TTFB": 40.9, "ERRORS": 30, "AVG_DURATION": 77.26667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 104, "MAX_DURATION": 146, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88183.0, "COUNT": 30}, {"from": 71000, "to": 72000, "AVG_TTFB": 34.53125, "ERRORS": 32, "AVG_DURATION": 70.46875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 44, "MAX_DURATION": 82, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94020.0, "COUNT": 32}, {"from": 72000, "to": 73000, "AVG_TTFB": 41.172413, "ERRORS": 29, "AVG_DURATION": 77.10345, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 104, "MAX_DURATION": 137, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85233.0, "COUNT": 29}, {"from": 73000, "to": 74000, "AVG_TTFB": 40.677418, "ERRORS": 31, "AVG_DURATION": 75.80645, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 115, "MAX_DURATION": 150, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91145.0, "COUNT": 31}, {"from": 74000, "to": 75000, "AVG_TTFB": 33.548386, "ERRORS": 31, "AVG_DURATION": 67.741936, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 44, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91073.0, "COUNT": 31}, {"from": 75000, "to": 76000, "AVG_TTFB": 35.941177, "ERRORS": 34, "AVG_DURATION": 70.44118, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 117, "MAX_DURATION": 152, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99940.0, "COUNT": 34}, {"from": 76000, "to": 77000, "AVG_TTFB": 35.25, "ERRORS": 32, "AVG_DURATION": 70.6875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 62, "MAX_DURATION": 94, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94050.0, "COUNT": 32}, {"from": 77000, "to": 78000, "AVG_TTFB": 34.41379, "ERRORS": 29, "AVG_DURATION": 70.034485, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 40, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85233.0, "COUNT": 29}, {"from": 78000, "to": 79000, "AVG_TTFB": 39.09375, "ERRORS": 32, "AVG_DURATION": 75.40625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 127, "MAX_DURATION": 167, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94049.0, "COUNT": 32}, {"from": 79000, "to": 80000, "AVG_TTFB": 34.242424, "ERRORS": 33, "AVG_DURATION": 69.42424, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 46, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96959.0, "COUNT": 33}, {"from": 80000, "to": 81000, "AVG_TTFB": 38.233334, "ERRORS": 30, "AVG_DURATION": 73.833336, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 98, "MAX_DURATION": 139, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88167.0, "COUNT": 30}, {"from": 81000, "to": 82000, "AVG_TTFB": 33.129032, "ERRORS": 31, "AVG_DURATION": 66.935486, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 40, "MAX_DURATION": 74, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91087.0, "COUNT": 31}, {"from": 82000, "to": 83000, "AVG_TTFB": 36.29032, "ERRORS": 31, "AVG_DURATION": 73.96774, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 51, "MAX_DURATION": 94, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91129.0, "COUNT": 31}, {"from": 83000, "to": 84000, "AVG_TTFB": 35.933334, "ERRORS": 30, "AVG_DURATION": 72.03333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 61, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88196.0, "COUNT": 30}, {"from": 84000, "to": 85000, "AVG_TTFB": 38.51613, "ERRORS": 31, "AVG_DURATION": 73.80645, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 64, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 145, "MAX_DURATION": 180, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91069.0, "COUNT": 31}, {"from": 85000, "to": 86000, "AVG_TTFB": 34.419353, "ERRORS": 31, "AVG_DURATION": 69.80645, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 50, "MAX_DURATION": 83, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91143.0, "COUNT": 31}, {"from": 86000, "to": 87000, "AVG_TTFB": 34.38889, "ERRORS": 36, "AVG_DURATION": 70.02778, "ERRORS_PER_SECOND": 36.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 44, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 36.0, "THROUGHPUT": 105785.0, "COUNT": 36}, {"from": 87000, "to": 88000, "AVG_TTFB": 37.16129, "ERRORS": 31, "AVG_DURATION": 71.09677, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 135, "MAX_DURATION": 171, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91099.0, "COUNT": 31}, {"from": 88000, "to": 89000, "AVG_TTFB": 34.451614, "ERRORS": 31, "AVG_DURATION": 68.645164, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 49, "MAX_DURATION": 89, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91099.0, "COUNT": 31}, {"from": 89000, "to": 90000, "AVG_TTFB": 56.77778, "ERRORS": 27, "AVG_DURATION": 111.888885, "ERRORS_PER_SECOND": 27.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 313, "MAX_DURATION": 401, "ELEMENTS_PER_SECOND": 27.0, "THROUGHPUT": 79346.0, "COUNT": 27}, {"from": 90000, "to": 91000, "AVG_TTFB": 36.741936, "ERRORS": 31, "AVG_DURATION": 71.935486, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 114, "MAX_DURATION": 155, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91077.0, "COUNT": 31}, {"from": 91000, "to": 92000, "AVG_TTFB": 32.545456, "ERRORS": 33, "AVG_DURATION": 67.242424, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 40, "MAX_DURATION": 74, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 97012.0, "COUNT": 33}, {"from": 92000, "to": 93000, "AVG_TTFB": 38.96875, "ERRORS": 32, "AVG_DURATION": 73.59375, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 117, "MAX_DURATION": 156, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94072.0, "COUNT": 32}, {"from": 93000, "to": 94000, "AVG_TTFB": 32.71875, "ERRORS": 32, "AVG_DURATION": 67.28125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 41, "MAX_DURATION": 82, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94057.0, "COUNT": 32}, {"from": 94000, "to": 95000, "AVG_TTFB": 34.0, "ERRORS": 30, "AVG_DURATION": 80.666664, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 42, "MAX_DURATION": 204, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88207.0, "COUNT": 30}, {"from": 95000, "to": 96000, "AVG_TTFB": 33.59375, "ERRORS": 32, "AVG_DURATION": 67.875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 28, "MAX_TTFB": 40, "MAX_DURATION": 78, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94008.0, "COUNT": 32}, {"from": 96000, "to": 97000, "AVG_TTFB": 36.096775, "ERRORS": 31, "AVG_DURATION": 72.064514, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 55, "MAX_DURATION": 89, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91084.0, "COUNT": 31}, {"from": 97000, "to": 98000, "AVG_TTFB": 36.727272, "ERRORS": 33, "AVG_DURATION": 71.57576, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 61, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 120, "MAX_DURATION": 152, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96981.0, "COUNT": 33}, {"from": 98000, "to": 99000, "AVG_TTFB": 37.83871, "ERRORS": 31, "AVG_DURATION": 73.258064, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 127, "MAX_DURATION": 160, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91111.0, "COUNT": 31}, {"from": 99000, "to": 100000, "AVG_TTFB": 36.13793, "ERRORS": 29, "AVG_DURATION": 72.27586, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 51, "MAX_DURATION": 95, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85265.0, "COUNT": 29}, {"from": 100000, "to": 101000, "AVG_TTFB": 42.933334, "ERRORS": 30, "AVG_DURATION": 82.933334, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 65, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 148, "MAX_DURATION": 199, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88171.0, "COUNT": 30}, {"from": 101000, "to": 102000, "AVG_TTFB": 45.636364, "ERRORS": 22, "AVG_DURATION": 96.681816, "ERRORS_PER_SECOND": 22.0, "MIN_DURATION": 67, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 83, "MAX_DURATION": 237, "ELEMENTS_PER_SECOND": 22.0, "THROUGHPUT": 64665.0, "COUNT": 22}, {"from": 102000, "to": 103000, "AVG_TTFB": 50.95238, "ERRORS": 21, "AVG_DURATION": 106.333336, "ERRORS_PER_SECOND": 21.0, "MIN_DURATION": 73, "ERROR_RATE": 100.0, "MIN_TTFB": 39, "MAX_TTFB": 68, "MAX_DURATION": 139, "ELEMENTS_PER_SECOND": 21.0, "THROUGHPUT": 61735.0, "COUNT": 21}, {"from": 103000, "to": 104000, "AVG_TTFB": 52.863636, "ERRORS": 22, "AVG_DURATION": 103.27273, "ERRORS_PER_SECOND": 22.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 42, "MAX_TTFB": 71, "MAX_DURATION": 117, "ELEMENTS_PER_SECOND": 22.0, "THROUGHPUT": 64664.0, "COUNT": 22}, {"from": 104000, "to": 105000, "AVG_TTFB": 65.411766, "ERRORS": 17, "AVG_DURATION": 128.94118, "ERRORS_PER_SECOND": 17.0, "MIN_DURATION": 100, "ERROR_RATE": 100.0, "MIN_TTFB": 46, "MAX_TTFB": 94, "MAX_DURATION": 161, "ELEMENTS_PER_SECOND": 17.0, "THROUGHPUT": 49941.0, "COUNT": 17}, {"from": 105000, "to": 106000, "AVG_TTFB": 60.666668, "ERRORS": 18, "AVG_DURATION": 117.44444, "ERRORS_PER_SECOND": 18.0, "MIN_DURATION": 77, "ERROR_RATE": 100.0, "MIN_TTFB": 40, "MAX_TTFB": 78, "MAX_DURATION": 144, "ELEMENTS_PER_SECOND": 18.0, "THROUGHPUT": 52902.0, "COUNT": 18}, {"from": 106000, "to": 107000, "AVG_TTFB": 37.07143, "ERRORS": 28, "AVG_DURATION": 78.10714, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 66, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 49, "MAX_DURATION": 103, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82292.0, "COUNT": 28}, {"from": 107000, "to": 108000, "AVG_TTFB": 36.964287, "ERRORS": 28, "AVG_DURATION": 75.0, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 67, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 43, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82312.0, "COUNT": 28}, {"from": 108000, "to": 109000, "AVG_TTFB": 42.96552, "ERRORS": 29, "AVG_DURATION": 85.89655, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 66, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 136, "MAX_DURATION": 185, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85253.0, "COUNT": 29}, {"from": 109000, "to": 110000, "AVG_TTFB": 42.46154, "ERRORS": 26, "AVG_DURATION": 85.34615, "ERRORS_PER_SECOND": 26.0, "MIN_DURATION": 72, "ERROR_RATE": 100.0, "MIN_TTFB": 33, "MAX_TTFB": 60, "MAX_DURATION": 99, "ELEMENTS_PER_SECOND": 26.0, "THROUGHPUT": 76398.0, "COUNT": 26}, {"from": 110000, "to": 111000, "AVG_TTFB": 42.958332, "ERRORS": 24, "AVG_DURATION": 89.166664, "ERRORS_PER_SECOND": 24.0, "MIN_DURATION": 67, "ERROR_RATE": 100.0, "MIN_TTFB": 31, "MAX_TTFB": 57, "MAX_DURATION": 111, "ELEMENTS_PER_SECOND": 24.0, "THROUGHPUT": 70482.0, "COUNT": 24}, {"from": 111000, "to": 112000, "AVG_TTFB": 37.19355, "ERRORS": 31, "AVG_DURATION": 73.41936, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 68, "MAX_DURATION": 103, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91074.0, "COUNT": 31}, {"from": 112000, "to": 113000, "AVG_TTFB": 43.67857, "ERRORS": 28, "AVG_DURATION": 83.35714, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 135, "MAX_DURATION": 184, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82252.0, "COUNT": 28}, {"from": 113000, "to": 114000, "AVG_TTFB": 37.827587, "ERRORS": 29, "AVG_DURATION": 77.48276, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 67, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 51, "MAX_DURATION": 90, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85214.0, "COUNT": 29}, {"from": 114000, "to": 115000, "AVG_TTFB": 44.260868, "ERRORS": 23, "AVG_DURATION": 87.82609, "ERRORS_PER_SECOND": 23.0, "MIN_DURATION": 65, "ERROR_RATE": 100.0, "MIN_TTFB": 32, "MAX_TTFB": 85, "MAX_DURATION": 123, "ELEMENTS_PER_SECOND": 23.0, "THROUGHPUT": 67598.0, "COUNT": 23}, {"from": 115000, "to": 116000, "AVG_TTFB": 47.75, "ERRORS": 24, "AVG_DURATION": 98.416664, "ERRORS_PER_SECOND": 24.0, "MIN_DURATION": 80, "ERROR_RATE": 100.0, "MIN_TTFB": 38, "MAX_TTFB": 58, "MAX_DURATION": 120, "ELEMENTS_PER_SECOND": 24.0, "THROUGHPUT": 70518.0, "COUNT": 24}, {"from": 116000, "to": 117000, "AVG_TTFB": 63.055557, "ERRORS": 18, "AVG_DURATION": 123.27778, "ERRORS_PER_SECOND": 18.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 44, "MAX_TTFB": 158, "MAX_DURATION": 208, "ELEMENTS_PER_SECOND": 18.0, "THROUGHPUT": 52890.0, "COUNT": 18}, {"from": 117000, "to": 118000, "AVG_TTFB": 64.0, "ERRORS": 18, "AVG_DURATION": 121.333336, "ERRORS_PER_SECOND": 18.0, "MIN_DURATION": 106, "ERROR_RATE": 100.0, "MIN_TTFB": 51, "MAX_TTFB": 94, "MAX_DURATION": 156, "ELEMENTS_PER_SECOND": 18.0, "THROUGHPUT": 52902.0, "COUNT": 18}, {"from": 118000, "to": 119000, "AVG_TTFB": 58.1, "ERRORS": 20, "AVG_DURATION": 117.85, "ERRORS_PER_SECOND": 20.0, "MIN_DURATION": 104, "ERROR_RATE": 100.0, "MIN_TTFB": 47, "MAX_TTFB": 66, "MAX_DURATION": 153, "ELEMENTS_PER_SECOND": 20.0, "THROUGHPUT": 58734.0, "COUNT": 20}, {"from": 119000, "to": 120000, "AVG_TTFB": 67.5, "ERRORS": 12, "AVG_DURATION": 135.3125, "ERRORS_PER_SECOND": 12.0, "MIN_DURATION": 109, "ERROR_RATE": 75.0, "MIN_TTFB": 53, "MAX_TTFB": 151, "MAX_DURATION": 242, "ELEMENTS_PER_SECOND": 16.0, "THROUGHPUT": 41426.0, "COUNT": 16}, {"from": 120000, "to": 121000, "AVG_TTFB": 74.0, "ERRORS": 0, "AVG_DURATION": 190.0, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 190, "ERROR_RATE": 0.0, "MIN_TTFB": 74, "MAX_TTFB": 74, "MAX_DURATION": 190, "ELEMENTS_PER_SECOND": 1.0, "THROUGHPUT": 1535.0, "COUNT": 1}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/elements/e66cb9ec-3c72-49b4-b3a2-5fa91e2fd911/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 1000, "AVG_TTFB": 104.2, "ERRORS": 0, "AVG_DURATION": 147.8, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 103, "ERROR_RATE": 0.0, "MIN_TTFB": 66, "MAX_TTFB": 162, "MAX_DURATION": 207, "ELEMENTS_PER_SECOND": 20.0, "THROUGHPUT": 30786.0, "COUNT": 20}, {"from": 1000, "to": 2000, "AVG_TTFB": 63.866665, "ERRORS": 30, "AVG_DURATION": 98.7, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 93, "ERROR_RATE": 100.0, "MIN_TTFB": 58, "MAX_TTFB": 70, "MAX_DURATION": 109, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88194.0, "COUNT": 30}, {"from": 2000, "to": 3000, "AVG_TTFB": 67.8, "ERRORS": 30, "AVG_DURATION": 104.2, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 93, "ERROR_RATE": 100.0, "MIN_TTFB": 61, "MAX_TTFB": 87, "MAX_DURATION": 125, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88156.0, "COUNT": 30}, {"from": 3000, "to": 4000, "AVG_TTFB": 63.3, "ERRORS": 30, "AVG_DURATION": 97.166664, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 70, "MAX_DURATION": 108, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88188.0, "COUNT": 30}, {"from": 4000, "to": 5000, "AVG_TTFB": 60.589745, "ERRORS": 39, "AVG_DURATION": 95.38461, "ERRORS_PER_SECOND": 39.0, "MIN_DURATION": 86, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 74, "MAX_DURATION": 114, "ELEMENTS_PER_SECOND": 39.0, "THROUGHPUT": 114613.0, "COUNT": 39}, {"from": 5000, "to": 6000, "AVG_TTFB": 66.935486, "ERRORS": 31, "AVG_DURATION": 104.09677, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 96, "MAX_DURATION": 144, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91120.0, "COUNT": 31}, {"from": 6000, "to": 7000, "AVG_TTFB": 63.966667, "ERRORS": 30, "AVG_DURATION": 98.26667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 83, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 76, "MAX_DURATION": 113, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88203.0, "COUNT": 30}, {"from": 7000, "to": 8000, "AVG_TTFB": 62.533333, "ERRORS": 30, "AVG_DURATION": 96.666664, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 57, "MAX_TTFB": 69, "MAX_DURATION": 106, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88170.0, "COUNT": 30}, {"from": 8000, "to": 9000, "AVG_TTFB": 61.605263, "ERRORS": 38, "AVG_DURATION": 97.39474, "ERRORS_PER_SECOND": 38.0, "MIN_DURATION": 86, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 74, "MAX_DURATION": 107, "ELEMENTS_PER_SECOND": 38.0, "THROUGHPUT": 111714.0, "COUNT": 38}, {"from": 9000, "to": 10000, "AVG_TTFB": 62.09375, "ERRORS": 32, "AVG_DURATION": 96.78125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 83, "MAX_DURATION": 116, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94001.0, "COUNT": 32}, {"from": 10000, "to": 11000, "AVG_TTFB": 61.032257, "ERRORS": 31, "AVG_DURATION": 94.96774, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 82, "MAX_DURATION": 116, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91092.0, "COUNT": 31}, {"from": 11000, "to": 12000, "AVG_TTFB": 64.125, "ERRORS": 32, "AVG_DURATION": 101.5, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 86, "ERROR_RATE": 100.0, "MIN_TTFB": 53, "MAX_TTFB": 101, "MAX_DURATION": 151, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94043.0, "COUNT": 32}, {"from": 12000, "to": 13000, "AVG_TTFB": 65.83871, "ERRORS": 31, "AVG_DURATION": 100.741936, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 85, "ERROR_RATE": 100.0, "MIN_TTFB": 53, "MAX_TTFB": 113, "MAX_DURATION": 150, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91099.0, "COUNT": 31}, {"from": 13000, "to": 14000, "AVG_TTFB": 63.411766, "ERRORS": 34, "AVG_DURATION": 97.588234, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 83, "ERROR_RATE": 100.0, "MIN_TTFB": 52, "MAX_TTFB": 88, "MAX_DURATION": 123, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99904.0, "COUNT": 34}, {"from": 14000, "to": 15000, "AVG_TTFB": 62.15625, "ERRORS": 32, "AVG_DURATION": 96.46875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 86, "ERROR_RATE": 100.0, "MIN_TTFB": 53, "MAX_TTFB": 73, "MAX_DURATION": 112, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94029.0, "COUNT": 32}, {"from": 15000, "to": 16000, "AVG_TTFB": 62.17647, "ERRORS": 34, "AVG_DURATION": 96.26471, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 86, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 87, "MAX_DURATION": 125, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99916.0, "COUNT": 34}, {"from": 16000, "to": 17000, "AVG_TTFB": 83.8421, "ERRORS": 19, "AVG_DURATION": 139.42105, "ERRORS_PER_SECOND": 19.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 57, "MAX_TTFB": 210, "MAX_DURATION": 398, "ELEMENTS_PER_SECOND": 19.0, "THROUGHPUT": 55847.0, "COUNT": 19}, {"from": 17000, "to": 18000, "AVG_TTFB": 65.741936, "ERRORS": 31, "AVG_DURATION": 102.51613, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 92, "ERROR_RATE": 100.0, "MIN_TTFB": 57, "MAX_TTFB": 92, "MAX_DURATION": 135, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91121.0, "COUNT": 31}, {"from": 18000, "to": 19000, "AVG_TTFB": 64.44118, "ERRORS": 34, "AVG_DURATION": 99.911766, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 57, "MAX_TTFB": 95, "MAX_DURATION": 133, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99948.0, "COUNT": 34}, {"from": 19000, "to": 20000, "AVG_TTFB": 68.26667, "ERRORS": 30, "AVG_DURATION": 106.03333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 86, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 132, "MAX_DURATION": 166, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88181.0, "COUNT": 30}, {"from": 20000, "to": 21000, "AVG_TTFB": 65.0, "ERRORS": 30, "AVG_DURATION": 102.76667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 83, "ERROR_RATE": 100.0, "MIN_TTFB": 52, "MAX_TTFB": 106, "MAX_DURATION": 145, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88153.0, "COUNT": 30}, {"from": 21000, "to": 22000, "AVG_TTFB": 69.67857, "ERRORS": 28, "AVG_DURATION": 115.03571, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 86, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 223, "MAX_DURATION": 273, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82346.0, "COUNT": 28}, {"from": 22000, "to": 23000, "AVG_TTFB": 62.969696, "ERRORS": 33, "AVG_DURATION": 96.666664, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 81, "MAX_DURATION": 124, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96980.0, "COUNT": 33}, {"from": 23000, "to": 24000, "AVG_TTFB": 65.84375, "ERRORS": 32, "AVG_DURATION": 101.25, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 93, "MAX_DURATION": 127, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94108.0, "COUNT": 32}, {"from": 24000, "to": 25000, "AVG_TTFB": 63.411766, "ERRORS": 34, "AVG_DURATION": 97.76471, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 82, "MAX_DURATION": 115, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99937.0, "COUNT": 34}, {"from": 25000, "to": 26000, "AVG_TTFB": 64.83871, "ERRORS": 31, "AVG_DURATION": 97.16129, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 85, "ERROR_RATE": 100.0, "MIN_TTFB": 53, "MAX_TTFB": 113, "MAX_DURATION": 147, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91111.0, "COUNT": 31}, {"from": 26000, "to": 27000, "AVG_TTFB": 67.44827, "ERRORS": 29, "AVG_DURATION": 106.0, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 132, "MAX_DURATION": 170, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85223.0, "COUNT": 29}, {"from": 27000, "to": 28000, "AVG_TTFB": 68.617645, "ERRORS": 34, "AVG_DURATION": 104.911766, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 106, "MAX_DURATION": 169, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99902.0, "COUNT": 34}, {"from": 28000, "to": 29000, "AVG_TTFB": 63.15625, "ERRORS": 32, "AVG_DURATION": 98.0, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 86, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 96, "MAX_DURATION": 132, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94064.0, "COUNT": 32}, {"from": 29000, "to": 30000, "AVG_TTFB": 65.083336, "ERRORS": 24, "AVG_DURATION": 100.916664, "ERRORS_PER_SECOND": 24.0, "MIN_DURATION": 93, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 79, "MAX_DURATION": 112, "ELEMENTS_PER_SECOND": 24.0, "THROUGHPUT": 70542.0, "COUNT": 24}, {"from": 30000, "to": 31000, "AVG_TTFB": 69.36667, "ERRORS": 30, "AVG_DURATION": 107.666664, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 91, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 120, "MAX_DURATION": 151, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88202.0, "COUNT": 30}, {"from": 31000, "to": 32000, "AVG_TTFB": 63.694443, "ERRORS": 36, "AVG_DURATION": 98.19444, "ERRORS_PER_SECOND": 36.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 87, "MAX_DURATION": 118, "ELEMENTS_PER_SECOND": 36.0, "THROUGHPUT": 105808.0, "COUNT": 36}, {"from": 32000, "to": 33000, "AVG_TTFB": 63.419353, "ERRORS": 31, "AVG_DURATION": 98.064514, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 109, "MAX_DURATION": 143, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91098.0, "COUNT": 31}, {"from": 33000, "to": 34000, "AVG_TTFB": 61.545456, "ERRORS": 33, "AVG_DURATION": 96.454544, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 89, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 72, "MAX_DURATION": 109, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 97029.0, "COUNT": 33}, {"from": 34000, "to": 35000, "AVG_TTFB": 66.625, "ERRORS": 32, "AVG_DURATION": 103.65625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 85, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 98, "MAX_DURATION": 139, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94075.0, "COUNT": 32}, {"from": 35000, "to": 36000, "AVG_TTFB": 70.4, "ERRORS": 30, "AVG_DURATION": 110.9, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 136, "MAX_DURATION": 173, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88197.0, "COUNT": 30}, {"from": 36000, "to": 37000, "AVG_TTFB": 64.666664, "ERRORS": 33, "AVG_DURATION": 100.84849, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 102, "MAX_DURATION": 133, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96964.0, "COUNT": 33}, {"from": 37000, "to": 38000, "AVG_TTFB": 63.4, "ERRORS": 30, "AVG_DURATION": 98.46667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 89, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 80, "MAX_DURATION": 115, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88198.0, "COUNT": 30}, {"from": 38000, "to": 39000, "AVG_TTFB": 62.235294, "ERRORS": 34, "AVG_DURATION": 96.0, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 73, "MAX_DURATION": 105, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99903.0, "COUNT": 34}, {"from": 39000, "to": 40000, "AVG_TTFB": 61.5625, "ERRORS": 32, "AVG_DURATION": 95.5, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 68, "MAX_DURATION": 106, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94064.0, "COUNT": 32}, {"from": 40000, "to": 41000, "AVG_TTFB": 65.09677, "ERRORS": 31, "AVG_DURATION": 101.70968, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 84, "ERROR_RATE": 100.0, "MIN_TTFB": 52, "MAX_TTFB": 115, "MAX_DURATION": 150, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91102.0, "COUNT": 31}, {"from": 41000, "to": 42000, "AVG_TTFB": 63.085712, "ERRORS": 35, "AVG_DURATION": 96.51428, "ERRORS_PER_SECOND": 35.0, "MIN_DURATION": 86, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 114, "MAX_DURATION": 143, "ELEMENTS_PER_SECOND": 35.0, "THROUGHPUT": 102851.0, "COUNT": 35}, {"from": 42000, "to": 43000, "AVG_TTFB": 64.1875, "ERRORS": 32, "AVG_DURATION": 98.71875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 84, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 119, "MAX_DURATION": 155, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94042.0, "COUNT": 32}, {"from": 43000, "to": 44000, "AVG_TTFB": 64.46875, "ERRORS": 32, "AVG_DURATION": 99.15625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 89, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 113, "MAX_DURATION": 149, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94036.0, "COUNT": 32}, {"from": 44000, "to": 45000, "AVG_TTFB": 64.2, "ERRORS": 30, "AVG_DURATION": 98.066666, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 57, "MAX_TTFB": 109, "MAX_DURATION": 144, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88169.0, "COUNT": 30}, {"from": 45000, "to": 46000, "AVG_TTFB": 63.8, "ERRORS": 35, "AVG_DURATION": 97.97143, "ERRORS_PER_SECOND": 35.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 57, "MAX_TTFB": 115, "MAX_DURATION": 146, "ELEMENTS_PER_SECOND": 35.0, "THROUGHPUT": 102895.0, "COUNT": 35}, {"from": 46000, "to": 47000, "AVG_TTFB": 62.9375, "ERRORS": 32, "AVG_DURATION": 97.75, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 52, "MAX_TTFB": 112, "MAX_DURATION": 148, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94048.0, "COUNT": 32}, {"from": 47000, "to": 48000, "AVG_TTFB": 66.32258, "ERRORS": 31, "AVG_DURATION": 103.870964, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 91, "ERROR_RATE": 100.0, "MIN_TTFB": 58, "MAX_TTFB": 85, "MAX_DURATION": 121, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91073.0, "COUNT": 31}, {"from": 48000, "to": 49000, "AVG_TTFB": 62.633335, "ERRORS": 30, "AVG_DURATION": 98.333336, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 89, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 75, "MAX_DURATION": 112, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88134.0, "COUNT": 30}, {"from": 49000, "to": 50000, "AVG_TTFB": 62.47059, "ERRORS": 34, "AVG_DURATION": 97.35294, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 76, "MAX_DURATION": 113, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99922.0, "COUNT": 34}, {"from": 50000, "to": 51000, "AVG_TTFB": 64.89655, "ERRORS": 29, "AVG_DURATION": 103.51724, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 96, "MAX_DURATION": 153, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85217.0, "COUNT": 29}, {"from": 51000, "to": 52000, "AVG_TTFB": 66.125, "ERRORS": 32, "AVG_DURATION": 101.625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 85, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 127, "MAX_DURATION": 159, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94014.0, "COUNT": 32}, {"from": 52000, "to": 53000, "AVG_TTFB": 75.92308, "ERRORS": 26, "AVG_DURATION": 112.92308, "ERRORS_PER_SECOND": 26.0, "MIN_DURATION": 94, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 208, "MAX_DURATION": 242, "ELEMENTS_PER_SECOND": 26.0, "THROUGHPUT": 76432.0, "COUNT": 26}, {"from": 53000, "to": 54000, "AVG_TTFB": 66.06061, "ERRORS": 33, "AVG_DURATION": 102.21212, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 85, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 128, "MAX_DURATION": 160, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96948.0, "COUNT": 33}, {"from": 54000, "to": 55000, "AVG_TTFB": 63.612904, "ERRORS": 31, "AVG_DURATION": 99.80645, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 91, "ERROR_RATE": 100.0, "MIN_TTFB": 58, "MAX_TTFB": 78, "MAX_DURATION": 117, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91076.0, "COUNT": 31}, {"from": 55000, "to": 56000, "AVG_TTFB": 63.80645, "ERRORS": 31, "AVG_DURATION": 99.16129, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 126, "MAX_DURATION": 164, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91069.0, "COUNT": 31}, {"from": 56000, "to": 57000, "AVG_TTFB": 62.484848, "ERRORS": 33, "AVG_DURATION": 97.84849, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 89, "ERROR_RATE": 100.0, "MIN_TTFB": 58, "MAX_TTFB": 71, "MAX_DURATION": 108, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96968.0, "COUNT": 33}, {"from": 57000, "to": 58000, "AVG_TTFB": 66.8, "ERRORS": 30, "AVG_DURATION": 103.36667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 58, "MAX_TTFB": 81, "MAX_DURATION": 119, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88141.0, "COUNT": 30}, {"from": 58000, "to": 59000, "AVG_TTFB": 70.23333, "ERRORS": 30, "AVG_DURATION": 108.13333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 91, "ERROR_RATE": 100.0, "MIN_TTFB": 58, "MAX_TTFB": 113, "MAX_DURATION": 153, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88164.0, "COUNT": 30}, {"from": 59000, "to": 60000, "AVG_TTFB": 68.413795, "ERRORS": 22, "AVG_DURATION": 106.93104, "ERRORS_PER_SECOND": 22.0, "MIN_DURATION": 90, "ERROR_RATE": 75.86207, "MIN_TTFB": 55, "MAX_TTFB": 110, "MAX_DURATION": 142, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 75416.0, "COUNT": 29}, {"from": 60000, "to": 61000, "AVG_TTFB": 65.066666, "ERRORS": 20, "AVG_DURATION": 103.566666, "ERRORS_PER_SECOND": 20.0, "MIN_DURATION": 91, "ERROR_RATE": 66.666664, "MIN_TTFB": 56, "MAX_TTFB": 105, "MAX_DURATION": 147, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 74159.0, "COUNT": 30}, {"from": 61000, "to": 62000, "AVG_TTFB": 66.28125, "ERRORS": 32, "AVG_DURATION": 101.6875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 89, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 126, "MAX_DURATION": 156, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94056.0, "COUNT": 32}, {"from": 62000, "to": 63000, "AVG_TTFB": 65.48276, "ERRORS": 29, "AVG_DURATION": 102.37931, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 89, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 135, "MAX_DURATION": 171, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85229.0, "COUNT": 29}, {"from": 63000, "to": 64000, "AVG_TTFB": 62.114285, "ERRORS": 35, "AVG_DURATION": 98.28571, "ERRORS_PER_SECOND": 35.0, "MIN_DURATION": 86, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 74, "MAX_DURATION": 116, "ELEMENTS_PER_SECOND": 35.0, "THROUGHPUT": 102866.0, "COUNT": 35}, {"from": 64000, "to": 65000, "AVG_TTFB": 63.4375, "ERRORS": 32, "AVG_DURATION": 97.21875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 136, "MAX_DURATION": 168, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94069.0, "COUNT": 32}, {"from": 65000, "to": 66000, "AVG_TTFB": 73.413795, "ERRORS": 29, "AVG_DURATION": 111.13793, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 140, "MAX_DURATION": 174, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85243.0, "COUNT": 29}, {"from": 66000, "to": 67000, "AVG_TTFB": 65.5, "ERRORS": 32, "AVG_DURATION": 102.125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 91, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 89, "MAX_DURATION": 127, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94013.0, "COUNT": 32}, {"from": 67000, "to": 68000, "AVG_TTFB": 65.90323, "ERRORS": 31, "AVG_DURATION": 102.354836, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 89, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 117, "MAX_DURATION": 149, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91099.0, "COUNT": 31}, {"from": 68000, "to": 69000, "AVG_TTFB": 65.451614, "ERRORS": 31, "AVG_DURATION": 100.70968, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 89, "ERROR_RATE": 100.0, "MIN_TTFB": 57, "MAX_TTFB": 82, "MAX_DURATION": 116, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91091.0, "COUNT": 31}, {"from": 69000, "to": 70000, "AVG_TTFB": 68.1, "ERRORS": 30, "AVG_DURATION": 104.066666, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 58, "MAX_TTFB": 147, "MAX_DURATION": 178, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88163.0, "COUNT": 30}, {"from": 70000, "to": 71000, "AVG_TTFB": 64.41936, "ERRORS": 31, "AVG_DURATION": 100.03226, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 91, "MAX_DURATION": 126, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91127.0, "COUNT": 31}, {"from": 71000, "to": 72000, "AVG_TTFB": 64.548386, "ERRORS": 31, "AVG_DURATION": 100.41936, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 57, "MAX_TTFB": 133, "MAX_DURATION": 164, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91077.0, "COUNT": 31}, {"from": 72000, "to": 73000, "AVG_TTFB": 70.2, "ERRORS": 30, "AVG_DURATION": 109.73333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 147, "MAX_DURATION": 184, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88210.0, "COUNT": 30}, {"from": 73000, "to": 74000, "AVG_TTFB": 70.064514, "ERRORS": 31, "AVG_DURATION": 104.6129, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 58, "MAX_TTFB": 134, "MAX_DURATION": 167, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91093.0, "COUNT": 31}, {"from": 74000, "to": 75000, "AVG_TTFB": 62.84375, "ERRORS": 32, "AVG_DURATION": 97.96875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 89, "ERROR_RATE": 100.0, "MIN_TTFB": 57, "MAX_TTFB": 74, "MAX_DURATION": 109, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94088.0, "COUNT": 32}, {"from": 75000, "to": 76000, "AVG_TTFB": 62.117645, "ERRORS": 34, "AVG_DURATION": 97.35294, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 70, "MAX_DURATION": 106, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99944.0, "COUNT": 34}, {"from": 76000, "to": 77000, "AVG_TTFB": 64.166664, "ERRORS": 30, "AVG_DURATION": 99.36667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 89, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 96, "MAX_DURATION": 130, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88136.0, "COUNT": 30}, {"from": 77000, "to": 78000, "AVG_TTFB": 61.78788, "ERRORS": 33, "AVG_DURATION": 97.606064, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 72, "MAX_DURATION": 115, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96957.0, "COUNT": 33}, {"from": 78000, "to": 79000, "AVG_TTFB": 67.64286, "ERRORS": 28, "AVG_DURATION": 106.07143, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 57, "MAX_TTFB": 78, "MAX_DURATION": 136, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82271.0, "COUNT": 28}, {"from": 79000, "to": 80000, "AVG_TTFB": 64.875, "ERRORS": 32, "AVG_DURATION": 99.40625, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 77, "MAX_DURATION": 111, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94042.0, "COUNT": 32}, {"from": 80000, "to": 81000, "AVG_TTFB": 64.41936, "ERRORS": 31, "AVG_DURATION": 100.58064, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 91, "ERROR_RATE": 100.0, "MIN_TTFB": 58, "MAX_TTFB": 74, "MAX_DURATION": 155, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91101.0, "COUNT": 31}, {"from": 81000, "to": 82000, "AVG_TTFB": 63.59375, "ERRORS": 32, "AVG_DURATION": 97.71875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 86, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 75, "MAX_DURATION": 109, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94053.0, "COUNT": 32}, {"from": 82000, "to": 83000, "AVG_TTFB": 72.064514, "ERRORS": 31, "AVG_DURATION": 108.80645, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 86, "ERROR_RATE": 100.0, "MIN_TTFB": 53, "MAX_TTFB": 150, "MAX_DURATION": 193, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91083.0, "COUNT": 31}, {"from": 83000, "to": 84000, "AVG_TTFB": 65.46667, "ERRORS": 30, "AVG_DURATION": 101.5, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 91, "ERROR_RATE": 100.0, "MIN_TTFB": 58, "MAX_TTFB": 81, "MAX_DURATION": 131, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88178.0, "COUNT": 30}, {"from": 84000, "to": 85000, "AVG_TTFB": 64.53125, "ERRORS": 32, "AVG_DURATION": 100.28125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 84, "ERROR_RATE": 100.0, "MIN_TTFB": 53, "MAX_TTFB": 81, "MAX_DURATION": 117, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94064.0, "COUNT": 32}, {"from": 85000, "to": 86000, "AVG_TTFB": 63.969696, "ERRORS": 33, "AVG_DURATION": 102.393936, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 96, "MAX_DURATION": 132, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96945.0, "COUNT": 33}, {"from": 86000, "to": 87000, "AVG_TTFB": 61.48387, "ERRORS": 31, "AVG_DURATION": 96.6129, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 85, "ERROR_RATE": 100.0, "MIN_TTFB": 53, "MAX_TTFB": 72, "MAX_DURATION": 107, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91040.0, "COUNT": 31}, {"from": 87000, "to": 88000, "AVG_TTFB": 68.242424, "ERRORS": 33, "AVG_DURATION": 102.454544, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 180, "MAX_DURATION": 214, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96987.0, "COUNT": 33}, {"from": 88000, "to": 89000, "AVG_TTFB": 61.18182, "ERRORS": 33, "AVG_DURATION": 95.72727, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 85, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 82, "MAX_DURATION": 115, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96963.0, "COUNT": 33}, {"from": 89000, "to": 90000, "AVG_TTFB": 87.916664, "ERRORS": 24, "AVG_DURATION": 121.666664, "ERRORS_PER_SECOND": 24.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 324, "MAX_DURATION": 367, "ELEMENTS_PER_SECOND": 24.0, "THROUGHPUT": 70513.0, "COUNT": 24}, {"from": 90000, "to": 91000, "AVG_TTFB": 61.727272, "ERRORS": 33, "AVG_DURATION": 95.51515, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 53, "MAX_TTFB": 76, "MAX_DURATION": 108, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96955.0, "COUNT": 33}, {"from": 91000, "to": 92000, "AVG_TTFB": 64.5, "ERRORS": 32, "AVG_DURATION": 98.1875, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 149, "MAX_DURATION": 188, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94055.0, "COUNT": 32}, {"from": 92000, "to": 93000, "AVG_TTFB": 61.78788, "ERRORS": 33, "AVG_DURATION": 95.36364, "ERRORS_PER_SECOND": 33.0, "MIN_DURATION": 89, "ERROR_RATE": 100.0, "MIN_TTFB": 53, "MAX_TTFB": 72, "MAX_DURATION": 104, "ELEMENTS_PER_SECOND": 33.0, "THROUGHPUT": 96979.0, "COUNT": 33}, {"from": 93000, "to": 94000, "AVG_TTFB": 63.28125, "ERRORS": 32, "AVG_DURATION": 97.75, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 155, "MAX_DURATION": 188, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94020.0, "COUNT": 32}, {"from": 94000, "to": 95000, "AVG_TTFB": 78.14286, "ERRORS": 28, "AVG_DURATION": 117.82143, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 86, "ERROR_RATE": 100.0, "MIN_TTFB": 54, "MAX_TTFB": 264, "MAX_DURATION": 302, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82290.0, "COUNT": 28}, {"from": 95000, "to": 96000, "AVG_TTFB": 65.64706, "ERRORS": 34, "AVG_DURATION": 100.411766, "ERRORS_PER_SECOND": 34.0, "MIN_DURATION": 84, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 155, "MAX_DURATION": 189, "ELEMENTS_PER_SECOND": 34.0, "THROUGHPUT": 99904.0, "COUNT": 34}, {"from": 96000, "to": 97000, "AVG_TTFB": 63.612904, "ERRORS": 31, "AVG_DURATION": 98.451614, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 58, "MAX_TTFB": 76, "MAX_DURATION": 118, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91108.0, "COUNT": 31}, {"from": 97000, "to": 98000, "AVG_TTFB": 68.4375, "ERRORS": 32, "AVG_DURATION": 102.3125, "ERRORS_PER_SECOND": 32.0, "MIN_DURATION": 87, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 183, "MAX_DURATION": 218, "ELEMENTS_PER_SECOND": 32.0, "THROUGHPUT": 94040.0, "COUNT": 32}, {"from": 98000, "to": 99000, "AVG_TTFB": 70.5, "ERRORS": 30, "AVG_DURATION": 105.46667, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 88, "ERROR_RATE": 100.0, "MIN_TTFB": 57, "MAX_TTFB": 164, "MAX_DURATION": 203, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88148.0, "COUNT": 30}, {"from": 99000, "to": 100000, "AVG_TTFB": 74.6129, "ERRORS": 31, "AVG_DURATION": 110.83871, "ERRORS_PER_SECOND": 31.0, "MIN_DURATION": 92, "ERROR_RATE": 100.0, "MIN_TTFB": 60, "MAX_TTFB": 163, "MAX_DURATION": 198, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 91111.0, "COUNT": 31}, {"from": 100000, "to": 101000, "AVG_TTFB": 67.413795, "ERRORS": 29, "AVG_DURATION": 106.68965, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 96, "ERROR_RATE": 100.0, "MIN_TTFB": 53, "MAX_TTFB": 81, "MAX_DURATION": 119, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85254.0, "COUNT": 29}, {"from": 101000, "to": 102000, "AVG_TTFB": 99.541664, "ERRORS": 24, "AVG_DURATION": 144.375, "ERRORS_PER_SECOND": 24.0, "MIN_DURATION": 92, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 314, "MAX_DURATION": 356, "ELEMENTS_PER_SECOND": 24.0, "THROUGHPUT": 70586.0, "COUNT": 24}, {"from": 102000, "to": 103000, "AVG_TTFB": 104.68421, "ERRORS": 19, "AVG_DURATION": 157.78947, "ERRORS_PER_SECOND": 19.0, "MIN_DURATION": 131, "ERROR_RATE": 100.0, "MIN_TTFB": 87, "MAX_TTFB": 125, "MAX_DURATION": 185, "ELEMENTS_PER_SECOND": 19.0, "THROUGHPUT": 55817.0, "COUNT": 19}, {"from": 103000, "to": 104000, "AVG_TTFB": 97.09524, "ERRORS": 21, "AVG_DURATION": 148.80952, "ERRORS_PER_SECOND": 21.0, "MIN_DURATION": 118, "ERROR_RATE": 100.0, "MIN_TTFB": 81, "MAX_TTFB": 124, "MAX_DURATION": 189, "ELEMENTS_PER_SECOND": 21.0, "THROUGHPUT": 61727.0, "COUNT": 21}, {"from": 104000, "to": 105000, "AVG_TTFB": 122.0, "ERRORS": 19, "AVG_DURATION": 184.3158, "ERRORS_PER_SECOND": 19.0, "MIN_DURATION": 147, "ERROR_RATE": 100.0, "MIN_TTFB": 93, "MAX_TTFB": 222, "MAX_DURATION": 308, "ELEMENTS_PER_SECOND": 19.0, "THROUGHPUT": 55823.0, "COUNT": 19}, {"from": 105000, "to": 106000, "AVG_TTFB": 121.8421, "ERRORS": 19, "AVG_DURATION": 175.1579, "ERRORS_PER_SECOND": 19.0, "MIN_DURATION": 128, "ERROR_RATE": 100.0, "MIN_TTFB": 87, "MAX_TTFB": 225, "MAX_DURATION": 275, "ELEMENTS_PER_SECOND": 19.0, "THROUGHPUT": 55811.0, "COUNT": 19}, {"from": 106000, "to": 107000, "AVG_TTFB": 68.92857, "ERRORS": 28, "AVG_DURATION": 105.96429, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 55, "MAX_TTFB": 152, "MAX_DURATION": 186, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82302.0, "COUNT": 28}, {"from": 107000, "to": 108000, "AVG_TTFB": 75.17857, "ERRORS": 28, "AVG_DURATION": 113.82143, "ERRORS_PER_SECOND": 28.0, "MIN_DURATION": 94, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 170, "MAX_DURATION": 213, "ELEMENTS_PER_SECOND": 28.0, "THROUGHPUT": 82313.0, "COUNT": 28}, {"from": 108000, "to": 109000, "AVG_TTFB": 76.03846, "ERRORS": 26, "AVG_DURATION": 117.42308, "ERRORS_PER_SECOND": 26.0, "MIN_DURATION": 94, "ERROR_RATE": 100.0, "MIN_TTFB": 57, "MAX_TTFB": 94, "MAX_DURATION": 140, "ELEMENTS_PER_SECOND": 26.0, "THROUGHPUT": 76464.0, "COUNT": 26}, {"from": 109000, "to": 110000, "AVG_TTFB": 81.965515, "ERRORS": 29, "AVG_DURATION": 124.68965, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 106, "ERROR_RATE": 100.0, "MIN_TTFB": 68, "MAX_TTFB": 107, "MAX_DURATION": 151, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85221.0, "COUNT": 29}, {"from": 110000, "to": 111000, "AVG_TTFB": 89.7619, "ERRORS": 21, "AVG_DURATION": 138.4762, "ERRORS_PER_SECOND": 21.0, "MIN_DURATION": 122, "ERROR_RATE": 100.0, "MIN_TTFB": 68, "MAX_TTFB": 112, "MAX_DURATION": 157, "ELEMENTS_PER_SECOND": 21.0, "THROUGHPUT": 61717.0, "COUNT": 21}, {"from": 111000, "to": 112000, "AVG_TTFB": 73.7, "ERRORS": 30, "AVG_DURATION": 111.333336, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 93, "ERROR_RATE": 100.0, "MIN_TTFB": 59, "MAX_TTFB": 106, "MAX_DURATION": 156, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88190.0, "COUNT": 30}, {"from": 112000, "to": 113000, "AVG_TTFB": 74.9, "ERRORS": 30, "AVG_DURATION": 112.63333, "ERRORS_PER_SECOND": 30.0, "MIN_DURATION": 89, "ERROR_RATE": 100.0, "MIN_TTFB": 56, "MAX_TTFB": 173, "MAX_DURATION": 206, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 88134.0, "COUNT": 30}, {"from": 113000, "to": 114000, "AVG_TTFB": 66.89655, "ERRORS": 29, "AVG_DURATION": 105.37931, "ERRORS_PER_SECOND": 29.0, "MIN_DURATION": 90, "ERROR_RATE": 100.0, "MIN_TTFB": 57, "MAX_TTFB": 85, "MAX_DURATION": 130, "ELEMENTS_PER_SECOND": 29.0, "THROUGHPUT": 85181.0, "COUNT": 29}, {"from": 114000, "to": 115000, "AVG_TTFB": 83.16, "ERRORS": 25, "AVG_DURATION": 133.16, "ERRORS_PER_SECOND": 25.0, "MIN_DURATION": 98, "ERROR_RATE": 100.0, "MIN_TTFB": 60, "MAX_TTFB": 126, "MAX_DURATION": 178, "ELEMENTS_PER_SECOND": 25.0, "THROUGHPUT": 73464.0, "COUNT": 25}, {"from": 115000, "to": 116000, "AVG_TTFB": 91.77273, "ERRORS": 22, "AVG_DURATION": 141.86363, "ERRORS_PER_SECOND": 22.0, "MIN_DURATION": 113, "ERROR_RATE": 100.0, "MIN_TTFB": 68, "MAX_TTFB": 116, "MAX_DURATION": 170, "ELEMENTS_PER_SECOND": 22.0, "THROUGHPUT": 64619.0, "COUNT": 22}, {"from": 116000, "to": 117000, "AVG_TTFB": 109.22222, "ERRORS": 18, "AVG_DURATION": 170.27777, "ERRORS_PER_SECOND": 18.0, "MIN_DURATION": 120, "ERROR_RATE": 100.0, "MIN_TTFB": 74, "MAX_TTFB": 133, "MAX_DURATION": 198, "ELEMENTS_PER_SECOND": 18.0, "THROUGHPUT": 52874.0, "COUNT": 18}, {"from": 117000, "to": 118000, "AVG_TTFB": 132.73685, "ERRORS": 19, "AVG_DURATION": 190.10527, "ERRORS_PER_SECOND": 19.0, "MIN_DURATION": 152, "ERROR_RATE": 100.0, "MIN_TTFB": 97, "MAX_TTFB": 219, "MAX_DURATION": 294, "ELEMENTS_PER_SECOND": 19.0, "THROUGHPUT": 55790.0, "COUNT": 19}, {"from": 118000, "to": 119000, "AVG_TTFB": 114.588234, "ERRORS": 17, "AVG_DURATION": 172.64706, "ERRORS_PER_SECOND": 17.0, "MIN_DURATION": 146, "ERROR_RATE": 100.0, "MIN_TTFB": 95, "MAX_TTFB": 139, "MAX_DURATION": 193, "ELEMENTS_PER_SECOND": 17.0, "THROUGHPUT": 49950.0, "COUNT": 17}, {"from": 119000, "to": 120000, "AVG_TTFB": 136.94444, "ERRORS": 14, "AVG_DURATION": 205.88889, "ERRORS_PER_SECOND": 14.0, "MIN_DURATION": 166, "ERROR_RATE": 77.77778, "MIN_TTFB": 107, "MAX_TTFB": 240, "MAX_DURATION": 331, "ELEMENTS_PER_SECOND": 18.0, "THROUGHPUT": 47310.0, "COUNT": 18}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/monitors':
            return '[{"id": "de6f330b-eecb-32cf-8582-63e0e1654735", "name": "User Load", "path": ["Controller", "User Load"]}, {"id": "1949328f-ad88-3770-933a-f104f026f14c", "name": "Throughput", "path": ["Controller", "Throughput"]}, {"id": "424155ec-15a1-3b91-8261-0ae15cda2a19", "name": "CPU Load", "path": ["Controller", "CPU Load"]}, {"id": "8f0f9bab-9cf2-3958-9085-4414999b5c0d", "name": "Memory Used", "path": ["Controller", "Memory Used"]}, {"id": "a471ab71-378c-3e5b-b119-6c9b20b86106", "name": "Thread Count", "path": ["Controller", "Thread Count"]}, {"id": "16099c2c-e210-3e3d-acfb-2898cf8c2e97", "name": "Connected", "path": ["Controller", "Load Generator Connections", "Connected"]}, {"id": "5bffd188-0ea4-389e-92f0-03c3056a54cf", "name": "Total Disconnections", "path": ["Controller", "Load Generator Connections", "Total Disconnections"]}, {"id": "26b4b692-5398-30dc-9734-4d3e27099bab", "name": "Total Reconnections", "path": ["Controller", "Load Generator Connections", "Total Reconnections"]}, {"id": "31df4f31-f46e-3ff7-b8e9-d331d8f5255e", "name": "Segments sent/sec.", "path": ["Controller", "Network", "Segments sent/sec."]}, {"id": "405c96a0-5e8d-3666-bd85-5efa626b8ecc", "name": "Segments retransmitted/sec.", "path": ["Controller", "Network", "Segments retransmitted/sec."]}, {"id": "4e3f6686-51c9-36b7-8d67-085976e5af37", "name": "Megabits received/sec.", "path": ["Controller", "Network", "eth7 (Intel(R) Ethernet Connection (2) I219-LM)", "Megabits received/sec."]}, {"id": "a066d995-670f-3b4a-95e2-50b89018dc21", "name": "Megabits sent/sec.", "path": ["Controller", "Network", "eth7 (Intel(R) Ethernet Connection (2) I219-LM)", "Megabits sent/sec."]}, {"id": "93ba2663-a5df-312e-9ec6-904c4cf636de", "name": "Megabits received/sec.", "path": ["Controller", "NeoLoad Web", "Megabits received/sec."]}, {"id": "2904ab8d-da00-343c-95c6-5dc8cd140339", "name": "Megabits sent/sec.", "path": ["Controller", "NeoLoad Web", "Megabits sent/sec."]}, {"id": "79d6d662-3504-3ba9-9e47-297e28f29db9", "name": "User Load", "path": ["Default zone", "User Load"]}, {"id": "ad6e903c-a684-3ef0-ae0a-b582eeb2d23f", "name": "User Load", "path": ["Default zone", "REST_API_Population", "User Load"]}, {"id": "601c9770-bc3c-3e1b-a3fc-99fdb37ea61e", "name": "User Load", "path": ["Default zone", "REST_API_Population", "kamoulox", "User Load"]}, {"id": "b9316efa-fcbc-3a13-b8df-d3af2ad7630d", "name": "CPU Load", "path": ["Default zone", "LG PC087:7110", "CPU Load"]}, {"id": "915f834b-10d4-3449-bef7-128e8258b9e9", "name": "Memory Used", "path": ["Default zone", "LG PC087:7110", "Memory Used"]}, {"id": "1b35682f-e0f3-3184-8b29-abd663698938", "name": "Throughput", "path": ["Default zone", "LG PC087:7110", "Throughput"]}, {"id": "db7a8e31-b67a-31f5-9f9c-194995e3fd0f", "name": "User Load", "path": ["Default zone", "LG PC087:7110", "User Load"]}, {"id": "5789ec0c-6dcb-3a83-a3fb-6f92089a8733", "name": "Thread Count", "path": ["Default zone", "LG PC087:7110", "Thread Count"]}, {"id": "3f147404-be99-38b9-bfd0-7eeb679d1c6e", "name": "Thread Pool used", "path": ["Default zone", "LG PC087:7110", "Http client", "Thread Pool used"]}, {"id": "3ef5f25e-2c6d-3132-a27e-268f4ae29e45", "name": "Task Pool used", "path": ["Default zone", "LG PC087:7110", "Http client", "Task Pool used"]}, {"id": "ad4e984e-630f-3468-bd33-9edb2bb4ef14", "name": "Ping Duration (ms)", "path": ["Default zone", "LG PC087:7110", "Controller Connection", "Ping Duration (ms)"]}, {"id": "c80f2b4f-6326-3597-86e7-1651b1cb91f8", "name": "Segments sent/sec.", "path": ["Default zone", "LG PC087:7110", "Network", "Segments sent/sec."]}, {"id": "7d0f46ef-d93f-3e5e-a359-f55ea728847f", "name": "Segments retransmitted/sec.", "path": ["Default zone", "LG PC087:7110", "Network", "Segments retransmitted/sec."]}, {"id": "73981a68-f29f-33cd-908d-a1b9d38a4146", "name": "Megabits received/sec.", "path": ["Default zone", "LG PC087:7110", "Network", "eth7 (Intel(R) Ethernet Connection (2) I219-LM)", "Megabits received/sec."]}, {"id": "565dfa78-aa2d-35e8-890d-cb14b56318f1", "name": "Megabits sent/sec.", "path": ["Default zone", "LG PC087:7110", "Network", "eth7 (Intel(R) Ethernet Connection (2) I219-LM)", "Megabits sent/sec."]}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/monitors/de6f330b-eecb-32cf-8582-63e0e1654735/points':
            return '[{"from":0,"to":1000,"AVG":0},{"from":1000,"to":2000,"AVG":10},{"from":2000,"to":3000,"AVG":10}]'

        elif endpoint.startswith \
                ("v3/workspaces/5f689c3f0860270001606902/test-results/c6ae22a9-9868-4966-8b1e-8439b985e792/monitors/"):
            return '[]'

        else:
            raise Exception('Endpoint NOT mocked : ' + endpoint)

    @pytest.mark.datafiles('tests/resources/report/raw_expected_report.json')
    def test_parse_source_data_spec_raw(self, monkeypatch, datafiles):
        runner = CliRunner()
        result_ws = runner.invoke(workspaces, ['use', '5f689c3f0860270001606902'])
        assert_success(result_ws)

        monkeypatch.setattr(rest_crud, 'get', lambda actual_endpoint: ast.literal_eval(self.__return_raw_json(actual_endpoint)))
        monkeypatch.setattr(time, 'localtime', lambda timestamp_utc: time.gmtime(timestamp_utc))
        monkeypatch.setattr(UserData, 'get_version', lambda version: 'SaaS')
        model = report.initialize_model("timespan=0%-100%", "")
        json_data = report.parse_source_data_spec(None, model, "single", "1c393548-1e78-4632-8dd1-b661fb6cd043")

        result_logout = runner.invoke(logout)
        assert_success(result_logout)

        with open('tests/resources/report/raw_actual_report.json', 'w', newline='\n') as f:
            f.write(json.dumps(json_data, indent=2))
        assert filecmp.cmp(datafiles.listdir()[0], 'tests/resources/report/raw_actual_report.json') is True, "Json output for the report (file tests/resources/report/raw_actual_report.json) is not the one expected (file tests/resources/report/raw_expected_report.json)"

    def __return_raw_json(self, endpoint):
        if endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043':
            return '{"id": "1c393548-1e78-4632-8dd1-b661fb6cd043", "name": "#12", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "rest_api", "scenario": "short_10s", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1613144874115, "endDate": 1613144884271, "duration": 10156, "testId": "274191ff-0f65-4b11-8828-2bd9e9b956ff"}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/slas/statistics':
            return '[{"kpi": "count", "status": "FAILED", "value": 2153.0, "failedThreshold": {"operator": ">=", "values": [0.0]}}, {"kpi": "throughput", "status": "FAILED", "value": 2.92, "failedThreshold": {"operator": ">=", "values": [0.0]}}, {"kpi": "error-rate", "status": "FAILED", "value": 46.7255, "failedThreshold": {"operator": ">=", "values": [0.0]}}, {"kpi": "errors-count", "status": "WARNING", "value": 1006.0, "warningThreshold": {"operator": ">=", "values": [0.0]}}, {"kpi": "avg-throughput-per-sec", "status": "WARNING", "value": 2.1263838, "warningThreshold": {"operator": ">=", "values": [0.0]}}, {"kpi": "avg-request-per-sec", "status": "WARNING", "value": 195.72728, "warningThreshold": {"operator": ">=", "values": [0.0]}}, {"kpi": "avg-request-resp-time", "status": "PASSED", "value": 0.044, "warningThreshold": {"operator": ">=", "values": [3600.0]}, "failedThreshold": {"operator": ">=", "values": [3601.0]}}, {"kpi": "avg-transaction-resp-time", "status": "PASSED", "value": 0.089, "warningThreshold": {"operator": ">=", "values": [3600.0]}, "failedThreshold": {"operator": ">=", "values": [3601.0]}}, {"kpi": "avg-page-resp-time", "status": "PASSED", "value": 0.049, "warningThreshold": {"operator": ">=", "values": [3600.0]}, "failedThreshold": {"operator": ">=", "values": [3601.0]}}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/slas/per-test':
            return '[{"kpi": "errors-count", "status": "FAILED", "value": 247.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "6a0ef850-8644-4236-80e5-53748fbd2d01", "name": "transaction_sla_failed", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "errors-count", "status": "FAILED", "value": 0.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d15f2c71-0e0b-454f-9802-2d9a5cc9baac", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "errors-count", "status": "FAILED", "value": 0.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7bf42114-d630-494f-bc13-fea402ce085d", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "errors-count", "status": "FAILED", "value": 247.0, "failedThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_failed"}}, {"kpi": "errors-count", "status": "WARNING", "value": 0.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "d319f4e8-d31e-4a1b-9187-b8238bc57007", "name": "/", "category": "REQUEST", "userpath": "kamoulox", "parent": "page"}}, {"kpi": "errors-count", "status": "WARNING", "value": 246.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a", "name": "transaction_sla_warning", "category": "TRANSACTION", "userpath": "kamoulox", "parent": "Actions"}}, {"kpi": "errors-count", "status": "WARNING", "value": 249.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "1034d624-451c-4523-b579-6f003ebd93f8", "name": "/api/webhooks/last", "category": "REQUEST", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "errors-count", "status": "WARNING", "value": 0.0, "warningThreshold": {"operator": ">=", "values": [0.0]}, "element": {"elementId": "b24becfd-0be5-4e27-93c9-c7ce4a7db5e3", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_warning"}}, {"kpi": "avg-page-resp-time", "status": "PASSED", "value": 0.038826413, "warningThreshold": {"operator": ">=", "values": [3600.0]}, "failedThreshold": {"operator": ">=", "values": [36000.0]}, "element": {"elementId": "1ea4878e-6df1-4c5b-bd5d-3c1f0566c91d", "name": "page", "category": "PAGE", "userpath": "kamoulox", "parent": "transaction_sla_passed"}}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/slas/per-interval':
            return '[]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/statistics':
            return '{"totalRequestCountSuccess": 1147, "totalRequestCountFailure": 1006, "totalRequestDurationAverage": 44.33024, "totalRequestCountPerSecond": 211.9929, "totalTransactionCountSuccess": 69, "totalTransactionCountFailure": 999, "totalTransactionDurationAverage": 88.656364, "totalTransactionCountPerSecond": 105.159515, "totalIterationCountSuccess": 10, "totalIterationCountFailure": 255, "totalGlobalDownloadedBytes": 3065803, "totalGlobalDownloadedBytesPerSecond": 301871.12, "totalGlobalCountFailure": 1006}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/events':
            return '[{"id": "1006", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10130, "type": "ERROR", "duration": 0, "source": "kamoulox#7 - Actions#28"}, {"id": "1004", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10129, "type": "ERROR", "duration": 0, "source": "kamoulox#4 - Actions#27"}, {"id": "1005", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10129, "type": "ERROR", "duration": 0, "source": "kamoulox#2 - Actions#28"}, {"id": "1003", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10126, "type": "ERROR", "duration": 0, "source": "kamoulox#5 - Actions#28"}, {"id": "1002", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10118, "type": "ERROR", "duration": 0, "source": "kamoulox#1 - Actions#28"}, {"id": "1000", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10103, "type": "ERROR", "duration": 0, "source": "kamoulox#6 - Actions#28"}, {"id": "1001", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10103, "type": "ERROR", "duration": 0, "source": "kamoulox#9 - Actions#28"}, {"id": "999", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10076, "type": "ERROR", "duration": 0, "source": "kamoulox#8 - Actions#24"}, {"id": "997", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10071, "type": "ERROR", "duration": 0, "source": "kamoulox#3 - Actions#28"}, {"id": "998", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10071, "type": "ERROR", "duration": 0, "source": "kamoulox#0 - Actions#28"}, {"id": "996", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10055, "type": "ERROR", "duration": 0, "source": "kamoulox#5 - Actions#28"}, {"id": "995", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10045, "type": "ERROR", "duration": 0, "source": "kamoulox#4 - Actions#27"}, {"id": "994", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10038, "type": "ERROR", "duration": 0, "source": "kamoulox#1 - Actions#28"}, {"id": "993", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10034, "type": "ERROR", "duration": 0, "source": "kamoulox#7 - Actions#27"}, {"id": "992", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10033, "type": "ERROR", "duration": 0, "source": "kamoulox#2 - Actions#27"}, {"id": "991", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 10007, "type": "ERROR", "duration": 0, "source": "kamoulox#8 - Actions#24"}, {"id": "989", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9986, "type": "ERROR", "duration": 0, "source": "kamoulox#9 - Actions#27"}, {"id": "990", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9986, "type": "ERROR", "duration": 0, "source": "kamoulox#6 - Actions#27"}, {"id": "988", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9965, "type": "ERROR", "duration": 0, "source": "kamoulox#0 - Actions#27"}, {"id": "987", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9964, "type": "ERROR", "duration": 0, "source": "kamoulox#3 - Actions#27"}, {"id": "986", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9954, "type": "ERROR", "duration": 0, "source": "kamoulox#7 - Actions#27"}, {"id": "985", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9952, "type": "ERROR", "duration": 0, "source": "kamoulox#2 - Actions#27"}, {"id": "984", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9945, "type": "ERROR", "duration": 0, "source": "kamoulox#5 - Actions#27"}, {"id": "983", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9942, "type": "ERROR", "duration": 0, "source": "kamoulox#4 - Actions#26"}, {"id": "982", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9935, "type": "ERROR", "duration": 0, "source": "kamoulox#8 - Actions#24"}, {"id": "981", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9932, "type": "ERROR", "duration": 0, "source": "kamoulox#1 - Actions#27"}, {"id": "980", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9920, "type": "ERROR", "duration": 0, "source": "kamoulox#6 - Actions#27"}, {"id": "979", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9912, "type": "ERROR", "duration": 0, "source": "kamoulox#9 - Actions#27"}, {"id": "978", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9902, "type": "ERROR", "duration": 0, "source": "kamoulox#0 - Actions#27"}, {"id": "977", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9901, "type": "ERROR", "duration": 0, "source": "kamoulox#3 - Actions#27"}, {"id": "976", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9887, "type": "ERROR", "duration": 0, "source": "kamoulox#2 - Actions#27"}, {"id": "975", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9882, "type": "ERROR", "duration": 0, "source": "kamoulox#7 - Actions#27"}, {"id": "974", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9877, "type": "ERROR", "duration": 0, "source": "kamoulox#5 - Actions#27"}, {"id": "973", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9875, "type": "ERROR", "duration": 0, "source": "kamoulox#4 - Actions#26"}, {"id": "972", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9865, "type": "ERROR", "duration": 0, "source": "kamoulox#1 - Actions#27"}, {"id": "971", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9848, "type": "ERROR", "duration": 0, "source": "kamoulox#6 - Actions#27"}, {"id": "970", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9845, "type": "ERROR", "duration": 0, "source": "kamoulox#9 - Actions#27"}, {"id": "969", "code": "429", "elementid": "b29523e3-6f58-40fc-98a4-8e24823d9390", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9839, "type": "ERROR", "duration": 0, "source": "kamoulox#8 - Actions#23"}, {"id": "968", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9831, "type": "ERROR", "duration": 0, "source": "kamoulox#3 - Actions#27"}, {"id": "967", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9817, "type": "ERROR", "duration": 0, "source": "kamoulox#0 - Actions#27"}, {"id": "966", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9810, "type": "ERROR", "duration": 0, "source": "kamoulox#2 - Actions#27"}, {"id": "965", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9804, "type": "ERROR", "duration": 0, "source": "kamoulox#7 - Actions#27"}, {"id": "963", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9803, "type": "ERROR", "duration": 0, "source": "kamoulox#4 - Actions#26"}, {"id": "964", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9803, "type": "ERROR", "duration": 0, "source": "kamoulox#5 - Actions#27"}, {"id": "962", "code": "429", "elementid": "1034d624-451c-4523-b579-6f003ebd93f8", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9790, "type": "ERROR", "duration": 0, "source": "kamoulox#1 - Actions#27"}, {"id": "961", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9780, "type": "ERROR", "duration": 0, "source": "kamoulox#9 - Actions#27"}, {"id": "960", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9761, "type": "ERROR", "duration": 0, "source": "kamoulox#6 - Actions#27"}, {"id": "959", "code": "429", "elementid": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9760, "type": "ERROR", "duration": 0, "source": "kamoulox#8 - Actions#23"}, {"id": "958", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9753, "type": "ERROR", "duration": 0, "source": "kamoulox#3 - Actions#27"}, {"id": "957", "code": "429", "elementid": "1a080b9f-05dc-4424-b575-866dedb2acb0", "fullname": "GET http://testing.intranet.neotys.com/api/webhooks/last", "offset": 9739, "type": "ERROR", "duration": 0, "source": "kamoulox#0 - Actions#27"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements?category=TRANSACTION':
            return '[{"id": "all-transactions", "name": "<all transactions>", "type": "ALL_TRANSACTIONS"}, {"id": "e66cb9ec-3c72-49b4-b3a2-5fa91e2fd911", "name": "transaction_no_sla", "path": ["kamoulox", "Actions", "transaction_no_sla"], "type": "TRANSACTION"}, {"id": "7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a", "name": "transaction_sla_warning", "path": ["kamoulox", "Actions", "transaction_sla_warning"], "type": "TRANSACTION"}, {"id": "6a0ef850-8644-4236-80e5-53748fbd2d01", "name": "transaction_sla_failed", "path": ["kamoulox", "Actions", "transaction_sla_failed"], "type": "TRANSACTION"}, {"id": "818d2419-daec-43d6-b004-b1195799df57", "name": "transaction_sla_passed", "path": ["kamoulox", "Actions", "transaction_sla_passed"], "type": "TRANSACTION"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/e66cb9ec-3c72-49b4-b3a2-5fa91e2fd911/values':
            return '{"count": 271, "elementPerSecond": 26.683733, "minDuration": 88, "maxDuration": 413, "sumDuration": 32531, "avgDuration": 120.04059, "minTTFB": 55, "maxTTFB": 297, "sumTTFB": 20653, "avgTTFB": 76.210335, "sumDownloadedBytes": 768408, "downloadedBytesPerSecond": 75660.5, "successCount": 20, "successPerSecond": 1.9692793, "successRate": 7.380074, "failureCount": 251, "failurePerSecond": 24.714455, "failureRate": 92.61993, "percentile50": 103.0, "percentile90": 130.0, "percentile95": 318.0, "percentile99": 375.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a/values':
            return '{"count": 266, "elementPerSecond": 26.191414, "minDuration": 61, "maxDuration": 215, "sumDuration": 20155, "avgDuration": 75.770676, "minTTFB": 29, "maxTTFB": 152, "sumTTFB": 10060, "avgTTFB": 37.81955, "sumDownloadedBytes": 753705, "downloadedBytesPerSecond": 74212.78, "successCount": 20, "successPerSecond": 1.9692793, "successRate": 7.518797, "failureCount": 246, "failurePerSecond": 24.222136, "failureRate": 92.4812, "percentile50": 72.0, "percentile90": 88.0, "percentile95": 96.0, "percentile99": 146.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/6a0ef850-8644-4236-80e5-53748fbd2d01/values':
            return '{"count": 266, "elementPerSecond": 26.191414, "minDuration": 59, "maxDuration": 1236, "sumDuration": 21174, "avgDuration": 79.6015, "minTTFB": 29, "maxTTFB": 1198, "sumTTFB": 11145, "avgTTFB": 41.898495, "sumDownloadedBytes": 754949, "downloadedBytesPerSecond": 74335.27, "successCount": 19, "successPerSecond": 1.8708153, "successRate": 7.142857, "failureCount": 247, "failurePerSecond": 24.320599, "failureRate": 92.85714, "percentile50": 72.0, "percentile90": 85.0, "percentile95": 89.0, "percentile99": 106.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/818d2419-daec-43d6-b004-b1195799df57/values':
            return '{"count": 265, "elementPerSecond": 26.09295, "minDuration": 60, "maxDuration": 514, "sumDuration": 20825, "avgDuration": 78.58491, "minTTFB": 29, "maxTTFB": 475, "sumTTFB": 10099, "avgTTFB": 38.109432, "sumDownloadedBytes": 764696, "downloadedBytesPerSecond": 75295.0, "successCount": 10, "successPerSecond": 0.98463964, "successRate": 3.7735848, "failureCount": 255, "failurePerSecond": 25.10831, "failureRate": 96.22642, "percentile50": 73.0, "percentile90": 87.0, "percentile95": 97.0, "percentile99": 283.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/818d2419-daec-43d6-b004-b1195799df57/raw?format=JSON':
            return '[{"Elapsed": 1025, "Time": "2021-02-12T15:47:55.140Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 119, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1026, "Time": "2021-02-12T15:47:55.141Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 119, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1027, "Time": "2021-02-12T15:47:55.142Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 119, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1027, "Time": "2021-02-12T15:47:55.142Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 125, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1028, "Time": "2021-02-12T15:47:55.143Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 126, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1065, "Time": "2021-02-12T15:47:55.180Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1065, "Time": "2021-02-12T15:47:55.180Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1078, "Time": "2021-02-12T15:47:55.193Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 158, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1083, "Time": "2021-02-12T15:47:55.198Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 141, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1083, "Time": "2021-02-12T15:47:55.198Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1451, "Time": "2021-02-12T15:47:55.566Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1451, "Time": "2021-02-12T15:47:55.566Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1459, "Time": "2021-02-12T15:47:55.574Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1459, "Time": "2021-02-12T15:47:55.574Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1460, "Time": "2021-02-12T15:47:55.575Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1469, "Time": "2021-02-12T15:47:55.584Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1469, "Time": "2021-02-12T15:47:55.584Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1470, "Time": "2021-02-12T15:47:55.585Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 86, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1495, "Time": "2021-02-12T15:47:55.610Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1536, "Time": "2021-02-12T15:47:55.651Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1806, "Time": "2021-02-12T15:47:55.921Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1809, "Time": "2021-02-12T15:47:55.924Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1816, "Time": "2021-02-12T15:47:55.931Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1817, "Time": "2021-02-12T15:47:55.932Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1818, "Time": "2021-02-12T15:47:55.933Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1828, "Time": "2021-02-12T15:47:55.943Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1828, "Time": "2021-02-12T15:47:55.943Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1858, "Time": "2021-02-12T15:47:55.973Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1888, "Time": "2021-02-12T15:47:56.003Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2110, "Time": "2021-02-12T15:47:56.225Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2119, "Time": "2021-02-12T15:47:56.234Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2123, "Time": "2021-02-12T15:47:56.238Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2135, "Time": "2021-02-12T15:47:56.250Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2135, "Time": "2021-02-12T15:47:56.250Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2144, "Time": "2021-02-12T15:47:56.259Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2145, "Time": "2021-02-12T15:47:56.260Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2193, "Time": "2021-02-12T15:47:56.308Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2194, "Time": "2021-02-12T15:47:56.309Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2425, "Time": "2021-02-12T15:47:56.540Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2426, "Time": "2021-02-12T15:47:56.541Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2438, "Time": "2021-02-12T15:47:56.553Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2448, "Time": "2021-02-12T15:47:56.563Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2451, "Time": "2021-02-12T15:47:56.566Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2454, "Time": "2021-02-12T15:47:56.569Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2503, "Time": "2021-02-12T15:47:56.618Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2515, "Time": "2021-02-12T15:47:56.630Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2738, "Time": "2021-02-12T15:47:56.853Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2746, "Time": "2021-02-12T15:47:56.861Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2760, "Time": "2021-02-12T15:47:56.875Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2765, "Time": "2021-02-12T15:47:56.880Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2778, "Time": "2021-02-12T15:47:56.893Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2784, "Time": "2021-02-12T15:47:56.899Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2837, "Time": "2021-02-12T15:47:56.952Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2839, "Time": "2021-02-12T15:47:56.954Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2899, "Time": "2021-02-12T15:47:57.014Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 514, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2962, "Time": "2021-02-12T15:47:57.077Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3059, "Time": "2021-02-12T15:47:57.174Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3071, "Time": "2021-02-12T15:47:57.186Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3073, "Time": "2021-02-12T15:47:57.188Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3088, "Time": "2021-02-12T15:47:57.203Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3092, "Time": "2021-02-12T15:47:57.207Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3111, "Time": "2021-02-12T15:47:57.226Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3149, "Time": "2021-02-12T15:47:57.264Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3157, "Time": "2021-02-12T15:47:57.272Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3186, "Time": "2021-02-12T15:47:57.301Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3276, "Time": "2021-02-12T15:47:57.391Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3381, "Time": "2021-02-12T15:47:57.496Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 83, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3386, "Time": "2021-02-12T15:47:57.501Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 89, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3401, "Time": "2021-02-12T15:47:57.516Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3421, "Time": "2021-02-12T15:47:57.536Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3435, "Time": "2021-02-12T15:47:57.550Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 89, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3452, "Time": "2021-02-12T15:47:57.567Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 86, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3469, "Time": "2021-02-12T15:47:57.584Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3507, "Time": "2021-02-12T15:47:57.622Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3520, "Time": "2021-02-12T15:47:57.635Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3608, "Time": "2021-02-12T15:47:57.723Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3732, "Time": "2021-02-12T15:47:57.847Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3740, "Time": "2021-02-12T15:47:57.855Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3743, "Time": "2021-02-12T15:47:57.858Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3768, "Time": "2021-02-12T15:47:57.883Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3808, "Time": "2021-02-12T15:47:57.923Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 98, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3816, "Time": "2021-02-12T15:47:57.931Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 92, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3828, "Time": "2021-02-12T15:47:57.943Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4068, "Time": "2021-02-12T15:47:58.183Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 283, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4084, "Time": "2021-02-12T15:47:58.199Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 310, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4265, "Time": "2021-02-12T15:47:58.380Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4351, "Time": "2021-02-12T15:47:58.466Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4358, "Time": "2021-02-12T15:47:58.473Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4374, "Time": "2021-02-12T15:47:58.489Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4411, "Time": "2021-02-12T15:47:58.526Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 88, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4421, "Time": "2021-02-12T15:47:58.536Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4429, "Time": "2021-02-12T15:47:58.544Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4430, "Time": "2021-02-12T15:47:58.545Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4447, "Time": "2021-02-12T15:47:58.562Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 89, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4458, "Time": "2021-02-12T15:47:58.573Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4649, "Time": "2021-02-12T15:47:58.764Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4753, "Time": "2021-02-12T15:47:58.868Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 88, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4763, "Time": "2021-02-12T15:47:58.878Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 83, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4786, "Time": "2021-02-12T15:47:58.901Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4799, "Time": "2021-02-12T15:47:58.914Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4799, "Time": "2021-02-12T15:47:58.914Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4817, "Time": "2021-02-12T15:47:58.932Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4819, "Time": "2021-02-12T15:47:58.934Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4831, "Time": "2021-02-12T15:47:58.946Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4836, "Time": "2021-02-12T15:47:58.951Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4977, "Time": "2021-02-12T15:47:59.092Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5056, "Time": "2021-02-12T15:47:59.171Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 60, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5092, "Time": "2021-02-12T15:47:59.207Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5095, "Time": "2021-02-12T15:47:59.210Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5113, "Time": "2021-02-12T15:47:59.228Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5124, "Time": "2021-02-12T15:47:59.239Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5124, "Time": "2021-02-12T15:47:59.239Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5130, "Time": "2021-02-12T15:47:59.245Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5132, "Time": "2021-02-12T15:47:59.247Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5135, "Time": "2021-02-12T15:47:59.250Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5304, "Time": "2021-02-12T15:47:59.419Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5387, "Time": "2021-02-12T15:47:59.502Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5394, "Time": "2021-02-12T15:47:59.509Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5414, "Time": "2021-02-12T15:47:59.529Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5422, "Time": "2021-02-12T15:47:59.537Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5424, "Time": "2021-02-12T15:47:59.539Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5438, "Time": "2021-02-12T15:47:59.553Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5443, "Time": "2021-02-12T15:47:59.558Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 62, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5444, "Time": "2021-02-12T15:47:59.559Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5451, "Time": "2021-02-12T15:47:59.566Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5666, "Time": "2021-02-12T15:47:59.781Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5781, "Time": "2021-02-12T15:47:59.896Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5781, "Time": "2021-02-12T15:47:59.896Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 94, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5784, "Time": "2021-02-12T15:47:59.899Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 83, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5786, "Time": "2021-02-12T15:47:59.901Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5791, "Time": "2021-02-12T15:47:59.906Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5799, "Time": "2021-02-12T15:47:59.914Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5818, "Time": "2021-02-12T15:47:59.933Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 83, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5840, "Time": "2021-02-12T15:47:59.955Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5862, "Time": "2021-02-12T15:47:59.977Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6004, "Time": "2021-02-12T15:48:00.119Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6094, "Time": "2021-02-12T15:48:00.209Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6095, "Time": "2021-02-12T15:48:00.210Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6115, "Time": "2021-02-12T15:48:00.230Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6117, "Time": "2021-02-12T15:48:00.232Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6117, "Time": "2021-02-12T15:48:00.232Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6131, "Time": "2021-02-12T15:48:00.246Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6168, "Time": "2021-02-12T15:48:00.283Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6168, "Time": "2021-02-12T15:48:00.283Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6169, "Time": "2021-02-12T15:48:00.284Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6333, "Time": "2021-02-12T15:48:00.448Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6424, "Time": "2021-02-12T15:48:00.539Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 61, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6430, "Time": "2021-02-12T15:48:00.545Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6440, "Time": "2021-02-12T15:48:00.555Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6441, "Time": "2021-02-12T15:48:00.556Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6448, "Time": "2021-02-12T15:48:00.563Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6459, "Time": "2021-02-12T15:48:00.574Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6477, "Time": "2021-02-12T15:48:00.592Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6491, "Time": "2021-02-12T15:48:00.606Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 60, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6492, "Time": "2021-02-12T15:48:00.607Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6636, "Time": "2021-02-12T15:48:00.751Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6726, "Time": "2021-02-12T15:48:00.841Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6732, "Time": "2021-02-12T15:48:00.847Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6745, "Time": "2021-02-12T15:48:00.860Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6750, "Time": "2021-02-12T15:48:00.865Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6751, "Time": "2021-02-12T15:48:00.866Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 60, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6759, "Time": "2021-02-12T15:48:00.874Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6774, "Time": "2021-02-12T15:48:00.889Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6783, "Time": "2021-02-12T15:48:00.898Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6786, "Time": "2021-02-12T15:48:00.901Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6946, "Time": "2021-02-12T15:48:01.061Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7040, "Time": "2021-02-12T15:48:01.155Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7049, "Time": "2021-02-12T15:48:01.164Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7060, "Time": "2021-02-12T15:48:01.175Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7063, "Time": "2021-02-12T15:48:01.178Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7065, "Time": "2021-02-12T15:48:01.180Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7074, "Time": "2021-02-12T15:48:01.189Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7085, "Time": "2021-02-12T15:48:01.200Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7093, "Time": "2021-02-12T15:48:01.208Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7111, "Time": "2021-02-12T15:48:01.226Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7237, "Time": "2021-02-12T15:48:01.352Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 60, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7346, "Time": "2021-02-12T15:48:01.461Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7358, "Time": "2021-02-12T15:48:01.473Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7375, "Time": "2021-02-12T15:48:01.490Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7389, "Time": "2021-02-12T15:48:01.504Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7389, "Time": "2021-02-12T15:48:01.504Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7389, "Time": "2021-02-12T15:48:01.504Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7398, "Time": "2021-02-12T15:48:01.513Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7399, "Time": "2021-02-12T15:48:01.514Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7412, "Time": "2021-02-12T15:48:01.527Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7549, "Time": "2021-02-12T15:48:01.664Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7659, "Time": "2021-02-12T15:48:01.774Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7679, "Time": "2021-02-12T15:48:01.794Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7686, "Time": "2021-02-12T15:48:01.801Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7702, "Time": "2021-02-12T15:48:01.817Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7704, "Time": "2021-02-12T15:48:01.819Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7708, "Time": "2021-02-12T15:48:01.823Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7709, "Time": "2021-02-12T15:48:01.824Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7711, "Time": "2021-02-12T15:48:01.826Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7732, "Time": "2021-02-12T15:48:01.847Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7880, "Time": "2021-02-12T15:48:01.995Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7993, "Time": "2021-02-12T15:48:02.108Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8008, "Time": "2021-02-12T15:48:02.123Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8010, "Time": "2021-02-12T15:48:02.125Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8020, "Time": "2021-02-12T15:48:02.135Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8030, "Time": "2021-02-12T15:48:02.145Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8043, "Time": "2021-02-12T15:48:02.158Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8045, "Time": "2021-02-12T15:48:02.160Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8049, "Time": "2021-02-12T15:48:02.164Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 88, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8058, "Time": "2021-02-12T15:48:02.173Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8218, "Time": "2021-02-12T15:48:02.333Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8328, "Time": "2021-02-12T15:48:02.443Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8335, "Time": "2021-02-12T15:48:02.450Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8359, "Time": "2021-02-12T15:48:02.474Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8359, "Time": "2021-02-12T15:48:02.474Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 89, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8360, "Time": "2021-02-12T15:48:02.475Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8366, "Time": "2021-02-12T15:48:02.481Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8371, "Time": "2021-02-12T15:48:02.486Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8379, "Time": "2021-02-12T15:48:02.494Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8399, "Time": "2021-02-12T15:48:02.514Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8557, "Time": "2021-02-12T15:48:02.672Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8645, "Time": "2021-02-12T15:48:02.760Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8667, "Time": "2021-02-12T15:48:02.782Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8668, "Time": "2021-02-12T15:48:02.783Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8673, "Time": "2021-02-12T15:48:02.788Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8678, "Time": "2021-02-12T15:48:02.793Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8678, "Time": "2021-02-12T15:48:02.793Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8690, "Time": "2021-02-12T15:48:02.805Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8694, "Time": "2021-02-12T15:48:02.809Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8729, "Time": "2021-02-12T15:48:02.844Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8888, "Time": "2021-02-12T15:48:03.003Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8985, "Time": "2021-02-12T15:48:03.100Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 86, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9001, "Time": "2021-02-12T15:48:03.116Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9005, "Time": "2021-02-12T15:48:03.120Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9005, "Time": "2021-02-12T15:48:03.120Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9012, "Time": "2021-02-12T15:48:03.127Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9017, "Time": "2021-02-12T15:48:03.132Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 62, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9018, "Time": "2021-02-12T15:48:03.133Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9025, "Time": "2021-02-12T15:48:03.140Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 62, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9050, "Time": "2021-02-12T15:48:03.165Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9201, "Time": "2021-02-12T15:48:03.316Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9296, "Time": "2021-02-12T15:48:03.411Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9311, "Time": "2021-02-12T15:48:03.426Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9311, "Time": "2021-02-12T15:48:03.426Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9318, "Time": "2021-02-12T15:48:03.433Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9318, "Time": "2021-02-12T15:48:03.433Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9324, "Time": "2021-02-12T15:48:03.439Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9344, "Time": "2021-02-12T15:48:03.459Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9345, "Time": "2021-02-12T15:48:03.460Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 88, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9364, "Time": "2021-02-12T15:48:03.479Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9514, "Time": "2021-02-12T15:48:03.629Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9619, "Time": "2021-02-12T15:48:03.734Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9629, "Time": "2021-02-12T15:48:03.744Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9630, "Time": "2021-02-12T15:48:03.745Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9630, "Time": "2021-02-12T15:48:03.745Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9640, "Time": "2021-02-12T15:48:03.755Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9669, "Time": "2021-02-12T15:48:03.784Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9677, "Time": "2021-02-12T15:48:03.792Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9700, "Time": "2021-02-12T15:48:03.815Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9706, "Time": "2021-02-12T15:48:03.821Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9841, "Time": "2021-02-12T15:48:03.956Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9932, "Time": "2021-02-12T15:48:04.047Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9942, "Time": "2021-02-12T15:48:04.057Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9945, "Time": "2021-02-12T15:48:04.060Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9964, "Time": "2021-02-12T15:48:04.079Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 62, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9965, "Time": "2021-02-12T15:48:04.080Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9986, "Time": "2021-02-12T15:48:04.101Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9986, "Time": "2021-02-12T15:48:04.101Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 10034, "Time": "2021-02-12T15:48:04.149Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 10034, "Time": "2021-02-12T15:48:04.149Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements?category=REQUEST':
            return '[{"id": "all-requests", "name": "<all requests>", "type": "ALL_REQUESTS"}, {"id": "54d47517-89bb-4e58-b22c-4fb9ac78000b", "name": "/", "path": ["kamoulox", "Actions", "transaction_no_sla", "page", "/"], "type": "REQUEST"}, {"id": "1a080b9f-05dc-4424-b575-866dedb2acb0", "name": "/api/webhooks/last", "path": ["kamoulox", "Actions", "transaction_no_sla", "/api/webhooks/last"], "type": "REQUEST"}, {"id": "d319f4e8-d31e-4a1b-9187-b8238bc57007", "name": "/", "path": ["kamoulox", "Actions", "transaction_sla_warning", "page", "/"], "type": "REQUEST"}, {"id": "1034d624-451c-4523-b579-6f003ebd93f8", "name": "/api/webhooks/last", "path": ["kamoulox", "Actions", "transaction_sla_warning", "/api/webhooks/last"], "type": "REQUEST"}, {"id": "7bf42114-d630-494f-bc13-fea402ce085d", "name": "/", "path": ["kamoulox", "Actions", "transaction_sla_failed", "page", "/"], "type": "REQUEST"}, {"id": "8f34d21e-89e0-4929-89b7-01198f3ad27d", "name": "/api/webhooks/last", "path": ["kamoulox", "Actions", "transaction_sla_failed", "/api/webhooks/last"], "type": "REQUEST"}, {"id": "1df02e91-0b8c-4032-9d14-0853e9724294", "name": "/", "path": ["kamoulox", "Actions", "transaction_sla_passed", "page", "/"], "type": "REQUEST"}, {"id": "b29523e3-6f58-40fc-98a4-8e24823d9390", "name": "/api/webhooks/last", "path": ["kamoulox", "Actions", "transaction_sla_passed", "/api/webhooks/last"], "type": "REQUEST"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/all-requests/values':
            return '{"count": 2153, "elementPerSecond": 211.9929, "minDuration": 29, "maxDuration": 1198, "sumDuration": 95443, "avgDuration": 44.33024, "minTTFB": 28, "maxTTFB": 1198, "sumTTFB": 92774, "avgTTFB": 43.090572, "sumDownloadedBytes": 3065803, "downloadedBytesPerSecond": 301871.12, "successCount": 1147, "successPerSecond": 112.93816, "successRate": 53.2745, "failureCount": 1006, "failurePerSecond": 99.05475, "failureRate": 46.7255}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/all-requests/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 1000, "AVG_TTFB": 77.75, "ERRORS": 0, "AVG_DURATION": 79.20588, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 32, "ERROR_RATE": 0.0, "MIN_TTFB": 30, "MAX_TTFB": 237, "MAX_DURATION": 241, "ELEMENTS_PER_SECOND": 68.0, "THROUGHPUT": 55504.0, "COUNT": 68}, {"from": 1000, "to": 2000, "AVG_TTFB": 44.668343, "ERRORS": 64, "AVG_DURATION": 46.758793, "ERRORS_PER_SECOND": 64.0, "MIN_DURATION": 30, "ERROR_RATE": 32.160805, "MIN_TTFB": 29, "MAX_TTFB": 98, "MAX_DURATION": 111, "ELEMENTS_PER_SECOND": 199.0, "THROUGHPUT": 239953.0, "COUNT": 199}, {"from": 2000, "to": 3000, "AVG_TTFB": 45.27397, "ERRORS": 107, "AVG_DURATION": 46.351597, "ERRORS_PER_SECOND": 107.0, "MIN_DURATION": 30, "ERROR_RATE": 48.858448, "MIN_TTFB": 28, "MAX_TTFB": 1198, "MAX_DURATION": 1198, "ELEMENTS_PER_SECOND": 219.0, "THROUGHPUT": 320300.0, "COUNT": 219}, {"from": 3000, "to": 4000, "AVG_TTFB": 41.44, "ERRORS": 101, "AVG_DURATION": 42.62, "ERRORS_PER_SECOND": 101.0, "MIN_DURATION": 29, "ERROR_RATE": 50.5, "MIN_TTFB": 28, "MAX_TTFB": 154, "MAX_DURATION": 156, "ELEMENTS_PER_SECOND": 200.0, "THROUGHPUT": 294480.0, "COUNT": 200}, {"from": 4000, "to": 5000, "AVG_TTFB": 54.89109, "ERRORS": 101, "AVG_DURATION": 56.282177, "ERRORS_PER_SECOND": 101.0, "MIN_DURATION": 30, "ERROR_RATE": 50.0, "MIN_TTFB": 29, "MAX_TTFB": 301, "MAX_DURATION": 301, "ELEMENTS_PER_SECOND": 202.0, "THROUGHPUT": 296830.0, "COUNT": 202}, {"from": 5000, "to": 6000, "AVG_TTFB": 40.565956, "ERRORS": 116, "AVG_DURATION": 41.582977, "ERRORS_PER_SECOND": 116.0, "MIN_DURATION": 29, "ERROR_RATE": 49.361702, "MIN_TTFB": 29, "MAX_TTFB": 75, "MAX_DURATION": 75, "ELEMENTS_PER_SECOND": 235.0, "THROUGHPUT": 344231.0, "COUNT": 235}, {"from": 6000, "to": 7000, "AVG_TTFB": 38.05098, "ERRORS": 129, "AVG_DURATION": 38.964706, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 29, "ERROR_RATE": 50.588234, "MIN_TTFB": 28, "MAX_TTFB": 89, "MAX_DURATION": 90, "ELEMENTS_PER_SECOND": 255.0, "THROUGHPUT": 375564.0, "COUNT": 255}, {"from": 7000, "to": 8000, "AVG_TTFB": 37.98016, "ERRORS": 126, "AVG_DURATION": 39.01984, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 29, "ERROR_RATE": 50.0, "MIN_TTFB": 29, "MAX_TTFB": 85, "MAX_DURATION": 85, "ELEMENTS_PER_SECOND": 252.0, "THROUGHPUT": 370277.0, "COUNT": 252}, {"from": 8000, "to": 9000, "AVG_TTFB": 39.48971, "ERRORS": 120, "AVG_DURATION": 40.979424, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 30, "ERROR_RATE": 49.382717, "MIN_TTFB": 29, "MAX_TTFB": 84, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 243.0, "THROUGHPUT": 356161.0, "COUNT": 243}, {"from": 9000, "to": 10000, "AVG_TTFB": 38.212852, "ERRORS": 126, "AVG_DURATION": 39.33735, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 30, "ERROR_RATE": 50.60241, "MIN_TTFB": 28, "MAX_TTFB": 76, "MAX_DURATION": 76, "ELEMENTS_PER_SECOND": 249.0, "THROUGHPUT": 366686.0, "COUNT": 249}, {"from": 10000, "to": 11000, "AVG_TTFB": 44.80645, "ERRORS": 16, "AVG_DURATION": 45.612904, "ERRORS_PER_SECOND": 16.0, "MIN_DURATION": 33, "ERROR_RATE": 51.612904, "MIN_TTFB": 32, "MAX_TTFB": 79, "MAX_DURATION": 79, "ELEMENTS_PER_SECOND": 31.0, "THROUGHPUT": 45817.0, "COUNT": 31}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements?category=TRANSACTION':
            return '[{"id": "all-transactions", "name": "<all transactions>", "type": "ALL_TRANSACTIONS"}, {"id": "e66cb9ec-3c72-49b4-b3a2-5fa91e2fd911", "name": "transaction_no_sla", "path": ["kamoulox", "Actions", "transaction_no_sla"], "type": "TRANSACTION"}, {"id": "7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a", "name": "transaction_sla_warning", "path": ["kamoulox", "Actions", "transaction_sla_warning"], "type": "TRANSACTION"}, {"id": "6a0ef850-8644-4236-80e5-53748fbd2d01", "name": "transaction_sla_failed", "path": ["kamoulox", "Actions", "transaction_sla_failed"], "type": "TRANSACTION"}, {"id": "818d2419-daec-43d6-b004-b1195799df57", "name": "transaction_sla_passed", "path": ["kamoulox", "Actions", "transaction_sla_passed"], "type": "TRANSACTION"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/all-transactions/values':
            return '{"count": 1068, "elementPerSecond": 105.159515, "minDuration": 59, "maxDuration": 1236, "sumDuration": 94685, "avgDuration": 88.656364, "minTTFB": 29, "maxTTFB": 1198, "sumTTFB": 51957, "avgTTFB": 48.648876, "sumDownloadedBytes": 3041758, "downloadedBytesPerSecond": 299503.53, "successCount": 69, "successPerSecond": 6.7940135, "successRate": 6.4606743, "failureCount": 999, "failurePerSecond": 98.3655, "failureRate": 93.53933, "percentile50": 72.0, "percentile90": 88.0, "percentile95": 96.0, "percentile99": 146.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/all-transactions/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 1000, "AVG_TTFB": 94.7, "ERRORS": 0, "AVG_DURATION": 169.66667, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 68, "ERROR_RATE": 0.0, "MIN_TTFB": 30, "MAX_TTFB": 237, "MAX_DURATION": 387, "ELEMENTS_PER_SECOND": 30.0, "THROUGHPUT": 46160.0, "COUNT": 30}, {"from": 1000, "to": 2000, "AVG_TTFB": 48.417477, "ERRORS": 64, "AVG_DURATION": 92.84466, "ERRORS_PER_SECOND": 64.0, "MIN_DURATION": 62, "ERROR_RATE": 62.13592, "MIN_TTFB": 30, "MAX_TTFB": 98, "MAX_DURATION": 158, "ELEMENTS_PER_SECOND": 103.0, "THROUGHPUT": 248123.0, "COUNT": 103}, {"from": 2000, "to": 3000, "AVG_TTFB": 57.46729, "ERRORS": 107, "AVG_DURATION": 93.19626, "ERRORS_PER_SECOND": 107.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 1198, "MAX_DURATION": 1236, "ELEMENTS_PER_SECOND": 107.0, "THROUGHPUT": 314458.0, "COUNT": 107}, {"from": 3000, "to": 4000, "AVG_TTFB": 43.653465, "ERRORS": 101, "AVG_DURATION": 83.21782, "ERRORS_PER_SECOND": 101.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 94, "MAX_DURATION": 250, "ELEMENTS_PER_SECOND": 101.0, "THROUGHPUT": 296826.0, "COUNT": 101}, {"from": 4000, "to": 5000, "AVG_TTFB": 64.04951, "ERRORS": 101, "AVG_DURATION": 114.544556, "ERRORS_PER_SECOND": 101.0, "MIN_DURATION": 63, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 301, "MAX_DURATION": 413, "ELEMENTS_PER_SECOND": 101.0, "THROUGHPUT": 296818.0, "COUNT": 101}, {"from": 5000, "to": 6000, "AVG_TTFB": 44.88793, "ERRORS": 116, "AVG_DURATION": 83.25, "ERRORS_PER_SECOND": 116.0, "MIN_DURATION": 59, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 75, "MAX_DURATION": 120, "ELEMENTS_PER_SECOND": 116.0, "THROUGHPUT": 340741.0, "COUNT": 116}, {"from": 6000, "to": 7000, "AVG_TTFB": 42.279068, "ERRORS": 129, "AVG_DURATION": 77.76744, "ERRORS_PER_SECOND": 129.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 89, "MAX_DURATION": 125, "ELEMENTS_PER_SECOND": 129.0, "THROUGHPUT": 379072.0, "COUNT": 129}, {"from": 7000, "to": 8000, "AVG_TTFB": 42.365078, "ERRORS": 126, "AVG_DURATION": 78.18254, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 60, "ERROR_RATE": 100.0, "MIN_TTFB": 29, "MAX_TTFB": 85, "MAX_DURATION": 122, "ELEMENTS_PER_SECOND": 126.0, "THROUGHPUT": 370269.0, "COUNT": 126}, {"from": 8000, "to": 9000, "AVG_TTFB": 44.3, "ERRORS": 120, "AVG_DURATION": 82.2, "ERRORS_PER_SECOND": 120.0, "MIN_DURATION": 64, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 84, "MAX_DURATION": 117, "ELEMENTS_PER_SECOND": 120.0, "THROUGHPUT": 352659.0, "COUNT": 120}, {"from": 9000, "to": 10000, "AVG_TTFB": 42.174603, "ERRORS": 126, "AVG_DURATION": 78.22222, "ERRORS_PER_SECOND": 126.0, "MIN_DURATION": 62, "ERROR_RATE": 100.0, "MIN_TTFB": 30, "MAX_TTFB": 76, "MAX_DURATION": 112, "ELEMENTS_PER_SECOND": 126.0, "THROUGHPUT": 370176.0, "COUNT": 126}, {"from": 10000, "to": 11000, "AVG_TTFB": 52.555557, "ERRORS": 9, "AVG_DURATION": 91.77778, "ERRORS_PER_SECOND": 9.0, "MIN_DURATION": 69, "ERROR_RATE": 100.0, "MIN_TTFB": 34, "MAX_TTFB": 71, "MAX_DURATION": 110, "ELEMENTS_PER_SECOND": 9.0, "THROUGHPUT": 26456.0, "COUNT": 9}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/818d2419-daec-43d6-b004-b1195799df57/values':
            return '{"count": 265, "elementPerSecond": 26.09295, "minDuration": 60, "maxDuration": 514, "sumDuration": 20825, "avgDuration": 78.58491, "minTTFB": 29, "maxTTFB": 475, "sumTTFB": 10099, "avgTTFB": 38.109432, "sumDownloadedBytes": 764696, "downloadedBytesPerSecond": 75295.0, "successCount": 10, "successPerSecond": 0.98463964, "successRate": 3.7735848, "failureCount": 255, "failurePerSecond": 25.10831, "failureRate": 96.22642, "percentile50": 73.0, "percentile90": 87.0, "percentile95": 97.0, "percentile99": 283.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/e66cb9ec-3c72-49b4-b3a2-5fa91e2fd911/values':
            return '{"count": 271, "elementPerSecond": 26.683733, "minDuration": 88, "maxDuration": 413, "sumDuration": 32531, "avgDuration": 120.04059, "minTTFB": 55, "maxTTFB": 297, "sumTTFB": 20653, "avgTTFB": 76.210335, "sumDownloadedBytes": 768408, "downloadedBytesPerSecond": 75660.5, "successCount": 20, "successPerSecond": 1.9692793, "successRate": 7.380074, "failureCount": 251, "failurePerSecond": 24.714455, "failureRate": 92.61993, "percentile50": 103.0, "percentile90": 130.0, "percentile95": 318.0, "percentile99": 375.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a/values':
            return '{"count": 266, "elementPerSecond": 26.191414, "minDuration": 61, "maxDuration": 215, "sumDuration": 20155, "avgDuration": 75.770676, "minTTFB": 29, "maxTTFB": 152, "sumTTFB": 10060, "avgTTFB": 37.81955, "sumDownloadedBytes": 753705, "downloadedBytesPerSecond": 74212.78, "successCount": 20, "successPerSecond": 1.9692793, "successRate": 7.518797, "failureCount": 246, "failurePerSecond": 24.222136, "failureRate": 92.4812, "percentile50": 72.0, "percentile90": 88.0, "percentile95": 96.0, "percentile99": 146.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/6a0ef850-8644-4236-80e5-53748fbd2d01/values':
            return '{"count": 266, "elementPerSecond": 26.191414, "minDuration": 59, "maxDuration": 1236, "sumDuration": 21174, "avgDuration": 79.6015, "minTTFB": 29, "maxTTFB": 1198, "sumTTFB": 11145, "avgTTFB": 41.898495, "sumDownloadedBytes": 754949, "downloadedBytesPerSecond": 74335.27, "successCount": 19, "successPerSecond": 1.8708153, "successRate": 7.142857, "failureCount": 247, "failurePerSecond": 24.320599, "failureRate": 92.85714, "percentile50": 72.0, "percentile90": 85.0, "percentile95": 89.0, "percentile99": 106.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/818d2419-daec-43d6-b004-b1195799df57/raw?format=JSON':
            return '[{"Elapsed": 1025, "Time": "2021-02-12T15:47:55.140Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 119, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1026, "Time": "2021-02-12T15:47:55.141Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 119, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1027, "Time": "2021-02-12T15:47:55.142Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 119, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1027, "Time": "2021-02-12T15:47:55.142Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 125, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1028, "Time": "2021-02-12T15:47:55.143Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 126, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1065, "Time": "2021-02-12T15:47:55.180Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1065, "Time": "2021-02-12T15:47:55.180Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1078, "Time": "2021-02-12T15:47:55.193Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 158, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1083, "Time": "2021-02-12T15:47:55.198Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 141, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1083, "Time": "2021-02-12T15:47:55.198Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1451, "Time": "2021-02-12T15:47:55.566Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1451, "Time": "2021-02-12T15:47:55.566Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1459, "Time": "2021-02-12T15:47:55.574Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1459, "Time": "2021-02-12T15:47:55.574Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1460, "Time": "2021-02-12T15:47:55.575Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1469, "Time": "2021-02-12T15:47:55.584Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1469, "Time": "2021-02-12T15:47:55.584Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1470, "Time": "2021-02-12T15:47:55.585Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 86, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1495, "Time": "2021-02-12T15:47:55.610Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1536, "Time": "2021-02-12T15:47:55.651Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1806, "Time": "2021-02-12T15:47:55.921Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1809, "Time": "2021-02-12T15:47:55.924Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1816, "Time": "2021-02-12T15:47:55.931Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1817, "Time": "2021-02-12T15:47:55.932Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1818, "Time": "2021-02-12T15:47:55.933Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1828, "Time": "2021-02-12T15:47:55.943Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1828, "Time": "2021-02-12T15:47:55.943Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1858, "Time": "2021-02-12T15:47:55.973Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1888, "Time": "2021-02-12T15:47:56.003Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2110, "Time": "2021-02-12T15:47:56.225Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2119, "Time": "2021-02-12T15:47:56.234Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2123, "Time": "2021-02-12T15:47:56.238Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2135, "Time": "2021-02-12T15:47:56.250Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2135, "Time": "2021-02-12T15:47:56.250Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2144, "Time": "2021-02-12T15:47:56.259Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2145, "Time": "2021-02-12T15:47:56.260Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2193, "Time": "2021-02-12T15:47:56.308Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2194, "Time": "2021-02-12T15:47:56.309Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2425, "Time": "2021-02-12T15:47:56.540Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2426, "Time": "2021-02-12T15:47:56.541Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2438, "Time": "2021-02-12T15:47:56.553Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2448, "Time": "2021-02-12T15:47:56.563Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2451, "Time": "2021-02-12T15:47:56.566Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2454, "Time": "2021-02-12T15:47:56.569Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2503, "Time": "2021-02-12T15:47:56.618Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2515, "Time": "2021-02-12T15:47:56.630Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2738, "Time": "2021-02-12T15:47:56.853Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2746, "Time": "2021-02-12T15:47:56.861Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2760, "Time": "2021-02-12T15:47:56.875Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2765, "Time": "2021-02-12T15:47:56.880Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2778, "Time": "2021-02-12T15:47:56.893Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2784, "Time": "2021-02-12T15:47:56.899Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2837, "Time": "2021-02-12T15:47:56.952Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2839, "Time": "2021-02-12T15:47:56.954Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2899, "Time": "2021-02-12T15:47:57.014Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 514, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2962, "Time": "2021-02-12T15:47:57.077Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3059, "Time": "2021-02-12T15:47:57.174Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3071, "Time": "2021-02-12T15:47:57.186Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3073, "Time": "2021-02-12T15:47:57.188Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3088, "Time": "2021-02-12T15:47:57.203Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3092, "Time": "2021-02-12T15:47:57.207Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3111, "Time": "2021-02-12T15:47:57.226Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3149, "Time": "2021-02-12T15:47:57.264Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3157, "Time": "2021-02-12T15:47:57.272Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3186, "Time": "2021-02-12T15:47:57.301Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3276, "Time": "2021-02-12T15:47:57.391Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3381, "Time": "2021-02-12T15:47:57.496Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 83, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3386, "Time": "2021-02-12T15:47:57.501Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 89, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3401, "Time": "2021-02-12T15:47:57.516Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3421, "Time": "2021-02-12T15:47:57.536Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3435, "Time": "2021-02-12T15:47:57.550Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 89, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3452, "Time": "2021-02-12T15:47:57.567Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 86, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3469, "Time": "2021-02-12T15:47:57.584Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3507, "Time": "2021-02-12T15:47:57.622Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3520, "Time": "2021-02-12T15:47:57.635Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3608, "Time": "2021-02-12T15:47:57.723Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3732, "Time": "2021-02-12T15:47:57.847Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3740, "Time": "2021-02-12T15:47:57.855Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3743, "Time": "2021-02-12T15:47:57.858Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3768, "Time": "2021-02-12T15:47:57.883Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3808, "Time": "2021-02-12T15:47:57.923Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 98, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3816, "Time": "2021-02-12T15:47:57.931Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 92, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3828, "Time": "2021-02-12T15:47:57.943Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4068, "Time": "2021-02-12T15:47:58.183Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 283, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4084, "Time": "2021-02-12T15:47:58.199Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 310, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4265, "Time": "2021-02-12T15:47:58.380Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4351, "Time": "2021-02-12T15:47:58.466Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4358, "Time": "2021-02-12T15:47:58.473Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4374, "Time": "2021-02-12T15:47:58.489Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4411, "Time": "2021-02-12T15:47:58.526Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 88, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4421, "Time": "2021-02-12T15:47:58.536Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4429, "Time": "2021-02-12T15:47:58.544Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4430, "Time": "2021-02-12T15:47:58.545Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4447, "Time": "2021-02-12T15:47:58.562Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 89, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4458, "Time": "2021-02-12T15:47:58.573Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4649, "Time": "2021-02-12T15:47:58.764Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4753, "Time": "2021-02-12T15:47:58.868Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 88, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4763, "Time": "2021-02-12T15:47:58.878Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 83, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4786, "Time": "2021-02-12T15:47:58.901Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4799, "Time": "2021-02-12T15:47:58.914Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4799, "Time": "2021-02-12T15:47:58.914Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4817, "Time": "2021-02-12T15:47:58.932Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4819, "Time": "2021-02-12T15:47:58.934Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4831, "Time": "2021-02-12T15:47:58.946Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4836, "Time": "2021-02-12T15:47:58.951Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4977, "Time": "2021-02-12T15:47:59.092Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5056, "Time": "2021-02-12T15:47:59.171Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 60, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5092, "Time": "2021-02-12T15:47:59.207Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5095, "Time": "2021-02-12T15:47:59.210Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5113, "Time": "2021-02-12T15:47:59.228Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5124, "Time": "2021-02-12T15:47:59.239Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5124, "Time": "2021-02-12T15:47:59.239Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5130, "Time": "2021-02-12T15:47:59.245Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5132, "Time": "2021-02-12T15:47:59.247Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5135, "Time": "2021-02-12T15:47:59.250Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5304, "Time": "2021-02-12T15:47:59.419Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5387, "Time": "2021-02-12T15:47:59.502Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5394, "Time": "2021-02-12T15:47:59.509Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5414, "Time": "2021-02-12T15:47:59.529Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5422, "Time": "2021-02-12T15:47:59.537Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5424, "Time": "2021-02-12T15:47:59.539Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5438, "Time": "2021-02-12T15:47:59.553Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5443, "Time": "2021-02-12T15:47:59.558Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 62, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5444, "Time": "2021-02-12T15:47:59.559Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5451, "Time": "2021-02-12T15:47:59.566Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5666, "Time": "2021-02-12T15:47:59.781Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5781, "Time": "2021-02-12T15:47:59.896Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5781, "Time": "2021-02-12T15:47:59.896Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 94, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5784, "Time": "2021-02-12T15:47:59.899Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 83, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5786, "Time": "2021-02-12T15:47:59.901Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5791, "Time": "2021-02-12T15:47:59.906Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5799, "Time": "2021-02-12T15:47:59.914Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5818, "Time": "2021-02-12T15:47:59.933Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 83, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5840, "Time": "2021-02-12T15:47:59.955Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5862, "Time": "2021-02-12T15:47:59.977Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6004, "Time": "2021-02-12T15:48:00.119Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6094, "Time": "2021-02-12T15:48:00.209Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6095, "Time": "2021-02-12T15:48:00.210Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6115, "Time": "2021-02-12T15:48:00.230Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6117, "Time": "2021-02-12T15:48:00.232Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6117, "Time": "2021-02-12T15:48:00.232Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6131, "Time": "2021-02-12T15:48:00.246Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6168, "Time": "2021-02-12T15:48:00.283Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6168, "Time": "2021-02-12T15:48:00.283Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6169, "Time": "2021-02-12T15:48:00.284Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6333, "Time": "2021-02-12T15:48:00.448Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6424, "Time": "2021-02-12T15:48:00.539Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 61, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6430, "Time": "2021-02-12T15:48:00.545Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6440, "Time": "2021-02-12T15:48:00.555Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6441, "Time": "2021-02-12T15:48:00.556Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6448, "Time": "2021-02-12T15:48:00.563Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6459, "Time": "2021-02-12T15:48:00.574Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6477, "Time": "2021-02-12T15:48:00.592Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6491, "Time": "2021-02-12T15:48:00.606Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 60, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6492, "Time": "2021-02-12T15:48:00.607Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6636, "Time": "2021-02-12T15:48:00.751Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6726, "Time": "2021-02-12T15:48:00.841Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6732, "Time": "2021-02-12T15:48:00.847Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6745, "Time": "2021-02-12T15:48:00.860Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6750, "Time": "2021-02-12T15:48:00.865Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6751, "Time": "2021-02-12T15:48:00.866Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 60, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6759, "Time": "2021-02-12T15:48:00.874Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6774, "Time": "2021-02-12T15:48:00.889Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6783, "Time": "2021-02-12T15:48:00.898Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6786, "Time": "2021-02-12T15:48:00.901Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6946, "Time": "2021-02-12T15:48:01.061Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7040, "Time": "2021-02-12T15:48:01.155Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7049, "Time": "2021-02-12T15:48:01.164Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7060, "Time": "2021-02-12T15:48:01.175Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7063, "Time": "2021-02-12T15:48:01.178Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7065, "Time": "2021-02-12T15:48:01.180Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7074, "Time": "2021-02-12T15:48:01.189Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7085, "Time": "2021-02-12T15:48:01.200Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7093, "Time": "2021-02-12T15:48:01.208Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7111, "Time": "2021-02-12T15:48:01.226Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7237, "Time": "2021-02-12T15:48:01.352Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 60, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7346, "Time": "2021-02-12T15:48:01.461Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7358, "Time": "2021-02-12T15:48:01.473Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7375, "Time": "2021-02-12T15:48:01.490Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7389, "Time": "2021-02-12T15:48:01.504Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7389, "Time": "2021-02-12T15:48:01.504Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7389, "Time": "2021-02-12T15:48:01.504Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7398, "Time": "2021-02-12T15:48:01.513Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7399, "Time": "2021-02-12T15:48:01.514Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7412, "Time": "2021-02-12T15:48:01.527Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7549, "Time": "2021-02-12T15:48:01.664Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7659, "Time": "2021-02-12T15:48:01.774Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7679, "Time": "2021-02-12T15:48:01.794Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7686, "Time": "2021-02-12T15:48:01.801Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7702, "Time": "2021-02-12T15:48:01.817Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7704, "Time": "2021-02-12T15:48:01.819Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7708, "Time": "2021-02-12T15:48:01.823Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7709, "Time": "2021-02-12T15:48:01.824Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7711, "Time": "2021-02-12T15:48:01.826Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7732, "Time": "2021-02-12T15:48:01.847Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7880, "Time": "2021-02-12T15:48:01.995Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7993, "Time": "2021-02-12T15:48:02.108Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8008, "Time": "2021-02-12T15:48:02.123Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8010, "Time": "2021-02-12T15:48:02.125Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8020, "Time": "2021-02-12T15:48:02.135Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8030, "Time": "2021-02-12T15:48:02.145Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8043, "Time": "2021-02-12T15:48:02.158Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8045, "Time": "2021-02-12T15:48:02.160Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8049, "Time": "2021-02-12T15:48:02.164Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 88, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8058, "Time": "2021-02-12T15:48:02.173Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8218, "Time": "2021-02-12T15:48:02.333Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8328, "Time": "2021-02-12T15:48:02.443Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8335, "Time": "2021-02-12T15:48:02.450Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8359, "Time": "2021-02-12T15:48:02.474Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8359, "Time": "2021-02-12T15:48:02.474Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 89, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8360, "Time": "2021-02-12T15:48:02.475Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8366, "Time": "2021-02-12T15:48:02.481Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8371, "Time": "2021-02-12T15:48:02.486Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8379, "Time": "2021-02-12T15:48:02.494Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8399, "Time": "2021-02-12T15:48:02.514Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8557, "Time": "2021-02-12T15:48:02.672Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8645, "Time": "2021-02-12T15:48:02.760Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8667, "Time": "2021-02-12T15:48:02.782Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8668, "Time": "2021-02-12T15:48:02.783Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8673, "Time": "2021-02-12T15:48:02.788Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8678, "Time": "2021-02-12T15:48:02.793Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8678, "Time": "2021-02-12T15:48:02.793Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8690, "Time": "2021-02-12T15:48:02.805Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8694, "Time": "2021-02-12T15:48:02.809Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8729, "Time": "2021-02-12T15:48:02.844Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8888, "Time": "2021-02-12T15:48:03.003Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8985, "Time": "2021-02-12T15:48:03.100Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 86, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9001, "Time": "2021-02-12T15:48:03.116Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9005, "Time": "2021-02-12T15:48:03.120Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9005, "Time": "2021-02-12T15:48:03.120Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9012, "Time": "2021-02-12T15:48:03.127Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9017, "Time": "2021-02-12T15:48:03.132Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 62, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9018, "Time": "2021-02-12T15:48:03.133Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9025, "Time": "2021-02-12T15:48:03.140Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 62, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9050, "Time": "2021-02-12T15:48:03.165Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9201, "Time": "2021-02-12T15:48:03.316Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9296, "Time": "2021-02-12T15:48:03.411Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9311, "Time": "2021-02-12T15:48:03.426Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9311, "Time": "2021-02-12T15:48:03.426Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9318, "Time": "2021-02-12T15:48:03.433Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9318, "Time": "2021-02-12T15:48:03.433Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9324, "Time": "2021-02-12T15:48:03.439Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9344, "Time": "2021-02-12T15:48:03.459Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9345, "Time": "2021-02-12T15:48:03.460Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 88, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9364, "Time": "2021-02-12T15:48:03.479Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9514, "Time": "2021-02-12T15:48:03.629Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9619, "Time": "2021-02-12T15:48:03.734Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9629, "Time": "2021-02-12T15:48:03.744Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9630, "Time": "2021-02-12T15:48:03.745Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9630, "Time": "2021-02-12T15:48:03.745Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9640, "Time": "2021-02-12T15:48:03.755Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9669, "Time": "2021-02-12T15:48:03.784Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9677, "Time": "2021-02-12T15:48:03.792Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9700, "Time": "2021-02-12T15:48:03.815Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9706, "Time": "2021-02-12T15:48:03.821Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9841, "Time": "2021-02-12T15:48:03.956Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9932, "Time": "2021-02-12T15:48:04.047Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9942, "Time": "2021-02-12T15:48:04.057Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9945, "Time": "2021-02-12T15:48:04.060Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9964, "Time": "2021-02-12T15:48:04.079Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 62, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9965, "Time": "2021-02-12T15:48:04.080Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9986, "Time": "2021-02-12T15:48:04.101Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9986, "Time": "2021-02-12T15:48:04.101Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 10034, "Time": "2021-02-12T15:48:04.149Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 10034, "Time": "2021-02-12T15:48:04.149Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_passed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/e66cb9ec-3c72-49b4-b3a2-5fa91e2fd911/raw?format=JSON':
            return '[{"Elapsed": 741, "Time": "2021-02-12T15:47:54.856Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 371, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 741, "Time": "2021-02-12T15:47:54.856Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 369, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 741, "Time": "2021-02-12T15:47:54.856Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 367, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 741, "Time": "2021-02-12T15:47:54.856Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 357, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 741, "Time": "2021-02-12T15:47:54.856Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 303, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 749, "Time": "2021-02-12T15:47:54.864Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 387, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 749, "Time": "2021-02-12T15:47:54.864Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 334, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 749, "Time": "2021-02-12T15:47:54.864Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 333, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 749, "Time": "2021-02-12T15:47:54.864Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 356, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 756, "Time": "2021-02-12T15:47:54.871Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 358, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1167, "Time": "2021-02-12T15:47:55.282Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 124, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1169, "Time": "2021-02-12T15:47:55.284Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 127, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1175, "Time": "2021-02-12T15:47:55.290Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 132, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1175, "Time": "2021-02-12T15:47:55.290Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 140, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1194, "Time": "2021-02-12T15:47:55.309Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 157, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1209, "Time": "2021-02-12T15:47:55.324Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 133, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1209, "Time": "2021-02-12T15:47:55.324Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 131, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1215, "Time": "2021-02-12T15:47:55.330Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 123, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1215, "Time": "2021-02-12T15:47:55.330Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 120, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1236, "Time": "2021-02-12T15:47:55.351Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 145, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1571, "Time": "2021-02-12T15:47:55.686Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 102, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1571, "Time": "2021-02-12T15:47:55.686Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 111, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1574, "Time": "2021-02-12T15:47:55.689Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 113, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1584, "Time": "2021-02-12T15:47:55.699Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 109, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1585, "Time": "2021-02-12T15:47:55.700Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 120, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1585, "Time": "2021-02-12T15:47:55.700Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 111, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1587, "Time": "2021-02-12T15:47:55.702Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 122, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1593, "Time": "2021-02-12T15:47:55.708Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 120, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1621, "Time": "2021-02-12T15:47:55.736Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 119, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1652, "Time": "2021-02-12T15:47:55.767Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 110, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1904, "Time": "2021-02-12T15:47:56.019Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1906, "Time": "2021-02-12T15:47:56.021Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1913, "Time": "2021-02-12T15:47:56.028Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1915, "Time": "2021-02-12T15:47:56.030Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 94, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1921, "Time": "2021-02-12T15:47:56.036Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 92, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1921, "Time": "2021-02-12T15:47:56.036Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 102, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1925, "Time": "2021-02-12T15:47:56.040Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1973, "Time": "2021-02-12T15:47:56.088Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 112, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1983, "Time": "2021-02-12T15:47:56.098Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 92, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2217, "Time": "2021-02-12T15:47:56.332Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2221, "Time": "2021-02-12T15:47:56.336Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2223, "Time": "2021-02-12T15:47:56.338Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2234, "Time": "2021-02-12T15:47:56.349Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2236, "Time": "2021-02-12T15:47:56.351Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 100, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2243, "Time": "2021-02-12T15:47:56.358Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2243, "Time": "2021-02-12T15:47:56.358Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2293, "Time": "2021-02-12T15:47:56.408Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 98, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2293, "Time": "2021-02-12T15:47:56.408Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2527, "Time": "2021-02-12T15:47:56.642Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 100, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2532, "Time": "2021-02-12T15:47:56.647Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 105, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2548, "Time": "2021-02-12T15:47:56.663Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2550, "Time": "2021-02-12T15:47:56.665Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 109, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2560, "Time": "2021-02-12T15:47:56.675Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 104, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2564, "Time": "2021-02-12T15:47:56.679Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 111, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2608, "Time": "2021-02-12T15:47:56.723Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 104, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2618, "Time": "2021-02-12T15:47:56.733Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2840, "Time": "2021-02-12T15:47:56.955Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2851, "Time": "2021-02-12T15:47:56.966Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 103, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2861, "Time": "2021-02-12T15:47:56.976Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2869, "Time": "2021-02-12T15:47:56.984Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 102, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2884, "Time": "2021-02-12T15:47:56.999Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2887, "Time": "2021-02-12T15:47:57.002Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 108, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2940, "Time": "2021-02-12T15:47:57.055Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2940, "Time": "2021-02-12T15:47:57.055Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 102, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2991, "Time": "2021-02-12T15:47:57.106Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 90, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3059, "Time": "2021-02-12T15:47:57.174Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3157, "Time": "2021-02-12T15:47:57.272Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3171, "Time": "2021-02-12T15:47:57.286Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 98, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3171, "Time": "2021-02-12T15:47:57.286Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 96, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3186, "Time": "2021-02-12T15:47:57.301Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 96, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3192, "Time": "2021-02-12T15:47:57.307Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 98, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3213, "Time": "2021-02-12T15:47:57.328Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3245, "Time": "2021-02-12T15:47:57.360Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3256, "Time": "2021-02-12T15:47:57.371Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 98, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3277, "Time": "2021-02-12T15:47:57.392Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 89, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3383, "Time": "2021-02-12T15:47:57.498Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 105, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3500, "Time": "2021-02-12T15:47:57.615Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 112, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3502, "Time": "2021-02-12T15:47:57.617Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 120, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3509, "Time": "2021-02-12T15:47:57.624Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 107, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3533, "Time": "2021-02-12T15:47:57.648Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 111, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3556, "Time": "2021-02-12T15:47:57.671Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 120, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3569, "Time": "2021-02-12T15:47:57.684Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 116, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3576, "Time": "2021-02-12T15:47:57.691Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 105, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3623, "Time": "2021-02-12T15:47:57.738Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 115, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3645, "Time": "2021-02-12T15:47:57.760Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 124, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3729, "Time": "2021-02-12T15:47:57.844Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 119, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3984, "Time": "2021-02-12T15:47:58.099Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 250, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4028, "Time": "2021-02-12T15:47:58.143Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 285, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4064, "Time": "2021-02-12T15:47:58.179Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 318, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4179, "Time": "2021-02-12T15:47:58.294Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 359, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4184, "Time": "2021-02-12T15:47:58.299Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 413, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4189, "Time": "2021-02-12T15:47:58.304Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 375, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4191, "Time": "2021-02-12T15:47:58.306Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 359, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4209, "Time": "2021-02-12T15:47:58.324Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 122, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4220, "Time": "2021-02-12T15:47:58.335Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 146, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4378, "Time": "2021-02-12T15:47:58.493Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 112, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4484, "Time": "2021-02-12T15:47:58.599Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 132, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4486, "Time": "2021-02-12T15:47:58.601Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 127, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4506, "Time": "2021-02-12T15:47:58.621Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 130, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4545, "Time": "2021-02-12T15:47:58.660Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 133, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4549, "Time": "2021-02-12T15:47:58.664Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 127, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4552, "Time": "2021-02-12T15:47:58.667Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 122, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4565, "Time": "2021-02-12T15:47:58.680Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 135, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4572, "Time": "2021-02-12T15:47:58.687Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 123, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4584, "Time": "2021-02-12T15:47:58.699Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 125, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4763, "Time": "2021-02-12T15:47:58.878Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 112, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4849, "Time": "2021-02-12T15:47:58.964Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4866, "Time": "2021-02-12T15:47:58.981Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4898, "Time": "2021-02-12T15:47:59.013Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 110, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4898, "Time": "2021-02-12T15:47:59.013Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 98, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4903, "Time": "2021-02-12T15:47:59.018Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 103, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4912, "Time": "2021-02-12T15:47:59.027Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 91, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4918, "Time": "2021-02-12T15:47:59.033Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4931, "Time": "2021-02-12T15:47:59.046Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4937, "Time": "2021-02-12T15:47:59.052Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 100, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5092, "Time": "2021-02-12T15:47:59.207Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 114, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5165, "Time": "2021-02-12T15:47:59.280Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 108, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5191, "Time": "2021-02-12T15:47:59.306Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5192, "Time": "2021-02-12T15:47:59.307Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 96, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5213, "Time": "2021-02-12T15:47:59.328Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5221, "Time": "2021-02-12T15:47:59.336Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5221, "Time": "2021-02-12T15:47:59.336Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 90, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5224, "Time": "2021-02-12T15:47:59.339Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 98, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5226, "Time": "2021-02-12T15:47:59.341Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 94, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5239, "Time": "2021-02-12T15:47:59.354Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 103, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5394, "Time": "2021-02-12T15:47:59.509Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 88, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5488, "Time": "2021-02-12T15:47:59.603Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5501, "Time": "2021-02-12T15:47:59.616Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 105, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5506, "Time": "2021-02-12T15:47:59.621Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 91, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5521, "Time": "2021-02-12T15:47:59.636Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 96, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5523, "Time": "2021-02-12T15:47:59.638Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5549, "Time": "2021-02-12T15:47:59.664Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 108, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5569, "Time": "2021-02-12T15:47:59.684Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 120, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5569, "Time": "2021-02-12T15:47:59.684Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 115, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5569, "Time": "2021-02-12T15:47:59.684Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 117, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5786, "Time": "2021-02-12T15:47:59.901Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 117, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5880, "Time": "2021-02-12T15:47:59.995Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 96, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5881, "Time": "2021-02-12T15:47:59.996Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 96, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5887, "Time": "2021-02-12T15:48:00.002Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5893, "Time": "2021-02-12T15:48:00.008Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 105, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5898, "Time": "2021-02-12T15:48:00.013Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 96, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5901, "Time": "2021-02-12T15:48:00.016Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 107, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5932, "Time": "2021-02-12T15:48:00.047Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 112, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5952, "Time": "2021-02-12T15:48:00.067Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 110, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5956, "Time": "2021-02-12T15:48:00.071Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 91, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6115, "Time": "2021-02-12T15:48:00.230Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 109, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6217, "Time": "2021-02-12T15:48:00.332Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6218, "Time": "2021-02-12T15:48:00.333Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 121, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6220, "Time": "2021-02-12T15:48:00.335Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 125, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6223, "Time": "2021-02-12T15:48:00.338Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 107, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6231, "Time": "2021-02-12T15:48:00.346Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6236, "Time": "2021-02-12T15:48:00.351Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 118, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6269, "Time": "2021-02-12T15:48:00.384Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6282, "Time": "2021-02-12T15:48:00.397Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 111, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6288, "Time": "2021-02-12T15:48:00.403Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 118, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6426, "Time": "2021-02-12T15:48:00.541Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 92, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6521, "Time": "2021-02-12T15:48:00.636Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 96, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6523, "Time": "2021-02-12T15:48:00.638Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6540, "Time": "2021-02-12T15:48:00.655Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 98, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6543, "Time": "2021-02-12T15:48:00.658Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6543, "Time": "2021-02-12T15:48:00.658Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6553, "Time": "2021-02-12T15:48:00.668Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 94, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6571, "Time": "2021-02-12T15:48:00.686Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6583, "Time": "2021-02-12T15:48:00.698Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 90, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6585, "Time": "2021-02-12T15:48:00.700Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6733, "Time": "2021-02-12T15:48:00.848Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 96, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6825, "Time": "2021-02-12T15:48:00.940Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6837, "Time": "2021-02-12T15:48:00.952Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 92, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6839, "Time": "2021-02-12T15:48:00.954Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6846, "Time": "2021-02-12T15:48:00.961Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 96, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6848, "Time": "2021-02-12T15:48:00.963Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6864, "Time": "2021-02-12T15:48:00.979Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 104, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6870, "Time": "2021-02-12T15:48:00.985Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 94, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6877, "Time": "2021-02-12T15:48:00.992Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6893, "Time": "2021-02-12T15:48:01.008Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7040, "Time": "2021-02-12T15:48:01.155Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7143, "Time": "2021-02-12T15:48:01.258Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7149, "Time": "2021-02-12T15:48:01.264Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7166, "Time": "2021-02-12T15:48:01.281Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7171, "Time": "2021-02-12T15:48:01.286Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 110, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7177, "Time": "2021-02-12T15:48:01.292Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 112, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7181, "Time": "2021-02-12T15:48:01.296Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 104, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7196, "Time": "2021-02-12T15:48:01.311Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 102, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7198, "Time": "2021-02-12T15:48:01.313Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 112, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7207, "Time": "2021-02-12T15:48:01.322Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 94, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7334, "Time": "2021-02-12T15:48:01.449Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7454, "Time": "2021-02-12T15:48:01.569Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7465, "Time": "2021-02-12T15:48:01.580Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7476, "Time": "2021-02-12T15:48:01.591Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7489, "Time": "2021-02-12T15:48:01.604Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7492, "Time": "2021-02-12T15:48:01.607Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 102, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7495, "Time": "2021-02-12T15:48:01.610Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 96, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7497, "Time": "2021-02-12T15:48:01.612Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 107, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7501, "Time": "2021-02-12T15:48:01.616Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 102, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7516, "Time": "2021-02-12T15:48:01.631Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 103, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7647, "Time": "2021-02-12T15:48:01.762Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7773, "Time": "2021-02-12T15:48:01.888Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 111, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7783, "Time": "2021-02-12T15:48:01.898Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 102, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7796, "Time": "2021-02-12T15:48:01.911Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 92, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7807, "Time": "2021-02-12T15:48:01.922Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7810, "Time": "2021-02-12T15:48:01.925Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 122, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7820, "Time": "2021-02-12T15:48:01.935Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 108, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7826, "Time": "2021-02-12T15:48:01.941Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 116, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7827, "Time": "2021-02-12T15:48:01.942Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 117, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7840, "Time": "2021-02-12T15:48:01.955Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7972, "Time": "2021-02-12T15:48:02.087Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 91, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8102, "Time": "2021-02-12T15:48:02.217Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 105, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8112, "Time": "2021-02-12T15:48:02.227Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 103, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8113, "Time": "2021-02-12T15:48:02.228Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8127, "Time": "2021-02-12T15:48:02.242Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 104, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8128, "Time": "2021-02-12T15:48:02.243Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8143, "Time": "2021-02-12T15:48:02.258Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8143, "Time": "2021-02-12T15:48:02.258Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8158, "Time": "2021-02-12T15:48:02.273Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 107, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8160, "Time": "2021-02-12T15:48:02.275Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 100, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8328, "Time": "2021-02-12T15:48:02.443Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 107, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8434, "Time": "2021-02-12T15:48:02.549Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 105, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8441, "Time": "2021-02-12T15:48:02.556Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 104, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8462, "Time": "2021-02-12T15:48:02.577Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8462, "Time": "2021-02-12T15:48:02.577Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8463, "Time": "2021-02-12T15:48:02.578Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8469, "Time": "2021-02-12T15:48:02.584Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 108, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8474, "Time": "2021-02-12T15:48:02.589Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 102, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8482, "Time": "2021-02-12T15:48:02.597Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 103, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8513, "Time": "2021-02-12T15:48:02.628Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 113, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8664, "Time": "2021-02-12T15:48:02.779Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8754, "Time": "2021-02-12T15:48:02.869Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 108, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8768, "Time": "2021-02-12T15:48:02.883Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 100, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8768, "Time": "2021-02-12T15:48:02.883Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8780, "Time": "2021-02-12T15:48:02.895Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 111, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8785, "Time": "2021-02-12T15:48:02.900Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8789, "Time": "2021-02-12T15:48:02.904Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 110, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8801, "Time": "2021-02-12T15:48:02.916Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8809, "Time": "2021-02-12T15:48:02.924Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 117, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8823, "Time": "2021-02-12T15:48:02.938Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 92, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8995, "Time": "2021-02-12T15:48:03.110Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 105, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9083, "Time": "2021-02-12T15:48:03.198Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9094, "Time": "2021-02-12T15:48:03.209Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 92, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9104, "Time": "2021-02-12T15:48:03.219Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 98, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9105, "Time": "2021-02-12T15:48:03.220Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9106, "Time": "2021-02-12T15:48:03.221Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 94, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9114, "Time": "2021-02-12T15:48:03.229Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9118, "Time": "2021-02-12T15:48:03.233Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 98, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9125, "Time": "2021-02-12T15:48:03.240Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9150, "Time": "2021-02-12T15:48:03.265Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9296, "Time": "2021-02-12T15:48:03.411Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9396, "Time": "2021-02-12T15:48:03.511Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 99, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9402, "Time": "2021-02-12T15:48:03.517Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 90, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9404, "Time": "2021-02-12T15:48:03.519Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 92, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9420, "Time": "2021-02-12T15:48:03.535Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 100, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9425, "Time": "2021-02-12T15:48:03.540Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9436, "Time": "2021-02-12T15:48:03.551Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 110, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9456, "Time": "2021-02-12T15:48:03.571Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 108, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9457, "Time": "2021-02-12T15:48:03.572Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 110, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9469, "Time": "2021-02-12T15:48:03.584Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 104, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9622, "Time": "2021-02-12T15:48:03.737Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9723, "Time": "2021-02-12T15:48:03.838Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 90, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9726, "Time": "2021-02-12T15:48:03.841Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9730, "Time": "2021-02-12T15:48:03.845Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 97, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9739, "Time": "2021-02-12T15:48:03.854Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 108, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9753, "Time": "2021-02-12T15:48:03.868Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 112, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9762, "Time": "2021-02-12T15:48:03.877Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 92, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9780, "Time": "2021-02-12T15:48:03.895Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 103, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9804, "Time": "2021-02-12T15:48:03.919Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 103, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9810, "Time": "2021-02-12T15:48:03.925Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 103, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9935, "Time": "2021-02-12T15:48:04.050Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 10038, "Time": "2021-02-12T15:48:04.153Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 105, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 10045, "Time": "2021-02-12T15:48:04.160Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 10055, "Time": "2021-02-12T15:48:04.170Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 110, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 10071, "Time": "2021-02-12T15:48:04.186Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 10071, "Time": "2021-02-12T15:48:04.186Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_no_sla", "Response time": 105, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/6a0ef850-8644-4236-80e5-53748fbd2d01/raw?format=JSON':
            return '[{"Elapsed": 896, "Time": "2021-02-12T15:47:55.011Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 896, "Time": "2021-02-12T15:47:55.011Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 901, "Time": "2021-02-12T15:47:55.016Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 901, "Time": "2021-02-12T15:47:55.016Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 901, "Time": "2021-02-12T15:47:55.016Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 905, "Time": "2021-02-12T15:47:55.020Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 75, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 916, "Time": "2021-02-12T15:47:55.031Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 916, "Time": "2021-02-12T15:47:55.031Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 78, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 916, "Time": "2021-02-12T15:47:55.031Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 85, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 931, "Time": "2021-02-12T15:47:55.046Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 77, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1357, "Time": "2021-02-12T15:47:55.472Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 75, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1357, "Time": "2021-02-12T15:47:55.472Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 75, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1357, "Time": "2021-02-12T15:47:55.472Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 99, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1367, "Time": "2021-02-12T15:47:55.482Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1377, "Time": "2021-02-12T15:47:55.492Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 87, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1381, "Time": "2021-02-12T15:47:55.496Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1381, "Time": "2021-02-12T15:47:55.496Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1390, "Time": "2021-02-12T15:47:55.505Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 81, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1415, "Time": "2021-02-12T15:47:55.530Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 85, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1451, "Time": "2021-02-12T15:47:55.566Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 88, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1729, "Time": "2021-02-12T15:47:55.844Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1731, "Time": "2021-02-12T15:47:55.846Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1741, "Time": "2021-02-12T15:47:55.856Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1741, "Time": "2021-02-12T15:47:55.856Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1748, "Time": "2021-02-12T15:47:55.863Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1749, "Time": "2021-02-12T15:47:55.864Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1750, "Time": "2021-02-12T15:47:55.865Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1789, "Time": "2021-02-12T15:47:55.904Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 90, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1809, "Time": "2021-02-12T15:47:55.924Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2045, "Time": "2021-02-12T15:47:56.160Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2045, "Time": "2021-02-12T15:47:56.160Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2056, "Time": "2021-02-12T15:47:56.171Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2062, "Time": "2021-02-12T15:47:56.177Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2066, "Time": "2021-02-12T15:47:56.181Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2076, "Time": "2021-02-12T15:47:56.191Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2076, "Time": "2021-02-12T15:47:56.191Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2128, "Time": "2021-02-12T15:47:56.243Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2128, "Time": "2021-02-12T15:47:56.243Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2356, "Time": "2021-02-12T15:47:56.471Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2359, "Time": "2021-02-12T15:47:56.474Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2366, "Time": "2021-02-12T15:47:56.481Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2375, "Time": "2021-02-12T15:47:56.490Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2382, "Time": "2021-02-12T15:47:56.497Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2382, "Time": "2021-02-12T15:47:56.497Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2385, "Time": "2021-02-12T15:47:56.500Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2433, "Time": "2021-02-12T15:47:56.548Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2441, "Time": "2021-02-12T15:47:56.556Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2669, "Time": "2021-02-12T15:47:56.784Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2675, "Time": "2021-02-12T15:47:56.790Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2688, "Time": "2021-02-12T15:47:56.803Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2697, "Time": "2021-02-12T15:47:56.812Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2703, "Time": "2021-02-12T15:47:56.818Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2708, "Time": "2021-02-12T15:47:56.823Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2755, "Time": "2021-02-12T15:47:56.870Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2763, "Time": "2021-02-12T15:47:56.878Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2897, "Time": "2021-02-12T15:47:57.012Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 1236, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2991, "Time": "2021-02-12T15:47:57.106Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3006, "Time": "2021-02-12T15:47:57.121Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3006, "Time": "2021-02-12T15:47:57.121Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3022, "Time": "2021-02-12T15:47:57.137Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3026, "Time": "2021-02-12T15:47:57.141Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3044, "Time": "2021-02-12T15:47:57.159Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 91, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3080, "Time": "2021-02-12T15:47:57.195Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3092, "Time": "2021-02-12T15:47:57.207Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3117, "Time": "2021-02-12T15:47:57.232Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3199, "Time": "2021-02-12T15:47:57.314Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3297, "Time": "2021-02-12T15:47:57.412Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3297, "Time": "2021-02-12T15:47:57.412Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3319, "Time": "2021-02-12T15:47:57.434Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3346, "Time": "2021-02-12T15:47:57.461Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3346, "Time": "2021-02-12T15:47:57.461Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 89, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3365, "Time": "2021-02-12T15:47:57.480Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3401, "Time": "2021-02-12T15:47:57.516Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3422, "Time": "2021-02-12T15:47:57.537Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3435, "Time": "2021-02-12T15:47:57.550Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3534, "Time": "2021-02-12T15:47:57.649Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3656, "Time": "2021-02-12T15:47:57.771Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3660, "Time": "2021-02-12T15:47:57.775Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3666, "Time": "2021-02-12T15:47:57.781Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 83, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3689, "Time": "2021-02-12T15:47:57.804Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3710, "Time": "2021-02-12T15:47:57.825Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3723, "Time": "2021-02-12T15:47:57.838Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3733, "Time": "2021-02-12T15:47:57.848Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3774, "Time": "2021-02-12T15:47:57.889Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3784, "Time": "2021-02-12T15:47:57.899Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4181, "Time": "2021-02-12T15:47:58.296Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 365, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4275, "Time": "2021-02-12T15:47:58.390Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4276, "Time": "2021-02-12T15:47:58.391Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4296, "Time": "2021-02-12T15:47:58.411Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4323, "Time": "2021-02-12T15:47:58.438Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4342, "Time": "2021-02-12T15:47:58.457Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4342, "Time": "2021-02-12T15:47:58.457Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4347, "Time": "2021-02-12T15:47:58.462Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4358, "Time": "2021-02-12T15:47:58.473Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4373, "Time": "2021-02-12T15:47:58.488Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4550, "Time": "2021-02-12T15:47:58.665Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4665, "Time": "2021-02-12T15:47:58.780Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4680, "Time": "2021-02-12T15:47:58.795Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 106, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4704, "Time": "2021-02-12T15:47:58.819Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4731, "Time": "2021-02-12T15:47:58.846Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 83, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4731, "Time": "2021-02-12T15:47:58.846Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 98, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4737, "Time": "2021-02-12T15:47:58.852Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 89, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4752, "Time": "2021-02-12T15:47:58.867Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 90, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4755, "Time": "2021-02-12T15:47:58.870Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4755, "Time": "2021-02-12T15:47:58.870Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4910, "Time": "2021-02-12T15:47:59.025Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4996, "Time": "2021-02-12T15:47:59.111Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5012, "Time": "2021-02-12T15:47:59.127Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5027, "Time": "2021-02-12T15:47:59.142Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5033, "Time": "2021-02-12T15:47:59.148Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5037, "Time": "2021-02-12T15:47:59.152Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5053, "Time": "2021-02-12T15:47:59.168Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5056, "Time": "2021-02-12T15:47:59.171Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5063, "Time": "2021-02-12T15:47:59.178Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 59, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5063, "Time": "2021-02-12T15:47:59.178Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5231, "Time": "2021-02-12T15:47:59.346Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5319, "Time": "2021-02-12T15:47:59.434Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5320, "Time": "2021-02-12T15:47:59.435Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5330, "Time": "2021-02-12T15:47:59.445Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5350, "Time": "2021-02-12T15:47:59.465Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5354, "Time": "2021-02-12T15:47:59.469Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5361, "Time": "2021-02-12T15:47:59.476Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5381, "Time": "2021-02-12T15:47:59.496Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5381, "Time": "2021-02-12T15:47:59.496Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5381, "Time": "2021-02-12T15:47:59.496Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5568, "Time": "2021-02-12T15:47:59.683Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 92, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5684, "Time": "2021-02-12T15:47:59.799Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5684, "Time": "2021-02-12T15:47:59.799Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5697, "Time": "2021-02-12T15:47:59.812Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5697, "Time": "2021-02-12T15:47:59.812Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5717, "Time": "2021-02-12T15:47:59.832Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5718, "Time": "2021-02-12T15:47:59.833Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5733, "Time": "2021-02-12T15:47:59.848Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5754, "Time": "2021-02-12T15:47:59.869Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5784, "Time": "2021-02-12T15:47:59.899Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 100, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5930, "Time": "2021-02-12T15:48:00.045Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6020, "Time": "2021-02-12T15:48:00.135Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6025, "Time": "2021-02-12T15:48:00.140Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6039, "Time": "2021-02-12T15:48:00.154Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 86, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6042, "Time": "2021-02-12T15:48:00.157Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6043, "Time": "2021-02-12T15:48:00.158Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6056, "Time": "2021-02-12T15:48:00.171Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6086, "Time": "2021-02-12T15:48:00.201Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6089, "Time": "2021-02-12T15:48:00.204Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6094, "Time": "2021-02-12T15:48:00.209Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6267, "Time": "2021-02-12T15:48:00.382Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6363, "Time": "2021-02-12T15:48:00.478Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6363, "Time": "2021-02-12T15:48:00.478Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6369, "Time": "2021-02-12T15:48:00.484Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6376, "Time": "2021-02-12T15:48:00.491Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6377, "Time": "2021-02-12T15:48:00.492Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6377, "Time": "2021-02-12T15:48:00.492Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6409, "Time": "2021-02-12T15:48:00.524Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6424, "Time": "2021-02-12T15:48:00.539Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6430, "Time": "2021-02-12T15:48:00.545Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6572, "Time": "2021-02-12T15:48:00.687Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6660, "Time": "2021-02-12T15:48:00.775Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6661, "Time": "2021-02-12T15:48:00.776Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6673, "Time": "2021-02-12T15:48:00.788Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6680, "Time": "2021-02-12T15:48:00.795Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6690, "Time": "2021-02-12T15:48:00.805Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6691, "Time": "2021-02-12T15:48:00.806Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6708, "Time": "2021-02-12T15:48:00.823Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6719, "Time": "2021-02-12T15:48:00.834Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6719, "Time": "2021-02-12T15:48:00.834Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6877, "Time": "2021-02-12T15:48:00.992Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6967, "Time": "2021-02-12T15:48:01.082Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6978, "Time": "2021-02-12T15:48:01.093Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6978, "Time": "2021-02-12T15:48:01.093Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6986, "Time": "2021-02-12T15:48:01.101Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6998, "Time": "2021-02-12T15:48:01.113Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7010, "Time": "2021-02-12T15:48:01.125Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7017, "Time": "2021-02-12T15:48:01.132Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7017, "Time": "2021-02-12T15:48:01.132Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7033, "Time": "2021-02-12T15:48:01.148Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7177, "Time": "2021-02-12T15:48:01.292Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7274, "Time": "2021-02-12T15:48:01.389Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7291, "Time": "2021-02-12T15:48:01.406Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7303, "Time": "2021-02-12T15:48:01.418Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7316, "Time": "2021-02-12T15:48:01.431Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7321, "Time": "2021-02-12T15:48:01.436Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7323, "Time": "2021-02-12T15:48:01.438Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7334, "Time": "2021-02-12T15:48:01.449Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7334, "Time": "2021-02-12T15:48:01.449Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7343, "Time": "2021-02-12T15:48:01.458Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7484, "Time": "2021-02-12T15:48:01.599Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7587, "Time": "2021-02-12T15:48:01.702Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7606, "Time": "2021-02-12T15:48:01.721Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7618, "Time": "2021-02-12T15:48:01.733Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7627, "Time": "2021-02-12T15:48:01.742Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7630, "Time": "2021-02-12T15:48:01.745Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7635, "Time": "2021-02-12T15:48:01.750Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7635, "Time": "2021-02-12T15:48:01.750Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7639, "Time": "2021-02-12T15:48:01.754Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7652, "Time": "2021-02-12T15:48:01.767Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7816, "Time": "2021-02-12T15:48:01.931Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7918, "Time": "2021-02-12T15:48:02.033Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7928, "Time": "2021-02-12T15:48:02.043Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7938, "Time": "2021-02-12T15:48:02.053Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7942, "Time": "2021-02-12T15:48:02.057Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7948, "Time": "2021-02-12T15:48:02.063Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7956, "Time": "2021-02-12T15:48:02.071Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7959, "Time": "2021-02-12T15:48:02.074Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7961, "Time": "2021-02-12T15:48:02.076Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7970, "Time": "2021-02-12T15:48:02.085Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8143, "Time": "2021-02-12T15:48:02.258Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8256, "Time": "2021-02-12T15:48:02.371Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8263, "Time": "2021-02-12T15:48:02.378Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8269, "Time": "2021-02-12T15:48:02.384Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8275, "Time": "2021-02-12T15:48:02.390Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8285, "Time": "2021-02-12T15:48:02.400Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8290, "Time": "2021-02-12T15:48:02.405Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8295, "Time": "2021-02-12T15:48:02.410Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8300, "Time": "2021-02-12T15:48:02.415Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8314, "Time": "2021-02-12T15:48:02.429Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8484, "Time": "2021-02-12T15:48:02.599Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8578, "Time": "2021-02-12T15:48:02.693Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8593, "Time": "2021-02-12T15:48:02.708Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8601, "Time": "2021-02-12T15:48:02.716Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8605, "Time": "2021-02-12T15:48:02.720Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8611, "Time": "2021-02-12T15:48:02.726Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8612, "Time": "2021-02-12T15:48:02.727Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8624, "Time": "2021-02-12T15:48:02.739Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8627, "Time": "2021-02-12T15:48:02.742Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8657, "Time": "2021-02-12T15:48:02.772Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8808, "Time": "2021-02-12T15:48:02.923Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8899, "Time": "2021-02-12T15:48:03.014Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8929, "Time": "2021-02-12T15:48:03.044Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8939, "Time": "2021-02-12T15:48:03.054Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 86, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8939, "Time": "2021-02-12T15:48:03.054Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8942, "Time": "2021-02-12T15:48:03.057Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8954, "Time": "2021-02-12T15:48:03.069Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8954, "Time": "2021-02-12T15:48:03.069Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8963, "Time": "2021-02-12T15:48:03.078Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8976, "Time": "2021-02-12T15:48:03.091Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9131, "Time": "2021-02-12T15:48:03.246Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9230, "Time": "2021-02-12T15:48:03.345Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9234, "Time": "2021-02-12T15:48:03.349Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9242, "Time": "2021-02-12T15:48:03.357Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9245, "Time": "2021-02-12T15:48:03.360Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9255, "Time": "2021-02-12T15:48:03.370Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9255, "Time": "2021-02-12T15:48:03.370Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9256, "Time": "2021-02-12T15:48:03.371Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9264, "Time": "2021-02-12T15:48:03.379Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9293, "Time": "2021-02-12T15:48:03.408Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9442, "Time": "2021-02-12T15:48:03.557Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9537, "Time": "2021-02-12T15:48:03.652Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9551, "Time": "2021-02-12T15:48:03.666Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9558, "Time": "2021-02-12T15:48:03.673Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9561, "Time": "2021-02-12T15:48:03.676Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9571, "Time": "2021-02-12T15:48:03.686Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9590, "Time": "2021-02-12T15:48:03.705Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9601, "Time": "2021-02-12T15:48:03.716Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9614, "Time": "2021-02-12T15:48:03.729Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9623, "Time": "2021-02-12T15:48:03.738Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9760, "Time": "2021-02-12T15:48:03.875Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9866, "Time": "2021-02-12T15:48:03.981Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9875, "Time": "2021-02-12T15:48:03.990Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9877, "Time": "2021-02-12T15:48:03.992Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9901, "Time": "2021-02-12T15:48:04.016Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9902, "Time": "2021-02-12T15:48:04.017Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9912, "Time": "2021-02-12T15:48:04.027Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9920, "Time": "2021-02-12T15:48:04.035Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9952, "Time": "2021-02-12T15:48:04.067Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9954, "Time": "2021-02-12T15:48:04.069Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 10076, "Time": "2021-02-12T15:48:04.191Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_failed", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/elements/7dea9e71-4777-4fdb-88d0-2a3dbb2eaa1a/raw?format=JSON':
            return '[{"Elapsed": 817, "Time": "2021-02-12T15:47:54.932Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 818, "Time": "2021-02-12T15:47:54.933Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 829, "Time": "2021-02-12T15:47:54.944Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 78, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 829, "Time": "2021-02-12T15:47:54.944Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 81, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 829, "Time": "2021-02-12T15:47:54.944Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 81, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 829, "Time": "2021-02-12T15:47:54.944Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 80, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 829, "Time": "2021-02-12T15:47:54.944Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 78, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 836, "Time": "2021-02-12T15:47:54.951Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 85, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 841, "Time": "2021-02-12T15:47:54.956Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 92, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 850, "Time": "2021-02-12T15:47:54.965Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 91, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1253, "Time": "2021-02-12T15:47:55.368Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 78, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1277, "Time": "2021-02-12T15:47:55.392Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 96, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1277, "Time": "2021-02-12T15:47:55.392Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 96, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1287, "Time": "2021-02-12T15:47:55.402Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 88, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1287, "Time": "2021-02-12T15:47:55.402Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 113, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1305, "Time": "2021-02-12T15:47:55.420Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 90, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1305, "Time": "2021-02-12T15:47:55.420Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 83, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1305, "Time": "2021-02-12T15:47:55.420Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 90, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1326, "Time": "2021-02-12T15:47:55.441Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 107, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1326, "Time": "2021-02-12T15:47:55.441Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 87, "Success": "yes", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1649, "Time": "2021-02-12T15:47:55.764Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1652, "Time": "2021-02-12T15:47:55.767Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1655, "Time": "2021-02-12T15:47:55.770Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1659, "Time": "2021-02-12T15:47:55.774Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1661, "Time": "2021-02-12T15:47:55.776Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1661, "Time": "2021-02-12T15:47:55.776Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1672, "Time": "2021-02-12T15:47:55.787Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1672, "Time": "2021-02-12T15:47:55.787Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1697, "Time": "2021-02-12T15:47:55.812Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1734, "Time": "2021-02-12T15:47:55.849Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1969, "Time": "2021-02-12T15:47:56.084Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 62, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1971, "Time": "2021-02-12T15:47:56.086Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1990, "Time": "2021-02-12T15:47:56.105Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1990, "Time": "2021-02-12T15:47:56.105Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1990, "Time": "2021-02-12T15:47:56.105Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 1995, "Time": "2021-02-12T15:47:56.110Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2003, "Time": "2021-02-12T15:47:56.118Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2045, "Time": "2021-02-12T15:47:56.160Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2058, "Time": "2021-02-12T15:47:56.173Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2287, "Time": "2021-02-12T15:47:56.402Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2291, "Time": "2021-02-12T15:47:56.406Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2296, "Time": "2021-02-12T15:47:56.411Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2305, "Time": "2021-02-12T15:47:56.420Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2307, "Time": "2021-02-12T15:47:56.422Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2312, "Time": "2021-02-12T15:47:56.427Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2312, "Time": "2021-02-12T15:47:56.427Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2361, "Time": "2021-02-12T15:47:56.476Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2361, "Time": "2021-02-12T15:47:56.476Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2599, "Time": "2021-02-12T15:47:56.714Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2602, "Time": "2021-02-12T15:47:56.717Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2616, "Time": "2021-02-12T15:47:56.731Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2618, "Time": "2021-02-12T15:47:56.733Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2628, "Time": "2021-02-12T15:47:56.743Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2633, "Time": "2021-02-12T15:47:56.748Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2687, "Time": "2021-02-12T15:47:56.802Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2688, "Time": "2021-02-12T15:47:56.803Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2918, "Time": "2021-02-12T15:47:57.033Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2929, "Time": "2021-02-12T15:47:57.044Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2930, "Time": "2021-02-12T15:47:57.045Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2951, "Time": "2021-02-12T15:47:57.066Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2951, "Time": "2021-02-12T15:47:57.066Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 2951, "Time": "2021-02-12T15:47:57.066Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3010, "Time": "2021-02-12T15:47:57.125Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3015, "Time": "2021-02-12T15:47:57.130Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3053, "Time": "2021-02-12T15:47:57.168Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 62, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3130, "Time": "2021-02-12T15:47:57.245Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3233, "Time": "2021-02-12T15:47:57.348Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3233, "Time": "2021-02-12T15:47:57.348Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 62, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3245, "Time": "2021-02-12T15:47:57.360Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3255, "Time": "2021-02-12T15:47:57.370Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3264, "Time": "2021-02-12T15:47:57.379Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3287, "Time": "2021-02-12T15:47:57.402Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3312, "Time": "2021-02-12T15:47:57.427Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3345, "Time": "2021-02-12T15:47:57.460Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3351, "Time": "2021-02-12T15:47:57.466Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3469, "Time": "2021-02-12T15:47:57.584Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3572, "Time": "2021-02-12T15:47:57.687Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3575, "Time": "2021-02-12T15:47:57.690Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3582, "Time": "2021-02-12T15:47:57.697Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3624, "Time": "2021-02-12T15:47:57.739Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 88, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3628, "Time": "2021-02-12T15:47:57.743Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3655, "Time": "2021-02-12T15:47:57.770Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3659, "Time": "2021-02-12T15:47:57.774Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3697, "Time": "2021-02-12T15:47:57.812Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3713, "Time": "2021-02-12T15:47:57.828Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 3815, "Time": "2021-02-12T15:47:57.930Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 86, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4200, "Time": "2021-02-12T15:47:58.315Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 215, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4209, "Time": "2021-02-12T15:47:58.324Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 179, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4211, "Time": "2021-02-12T15:47:58.326Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 146, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4246, "Time": "2021-02-12T15:47:58.361Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4260, "Time": "2021-02-12T15:47:58.375Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4260, "Time": "2021-02-12T15:47:58.375Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4265, "Time": "2021-02-12T15:47:58.380Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4278, "Time": "2021-02-12T15:47:58.393Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4296, "Time": "2021-02-12T15:47:58.411Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4465, "Time": "2021-02-12T15:47:58.580Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4571, "Time": "2021-02-12T15:47:58.686Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4573, "Time": "2021-02-12T15:47:58.688Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4602, "Time": "2021-02-12T15:47:58.717Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 96, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4633, "Time": "2021-02-12T15:47:58.748Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 84, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4647, "Time": "2021-02-12T15:47:58.762Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 102, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4647, "Time": "2021-02-12T15:47:58.762Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4662, "Time": "2021-02-12T15:47:58.777Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 96, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4674, "Time": "2021-02-12T15:47:58.789Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 101, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4675, "Time": "2021-02-12T15:47:58.790Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 90, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4838, "Time": "2021-02-12T15:47:58.953Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4931, "Time": "2021-02-12T15:47:59.046Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4940, "Time": "2021-02-12T15:47:59.055Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4963, "Time": "2021-02-12T15:47:59.078Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4963, "Time": "2021-02-12T15:47:59.078Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4971, "Time": "2021-02-12T15:47:59.086Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4984, "Time": "2021-02-12T15:47:59.099Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4987, "Time": "2021-02-12T15:47:59.102Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 4997, "Time": "2021-02-12T15:47:59.112Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5003, "Time": "2021-02-12T15:47:59.118Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5159, "Time": "2021-02-12T15:47:59.274Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5244, "Time": "2021-02-12T15:47:59.359Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5255, "Time": "2021-02-12T15:47:59.370Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5264, "Time": "2021-02-12T15:47:59.379Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5278, "Time": "2021-02-12T15:47:59.393Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5286, "Time": "2021-02-12T15:47:59.401Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5297, "Time": "2021-02-12T15:47:59.412Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5297, "Time": "2021-02-12T15:47:59.412Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5298, "Time": "2021-02-12T15:47:59.413Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5308, "Time": "2021-02-12T15:47:59.423Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5473, "Time": "2021-02-12T15:47:59.588Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5601, "Time": "2021-02-12T15:47:59.716Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 98, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5603, "Time": "2021-02-12T15:47:59.718Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 95, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5603, "Time": "2021-02-12T15:47:59.718Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 110, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5615, "Time": "2021-02-12T15:47:59.730Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 92, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5644, "Time": "2021-02-12T15:47:59.759Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 117, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5645, "Time": "2021-02-12T15:47:59.760Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 93, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5652, "Time": "2021-02-12T15:47:59.767Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5669, "Time": "2021-02-12T15:47:59.784Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 90, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5681, "Time": "2021-02-12T15:47:59.796Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 111, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5859, "Time": "2021-02-12T15:47:59.974Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5949, "Time": "2021-02-12T15:48:00.064Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5952, "Time": "2021-02-12T15:48:00.067Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5953, "Time": "2021-02-12T15:48:00.068Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5965, "Time": "2021-02-12T15:48:00.080Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5966, "Time": "2021-02-12T15:48:00.081Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 5981, "Time": "2021-02-12T15:48:00.096Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6007, "Time": "2021-02-12T15:48:00.122Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6017, "Time": "2021-02-12T15:48:00.132Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6025, "Time": "2021-02-12T15:48:00.140Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6190, "Time": "2021-02-12T15:48:00.305Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6300, "Time": "2021-02-12T15:48:00.415Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6300, "Time": "2021-02-12T15:48:00.415Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6300, "Time": "2021-02-12T15:48:00.415Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6304, "Time": "2021-02-12T15:48:00.419Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 87, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6308, "Time": "2021-02-12T15:48:00.423Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6310, "Time": "2021-02-12T15:48:00.425Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6340, "Time": "2021-02-12T15:48:00.455Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6355, "Time": "2021-02-12T15:48:00.470Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6358, "Time": "2021-02-12T15:48:00.473Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6500, "Time": "2021-02-12T15:48:00.615Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6591, "Time": "2021-02-12T15:48:00.706Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6592, "Time": "2021-02-12T15:48:00.707Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6604, "Time": "2021-02-12T15:48:00.719Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6609, "Time": "2021-02-12T15:48:00.724Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6611, "Time": "2021-02-12T15:48:00.726Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6622, "Time": "2021-02-12T15:48:00.737Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6637, "Time": "2021-02-12T15:48:00.752Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6651, "Time": "2021-02-12T15:48:00.766Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6652, "Time": "2021-02-12T15:48:00.767Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6805, "Time": "2021-02-12T15:48:00.920Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6895, "Time": "2021-02-12T15:48:01.010Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6903, "Time": "2021-02-12T15:48:01.018Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6913, "Time": "2021-02-12T15:48:01.028Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6914, "Time": "2021-02-12T15:48:01.029Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6917, "Time": "2021-02-12T15:48:01.032Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6928, "Time": "2021-02-12T15:48:01.043Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6948, "Time": "2021-02-12T15:48:01.063Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6950, "Time": "2021-02-12T15:48:01.065Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 6966, "Time": "2021-02-12T15:48:01.081Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7111, "Time": "2021-02-12T15:48:01.226Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7204, "Time": "2021-02-12T15:48:01.319Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 61, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7222, "Time": "2021-02-12T15:48:01.337Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7230, "Time": "2021-02-12T15:48:01.345Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7243, "Time": "2021-02-12T15:48:01.358Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7249, "Time": "2021-02-12T15:48:01.364Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7250, "Time": "2021-02-12T15:48:01.365Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7260, "Time": "2021-02-12T15:48:01.375Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7268, "Time": "2021-02-12T15:48:01.383Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7276, "Time": "2021-02-12T15:48:01.391Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7413, "Time": "2021-02-12T15:48:01.528Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7519, "Time": "2021-02-12T15:48:01.634Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7539, "Time": "2021-02-12T15:48:01.654Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7549, "Time": "2021-02-12T15:48:01.664Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7556, "Time": "2021-02-12T15:48:01.671Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7564, "Time": "2021-02-12T15:48:01.679Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7565, "Time": "2021-02-12T15:48:01.680Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7566, "Time": "2021-02-12T15:48:01.681Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7570, "Time": "2021-02-12T15:48:01.685Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7582, "Time": "2021-02-12T15:48:01.697Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7729, "Time": "2021-02-12T15:48:01.844Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7846, "Time": "2021-02-12T15:48:01.961Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7857, "Time": "2021-02-12T15:48:01.972Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7862, "Time": "2021-02-12T15:48:01.977Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7876, "Time": "2021-02-12T15:48:01.991Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7876, "Time": "2021-02-12T15:48:01.991Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7887, "Time": "2021-02-12T15:48:02.002Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7887, "Time": "2021-02-12T15:48:02.002Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 61, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7895, "Time": "2021-02-12T15:48:02.010Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 7906, "Time": "2021-02-12T15:48:02.021Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8062, "Time": "2021-02-12T15:48:02.177Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 90, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8188, "Time": "2021-02-12T15:48:02.303Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 85, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8189, "Time": "2021-02-12T15:48:02.304Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8192, "Time": "2021-02-12T15:48:02.307Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8202, "Time": "2021-02-12T15:48:02.317Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8208, "Time": "2021-02-12T15:48:02.323Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8216, "Time": "2021-02-12T15:48:02.331Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8226, "Time": "2021-02-12T15:48:02.341Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8233, "Time": "2021-02-12T15:48:02.348Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8236, "Time": "2021-02-12T15:48:02.351Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8412, "Time": "2021-02-12T15:48:02.527Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 83, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8503, "Time": "2021-02-12T15:48:02.618Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8517, "Time": "2021-02-12T15:48:02.632Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8528, "Time": "2021-02-12T15:48:02.643Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8530, "Time": "2021-02-12T15:48:02.645Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8534, "Time": "2021-02-12T15:48:02.649Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8536, "Time": "2021-02-12T15:48:02.651Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8545, "Time": "2021-02-12T15:48:02.660Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8551, "Time": "2021-02-12T15:48:02.666Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8584, "Time": "2021-02-12T15:48:02.699Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8734, "Time": "2021-02-12T15:48:02.849Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8822, "Time": "2021-02-12T15:48:02.937Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8850, "Time": "2021-02-12T15:48:02.965Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 81, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8850, "Time": "2021-02-12T15:48:02.965Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8854, "Time": "2021-02-12T15:48:02.969Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 74, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8855, "Time": "2021-02-12T15:48:02.970Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8867, "Time": "2021-02-12T15:48:02.982Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8875, "Time": "2021-02-12T15:48:02.990Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8888, "Time": "2021-02-12T15:48:03.003Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 78, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 8900, "Time": "2021-02-12T15:48:03.015Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9063, "Time": "2021-02-12T15:48:03.178Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9159, "Time": "2021-02-12T15:48:03.274Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 75, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9163, "Time": "2021-02-12T15:48:03.278Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9171, "Time": "2021-02-12T15:48:03.286Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 64, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9178, "Time": "2021-02-12T15:48:03.293Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9179, "Time": "2021-02-12T15:48:03.294Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9181, "Time": "2021-02-12T15:48:03.296Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9189, "Time": "2021-02-12T15:48:03.304Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 63, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9189, "Time": "2021-02-12T15:48:03.304Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 70, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9222, "Time": "2021-02-12T15:48:03.337Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 72, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9363, "Time": "2021-02-12T15:48:03.478Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9466, "Time": "2021-02-12T15:48:03.581Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9480, "Time": "2021-02-12T15:48:03.595Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9485, "Time": "2021-02-12T15:48:03.600Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 80, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9488, "Time": "2021-02-12T15:48:03.603Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 66, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9504, "Time": "2021-02-12T15:48:03.619Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9518, "Time": "2021-02-12T15:48:03.633Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 82, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9526, "Time": "2021-02-12T15:48:03.641Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 69, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9537, "Time": "2021-02-12T15:48:03.652Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 79, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9537, "Time": "2021-02-12T15:48:03.652Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 68, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9690, "Time": "2021-02-12T15:48:03.805Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9790, "Time": "2021-02-12T15:48:03.905Z", "User Path": "kamoulox", "Virtual User ID": "0-1", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 67, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9804, "Time": "2021-02-12T15:48:03.919Z", "User Path": "kamoulox", "Virtual User ID": "0-4", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 73, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9805, "Time": "2021-02-12T15:48:03.920Z", "User Path": "kamoulox", "Virtual User ID": "0-5", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9818, "Time": "2021-02-12T15:48:03.933Z", "User Path": "kamoulox", "Virtual User ID": "0-0", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9831, "Time": "2021-02-12T15:48:03.946Z", "User Path": "kamoulox", "Virtual User ID": "0-3", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9845, "Time": "2021-02-12T15:48:03.960Z", "User Path": "kamoulox", "Virtual User ID": "0-9", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 65, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9848, "Time": "2021-02-12T15:48:03.963Z", "User Path": "kamoulox", "Virtual User ID": "0-6", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 83, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9882, "Time": "2021-02-12T15:48:03.997Z", "User Path": "kamoulox", "Virtual User ID": "0-7", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 77, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 9887, "Time": "2021-02-12T15:48:04.002Z", "User Path": "kamoulox", "Virtual User ID": "0-2", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 76, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}, {"Elapsed": 10007, "Time": "2021-02-12T15:48:04.122Z", "User Path": "kamoulox", "Virtual User ID": "0-8", "Parent": "Actions", "Element": "transaction_sla_warning", "Response time": 71, "Success": "no", "Population": "REST_API_Population", "Zone": "Guillaume"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/monitors':
            return '[{"id": "de6f330b-eecb-32cf-8582-63e0e1654735", "name": "User Load", "path": ["Controller", "User Load"]}, {"id": "1949328f-ad88-3770-933a-f104f026f14c", "name": "Throughput", "path": ["Controller", "Throughput"]}, {"id": "424155ec-15a1-3b91-8261-0ae15cda2a19", "name": "CPU Load", "path": ["Controller", "CPU Load"]}, {"id": "8f0f9bab-9cf2-3958-9085-4414999b5c0d", "name": "Memory Used", "path": ["Controller", "Memory Used"]}, {"id": "a471ab71-378c-3e5b-b119-6c9b20b86106", "name": "Thread Count", "path": ["Controller", "Thread Count"]}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/monitors/de6f330b-eecb-32cf-8582-63e0e1654735/points':
            return '[{"from": 0, "to": 1000, "AVG": 0.0}, {"from": 1000, "to": 2000, "AVG": 10.0}, {"from": 2000, "to": 3000, "AVG": 10.0}, {"from": 3000, "to": 4000, "AVG": 10.0}, {"from": 4000, "to": 5000, "AVG": 10.0}, {"from": 5000, "to": 6000, "AVG": 10.0}, {"from": 6000, "to": 7000, "AVG": 10.0}, {"from": 7000, "to": 8000, "AVG": 10.0}, {"from": 8000, "to": 9000, "AVG": 10.0}, {"from": 9000, "to": 10000, "AVG": 10.0}, {"from": 10000, "to": 11000, "AVG": 10.0}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/monitors/1949328f-ad88-3770-933a-f104f026f14c/points':
            return '[{"from": 0, "to": 1000, "AVG": 0.0}, {"from": 1000, "to": 2000, "AVG": 0.08897915}, {"from": 2000, "to": 3000, "AVG": 1.3978566}, {"from": 3000, "to": 4000, "AVG": 2.413369}, {"from": 4000, "to": 5000, "AVG": 2.6264877}, {"from": 5000, "to": 6000, "AVG": 1.7971404}, {"from": 6000, "to": 7000, "AVG": 2.6735117}, {"from": 7000, "to": 8000, "AVG": 2.764439}, {"from": 8000, "to": 9000, "AVG": 2.8976302}, {"from": 9000, "to": 10000, "AVG": 2.7339554}, {"from": 10000, "to": 11000, "AVG": 2.7284}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/monitors/424155ec-15a1-3b91-8261-0ae15cda2a19/points':
            return '[{"from": 0, "to": 1000, "AVG": 14.0}, {"from": 1000, "to": 2000, "AVG": 27.0}, {"from": 2000, "to": 3000, "AVG": 40.0}, {"from": 3000, "to": 4000, "AVG": 23.0}, {"from": 4000, "to": 5000, "AVG": 29.0}, {"from": 5000, "to": 6000, "AVG": 22.0}, {"from": 6000, "to": 7000, "AVG": 32.0}, {"from": 7000, "to": 8000, "AVG": 41.0}, {"from": 8000, "to": 9000, "AVG": 20.0}, {"from": 9000, "to": 10000, "AVG": 46.0}, {"from": 10000, "to": 11000, "AVG": 42.0}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/monitors/8f0f9bab-9cf2-3958-9085-4414999b5c0d/points':
            return '[{"from": 0, "to": 1000, "AVG": 2.0}, {"from": 1000, "to": 2000, "AVG": 3.0}, {"from": 2000, "to": 3000, "AVG": 3.0}, {"from": 3000, "to": 4000, "AVG": 3.0}, {"from": 4000, "to": 5000, "AVG": 3.0}, {"from": 5000, "to": 6000, "AVG": 3.0}, {"from": 6000, "to": 7000, "AVG": 4.0}, {"from": 7000, "to": 8000, "AVG": 4.0}, {"from": 8000, "to": 9000, "AVG": 4.0}, {"from": 9000, "to": 10000, "AVG": 4.0}, {"from": 10000, "to": 11000, "AVG": 4.0}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/1c393548-1e78-4632-8dd1-b661fb6cd043/monitors/a471ab71-378c-3e5b-b119-6c9b20b86106/points':
            return '[{"from": 0, "to": 1000, "AVG": 247.0}, {"from": 1000, "to": 2000, "AVG": 270.0}, {"from": 2000, "to": 3000, "AVG": 270.0}, {"from": 3000, "to": 4000, "AVG": 272.0}, {"from": 4000, "to": 5000, "AVG": 272.0}, {"from": 5000, "to": 6000, "AVG": 271.0}, {"from": 6000, "to": 7000, "AVG": 271.0}, {"from": 7000, "to": 8000, "AVG": 271.0}, {"from": 8000, "to": 9000, "AVG": 271.0}, {"from": 9000, "to": 10000, "AVG": 271.0}, {"from": 10000, "to": 11000, "AVG": 272.0}]'

        else:
            raise Exception('Endpoint NOT mocked : ' + endpoint)
