# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['bas_metadata_library',
 'bas_metadata_library.standards',
 'bas_metadata_library.standards.iso_19115_1_v1',
 'bas_metadata_library.standards.iso_19115_1_v1.profiles',
 'bas_metadata_library.standards.iso_19115_2_v1',
 'bas_metadata_library.standards.iso_19115_2_v1.profiles',
 'bas_metadata_library.standards.iso_19115_common',
 'bas_metadata_library.standards_schemas',
 'bas_metadata_library.standards_schemas.iso_19115_1_v1',
 'bas_metadata_library.standards_schemas.iso_19115_1_v1.profiles',
 'bas_metadata_library.standards_schemas.iso_19115_1_v1.profiles.inspire_v1_3',
 'bas_metadata_library.standards_schemas.iso_19115_1_v1.profiles.uk_pdc_discovery_v1',
 'bas_metadata_library.standards_schemas.iso_19115_2_v1',
 'bas_metadata_library.standards_schemas.iso_19115_2_v1.profiles',
 'bas_metadata_library.standards_schemas.iso_19115_2_v1.profiles.inspire_v1_3',
 'bas_metadata_library.standards_schemas.iso_19115_2_v1.profiles.uk_pdc_discovery_v1']

package_data = \
{'': ['*']}

install_requires = \
['backports-datetime-fromisoformat>=1.0.0,<2.0.0',
 'importlib-resources>=3.0.0,<4.0.0',
 'jsonschema>=3.2.0,<4.0.0',
 'lxml==4.5.2',
 'rfc3987>=1.3.8,<2.0.0',
 'strict-rfc3339>=0.7,<0.8']

setup_kwargs = {
    'name': 'bas-metadata-library',
    'version': '0.4.0',
    'description': 'Python library for generating metadata records',
    'long_description': '# BAS Metadata Library\n\nPython library for generating metadata records.\n\n## Purpose\n\nThis library is designed to assist in generating metadata records for the discovery of datasets. As a library, this\npackage is intended to be embedded within other tools and services, to avoid the need to implement\nthe complexity and verbosity of specific metadata standards.\n\nThis library is built around the needs of the British Antarctic Survey and NERC Polar Data Centre. This means only\nstandards, and elements of these standards, used by BAS or the UK PDC are supported. Additions that would enable this\nlibrary to be useful to others are welcome as contributions.\n\n### Supported standards\n\n| Standard                                                    | Implementation                                              | Library Namespace                               | Introduced In                                                                                    |\n| ----------------------------------------------------------- | ----------------------------------------------------------- | ----------------------------------------------- | ------------------------------------------------------------------------------------------------ |\n| [ISO 19115:2003](https://www.iso.org/standard/26020.html)   | [ISO 19139:2007](https://www.iso.org/standard/32557.html)   | `bas_metadata_library.standards.iso_19115_1_v1` | [#46](https://gitlab.data.bas.ac.uk/uk-pdc/metadata-infrastructure/metadata-generator/issues/46) |\n| [ISO 19115-2:2009](https://www.iso.org/standard/39229.html) | [ISO 19139-2:2012](https://www.iso.org/standard/57104.html) | `bas_metadata_library.standards.iso_19115_2_v1` | [#50](https://gitlab.data.bas.ac.uk/uk-pdc/metadata-infrastructure/metadata-generator/issues/50) |\n\n**Note:** In this library *ISO 19115:2003* is referred to as *ISO-19115-1* (`iso_19115_1_v1`) for consistency with\n*ISO 19115-2:2009* (referred to as *ISO-19115-2*, `iso_19115_2_v1`). As ISO have subsequently created\n[ISO 19115-1:2014](https://www.iso.org/standard/53798.html) this creates a conflict/ambiguity. To resolve this\nwithout making breaking changes, *ISO 19115-1:2014* will be referred to as *ISO-19115-3* when added to this library.\n\n### Supported profiles\n\n| Standard         | Profile                                    | Implementation                                                               | Library Namespace                                                            | Introduced In                                                                                    |\n| ---------------- | ------------------------------------------ | ---------------------------------------------------------------------------- | ---------------------------------------------------------------------------- | ------------------------------------------------------------------------------------------------ |\n| ISO 19115:2003   | [EU Inspire](https://inspire.ec.europa.eu) | [UK Gemini](https://www.agi.org.uk/agi-groups/standards-committee/uk-gemini) | `bas_metadata_library.standards.iso_19115_1_v1.profiles.inspire_v1_3`        | [#40](https://gitlab.data.bas.ac.uk/uk-pdc/metadata-infrastructure/metadata-generator/issues/40) |\n| ISO 19115:2003   | UK Polar Data Centre Discovery Metadata    | -                                                                            | `bas_metadata_library.standards.iso_19115_1_v1.profiles.uk_pdc_discovery_v1` | [#45](https://gitlab.data.bas.ac.uk/uk-pdc/metadata-infrastructure/metadata-generator/issues/45) |\n| ISO 19115-2:2009 | [EU Inspire](https://inspire.ec.europa.eu) | [UK Gemini](https://www.agi.org.uk/agi-groups/standards-committee/uk-gemini) | `bas_metadata_library.standards.iso_19115_2_v1.profiles.inspire_v1_3`        | [#40](https://gitlab.data.bas.ac.uk/uk-pdc/metadata-infrastructure/metadata-generator/issues/50) |\n| ISO 19115-2:2009 | UK Polar Data Centre Discovery Metadata    | -                                                                            | `bas_metadata_library.standards.iso_19115_2_v1.profiles.uk_pdc_discovery_v1` | [#45](https://gitlab.data.bas.ac.uk/uk-pdc/metadata-infrastructure/metadata-generator/issues/50) |\n\n## Installation\n\nThis package can be installed using Pip from [PyPi](https://pypi.org/project/bas-metadata-library):\n\n```\n$ pip install bas-metadata-library\n```\n\n## Usage\n\nTo generate an ISO 19115 metadata record and return it as an XML document:\n\n```python\nfrom datetime import date\nfrom bas_metadata_library.standards.iso_19115_2_v1 import MetadataRecordConfig, MetadataRecord\n\nminimal_record_config = {\n    "language": "eng",\n    "character_set": "utf-8",\n    "hierarchy_level": "dataset",\n    "contacts": [{"organisation": {"name": "UK Polar Data Centre"}, "role": ["pointOfContact"]}],\n    "date_stamp": date(2018, 10, 18),\n    "resource": {\n        "title": {"value": "Test Record"},\n        "dates": [{"date": date(2018, 1, 1), "date_precision": "year", "date_type": "creation"}],\n        "abstract": "Test Record for ISO 19115 metadata standard (no profile) with required properties only.",\n        "character_set": "utf-8",\n        "language": "eng",\n        "topics": ["environment", "climatologyMeteorologyAtmosphere"],\n        "extent": {\n            "geographic": {\n                "bounding_box": {\n                    "west_longitude": -45.61521,\n                    "east_longitude": -27.04976,\n                    "south_latitude": -68.1511,\n                    "north_latitude": -54.30761,\n                }\n            }\n        },\n    },\n}\nconfiguration = MetadataRecordConfig(**minimal_record_config)\nrecord = MetadataRecord(configuration=configuration)\ndocument = record.generate_xml_document()\n\n# output document\nprint(document)\n```\n\nWhere `metadata_configs.record` is a Python dictionary implementing the BAS metadata generic schema, documented in the\n[BAS Metadata Standards](https://metadata-standards.data.bas.ac.uk) project.\n\nTo reverse this process and convert an XML record into a configuration object:\n\n```python\nfrom bas_metadata_library.standards.iso_19115_2_v1 import MetadataRecord\n\nwith open(f"minimal-record.xml") as record_file:\n    record_data = record_file.read()\n\nrecord = MetadataRecord(record=record_data)\nconfiguration = record.make_config()\nminimal_record_config = configuration.config\n\n# output configuration\nprint(minimal_record_config)\n```\n\n### HTML entities\n\nDo not include HTML entities in input to this generator, as it will be douple escaped by [Lxml](https://lxml.de), the\nunderlying XML processing library.\n\nThis means `&gt;`, the HTML entity for `>`, will be escaped again to `&amp;gt;` which will not be correctly\ninterpreted when decoded. Instead the literal character should be used (e.g. `>`), which Lxml will escape if needed.\n\nThis applies to any unicode character, such as accents (e.g. `å`) and symbols (e.g. `µ`).\n\n## Implementation\n\nThis library consists of a set of base classes using [lxml](https://lxml.de) for generating XML based metadata records\nfrom a configuration object, or generating a configuration object from an XML record.\n\nEach [supported standard](#supported-standards) implements these classes for supported elements as per their respective\nstandard. Two methods are implemented, `make_element()` builds an XML element using values from a configuration object,\n`make_config()` typically uses XPath expressions to build a configuration object from XML. These element classes are\ncombined to generate complete metadata records or configuration objects.\n\nConfiguration objects are python dicts, the properties and values of which are defined by, and validated against, a\n[JSON Schema](https://json-schema.org).\n\nSee the [development](#development) section for more information on the [base classes](#library-base-classes) used\nacross all standards and how to [add a new standard](#adding-a-new-standard).\n\n## Setup\n\n### Terraform\n\nTerraform is used to provision resources required to operate this application in staging and production environments.\n\nThese resources allow [Configuration schemas](#configuration-schemas) for each standard to be accessed externally.\n\nAccess to the [BAS AWS account](https://gitlab.data.bas.ac.uk/WSF/bas-aws) is needed to provisioning these resources.\n\n**Note:** This provisioning should have already been performed (and applies globally). If changes are made to this\nprovisioning it only needs to be applied once.\n\n```shell\n# start terraform inside a docker container\n$ cd provisioning/terraform\n$ docker compose run terraform\n# setup terraform\n$ terraform init\n# apply changes\n$ terraform validate\n$ terraform fmt\n$ terraform apply\n# exit container\n$ exit\n$ docker compose down\n```\n\n#### Terraform remote state\n\nState information for this project is stored remotely using a\n[Backend](https://www.terraform.io/docs/backends/index.html).\n\nSpecifically the [AWS S3](https://www.terraform.io/docs/backends/types/s3.html) backend as part of the\n[BAS Terraform Remote State](https://gitlab.data.bas.ac.uk/WSF/terraform-remote-state) project.\n\nRemote state storage will be automatically initialised when running `terraform init`. Any changes to remote state will\nbe automatically saved to the remote backend, there is no need to push or pull changes.\n\n##### Remote state authentication\n\nPermission to read and/or write remote state information for this project is restricted to authorised users. Contact\nthe [BAS Web & Applications Team](mailto:servicedesk@bas.ac.uk) to request access.\n\nSee the [BAS Terraform Remote State](https://gitlab.data.bas.ac.uk/WSF/terraform-remote-state) project for how these\npermissions to remote state are enforced.\n\n## Development\n\nThis API is developed as a Python library. A bundled Flask application is used to simulate its usage and to act as\nframework for running tests etc.\n\n```shell\n$ git clone https://gitlab.data.bas.ac.uk/uk-pdc/metadata-infrastructure/metadata-generator.git\n$ cd metadata-generator\n```\n\n### Development environment\n\nDocker and Docker Compose are required to setup a local development environment of this application.\n\nIf you have access to the [BAS GitLab instance](https://gitlab.data.bas.ac.uk), you can pull the application Docker\nimage from the BAS Docker Registry. Otherwise you will need to build the Docker image locally.\n\n```shell\n# If you have access to gitlab.data.bas.ac.uk:\n$ docker login docker-registry.data.bas.ac.uk\n$ docker compose pull\n# If you don\'t have access:\n$ docker compose build\n```\n\nTo run the application using the Flask development server (which reloads automatically if source files are changed):\n\n```shell\n$ docker compose up\n```\n\nTo run other commands against the Flask application (such as [Integration tests](#integration-tests)):\n\n```shell\n# in a separate terminal to `docker compose up`\n$ docker compose run app flask [command]\n# E.g.\n$ docker compose run app flask test\n# List all available commands\n$ docker compose run app flask\n```\n\n### Library base classes\n\nThe `bas_metadata_library` module defines a series of modules for each standard (in `bas_metadata_library.standards`)\nas well  as *base* classes used across all standards, that providing common functionality. See existing standards for\nhow these are used.\n\n### Configuration schemas\n\nThis library accepts a \'configuration\' for each metadata record. This contains values for elements, or values that are\nused to compute values. For example, a *title* element would use a value taken directly from the record configuration.\n\nTo ensure all required configuration attributes are included, and where relevant that their values are allowed, this\nconfiguration is validated against a schema. This schema uses the [JSON Schema](https://json-schema.org) standard.\n\nConfiguration schemas are stored as JSON files in `bas_metadata_library.standards_schemas` and loaded as resource files\nwithin this package. Schemas are also made available externally through the BAS Metadata Standards website\n[metadata-standards.data.bas.ac.uk](https://metadata-standards.data.bas.ac.uk) to allow:\n\n1. other applications to ensure their output will be compatible with this library, but that can\'t use this library\n2. to allow schema inheritance/extension where used for standards that inherit from other standards (such as profiles)\n\nJSON Schema\'s can be developed using [jsonschemavalidator.net](https://www.jsonschemavalidator.net).\n\n### Adding a new standard\n\nTo add a new standard:\n\n1. create a new module in `bas_metadata_library.standards/` e.g. `bas_metadata_library.standards.foo_v1/__init__.py`\n2. in this module, overload the `Namespaces`, `MetadataRecordConfig` and `MetadataRecord` classes as needed\n3. create a suitable metadata configuration JSON schema in `bas_metadata_library.standards_schemas/`\n   e.g. `bas_metadata_library.standards_schemas/foo_v1/configuration-schema.json`\n4. add a script line to the `publish-schemas-stage` and `publish-schemas-prod` to copy the configuration schema to the\n   relevant S3 buckets for external access\n5. define a series of test configurations (e.g. minimal, typical and complete) for generating test records in\n   `tests/resources/configs/` e.g. `tests/resources/configs/foo_v1_standard.py`\n6. update the inbuilt Flask application in `app.py` with a route for generating test records for the new standard\n7. use the inbuilt Flask application to generate the test records and save to `tests/resources/records/`\n8. add relevant [tests](#testing) with methods to test each metadata element class and test records\n\n### Code Style\n\nPEP-8 style and formatting guidelines must be used for this project, with the exception of the 80 character line limit.\n\n[Black](https://github.com/psf/black) is used to ensure compliance, configured in `pyproject.toml`.\n\nBlack can be [integrated](https://black.readthedocs.io/en/stable/editor_integration.html#pycharm-intellij-idea) with a\nrange of editors, such as PyCharm, to perform formatting automatically.\n\nTo apply formatting manually:\n\n```shell\n$ docker compose run app black bas_metadata_library/\n```\n\nTo check compliance manually:\n\n```shell\n$ docker compose run app black --check bas_metadata_library/\n```\n\nChecks are ran automatically in [Continuous Integration](#continuous-integration).\n\n### Dependencies\n\nPython dependencies for this project are managed with [Poetry](https://python-poetry.org) in `pyproject.toml`.\n\nNon-code files, such as static files, can also be included in the [Python package](#python-package) using the\n`include` key in `pyproject.toml`.\n\n#### Adding new dependencies\n\nTo add a new (development) dependency:\n\n```shell\n$ docker compose run app ash\n$ poetry add [dependency] (--dev)\n```\n\nThen rebuild the development container, and if you can, push to GitLab:\n\n```shell\n$ docker compose build app\n$ docker compose push app\n```\n\n#### Updating dependencies\n\n```shell\n$ docker compose run app ash\n$ poetry update\n```\n\nThen rebuild the development container, and if you can, push to GitLab:\n\n```shell\n$ docker compose build app\n$ docker compose push app\n```\n\n### Static security scanning\n\nTo ensure the security of this API, source code is checked against [Bandit](https://github.com/PyCQA/bandit) for issues\nsuch as not sanitising user inputs or using weak cryptography.\n\n**Warning:** Bandit is a static analysis tool and can\'t check for issues that are only be detectable when running the\napplication. As with all security tools, Bandit is an aid for spotting common mistakes, not a guarantee of secure code.\n\nThrough [Continuous Integration](#continuous-integration), each commit is tested.\n\nTo check locally:\n\n```shell\n$ docker compose run app bandit -r .\n```\n\n### Editor support\n\n#### PyCharm\n\nA run/debug configuration, *App*, is included in the project.\n\n## Testing\n\nAll code in the `bas_metadata_library` module must be covered by tests, defined in `tests/`. This project uses\n[PyTest](https://docs.pytest.org/en/latest/) which should be ran in a random order using\n[pytest-random-order](https://pypi.org/project/pytest-random-order/).\n\nTests are written to create metadata records based on a series of configurations defined in `tests/config.py`. These\ndefine \'minimal\' to \'complete\' test records, intended to test different ways a standard can be used, both for\nindividual elements and whole records. These tests are designed to ensure that records are generally well-formed and\nthat where config options are used the corresponding elements in the metadata record are generated.\n\nAs this library does not seek to support all possible elements and variations within each standard, these tests are\nsimilarly not exhaustive, nor are they a substitute for formal metadata validation.\n\nTest methods are used to test individual elements are formed correctly. Comparisons against static records are used to\ntest the structure of whole records.\n\nTo run tests manually from the command line:\n\n```shell\n$ docker compose run app pytest --random-order\n```\n\nTo run tests manually using PyCharm, use the included *App (Tests)* run/debug configuration.\n\nTests are ran automatically in [Continuous Integration](#continuous-integration).\n\n### Capturing static test records\n\nTo capture static test records, which verify complete records are assembled correctly, a custom Flask CLI command,\n`capture-test-records` is available. This requires the Flask application to first be running. The Requests library is\nused to make requests against the Flask app save responses to a relevant directory in `tests/resources/records`.\n\n```shell\n# start Flask application:\n$ docker compose up\n# then in a separate terminal:\n$ docker compose run app flask capture-test-records\n```\n\nIt is intended that this command will update pre-existing static records, with differences captured in version control\nand reviewed manually to ensure they are correct.\n\n### Test coverage\n\n[pytest-cov](https://pypi.org/project/pytest-cov/) is used to measure test coverage.\n\nTo prevent noise, `.coveragerc` is used to omit empty `__init__.py` files from reports.\n\nTo measure coverage manually:\n\n```shell\n$ docker compose run app pytest --random-order --cov=bas_metadata_library --cov-fail-under=100 --cov-report=html .\n```\n\n[Continuous Integration](#continuous-integration) will check coverage automatically and fail if less than 100%.\n\n### Continuous Integration\n\nAll commits will trigger a Continuous Integration process using GitLab\'s CI/CD platform, configured in `.gitlab-ci.yml`.\n\n## Deployment\n\n### Python package\n\nThis project is distributed as a Python package, hosted in [PyPi](https://pypi.org/project/bas-metadata-library).\n\nSource and binary packages are built and published automatically using\n[Poetry](https://python-poetry.org/docs/cli/#publish) in [Continuous Delivery](#continuous-deployment).\n\nPackage versions are determined automatically using the `support/python-packaging/parse_version.py` script.\n\n### Continuous Deployment\n\nA Continuous Deployment process using GitLab\'s CI/CD platform is configured in `.gitlab-ci.yml`.\n\n## Release procedure\n\nFor all releases:\n\n1. create a release branch\n2. close release in `CHANGELOG.md`\n3. push changes, merge the release branch into `master` and tag with version\n\n## Feedback\n\nThe maintainer of this project is the BAS Web & Applications Team, they can be contacted at:\n[servicedesk@bas.ac.uk](mailto:servicedesk@bas.ac.uk).\n\n## Issue tracking\n\nThis project uses issue tracking, see the\n[Issue tracker](https://gitlab.data.bas.ac.uk/uk-pdc/metadata-infrastructure/metadata-generator/issues) for more\ninformation.\n\n**Note:** Read & write access to this issue tracker is restricted. Contact the project maintainer to request access.\n\n## License\n\n© UK Research and Innovation (UKRI), 2019 - 2021, British Antarctic Survey.\n\nYou may use and re-use this software and associated documentation files free of charge in any format or medium, under\nthe terms of the Open Government Licence v3.0.\n\nYou may obtain a copy of the Open Government Licence at http://www.nationalarchives.gov.uk/doc/open-government-licence/\n',
    'author': 'Felix Fennell',
    'author_email': 'felnne@bas.ac.uk',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/antarctica/metadata-library',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
