# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['seqal', 'seqal.samplers', 'seqal.stoppers']

package_data = \
{'': ['*']}

install_requires = \
['flair==0.10', 'spacy>=3.4.1,<4.0.0', 'torch==1.9.1']

setup_kwargs = {
    'name': 'seqal',
    'version': '0.3.1',
    'description': 'Sequence labeling active learning framework for Python',
    'long_description': '# SeqAL\n\n<!-- <p align="center">\n  <a href="https://github.com/BrambleXu/seqal/actions?query=workflow%3ACI">\n    <img src="https://img.shields.io/github/workflow/status/BrambleXu/seqal/CI/main?label=CI&logo=github&style=flat-square" alt="CI Status" >\n  </a>\n  <a href="https://seqal.readthedocs.io">\n    <img src="https://img.shields.io/readthedocs/seqal.svg?logo=read-the-docs&logoColor=fff&style=flat-square" alt="Documentation Status">\n  </a>\n  <a href="https://codecov.io/gh/BrambleXu/seqal">\n    <img src="https://img.shields.io/codecov/c/github/BrambleXu/seqal.svg?logo=codecov&logoColor=fff&style=flat-square" alt="Test coverage percentage">\n  </a>\n</p> -->\n<p align="center">\n  <a href="https://github.com/BrambleXu/seqal/actions?query=workflow%3ACI">\n    <img src="https://img.shields.io/github/workflow/status/BrambleXu/seqal/CI/main?label=CI&logo=github&style=flat-square" alt="CI Status" >\n  </a>\n  <a href="https://python-poetry.org/">\n    <img src="https://img.shields.io/badge/packaging-poetry-299bd7?style=flat-square&logo=data:image/png" alt="Poetry">\n  </a>\n  <a href="https://github.com/ambv/black">\n    <img src="https://img.shields.io/badge/code%20style-black-000000.svg?style=flat-square" alt="black">\n  </a>\n  <a href="https://github.com/pre-commit/pre-commit">\n    <img src="https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white&style=flat-square" alt="pre-commit">\n  </a>\n</p>\n<p align="center">\n  <a href="https://pypi.org/project/seqal/">\n    <img src="https://img.shields.io/pypi/v/seqal.svg?logo=python&logoColor=fff&style=flat-square" alt="PyPI Version">\n  </a>\n  <img src="https://img.shields.io/pypi/pyversions/seqal.svg?style=flat-square&logo=python&amp;logoColor=fff" alt="Supported Python versions">\n  <img src="https://img.shields.io/pypi/l/seqal.svg?style=flat-square" alt="License">\n</p>\n\nSeqAL is a sequence labeling active learning framework based on Flair.\n\n## Installation\n\nInstall this via pip (or your favourite package manager):\n\n`pip install seqal`\n\n\n## Usage\n\nTo understand what SeqAL can do, we first introduce the pool-based active learning cycle.\n\n![al_cycle](./docs/images/al_cycle.png)\n\n- Step 0: Prepare seed data (a small number of labeled data used for training)\n- Step 1: Train the model with seed data\n  - Step 2: Predict unlabeled data with the trained model\n  - Step 3: Query informative samples based on predictions\n  - Step 4: Annotator (Oracle) annotate the selected samples\n  - Step 5: Input the new labeled samples to labeled dataset\n  - Step 6: Retrain model\n- Repeat step2~step6 until the f1 score of the model beyond the threshold or annotation budget is no left\n\nSeqAL can cover all steps except step 0 and step 4. Below is a simple script to demonstrate how to use SeqAL to implement the work flow.\n\n```python\nfrom seqal.active_learner import ActiveLearner\nfrom seqal.samplers import LeastConfidenceSampler\nfrom seqal.alinger import Alinger\nfrom seqal.datasets import ColumnCorpus\nfrom seqal.utils import load_plain_text\nfrom xxxx import annotate_by_human  # User need to prepare this method\n\n\n# Step 0: Preparation\n## Prepare Seed data, valid data, and test data\ncolumns = {0: "text", 1: "pos", 2: "syntactic_chunk", 3: "ner"}\ndata_folder = "./datasets/conll"\ncorpus = ColumnCorpus(\n    data_folder,\n    columns,\n    train_file="train_seed.txt",\n    dev_file="valid.txt",\n    test_file="test.txt",\n)\n\n## Unlabeled data pool\nfile_path = "./datasets/conll/train_datapool.txt"\nunlabeled_sentences = load_plain_text(file_path)\n\n## Initilize ActiveLearner\nlearner = ActiveLearner(\n  tagger_params=tagger_params,   # Model parameters (hidden size, embedding, etc.)\n  query_strategy=LeastConfidenceSampler(),  # Query algorithm\n  corpus=corpus,                 # Corpus contains training, validation, test data\n  trainer_params=trainer_params  # Trainer parameters (epoch, batch size, etc.)\n)\n\n# Step 1: Initial training on model\nlearner.initialize()\n\n# Step 2&3: Predict on unlabeled data and query informative data\n_, queried_samples = learner.query(data_pool)\nqueried_samples = [{"text": sent.to_plain_string()} for sent in queried_samples]  # Convert sentence class to plain text\n# queried_samples:\n# [\n#   {\n#     "text": "Tokyo is a city"\n#   }\n# ]\n\n# Step 4: Annotator annotate the selected samples\nnew_labels = annotate_by_human(queried_samples)\n# new_labels:\n# [\n#   {\n#     "text": [\'Tokyo\', \'is\', \'a\', \'city\'],\n#     "labels": [\'B-LOC\', \'O\', \'O\', \'O\']\n#   }\n# ]\n\n## Convert data to the suitable format\nalinger = Alinger()\nnew_labeled_samples = alinger.add_tags_on_token(new_labels, \'ner\')\n\n# Step 5&6: Add new labeled samples to training and retrain model\nlearner.teach(new_labeled_samples)\n```\n\n## Tutorials\n\nWe provide a set of quick tutorials to get you started with the library.\n\n- [Tutorial 1: Introduction](docs/TUTORIAL_1_Introduction.md)\n- [Tutorial 2: Prepare Corpus](docs/TUTORIAL_2_Prepare_Corpus.md)\n- [Tutorial 3: Active Learner Setup](docs/TUTORIAL_3_Active_Learner_Setup.md)\n- [Tutorial 4: Prepare Data Pool](docs/TUTORIAL_4_Prepare_Data_Pool.md)\n- [Tutorial 5: Research and Annotation Mode](docs/TUTORIAL_5_Research_and_Annotation_Mode.md)\n- [Tutorial 6: Query Setup](docs/TUTORIAL_6_Query_Setup.md)\n- [Tutorial 7: Annotated Data](docs/TUTORIAL_7_Annotated_Data.md)\n- [Tutorial 8: Stopper](docs/TUTORIAL_8_Stopper.md)\n- [Tutorial 9: Ouput Labeled Data](docs/TUTORIAL_9_Ouput_Labeled_Data.md)\n- [Tutorial 10: Performance Recorder](docs/TUTORIAL_10_Performance_Recorder.md)\n- [Tutorial 11: Multiple Language Support](docs/TUTORIAL_11_Multiple_Language_Support.md)\n\n## Performance\n\nActive learning algorithms achieve 97% performance of the best deep model trained on full data using only 30% of the training data on the CoNLL 2003 English dataset. The CPU model can decrease the time cost greatly only sacrificing a little performance.\n\nSee [performance](./docs/performance.md) for more detail about performance and time cost.\n\n\n## Construct envirement locally\n\nIf you want to make a PR or implement something locally, you can follow bellow instruction to construct the development envirement locally.\n\nFirst we create a environment "seqal" based on the `environment.yml` file.\n\nWe use conda as envirement management tool, so install it first.\n\n```\nconda env create -f environment.yml\n```\n\nThen we activate the environment.\n\n```\nconda activate seqal\n```\n\nInstall poetry for dependency management.\n\n```\ncurl -sSL https://raw.githubusercontent.com/python-poetry/poetry/master/get-poetry.py | python -\n```\n\nAdd poetry path in your shell configure file (`bashrc`, `zshrc`, etc.)\n```\nexport PATH="$HOME/.poetry/bin:$PATH"\n```\n\nInstalling dependencies from `pyproject.toml`.\n\n```\npoetry install\n```\n\nYou can make development locally now.\n\nIf you want to delete the local envirement, run below command.\n```\nconda remove --name seqal --all\n```\n\n## Credits\n\n- [Cookiecutter](https://github.com/audreyr/cookiecutter)\n- [browniebroke/cookiecutter-pypackage](https://github.com/browniebroke/cookiecutter-pypackage)\n- [flairNLP/flair](https://github.com/flairNLP/flair)\n- [modal](https://github.com/modAL-python/modAL)\n',
    'author': 'Xu Liang',
    'author_email': 'liangxu006@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/BrambleXu/seqal',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
