"use strict";
/*
 * IMPORTANT: This file is used both by bundled lambda functions and by
 * constructs. This means it should not directly or transitively import anything
 * not part of the package's `dependencies`, such as the `aws-sdk`.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ELIGIBLE_LICENSES = exports.STORAGE_KEY_FORMAT_REGEX = exports.CATALOG_KEY = exports.NOT_SUPPORTED_SUFFIX = exports.docsKeySuffix = exports.getObjectKeys = exports.getObjectKeyPrefix = exports.DOCS_KEY_SUFFIX_ANY = exports.DOCS_KEY_SUFFIX_PYTHON = exports.DOCS_KEY_SUFFIX_TYPESCRIPT = exports.ASSEMBLY_KEY_SUFFIX = exports.METADATA_KEY_SUFFIX = exports.PACKAGE_KEY_SUFFIX = exports.STORAGE_KEY_PREFIX = void 0;
const language_1 = require("./language");
/**
 * Key prefix for the package data storage.
 */
exports.STORAGE_KEY_PREFIX = 'data/';
/**
 * Key suffix for storing npm package bundles.
 */
exports.PACKAGE_KEY_SUFFIX = '/package.tgz';
/**
 * Key suffix for storing npm package metadata.
 */
exports.METADATA_KEY_SUFFIX = '/metadata.json';
/**
 * The key suffix for (TypeScript) assembly files
 */
exports.ASSEMBLY_KEY_SUFFIX = '/assembly.json';
/**
 * The key suffix for a TypeScript doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_TYPESCRIPT = docsKeySuffix(language_1.DocumentationLanguage.TYPESCRIPT);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_PYTHON = docsKeySuffix(language_1.DocumentationLanguage.PYTHON);
/**
 * The key suffix matching any documentation artifact.
 */
exports.DOCS_KEY_SUFFIX_ANY = docsKeySuffix('*');
/**
 * Return the S3 object key prefix for a specific package name and optionally a
 * version. Note that the prefix does not end with a "/" so you will likely want
 * to add that if you want to match a specific set of objects.
 */
function getObjectKeyPrefix(packageName, packageVersion) {
    let key = `${exports.STORAGE_KEY_PREFIX}${packageName}`;
    if (packageVersion) {
        key += `/v${packageVersion}`;
    }
    return key;
}
exports.getObjectKeyPrefix = getObjectKeyPrefix;
/**
 * Resolves the set of S3 object keys use for a specific package/version.
 */
function getObjectKeys(packageName, packageVersion) {
    const prefix = getObjectKeyPrefix(packageName, packageVersion);
    return {
        assemblyKey: `${prefix}${exports.ASSEMBLY_KEY_SUFFIX}`,
        packageKey: `${prefix}${exports.PACKAGE_KEY_SUFFIX}`,
        metadataKey: `${prefix}${exports.METADATA_KEY_SUFFIX}`,
    };
}
exports.getObjectKeys = getObjectKeys;
/**
 * The key suffix for documentation artifacts by language and submodule.
 */
function docsKeySuffix(lang, submodule) {
    return `/docs-${submodule ? `${submodule}-` : ''}${lang}.md`;
}
exports.docsKeySuffix = docsKeySuffix;
/**
 * Key suffix for beacon files when a particular feature is not supported for
 * the particular package (i.e: Python docs for a package that does not have a
 * Python target configured).
 */
exports.NOT_SUPPORTED_SUFFIX = '.not-supported';
/**
 * The key for the catalog document.
 */
exports.CATALOG_KEY = 'catalog.json';
/**
 * A regular expression that can be used to parse out a storage key.
 */
exports.STORAGE_KEY_FORMAT_REGEX = new RegExp(`^${exports.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`);
// Capture groups:                                                        ┗━━━━━━━━1━━━━━━━━┛  ┗━━2━━┛
/**
 * A list of case-normalized (upper) SPDX license identifiers that are deemed
 * eligible for listing on the Construct Hub.
 *
 * @see https://github.com/cdklabs/construct-hub/issues/145
 */
exports.ELIGIBLE_LICENSES = new Set([
    // Apache Licenses
    'Apache-1.0',
    'Apache-1.1',
    'Apache-2.0',
    // BSD Licenses
    '0BSD',
    'BSD-1-Clause',
    'BSD-2-Clause',
    'BSD-2-Clause-Patent',
    'BSD-2-Clause-Views',
    'BSD-3-Clause',
    'BSD-3-Clause-Attribution',
    'BSD-3-Clause-Clear',
    'BSD-3-Clause-LBNL',
    'BSD-3-Clause-Modification',
    'BSD-3-Clause-No-Military-License',
    'BSD-3-Clause-No-Nuclear-License',
    'BSD-3-Clause-No-Nuclear-License-2014',
    'BSD-3-Clause-No-Nuclear-Warranty',
    'BSD-3-Clause-Open-MPI',
    'BSD-4-Clause',
    'BSD-4-Clause-Shortened',
    'BSD-4-Clause-UC',
    'BSD-Protection',
    'BSD-Source-Code',
    // MIT Licenses
    'MIT',
    'MIT-0',
    'MIT-advertising',
    'MIT-CMU',
    'MIT-enna',
    'MIT-feh',
    'MIT-Modern-Variant',
    'MIT-open-group',
    'MITNFA',
].map((s) => s.toUpperCase()));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29uc3RhbnRzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL2JhY2tlbmQvc2hhcmVkL2NvbnN0YW50cy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7R0FJRzs7O0FBRUgseUNBQW1EO0FBRW5EOztHQUVHO0FBQ1UsUUFBQSxrQkFBa0IsR0FBRyxPQUFPLENBQUM7QUFFMUM7O0dBRUc7QUFDVSxRQUFBLGtCQUFrQixHQUFHLGNBQWMsQ0FBQztBQUVqRDs7R0FFRztBQUNVLFFBQUEsbUJBQW1CLEdBQUcsZ0JBQWdCLENBQUM7QUFFcEQ7O0dBRUc7QUFDVSxRQUFBLG1CQUFtQixHQUFHLGdCQUFnQixDQUFDO0FBRXBEOztHQUVHO0FBQ1UsUUFBQSwwQkFBMEIsR0FBRyxhQUFhLENBQUMsZ0NBQXFCLENBQUMsVUFBVSxDQUFDLENBQUM7QUFFMUY7O0dBRUc7QUFDVSxRQUFBLHNCQUFzQixHQUFHLGFBQWEsQ0FBQyxnQ0FBcUIsQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUVsRjs7R0FFRztBQUNVLFFBQUEsbUJBQW1CLEdBQUcsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0FBRXREOzs7O0dBSUc7QUFDSCxTQUFnQixrQkFBa0IsQ0FBQyxXQUFtQixFQUFFLGNBQXVCO0lBQzdFLElBQUksR0FBRyxHQUFHLEdBQUcsMEJBQWtCLEdBQUcsV0FBVyxFQUFFLENBQUM7SUFDaEQsSUFBSSxjQUFjLEVBQUU7UUFDbEIsR0FBRyxJQUFJLEtBQUssY0FBYyxFQUFFLENBQUM7S0FDOUI7SUFFRCxPQUFPLEdBQUcsQ0FBQztBQUNiLENBQUM7QUFQRCxnREFPQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsYUFBYSxDQUFDLFdBQW1CLEVBQUUsY0FBc0I7SUFDdkUsTUFBTSxNQUFNLEdBQUcsa0JBQWtCLENBQUMsV0FBVyxFQUFFLGNBQWMsQ0FBQyxDQUFDO0lBQy9ELE9BQU87UUFDTCxXQUFXLEVBQUUsR0FBRyxNQUFNLEdBQUcsMkJBQW1CLEVBQUU7UUFDOUMsVUFBVSxFQUFFLEdBQUcsTUFBTSxHQUFHLDBCQUFrQixFQUFFO1FBQzVDLFdBQVcsRUFBRSxHQUFHLE1BQU0sR0FBRywyQkFBbUIsRUFBRTtLQUMvQyxDQUFDO0FBQ0osQ0FBQztBQVBELHNDQU9DO0FBRUQ7O0dBRUc7QUFDSCxTQUFnQixhQUFhLENBQUMsSUFBa0MsRUFBRSxTQUFrQjtJQUNsRixPQUFPLFNBQVMsU0FBUyxDQUFDLENBQUMsQ0FBQyxHQUFHLFNBQVMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEdBQUcsSUFBSSxLQUFLLENBQUM7QUFDL0QsQ0FBQztBQUZELHNDQUVDO0FBRUQ7Ozs7R0FJRztBQUNVLFFBQUEsb0JBQW9CLEdBQUcsZ0JBQWdCLENBQUM7QUFFckQ7O0dBRUc7QUFDVSxRQUFBLFdBQVcsR0FBRyxjQUFjLENBQUM7QUFFMUM7O0dBRUc7QUFDVSxRQUFBLHdCQUF3QixHQUFHLElBQUksTUFBTSxDQUFDLElBQUksMEJBQWtCLGtDQUFrQyxDQUFDLENBQUM7QUFDN0csc0dBQXNHO0FBRXRHOzs7OztHQUtHO0FBQ1UsUUFBQSxpQkFBaUIsR0FBd0IsSUFBSSxHQUFHLENBQUM7SUFDNUQsa0JBQWtCO0lBQ2xCLFlBQVk7SUFDWixZQUFZO0lBQ1osWUFBWTtJQUNaLGVBQWU7SUFDZixNQUFNO0lBQ04sY0FBYztJQUNkLGNBQWM7SUFDZCxxQkFBcUI7SUFDckIsb0JBQW9CO0lBQ3BCLGNBQWM7SUFDZCwwQkFBMEI7SUFDMUIsb0JBQW9CO0lBQ3BCLG1CQUFtQjtJQUNuQiwyQkFBMkI7SUFDM0Isa0NBQWtDO0lBQ2xDLGlDQUFpQztJQUNqQyxzQ0FBc0M7SUFDdEMsa0NBQWtDO0lBQ2xDLHVCQUF1QjtJQUN2QixjQUFjO0lBQ2Qsd0JBQXdCO0lBQ3hCLGlCQUFpQjtJQUNqQixnQkFBZ0I7SUFDaEIsaUJBQWlCO0lBQ2pCLGVBQWU7SUFDZixLQUFLO0lBQ0wsT0FBTztJQUNQLGlCQUFpQjtJQUNqQixTQUFTO0lBQ1QsVUFBVTtJQUNWLFNBQVM7SUFDVCxvQkFBb0I7SUFDcEIsZ0JBQWdCO0lBQ2hCLFFBQVE7Q0FDVCxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qXG4gKiBJTVBPUlRBTlQ6IFRoaXMgZmlsZSBpcyB1c2VkIGJvdGggYnkgYnVuZGxlZCBsYW1iZGEgZnVuY3Rpb25zIGFuZCBieVxuICogY29uc3RydWN0cy4gVGhpcyBtZWFucyBpdCBzaG91bGQgbm90IGRpcmVjdGx5IG9yIHRyYW5zaXRpdmVseSBpbXBvcnQgYW55dGhpbmdcbiAqIG5vdCBwYXJ0IG9mIHRoZSBwYWNrYWdlJ3MgYGRlcGVuZGVuY2llc2AsIHN1Y2ggYXMgdGhlIGBhd3Mtc2RrYC5cbiAqL1xuXG5pbXBvcnQgeyBEb2N1bWVudGF0aW9uTGFuZ3VhZ2UgfSBmcm9tICcuL2xhbmd1YWdlJztcblxuLyoqXG4gKiBLZXkgcHJlZml4IGZvciB0aGUgcGFja2FnZSBkYXRhIHN0b3JhZ2UuXG4gKi9cbmV4cG9ydCBjb25zdCBTVE9SQUdFX0tFWV9QUkVGSVggPSAnZGF0YS8nO1xuXG4vKipcbiAqIEtleSBzdWZmaXggZm9yIHN0b3JpbmcgbnBtIHBhY2thZ2UgYnVuZGxlcy5cbiAqL1xuZXhwb3J0IGNvbnN0IFBBQ0tBR0VfS0VZX1NVRkZJWCA9ICcvcGFja2FnZS50Z3onO1xuXG4vKipcbiAqIEtleSBzdWZmaXggZm9yIHN0b3JpbmcgbnBtIHBhY2thZ2UgbWV0YWRhdGEuXG4gKi9cbmV4cG9ydCBjb25zdCBNRVRBREFUQV9LRVlfU1VGRklYID0gJy9tZXRhZGF0YS5qc29uJztcblxuLyoqXG4gKiBUaGUga2V5IHN1ZmZpeCBmb3IgKFR5cGVTY3JpcHQpIGFzc2VtYmx5IGZpbGVzXG4gKi9cbmV4cG9ydCBjb25zdCBBU1NFTUJMWV9LRVlfU1VGRklYID0gJy9hc3NlbWJseS5qc29uJztcblxuLyoqXG4gKiBUaGUga2V5IHN1ZmZpeCBmb3IgYSBUeXBlU2NyaXB0IGRvYyBhcnRpZmFjdCAocm9vdCBtb2R1bGUpLlxuICovXG5leHBvcnQgY29uc3QgRE9DU19LRVlfU1VGRklYX1RZUEVTQ1JJUFQgPSBkb2NzS2V5U3VmZml4KERvY3VtZW50YXRpb25MYW5ndWFnZS5UWVBFU0NSSVBUKTtcblxuLyoqXG4gKiBUaGUga2V5IHN1ZmZpeCBmb3IgYSBQeXRob24gZG9jIGFydGlmYWN0IChyb290IG1vZHVsZSkuXG4gKi9cbmV4cG9ydCBjb25zdCBET0NTX0tFWV9TVUZGSVhfUFlUSE9OID0gZG9jc0tleVN1ZmZpeChEb2N1bWVudGF0aW9uTGFuZ3VhZ2UuUFlUSE9OKTtcblxuLyoqXG4gKiBUaGUga2V5IHN1ZmZpeCBtYXRjaGluZyBhbnkgZG9jdW1lbnRhdGlvbiBhcnRpZmFjdC5cbiAqL1xuZXhwb3J0IGNvbnN0IERPQ1NfS0VZX1NVRkZJWF9BTlkgPSBkb2NzS2V5U3VmZml4KCcqJyk7XG5cbi8qKlxuICogUmV0dXJuIHRoZSBTMyBvYmplY3Qga2V5IHByZWZpeCBmb3IgYSBzcGVjaWZpYyBwYWNrYWdlIG5hbWUgYW5kIG9wdGlvbmFsbHkgYVxuICogdmVyc2lvbi4gTm90ZSB0aGF0IHRoZSBwcmVmaXggZG9lcyBub3QgZW5kIHdpdGggYSBcIi9cIiBzbyB5b3Ugd2lsbCBsaWtlbHkgd2FudFxuICogdG8gYWRkIHRoYXQgaWYgeW91IHdhbnQgdG8gbWF0Y2ggYSBzcGVjaWZpYyBzZXQgb2Ygb2JqZWN0cy5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldE9iamVjdEtleVByZWZpeChwYWNrYWdlTmFtZTogc3RyaW5nLCBwYWNrYWdlVmVyc2lvbj86IHN0cmluZykge1xuICBsZXQga2V5ID0gYCR7U1RPUkFHRV9LRVlfUFJFRklYfSR7cGFja2FnZU5hbWV9YDtcbiAgaWYgKHBhY2thZ2VWZXJzaW9uKSB7XG4gICAga2V5ICs9IGAvdiR7cGFja2FnZVZlcnNpb259YDtcbiAgfVxuXG4gIHJldHVybiBrZXk7XG59XG5cbi8qKlxuICogUmVzb2x2ZXMgdGhlIHNldCBvZiBTMyBvYmplY3Qga2V5cyB1c2UgZm9yIGEgc3BlY2lmaWMgcGFja2FnZS92ZXJzaW9uLlxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0T2JqZWN0S2V5cyhwYWNrYWdlTmFtZTogc3RyaW5nLCBwYWNrYWdlVmVyc2lvbjogc3RyaW5nKSB7XG4gIGNvbnN0IHByZWZpeCA9IGdldE9iamVjdEtleVByZWZpeChwYWNrYWdlTmFtZSwgcGFja2FnZVZlcnNpb24pO1xuICByZXR1cm4ge1xuICAgIGFzc2VtYmx5S2V5OiBgJHtwcmVmaXh9JHtBU1NFTUJMWV9LRVlfU1VGRklYfWAsXG4gICAgcGFja2FnZUtleTogYCR7cHJlZml4fSR7UEFDS0FHRV9LRVlfU1VGRklYfWAsXG4gICAgbWV0YWRhdGFLZXk6IGAke3ByZWZpeH0ke01FVEFEQVRBX0tFWV9TVUZGSVh9YCxcbiAgfTtcbn1cblxuLyoqXG4gKiBUaGUga2V5IHN1ZmZpeCBmb3IgZG9jdW1lbnRhdGlvbiBhcnRpZmFjdHMgYnkgbGFuZ3VhZ2UgYW5kIHN1Ym1vZHVsZS5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGRvY3NLZXlTdWZmaXgobGFuZz86IERvY3VtZW50YXRpb25MYW5ndWFnZSB8ICcqJywgc3VibW9kdWxlPzogc3RyaW5nKSB7XG4gIHJldHVybiBgL2RvY3MtJHtzdWJtb2R1bGUgPyBgJHtzdWJtb2R1bGV9LWAgOiAnJ30ke2xhbmd9Lm1kYDtcbn1cblxuLyoqXG4gKiBLZXkgc3VmZml4IGZvciBiZWFjb24gZmlsZXMgd2hlbiBhIHBhcnRpY3VsYXIgZmVhdHVyZSBpcyBub3Qgc3VwcG9ydGVkIGZvclxuICogdGhlIHBhcnRpY3VsYXIgcGFja2FnZSAoaS5lOiBQeXRob24gZG9jcyBmb3IgYSBwYWNrYWdlIHRoYXQgZG9lcyBub3QgaGF2ZSBhXG4gKiBQeXRob24gdGFyZ2V0IGNvbmZpZ3VyZWQpLlxuICovXG5leHBvcnQgY29uc3QgTk9UX1NVUFBPUlRFRF9TVUZGSVggPSAnLm5vdC1zdXBwb3J0ZWQnO1xuXG4vKipcbiAqIFRoZSBrZXkgZm9yIHRoZSBjYXRhbG9nIGRvY3VtZW50LlxuICovXG5leHBvcnQgY29uc3QgQ0FUQUxPR19LRVkgPSAnY2F0YWxvZy5qc29uJztcblxuLyoqXG4gKiBBIHJlZ3VsYXIgZXhwcmVzc2lvbiB0aGF0IGNhbiBiZSB1c2VkIHRvIHBhcnNlIG91dCBhIHN0b3JhZ2Uga2V5LlxuICovXG5leHBvcnQgY29uc3QgU1RPUkFHRV9LRVlfRk9STUFUX1JFR0VYID0gbmV3IFJlZ0V4cChgXiR7U1RPUkFHRV9LRVlfUFJFRklYfSgoPzpAW14vXSsvKT9bXi9dKykvdihbXi9dKykvLiokYCk7XG4vLyBDYXB0dXJlIGdyb3VwczogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIOKUl+KUgeKUgeKUgeKUgeKUgeKUgeKUgeKUgTHilIHilIHilIHilIHilIHilIHilIHilIHilJsgIOKUl+KUgeKUgTLilIHilIHilJtcblxuLyoqXG4gKiBBIGxpc3Qgb2YgY2FzZS1ub3JtYWxpemVkICh1cHBlcikgU1BEWCBsaWNlbnNlIGlkZW50aWZpZXJzIHRoYXQgYXJlIGRlZW1lZFxuICogZWxpZ2libGUgZm9yIGxpc3Rpbmcgb24gdGhlIENvbnN0cnVjdCBIdWIuXG4gKlxuICogQHNlZSBodHRwczovL2dpdGh1Yi5jb20vY2RrbGFicy9jb25zdHJ1Y3QtaHViL2lzc3Vlcy8xNDVcbiAqL1xuZXhwb3J0IGNvbnN0IEVMSUdJQkxFX0xJQ0VOU0VTOiBSZWFkb25seVNldDxzdHJpbmc+ID0gbmV3IFNldChbXG4gIC8vIEFwYWNoZSBMaWNlbnNlc1xuICAnQXBhY2hlLTEuMCcsXG4gICdBcGFjaGUtMS4xJyxcbiAgJ0FwYWNoZS0yLjAnLFxuICAvLyBCU0QgTGljZW5zZXNcbiAgJzBCU0QnLFxuICAnQlNELTEtQ2xhdXNlJyxcbiAgJ0JTRC0yLUNsYXVzZScsXG4gICdCU0QtMi1DbGF1c2UtUGF0ZW50JyxcbiAgJ0JTRC0yLUNsYXVzZS1WaWV3cycsXG4gICdCU0QtMy1DbGF1c2UnLFxuICAnQlNELTMtQ2xhdXNlLUF0dHJpYnV0aW9uJyxcbiAgJ0JTRC0zLUNsYXVzZS1DbGVhcicsXG4gICdCU0QtMy1DbGF1c2UtTEJOTCcsXG4gICdCU0QtMy1DbGF1c2UtTW9kaWZpY2F0aW9uJyxcbiAgJ0JTRC0zLUNsYXVzZS1Oby1NaWxpdGFyeS1MaWNlbnNlJyxcbiAgJ0JTRC0zLUNsYXVzZS1Oby1OdWNsZWFyLUxpY2Vuc2UnLFxuICAnQlNELTMtQ2xhdXNlLU5vLU51Y2xlYXItTGljZW5zZS0yMDE0JyxcbiAgJ0JTRC0zLUNsYXVzZS1Oby1OdWNsZWFyLVdhcnJhbnR5JyxcbiAgJ0JTRC0zLUNsYXVzZS1PcGVuLU1QSScsXG4gICdCU0QtNC1DbGF1c2UnLFxuICAnQlNELTQtQ2xhdXNlLVNob3J0ZW5lZCcsXG4gICdCU0QtNC1DbGF1c2UtVUMnLFxuICAnQlNELVByb3RlY3Rpb24nLFxuICAnQlNELVNvdXJjZS1Db2RlJyxcbiAgLy8gTUlUIExpY2Vuc2VzXG4gICdNSVQnLFxuICAnTUlULTAnLFxuICAnTUlULWFkdmVydGlzaW5nJyxcbiAgJ01JVC1DTVUnLFxuICAnTUlULWVubmEnLFxuICAnTUlULWZlaCcsXG4gICdNSVQtTW9kZXJuLVZhcmlhbnQnLFxuICAnTUlULW9wZW4tZ3JvdXAnLFxuICAnTUlUTkZBJyxcbl0ubWFwKChzKSA9PiBzLnRvVXBwZXJDYXNlKCkpKTtcbiJdfQ==