"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphqlApi = exports.IamResource = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const appsync_generated_1 = require("./appsync.generated");
const graphqlapi_base_1 = require("./graphqlapi-base");
const schema_1 = require("./schema");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
    /**
     * Lambda authorization type
     */
    AuthorizationType["LAMBDA"] = "AWS_LAMBDA";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * A class used to generate resource arns for AppSync
 */
class IamResource {
    constructor(arns) {
        this.arns = arns;
    }
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns) {
        if (arns.length === 0) {
            throw new Error('At least 1 custom ARN must be provided.');
        }
        return new IamResource(arns);
    }
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type, ...fields) {
        const arns = fields.length ? fields.map((field) => `types/${type}/fields/${field}`) : [`types/${type}/*`];
        return new IamResource(arns);
    }
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all() {
        return new IamResource(['*']);
    }
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api) {
        jsiiDeprecationWarnings._aws_cdk_aws_appsync_GraphqlApi(api);
        return this.arns.map((arn) => core_1.Stack.of(api).formatArn({
            service: 'appsync',
            resource: `apis/${api.apiId}`,
            arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
            resourceName: `${arn}`,
        }));
    }
}
exports.IamResource = IamResource;
_a = JSII_RTTI_SYMBOL_1;
IamResource[_a] = { fqn: "@aws-cdk/aws-appsync.IamResource", version: "1.146.0" };
/**
 * An AppSync GraphQL API
 *
 * @resource AWS::AppSync::GraphQLApi
 */
class GraphqlApi extends graphqlapi_base_1.GraphqlApiBase {
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_appsync_GraphqlApiProps(props);
        const defaultMode = (_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) !== null && _d !== void 0 ? _d : { authorizationType: AuthorizationType.API_KEY };
        const additionalModes = (_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.additionalAuthorizationModes) !== null && _f !== void 0 ? _f : [];
        const modes = [defaultMode, ...additionalModes];
        this.modes = modes.map((mode) => mode.authorizationType);
        this.validateAuthorizationProps(modes);
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultMode.authorizationType,
            logConfig: this.setupLogConfig(props.logConfig),
            openIdConnectConfig: this.setupOpenIdConnectConfig(defaultMode.openIdConnectConfig),
            userPoolConfig: this.setupUserPoolConfig(defaultMode.userPoolConfig),
            lambdaAuthorizerConfig: this.setupLambdaAuthorizerConfig(defaultMode.lambdaAuthorizerConfig),
            additionalAuthenticationProviders: this.setupAdditionalAuthorizationModes(additionalModes),
            xrayEnabled: props.xrayEnabled,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphqlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        this.schema = (_g = props.schema) !== null && _g !== void 0 ? _g : new schema_1.Schema();
        this.schemaResource = this.schema.bind(this);
        if (modes.some((mode) => mode.authorizationType === AuthorizationType.API_KEY)) {
            const config = (_h = modes.find((mode) => {
                return mode.authorizationType === AuthorizationType.API_KEY && mode.apiKeyConfig;
            })) === null || _h === void 0 ? void 0 : _h.apiKeyConfig;
            this.apiKeyResource = this.createAPIKey(config);
            this.apiKeyResource.addDependsOn(this.schemaResource);
            this.apiKey = this.apiKeyResource.attrApiKey;
        }
    }
    /**
     * Import a GraphQL API through this function
     *
     * @param scope scope
     * @param id id
     * @param attrs GraphQL API Attributes of an API
     */
    static fromGraphqlApiAttributes(scope, id, attrs) {
        var _c;
        jsiiDeprecationWarnings._aws_cdk_aws_appsync_GraphqlApiAttributes(attrs);
        const arn = (_c = attrs.graphqlApiArn) !== null && _c !== void 0 ? _c : core_1.Stack.of(scope).formatArn({
            service: 'appsync',
            resource: `apis/${attrs.graphqlApiId}`,
        });
        class Import extends graphqlapi_base_1.GraphqlApiBase {
            constructor(s, i) {
                super(s, i);
                this.apiId = attrs.graphqlApiId;
                this.arn = arn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee, resources, ...actions) {
        jsiiDeprecationWarnings._aws_cdk_aws_appsync_IamResource(resources);
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: resources.resourceArns(this),
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Mutation', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Query', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Subscription', ...fields), 'appsync:GraphQL');
    }
    validateAuthorizationProps(modes) {
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.LAMBDA).length > 1) {
            throw new Error('You can only have a single AWS Lambda function configured to authorize your API.');
        }
        modes.map((mode) => {
            if (mode.authorizationType === AuthorizationType.OIDC && !mode.openIdConnectConfig) {
                throw new Error('Missing OIDC Configuration');
            }
            if (mode.authorizationType === AuthorizationType.USER_POOL && !mode.userPoolConfig) {
                throw new Error('Missing User Pool Configuration');
            }
            if (mode.authorizationType === AuthorizationType.LAMBDA && !mode.lambdaAuthorizerConfig) {
                throw new Error('Missing Lambda Configuration');
            }
        });
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.API_KEY).length > 1) {
            throw new Error('You can\'t duplicate API_KEY configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.IAM).length > 1) {
            throw new Error('You can\'t duplicate IAM configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
    }
    /**
     * Add schema dependency to a given construct
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct) {
        construct.addDependsOn(this.schemaResource);
        return true;
    }
    setupLogConfig(config) {
        var _c, _d;
        if (!config)
            return undefined;
        const logsRoleArn = (_d = (_c = config.role) === null || _c === void 0 ? void 0 : _c.roleArn) !== null && _d !== void 0 ? _d : new aws_iam_1.Role(this, 'ApiLogsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('appsync.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'),
            ],
        }).roleArn;
        return {
            cloudWatchLogsRoleArn: logsRoleArn,
            excludeVerboseContent: config.excludeVerboseContent,
            fieldLogLevel: config.fieldLogLevel,
        };
    }
    setupOpenIdConnectConfig(config) {
        if (!config)
            return undefined;
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    setupUserPoolConfig(config) {
        if (!config)
            return undefined;
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || UserPoolDefaultAction.ALLOW,
        };
    }
    setupLambdaAuthorizerConfig(config) {
        var _c;
        if (!config)
            return undefined;
        return {
            authorizerResultTtlInSeconds: (_c = config.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            authorizerUri: config.handler.functionArn,
            identityValidationExpression: config.validationRegex,
        };
    }
    setupAdditionalAuthorizationModes(modes) {
        if (!modes || modes.length === 0)
            return undefined;
        return modes.reduce((acc, mode) => [
            ...acc, {
                authenticationType: mode.authorizationType,
                userPoolConfig: this.setupUserPoolConfig(mode.userPoolConfig),
                openIdConnectConfig: this.setupOpenIdConnectConfig(mode.openIdConnectConfig),
                lambdaAuthorizerConfig: this.setupLambdaAuthorizerConfig(mode.lambdaAuthorizerConfig),
            },
        ], []);
    }
    createAPIKey(config) {
        var _c, _d;
        if (((_c = config === null || config === void 0 ? void 0 : config.expires) === null || _c === void 0 ? void 0 : _c.isBefore(core_1.Duration.days(1))) || ((_d = config === null || config === void 0 ? void 0 : config.expires) === null || _d === void 0 ? void 0 : _d.isAfter(core_1.Duration.days(365)))) {
            throw Error('API key expiration must be between 1 and 365 days.');
        }
        const expires = (config === null || config === void 0 ? void 0 : config.expires) ? config === null || config === void 0 ? void 0 : config.expires.toEpoch() : undefined;
        return new appsync_generated_1.CfnApiKey(this, `${(config === null || config === void 0 ? void 0 : config.name) || 'Default'}ApiKey`, {
            expires,
            description: config === null || config === void 0 ? void 0 : config.description,
            apiId: this.apiId,
        });
    }
    /**
     * Escape hatch to append to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     */
    addToSchema(addition, delimiter) {
        this.schema.addToSchema(addition, delimiter);
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     */
    addType(type) {
        jsiiDeprecationWarnings._aws_cdk_aws_appsync_IIntermediateType(type);
        return this.schema.addType(type);
    }
    /**
     * Add a query field to the schema's Query. CDK will create an
     * Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName, field) {
        jsiiDeprecationWarnings._aws_cdk_aws_appsync_ResolvableField(field);
        return this.schema.addQuery(fieldName, field);
    }
    /**
     * Add a mutation field to the schema's Mutation. CDK will create an
     * Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName, field) {
        jsiiDeprecationWarnings._aws_cdk_aws_appsync_ResolvableField(field);
        return this.schema.addMutation(fieldName, field);
    }
    /**
     * Add a subscription field to the schema's Subscription. CDK will create an
     * Object Type called 'Subscription'. For example,
     *
     * type Subscription {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription
     * @param field the resolvable field to for this Subscription
     */
    addSubscription(fieldName, field) {
        jsiiDeprecationWarnings._aws_cdk_aws_appsync_ResolvableField(field);
        return this.schema.addSubscription(fieldName, field);
    }
}
exports.GraphqlApi = GraphqlApi;
_b = JSII_RTTI_SYMBOL_1;
GraphqlApi[_b] = { fqn: "@aws-cdk/aws-appsync.GraphqlApi", version: "1.146.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ3JhcGhxbGFwaS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImdyYXBocWxhcGkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQ0EsOENBQW1HO0FBRW5HLHdDQUFpRztBQUVqRywyREFBaUY7QUFDakYsdURBQWdFO0FBQ2hFLHFDQUFrQztBQUtsQzs7R0FFRztBQUNILElBQVksaUJBcUJYO0FBckJELFdBQVksaUJBQWlCO0lBQzNCOztPQUVHO0lBQ0gsd0NBQW1CLENBQUE7SUFDbkI7O09BRUc7SUFDSCxvQ0FBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCw0REFBdUMsQ0FBQTtJQUN2Qzs7T0FFRztJQUNILDRDQUF1QixDQUFBO0lBQ3ZCOztPQUVHO0lBQ0gsMENBQXFCLENBQUE7QUFDdkIsQ0FBQyxFQXJCVyxpQkFBaUIsR0FBakIseUJBQWlCLEtBQWpCLHlCQUFpQixRQXFCNUI7QUFvQ0Q7O0dBRUc7QUFDSCxJQUFZLHFCQVNYO0FBVEQsV0FBWSxxQkFBcUI7SUFDL0I7O09BRUc7SUFDSCx3Q0FBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCxzQ0FBYSxDQUFBO0FBQ2YsQ0FBQyxFQVRXLHFCQUFxQixHQUFyQiw2QkFBcUIsS0FBckIsNkJBQXFCLFFBU2hDO0FBaUlEOztHQUVHO0FBQ0gsSUFBWSxhQWFYO0FBYkQsV0FBWSxhQUFhO0lBQ3ZCOztPQUVHO0lBQ0gsOEJBQWEsQ0FBQTtJQUNiOztPQUVHO0lBQ0gsZ0NBQWUsQ0FBQTtJQUNmOztPQUVHO0lBQ0gsNEJBQVcsQ0FBQTtBQUNiLENBQUMsRUFiVyxhQUFhLEdBQWIscUJBQWEsS0FBYixxQkFBYSxRQWF4QjtBQW1FRDs7R0FFRztBQUNILE1BQWEsV0FBVztJQXFDdEIsWUFBb0IsSUFBYztRQUNoQyxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztLQUNsQjtJQXRDRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQUcsSUFBYztRQUNwQyxJQUFJLElBQUksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3JCLE1BQU0sSUFBSSxLQUFLLENBQUMseUNBQXlDLENBQUMsQ0FBQztTQUM1RDtRQUNELE9BQU8sSUFBSSxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7S0FDOUI7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFZLEVBQUUsR0FBRyxNQUFnQjtRQUNwRCxNQUFNLElBQUksR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxTQUFTLElBQUksV0FBVyxLQUFLLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUMsQ0FBQztRQUMxRyxPQUFPLElBQUksV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO0tBQzlCO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsR0FBRztRQUNmLE9BQU8sSUFBSSxXQUFXLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0tBQy9CO0lBUUQ7Ozs7T0FJRztJQUNJLFlBQVksQ0FBQyxHQUFlOztRQUNqQyxPQUFPLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxZQUFLLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUNwRCxPQUFPLEVBQUUsU0FBUztZQUNsQixRQUFRLEVBQUUsUUFBUSxHQUFHLENBQUMsS0FBSyxFQUFFO1lBQzdCLFNBQVMsRUFBRSxnQkFBUyxDQUFDLG1CQUFtQjtZQUN4QyxZQUFZLEVBQUUsR0FBRyxHQUFHLEVBQUU7U0FDdkIsQ0FBQyxDQUFDLENBQUM7S0FDTDs7QUFyREgsa0NBc0RDOzs7QUFtQkQ7Ozs7R0FJRztBQUNILE1BQWEsVUFBVyxTQUFRLGdDQUFjO0lBbUU1QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCOztRQUM5RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDOztRQUVqQixNQUFNLFdBQVcsZUFBRyxLQUFLLENBQUMsbUJBQW1CLDBDQUFFLG9CQUFvQixtQ0FDakUsRUFBRSxpQkFBaUIsRUFBRSxpQkFBaUIsQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUNuRCxNQUFNLGVBQWUsZUFBRyxLQUFLLENBQUMsbUJBQW1CLDBDQUFFLDRCQUE0QixtQ0FBSSxFQUFFLENBQUM7UUFDdEYsTUFBTSxLQUFLLEdBQUcsQ0FBQyxXQUFXLEVBQUUsR0FBRyxlQUFlLENBQUMsQ0FBQztRQUVoRCxJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBRSxDQUFDO1FBRTFELElBQUksQ0FBQywwQkFBMEIsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUV2QyxJQUFJLENBQUMsR0FBRyxHQUFHLElBQUksaUNBQWEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzdDLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSTtZQUNoQixrQkFBa0IsRUFBRSxXQUFXLENBQUMsaUJBQWlCO1lBQ2pELFNBQVMsRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUM7WUFDL0MsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxtQkFBbUIsQ0FBQztZQUNuRixjQUFjLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUM7WUFDcEUsc0JBQXNCLEVBQUUsSUFBSSxDQUFDLDJCQUEyQixDQUFDLFdBQVcsQ0FBQyxzQkFBc0IsQ0FBQztZQUM1RixpQ0FBaUMsRUFBRSxJQUFJLENBQUMsaUNBQWlDLENBQUMsZUFBZSxDQUFDO1lBQzFGLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztTQUMvQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDO1FBQ2hDLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUM7UUFDNUIsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQztRQUMxQyxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDO1FBQzFCLElBQUksQ0FBQyxNQUFNLFNBQUcsS0FBSyxDQUFDLE1BQU0sbUNBQUksSUFBSSxlQUFNLEVBQUUsQ0FBQztRQUMzQyxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTdDLElBQUksS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQzlFLE1BQU0sTUFBTSxTQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUF1QixFQUFFLEVBQUU7Z0JBQ3BELE9BQU8sSUFBSSxDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLE9BQU8sSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDO1lBQ25GLENBQUMsQ0FBQywwQ0FBRSxZQUFZLENBQUM7WUFDakIsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ2hELElBQUksQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztZQUN0RCxJQUFJLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsVUFBVSxDQUFDO1NBQzlDO0tBQ0Y7SUF4R0Q7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLHdCQUF3QixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTJCOzs7UUFDOUYsTUFBTSxHQUFHLFNBQUcsS0FBSyxDQUFDLGFBQWEsbUNBQUksWUFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxTQUFTLENBQUM7WUFDM0QsT0FBTyxFQUFFLFNBQVM7WUFDbEIsUUFBUSxFQUFFLFFBQVEsS0FBSyxDQUFDLFlBQVksRUFBRTtTQUN2QyxDQUFDLENBQUM7UUFDSCxNQUFNLE1BQU8sU0FBUSxnQ0FBYztZQUdqQyxZQUFhLENBQVksRUFBRSxDQUFTO2dCQUNsQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO2dCQUhFLFVBQUssR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO2dCQUMzQixRQUFHLEdBQUcsR0FBRyxDQUFDO1lBRzFCLENBQUM7U0FDRjtRQUNELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0tBQzlCO0lBc0ZEOzs7Ozs7O09BT0c7SUFDSSxLQUFLLENBQUMsT0FBbUIsRUFBRSxTQUFzQixFQUFFLEdBQUcsT0FBaUI7O1FBQzVFLE9BQU8sZUFBSyxDQUFDLGNBQWMsQ0FBQztZQUMxQixPQUFPO1lBQ1AsT0FBTztZQUNQLFlBQVksRUFBRSxTQUFTLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQztZQUMxQyxLQUFLLEVBQUUsSUFBSTtTQUNaLENBQUMsQ0FBQztLQUNKO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYSxDQUFDLE9BQW1CLEVBQUUsR0FBRyxNQUFnQjtRQUMzRCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLFdBQVcsQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFLEdBQUcsTUFBTSxDQUFDLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztLQUMxRjtJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVUsQ0FBQyxPQUFtQixFQUFFLEdBQUcsTUFBZ0I7UUFDeEQsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsTUFBTSxDQUFDLE9BQU8sRUFBRSxHQUFHLE1BQU0sQ0FBQyxFQUFFLGlCQUFpQixDQUFDLENBQUM7S0FDdkY7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUIsQ0FBQyxPQUFtQixFQUFFLEdBQUcsTUFBZ0I7UUFDL0QsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsTUFBTSxDQUFDLGNBQWMsRUFBRSxHQUFHLE1BQU0sQ0FBQyxFQUFFLGlCQUFpQixDQUFDLENBQUM7S0FDOUY7SUFFTywwQkFBMEIsQ0FBQyxLQUEwQjtRQUMzRCxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsS0FBSyxpQkFBaUIsQ0FBQyxNQUFNLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzFGLE1BQU0sSUFBSSxLQUFLLENBQUMsa0ZBQWtGLENBQUMsQ0FBQztTQUNyRztRQUNELEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRTtZQUNqQixJQUFJLElBQUksQ0FBQyxpQkFBaUIsS0FBSyxpQkFBaUIsQ0FBQyxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsbUJBQW1CLEVBQUU7Z0JBQ2xGLE1BQU0sSUFBSSxLQUFLLENBQUMsNEJBQTRCLENBQUMsQ0FBQzthQUMvQztZQUNELElBQUksSUFBSSxDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLFNBQVMsSUFBSSxDQUFDLElBQUksQ0FBQyxjQUFjLEVBQUU7Z0JBQ2xGLE1BQU0sSUFBSSxLQUFLLENBQUMsaUNBQWlDLENBQUMsQ0FBQzthQUNwRDtZQUNELElBQUksSUFBSSxDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLE1BQU0sSUFBSSxDQUFDLElBQUksQ0FBQyxzQkFBc0IsRUFBRTtnQkFDdkYsTUFBTSxJQUFJLEtBQUssQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO2FBQ2pEO1FBQ0gsQ0FBQyxDQUFDLENBQUM7UUFDSCxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsS0FBSyxpQkFBaUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzNGLE1BQU0sSUFBSSxLQUFLLENBQUMsbUhBQW1ILENBQUMsQ0FBQztTQUN0STtRQUNELElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDdkYsTUFBTSxJQUFJLEtBQUssQ0FBQywrR0FBK0csQ0FBQyxDQUFDO1NBQ2xJO0tBQ0Y7SUFFRDs7OztPQUlHO0lBQ0ksbUJBQW1CLENBQUMsU0FBc0I7UUFDL0MsU0FBUyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDNUMsT0FBTyxJQUFJLENBQUM7S0FDYjtJQUVPLGNBQWMsQ0FBQyxNQUFrQjs7UUFDdkMsSUFBSSxDQUFDLE1BQU07WUFBRSxPQUFPLFNBQVMsQ0FBQztRQUM5QixNQUFNLFdBQVcsZUFBVyxNQUFNLENBQUMsSUFBSSwwQ0FBRSxPQUFPLG1DQUFJLElBQUksY0FBSSxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDaEYsU0FBUyxFQUFFLElBQUksMEJBQWdCLENBQUMsdUJBQXVCLENBQUM7WUFDeEQsZUFBZSxFQUFFO2dCQUNmLHVCQUFhLENBQUMsd0JBQXdCLENBQUMsNkNBQTZDLENBQUM7YUFDdEY7U0FDRixDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ1gsT0FBTztZQUNMLHFCQUFxQixFQUFFLFdBQVc7WUFDbEMscUJBQXFCLEVBQUUsTUFBTSxDQUFDLHFCQUFxQjtZQUNuRCxhQUFhLEVBQUUsTUFBTSxDQUFDLGFBQWE7U0FDcEMsQ0FBQztLQUNIO0lBRU8sd0JBQXdCLENBQUMsTUFBNEI7UUFDM0QsSUFBSSxDQUFDLE1BQU07WUFBRSxPQUFPLFNBQVMsQ0FBQztRQUM5QixPQUFPO1lBQ0wsT0FBTyxFQUFFLE1BQU0sQ0FBQyxtQkFBbUI7WUFDbkMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxRQUFRO1lBQ3pCLE1BQU0sRUFBRSxNQUFNLENBQUMsb0JBQW9CO1lBQ25DLE1BQU0sRUFBRSxNQUFNLENBQUMsWUFBWTtTQUM1QixDQUFDO0tBQ0g7SUFFTyxtQkFBbUIsQ0FBQyxNQUF1QjtRQUNqRCxJQUFJLENBQUMsTUFBTTtZQUFFLE9BQU8sU0FBUyxDQUFDO1FBQzlCLE9BQU87WUFDTCxVQUFVLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxVQUFVO1lBQ3RDLFNBQVMsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxNQUFNO1lBQ3ZDLGdCQUFnQixFQUFFLE1BQU0sQ0FBQyxnQkFBZ0I7WUFDekMsYUFBYSxFQUFFLE1BQU0sQ0FBQyxhQUFhLElBQUkscUJBQXFCLENBQUMsS0FBSztTQUNuRSxDQUFDO0tBQ0g7SUFFTywyQkFBMkIsQ0FBQyxNQUErQjs7UUFDakUsSUFBSSxDQUFDLE1BQU07WUFBRSxPQUFPLFNBQVMsQ0FBQztRQUM5QixPQUFPO1lBQ0wsNEJBQTRCLFFBQUUsTUFBTSxDQUFDLGVBQWUsMENBQUUsU0FBUyxFQUFFO1lBQ2pFLGFBQWEsRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7WUFDekMsNEJBQTRCLEVBQUUsTUFBTSxDQUFDLGVBQWU7U0FDckQsQ0FBQztLQUNIO0lBRU8saUNBQWlDLENBQUMsS0FBMkI7UUFDbkUsSUFBSSxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsTUFBTSxLQUFLLENBQUM7WUFBRSxPQUFPLFNBQVMsQ0FBQztRQUNuRCxPQUFPLEtBQUssQ0FBQyxNQUFNLENBQTJELENBQUMsR0FBRyxFQUFFLElBQUksRUFBRSxFQUFFLENBQUM7WUFDM0YsR0FBRyxHQUFHLEVBQUU7Z0JBQ04sa0JBQWtCLEVBQUUsSUFBSSxDQUFDLGlCQUFpQjtnQkFDMUMsY0FBYyxFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDO2dCQUM3RCxtQkFBbUIsRUFBRSxJQUFJLENBQUMsd0JBQXdCLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDO2dCQUM1RSxzQkFBc0IsRUFBRSxJQUFJLENBQUMsMkJBQTJCLENBQUMsSUFBSSxDQUFDLHNCQUFzQixDQUFDO2FBQ3RGO1NBQ0YsRUFBRSxFQUFFLENBQUMsQ0FBQztLQUNSO0lBRU8sWUFBWSxDQUFDLE1BQXFCOztRQUN4QyxJQUFJLE9BQUEsTUFBTSxhQUFOLE1BQU0sdUJBQU4sTUFBTSxDQUFFLE9BQU8sMENBQUUsUUFBUSxDQUFDLGVBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLGFBQUssTUFBTSxhQUFOLE1BQU0sdUJBQU4sTUFBTSxDQUFFLE9BQU8sMENBQUUsT0FBTyxDQUFDLGVBQVEsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUMsRUFBRTtZQUMvRixNQUFNLEtBQUssQ0FBQyxvREFBb0QsQ0FBQyxDQUFDO1NBQ25FO1FBQ0QsTUFBTSxPQUFPLEdBQUcsQ0FBQSxNQUFNLGFBQU4sTUFBTSx1QkFBTixNQUFNLENBQUUsT0FBTyxFQUFDLENBQUMsQ0FBQyxNQUFNLGFBQU4sTUFBTSx1QkFBTixNQUFNLENBQUUsT0FBTyxDQUFDLE9BQU8sR0FBRyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBQ3hFLE9BQU8sSUFBSSw2QkFBUyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUEsTUFBTSxhQUFOLE1BQU0sdUJBQU4sTUFBTSxDQUFFLElBQUksS0FBSSxTQUFTLFFBQVEsRUFBRTtZQUMvRCxPQUFPO1lBQ1AsV0FBVyxFQUFFLE1BQU0sYUFBTixNQUFNLHVCQUFOLE1BQU0sQ0FBRSxXQUFXO1lBQ2hDLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSztTQUNsQixDQUFDLENBQUM7S0FDSjtJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksV0FBVyxDQUFDLFFBQWdCLEVBQUUsU0FBa0I7UUFDckQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsUUFBUSxFQUFFLFNBQVMsQ0FBQyxDQUFDO0tBQzlDO0lBRUQ7Ozs7O09BS0c7SUFDSSxPQUFPLENBQUMsSUFBdUI7O1FBQ3BDLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7S0FDbEM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksUUFBUSxDQUFDLFNBQWlCLEVBQUUsS0FBc0I7O1FBQ3ZELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO0tBQy9DO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFdBQVcsQ0FBQyxTQUFpQixFQUFFLEtBQXNCOztRQUMxRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQztLQUNsRDtJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxlQUFlLENBQUMsU0FBaUIsRUFBRSxLQUFzQjs7UUFDOUQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGVBQWUsQ0FBQyxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUM7S0FDdEQ7O0FBcFVILGdDQXFVQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElVc2VyUG9vbCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2duaXRvJztcbmltcG9ydCB7IE1hbmFnZWRQb2xpY3ksIFJvbGUsIElSb2xlLCBTZXJ2aWNlUHJpbmNpcGFsLCBHcmFudCwgSUdyYW50YWJsZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgSUZ1bmN0aW9uIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBBcm5Gb3JtYXQsIENmblJlc291cmNlLCBEdXJhdGlvbiwgRXhwaXJhdGlvbiwgSVJlc29sdmFibGUsIFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENmbkFwaUtleSwgQ2ZuR3JhcGhRTEFwaSwgQ2ZuR3JhcGhRTFNjaGVtYSB9IGZyb20gJy4vYXBwc3luYy5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSUdyYXBocWxBcGksIEdyYXBocWxBcGlCYXNlIH0gZnJvbSAnLi9ncmFwaHFsYXBpLWJhc2UnO1xuaW1wb3J0IHsgU2NoZW1hIH0gZnJvbSAnLi9zY2hlbWEnO1xuaW1wb3J0IHsgSUludGVybWVkaWF0ZVR5cGUgfSBmcm9tICcuL3NjaGVtYS1iYXNlJztcbmltcG9ydCB7IFJlc29sdmFibGVGaWVsZCB9IGZyb20gJy4vc2NoZW1hLWZpZWxkJztcbmltcG9ydCB7IE9iamVjdFR5cGUgfSBmcm9tICcuL3NjaGVtYS1pbnRlcm1lZGlhdGUnO1xuXG4vKipcbiAqIGVudW0gd2l0aCBhbGwgcG9zc2libGUgdmFsdWVzIGZvciBBcHBTeW5jIGF1dGhvcml6YXRpb24gdHlwZVxuICovXG5leHBvcnQgZW51bSBBdXRob3JpemF0aW9uVHlwZSB7XG4gIC8qKlxuICAgKiBBUEkgS2V5IGF1dGhvcml6YXRpb24gdHlwZVxuICAgKi9cbiAgQVBJX0tFWSA9ICdBUElfS0VZJyxcbiAgLyoqXG4gICAqIEFXUyBJQU0gYXV0aG9yaXphdGlvbiB0eXBlLiBDYW4gYmUgdXNlZCB3aXRoIENvZ25pdG8gSWRlbnRpdHkgUG9vbCBmZWRlcmF0ZWQgY3JlZGVudGlhbHNcbiAgICovXG4gIElBTSA9ICdBV1NfSUFNJyxcbiAgLyoqXG4gICAqIENvZ25pdG8gVXNlciBQb29sIGF1dGhvcml6YXRpb24gdHlwZVxuICAgKi9cbiAgVVNFUl9QT09MID0gJ0FNQVpPTl9DT0dOSVRPX1VTRVJfUE9PTFMnLFxuICAvKipcbiAgICogT3BlbklEIENvbm5lY3QgYXV0aG9yaXphdGlvbiB0eXBlXG4gICAqL1xuICBPSURDID0gJ09QRU5JRF9DT05ORUNUJyxcbiAgLyoqXG4gICAqIExhbWJkYSBhdXRob3JpemF0aW9uIHR5cGVcbiAgICovXG4gIExBTUJEQSA9ICdBV1NfTEFNQkRBJyxcbn1cblxuLyoqXG4gKiBJbnRlcmZhY2UgdG8gc3BlY2lmeSBkZWZhdWx0IG9yIGFkZGl0aW9uYWwgYXV0aG9yaXphdGlvbihzKVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEF1dGhvcml6YXRpb25Nb2RlIHtcbiAgLyoqXG4gICAqIE9uZSBvZiBwb3NzaWJsZSBmb3VyIHZhbHVlcyBBcHBTeW5jIHN1cHBvcnRzXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L2Rldmd1aWRlL3NlY3VyaXR5Lmh0bWxcbiAgICpcbiAgICogQGRlZmF1bHQgLSBgQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWWBcbiAgICovXG4gIHJlYWRvbmx5IGF1dGhvcml6YXRpb25UeXBlOiBBdXRob3JpemF0aW9uVHlwZTtcbiAgLyoqXG4gICAqIElmIGF1dGhvcml6YXRpb25UeXBlIGlzIGBBdXRob3JpemF0aW9uVHlwZS5VU0VSX1BPT0xgLCB0aGlzIG9wdGlvbiBpcyByZXF1aXJlZC5cbiAgICogQGRlZmF1bHQgLSBub25lXG4gICAqL1xuICByZWFkb25seSB1c2VyUG9vbENvbmZpZz86IFVzZXJQb29sQ29uZmlnO1xuICAvKipcbiAgICogSWYgYXV0aG9yaXphdGlvblR5cGUgaXMgYEF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVlgLCB0aGlzIG9wdGlvbiBjYW4gYmUgY29uZmlndXJlZC5cbiAgICogQGRlZmF1bHQgLSBuYW1lOiAnRGVmYXVsdEFQSUtleScgfCBkZXNjcmlwdGlvbjogJ0RlZmF1bHQgQVBJIEtleSBjcmVhdGVkIGJ5IENESydcbiAgICovXG4gIHJlYWRvbmx5IGFwaUtleUNvbmZpZz86IEFwaUtleUNvbmZpZztcbiAgLyoqXG4gICAqIElmIGF1dGhvcml6YXRpb25UeXBlIGlzIGBBdXRob3JpemF0aW9uVHlwZS5PSURDYCwgdGhpcyBvcHRpb24gaXMgcmVxdWlyZWQuXG4gICAqIEBkZWZhdWx0IC0gbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgb3BlbklkQ29ubmVjdENvbmZpZz86IE9wZW5JZENvbm5lY3RDb25maWc7XG4gIC8qKlxuICAgKiBJZiBhdXRob3JpemF0aW9uVHlwZSBpcyBgQXV0aG9yaXphdGlvblR5cGUuTEFNQkRBYCwgdGhpcyBvcHRpb24gaXMgcmVxdWlyZWQuXG4gICAqIEBkZWZhdWx0IC0gbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgbGFtYmRhQXV0aG9yaXplckNvbmZpZz86IExhbWJkYUF1dGhvcml6ZXJDb25maWc7XG59XG5cbi8qKlxuICogZW51bSB3aXRoIGFsbCBwb3NzaWJsZSB2YWx1ZXMgZm9yIENvZ25pdG8gdXNlci1wb29sIGRlZmF1bHQgYWN0aW9uc1xuICovXG5leHBvcnQgZW51bSBVc2VyUG9vbERlZmF1bHRBY3Rpb24ge1xuICAvKipcbiAgICogQUxMT1cgYWNjZXNzIHRvIEFQSVxuICAgKi9cbiAgQUxMT1cgPSAnQUxMT1cnLFxuICAvKipcbiAgICogREVOWSBhY2Nlc3MgdG8gQVBJXG4gICAqL1xuICBERU5ZID0gJ0RFTlknLFxufVxuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIENvZ25pdG8gdXNlci1wb29scyBpbiBBcHBTeW5jXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVXNlclBvb2xDb25maWcge1xuICAvKipcbiAgICogVGhlIENvZ25pdG8gdXNlciBwb29sIHRvIHVzZSBhcyBpZGVudGl0eSBzb3VyY2VcbiAgICovXG4gIHJlYWRvbmx5IHVzZXJQb29sOiBJVXNlclBvb2w7XG4gIC8qKlxuICAgKiB0aGUgb3B0aW9uYWwgYXBwIGlkIHJlZ2V4XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGFwcElkQ2xpZW50UmVnZXg/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBEZWZhdWx0IGF1dGggYWN0aW9uXG4gICAqXG4gICAqIEBkZWZhdWx0IEFMTE9XXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0QWN0aW9uPzogVXNlclBvb2xEZWZhdWx0QWN0aW9uO1xufVxuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIEFQSSBLZXkgYXV0aG9yaXphdGlvbiBpbiBBcHBTeW5jXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBpS2V5Q29uZmlnIHtcbiAgLyoqXG4gICAqIFVuaXF1ZSBuYW1lIG9mIHRoZSBBUEkgS2V5XG4gICAqIEBkZWZhdWx0IC0gJ0RlZmF1bHRBUElLZXknXG4gICAqL1xuICByZWFkb25seSBuYW1lPzogc3RyaW5nO1xuICAvKipcbiAgICogRGVzY3JpcHRpb24gb2YgQVBJIGtleVxuICAgKiBAZGVmYXVsdCAtICdEZWZhdWx0IEFQSSBLZXkgY3JlYXRlZCBieSBDREsnXG4gICAqL1xuICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHRpbWUgZnJvbSBjcmVhdGlvbiB0aW1lIGFmdGVyIHdoaWNoIHRoZSBBUEkga2V5IGV4cGlyZXMuXG4gICAqIEl0IG11c3QgYmUgYSBtaW5pbXVtIG9mIDEgZGF5IGFuZCBhIG1heGltdW0gb2YgMzY1IGRheXMgZnJvbSBkYXRlIG9mIGNyZWF0aW9uLlxuICAgKiBSb3VuZGVkIGRvd24gdG8gdGhlIG5lYXJlc3QgaG91ci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSA3IGRheXMgcm91bmRlZCBkb3duIHRvIG5lYXJlc3QgaG91clxuICAgKi9cbiAgcmVhZG9ubHkgZXhwaXJlcz86IEV4cGlyYXRpb247XG59XG5cbi8qKlxuICogQ29uZmlndXJhdGlvbiBmb3IgT3BlbklEIENvbm5lY3QgYXV0aG9yaXphdGlvbiBpbiBBcHBTeW5jXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgT3BlbklkQ29ubmVjdENvbmZpZyB7XG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIG1pbGxpc2Vjb25kcyBhbiBPSURDIHRva2VuIGlzIHZhbGlkIGFmdGVyIGJlaW5nIGF1dGhlbnRpY2F0ZWQgYnkgT0lEQyBwcm92aWRlci5cbiAgICogYGF1dGhfdGltZWAgY2xhaW0gaW4gT0lEQyB0b2tlbiBpcyByZXF1aXJlZCBmb3IgdGhpcyB2YWxpZGF0aW9uIHRvIHdvcmsuXG4gICAqIEBkZWZhdWx0IC0gbm8gdmFsaWRhdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgdG9rZW5FeHBpcnlGcm9tQXV0aD86IG51bWJlcjtcbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgbWlsbGlzZWNvbmRzIGFuIE9JREMgdG9rZW4gaXMgdmFsaWQgYWZ0ZXIgYmVpbmcgaXNzdWVkIHRvIGEgdXNlci5cbiAgICogVGhpcyB2YWxpZGF0aW9uIHVzZXMgYGlhdGAgY2xhaW0gb2YgT0lEQyB0b2tlbi5cbiAgICogQGRlZmF1bHQgLSBubyB2YWxpZGF0aW9uXG4gICAqL1xuICByZWFkb25seSB0b2tlbkV4cGlyeUZyb21Jc3N1ZT86IG51bWJlcjtcbiAgLyoqXG4gICAqIFRoZSBjbGllbnQgaWRlbnRpZmllciBvZiB0aGUgUmVseWluZyBwYXJ0eSBhdCB0aGUgT3BlbklEIGlkZW50aXR5IHByb3ZpZGVyLlxuICAgKiBBIHJlZ3VsYXIgZXhwcmVzc2lvbiBjYW4gYmUgc3BlY2lmaWVkIHNvIEFwcFN5bmMgY2FuIHZhbGlkYXRlIGFnYWluc3QgbXVsdGlwbGUgY2xpZW50IGlkZW50aWZpZXJzIGF0IGEgdGltZS5cbiAgICogQGV4YW1wbGUgLSAnQUJDRHxDREVGJyAvLyB3aGVyZSBBQkNEIGFuZCBDREVGIGFyZSB0d28gZGlmZmVyZW50IGNsaWVudElkXG4gICAqIEBkZWZhdWx0IC0gKiAoQWxsKVxuICAgKi9cbiAgcmVhZG9ubHkgY2xpZW50SWQ/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgaXNzdWVyIGZvciB0aGUgT0lEQyBjb25maWd1cmF0aW9uLiBUaGUgaXNzdWVyIHJldHVybmVkIGJ5IGRpc2NvdmVyeSBtdXN0IGV4YWN0bHkgbWF0Y2ggdGhlIHZhbHVlIG9mIGBpc3NgIGluIHRoZSBPSURDIHRva2VuLlxuICAgKi9cbiAgcmVhZG9ubHkgb2lkY1Byb3ZpZGVyOiBzdHJpbmc7XG59XG5cbi8qKlxuICogQ29uZmlndXJhdGlvbiBmb3IgTGFtYmRhIGF1dGhvcml6YXRpb24gaW4gQXBwU3luYy4gTm90ZSB0aGF0IHlvdSBjYW4gb25seSBoYXZlIGEgc2luZ2xlIEFXUyBMYW1iZGEgZnVuY3Rpb24gY29uZmlndXJlZCB0byBhdXRob3JpemUgeW91ciBBUEkuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGFtYmRhQXV0aG9yaXplckNvbmZpZyB7XG4gIC8qKlxuICAgKiBUaGUgYXV0aG9yaXplciBsYW1iZGEgZnVuY3Rpb24uXG4gICAqIE5vdGU6IFRoaXMgTGFtYmRhIGZ1bmN0aW9uIG11c3QgaGF2ZSB0aGUgZm9sbG93aW5nIHJlc291cmNlLWJhc2VkIHBvbGljeSBhc3NpZ25lZCB0byBpdC5cbiAgICogV2hlbiBjb25maWd1cmluZyBMYW1iZGEgYXV0aG9yaXplcnMgaW4gdGhlIGNvbnNvbGUsIHRoaXMgaXMgZG9uZSBmb3IgeW91LlxuICAgKiBUbyBkbyBzbyB3aXRoIHRoZSBBV1MgQ0xJLCBydW4gdGhlIGZvbGxvd2luZzpcbiAgICpcbiAgICogYGF3cyBsYW1iZGEgYWRkLXBlcm1pc3Npb24gLS1mdW5jdGlvbi1uYW1lIFwiYXJuOmF3czpsYW1iZGE6dXMtZWFzdC0yOjExMTEyMjIyMzMzMzpmdW5jdGlvbjpteS1mdW5jdGlvblwiIC0tc3RhdGVtZW50LWlkIFwiYXBwc3luY1wiIC0tcHJpbmNpcGFsIGFwcHN5bmMuYW1hem9uYXdzLmNvbSAtLWFjdGlvbiBsYW1iZGE6SW52b2tlRnVuY3Rpb25gXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXByb3BlcnRpZXMtYXBwc3luYy1ncmFwaHFsYXBpLWxhbWJkYWF1dGhvcml6ZXJjb25maWcuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgaGFuZGxlcjogSUZ1bmN0aW9uO1xuXG4gIC8qKlxuICAgKiBIb3cgbG9uZyB0aGUgcmVzdWx0cyBhcmUgY2FjaGVkLlxuICAgKiBEaXNhYmxlIGNhY2hpbmcgYnkgc2V0dGluZyB0aGlzIHRvIDAuXG4gICAqXG4gICAqIEBkZWZhdWx0IER1cmF0aW9uLm1pbnV0ZXMoNSlcbiAgICovXG4gIHJlYWRvbmx5IHJlc3VsdHNDYWNoZVR0bD86IER1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBBIHJlZ3VsYXIgZXhwcmVzc2lvbiBmb3IgdmFsaWRhdGlvbiBvZiB0b2tlbnMgYmVmb3JlIHRoZSBMYW1iZGEgZnVuY3Rpb24gaXMgY2FsbGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIHJlZ2V4IGZpbHRlciB3aWxsIGJlIGFwcGxpZWQuXG4gICAqL1xuICByZWFkb25seSB2YWxpZGF0aW9uUmVnZXg/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogQ29uZmlndXJhdGlvbiBvZiB0aGUgQVBJIGF1dGhvcml6YXRpb24gbW9kZXMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXV0aG9yaXphdGlvbkNvbmZpZyB7XG4gIC8qKlxuICAgKiBPcHRpb25hbCBhdXRob3JpemF0aW9uIGNvbmZpZ3VyYXRpb25cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBUEkgS2V5IGF1dGhvcml6YXRpb25cbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRBdXRob3JpemF0aW9uPzogQXV0aG9yaXphdGlvbk1vZGU7XG5cbiAgLyoqXG4gICAqIEFkZGl0aW9uYWwgYXV0aG9yaXphdGlvbiBtb2Rlc1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIG90aGVyIG1vZGVzXG4gICAqL1xuICByZWFkb25seSBhZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzPzogQXV0aG9yaXphdGlvbk1vZGVbXTtcbn1cblxuLyoqXG4gKiBsb2ctbGV2ZWwgZm9yIGZpZWxkcyBpbiBBcHBTeW5jXG4gKi9cbmV4cG9ydCBlbnVtIEZpZWxkTG9nTGV2ZWwge1xuICAvKipcbiAgICogTm8gbG9nZ2luZ1xuICAgKi9cbiAgTk9ORSA9ICdOT05FJyxcbiAgLyoqXG4gICAqIEVycm9yIGxvZ2dpbmdcbiAgICovXG4gIEVSUk9SID0gJ0VSUk9SJyxcbiAgLyoqXG4gICAqIEFsbCBsb2dnaW5nXG4gICAqL1xuICBBTEwgPSAnQUxMJyxcbn1cblxuLyoqXG4gKiBMb2dnaW5nIGNvbmZpZ3VyYXRpb24gZm9yIEFwcFN5bmNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMb2dDb25maWcge1xuICAvKipcbiAgICogZXhjbHVkZSB2ZXJib3NlIGNvbnRlbnRcbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGV4Y2x1ZGVWZXJib3NlQ29udGVudD86IGJvb2xlYW4gfCBJUmVzb2x2YWJsZTtcbiAgLyoqXG4gICAqIGxvZyBsZXZlbCBmb3IgZmllbGRzXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVXNlIEFwcFN5bmMgZGVmYXVsdFxuICAgKi9cbiAgcmVhZG9ubHkgZmllbGRMb2dMZXZlbD86IEZpZWxkTG9nTGV2ZWw7XG5cbiAgLyoqXG4gICAqIFRoZSByb2xlIGZvciBDbG91ZFdhdGNoIExvZ3NcbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSByb2xlPzogSVJvbGU7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYW4gQXBwU3luYyBHcmFwaFFMIEFQSVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEdyYXBocWxBcGlQcm9wcyB7XG4gIC8qKlxuICAgKiB0aGUgbmFtZSBvZiB0aGUgR3JhcGhRTCBBUElcbiAgICovXG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogT3B0aW9uYWwgYXV0aG9yaXphdGlvbiBjb25maWd1cmF0aW9uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQVBJIEtleSBhdXRob3JpemF0aW9uXG4gICAqL1xuICByZWFkb25seSBhdXRob3JpemF0aW9uQ29uZmlnPzogQXV0aG9yaXphdGlvbkNvbmZpZztcblxuICAvKipcbiAgICogTG9nZ2luZyBjb25maWd1cmF0aW9uIGZvciB0aGlzIGFwaVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGxvZ0NvbmZpZz86IExvZ0NvbmZpZztcblxuICAvKipcbiAgICogR3JhcGhRTCBzY2hlbWEgZGVmaW5pdGlvbi4gU3BlY2lmeSBob3cgeW91IHdhbnQgdG8gZGVmaW5lIHlvdXIgc2NoZW1hLlxuICAgKlxuICAgKiBTY2hlbWEuZnJvbUZpbGUoZmlsZVBhdGg6IHN0cmluZykgYWxsb3dzIHNjaGVtYSBkZWZpbml0aW9uIHRocm91Z2ggc2NoZW1hLmdyYXBocWwgZmlsZVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHNjaGVtYSB3aWxsIGJlIGdlbmVyYXRlZCBjb2RlLWZpcnN0IChpLmUuIGFkZFR5cGUsIGFkZE9iamVjdFR5cGUsIGV0Yy4pXG4gICAqXG4gICAqL1xuICByZWFkb25seSBzY2hlbWE/OiBTY2hlbWE7XG4gIC8qKlxuICAgKiBBIGZsYWcgaW5kaWNhdGluZyB3aGV0aGVyIG9yIG5vdCBYLVJheSB0cmFjaW5nIGlzIGVuYWJsZWQgZm9yIHRoZSBHcmFwaFFMIEFQSS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgeHJheUVuYWJsZWQ/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIEEgY2xhc3MgdXNlZCB0byBnZW5lcmF0ZSByZXNvdXJjZSBhcm5zIGZvciBBcHBTeW5jXG4gKi9cbmV4cG9ydCBjbGFzcyBJYW1SZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBHZW5lcmF0ZSB0aGUgcmVzb3VyY2UgbmFtZXMgZ2l2ZW4gY3VzdG9tIGFybnNcbiAgICpcbiAgICogQHBhcmFtIGFybnMgVGhlIGN1c3RvbSBhcm5zIHRoYXQgbmVlZCB0byBiZSBwZXJtaXNzaW9uZWRcbiAgICpcbiAgICogRXhhbXBsZTogY3VzdG9tKCcvdHlwZXMvUXVlcnkvZmllbGRzL2dldEV4YW1wbGUnKVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBjdXN0b20oLi4uYXJuczogc3RyaW5nW10pOiBJYW1SZXNvdXJjZSB7XG4gICAgaWYgKGFybnMubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F0IGxlYXN0IDEgY3VzdG9tIEFSTiBtdXN0IGJlIHByb3ZpZGVkLicpO1xuICAgIH1cbiAgICByZXR1cm4gbmV3IElhbVJlc291cmNlKGFybnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlIHRoZSByZXNvdXJjZSBuYW1lcyBnaXZlbiBhIHR5cGUgYW5kIGZpZWxkc1xuICAgKlxuICAgKiBAcGFyYW0gdHlwZSBUaGUgdHlwZSB0aGF0IG5lZWRzIHRvIGJlIGFsbG93ZWRcbiAgICogQHBhcmFtIGZpZWxkcyBUaGUgZmllbGRzIHRoYXQgbmVlZCB0byBiZSBhbGxvd2VkLCBpZiBlbXB0eSBncmFudCBwZXJtaXNzaW9ucyB0byBBTEwgZmllbGRzXG4gICAqXG4gICAqIEV4YW1wbGU6IG9mVHlwZSgnUXVlcnknLCAnR2V0RXhhbXBsZScpXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIG9mVHlwZSh0eXBlOiBzdHJpbmcsIC4uLmZpZWxkczogc3RyaW5nW10pOiBJYW1SZXNvdXJjZSB7XG4gICAgY29uc3QgYXJucyA9IGZpZWxkcy5sZW5ndGggPyBmaWVsZHMubWFwKChmaWVsZCkgPT4gYHR5cGVzLyR7dHlwZX0vZmllbGRzLyR7ZmllbGR9YCkgOiBbYHR5cGVzLyR7dHlwZX0vKmBdO1xuICAgIHJldHVybiBuZXcgSWFtUmVzb3VyY2UoYXJucyk7XG4gIH1cblxuICAvKipcbiAgICogR2VuZXJhdGUgdGhlIHJlc291cmNlIG5hbWVzIHRoYXQgYWNjZXB0cyBhbGwgdHlwZXM6IGAqYFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhbGwoKTogSWFtUmVzb3VyY2Uge1xuICAgIHJldHVybiBuZXcgSWFtUmVzb3VyY2UoWycqJ10pO1xuICB9XG5cbiAgcHJpdmF0ZSBhcm5zOiBzdHJpbmdbXTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKGFybnM6IHN0cmluZ1tdKSB7XG4gICAgdGhpcy5hcm5zID0gYXJucztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gdGhlIFJlc291cmNlIEFSTlxuICAgKlxuICAgKiBAcGFyYW0gYXBpIFRoZSBHcmFwaFFMIEFQSSB0byBnaXZlIHBlcm1pc3Npb25zXG4gICAqL1xuICBwdWJsaWMgcmVzb3VyY2VBcm5zKGFwaTogR3JhcGhxbEFwaSk6IHN0cmluZ1tdIHtcbiAgICByZXR1cm4gdGhpcy5hcm5zLm1hcCgoYXJuKSA9PiBTdGFjay5vZihhcGkpLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnYXBwc3luYycsXG4gICAgICByZXNvdXJjZTogYGFwaXMvJHthcGkuYXBpSWR9YCxcbiAgICAgIGFybkZvcm1hdDogQXJuRm9ybWF0LlNMQVNIX1JFU09VUkNFX05BTUUsXG4gICAgICByZXNvdXJjZU5hbWU6IGAke2Fybn1gLFxuICAgIH0pKTtcbiAgfVxufVxuXG4vKipcbiAqIEF0dHJpYnV0ZXMgZm9yIEdyYXBoUUwgaW1wb3J0c1xuICovXG5leHBvcnQgaW50ZXJmYWNlIEdyYXBocWxBcGlBdHRyaWJ1dGVzIHtcbiAgLyoqXG4gICAqIGFuIHVuaXF1ZSBBV1MgQXBwU3luYyBHcmFwaFFMIEFQSSBpZGVudGlmaWVyXG4gICAqIGkuZS4gJ2x4ejc3NWx3ZHJnY25kZ3ozbnVydmFjN29hJ1xuICAgKi9cbiAgcmVhZG9ubHkgZ3JhcGhxbEFwaUlkOiBzdHJpbmcsXG5cbiAgLyoqXG4gICAqIHRoZSBhcm4gZm9yIHRoZSBHcmFwaFFMIEFwaVxuICAgKiBAZGVmYXVsdCAtIGF1dG9nZW5lcmF0ZWQgYXJuXG4gICAqL1xuICByZWFkb25seSBncmFwaHFsQXBpQXJuPzogc3RyaW5nLFxufVxuXG4vKipcbiAqIEFuIEFwcFN5bmMgR3JhcGhRTCBBUElcbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpBcHBTeW5jOjpHcmFwaFFMQXBpXG4gKi9cbmV4cG9ydCBjbGFzcyBHcmFwaHFsQXBpIGV4dGVuZHMgR3JhcGhxbEFwaUJhc2Uge1xuICAvKipcbiAgICogSW1wb3J0IGEgR3JhcGhRTCBBUEkgdGhyb3VnaCB0aGlzIGZ1bmN0aW9uXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSBzY29wZVxuICAgKiBAcGFyYW0gaWQgaWRcbiAgICogQHBhcmFtIGF0dHJzIEdyYXBoUUwgQVBJIEF0dHJpYnV0ZXMgb2YgYW4gQVBJXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21HcmFwaHFsQXBpQXR0cmlidXRlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogR3JhcGhxbEFwaUF0dHJpYnV0ZXMpOiBJR3JhcGhxbEFwaSB7XG4gICAgY29uc3QgYXJuID0gYXR0cnMuZ3JhcGhxbEFwaUFybiA/PyBTdGFjay5vZihzY29wZSkuZm9ybWF0QXJuKHtcbiAgICAgIHNlcnZpY2U6ICdhcHBzeW5jJyxcbiAgICAgIHJlc291cmNlOiBgYXBpcy8ke2F0dHJzLmdyYXBocWxBcGlJZH1gLFxuICAgIH0pO1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIEdyYXBocWxBcGlCYXNlIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBhcGlJZCA9IGF0dHJzLmdyYXBocWxBcGlJZDtcbiAgICAgIHB1YmxpYyByZWFkb25seSBhcm4gPSBhcm47XG4gICAgICBjb25zdHJ1Y3RvciAoczogQ29uc3RydWN0LCBpOiBzdHJpbmcpIHtcbiAgICAgICAgc3VwZXIocywgaSk7XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICAvKipcbiAgICogYW4gdW5pcXVlIEFXUyBBcHBTeW5jIEdyYXBoUUwgQVBJIGlkZW50aWZpZXJcbiAgICogaS5lLiAnbHh6Nzc1bHdkcmdjbmRnejNudXJ2YWM3b2EnXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXBpSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogdGhlIEFSTiBvZiB0aGUgQVBJXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIHRoZSBVUkwgb2YgdGhlIGVuZHBvaW50IGNyZWF0ZWQgYnkgQXBwU3luY1xuICAgKlxuICAgKiBAYXR0cmlidXRlIEdyYXBoUWxVcmxcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBncmFwaHFsVXJsOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIHRoZSBuYW1lIG9mIHRoZSBBUElcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIHRoZSBzY2hlbWEgYXR0YWNoZWQgdG8gdGhpcyBhcGlcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzY2hlbWE6IFNjaGVtYTtcblxuICAvKipcbiAgICogVGhlIEF1dGhvcml6YXRpb24gVHlwZXMgZm9yIHRoaXMgR3JhcGhRTCBBcGlcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBtb2RlczogQXV0aG9yaXphdGlvblR5cGVbXTtcblxuICAvKipcbiAgICogdGhlIGNvbmZpZ3VyZWQgQVBJIGtleSwgaWYgcHJlc2VudFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGFwaSBrZXlcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhcGlLZXk/OiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSBzY2hlbWFSZXNvdXJjZTogQ2ZuR3JhcGhRTFNjaGVtYTtcbiAgcHJpdmF0ZSBhcGk6IENmbkdyYXBoUUxBcGk7XG4gIHByaXZhdGUgYXBpS2V5UmVzb3VyY2U/OiBDZm5BcGlLZXk7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEdyYXBocWxBcGlQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBkZWZhdWx0TW9kZSA9IHByb3BzLmF1dGhvcml6YXRpb25Db25maWc/LmRlZmF1bHRBdXRob3JpemF0aW9uID8/XG4gICAgICB7IGF1dGhvcml6YXRpb25UeXBlOiBBdXRob3JpemF0aW9uVHlwZS5BUElfS0VZIH07XG4gICAgY29uc3QgYWRkaXRpb25hbE1vZGVzID0gcHJvcHMuYXV0aG9yaXphdGlvbkNvbmZpZz8uYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlcyA/PyBbXTtcbiAgICBjb25zdCBtb2RlcyA9IFtkZWZhdWx0TW9kZSwgLi4uYWRkaXRpb25hbE1vZGVzXTtcblxuICAgIHRoaXMubW9kZXMgPSBtb2Rlcy5tYXAoKG1vZGUpID0+IG1vZGUuYXV0aG9yaXphdGlvblR5cGUgKTtcblxuICAgIHRoaXMudmFsaWRhdGVBdXRob3JpemF0aW9uUHJvcHMobW9kZXMpO1xuXG4gICAgdGhpcy5hcGkgPSBuZXcgQ2ZuR3JhcGhRTEFwaSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBuYW1lOiBwcm9wcy5uYW1lLFxuICAgICAgYXV0aGVudGljYXRpb25UeXBlOiBkZWZhdWx0TW9kZS5hdXRob3JpemF0aW9uVHlwZSxcbiAgICAgIGxvZ0NvbmZpZzogdGhpcy5zZXR1cExvZ0NvbmZpZyhwcm9wcy5sb2dDb25maWcpLFxuICAgICAgb3BlbklkQ29ubmVjdENvbmZpZzogdGhpcy5zZXR1cE9wZW5JZENvbm5lY3RDb25maWcoZGVmYXVsdE1vZGUub3BlbklkQ29ubmVjdENvbmZpZyksXG4gICAgICB1c2VyUG9vbENvbmZpZzogdGhpcy5zZXR1cFVzZXJQb29sQ29uZmlnKGRlZmF1bHRNb2RlLnVzZXJQb29sQ29uZmlnKSxcbiAgICAgIGxhbWJkYUF1dGhvcml6ZXJDb25maWc6IHRoaXMuc2V0dXBMYW1iZGFBdXRob3JpemVyQ29uZmlnKGRlZmF1bHRNb2RlLmxhbWJkYUF1dGhvcml6ZXJDb25maWcpLFxuICAgICAgYWRkaXRpb25hbEF1dGhlbnRpY2F0aW9uUHJvdmlkZXJzOiB0aGlzLnNldHVwQWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlcyhhZGRpdGlvbmFsTW9kZXMpLFxuICAgICAgeHJheUVuYWJsZWQ6IHByb3BzLnhyYXlFbmFibGVkLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hcGlJZCA9IHRoaXMuYXBpLmF0dHJBcGlJZDtcbiAgICB0aGlzLmFybiA9IHRoaXMuYXBpLmF0dHJBcm47XG4gICAgdGhpcy5ncmFwaHFsVXJsID0gdGhpcy5hcGkuYXR0ckdyYXBoUWxVcmw7XG4gICAgdGhpcy5uYW1lID0gdGhpcy5hcGkubmFtZTtcbiAgICB0aGlzLnNjaGVtYSA9IHByb3BzLnNjaGVtYSA/PyBuZXcgU2NoZW1hKCk7XG4gICAgdGhpcy5zY2hlbWFSZXNvdXJjZSA9IHRoaXMuc2NoZW1hLmJpbmQodGhpcyk7XG5cbiAgICBpZiAobW9kZXMuc29tZSgobW9kZSkgPT4gbW9kZS5hdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWSkpIHtcbiAgICAgIGNvbnN0IGNvbmZpZyA9IG1vZGVzLmZpbmQoKG1vZGU6IEF1dGhvcml6YXRpb25Nb2RlKSA9PiB7XG4gICAgICAgIHJldHVybiBtb2RlLmF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5BUElfS0VZICYmIG1vZGUuYXBpS2V5Q29uZmlnO1xuICAgICAgfSk/LmFwaUtleUNvbmZpZztcbiAgICAgIHRoaXMuYXBpS2V5UmVzb3VyY2UgPSB0aGlzLmNyZWF0ZUFQSUtleShjb25maWcpO1xuICAgICAgdGhpcy5hcGlLZXlSZXNvdXJjZS5hZGREZXBlbmRzT24odGhpcy5zY2hlbWFSZXNvdXJjZSk7XG4gICAgICB0aGlzLmFwaUtleSA9IHRoaXMuYXBpS2V5UmVzb3VyY2UuYXR0ckFwaUtleTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiBJQU0gcG9saWN5IHN0YXRlbWVudCBhc3NvY2lhdGVkIHdpdGggdGhpcyBHcmFwaFFMQXBpIHRvIGFuIElBTVxuICAgKiBwcmluY2lwYWwncyBwb2xpY3kuXG4gICAqXG4gICAqIEBwYXJhbSBncmFudGVlIFRoZSBwcmluY2lwYWxcbiAgICogQHBhcmFtIHJlc291cmNlcyBUaGUgc2V0IG9mIHJlc291cmNlcyB0byBhbGxvdyAoaS5lLiAuLi46W3JlZ2lvbl06W2FjY291bnRJZF06YXBpcy9HcmFwaFFMSWQvLi4uKVxuICAgKiBAcGFyYW0gYWN0aW9ucyBUaGUgYWN0aW9ucyB0aGF0IHNob3VsZCBiZSBncmFudGVkIHRvIHRoZSBwcmluY2lwYWwgKGkuZS4gYXBwc3luYzpncmFwaHFsIClcbiAgICovXG4gIHB1YmxpYyBncmFudChncmFudGVlOiBJR3JhbnRhYmxlLCByZXNvdXJjZXM6IElhbVJlc291cmNlLCAuLi5hY3Rpb25zOiBzdHJpbmdbXSk6IEdyYW50IHtcbiAgICByZXR1cm4gR3JhbnQuYWRkVG9QcmluY2lwYWwoe1xuICAgICAgZ3JhbnRlZSxcbiAgICAgIGFjdGlvbnMsXG4gICAgICByZXNvdXJjZUFybnM6IHJlc291cmNlcy5yZXNvdXJjZUFybnModGhpcyksXG4gICAgICBzY29wZTogdGhpcyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGFuIElBTSBwb2xpY3kgc3RhdGVtZW50IGZvciBNdXRhdGlvbiBhY2Nlc3MgdG8gdGhpcyBHcmFwaFFMQXBpIHRvIGFuIElBTVxuICAgKiBwcmluY2lwYWwncyBwb2xpY3kuXG4gICAqXG4gICAqIEBwYXJhbSBncmFudGVlIFRoZSBwcmluY2lwYWxcbiAgICogQHBhcmFtIGZpZWxkcyBUaGUgZmllbGRzIHRvIGdyYW50IGFjY2VzcyB0byB0aGF0IGFyZSBNdXRhdGlvbnMgKGxlYXZlIGJsYW5rIGZvciBhbGwpXG4gICAqL1xuICBwdWJsaWMgZ3JhbnRNdXRhdGlvbihncmFudGVlOiBJR3JhbnRhYmxlLCAuLi5maWVsZHM6IHN0cmluZ1tdKTogR3JhbnQge1xuICAgIHJldHVybiB0aGlzLmdyYW50KGdyYW50ZWUsIElhbVJlc291cmNlLm9mVHlwZSgnTXV0YXRpb24nLCAuLi5maWVsZHMpLCAnYXBwc3luYzpHcmFwaFFMJyk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiBJQU0gcG9saWN5IHN0YXRlbWVudCBmb3IgUXVlcnkgYWNjZXNzIHRvIHRoaXMgR3JhcGhRTEFwaSB0byBhbiBJQU1cbiAgICogcHJpbmNpcGFsJ3MgcG9saWN5LlxuICAgKlxuICAgKiBAcGFyYW0gZ3JhbnRlZSBUaGUgcHJpbmNpcGFsXG4gICAqIEBwYXJhbSBmaWVsZHMgVGhlIGZpZWxkcyB0byBncmFudCBhY2Nlc3MgdG8gdGhhdCBhcmUgUXVlcmllcyAobGVhdmUgYmxhbmsgZm9yIGFsbClcbiAgICovXG4gIHB1YmxpYyBncmFudFF1ZXJ5KGdyYW50ZWU6IElHcmFudGFibGUsIC4uLmZpZWxkczogc3RyaW5nW10pOiBHcmFudCB7XG4gICAgcmV0dXJuIHRoaXMuZ3JhbnQoZ3JhbnRlZSwgSWFtUmVzb3VyY2Uub2ZUeXBlKCdRdWVyeScsIC4uLmZpZWxkcyksICdhcHBzeW5jOkdyYXBoUUwnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGFuIElBTSBwb2xpY3kgc3RhdGVtZW50IGZvciBTdWJzY3JpcHRpb24gYWNjZXNzIHRvIHRoaXMgR3JhcGhRTEFwaSB0byBhbiBJQU1cbiAgICogcHJpbmNpcGFsJ3MgcG9saWN5LlxuICAgKlxuICAgKiBAcGFyYW0gZ3JhbnRlZSBUaGUgcHJpbmNpcGFsXG4gICAqIEBwYXJhbSBmaWVsZHMgVGhlIGZpZWxkcyB0byBncmFudCBhY2Nlc3MgdG8gdGhhdCBhcmUgU3Vic2NyaXB0aW9ucyAobGVhdmUgYmxhbmsgZm9yIGFsbClcbiAgICovXG4gIHB1YmxpYyBncmFudFN1YnNjcmlwdGlvbihncmFudGVlOiBJR3JhbnRhYmxlLCAuLi5maWVsZHM6IHN0cmluZ1tdKTogR3JhbnQge1xuICAgIHJldHVybiB0aGlzLmdyYW50KGdyYW50ZWUsIElhbVJlc291cmNlLm9mVHlwZSgnU3Vic2NyaXB0aW9uJywgLi4uZmllbGRzKSwgJ2FwcHN5bmM6R3JhcGhRTCcpO1xuICB9XG5cbiAgcHJpdmF0ZSB2YWxpZGF0ZUF1dGhvcml6YXRpb25Qcm9wcyhtb2RlczogQXV0aG9yaXphdGlvbk1vZGVbXSkge1xuICAgIGlmIChtb2Rlcy5maWx0ZXIoKG1vZGUpID0+IG1vZGUuYXV0aG9yaXphdGlvblR5cGUgPT09IEF1dGhvcml6YXRpb25UeXBlLkxBTUJEQSkubGVuZ3RoID4gMSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgY2FuIG9ubHkgaGF2ZSBhIHNpbmdsZSBBV1MgTGFtYmRhIGZ1bmN0aW9uIGNvbmZpZ3VyZWQgdG8gYXV0aG9yaXplIHlvdXIgQVBJLicpO1xuICAgIH1cbiAgICBtb2Rlcy5tYXAoKG1vZGUpID0+IHtcbiAgICAgIGlmIChtb2RlLmF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5PSURDICYmICFtb2RlLm9wZW5JZENvbm5lY3RDb25maWcpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdNaXNzaW5nIE9JREMgQ29uZmlndXJhdGlvbicpO1xuICAgICAgfVxuICAgICAgaWYgKG1vZGUuYXV0aG9yaXphdGlvblR5cGUgPT09IEF1dGhvcml6YXRpb25UeXBlLlVTRVJfUE9PTCAmJiAhbW9kZS51c2VyUG9vbENvbmZpZykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ01pc3NpbmcgVXNlciBQb29sIENvbmZpZ3VyYXRpb24nKTtcbiAgICAgIH1cbiAgICAgIGlmIChtb2RlLmF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5MQU1CREEgJiYgIW1vZGUubGFtYmRhQXV0aG9yaXplckNvbmZpZykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ01pc3NpbmcgTGFtYmRhIENvbmZpZ3VyYXRpb24nKTtcbiAgICAgIH1cbiAgICB9KTtcbiAgICBpZiAobW9kZXMuZmlsdGVyKChtb2RlKSA9PiBtb2RlLmF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5BUElfS0VZKS5sZW5ndGggPiAxKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBjYW5cXCd0IGR1cGxpY2F0ZSBBUElfS0VZIGNvbmZpZ3VyYXRpb24uIFNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvZGV2Z3VpZGUvc2VjdXJpdHkuaHRtbCcpO1xuICAgIH1cbiAgICBpZiAobW9kZXMuZmlsdGVyKChtb2RlKSA9PiBtb2RlLmF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5JQU0pLmxlbmd0aCA+IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignWW91IGNhblxcJ3QgZHVwbGljYXRlIElBTSBjb25maWd1cmF0aW9uLiBTZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L2Rldmd1aWRlL3NlY3VyaXR5Lmh0bWwnKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQWRkIHNjaGVtYSBkZXBlbmRlbmN5IHRvIGEgZ2l2ZW4gY29uc3RydWN0XG4gICAqXG4gICAqIEBwYXJhbSBjb25zdHJ1Y3QgdGhlIGRlcGVuZGVlXG4gICAqL1xuICBwdWJsaWMgYWRkU2NoZW1hRGVwZW5kZW5jeShjb25zdHJ1Y3Q6IENmblJlc291cmNlKTogYm9vbGVhbiB7XG4gICAgY29uc3RydWN0LmFkZERlcGVuZHNPbih0aGlzLnNjaGVtYVJlc291cmNlKTtcbiAgICByZXR1cm4gdHJ1ZTtcbiAgfVxuXG4gIHByaXZhdGUgc2V0dXBMb2dDb25maWcoY29uZmlnPzogTG9nQ29uZmlnKSB7XG4gICAgaWYgKCFjb25maWcpIHJldHVybiB1bmRlZmluZWQ7XG4gICAgY29uc3QgbG9nc1JvbGVBcm46IHN0cmluZyA9IGNvbmZpZy5yb2xlPy5yb2xlQXJuID8/IG5ldyBSb2xlKHRoaXMsICdBcGlMb2dzUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IFNlcnZpY2VQcmluY2lwYWwoJ2FwcHN5bmMuYW1hem9uYXdzLmNvbScpLFxuICAgICAgbWFuYWdlZFBvbGljaWVzOiBbXG4gICAgICAgIE1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQVdTQXBwU3luY1B1c2hUb0Nsb3VkV2F0Y2hMb2dzJyksXG4gICAgICBdLFxuICAgIH0pLnJvbGVBcm47XG4gICAgcmV0dXJuIHtcbiAgICAgIGNsb3VkV2F0Y2hMb2dzUm9sZUFybjogbG9nc1JvbGVBcm4sXG4gICAgICBleGNsdWRlVmVyYm9zZUNvbnRlbnQ6IGNvbmZpZy5leGNsdWRlVmVyYm9zZUNvbnRlbnQsXG4gICAgICBmaWVsZExvZ0xldmVsOiBjb25maWcuZmllbGRMb2dMZXZlbCxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBzZXR1cE9wZW5JZENvbm5lY3RDb25maWcoY29uZmlnPzogT3BlbklkQ29ubmVjdENvbmZpZykge1xuICAgIGlmICghY29uZmlnKSByZXR1cm4gdW5kZWZpbmVkO1xuICAgIHJldHVybiB7XG4gICAgICBhdXRoVHRsOiBjb25maWcudG9rZW5FeHBpcnlGcm9tQXV0aCxcbiAgICAgIGNsaWVudElkOiBjb25maWcuY2xpZW50SWQsXG4gICAgICBpYXRUdGw6IGNvbmZpZy50b2tlbkV4cGlyeUZyb21Jc3N1ZSxcbiAgICAgIGlzc3VlcjogY29uZmlnLm9pZGNQcm92aWRlcixcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBzZXR1cFVzZXJQb29sQ29uZmlnKGNvbmZpZz86IFVzZXJQb29sQ29uZmlnKSB7XG4gICAgaWYgKCFjb25maWcpIHJldHVybiB1bmRlZmluZWQ7XG4gICAgcmV0dXJuIHtcbiAgICAgIHVzZXJQb29sSWQ6IGNvbmZpZy51c2VyUG9vbC51c2VyUG9vbElkLFxuICAgICAgYXdzUmVnaW9uOiBjb25maWcudXNlclBvb2wuc3RhY2sucmVnaW9uLFxuICAgICAgYXBwSWRDbGllbnRSZWdleDogY29uZmlnLmFwcElkQ2xpZW50UmVnZXgsXG4gICAgICBkZWZhdWx0QWN0aW9uOiBjb25maWcuZGVmYXVsdEFjdGlvbiB8fCBVc2VyUG9vbERlZmF1bHRBY3Rpb24uQUxMT1csXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgc2V0dXBMYW1iZGFBdXRob3JpemVyQ29uZmlnKGNvbmZpZz86IExhbWJkYUF1dGhvcml6ZXJDb25maWcpIHtcbiAgICBpZiAoIWNvbmZpZykgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICByZXR1cm4ge1xuICAgICAgYXV0aG9yaXplclJlc3VsdFR0bEluU2Vjb25kczogY29uZmlnLnJlc3VsdHNDYWNoZVR0bD8udG9TZWNvbmRzKCksXG4gICAgICBhdXRob3JpemVyVXJpOiBjb25maWcuaGFuZGxlci5mdW5jdGlvbkFybixcbiAgICAgIGlkZW50aXR5VmFsaWRhdGlvbkV4cHJlc3Npb246IGNvbmZpZy52YWxpZGF0aW9uUmVnZXgsXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgc2V0dXBBZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzKG1vZGVzPzogQXV0aG9yaXphdGlvbk1vZGVbXSkge1xuICAgIGlmICghbW9kZXMgfHwgbW9kZXMubGVuZ3RoID09PSAwKSByZXR1cm4gdW5kZWZpbmVkO1xuICAgIHJldHVybiBtb2Rlcy5yZWR1Y2U8Q2ZuR3JhcGhRTEFwaS5BZGRpdGlvbmFsQXV0aGVudGljYXRpb25Qcm92aWRlclByb3BlcnR5W10+KChhY2MsIG1vZGUpID0+IFtcbiAgICAgIC4uLmFjYywge1xuICAgICAgICBhdXRoZW50aWNhdGlvblR5cGU6IG1vZGUuYXV0aG9yaXphdGlvblR5cGUsXG4gICAgICAgIHVzZXJQb29sQ29uZmlnOiB0aGlzLnNldHVwVXNlclBvb2xDb25maWcobW9kZS51c2VyUG9vbENvbmZpZyksXG4gICAgICAgIG9wZW5JZENvbm5lY3RDb25maWc6IHRoaXMuc2V0dXBPcGVuSWRDb25uZWN0Q29uZmlnKG1vZGUub3BlbklkQ29ubmVjdENvbmZpZyksXG4gICAgICAgIGxhbWJkYUF1dGhvcml6ZXJDb25maWc6IHRoaXMuc2V0dXBMYW1iZGFBdXRob3JpemVyQ29uZmlnKG1vZGUubGFtYmRhQXV0aG9yaXplckNvbmZpZyksXG4gICAgICB9LFxuICAgIF0sIFtdKTtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlQVBJS2V5KGNvbmZpZz86IEFwaUtleUNvbmZpZykge1xuICAgIGlmIChjb25maWc/LmV4cGlyZXM/LmlzQmVmb3JlKER1cmF0aW9uLmRheXMoMSkpIHx8IGNvbmZpZz8uZXhwaXJlcz8uaXNBZnRlcihEdXJhdGlvbi5kYXlzKDM2NSkpKSB7XG4gICAgICB0aHJvdyBFcnJvcignQVBJIGtleSBleHBpcmF0aW9uIG11c3QgYmUgYmV0d2VlbiAxIGFuZCAzNjUgZGF5cy4nKTtcbiAgICB9XG4gICAgY29uc3QgZXhwaXJlcyA9IGNvbmZpZz8uZXhwaXJlcyA/IGNvbmZpZz8uZXhwaXJlcy50b0Vwb2NoKCkgOiB1bmRlZmluZWQ7XG4gICAgcmV0dXJuIG5ldyBDZm5BcGlLZXkodGhpcywgYCR7Y29uZmlnPy5uYW1lIHx8ICdEZWZhdWx0J31BcGlLZXlgLCB7XG4gICAgICBleHBpcmVzLFxuICAgICAgZGVzY3JpcHRpb246IGNvbmZpZz8uZGVzY3JpcHRpb24sXG4gICAgICBhcGlJZDogdGhpcy5hcGlJZCxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFc2NhcGUgaGF0Y2ggdG8gYXBwZW5kIHRvIFNjaGVtYSBhcyBkZXNpcmVkLiBXaWxsIGFsd2F5cyByZXN1bHRcbiAgICogaW4gYSBuZXdsaW5lLlxuICAgKlxuICAgKiBAcGFyYW0gYWRkaXRpb24gdGhlIGFkZGl0aW9uIHRvIGFkZCB0byBzY2hlbWFcbiAgICogQHBhcmFtIGRlbGltaXRlciB0aGUgZGVsaW1pdGVyIGJldHdlZW4gc2NoZW1hIGFuZCBhZGRpdGlvblxuICAgKiBAZGVmYXVsdCAtICcnXG4gICAqXG4gICAqL1xuICBwdWJsaWMgYWRkVG9TY2hlbWEoYWRkaXRpb246IHN0cmluZywgZGVsaW1pdGVyPzogc3RyaW5nKTogdm9pZCB7XG4gICAgdGhpcy5zY2hlbWEuYWRkVG9TY2hlbWEoYWRkaXRpb24sIGRlbGltaXRlcik7XG4gIH1cblxuICAvKipcbiAgICogQWRkIHR5cGUgdG8gdGhlIHNjaGVtYVxuICAgKlxuICAgKiBAcGFyYW0gdHlwZSB0aGUgaW50ZXJtZWRpYXRlIHR5cGUgdG8gYWRkIHRvIHRoZSBzY2hlbWFcbiAgICpcbiAgICovXG4gIHB1YmxpYyBhZGRUeXBlKHR5cGU6IElJbnRlcm1lZGlhdGVUeXBlKTogSUludGVybWVkaWF0ZVR5cGUge1xuICAgIHJldHVybiB0aGlzLnNjaGVtYS5hZGRUeXBlKHR5cGUpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIHF1ZXJ5IGZpZWxkIHRvIHRoZSBzY2hlbWEncyBRdWVyeS4gQ0RLIHdpbGwgY3JlYXRlIGFuXG4gICAqIE9iamVjdCBUeXBlIGNhbGxlZCAnUXVlcnknLiBGb3IgZXhhbXBsZSxcbiAgICpcbiAgICogdHlwZSBRdWVyeSB7XG4gICAqICAgZmllbGROYW1lOiBGaWVsZC5yZXR1cm5UeXBlXG4gICAqIH1cbiAgICpcbiAgICogQHBhcmFtIGZpZWxkTmFtZSB0aGUgbmFtZSBvZiB0aGUgcXVlcnlcbiAgICogQHBhcmFtIGZpZWxkIHRoZSByZXNvbHZhYmxlIGZpZWxkIHRvIGZvciB0aGlzIHF1ZXJ5XG4gICAqL1xuICBwdWJsaWMgYWRkUXVlcnkoZmllbGROYW1lOiBzdHJpbmcsIGZpZWxkOiBSZXNvbHZhYmxlRmllbGQpOiBPYmplY3RUeXBlIHtcbiAgICByZXR1cm4gdGhpcy5zY2hlbWEuYWRkUXVlcnkoZmllbGROYW1lLCBmaWVsZCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgbXV0YXRpb24gZmllbGQgdG8gdGhlIHNjaGVtYSdzIE11dGF0aW9uLiBDREsgd2lsbCBjcmVhdGUgYW5cbiAgICogT2JqZWN0IFR5cGUgY2FsbGVkICdNdXRhdGlvbicuIEZvciBleGFtcGxlLFxuICAgKlxuICAgKiB0eXBlIE11dGF0aW9uIHtcbiAgICogICBmaWVsZE5hbWU6IEZpZWxkLnJldHVyblR5cGVcbiAgICogfVxuICAgKlxuICAgKiBAcGFyYW0gZmllbGROYW1lIHRoZSBuYW1lIG9mIHRoZSBNdXRhdGlvblxuICAgKiBAcGFyYW0gZmllbGQgdGhlIHJlc29sdmFibGUgZmllbGQgdG8gZm9yIHRoaXMgTXV0YXRpb25cbiAgICovXG4gIHB1YmxpYyBhZGRNdXRhdGlvbihmaWVsZE5hbWU6IHN0cmluZywgZmllbGQ6IFJlc29sdmFibGVGaWVsZCk6IE9iamVjdFR5cGUge1xuICAgIHJldHVybiB0aGlzLnNjaGVtYS5hZGRNdXRhdGlvbihmaWVsZE5hbWUsIGZpZWxkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBzdWJzY3JpcHRpb24gZmllbGQgdG8gdGhlIHNjaGVtYSdzIFN1YnNjcmlwdGlvbi4gQ0RLIHdpbGwgY3JlYXRlIGFuXG4gICAqIE9iamVjdCBUeXBlIGNhbGxlZCAnU3Vic2NyaXB0aW9uJy4gRm9yIGV4YW1wbGUsXG4gICAqXG4gICAqIHR5cGUgU3Vic2NyaXB0aW9uIHtcbiAgICogICBmaWVsZE5hbWU6IEZpZWxkLnJldHVyblR5cGVcbiAgICogfVxuICAgKlxuICAgKiBAcGFyYW0gZmllbGROYW1lIHRoZSBuYW1lIG9mIHRoZSBTdWJzY3JpcHRpb25cbiAgICogQHBhcmFtIGZpZWxkIHRoZSByZXNvbHZhYmxlIGZpZWxkIHRvIGZvciB0aGlzIFN1YnNjcmlwdGlvblxuICAgKi9cbiAgcHVibGljIGFkZFN1YnNjcmlwdGlvbihmaWVsZE5hbWU6IHN0cmluZywgZmllbGQ6IFJlc29sdmFibGVGaWVsZCk6IE9iamVjdFR5cGUge1xuICAgIHJldHVybiB0aGlzLnNjaGVtYS5hZGRTdWJzY3JpcHRpb24oZmllbGROYW1lLCBmaWVsZCk7XG4gIH1cbn1cbiJdfQ==