"""Serial Functions.

The following functions give you access to read and write through serial
ports.
"""

__all__ = [
    "closeSerialPort",
    "configureSerialPort",
    "openSerialPort",
    "readBytes",
    "readBytesAsString",
    "readLine",
    "readUntil",
    "sendBreak",
    "write",
    "writeBytes",
]

import pprint

from com.inductiveautomation.ignition.modules.serial.scripting import (
    SerialScriptModule,
)

# Bit rate constants.
BIT_RATE_110 = 110
BIT_RATE_150 = 150
BIT_RATE_300 = 300
BIT_RATE_600 = 600
BIT_RATE_1200 = 1200
BIT_RATE_2400 = 2400
BIT_RATE_4800 = 4800
BIT_RATE_9600 = 9600
BIT_RATE_19200 = 19200
BIT_RATE_38400 = 38400
BIT_RATE_57600 = 57600
BIT_RATE_115200 = 115200
BIT_RATE_230400 = 230400
BIT_RATE_460800 = 460800
BIT_RATE_921600 = 921600

# Data bits constants.
DATA_BITS_5 = 5
DATA_BITS_6 = 6
DATA_BITS_7 = 7
DATA_BITS_8 = 8

# Handshake constants.
HANDSHAKE_CTS_DTR = 4112
HANDSHAKE_CTS_RTS = 17
HANDSHAKE_DSR_DTR = 4352
HANDSHAKE_HARD_IN = 0
HANDSHAKE_HARD_OUT = 0
HANDSHAKE_NONE = 0
HANDSHAKE_SOFT_IN = 65536
HANDSHAKE_SOFT_OUT = 1048576
HANDSHAKE_SPLIT_MASK = 0
HANDSHAKE_XON_XOFF = 1114112

# Parity constants.
PARITY_EVEN = 2
PARITY_ODD = 1
PARITY_MARK = 3
PARITY_SPACE = 4
PARITY_NONE = 0

# Stop bits constants.
STOP_BITS_1 = 1
STOP_BITS_2 = 3


def closeSerialPort(port):
    """Closes a previously opened serial port.

    Returns without doing anything if the named serial port is not
    currently open. Will throw an exception if the port is open and
    cannot be closed.

    Args:
        port (str): The name of the serial port, e.g., "COM1" or
            "dev/ttyS0".
    """
    pprint.pprint(port)


def configureSerialPort(
    port, bitRate, dataBits, handshake, hardwareFlowControl, parity, stopBits
):
    """Configure a serial port for use in a later call.

    This only needs to be done once unless the configuration has changed
    after the initial call. All access to constants must be prefixed by
    "system.serial.".

    Args:
        port (str): The name of the serial port, e.g., "COM1" or
            "/dev/ttyS0". This parameter is required.
        bitRate (int): Configure the bit rate. Valid values are defined
            by the following constants: BIT_RATE_110, BIT_RATE_150,
            BIT_RATE_300, BIT_RATE_600, BIT_RATE_1200, BIT_RATE_2400,
            BIT_RATE_4800, BIT_RATE_9600, BIT_RATE_19200,
            BIT_RATE_38400, BIT_RATE_57600, BIT_RATE_115200,
            BIT_RATE_230400, BIT_RATE_460800, BIT_RATE_921600.
        dataBits (int): Configure the data bits. Valid values are
            defined by the following constants: DATA_BITS_5,
            DATA_BITS_6, DATA_BITS_7, DATA_BITS_8.
        handshake (int): Configure the handshake. Valid values are
            defined by the following constants: HANDSHAKE_CTS_DTR,
            HANDSHAKE_CTS_RTS, HANDSHAKE_DSR_DTR, HANDSHAKE_HARD_IN,
            HANDSHAKE_HARD_OUT, HANDSHAKE_NONE, HANDSHAKE_SOFT_IN,
            HANDSHAKE_SOFT_OUT, HANDSHAKE_SPLIT_MASK,
            HANDSHAKE_XON_XOFF.
        hardwareFlowControl (bool): Configure hardware flow control. On
            or off.
        parity (int): Configure parity. Valid values are defined by the
            following constants: PARITY_EVEN, PARITY_ODD, PARITY_MARK,
            PARITY_SPACE, PARITY_NONE.
        stopBits (int): Configure stop bits. Valid values are defined by
            the following constants: STOP_BITS_1, STOP_BITS_2.

    Returns:
        SerialConfigurator: A SerialConfigurator that can be used to
            configure the serial port instead of or in addition to the
            given keyword arguments.
    """
    pprint.pprint(
        [
            port,
            bitRate,
            dataBits,
            handshake,
            hardwareFlowControl,
            parity,
            stopBits,
        ]
    )
    return SerialScriptModule.SerialConfigurator()


def openSerialPort(port):
    """Opens a previously configured serial port for use.

    Will throw an exception if the serial port cannot be opened.

    Args:
        port (str): The name of the serial port, e.g., "COM1" or
            "dev/ttyS0".
    """
    pprint.pprint(port)


def readBytes(port, numberOfBytes, timeout=5000):
    """Read numberOfBytes bytes from a serial port.

    Args:
        port (str): The previously configured serial port to use.
        numberOfBytes (int): The number of bytes to read.
        timeout (int): Maximum amount of time, in milliseconds, to block
            before returning. Default is 5000. Optional.

    Returns:
        object: A byte[] containing bytes read from the serial port.
    """
    pprint.pprint([port, numberOfBytes, timeout])


def readBytesAsString(port, numberOfBytes, timeout=5000):
    """Read numberOfBytes bytes from a serial port and convert them to a
    String.

    If a specific encoding is needed to match the source of the
    data, use system.serial.readBytes and use the desired encoding to
    decode the byte array returned.

    Args:
        port (str): The previously configured serial port to use.
        numberOfBytes (int): The number of bytes to read.
        timeout (int): Maximum amount of time, in milliseconds, to block
            before returning. Default is 5000. Optional.

    Returns:
        str: A String created from the bytes read.
    """
    pprint.pprint([port, numberOfBytes, timeout])
    return ""


def readLine(port, timeout=5000, encoding="utf-8"):
    r"""Attempts to read a line from a serial port.

    A "line" is considered to be terminated by either a line feed
    ('\n'), a carriage return ('\r'), or a carriage return followed
    immediately by a line feed.

    The function will wait until the timeout period for a terminator. If
    the timeout is reached before the line is properly terminated, then
    the buffer will be dumped, possibly resulting in data loss.

    Args:
        port (str): The previously configured serial port to use.
        timeout (int): Maximum amount of time, in milliseconds, to block
            before returning. Default is 5000. Optional.
        encoding (str): The String encoding to use. Default is UTF8.
            Optional.

    Returns:
        str: A line of text.
    """
    pprint.pprint([port, timeout, encoding])
    return ""


def readUntil(port, delimiter, includeDelimiter, timeout=5000):
    """Reads a byte at a time from a serial port until a delimiter
    character is encountered.

    The read will block for up to timeout milliseconds before returning.

    If the delimiter is not found before the timeout period, then the
    buffer will dump, potentially resulting in data loss.

    Args:
        port (str): The previously configured serial port to use.
        delimiter (str): The delimiter to read until.
        includeDelimiter (bool): If True, the delimiter will be included
            in the return value.
        timeout (int): Optional timeout in milliseconds. Default is
            5000.

    Returns:
        str: Returns a String containing all 8-bit ASCII characters read
            until the delimiter was reached, and including the delimiter
            if the "includeDelimiter" parameter was True.
    """
    pprint.pprint([port, delimiter, includeDelimiter, timeout])
    return ""


def sendBreak(port, millis):
    """Sends a break signal for approximately millis milliseconds.

    Args:
        port (str): The name of the serial port, e.g., "COM1" or
            "dev/ttyS0".
        millis (int): Approximate length of break signal, in
            milliseconds.
    """
    pprint.pprint([port, millis])


def write(port, toWrite):
    """Write a String to a serial port using the platforms default
    character encoding.

    Args:
        port (str): The previously configured serial port to use.
        toWrite (str): The String to write.
    """
    pprint.pprint([port, toWrite])


def writeBytes(port, toWrite):
    """Write a byte[] to a serial port.

    Args:
        port (str): The previously configured serial port to use.
        toWrite (object): The byte[] to write.
    """
    pprint.pprint([port, toWrite])
