# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['notion_nlp', 'notion_nlp.core', 'notion_nlp.parameter']

package_data = \
{'': ['*']}

install_requires = \
['arrow==1.2.3',
 'jieba==0.42.1',
 'matplotlib==3.7.0',
 'pandas==1.5.3',
 'pydantic==1.10.5',
 'pyfunctional==1.4.3',
 'requests==2.28.2',
 'ruamel-yaml==0.17.21',
 'scikit-learn==1.2.1',
 'tabulate==0.9.0',
 'tqdm==4.64.1',
 'typer==0.7.0',
 'wcwidth==0.2.6',
 'wordcloud==1.8.2.2']

setup_kwargs = {
    'name': 'notion-nlp',
    'version': '1.0.4',
    'description': 'Reading rich text information from a Notion database and performing simple NLP analysis.',
    'long_description': '<p align="center">\n  <img width="100px" src="https://img.icons8.com/ios/250/FFFFFF/share-2.png" align="center" alt="Notion Rich Text Data Analysis" />\n  <h1 align="center">\n    Notion NLP\n  </h1>\n  <p align="center">\n    To read text from a Notion database and perform natural language processing analysis.\n  </p>\n</p>\n\n  <p align="center">\n    <a href="https://github.com/dario-github/notion-nlp/actions">\n      <img alt="Tests Passing" src="https://github.com/dario-github/notion-nlp/actions/workflows/main.yml/badge.svg" />\n    </a>\n    <a href="https://codecov.io/gh/dario-github/notion-nlp">\n      <img alt="codecov" src="https://codecov.io/gh/dario-github/notion-nlp/branch/main/graph/badge.svg?token=ehzYhousD3" />\n    </a>\n    <a href="https://github.com/dario-github/notion-nlp/graphs/contributors">\n      <img alt="GitHub Contributors" src="https://img.shields.io/github/contributors/dario-github/notion-nlp" />\n    </a>\n    <a href="https://visitorbadge.io/status?path=https%3A%2F%2Fgithub.com%2Fdario-github%2Fnotion-nlp">\n      <img alt="visitors" src="https://api.visitorbadge.io/api/visitors?path=https%3A%2F%2Fgithub.com%2Fdario-github%2Fnotion-nlp&countColor=%2337d67a&style=flat" />\n    </a>\n  </p>\n  \n  <p align="center">\n    <a href="README.md">English</a>\n    /\n    <a href="README.zh.md">简体中文</a>\n  </p>\n\n## Introduction\n\nTo achieve functionality similar to flomo, I have created a database using Notion, where I have recorded my thoughts and insights over the years, accumulating a rich corpus of language. However, the random roaming feature of flomo did not meet my needs, so I decided to develop a small tool that integrates with the Notion API and performs NLP analysis.\n\nNow, the tool can:\n\n- Output intuitive and visually appealing word cloud images.\n  \n  ![Harry Potter\'s Story Beginning](./docs/pictures/colormap_cividis.en.png)\n\n- Generate thematic summaries of your Notion notes.\n\n  ===> [Example thematic summary]((./results/en_unit_testing_task.tf_idf.analysis_result.top5_word_with_sentences.md))\n\n- Support multiple languages. I have added stopword lists for several languages including Chinese, English, Russian, French, Japanese, and German. Users can also customize their own stopword lists.\n\n  ===> [Stopword lists for multiple languages](./resources/stopwords/)\n\n- Support multiple tasks. Users can configure multiple databases and corresponding filtering and sorting conditions to create rich analysis tasks.\n\n  ===> [Example configuration file](./configs/config.sample.yaml)\n\n  For example, I have added the following tasks:\n\n    - Reflections from the past year\n    - Optimization of annual summaries for the current year\n    - Self-admonitions from all time periods\n  \nI am pleased to share this tool and hope it can be helpful to you. :laughing:\n\n## Pipline\n\n<div style="text-align:center;">\n\n```mermaid\nflowchart TB\n\nA[(Notion Database)] --> B([read rich text via API]) --> C([split word / cleaning / word-phrase mapping]) --> D[/calculate TF-IDF/] --> E[[Output the top-n keywords and their corresponding sentences in markdown format]]\n```\n\n</div>\n\n## Installation\n\n```shell\npython3.8 -m pip install notion-nlp\n```\n\n## Quick use\n\nConfiguration file reference ``configs/config.sample.yaml`` (hereinafter config, please rename to ``config.yaml`` as your own configuration file)\n\n### Get the integration token\n\nIn [notion integrations](https://www.notion.so/my-integrations/) create a new integration, get your own token and fill in the token in the config.yaml file afterwards.\n\n> Graphic Tutorial: [tango](https://app.tango.us/app/workflow/6e53c348-79b6-4ed3-8c75-46f5ddb996da?utm_source=markdown&utm_medium=markdown&utm_campaign=workflow%20export%20links) / [markdown](./docs/tango/get_the_integration_token.md)\n\n### Add integration to database/get database ID\n\nIf you open the notion database page in your browser or click on the share copy link, you will see the database id in the address link (similar to a string of jumbles) and fill in the database_id under the task of config.\n\n> Graphic Tutorial: [tango](https://app.tango.us/app/workflow/7e95c7df-af73-4748-9bf7-11efc8e24f2a?utm_source=markdown&utm_medium=markdown&utm_campaign=workflow%20export%20links) / [markdown](./docs/tango/add_integration_to_database.md)\n\n### Configure the filter sort database entry extra parameter\n\nThe task\'s extra is used to filter and sort the database, see [notion filter API](https://developers.notion.com/reference/post-database-query-filter#property-filter-object) for format and content, the [config.sample.yaml](./configs/config.sample.yaml) file already provides 2 configurations.\n\n### Run all tasks\n\n```Shell\n# Run from command line\npython3.8 -m notion_nlp run-all-tasks --config-file /path/to/your/config/file\n```\n\n```Python\n# Run from Python code\nfrom notion_nlp import run_all_tasks\nconfig_file = "./configs/config.yaml"\nrun_all_tasks(config_file)\n```\n\n### Run a single task\n\nIn the `run_task` command, you can specify the task in several ways, including:\n\n- `task`: an instance of `TaskParams`;\n- `task_json`: a JSON string representing the task information;\n- `task_name`: the name of the task.\n\nIf `config_file` exists, you can use `task_name` to specify the task. Note that the task needs to be activated, otherwise an exception will be thrown. If `config_file` does not exist, you need to provide a `token` and either `TaskParams` or `task_json`.\n\n#### With an existing `config` file, pass in `task name`/`task json`/`task parameter class`\n\n```shell\n# Option 1\npython3.8 -m notion_nlp run-task --task-name task_1 --config-file /path/to/your/config/file\n\n# Option 2\npython3.8 -m notion_nlp run-task --task-json \'{"name": "task_1", "database_id": "your_database_id"}\' --config-file /path/to/your/config/file\n```\n\n```python\nfrom notion_nlp import run_task\ntask_name = "task_1"\ndatabase_id = "your_database_id"\nconfig_file="./configs/config.yaml"\n\n# Option 1\nrun_task(task_name=task_name, config_file=config_file)\n\n# Option 2 (not recommended for Python code)\nimport json\ntask_info = {"name": task_name, "database_id": database_id}\nrun_task(task_json=json.dumps(task_info, ensure_ascii=False), config_file=config_file)\n\n# Option 3 (recommended)\nfrom notion_nlp.parameter.config import TaskParams\ntask = TaskParams(name=task_name, database_id=database_id)\nrun_task(task=task, config_file=config_file)\n```\n\n#### Without a `config` file, pass in `token` and `task json`/`task parameter class`\n\n```shell\n# Option 1\npython3.8 -m notion_nlp run-task --task-json \'{"name": "task_1", "database_id": "your_database_id"}\' --token \'your_notion_integration_token\'\n```\n\n```python\nfrom notion_nlp import run_task\ntask_name = "task_1"\ndatabase_id = "your_database_id"\nnotion_token = "your_notion_integration_token"\n\n# Option 1 (not recommended for Python code)\nimport json\ntask_info = {"name": task_name, "database_id": database_id}\nrun_task(task_json=json.dumps(task_info, ensure_ascii=False), token=notion_token)\n\n# Option 2 (recommended)\nfrom notion_nlp.parameter.config import TaskParams\ntask = TaskParams(name=task_name, database_id=database_id)\nrun_task(task=task, token=notion_token)\n```\n\n## Development\n\nWelcome to fork and add new features/fix bugs.\n\n- After cloning the project, use the `create_python_env_in_new_machine.sh` script to create a Poetry virtual environment.\n\n- After completing the code development, use the invoke command to perform a series of formatting tasks, including black/isort tasks added in task.py.\n  \n    ```shell\n    invoke check\n    ```\n\n- After submitting the formatted changes, run unit tests to check coverage.\n\n    ```shell\n    poetry run tox\n\n    ```\n\n## Note\n\n- The word segmentation tool has two built-in options: jieba/pkuseg. (Considering adding language analysis to automatically select the most suitable word segmentation tool for that language.)\n\n  - jieba is used by default.\n  - pkuseg cannot be installed with poetry and needs to be installed manually with pip. In addition, this library is slow and requires high memory usage. It has been tested that a VPS with less than 1G memory needs to load virtual memory to use it.\n\n- The analysis method using tf-idf is too simple. Consider integrating the API of LLM (such as chatGPT) for further analysis.\n\n## Contributions\n\n- scikit-learn - [https://github.com/scikit-learn/scikit-learn](https://github.com/scikit-learn/scikit-learn)\n- Alir3z4/stop-words - [https://github.com/Alir3z4/stop-words](https://github.com/Alir3z4/stop-words)\n\n## License and Copyright\n\n- [MIT License](./LICENSE)\n  - The MIT License is a permissive open-source software license. This means that anyone is free to use, copy, modify, and distribute your software, as long as they include the original copyright notice and license in their derivative works.\n\n  - However, the MIT License comes with no warranty or liability, meaning that you cannot be held liable for any damages or losses arising from the use or distribution of your software.\n\n  - By using this software, you agree to the terms and conditions of the MIT License.\n\n## Contact information\n\n- See more at my [HomePage](https://github.com/dario-github)\n',
    'author': 'Dario Zhang',
    'author_email': 'zdclink@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8.10,<3.9.0',
}


setup(**setup_kwargs)
