"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const fs_1 = require("fs");
const appsync_generated_1 = require("./appsync.generated");
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
function isUserPoolConfig(obj) {
    return obj.userPool !== undefined;
}
function isApiKeyConfig(obj) {
    return obj.apiKeyDesc !== undefined;
}
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * An AppSync GraphQL API
 */
class GraphQLApi extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let apiLogsRole;
        if (props.logConfig) {
            apiLogsRole = new aws_iam_1.Role(this, 'ApiLogsRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
            apiLogsRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'));
        }
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: 'API_KEY',
            ...props.logConfig && {
                logConfig: {
                    cloudWatchLogsRoleArn: apiLogsRole ? apiLogsRole.roleArn : undefined,
                    excludeVerboseContent: props.logConfig.excludeVerboseContent,
                    fieldLogLevel: props.logConfig.fieldLogLevel ? props.logConfig.fieldLogLevel.toString() : undefined,
                },
            }
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphQlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        if (props.authorizationConfig) {
            this.setupAuth(props.authorizationConfig);
        }
        let definition;
        if (props.schemaDefinition) {
            definition = props.schemaDefinition;
        }
        else if (props.schemaDefinitionFile) {
            definition = fs_1.readFileSync(props.schemaDefinitionFile).toString('UTF-8');
        }
        else {
            throw new Error('Missing Schema definition. Provide schemaDefinition or schemaDefinitionFile');
        }
        this.schema = new appsync_generated_1.CfnGraphQLSchema(this, 'Schema', {
            apiId: this.apiId,
            definition,
        });
    }
    /**
     * add a new DynamoDB data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param table The DynamoDB table backing this data source [disable-awslint:ref-via-interface]
     */
    addDynamoDbDataSource(name, description, table) {
        return new DynamoDbDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            table
        });
    }
    /**
     * add a new Lambda data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param lambdaFunction The Lambda function to call to interact with this data source
     */
    addLambdaDataSource(name, description, lambdaFunction) {
        return new LambdaDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            lambdaFunction
        });
    }
    setupAuth(auth) {
        if (isUserPoolConfig(auth.defaultAuthorization)) {
            const { authenticationType, userPoolConfig } = this.userPoolDescFrom(auth.defaultAuthorization);
            this.api.authenticationType = authenticationType;
            this.api.userPoolConfig = userPoolConfig;
        }
        else if (isApiKeyConfig(auth.defaultAuthorization)) {
            this.api.authenticationType = this.apiKeyDesc(auth.defaultAuthorization).authenticationType;
        }
        this.api.additionalAuthenticationProviders = [];
        for (const mode of (auth.additionalAuthorizationModes || [])) {
            if (isUserPoolConfig(mode)) {
                this.api.additionalAuthenticationProviders.push(this.userPoolDescFrom(mode));
            }
            else if (isApiKeyConfig(mode)) {
                this.api.additionalAuthenticationProviders.push(this.apiKeyDesc(mode));
            }
        }
    }
    userPoolDescFrom(upConfig) {
        return {
            authenticationType: 'AMAZON_COGNITO_USER_POOLS',
            userPoolConfig: {
                appIdClientRegex: upConfig.appIdClientRegex,
                userPoolId: upConfig.userPool.userPoolId,
                awsRegion: upConfig.userPool.stack.region,
                defaultAction: upConfig.defaultAction ? upConfig.defaultAction.toString() : 'ALLOW',
            }
        };
    }
    apiKeyDesc(akConfig) {
        let expires;
        if (akConfig.expires) {
            expires = new Date(akConfig.expires).valueOf();
            const now = Date.now();
            const days = (d) => now + core_1.Duration.days(d).toMilliseconds();
            if (expires < days(1) || expires > days(365)) {
                throw Error("API key expiration must be between 1 and 365 days.");
            }
            expires = Math.round(expires / 1000);
        }
        new appsync_generated_1.CfnApiKey(this, `${akConfig.apiKeyDesc || ''}ApiKey`, {
            expires,
            description: akConfig.apiKeyDesc,
            apiId: this.apiId,
        });
        return { authenticationType: 'API_KEY' };
    }
}
exports.GraphQLApi = GraphQLApi;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for concrete datasources
 */
class BaseDataSource extends core_1.Construct {
    constructor(scope, id, props, extended) {
        super(scope, id);
        this.serviceRole = props.serviceRole || new aws_iam_1.Role(this, 'ServiceRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
        this.grantPrincipal = this.serviceRole;
        this.ds = new appsync_generated_1.CfnDataSource(this, 'Resource', {
            apiId: props.api.apiId,
            name: props.name,
            description: props.description,
            serviceRoleArn: this.serviceRole.roleArn,
            ...extended,
        });
        this.name = props.name;
        this.api = props.api;
    }
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props) {
        return new Resolver(this, `${props.typeName}${props.fieldName}Resolver`, {
            api: this.api,
            dataSource: this,
            ...props,
        });
    }
}
exports.BaseDataSource = BaseDataSource;
/**
 * An AppSync datasource backed by a DynamoDB table
 */
class DynamoDbDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AMAZON_DYNAMODB',
            dynamoDbConfig: {
                tableName: props.table.tableName,
                awsRegion: props.table.stack.region,
                useCallerCredentials: props.useCallerCredentials,
            },
        });
        if (props.readOnlyAccess) {
            props.table.grantReadData(this.serviceRole);
        }
        else {
            props.table.grantReadWriteData(this.serviceRole);
        }
    }
}
exports.DynamoDbDataSource = DynamoDbDataSource;
/**
 * An AppSync datasource backed by a Lambda function
 */
class LambdaDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AWS_LAMBDA',
            lambdaConfig: {
                lambdaFunctionArn: props.lambdaFunction.functionArn,
            },
        });
        props.lambdaFunction.grantInvoke(this.serviceRole);
    }
}
exports.LambdaDataSource = LambdaDataSource;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(", ");
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(", ");
    }
}
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
/**
 * Factory class for DynamoDB key conditions.
 */
class KeyCondition {
    constructor(cond) {
        this.cond = cond;
    }
    /**
     * Condition k = arg, true if the key attribute k is equal to the Query argument
     */
    static eq(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '=', arg));
    }
    /**
     * Condition k < arg, true if the key attribute k is less than the Query argument
     */
    static lt(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '<', arg));
    }
    /**
     * Condition k <= arg, true if the key attribute k is less than or equal to the Query argument
     */
    static le(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '<=', arg));
    }
    /**
     * Condition k > arg, true if the key attribute k is greater than the the Query argument
     */
    static gt(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '>', arg));
    }
    /**
     * Condition k >= arg, true if the key attribute k is greater or equal to the Query argument
     */
    static ge(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '>=', arg));
    }
    /**
     * Condition (k, arg). True if the key attribute k begins with the Query argument.
     */
    static beginsWith(keyName, arg) {
        return new KeyCondition(new BeginsWith(keyName, arg));
    }
    /**
     * Condition k BETWEEN arg1 AND arg2, true if k >= arg1 and k <= arg2.
     */
    static between(keyName, arg1, arg2) {
        return new KeyCondition(new Between(keyName, arg1, arg2));
    }
    /**
     * Conjunction between two conditions.
     */
    and(keyCond) {
        return new KeyCondition(this.cond.and(keyCond.cond));
    }
    /**
     * Renders the key condition to a VTL string.
     */
    renderTemplate() {
        return `"query" : {
            "expression" : "${this.cond.renderCondition()}",
            "expressionNames" : {
                ${this.cond.renderExpressionNames()}
            },
            "expressionValues" : {
                ${this.cond.renderExpressionValues()}
            }
        }`;
    }
}
exports.KeyCondition = KeyCondition;
/**
 * Utility class representing the assigment of a value to an attribute.
 */
class Assign {
    constructor(attr, arg) {
        this.attr = attr;
        this.arg = arg;
    }
    /**
     * Renders the assignment as a VTL string.
     */
    renderAsAssignment() {
        return `"${this.attr}" : $util.dynamodb.toDynamoDBJson(${this.arg})`;
    }
    /**
     * Renders the assignment as a map element.
     */
    putInMap(map) {
        return `$util.qr($${map}.put("${this.attr}", "${this.arg}"))`;
    }
}
exports.Assign = Assign;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a partition key.
 */
class PartitionKeyStep {
    constructor(key) {
        this.key = key;
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    is(val) {
        return new PartitionKey(new Assign(this.key, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    auto() {
        return new PartitionKey(new Assign(this.key, '$util.autoId()'));
    }
}
exports.PartitionKeyStep = PartitionKeyStep;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a sort key.
 */
class SortKeyStep {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    is(val) {
        return new PrimaryKey(this.pkey, new Assign(this.skey, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    auto() {
        return new PrimaryKey(this.pkey, new Assign(this.skey, '$util.autoId()'));
    }
}
exports.SortKeyStep = SortKeyStep;
/**
 * Specifies the assignment to the primary key. It either
 * contains the full primary key or only the partition key.
 */
class PrimaryKey {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Allows assigning a value to the partition key.
     */
    static partition(key) {
        return new PartitionKeyStep(key);
    }
    /**
     * Renders the key assignment to a VTL string.
     */
    renderTemplate() {
        const assignments = [this.pkey.renderAsAssignment()];
        if (this.skey) {
            assignments.push(this.skey.renderAsAssignment());
        }
        return `"key" : {
            ${assignments.join(",")}
        }`;
    }
}
exports.PrimaryKey = PrimaryKey;
/**
 * Specifies the assignment to the partition key. It can be
 * enhanced with the assignment of the sort key.
 */
class PartitionKey extends PrimaryKey {
    constructor(pkey) {
        super(pkey);
    }
    /**
     * Allows assigning a value to the sort key.
     */
    sort(key) {
        return new SortKeyStep(this.pkey, key);
    }
}
exports.PartitionKey = PartitionKey;
/**
 * Specifies the attribute value assignments.
 */
class AttributeValues {
    constructor(container, assignments = []) {
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    attribute(attr) {
        return new AttributeValuesStep(attr, this.container, this.assignments);
    }
    /**
     * Renders the attribute value assingments to a VTL string.
     */
    renderTemplate() {
        return `
            #set($input = ${this.container})
            ${this.assignments.map(a => a.putInMap("input")).join("\n")}
            "attributeValues": $util.dynamodb.toMapValuesJson($input)`;
    }
}
exports.AttributeValues = AttributeValues;
/**
 * Utility class to allow assigning a value to an attribute.
 */
class AttributeValuesStep {
    constructor(attr, container, assignments) {
        this.attr = attr;
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Assign the value to the current attribute.
     */
    is(val) {
        this.assignments.push(new Assign(this.attr, val));
        return new AttributeValues(this.container, this.assignments);
    }
}
exports.AttributeValuesStep = AttributeValuesStep;
/**
 * Factory class for attribute value assignments.
 */
class Values {
    /**
     * Treats the specified object as a map of assignments, where the property
     * names represent attribute names. It’s opinionated about how it represents
     * some of the nested objects: e.g., it will use lists (“L”) rather than sets
     * (“SS”, “NS”, “BS”). By default it projects the argument container ("$ctx.args").
     */
    static projecting(arg) {
        return new AttributeValues('$ctx.args' + (arg ? `.${arg}` : ''));
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    static attribute(attr) {
        return new AttributeValues('{}').attribute(attr);
    }
}
exports.Values = Values;
/**
 * MappingTemplates for AppSync resolvers
 */
class MappingTemplate {
    /**
     * Create a mapping template from the given string
     */
    static fromString(template) {
        return new StringMappingTemplate(template);
    }
    /**
     * Create a mapping template from the given file
     */
    static fromFile(fileName) {
        return new StringMappingTemplate(fs_1.readFileSync(fileName).toString('UTF-8'));
    }
    /**
     * Mapping template for a result list from DynamoDB
     */
    static dynamoDbResultList() {
        return this.fromString('$util.toJson($ctx.result.items)');
    }
    /**
     * Mapping template for a single result item from DynamoDB
     */
    static dynamoDbResultItem() {
        return this.fromString('$util.toJson($ctx.result)');
    }
    /**
     * Mapping template to scan a DynamoDB table to fetch all entries
     */
    static dynamoDbScanTable() {
        return this.fromString('{"version" : "2017-02-28", "operation" : "Scan"}');
    }
    /**
     * Mapping template to query a set of items from a DynamoDB table
     *
     * @param cond the key condition for the query
     */
    static dynamoDbQuery(cond) {
        return this.fromString(`{"version" : "2017-02-28", "operation" : "Query", ${cond.renderTemplate()}}`);
    }
    /**
     * Mapping template to get a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Query argument
     */
    static dynamoDbGetItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "GetItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to delete a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Mutation argument
     */
    static dynamoDbDeleteItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "DeleteItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to save a single item to a DynamoDB table
     *
     * @param key the assigment of Mutation values to the primary key
     * @param values the assignment of Mutation values to the table attributes
     */
    static dynamoDbPutItem(key, values) {
        return this.fromString(`{
            "version" : "2017-02-28",
            "operation" : "PutItem",
            ${key.renderTemplate()},
            ${values.renderTemplate()}
        }`);
    }
    /**
     * Mapping template to invoke a Lambda function
     *
     * @param payload the VTL template snippet of the payload to send to the lambda.
     * If no payload is provided all available context fields are sent to the Lambda function
     */
    static lambdaRequest(payload = '$util.toJson($ctx)') {
        return this.fromString(`{"version": "2017-02-28", "operation": "Invoke", "payload": ${payload}}`);
    }
    /**
     * Mapping template to return the Lambda result to the caller
     */
    static lambdaResult() {
        return this.fromString('$util.toJson($ctx.result)');
    }
}
exports.MappingTemplate = MappingTemplate;
class StringMappingTemplate extends MappingTemplate {
    constructor(template) {
        super();
        this.template = template;
    }
    renderTemplate() {
        return this.template;
    }
}
/**
 * An AppSync resolver
 */
class Resolver extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.resolver = new appsync_generated_1.CfnResolver(this, 'Resource', {
            apiId: props.api.apiId,
            typeName: props.typeName,
            fieldName: props.fieldName,
            dataSourceName: props.dataSource ? props.dataSource.name : undefined,
            kind: props.pipelineConfig ? 'PIPELINE' : 'UNIT',
            requestMappingTemplate: props.requestMappingTemplate ? props.requestMappingTemplate.renderTemplate() : undefined,
            responseMappingTemplate: props.responseMappingTemplate ? props.responseMappingTemplate.renderTemplate() : undefined,
        });
        this.resolver.addDependsOn(props.api.schema);
        if (props.dataSource) {
            this.resolver.addDependsOn(props.dataSource.ds);
        }
        this.arn = this.resolver.attrResolverArn;
    }
}
exports.Resolver = Resolver;
//# sourceMappingURL=data:application/json;base64,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