import requests
from . import utils
from codat.models import operations
from typing import Optional

class Webhooks:
    _client: requests.Session
    _security_client: requests.Session
    _server_url: str
    _language: str
    _sdk_version: str
    _gen_version: str

    def __init__(self, client: requests.Session, security_client: requests.Session, server_url: str, language: str, sdk_version: str, gen_version: str) -> None:
        self._client = client
        self._security_client = security_client
        self._server_url = server_url
        self._language = language
        self._sdk_version = sdk_version
        self._gen_version = gen_version

    
    def get_webhook(self, request: operations.GetWebhookRequest) -> operations.GetWebhookResponse:
        r"""Get webhook
        Get a single webhook
        """
        
        base_url = self._server_url
        
        url = utils.generate_url(base_url, "/rules/{ruleId}", request.path_params)
        
        
        client = self._security_client
        
        r = client.request("GET", url)
        content_type = r.headers.get("Content-Type")

        res = operations.GetWebhookResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.GetWebhookWebhook])
                res.webhook = out
        elif r.status_code == 401:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.GetWebhook401ApplicationJSON])
                res.get_webhook_401_application_json_object = out
        elif r.status_code == 404:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.GetWebhook404ApplicationJSON])
                res.get_webhook_404_application_json_object = out

        return res

    
    def list_rules(self, request: operations.ListRulesRequest) -> operations.ListRulesResponse:
        r"""List webhooks
        List webhooks that you are subscribed to.
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/rules"
        
        query_params = utils.get_query_params(request.query_params)
        
        client = self._security_client
        
        r = client.request("GET", url, params=query_params)
        content_type = r.headers.get("Content-Type")

        res = operations.ListRulesResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.ListRulesLinks])
                res.links = out
        elif r.status_code == 400:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.ListRules400ApplicationJSON])
                res.list_rules_400_application_json_object = out
        elif r.status_code == 401:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.ListRules401ApplicationJSON])
                res.list_rules_401_application_json_object = out

        return res

    
    def post_rules(self, request: operations.PostRulesRequest) -> operations.PostRulesResponse:
        r"""Create webhook
        Create a new webhook configuration
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/rules"
        
        headers = {}
        req_content_type, data, form = utils.serialize_request_body(request)
        if req_content_type != "multipart/form-data" and req_content_type != "multipart/mixed":
            headers["content-type"] = req_content_type
        
        client = self._security_client
        
        r = client.request("POST", url, data=data, files=form, headers=headers)
        content_type = r.headers.get("Content-Type")

        res = operations.PostRulesResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.PostRulesWebhook])
                res.webhook = out
        elif r.status_code == 401:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.PostRules401ApplicationJSON])
                res.post_rules_401_application_json_object = out

        return res

    