import getpass
import grp
import logging
import os
import time as ttime
from collections import deque

import h5py
from ophyd.sim import NullStatus
from ophyd.status import SubscriptionStatus

logger = logging.getLogger(__name__)
DEFAULT_DATUM_DICT = {"data": None, "omega": None}


class MXFlyer:
    def __init__(self, vector, zebra, detector=None) -> None:
        self.name = "MXFlyer"
        self.vector = vector
        self.zebra = zebra
        self.detector = detector

        self._asset_docs_cache = deque()
        self._resource_uids = []
        self._datum_counter = None
        self._datum_ids = DEFAULT_DATUM_DICT
        self._master_file = None
        self._master_metadata = []

        self._collection_dictionary = None

    def read_configuration(self):
        return {}

    def describe_configuration(self):
        return {}

    def kickoff(self):
        self.detector.stage()
        self.vector.go.put(1)

        return NullStatus()

    def complete(self):
        def callback_motion(value, old_value, **kwargs):
            print(f"old: {old_value} -> new: {value}")
            if int(round(old_value)) == 1 and int(round(value)) == 0:
                return True
            else:
                return False

        motion_status = SubscriptionStatus(self.vector.active, callback_motion, run=False)
        return motion_status

    def describe_collect(self):
        return_dict = {}
        return_dict["primary"] = {
            f"{self.detector.name}_image": {
                "source": f"{self.detector.name}_data",
                "dtype": "array",
                "shape": [
                    self.detector.cam.num_images.get(),
                    self.detector.cam.array_size.array_size_y.get(),
                    self.detector.cam.array_size.array_size_x.get(),
                ],
                "dims": ["images", "row", "column"],
                "external": "FILESTORE:",
            },
            "omega": {
                "source": f"{self.detector.name}_omega",
                "dtype": "array",
                "shape": [self.detector.cam.num_images.get()],
                "dims": ["images"],
                "external": "FILESTORE:",
            },
        }
        return return_dict

    def collect(self):
        self.unstage()

        now = ttime.time()
        self._master_metadata = self._extract_metadata()
        data = {f"{self.detector.name}_image": self._datum_ids["data"], "omega": self._datum_ids["omega"]}
        yield {
            "data": data,
            "timestamps": {key: now for key in data},
            "time": now,
            "filled": {key: False for key in data},
        }

    # def collect_asset_docs(self):
    #     # items = list(self._asset_docs_cache)
    #     items = list(self.detector.file._asset_docs_cache)
    #     print(f"{print_now()} items:\n{items}")
    #     self.detector.file._asset_docs_cache.clear()
    #     for item in items:
    #         yield item

    def collect_asset_docs(self):

        asset_docs_cache = []

        # Get the Resource which was produced when the detector was staged.
        ((name, resource),) = self.detector.file.collect_asset_docs()

        asset_docs_cache.append(("resource", resource))
        self._datum_ids = DEFAULT_DATUM_DICT
        # Generate Datum documents from scratch here, because the detector was
        # triggered externally by the DeltaTau, never by ophyd.
        resource_uid = resource["uid"]
        # num_points = int(math.ceil(self.detector.cam.num_images.get() /
        #                 self.detector.cam.fw_num_images_per_file.get()))

        # We are currently generating only one datum document for all frames, that's why
        #   we use the 0th index below.
        #
        # Uncomment & update the line below if more datum documents are needed:
        # for i in range(num_points):

        seq_id = self.detector.cam.sequence_id.get()

        self._master_file = f"{resource['root']}/{resource['resource_path']}_{seq_id}_master.h5"
        if not os.path.isfile(self._master_file):
            raise RuntimeError(f"File {self._master_file} does not exist")

        # The pseudocode below is from Tom Caswell explaining the relationship between resource, datum, and events.
        #
        # resource = {
        #     "resource_id": "RES",
        #     "resource_kwargs": {},  # this goes to __init__
        #     "spec": "AD-EIGER-MX",
        #     ...: ...,
        # }
        # datum = {
        #     "datum_id": "a",
        #     "datum_kwargs": {"data_key": "data"},  # this goes to __call__
        #     "resource": "RES",
        #     ...: ...,
        # }
        # datum = {
        #     "datum_id": "b",
        #     "datum_kwargs": {"data_key": "omega"},
        #     "resource": "RES",
        #     ...: ...,
        # }

        # event = {...: ..., "data": {"detector_img": "a", "omega": "b"}}

        for data_key in self._datum_ids.keys():
            datum_id = f"{resource_uid}/{data_key}"
            self._datum_ids[data_key] = datum_id
            datum = {
                "resource": resource_uid,
                "datum_id": datum_id,
                "datum_kwargs": {"data_key": data_key},
            }
            asset_docs_cache.append(("datum", datum))
        return tuple(asset_docs_cache)

    def _extract_metadata(self, field="omega"):
        with h5py.File(self._master_file, "r") as hf:
            return hf.get(f"entry/sample/goniometer/{field}")[()]

    def unstage(self):
        ttime.sleep(1.0)
        self.detector.unstage()
        self.detector.cam.acquire.put(0)

    def update_parameters(self, *args, **kwargs):
        self.detector_arm(**kwargs)
        self.configure_detector(**kwargs)
        self.configure_vector(**kwargs)
        self.configure_zebra(**kwargs)

    def configure_detector(self, **kwargs):
        file_prefix = kwargs["file_prefix"]
        data_directory_name = kwargs["data_directory_name"]
        self.detector.file.external_name.put(file_prefix)
        self.detector.file.write_path_template = data_directory_name

    def configure_vector(self, *args, **kwargs):
        angle_start = kwargs["angle_start"]
        scanWidth = kwargs["scan_width"]
        imgWidth = kwargs["img_width"]
        exposurePeriodPerImage = kwargs["exposure_period_per_image"]
        # scan encoder 0=x, 1=y,2=z,3=omega

        self.vector.sync.put(1)
        self.vector.expose.put(1)

        if imgWidth == 0:
            angle_end = angle_start
            numImages = scanWidth
        else:
            angle_end = angle_start + scanWidth
            numImages = int(round(scanWidth / imgWidth))
        total_exposure_time = exposurePeriodPerImage * numImages
        if total_exposure_time < 1.0:
            self.vector.buffer_time.put(1000)
        else:
            self.vector.buffer_time.put(3)
            pass
        self.setup_vector_program(
            num_images=numImages,
            angle_start=angle_start,
            angle_end=angle_end,
            exposure_period_per_image=exposurePeriodPerImage,
        )

    def configure_zebra(self, *args, **kwargs):
        angle_start = kwargs["angle_start"]
        exposurePeriodPerImage = kwargs["exposure_period_per_image"]
        detector_dead_time = kwargs["detector_dead_time"]
        scanWidth = kwargs["scan_width"]
        imgWidth = kwargs["img_width"]
        numImages = kwargs["num_images"]
        self.zebra_daq_prep()
        ttime.sleep(1.0)

        PW = (exposurePeriodPerImage - detector_dead_time) * 1000
        PS = (exposurePeriodPerImage) * 1000
        GW = scanWidth - (1.0 - (PW / PS)) * (imgWidth / 2.0)
        self.setup_zebra_vector_scan(
            angle_start=angle_start,
            gate_width=GW,
            scan_width=scanWidth,
            pulse_width=PW,
            pulse_step=PS,
            exposure_period_per_image=exposurePeriodPerImage,
            num_images=numImages,
            is_still=imgWidth == 0,
        )

    def detector_arm(self, **kwargs):
        start = kwargs["angle_start"]
        width = kwargs["img_width"]
        num_images = kwargs["num_images"]
        exposure_per_image = kwargs["exposure_period_per_image"]
        file_prefix = kwargs["file_prefix"]
        data_directory_name = kwargs["data_directory_name"]
        file_number_start = kwargs["file_number_start"]
        x_beam = kwargs["x_beam"]
        y_beam = kwargs["y_beam"]
        wavelength = kwargs["wavelength"]
        det_distance_m = kwargs["det_distance_m"]
        transmission = kwargs["transmission"]

        self.detector.cam.save_files.put(1)
        self.detector.cam.file_owner.put(getpass.getuser())
        self.detector.cam.file_owner_grp.put(grp.getgrgid(os.getgid())[0])
        self.detector.cam.file_perms.put(420)
        file_prefix_minus_directory = str(file_prefix)
        file_prefix_minus_directory = file_prefix_minus_directory.split("/")[-1]

        self.detector.cam.acquire_time.put(exposure_per_image)
        self.detector.cam.acquire_period.put(exposure_per_image)
        self.detector.cam.num_images.put(num_images)
        self.detector.cam.file_path.put(data_directory_name)
        self.detector.cam.fw_name_pattern.put(f"{file_prefix_minus_directory}_$id")

        # TODO: change it back to detector.cam.sequence_id once the ophyd PR
        # https://github.com/bluesky/ophyd/pull/1001 is merged/released.
        self.detector.file.sequence_id.put(file_number_start)

        # originally from detector_set_fileheader
        self.detector.cam.beam_center_x.put(x_beam)
        self.detector.cam.beam_center_y.put(y_beam)
        self.detector.cam.omega_incr.put(width)
        self.detector.cam.omega_start.put(start)
        self.detector.cam.wavelength.put(wavelength)
        self.detector.cam.det_distance.put(det_distance_m * 1000)
        self.detector.cam.filter_transm.put(transmission)

        start_arm = ttime.time()
        self.detector.cam.acquire.put(1)
        logger.info(f"arm time = {ttime.time() - start_arm}")

    def setup_vector_program(self, num_images, angle_start, angle_end, exposure_period_per_image):
        self.vector.num_frames.put(num_images)
        self.vector.start.omega.put(angle_start)
        self.vector.end.omega.put(angle_end)
        self.vector.frame_exptime.put(exposure_period_per_image * 1000.0)
        self.vector.hold.put(0)

    def zebra_daq_prep(self):
        self.zebra.reset.put(1)
        ttime.sleep(2.0)
        self.zebra.out1.put(31)
        self.zebra.m1_set_pos.put(1)
        self.zebra.m2_set_pos.put(1)
        self.zebra.m3_set_pos.put(1)
        self.zebra.pc.arm.trig_source.put(1)

    # expected zebra setup:
    #     time in ms
    #     Posn direction: positive
    #     gate trig source - Position
    #     pulse trig source - Time
    def setup_zebra_vector_scan(
        self,
        angle_start,
        gate_width,
        scan_width,
        pulse_width,
        pulse_step,
        exposure_period_per_image,
        num_images,
        is_still=False,
    ):
        self.zebra.pc.gate.start.put(angle_start)
        if is_still is False:
            self.zebra.pc.gate.width.put(gate_width)
            self.zebra.pc.gate.step.put(scan_width)
        self.zebra.pc.gate.num_gates.put(1)
        self.zebra.pc.pulse.start.put(0)
        self.zebra.pc.pulse.width.put(pulse_width)
        self.zebra.pc.pulse.step.put(pulse_step)
        self.zebra.pc.pulse.delay.put(exposure_period_per_image / 2 * 1000)
        self.zebra.pc.pulse.max.put(num_images)
