from ...entities._comparator import BenchmarkComparator, BenchmarkListComparator


def test_compare_no_change():
    baseline = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1000",
        "unit": "i/s",
        "id": "some-benchmark-id-1",
        "batch_id": "some-batch-id-1",
        "run_id": "some-run-id-1",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "0.0",
    }
    contender = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1000",
        "unit": "i/s",
        "id": "some-benchmark-id-2",
        "batch_id": "some-batch-id-2",
        "run_id": "some-run-id-2",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "0.0",
    }

    result = BenchmarkComparator(baseline, contender).compare()
    formatted = BenchmarkComparator(baseline, contender).formatted()

    assert result == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "0.000",
        "threshold": "5.000",
        "regression": False,
        "improvement": False,
        "threshold_z": "5.000",
        "baseline_z_score": "0.000",
        "contender_z_score": "0.000",
        "baseline_z_regression": False,
        "baseline_z_improvement": False,
        "contender_z_regression": False,
        "contender_z_improvement": False,
        "baseline": "1000.000",
        "contender": "1000.000",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }
    assert formatted == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "0.000%",
        "threshold": "5.000%",
        "regression": False,
        "improvement": False,
        "threshold_z": "5.000",
        "baseline_z_score": "0.000",
        "contender_z_score": "0.000",
        "baseline_z_regression": False,
        "baseline_z_improvement": False,
        "contender_z_regression": False,
        "contender_z_improvement": False,
        "baseline": "1.000 K i/s",
        "contender": "1.000 K i/s",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }


def test_compare_regression():
    baseline = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1000",
        "unit": "i/s",
        "id": "some-benchmark-id-1",
        "batch_id": "some-batch-id-1",
        "run_id": "some-run-id-1",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "-6.0",
    }
    contender = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "language": "Python",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "value": "940",
        "unit": "i/s",
        "id": "some-benchmark-id-2",
        "batch_id": "some-batch-id-2",
        "run_id": "some-run-id-2",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "-6.0",
    }

    result = BenchmarkComparator(baseline, contender).compare()
    formatted = BenchmarkComparator(baseline, contender).formatted()

    assert result == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "-6.000",
        "threshold": "5.000",
        "regression": True,
        "improvement": False,
        "threshold_z": "5.000",
        "baseline_z_score": "-6.000",
        "contender_z_score": "-6.000",
        "baseline_z_regression": True,
        "baseline_z_improvement": False,
        "contender_z_regression": True,
        "contender_z_improvement": False,
        "baseline": "1000.000",
        "contender": "940.000",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }
    assert formatted == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "-6.000%",
        "threshold": "5.000%",
        "regression": True,
        "improvement": False,
        "threshold_z": "5.000",
        "baseline_z_score": "-6.000",
        "contender_z_score": "-6.000",
        "baseline_z_regression": True,
        "baseline_z_improvement": False,
        "contender_z_regression": True,
        "contender_z_improvement": False,
        "baseline": "1.000 K i/s",
        "contender": "940 i/s",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }


def test_compare_regression_less_is_better():
    baseline = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1000",
        "unit": "s",
        "id": "some-benchmark-id-1",
        "batch_id": "some-batch-id-1",
        "run_id": "some-run-id-1",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "-6.0",
    }
    contender = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1060",
        "unit": "s",
        "id": "some-benchmark-id-2",
        "batch_id": "some-batch-id-2",
        "run_id": "some-run-id-2",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "-6.0",
    }

    result = BenchmarkComparator(baseline, contender).compare()
    formatted = BenchmarkComparator(baseline, contender).formatted()

    assert result == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "-6.000",
        "threshold": "5.000",
        "regression": True,
        "improvement": False,
        "threshold_z": "5.000",
        "baseline_z_score": "-6.000",
        "contender_z_score": "-6.000",
        "baseline_z_regression": True,
        "baseline_z_improvement": False,
        "contender_z_regression": True,
        "contender_z_improvement": False,
        "baseline": "1000.000",
        "contender": "1060.000",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "s",
        "less_is_better": True,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }
    assert formatted == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "-6.000%",
        "threshold": "5.000%",
        "regression": True,
        "improvement": False,
        "threshold_z": "5.000",
        "baseline_z_score": "-6.000",
        "contender_z_score": "-6.000",
        "baseline_z_regression": True,
        "baseline_z_improvement": False,
        "contender_z_regression": True,
        "contender_z_improvement": False,
        "baseline": "1000.000 s",
        "contender": "1060.000 s",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "s",
        "less_is_better": True,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }


def test_compare_regression_but_under_threshold():
    baseline = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1000",
        "unit": "i/s",
        "id": "some-benchmark-id-1",
        "batch_id": "some-batch-id-1",
        "run_id": "some-run-id-1",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "-5.0",
    }
    contender = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "950",
        "unit": "i/s",
        "id": "some-benchmark-id-2",
        "batch_id": "some-batch-id-2",
        "run_id": "some-run-id-2",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "-5.0",
    }

    result = BenchmarkComparator(baseline, contender).compare()
    formatted = BenchmarkComparator(baseline, contender).formatted()

    assert result == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "-5.000",
        "threshold": "5.000",
        "regression": False,
        "improvement": False,
        "threshold_z": "5.000",
        "baseline_z_score": "-5.000",
        "contender_z_score": "-5.000",
        "baseline_z_regression": False,
        "baseline_z_improvement": False,
        "contender_z_regression": False,
        "contender_z_improvement": False,
        "baseline": "1000.000",
        "contender": "950.000",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }
    assert formatted == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "-5.000%",
        "threshold": "5.000%",
        "regression": False,
        "improvement": False,
        "threshold_z": "5.000",
        "baseline_z_score": "-5.000",
        "contender_z_score": "-5.000",
        "baseline_z_regression": False,
        "baseline_z_improvement": False,
        "contender_z_regression": False,
        "contender_z_improvement": False,
        "baseline": "1.000 K i/s",
        "contender": "950 i/s",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }


def test_compare_regression_custom_threshold_and_threshold_z():
    baseline = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1000",
        "unit": "i/s",
        "id": "some-benchmark-id-1",
        "batch_id": "some-batch-id-1",
        "run_id": "some-run-id-1",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "-5.0",
    }
    contender = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "950",
        "unit": "i/s",
        "id": "some-benchmark-id-2",
        "batch_id": "some-batch-id-2",
        "run_id": "some-run-id-2",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "-5.0",
    }

    threshold, threshold_z = 4, 1
    result = BenchmarkComparator(baseline, contender, threshold, threshold_z).compare()
    formatted = BenchmarkComparator(
        baseline, contender, threshold, threshold_z
    ).formatted()

    assert result == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "-5.000",
        "threshold": "4.000",
        "regression": True,
        "improvement": False,
        "threshold_z": "1.000",
        "baseline_z_score": "-5.000",
        "contender_z_score": "-5.000",
        "baseline_z_regression": True,
        "baseline_z_improvement": False,
        "contender_z_regression": True,
        "contender_z_improvement": False,
        "baseline": "1000.000",
        "contender": "950.000",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }
    assert formatted == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "-5.000%",
        "threshold": "4.000%",
        "regression": True,
        "improvement": False,
        "threshold_z": "1.000",
        "baseline_z_score": "-5.000",
        "contender_z_score": "-5.000",
        "baseline_z_regression": True,
        "baseline_z_improvement": False,
        "contender_z_regression": True,
        "contender_z_improvement": False,
        "baseline": "1.000 K i/s",
        "contender": "950 i/s",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }


def test_compare_improvement():
    baseline = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1000",
        "unit": "i/s",
        "id": "some-benchmark-id-1",
        "batch_id": "some-batch-id-1",
        "run_id": "some-run-id-1",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "6.0",
    }
    contender = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1060",
        "unit": "i/s",
        "id": "some-benchmark-id-2",
        "batch_id": "some-batch-id-2",
        "run_id": "some-run-id-2",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "6.0",
    }

    result = BenchmarkComparator(baseline, contender).compare()
    formatted = BenchmarkComparator(baseline, contender).formatted()

    assert result == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "6.000",
        "threshold": "5.000",
        "regression": False,
        "improvement": True,
        "threshold_z": "5.000",
        "baseline_z_score": "6.000",
        "contender_z_score": "6.000",
        "baseline_z_regression": False,
        "baseline_z_improvement": True,
        "contender_z_regression": False,
        "contender_z_improvement": True,
        "baseline": "1000.000",
        "contender": "1060.000",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }
    assert formatted == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "6.000%",
        "threshold": "5.000%",
        "regression": False,
        "improvement": True,
        "threshold_z": "5.000",
        "baseline_z_score": "6.000",
        "contender_z_score": "6.000",
        "baseline_z_regression": False,
        "baseline_z_improvement": True,
        "contender_z_regression": False,
        "contender_z_improvement": True,
        "baseline": "1.000 K i/s",
        "contender": "1.060 K i/s",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }


def test_compare_improvement_less_is_better():
    baseline = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1000",
        "unit": "s",
        "id": "some-benchmark-id-1",
        "batch_id": "some-batch-id-1",
        "run_id": "some-run-id-1",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "6.0",
    }
    contender = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "940",
        "unit": "s",
        "id": "some-benchmark-id-2",
        "batch_id": "some-batch-id-2",
        "run_id": "some-run-id-2",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "6.0",
    }

    result = BenchmarkComparator(baseline, contender).compare()
    formatted = BenchmarkComparator(baseline, contender).formatted()

    assert result == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "6.000",
        "threshold": "5.000",
        "regression": False,
        "improvement": True,
        "threshold_z": "5.000",
        "baseline_z_score": "6.000",
        "contender_z_score": "6.000",
        "baseline_z_regression": False,
        "baseline_z_improvement": True,
        "contender_z_regression": False,
        "contender_z_improvement": True,
        "baseline": "1000.000",
        "contender": "940.000",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "s",
        "less_is_better": True,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }
    assert formatted == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "6.000%",
        "threshold": "5.000%",
        "regression": False,
        "improvement": True,
        "threshold_z": "5.000",
        "baseline_z_score": "6.000",
        "contender_z_score": "6.000",
        "baseline_z_regression": False,
        "baseline_z_improvement": True,
        "contender_z_regression": False,
        "contender_z_improvement": True,
        "baseline": "1000.000 s",
        "contender": "940.000 s",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "s",
        "less_is_better": True,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }


def test_compare_improvement_but_under_threshold():
    baseline = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1000",
        "unit": "i/s",
        "id": "some-benchmark-id-1",
        "batch_id": "some-batch-id-1",
        "run_id": "some-run-id-1",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "5.0",
    }
    contender = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1050",
        "unit": "i/s",
        "id": "some-benchmark-id-2",
        "batch_id": "some-batch-id-2",
        "run_id": "some-run-id-2",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "5.0",
    }

    threshold, threshold_z = 4, 1
    result = BenchmarkComparator(baseline, contender, threshold, threshold_z).compare()
    formatted = BenchmarkComparator(
        baseline, contender, threshold, threshold_z
    ).formatted()

    assert result == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "5.000",
        "threshold": "4.000",
        "regression": False,
        "improvement": True,
        "threshold_z": "1.000",
        "baseline_z_score": "5.000",
        "contender_z_score": "5.000",
        "baseline_z_regression": False,
        "baseline_z_improvement": True,
        "contender_z_regression": False,
        "contender_z_improvement": True,
        "baseline": "1000.000",
        "contender": "1050.000",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }
    assert formatted == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "5.000%",
        "threshold": "4.000%",
        "regression": False,
        "improvement": True,
        "threshold_z": "1.000",
        "baseline_z_score": "5.000",
        "contender_z_score": "5.000",
        "baseline_z_regression": False,
        "baseline_z_improvement": True,
        "contender_z_regression": False,
        "contender_z_improvement": True,
        "baseline": "1.000 K i/s",
        "contender": "1.050 K i/s",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }


def test_compare_improvement_custom_threshold_and_threshold_z():
    baseline = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1000",
        "unit": "i/s",
        "id": "some-benchmark-id-1",
        "batch_id": "some-batch-id-1",
        "run_id": "some-run-id-1",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "5.0",
    }
    contender = {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "value": "1050",
        "unit": "i/s",
        "id": "some-benchmark-id-2",
        "batch_id": "some-batch-id-2",
        "run_id": "some-run-id-2",
        "tags": {"tag_one": "one", "tag_two": "two"},
        "z_score": "5.0",
    }

    result = BenchmarkComparator(baseline, contender).compare()
    formatted = BenchmarkComparator(baseline, contender).formatted()

    assert result == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "5.000",
        "threshold": "5.000",
        "regression": False,
        "improvement": False,
        "threshold_z": "5.000",
        "baseline_z_score": "5.000",
        "contender_z_score": "5.000",
        "baseline_z_regression": False,
        "baseline_z_improvement": False,
        "contender_z_regression": False,
        "contender_z_improvement": False,
        "baseline": "1000.000",
        "contender": "1050.000",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }
    assert formatted == {
        "batch": "arrow-compute-scalar-cast-benchmark",
        "benchmark": "CastUInt32ToInt32Safe/262144/1000",
        "language": "Python",
        "change": "5.000%",
        "threshold": "5.000%",
        "regression": False,
        "improvement": False,
        "threshold_z": "5.000",
        "baseline_z_score": "5.000",
        "contender_z_score": "5.000",
        "baseline_z_regression": False,
        "baseline_z_improvement": False,
        "contender_z_regression": False,
        "contender_z_improvement": False,
        "baseline": "1.000 K i/s",
        "contender": "1.050 K i/s",
        "baseline_id": "some-benchmark-id-1",
        "contender_id": "some-benchmark-id-2",
        "baseline_batch_id": "some-batch-id-1",
        "contender_batch_id": "some-batch-id-2",
        "baseline_run_id": "some-run-id-1",
        "contender_run_id": "some-run-id-2",
        "unit": "i/s",
        "less_is_better": False,
        "tags": {"tag_one": "one", "tag_two": "two"},
    }


def test_compare_list():
    pairs = {
        "some-case-id-1": {
            "baseline": {
                "batch": "math",
                "benchmark": "addition",
                "language": "Python",
                "unit": "s",
                "value": "0.036369",
                "id": "some-benchmark-id-1",
                "batch_id": "some-batch-id-1",
                "run_id": "some-run-id-1",
                "tags": {"tag_one": "one", "tag_two": "two"},
                "z_score": "0.0",
            },
            "contender": {
                "batch": "math",
                "benchmark": "addition",
                "language": "Python",
                "unit": "s",
                "value": "0.036369",
                "id": "some-benchmark-id-2",
                "batch_id": "some-batch-id-2",
                "run_id": "some-run-id-2",
                "tags": {"tag_one": "one", "tag_two": "two"},
                "z_score": "0.0",
            },
        },
        "some-case-id-2": {
            "baseline": {
                "batch": "math",
                "benchmark": "subtraction",
                "language": "Python",
                "unit": "s",
                "value": "1.036369",
                "id": "some-benchmark-id-3",
                "batch_id": "some-batch-id-1",
                "run_id": "some-run-id-1",
                "tags": {"tag_one": "1", "tag_two": "2"},
                "z_score": "-6.0",
            },
            "contender": {
                "batch": "math",
                "benchmark": "subtraction",
                "language": "Python",
                "unit": "s",
                "value": "0.036369",
                "id": "some-benchmark-id-4",
                "batch_id": "some-batch-id-2",
                "run_id": "some-run-id-2",
                "tags": {"tag_one": "1", "tag_two": "2"},
                "z_score": "6.0",
            },
        },
    }

    result = BenchmarkListComparator(pairs).compare()
    formatted = BenchmarkListComparator(pairs).formatted()

    assert list(result) == [
        {
            "batch": "math",
            "benchmark": "addition",
            "language": "Python",
            "change": "0.000",
            "threshold": "5.000",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "0.036",
            "contender": "0.036",
            "baseline_id": "some-benchmark-id-1",
            "contender_id": "some-benchmark-id-2",
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "one", "tag_two": "two"},
        },
        {
            "batch": "math",
            "benchmark": "subtraction",
            "language": "Python",
            "change": "96.491",
            "threshold": "5.000",
            "regression": False,
            "improvement": True,
            "threshold_z": "5.000",
            "baseline_z_score": "-6.000",
            "contender_z_score": "6.000",
            "baseline_z_regression": True,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": True,
            "baseline": "1.036",
            "contender": "0.036",
            "baseline_id": "some-benchmark-id-3",
            "contender_id": "some-benchmark-id-4",
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "1", "tag_two": "2"},
        },
    ]
    assert list(formatted) == [
        {
            "batch": "math",
            "benchmark": "addition",
            "language": "Python",
            "change": "0.000%",
            "threshold": "5.000%",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "0.036 s",
            "contender": "0.036 s",
            "baseline_id": "some-benchmark-id-1",
            "contender_id": "some-benchmark-id-2",
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "one", "tag_two": "two"},
        },
        {
            "batch": "math",
            "benchmark": "subtraction",
            "language": "Python",
            "change": "96.491%",
            "threshold": "5.000%",
            "regression": False,
            "improvement": True,
            "threshold_z": "5.000",
            "baseline_z_score": "-6.000",
            "contender_z_score": "6.000",
            "baseline_z_regression": True,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": True,
            "baseline": "1.036 s",
            "contender": "0.036 s",
            "baseline_id": "some-benchmark-id-3",
            "contender_id": "some-benchmark-id-4",
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "1", "tag_two": "2"},
        },
    ]


def test_compare_list_missing_contender():
    pairs = {
        "some-case-id-1": {
            "baseline": {
                "batch": "math",
                "benchmark": "addition",
                "language": "Python",
                "unit": "s",
                "value": "0.036369",
                "id": "some-benchmark-id-1",
                "batch_id": "some-batch-id-1",
                "run_id": "some-run-id-1",
                "tags": {"tag_one": "one", "tag_two": "two"},
                "z_score": "0.0",
            },
        },
        "some-case-id-2": {
            "baseline": {
                "batch": "math",
                "benchmark": "subtraction",
                "language": "Python",
                "unit": "s",
                "value": "1.036369",
                "id": "some-benchmark-id-3",
                "batch_id": "some-batch-id-1",
                "run_id": "some-run-id-1",
                "tags": {"tag_one": "1", "tag_two": "2"},
                "z_score": "0.0",
            },
            "contender": {
                "batch": "math",
                "benchmark": "subtraction",
                "language": "Python",
                "unit": "s",
                "value": "0.036369",
                "id": "some-benchmark-id-4",
                "batch_id": "some-batch-id-2",
                "run_id": "some-run-id-2",
                "tags": {"tag_one": "1", "tag_two": "2"},
                "z_score": "0.0",
            },
        },
    }

    result = BenchmarkListComparator(pairs).compare()
    formatted = BenchmarkListComparator(pairs).formatted()

    assert list(result) == [
        {
            "batch": "math",
            "benchmark": "addition",
            "language": "Python",
            "change": "0.000",
            "threshold": "5.000",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": None,
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "0.036",
            "contender": None,
            "baseline_id": "some-benchmark-id-1",
            "contender_id": None,
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": None,
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": None,
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "one", "tag_two": "two"},
        },
        {
            "batch": "math",
            "benchmark": "subtraction",
            "language": "Python",
            "change": "96.491",
            "threshold": "5.000",
            "regression": False,
            "improvement": True,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "1.036",
            "contender": "0.036",
            "baseline_id": "some-benchmark-id-3",
            "contender_id": "some-benchmark-id-4",
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "1", "tag_two": "2"},
        },
    ]
    assert list(formatted) == [
        {
            "batch": "math",
            "benchmark": "addition",
            "language": "Python",
            "change": "0.000%",
            "threshold": "5.000%",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": None,
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "0.036 s",
            "contender": None,
            "baseline_id": "some-benchmark-id-1",
            "contender_id": None,
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": None,
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": None,
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "one", "tag_two": "two"},
        },
        {
            "batch": "math",
            "benchmark": "subtraction",
            "language": "Python",
            "change": "96.491%",
            "threshold": "5.000%",
            "regression": False,
            "improvement": True,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "1.036 s",
            "contender": "0.036 s",
            "baseline_id": "some-benchmark-id-3",
            "contender_id": "some-benchmark-id-4",
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "1", "tag_two": "2"},
        },
    ]


def test_compare_list_empty_contender():
    pairs = {
        "some-case-id-1": {
            "baseline": {
                "batch": "math",
                "benchmark": "addition",
                "language": "Python",
                "unit": "s",
                "value": "0.036369",
                "id": "some-benchmark-id-1",
                "batch_id": "some-batch-id-1",
                "run_id": "some-run-id-1",
                "tags": {"tag_one": "one", "tag_two": "two"},
                "z_score": "0.0",
            },
            "contender": {},
        },
        "some-case-id-2": {
            "baseline": {
                "batch": "math",
                "benchmark": "subtraction",
                "language": "Python",
                "unit": "s",
                "value": "1.036369",
                "id": "some-benchmark-id-3",
                "batch_id": "some-batch-id-1",
                "run_id": "some-run-id-1",
                "tags": {"tag_one": "1", "tag_two": "2"},
                "z_score": "0.0",
            },
            "contender": {
                "batch": "math",
                "benchmark": "subtraction",
                "language": "Python",
                "unit": "s",
                "value": "0.036369",
                "id": "some-benchmark-id-4",
                "batch_id": "some-batch-id-2",
                "run_id": "some-run-id-2",
                "tags": {"tag_one": "1", "tag_two": "2"},
                "z_score": "0.0",
            },
        },
    }

    result = BenchmarkListComparator(pairs).compare()
    formatted = BenchmarkListComparator(pairs).formatted()

    assert list(result) == [
        {
            "batch": "math",
            "benchmark": "addition",
            "language": "Python",
            "change": "0.000",
            "threshold": "5.000",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": None,
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "0.036",
            "contender": None,
            "baseline_id": "some-benchmark-id-1",
            "contender_id": None,
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": None,
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": None,
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "one", "tag_two": "two"},
        },
        {
            "batch": "math",
            "benchmark": "subtraction",
            "language": "Python",
            "change": "96.491",
            "threshold": "5.000",
            "regression": False,
            "improvement": True,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "1.036",
            "contender": "0.036",
            "baseline_id": "some-benchmark-id-3",
            "contender_id": "some-benchmark-id-4",
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "1", "tag_two": "2"},
        },
    ]
    assert list(formatted) == [
        {
            "batch": "math",
            "benchmark": "addition",
            "language": "Python",
            "change": "0.000%",
            "threshold": "5.000%",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": None,
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "0.036 s",
            "contender": None,
            "baseline_id": "some-benchmark-id-1",
            "contender_id": None,
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": None,
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": None,
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "one", "tag_two": "two"},
        },
        {
            "batch": "math",
            "benchmark": "subtraction",
            "language": "Python",
            "change": "96.491%",
            "threshold": "5.000%",
            "regression": False,
            "improvement": True,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "1.036 s",
            "contender": "0.036 s",
            "baseline_id": "some-benchmark-id-3",
            "contender_id": "some-benchmark-id-4",
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "1", "tag_two": "2"},
        },
    ]


def test_compare_list_missing_baseline():
    pairs = {
        "some-case-id-1": {
            "contender": {
                "batch": "math",
                "benchmark": "addition",
                "language": "Python",
                "unit": "s",
                "value": "0.036369",
                "id": "some-benchmark-id-2",
                "batch_id": "some-batch-id-2",
                "run_id": "some-run-id-2",
                "tags": {"tag_one": "one", "tag_two": "two"},
                "z_score": "0.0",
            },
        },
        "some-case-id-2": {
            "baseline": {
                "batch": "math",
                "benchmark": "subtraction",
                "language": "Python",
                "unit": "s",
                "value": "1.036369",
                "id": "some-benchmark-id-3",
                "batch_id": "some-batch-id-1",
                "run_id": "some-run-id-1",
                "tags": {"tag_one": "1", "tag_two": "2"},
                "z_score": "0.0",
            },
            "contender": {
                "batch": "math",
                "benchmark": "subtraction",
                "language": "Python",
                "unit": "s",
                "value": "0.036369",
                "id": "some-benchmark-id-4",
                "batch_id": "some-batch-id-2",
                "run_id": "some-run-id-2",
                "tags": {"tag_one": "1", "tag_two": "2"},
                "z_score": "0.0",
            },
        },
    }

    result = BenchmarkListComparator(pairs).compare()
    formatted = BenchmarkListComparator(pairs).formatted()

    assert list(result) == [
        {
            "batch": "math",
            "benchmark": "addition",
            "language": "Python",
            "change": "0.000",
            "threshold": "5.000",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": None,
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": None,
            "contender": "0.036",
            "baseline_id": None,
            "contender_id": "some-benchmark-id-2",
            "baseline_batch_id": None,
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": None,
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "one", "tag_two": "two"},
        },
        {
            "batch": "math",
            "benchmark": "subtraction",
            "language": "Python",
            "change": "96.491",
            "threshold": "5.000",
            "regression": False,
            "improvement": True,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "1.036",
            "contender": "0.036",
            "baseline_id": "some-benchmark-id-3",
            "contender_id": "some-benchmark-id-4",
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "1", "tag_two": "2"},
        },
    ]
    assert list(formatted) == [
        {
            "batch": "math",
            "benchmark": "addition",
            "language": "Python",
            "change": "0.000%",
            "threshold": "5.000%",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": None,
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": None,
            "contender": "0.036 s",
            "baseline_id": None,
            "contender_id": "some-benchmark-id-2",
            "baseline_batch_id": None,
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": None,
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "one", "tag_two": "two"},
        },
        {
            "batch": "math",
            "benchmark": "subtraction",
            "language": "Python",
            "change": "96.491%",
            "threshold": "5.000%",
            "regression": False,
            "improvement": True,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "1.036 s",
            "contender": "0.036 s",
            "baseline_id": "some-benchmark-id-3",
            "contender_id": "some-benchmark-id-4",
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "1", "tag_two": "2"},
        },
    ]


def test_compare_list_empty_baseline():
    pairs = {
        "some-case-id-1": {
            "baseline": {},
            "contender": {
                "batch": "math",
                "benchmark": "addition",
                "language": "Python",
                "unit": "s",
                "value": "0.036369",
                "id": "some-benchmark-id-2",
                "batch_id": "some-batch-id-2",
                "run_id": "some-run-id-2",
                "tags": {"tag_one": "one", "tag_two": "two"},
                "z_score": "0.0",
            },
        },
        "some-case-id-2": {
            "baseline": {
                "batch": "math",
                "benchmark": "subtraction",
                "language": "Python",
                "unit": "s",
                "value": "1.036369",
                "id": "some-benchmark-id-3",
                "batch_id": "some-batch-id-1",
                "run_id": "some-run-id-1",
                "tags": {"tag_one": "1", "tag_two": "2"},
                "z_score": "0.0",
            },
            "contender": {
                "batch": "math",
                "benchmark": "subtraction",
                "language": "Python",
                "unit": "s",
                "value": "0.036369",
                "id": "some-benchmark-id-4",
                "batch_id": "some-batch-id-2",
                "run_id": "some-run-id-2",
                "tags": {"tag_one": "1", "tag_two": "2"},
                "z_score": "0.0",
            },
        },
    }

    result = BenchmarkListComparator(pairs).compare()
    formatted = BenchmarkListComparator(pairs).formatted()

    assert list(result) == [
        {
            "batch": "math",
            "benchmark": "addition",
            "language": "Python",
            "change": "0.000",
            "threshold": "5.000",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": None,
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": None,
            "contender": "0.036",
            "baseline_id": None,
            "contender_id": "some-benchmark-id-2",
            "baseline_batch_id": None,
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": None,
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "one", "tag_two": "two"},
        },
        {
            "batch": "math",
            "benchmark": "subtraction",
            "language": "Python",
            "change": "96.491",
            "threshold": "5.000",
            "regression": False,
            "improvement": True,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "1.036",
            "contender": "0.036",
            "baseline_id": "some-benchmark-id-3",
            "contender_id": "some-benchmark-id-4",
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "1", "tag_two": "2"},
        },
    ]
    assert list(formatted) == [
        {
            "batch": "math",
            "benchmark": "addition",
            "language": "Python",
            "change": "0.000%",
            "threshold": "5.000%",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": None,
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": None,
            "contender": "0.036 s",
            "baseline_id": None,
            "contender_id": "some-benchmark-id-2",
            "baseline_batch_id": None,
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": None,
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "one", "tag_two": "two"},
        },
        {
            "batch": "math",
            "benchmark": "subtraction",
            "language": "Python",
            "change": "96.491%",
            "threshold": "5.000%",
            "regression": False,
            "improvement": True,
            "threshold_z": "5.000",
            "baseline_z_score": "0.000",
            "contender_z_score": "0.000",
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": "1.036 s",
            "contender": "0.036 s",
            "baseline_id": "some-benchmark-id-3",
            "contender_id": "some-benchmark-id-4",
            "baseline_batch_id": "some-batch-id-1",
            "contender_batch_id": "some-batch-id-2",
            "baseline_run_id": "some-run-id-1",
            "contender_run_id": "some-run-id-2",
            "unit": "s",
            "less_is_better": True,
            "tags": {"tag_one": "1", "tag_two": "2"},
        },
    ]


def test_compare_list_empty_pair():
    pairs = {
        "some-case-id-1": {
            "baseline": {},
            "contender": {},
        },
        "some-case-id-2": {
            "baseline": {},
            "contender": {},
        },
    }

    result = BenchmarkListComparator(pairs).compare()
    formatted = BenchmarkListComparator(pairs).formatted()

    assert list(result) == [
        {
            "batch": "unknown",
            "benchmark": "unknown",
            "language": "unknown",
            "change": "0.000",
            "threshold": "5.000",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": None,
            "contender_z_score": None,
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": None,
            "contender": None,
            "baseline_id": None,
            "contender_id": None,
            "baseline_batch_id": None,
            "contender_batch_id": None,
            "baseline_run_id": None,
            "contender_run_id": None,
            "unit": "unknown",
            "less_is_better": True,
            "tags": "unknown",
        },
        {
            "batch": "unknown",
            "benchmark": "unknown",
            "language": "unknown",
            "change": "0.000",
            "threshold": "5.000",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": None,
            "contender_z_score": None,
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": None,
            "contender": None,
            "baseline_id": None,
            "contender_id": None,
            "baseline_batch_id": None,
            "contender_batch_id": None,
            "baseline_run_id": None,
            "contender_run_id": None,
            "unit": "unknown",
            "less_is_better": True,
            "tags": "unknown",
        },
    ]
    assert list(formatted) == [
        {
            "batch": "unknown",
            "benchmark": "unknown",
            "language": "unknown",
            "change": "0.000%",
            "threshold": "5.000%",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": None,
            "contender_z_score": None,
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": None,
            "contender": None,
            "baseline_id": None,
            "contender_id": None,
            "baseline_batch_id": None,
            "contender_batch_id": None,
            "baseline_run_id": None,
            "contender_run_id": None,
            "unit": "unknown",
            "less_is_better": True,
            "tags": "unknown",
        },
        {
            "batch": "unknown",
            "benchmark": "unknown",
            "language": "unknown",
            "change": "0.000%",
            "threshold": "5.000%",
            "regression": False,
            "improvement": False,
            "threshold_z": "5.000",
            "baseline_z_score": None,
            "contender_z_score": None,
            "baseline_z_regression": False,
            "baseline_z_improvement": False,
            "contender_z_regression": False,
            "contender_z_improvement": False,
            "baseline": None,
            "contender": None,
            "baseline_id": None,
            "contender_id": None,
            "baseline_batch_id": None,
            "contender_batch_id": None,
            "baseline_run_id": None,
            "contender_run_id": None,
            "unit": "unknown",
            "less_is_better": True,
            "tags": "unknown",
        },
    ]
