"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const defaults = require("@aws-solutions-constructs/core");
const lambda = require("@aws-cdk/aws-lambda");
const iam = require("@aws-cdk/aws-iam");
require("@aws-cdk/assert/jest");
// -----------------------------------------------------------------------------------------
// Pattern deployment with new Lambda function, new Sagemaker endpoint and deployVpc = true
// -----------------------------------------------------------------------------------------
test('Pattern deployment with new Lambda function, new Sagemaker endpoint, deployVpc = true', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const constructProps = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        lambdaFunctionProps: {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(5),
            memorySize: 128,
        },
        deployVpc: true,
    };
    new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: {
                SAGEMAKER_ENDPOINT_NAME: {
                    'Fn::GetAtt': ['testlambdasagemakerSagemakerEndpoint12803730', 'EndpointName'],
                },
            },
        },
        VpcConfig: {
            SecurityGroupIds: [
                {
                    'Fn::GetAtt': ['testlambdasagemakerReplaceDefaultSecurityGroupsecuritygroupB2FD7810', 'GroupId'],
                },
            ],
            SubnetIds: [
                {
                    Ref: 'VpcisolatedSubnet1SubnetE62B1B9B',
                },
                {
                    Ref: 'VpcisolatedSubnet2Subnet39217055',
                },
            ],
        },
    });
    // Assertion 3
    expect(stack).toHaveResourceLike('AWS::SageMaker::Model', {
        ExecutionRoleArn: {
            'Fn::GetAtt': ['testlambdasagemakerSagemakerRoleD84546B8', 'Arn'],
        },
        PrimaryContainer: {
            Image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
            ModelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
        },
        VpcConfig: {
            SecurityGroupIds: [
                {
                    'Fn::GetAtt': ['testlambdasagemakerReplaceModelDefaultSecurityGroup7284AA24', 'GroupId'],
                },
            ],
            Subnets: [
                {
                    Ref: 'VpcisolatedSubnet1SubnetE62B1B9B',
                },
                {
                    Ref: 'VpcisolatedSubnet2Subnet39217055',
                },
            ],
        },
    });
    // Assertion 4
    expect(stack).toHaveResourceLike('AWS::SageMaker::EndpointConfig', {
        ProductionVariants: [
            {
                InitialInstanceCount: 1,
                InitialVariantWeight: 1,
                InstanceType: 'ml.m4.xlarge',
                ModelName: {
                    'Fn::GetAtt': ['testlambdasagemakerSagemakerModelEC3E4E39', 'ModelName'],
                },
                VariantName: 'AllTraffic',
            },
        ],
        KmsKeyId: {
            Ref: 'testlambdasagemakerEncryptionKey2AACF9E0',
        },
    });
    // Assertion 5
    expect(stack).toHaveResourceLike('AWS::SageMaker::Endpoint', {
        EndpointConfigName: {
            'Fn::GetAtt': ['testlambdasagemakerSagemakerEndpointConfig6BABA334', 'EndpointConfigName'],
        },
    });
});
// ----------------------------------------------------------------------------------------------
// Pattern deployment with existing Lambda function, new Sagemaker endpoint and deployVpc = false
// ----------------------------------------------------------------------------------------------
test('Pattern deployment with existing Lambda function, new Sagemaker endpoint, deployVpc = false', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    // deploy lambda function
    const fn = defaults.deployLambdaFunction(stack, {
        runtime: lambda.Runtime.PYTHON_3_8,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        handler: 'index.handler',
        timeout: core_1.Duration.minutes(5),
        memorySize: 128,
    });
    const constructProps = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        existingLambdaObj: fn,
    };
    new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    expect(stack).toHaveResourceLike('AWS::SageMaker::Model', {
        ExecutionRoleArn: {
            'Fn::GetAtt': ['testlambdasagemakerSagemakerRoleD84546B8', 'Arn'],
        },
        PrimaryContainer: {
            Image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
            ModelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
        },
    });
    // Assertion 3
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: {
                SAGEMAKER_ENDPOINT_NAME: {
                    'Fn::GetAtt': ['testlambdasagemakerSagemakerEndpoint12803730', 'EndpointName'],
                },
            },
        },
    });
});
// ------------------------------------------------------------------------------------------------------------------
// Pattern deployment with new Lambda function, new Sagemaker endpoint, deployVpc = true, and custom role
// ------------------------------------------------------------------------------------------------------------------
test('Pattern deployment with new Lambda function, new Sagemaker endpoint, deployVpc = true, and custom role', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    // Create IAM Role to be assumed by SageMaker
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    sagemakerRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'));
    const constructProps = {
        modelProps: {
            executionRoleArn: sagemakerRole.roleArn,
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        deployVpc: true,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(5),
            memorySize: 128,
        },
    };
    new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    // Assertion 1
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                    Principal: {
                        Service: 'sagemaker.amazonaws.com',
                    },
                },
            ],
            Version: '2012-10-17',
        },
    });
    // Assertion 2: ReplaceDefaultSecurityGroup, ReplaceEndpointDefaultSecurityGroup, and ReplaceModelDefaultSecurityGroup
    expect(stack).toCountResources('AWS::EC2::SecurityGroup', 3);
    // Assertion 3
    expect(stack).toCountResources('AWS::EC2::Subnet', 2);
    // Assertion 4
    expect(stack).toCountResources('AWS::EC2::InternetGateway', 0);
    // Assertion 5: SAGEMAKER_RUNTIME VPC Interface
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Interface',
    });
    // Assertion 6: S3 VPC Endpoint
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Gateway',
    });
    // Assertion 7
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
// ---------------------------------------------------------------------------------
// Test for error when existing Lambda function does not have vpc and deployVpc = true
// ---------------------------------------------------------------------------------
test('Test for errot when existing Lambda function does not have vpc and deployVpc = true ', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    // deploy lambda function
    const fn = defaults.deployLambdaFunction(stack, {
        runtime: lambda.Runtime.PYTHON_3_8,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        handler: 'index.handler',
        timeout: core_1.Duration.minutes(5),
        memorySize: 128,
    });
    const constructProps = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        deployVpc: true,
        existingLambdaObj: fn,
    };
    const app = () => {
        new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -------------------------------------------------------------------------------------------------------
// Pattern deployment with existing Lambda function (with VPC), new Sagemaker endpoint, and existingVpc
// -------------------------------------------------------------------------------------------------------
test('Pattern deployment with existing Lambda function (with VPC), new Sagemaker endpoint, and existingVpc', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    // Add S3 VPC Gateway Endpint, required by Sagemaker to access Models artifacts via AWS private network
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.S3);
    // Add SAGEMAKER_RUNTIME VPC Interface Endpint, required by the lambda function to invoke the SageMaker endpoint
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.SAGEMAKER_RUNTIME);
    // deploy lambda function
    const fn = defaults.deployLambdaFunction(stack, {
        runtime: lambda.Runtime.PYTHON_3_8,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        handler: 'index.handler',
        timeout: core_1.Duration.minutes(5),
        memorySize: 128,
        vpc,
    });
    const constructProps = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        existingVpc: vpc,
        existingLambdaObj: fn,
    };
    new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    // Assertion 2: ReplaceDefaultSecurityGroup, ReplaceEndpointDefaultSecurityGroup, and ReplaceModelDefaultSecurityGroup
    expect(stack).toCountResources('AWS::EC2::SecurityGroup', 3);
    // Assertion 3
    expect(stack).toCountResources('AWS::EC2::Subnet', 2);
    // Assertion 4
    expect(stack).toCountResources('AWS::EC2::InternetGateway', 0);
    // Assertion 5: SAGEMAKER_RUNTIME VPC Interface
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Interface',
    });
    // Assertion 6: S3 VPC Endpoint
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Gateway',
    });
    // Assertion 7
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
// -----------------------------------------------------------------------------------------
// Test for error with existingLambdaObj/lambdaFunctionProps=undefined (not supplied by user)
// -----------------------------------------------------------------------------------------
test('Test for error with existingLambdaObj/lambdaFunctionProps=undefined (not supplied by user)', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
    };
    const app = () => {
        new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', props);
    };
    // Assertion 1
    expect(app).toThrowError();
});
// --------------------------------------------------------------------
// Test for error with (props.deployVpc && props.existingVpc) is true
// --------------------------------------------------------------------
test('Test for error with (props.deployVpc && props.existingVpc) is true', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    // Add S3 VPC Gateway Endpint, required by Sagemaker to access Models artifacts via AWS private network
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.S3);
    // Add SAGEMAKER_RUNTIME VPC Interface Endpint, required by the lambda function to invoke the SageMaker endpoint
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.SAGEMAKER_RUNTIME);
    const constructProps = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        deployVpc: true,
        existingVpc: vpc,
    };
    const app = () => {
        new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    };
    // Assertion 1
    expect(app).toThrowError();
});
// ----------------------------------------------------------------------------------------------------------
// Test for error with primaryContainer=undefined (not supplied by user), and no existingSageMakerEndpointObj
// ----------------------------------------------------------------------------------------------------------
test('Test for error with primaryContainer=undefined (not supplied by user)', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    // deploy lambda function
    const fn = defaults.deployLambdaFunction(stack, {
        runtime: lambda.Runtime.PYTHON_3_8,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        handler: 'index.handler',
        timeout: core_1.Duration.minutes(5),
        memorySize: 128,
    });
    const constructProps = {
        modelProps: {},
        deployVpc: true,
        existingLambdaObj: fn,
    };
    const app = () => {
        new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -------------------------------------------------------------------------------------------------
// Test getter methods: existing Lambda function (with VPC), new Sagemaker endpoint, and existingVpc
// -------------------------------------------------------------------------------------------------
test('Test getter methods: existing Lambda function (with VPC), new Sagemaker endpoint, and existingVpc', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    // Add S3 VPC Gateway Endpint, required by Sagemaker to access Models artifacts via AWS private network
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.S3);
    // Add SAGEMAKER_RUNTIME VPC Interface Endpint, required by the lambda function to invoke the SageMaker endpoint
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.SAGEMAKER_RUNTIME);
    // deploy lambda function
    const fn = defaults.deployLambdaFunction(stack, {
        runtime: lambda.Runtime.PYTHON_3_8,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        handler: 'index.handler',
        timeout: core_1.Duration.minutes(5),
        memorySize: 128,
        vpc,
    });
    const constructProps = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        existingVpc: vpc,
        existingLambdaObj: fn,
    };
    const app = new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    // Assertions
    expect(app.lambdaFunction).toBeDefined();
    expect(app.sagemakerEndpoint).toBeDefined();
    expect(app.sagemakerEndpointConfig).toBeDefined();
    expect(app.sagemakerModel).toBeDefined();
    expect(app.vpc).toBeDefined();
});
// --------------------------------------------------------------------------------------------
// Test getter methods: new Lambda function, existingSagemakerendpointObj (no vpc)
// --------------------------------------------------------------------------------------------
test('Test getter methods: new Lambda function, existingSagemakerendpointObj (no vpc)', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const [sagemakerEndpoint] = defaults.deploySagemakerEndpoint(stack, {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
    });
    const constructProps = {
        existingSagemakerEndpointObj: sagemakerEndpoint,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(5),
            memorySize: 128,
        },
    };
    const app = new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    // Assertions
    expect(app.lambdaFunction).toBeDefined();
    expect(app.sagemakerEndpoint).toBeDefined();
    expect(app.sagemakerEndpointConfig).toBeUndefined();
    expect(app.sagemakerModel).toBeUndefined();
    expect(app.vpc).toBeUndefined();
});
// --------------------------------------------------------------------------------------------
// Test getter methods: new Lambda function, existingSagemakerendpointObj and deployVpc = true
// --------------------------------------------------------------------------------------------
test('Test getter methods: new Lambda function, existingSagemakerendpointObj and deployVpc = true', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const [sagemakerEndpoint] = defaults.deploySagemakerEndpoint(stack, {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
    });
    const constructProps = {
        existingSagemakerEndpointObj: sagemakerEndpoint,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(5),
            memorySize: 128,
        },
        deployVpc: true,
    };
    const app = new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    // Assertions
    expect(app.lambdaFunction).toBeDefined();
    expect(app.sagemakerEndpoint).toBeDefined();
    expect(app.sagemakerEndpointConfig).toBeUndefined();
    expect(app.sagemakerModel).toBeUndefined();
    expect(app.vpc).toBeDefined();
});
// --------------------------------------------------------------
// Test lambda function custom environment variable
// --------------------------------------------------------------
test('Test lambda function custom environment variable', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const [sagemakerEndpoint] = defaults.deploySagemakerEndpoint(stack, {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
    });
    new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', {
        existingSagemakerEndpointObj: sagemakerEndpoint,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.PYTHON_3_8,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        sagemakerEnvironmentVariableName: 'CUSTOM_SAGEMAKER_ENDPOINT'
    });
    // Assertion
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'python3.8',
        Environment: {
            Variables: {
                CUSTOM_SAGEMAKER_ENDPOINT: {
                    'Fn::GetAtt': [
                        'SagemakerEndpoint',
                        'EndpointName'
                    ]
                }
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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