# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['rgsync', 'rgsync.Connectors']

package_data = \
{'': ['*']}

install_requires = \
['SQLAlchemy==1.3.24', 'pymongo>=3.12.0,<4.0.0', 'redis==3.5.3']

setup_kwargs = {
    'name': 'rgsync',
    'version': '1.1.3',
    'description': 'RedisGears synchronization recipe',
    'long_description': '[![license](https://img.shields.io/github/license/RedisGears/rgsync.svg)](https://github.com/RedisGears/rgsync)\n[![CircleCI](https://circleci.com/gh/RedisGears/rgsync/tree/master.svg?style=svg)](https://circleci.com/gh/RedisGears/rgsync/tree/master)\n![GitHub Actions](https://github.com/redisgears/rgsync/actions/workflows/tox-tests.yml/badge.svg)\n[![PyPI version](https://badge.fury.io/py/rgsync.svg)](https://badge.fury.io/py/rgsync)\n[![Known Vulnerabilities](https://snyk.io/test/github/RedisGears/rgsync/badge.svg?targetFile=requirements.txt)](https://snyk.io/test/github/RedisGears/rgsync?targetFile=requirements.txt)\n[![Language grade: Python](https://img.shields.io/lgtm/grade/python/g/RedisGears/rgsync.svg?logo=lgtm&logoWidth=18)](https://lgtm.com/projects/g/RedisGears/rgsync/context:python)\n\n\n# RGSync\n[![Forum](https://img.shields.io/badge/Forum-RedisGears-blue)](https://forum.redislabs.com/c/modules/redisgears)\n[![Discord](https://img.shields.io/discord/697882427875393627?style=flat-square)](https://discord.gg/6yaVTtp)\n\nA _Write Behind_ and _Write Through_ Recipe for [RedisGears](https://github.com/RedisGears/RedisGears)\n\n## Demo\n![WriteBehind demo](demo/WriteBehindDemo.gif)\n\n## Example\nThe following is a RedisGears recipe that shows how to use the _Write Behind_ pattern to map data from Redis Hashes to MySQL tables. The recipe maps all Redis Hashes with the prefix `person:<id>` to the MySQL table `persons`, with `<id>` being the primary key and mapped to the `person_id` column. Similarly, it maps all Hashes with the prefix `car:<id>` to the `cars` table.\n\n```python\nfrom rgsync import RGWriteBehind, RGWriteThrough\nfrom rgsync.Connectors import MySqlConnector, MySqlConnection\n\n\'\'\'\nCreate MySQL connection object\n\'\'\'\nconnection = MySqlConnection(\'demouser\', \'Password123!\', \'localhost:3306/test\')\n\n\'\'\'\nCreate MySQL persons connector\n\'\'\'\npersonsConnector = MySqlConnector(connection, \'persons\', \'person_id\')\n\npersonsMappings = {\n\t\'first_name\':\'first\',\n\t\'last_name\':\'last\',\n\t\'age\':\'age\'\n}\n\nRGWriteBehind(GB,  keysPrefix=\'person\', mappings=personsMappings, connector=personsConnector, name=\'PersonsWriteBehind\',  version=\'99.99.99\')\n\n\'\'\'\nCreate MySQL cars connector\n\'\'\'\ncarsConnector = MySqlConnector(connection, \'cars\', \'car_id\')\n\ncarsMappings = {\n\t\'id\':\'id\',\n\t\'color\':\'color\'\n}\n\nRGWriteBehind(GB, keysPrefix=\'car\', mappings=carsMappings, connector=carsConnector, name=\'CarsWriteBehind\', version=\'99.99.99\')\n```\n\n## Running the recipe\nYou can use [this utility](https://github.com/RedisGears/gears-cli) to send a RedisGears recipe for execution. For example, run this repository\'s [example.py recipe](examples/mysql/example.py) and install its dependencies with the following command:\n\n```bash\ngears-cli --host <host> --port <post> --password <password> run example.py REQUIREMENTS rgsync PyMySQL cryptography\n```\n\n## Overview of the recipe\'s operation\nThe [`RGWriteBehind()` class](rgsync/redis_gears_write_behind.py) implements the _Write Behind_ recipe, that mainly consists of two RedisGears functions and operates as follows:\n1. A write operation to a Redis Hash key triggers the execution of a RedisGears function.\n1. That RedisGears function reads the data from the Hash and writes into a Redis Stream.\n1. Another RedisGears function is executed asynchronously in the background and writes the changes to the target database.\n\n### The motivation for using a Redis Stream\nThe use of a Redis Stream in the _Write Behind_ recipe implementation is to ensure the persistence of captured changes while mitigating the performance penalty associated with shipping them to the target database.\n\nThe recipe\'s first RedisGears function is registered to run synchronously, which means that the function runs in the same main Redis thread in which the command was executed. This mode of execution is needed so changes events are recorded in order and to eliminate the possibility of losing events in case of failure.\n\nApplying the changes to the target database is usually much slower, effectively excluding the possibility of doing that in the main thread. The second RedisGears function is executed asynchronously on batches and in intervals to do that.\n\nThe Redis Stream is the channel through which both of the recipe\'s parts communicate, where the changes are persisted in order synchronously and are later processed in the background asynchronously.\n\n## Controlling what gets replicated\nSometimes you want to modify the data in Redis without replicating it to the target. For that purpose, the recipe can be customized by adding the special field `#` to your Hash\'s fields and setting it to one of these values:\n* `+` - Adds the data but does not replicate it to the target\n* `=` - Adds the data with and replicates it (the default behavior)\n* `-` - Deletes the data but does not replicate\n* `~` - Deletes the data from Redis and the target (the default behavior when using `del` command)\n\nWhen the Hash\'s value contains the `#` field, the recipe will act according to its value and will delete the `#` field from the Hash afterward. For example, the following shows how to delete a Hash without replicating the delete operation:\n\n```\nredis> HSET person:1 # -\n```\n\nAlternatively, to add a Hash without having it replicated:\n```\nredis> HSET person:007 first_name James last_name Bond age 42 # +\n```\n\n## At Least Once and Exactly Once semantics\nBy default the _Write Behind_ recipe provides the _At Least Once_ property for writes, meaning that data will be written once to the target, but possibly more than that in cases of failure.\n\nIt is possible to have the recipe provide _Exactly Once_ delivery semantics by using the Stream\'s message ID as an increasing ID of the operations. The writer RedisGears function can use that ID and record it in another table in the target to ensure that any given ID is only be written once.\n\nAll of the recipe\'s SQL connectors support this capability. To use it, you need to provide the connector with the name of the "exactly once" table. This table should contain 2 columns, the `id` which represents some unique ID of the writer (used to distinguish between shards for example) and `val` which is the last Stream ID written to the target. The "exactly once" table\'s name can be specified to the connector in the constructor via the optional `exactlyOnceTableName` variable.\n\n## Getting write acknowledgment\nIt is possible to use the recipe and get an acknowledgment of successful writes to the target. Follow these steps to do so:\n1. For each data-changing operation generate a `uuid`.\n2. Add the operation\'s `uuid` immediately after the value in the special `#` field, that is after the `+`/`=`/`-`/`~` character. Enabling write acknowledgment requires the use of the special `#`.\n3. After performing the operation, perform an `XREAD BLOCK <timeout> STREAMS {<hash key>}<uuid> 0-0`. Once the recipe has written to the target, it will create a message in that (`{<hash key>}<uuid>`) Stream that has a single field named \'status\' with the value \'done\'.\n4. For housekeeping purposes, it is recommended to delete that Stream after getting the acknowledgment. This is not a must, however, as these Streams are created with TTL of one hour.\n\n### Acknowledgment example\n```\n127.0.0.1:6379> hset person:007 first_name James last_name Bond age 42 # =6ce0c902-30c2-4ac9-8342-2f04fb359a94\n(integer) 1\n127.0.0.1:6379> XREAD BLOCK 2000 STREAMS {person:1}6ce0c902-30c2-4ac9-8342-2f04fb359a94 0-0\n1) 1) "{person:1}6ce0c902-30c2-4ac9-8342-2f04fb359a94"\n   2) 1) 1) "1581927201056-0"\n         2) 1) "status"\n            2) "done"\n```\n\n## Write Through\n_Write Through_ is done by using a temporary key. The recipe registers to changes of that key and writes them to the target. Writing to the target is executed in the Server\'s main thread, in synchronous mode, which means that the server will be blocked at that time and the client will not get the reply until it is finished.\n\nWriting the changes to the target may succeed or fail. If successful, the recipe renames the temporary key to its intended final name. A failure will prevent the rename. In either case, the temporary key is deleted.\n\nThe semantics of the acknowledgment Stream remains nearly the same as _Write Behind_. The only change is in the message\'s structure. Failed writes create a message in that (`{<hash key>}<uuid>`) Stream that has:\n\n  * A \'status\' field with the value \'failed\'\n  * An \'error\' field containing the error\'s description\n\nNote that when using _Write Through_ it is mandatory to supply a `uuid` and read the acknowledgment Stream. That is the only way to tell whether the write had succeeded.\n\n_Write Through_ is registered using the `RGWriteThrough` class:\n```python\nRGWriteThrough(GB, keysPrefix, mappings, connector, name, version)\n```\n\nThe `keysPrefix` argument is the prefix of the key on which the writes will be triggered. The temporary key\'s name will be in the following format:\n```\n<keysPrefix>{<realKeyName>}\n```\nUpon success, the key is renamed to `<realKeyName>`.\n\nAny failure in writing to the target will cause the recipe to abort. In such cases, the temporary key is not renamed and is deleted.\n\nNote that in some cases, such as connection failures, it is impossible to tell whether the operation had succeeded or failed on the target. The recipe considers these as failures, although in reality, the write may have succeeded.\n\n### Example\nThese examples assume that the `keysPrefix` is set to "__". The first shows a successful write:\n\n```\n127.0.0.1:6379> HSET __{person:1} first_name foo last_name bar age 20 # =6ce0c902-30c2-4ac9-8342-2f04fb359a94\n(integer) 4\n127.0.0.1:6379> XREAD BLOCK 2000 STREAMS {person:1}6ce0c902-30c2-4ac9-8342-2f04fb359a94 0-0\n1) 1) "{person:1}6ce0c902-30c2-4ac9-8342-2f04fb359a94"\n   2) 1) 1) "1583321726502-0"\n         2) 1) "status"\n            2) "done"\n127.0.0.1:6379> HGETALL person:1\n1) "age"\n2) "20"\n3) "last_name"\n4) "bar"\n5) "first_name"\n6) "foo"\n```\n\nAn a example of a failed _Write Through_:\n\n```\n127.0.0.1:6379> HSET __{person:1} first_name foo last_name bar age 20 # =6ce0c902-30c2-4ac9-8342-2f04fb359a94\n(integer) 4\n127.0.0.1:6379> XREAD BLOCK 2000 STREAMS {person:1}6ce0c902-30c2-4ac9-8342-2f04fb359a94 0-0\n1) 1) "{person:1}6ce0c902-30c2-4ac9-8342-2f04fb359a94"\n   2) 1) 1) "1583322141455-0"\n         2) 1) "status"\n            2) "failed"\n            3) "error"\n            4) "Failed connecting to SQL database, error=\\"(pymysql.err.OperationalError) (2003, \\"Can\'t connect to MySQL server on \'localhost\' ([Errno 111] Connection refused)\\")\\n(Background on this error at: http://sqlalche.me/e/e3q8)\\""\n\n```\n\n### Speed Improvements\nTo improve the speed of write through updates, users should think about adding indexing to their write through database. This index would be created based on the column containing the *redis* key id being replicated. Using the example above, a *person_id* column will be created, regardless of the back-end database chosen for write through. As such, an index on the *person_id* column may be prudent, depending on your data volume and architecture.\n\n## Data persistence and availability\nTo avoid data loss in Redis and the resulting inconsistencies with the target databases, it is recommended to employ and use this recipe only with a highly-available Redis environment. In such environments, the failure of a master node will cause the replica that replaced it to continue the recipe\'s execution from the point it was stopped.\n\nFurthermore, Redis\' AOF should be used alongside replication to protect against data loss during system-wide failures.\n\n## Monitoring the RedisGears function registrations\nUse [this](https://github.com/RedisGears/RedisGearsMonitor) to monitor RedisGear\'s function registrations.\n',
    'author': 'Redis OSS',
    'author_email': 'oss@redis.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
