#! /usr/bin/python3
# -*- coding: utf-8 -*-

#####################################################################
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.
#####################################################################
"""Testing PyBen OOP implementation.

Example of single file .torrent metadata:

    tfile={
        "announce": "http://ubuntu.com/announce",
        "info": {
            "name": "ubuntu.iso",
            "length": 12845678
            "piece length": 262144,
            "private": 1,
            "source": "ubuntu",
        },
    "created by": "mktorrent",
    "creation date": 20398488923,
    }

"""

import os

import pytest

from pyben.classes import Bendecoder, Benencoder
from pyben.exceptions import DecodeError, EncodeError

from . import dicts, ints, lists, rmpath, strings, testfile, testmeta


@pytest.fixture
def tfile():
    """Testfile Pytest fixture."""
    tempfile = testfile()
    yield tempfile
    rmpath(tempfile)


@pytest.fixture
def tmeta():
    """Metadata Pytest fixture."""
    info = testmeta()
    return info


@pytest.fixture
def types():
    """Return fixtures."""
    return [lists, ints, strings, dicts]


def test_types(types):
    """Test fixture functionality."""
    assert len(types) > 1  # nosec


def test_decoder_constructor(strings):
    """Test Bendecoder constructor."""
    for string, benstring in strings:
        decoder = Bendecoder(benstring)
        decoded = decoder.decode()
        assert decoded == string  # nosec


def test_malformed_data():
    """Test data that cannot be interpreted by decoder."""
    data = b"li92e12:hello world!::e"
    decoder = Bendecoder(data)
    try:
        _ = decoder.decode()
    except DecodeError:
        assert True  # nosec


def test_improper_type():
    """Test type that isn't interpreted by encoder."""
    vals = [1, 2, 3, 4, 5]
    data = [12, "hello world!", set(vals)]
    encoder = Benencoder(data)
    try:
        _ = encoder.encode()
    except EncodeError:
        assert True  # nosec


def test_encode_tuple_to_list():
    """Test encoding tuple to list."""
    data = ((130, "foobar", "foo:bar"), b"li130e6:foobar7:foo:bare")
    assert Benencoder()._encode_list(data[0]) == data[1]  # nosec


def test_encode_tuple_cast():
    """Test encoding tuple to list main method."""
    data = ((130, "foobar", "foo:bar"), b"li130e6:foobar7:foo:bare")
    assert Benencoder().encode(data[0]) == data[1]  # nosec


def test_decode_str(strings):
    """Test string decoding."""
    decoder = Bendecoder()
    for string, benstring in strings:
        text, feed = decoder._decode_str(benstring)
        assert string == text  # nosec
        assert feed == len(benstring)  # nosec


def test_decode_int_class(ints):
    """Test integer decoding."""
    decoder = Bendecoder()
    for num, benint in ints:
        real, feed = decoder._decode_int(benint)
        assert real == num  # nosec
        assert feed == len(benint)  # nosec


def test_decode_list_class(lists):
    """Test list decoding."""
    decoder = Bendecoder()
    for lst, benlist in lists:
        decoded, _ = decoder._decode_list(benlist)
        assert decoded == lst  # nosec


def test_decode_dict_class(dicts):
    """Test dictionary decoding."""
    decoder = Bendecoder()
    for dct, bendict in dicts:
        decoded, _ = decoder._decode_dict(bendict)
        assert dct == decoded  # nosec


def test_decode_class(ints, strings, lists, dicts):
    """Test decoding."""
    data = [lists, strings, ints, dicts]
    decoder = Bendecoder()
    for val in data:
        for item, benitem in val:
            decoded = decoder.decode(benitem)
            assert decoded == item  # nosec


def test_decode_load(tfile):
    """Test inline decoding."""
    decoder = Bendecoder()
    output = decoder.load(tfile)
    assert isinstance(output, dict)  # nosec


def test_decode_loads(tfile):
    """Test from file decoding."""
    with open(tfile, "rb") as _fd:
        inp = _fd.read()
    decoder = Bendecoder()
    out = decoder.loads(inp)
    assert out["info"]["length"] == 12845738  # nosec


def test_bencode_str(strings):
    """Test string encoding."""
    encoder = Benencoder()
    for string, benstring in strings:
        text = encoder._encode_str(string)
        assert benstring == text  # nosec


def test_bencode_int(ints):
    """Test integer encoding."""
    encoder = Benencoder()
    for num, benint in ints:
        real = encoder._encode_int(num)
        assert real == benint  # nosec


def test_bencode_list(lists):
    """Test list encoding."""
    encoder = Benencoder()
    for lst, benlist in lists:
        encoded = encoder._encode_list(lst)
        assert encoded == benlist  # nosec


def test_bencode_dump(tmeta, tfile):
    """Test to file encoding with path string."""
    encoder = Benencoder()
    encoder.dump(tmeta, tfile)
    assert os.path.exists(tfile)  # nosec


def test_bencode_dump1(tmeta, tfile):
    """Test to file encoding with FileIO."""
    encoder = Benencoder()
    with open(tfile, "wb") as a:
        encoder.dump(tmeta, a)
    assert os.path.exists(tfile)  # nosec


def test_bencode_dumps(tmeta):
    """Test inline encoding."""
    encoder = Benencoder()
    reg = encoder.dumps(tmeta)
    assert isinstance(reg, bytes)  # nosec


def test_encode_dict(dicts):
    """Test dictionary encoding."""
    encoder = Benencoder()
    for dct, bendict in dicts:
        encoded = encoder._encode_dict(dct)
        assert bendict == encoded  # nosec


def test_encode(ints, strings, lists, dicts):
    """Test encoding."""
    data = [ints, strings, lists, dicts]
    encoder = Benencoder()
    for val in data:
        for item, benitem in val:
            encoded = encoder.encode(item)
            assert encoded == benitem  # nosec


def test_bendecoder_load(tfile):
    """Test inline decoding."""
    decoder = Bendecoder()
    with open(tfile, "rb") as _fd:
        data = decoder.load(_fd)
    assert data is not None  # nosec
