# AUTOGENERATED! DO NOT EDIT! File to edit: 02_network_class.ipynb (unless otherwise specified).

__all__ = ['bn_integral', 'bn_integral_risefall', 'fig_params', 'force_aspect', 'logic_colors', 'plot_graph',
           'differential_labels', 'plot_timeseries', 'perturb', 'stack', 'perturb_init', 'BooleanNetwork', 'ring_bn',
           'random_bn', 'mpx_edges', 'delay_edges', 'plot_comparison']

# Cell
import warnings
with warnings.catch_warnings(): #ignore warnings
    warnings.simplefilter("ignore")
    import networkx as nx
    import numpy as np
    import sidis
    rng=sidis.RNG(0)
    import matplotlib.pyplot as plt
    import typing
    from typing import Optional, Tuple, Dict, Callable, Union, Mapping, Sequence, Iterable, Hashable, List, Any
    from collections import namedtuple
    import numba
    from numba import njit
    from .graph_functions import *
    from .model_functions import *
    import matplotlib as mpl

# Cell
@njit
def bn_integral(iterator,
               time_delays,
               sigmoid_constants,
               time_constants,
               predecessors,
               initial_condition_matrix,
               hold_times,
               dt,
               T,
               noise_scale,
               decimation,
               repeats=1
              ):
    '''
    Jit accelerated integration 'BooleanNetwork' class. Nearly identical to
    `bool_integral` with a loop over repeated `initial_conditions` and without
    a noise argument, in favor of generating noise internal to the function.
    '''
    C,N=initial_condition_matrix.shape #challenges,repeats,nodes
    responses = np.zeros((C,repeats,int(T/decimation),N))

    for c,r in np.ndindex(C,repeats): #diff inits
        initial_conditions=initial_condition_matrix[c]
        x=np.zeros((T,N)).astype(np.longdouble)
        dx=np.zeros(x.shape[-1]).astype(np.longdouble)
        for t in range(x.shape[0]-1):
            noise=np.empty(x.shape[1])
            for n in range(x.shape[1]):
                noise[n]=np.random.random()*noise_scale
            #noise=noise*noise_scale
            edge_index=0
            if t<max(hold_times):
                for n in range(x.shape[-1]):
                    if hold_times[n]>=t:
                        x[t,n]=initial_conditions[n]
            for i in range(len(iterator)):
                n1,n2,deg,mask=iterator[i]
                d=-time_delays[edge_index:edge_index+(n2-n1)*deg].reshape((n2-n1,deg))
                d+=t
                p=predecessors[edge_index:edge_index+(n2-n1)*deg].reshape((n2-n1,deg))
                a=sigmoid_constants[n1:n2].reshape((n2-n1,1))
                edge_index+=(n2-n1)*deg
                y=np.zeros((n2-n1,deg))#.astype(np.longdouble)
                for k in range(n2-n1):
                    for j in range(deg):
                        de=d[k,j]
                        pr=p[k,j]
                        y[k,j]=x[de,pr]
                y=sigmoid(x=y,a=a)
                dx[n1:n2]=BOOL(y,mask)
            dx=sigmoid(dx,sigmoid_constants)
            dxdt=(-x[t]+dx+noise)/time_constants
            x[t+1]=x[t]+dt*dxdt

        responses[c,r]=x[::decimation]

    return responses

#export
@njit
def bn_integral_risefall(iterator,
               time_delays,
               sigmoid_constants,
               time_constants,
               predecessors,
               initial_condition_matrix,
               hold_times,
               dt,
               T,
               noise_scale,
               decimation,
               repeats=1
              ):
    '''
    Jit accelerated integration 'BooleanNetwork' class. Nearly identical to
    `bool_integral_risefall` with a loop over repeated `initial_conditions` and without
    a noise argument, in favor of generating noise internal to the function.
    '''
    C,N=initial_condition_matrix.shape #challenges,repeats,nodes
    responses = np.zeros((C,repeats,int(T/decimation),N))

    for c,r in np.ndindex(C,repeats): #diff inits
        initial_conditions=initial_condition_matrix[c]
        x=np.zeros((T,N)).astype(np.longdouble)
        dx=np.zeros(x.shape[-1]).astype(np.longdouble)
        for t in range(x.shape[0]-1):
            noise=np.empty(x.shape[1])
            for n in range(x.shape[1]):
                noise[n]=np.random.random()*noise_scale
            #noise=noise*noise_scale
            edge_index=0
            if t<max(hold_times):
                for n in range(x.shape[-1]):
                    if hold_times[n]>=t:
                        x[t,n]=initial_conditions[n]
            for i in range(len(iterator)):
                n1,n2,deg,mask=iterator[i]
                d=-time_delays[edge_index:edge_index+(n2-n1)*deg].reshape((n2-n1,deg))
                d+=t
                p=predecessors[edge_index:edge_index+(n2-n1)*deg].reshape((n2-n1,deg))
                a=sigmoid_constants[n1:n2].reshape((n2-n1,1))
                edge_index+=(n2-n1)*deg
                y=np.zeros((n2-n1,deg))#.astype(np.longdouble)
                for k in range(n2-n1):
                    for j in range(deg):
                        de=d[k,j]
                        pr=p[k,j]
                        y[k,j]=x[de,pr]
                y=sigmoid(x=y,a=a)
                dx[n1:n2]=BOOL(y,mask)
            dx=sigmoid(dx,sigmoid_constants)
            tau=time_constants[:,0]+(time_constants[:,1]-time_constants[:,0]
                                    )*sigmoid(x[t],sigmoid_constants)
            dxdt=(-x[t]+dx+noise)/tau
            x[t+1]=x[t]+dt*dxdt

        responses[c,r]=x[::decimation]

    return responses

# Cell
def fig_params(X : float = 3.5,Y : float = 3,
              hspace : float = 0.0,
              offset : float = -.4,
              font : str = 'Times New Roman',
              fontsize : int = 12,
              ticksize : int = 6,
              tickwidth : int = 2,
              linewidth : int = 2,
              reset : bool = False,
             ):
    '''
    Changes the `rcParams` for plotting, with the option to `reset` to default.
    '''
    if reset:
        mpl.rcParams.update(mpl.rcParamsDefault)
        return
    else:
        #figure font Times New Roman,Helvetica, Arial, Cambria, or Symbol
        mpl.rcParams['font.family'] = font
        mpl.rcParams['font.size'] = fontsize
        mpl.rcParams['axes.titlesize'] = fontsize
        mpl.rcParams['axes.linewidth'] = linewidth
        mpl.rcParams['axes.titley'] = offset


        mpl.rcParams['xtick.major.size'] = ticksize
        mpl.rcParams['xtick.major.width'] = tickwidth
        mpl.rcParams['xtick.minor.size'] = ticksize//2
        mpl.rcParams['xtick.minor.width'] = tickwidth//2
        mpl.rcParams['xtick.direction']='out'

        mpl.rcParams['ytick.major.size'] = ticksize
        mpl.rcParams['ytick.major.width'] = tickwidth
        mpl.rcParams['ytick.minor.size'] = ticksize//2
        mpl.rcParams['ytick.minor.width'] = tickwidth//2
        mpl.rcParams['ytick.direction']='out'

        mpl.rcParams['figure.figsize'] = [X,Y]
        mpl.rcParams['figure.subplot.hspace'] = hspace

# Cell
def force_aspect(ax,aspect=1):
    '''
    Helper function to force the aspect of the matplotlib 'ax' axes object.
    '''
    try:
        im = ax.get_images()
        extent =  im[0].get_extent()
    except:
        x,y=ax.get_xlim(),ax.get_ylim()
        extent = [x[0],x[1],y[0],y[1]]
    ax.set_aspect(abs((extent[1]-extent[0])/(extent[3]-extent[2]))/aspect)

# Cell
logic_colors={XOR:'r',NOT:'b',MPX:'g',AND:'y',OR:'m',COPY:'c',XNOR:'k',NAND:'w'}

# Cell
def plot_graph(b,XY=(5,5),
               layout=nx.kamada_kawai_layout,layout_kwargs=dict(weight='delay'),
               colors=logic_colors,
               ls='--',lw=0.1,head_width=0.035,color='black',shape='full'):
    '''
    Plot the boolean network 'b' using 'layout'.
    Defaults to using the path-length cost function 'nx.kamada_kawai_layout'
    which attempts to force the path between each node to be scaled by the
    'weight', in this case the time-delay. Other args are for plotting arrows.
    '''
    #mpl.rcParams['figure.figsize'] = [XY[0],XY[1]]
    try:
        d=b.layout
    except:
        d=layout(b,**layout_kwargs)
        b.layout=d

    xy=np.array(list(d.values()))
    fig,ax=plt.subplots(figsize=XY)
    for func,nodes in b.funcs.items():
        ax.scatter(xy[nodes,0],xy[nodes,1],c=[colors[b.f[n]] for n in nodes],label=func.__name__)
    force_aspect(ax)
    for v,(x,y) in d.items():
        p=b.in_edges(v,keys=True)
        for e in p:
            u=e[0]
            x0,y0=d[u]
            dx,dy=x0-x,y0-y
            ax.arrow(x,y,dx,dy,
                      shape=shape,
                      color=color,
                      linestyle=ls,
                      lw=lw,
                      length_includes_head=True,
                      head_width=head_width)

    #from matplotlib.font_manager import FontProperties
    #fontP = FontProperties()
    #fontP.set_size(set_size)
    lgd=fig.legend(loc='lower center',
                   #bbox_to_anchor=(0.5,1.1),
                   ncol=len(b.funcs))
                   #prop=fontP,
                   #fancybox=True, shadow=True)
    #ax.set_title(layout.__name__)
    ax.axis('off')
    plt.show()
    #fig_params(reset=True)

# Cell
def differential_labels(bn,nodes=None):
    '''
    Generate text of the differential equations for each node
    '''
    if nodes is None:
        nodes=bn.nodes
    template=lambda i,p: r'$\tau_{i} \.x_{i} = -x_{i}(t)+{p}{eps}$'.format(i=i,p=p,
                          eps='+\epsilon' if bn.noise else '')
    labels=[]
    delay='delay'
    f='f'
    for i in nodes:
        p=','.join([f'X_{e[0]}(t-{bn.edges[e][delay]:.1f})' for e in bn.in_edges(i,keys=True)])
        p=f'{bn.nodes[i][f].__name__}['+p
        p+=']'
        labels+=[template(i=i,p=p)]
    return labels

# Cell
def plot_timeseries(xs,labels,colors,t0=0,t1=-1,alpha=0.3,
                    loc='upper center',
                    bbox_to_anchor=(0.5, 1.3),
                    ncol=1,
                    set_size='small',
                    fname=None,
                    shrink=0.8,
                     dt=0.01,
                     ls='-',
                     xy=(3,3),
                     handlelength=0,
                     handletextpad=0,
                     fancybox=True,
                     shadow=True,
                     left=None, bottom=None, right=None, top=None, wspace=0.5, hspace=0.1,
                     fontsize=10
                   ):
    from matplotlib.font_manager import FontProperties
    fontP = FontProperties()
    fontP.set_size(set_size)

    clip = lambda v,t0=t0,t1=t1: np.clip(v[t0:t1],0,1)
    xs = [clip(x) for x in xs]
    I=len(xs)
    fig,ax=plt.subplots(nrows=I,ncols=1,sharex=True, sharey=False,figsize=xy)
    time_axis=np.around((np.arange(xs[0].shape[0])*dt),2)
    plots=[]
    for i,x in enumerate(xs):
        p,=ax[i].plot(time_axis,x,color=colors[i],
                      ls=ls,
                      label=labels[i])
        plots+=[p]
        #ax[i].spines['right'].set_visible(False)
        #ax[i].spines['top'].set_visible(False)
        #ax[i].spines['bottom'].set_visible(False if i!=I-1 else True)
        ax[i].xaxis.set_visible(False if i!=I-1 else True)
        #ax[i].legend(loc=loc, bbox_to_anchor=bbox_to_anchor,
        #             ncol=ncol, fancybox=True, shadow=True,prop=fontP)
        #leg = ax[i].legend(handlelength=handlelength,
        #                   handletextpad=handletextpad, loc=loc, bbox_to_anchor=bbox_to_anchor,
        #             ncol=ncol, fancybox=fancybox, shadow=shadow,prop=fontP)#,mode="expand", borderaxespad=0.)
        #for item in leg.legendHandles:
        #    item.set_visible(False)
        ax[i].set_title(labels[i],fontsize=fontsize)
        ax[i].set_ylabel(r'$x_{0}$'.format(i))
        #force_aspect(ax[i],len(xs))
        #box = ax[i].get_position()
        #ax[i].set_position([box.x0, box.y0, box.width * shrink, box.height])

    plt.xlabel(r'Time ($\tau$)')



    #lgd=fig.legend(handles=plots,loc=loc, bbox_to_anchor=bbox_to_anchor,
    #      ncol=ncol, fancybox=True, shadow=True,prop=fontP)

    if fname:
        plt.savefig(fname=fname,bbox_inches='tight',dpi=600)
    fig.subplots_adjust(left=left, bottom=bottom, right=right, top=top, wspace=wspace, hspace=hspace)
    plt.tight_layout()
    plt.show()


# Cell
def perturb(a,scale=0.1):
    '''
    Helper function for randomly shuffling entries of an array 'a'
    by the fraction 'scale'.
    '''
    return (a*(1.+rng.random(-scale/2.,scale/2.,shape=a.shape))).astype(a.dtype)

# Cell
def stack(a,times=1,axis=0):
    '''
    Helper function for stacking an array 'a' along its 'axis'
    a certain number of 'times'.
    '''
    return np.repeat(np.expand_dims(a,axis),times,axis=axis).astype(a.dtype)

# Cell
def perturb_init(init):
    '''
    Helper function for generating all single bit flips
    of a binary array 'init', e.g, init=[0,0] returns
    [0,0],[1,0],[0,1].
    '''
    I=len(init)
    inits=stack(init,axis=0,times=I+1)
    for n in range(I):
        inits[n+1][n]=sidis.NOT(inits[n+1][n])
    return inits

# Cell
class BooleanNetwork(nx.MultiDiGraph):
    '''
    Model the dynamics of the graph `g` by giving the node attributes.

        f : logical function.

        a : sigmoid function.

        tau : time constant.

    and edge attributes

        delay : time-delay

    Each can be dict keyed by node, a single value, or tuple of callable/args choosing val;

    see `parse_kwargs`. Converts any edges with the given `edge_replacements` (see

    `convert_edges` function for arguments); useful for `MPX`.

    Sorts the graph in place using `sort_graph` in order to produce

    an iterable with `bool_model_iter`. See also `setup_bool_integral`.

    Initializes the dynamics to `init` using the hold times `hold`;

    see `bn_initial_conditions`. Integrates with `bool_integral`

    and `bn_integral_risefall` if `tau` is given as an array of

    [rise_time,fall_time]. Returns the node state array `x`

    and optionally plots the resulting dynamics with `plot_graph`.

    Includes most functions from the `graph_functions` library.

    Other args:

        edge_replacements: dict of replacements for edges

        T: integration time

        dt: integration timestep

        noise: amplitude of noise fluctuation at each timestep

        steady: whether to map boolean functions over initial condition first

        view: whether to change default edge view (in/out)

        plot: whether to automatically integrate and plot the given vals

        init: default initial condition to apply

        hold: default hold times for initial conditions

        take_attrs: whether to override the above and absorb any attributes from `g`

    '''

    def __init__(self,
                 g : nx.MultiDiGraph = ring(N=3,right=True,left=False,loop=False),
                 a : Union[float,Dict[Any,float],Tuple[Union[Callable,float]]] = np.inf,
                 tau : Union[float,Dict[Any,float],Tuple[Union[Callable,float]]] = 1.,
                 f : Union[Callable,Dict[Any,float],Tuple[Union[Callable,float]]] = NOT,
                 delay : Union[float,Dict[Any,float],Tuple[Union[Callable,float]]] = 0.,
                 edge_replacements : Optional[dict] = None,
                 T : int = 25,
                 dt : Union[float,int] = 0.01,
                 noise : Union[float,int] = 0.,
                 steady : bool = False,
                 view : str = 'out',
                 plot : bool = False,
                 init : Union[float,Dict[Any,float],Tuple[Union[Callable,float]]] = None,
                 hold : Union[float,Dict[Any,float],Tuple[Union[Callable,float]]] = None,
                 take_attrs : bool = True
                ):

        #super init
        super().__init__(g)
        #change edge view, if desired
        self.view(view)
        #give node/edge attrs
        self.give_nodes(a=a,tau=tau,f=f)
        self.give_edges(delay=delay)
        #replace them with current data, if available
        if take_attrs:
            self.take_attrs(g)
        #list original topology
        self.original_nodes=list(g.nodes)
        self.original_edges=list(g.edges)
        #replace edges with nodes if applicable
        if edge_replacements:
            self.convert_edges(edge_replacements)
            self.new_nodes=[n for n in self.nodes if n not in self.original_nodes]
            self.new_edges=[n for n in self.edges if n not in self.original_edges]
        else:
            self.new_nodes=None
            self.new_edges=None
        #sort the graph object in place and convert node labels to integers
        self.sort(relabel=True)
        #create jit integrator iterator and store groups of funcs
        self.iter,self.funcs=bool_model_iter(self,return_funcs=True)
        #set aliases for functions as self attrs
        for k,v in self.funcs.items():
            setattr(self,k.__name__,v)
        #set nodes to be forced for initial conds
        try:
            self.dynamical_nodes=self.MPX
        except:
            self.dynamical_nodes=self.original_nodes
        #set noise and timescales
        self.noise=noise
        self.dt=dt
        self.T=int(T/dt)
        #save node/edge data as self attr arrays
        self.node_data(save=True)
        self.edge_data(save=True)
        #ensure float type to prevent integer division
        self.a=self.a.astype(float)
        self.tau=self.tau.astype(float)
        self.delay=self.delay.astype(float)
        #if hold given as a float, treat it as a constant
        if isinstance(hold,(int,float)):
            hold=[hold for n in self.dynamical_nodes]
        #if hold given as a tuple, treat it as a delayed func eval
        elif isinstance(hold,tuple):
            hold=[parse_kwargs(h=hold)['h'] for n in self.dynamical_nodes]
        #if init given as a tuple, treat it as a delayed func eval
        #elif isinstance(hold,tuple):
        #    init=[parse_kwargs(i=init)['i'] for n in self.dynamical_nodes]
        #convert the initial conditions to match the dynamical nodes, steady state optional
        self.init,self.hold=bool_initial_conditions(self,init=init,hold=hold,steady=steady)
        self.init=np.array(self.init).astype(np.float64)
        #predecessor array alias
        self.predar=np.concatenate([list(self.predecessors(n)) for n in self.nodes])
        #choose integral based on if different rise/falls exist
        if len(self.tau.shape)!=1:
            self.integral=bn_integral_risefall
        else:
            self.integral=bn_integral
        #if plot, integrate default and show timeseries
        if plot:
            self.x=self.integrate()
            self.plot(self.x)

    def take_attrs(self,
                   g,
                   node_attrs=['a','tau','f'],
                   edge_attrs=['delay']
                  ):
        '''
        Absorb the 'node_attrs' and 'edge_attrs' from graph 'g'.
        '''
        for v in g.nodes:
            for a in node_attrs:
                try:
                    self.nodes[v][a]=g.nodes[v][a]
                except:
                    pass
        for e in g.edges:
            for a in edge_attrs:
                try:
                    self.edges[e][a]=g.edges[e][a]
                except:
                    pass


    def parse_initial_condition(self,
                                init=None,
                                hold=None,
                                steady=False,
                                override=False):
        '''
        Store any given 'init' and 'hold' values as self attrs.
        '''
        if init is not None and hold is not None: #if both given, replace current
            self.init,self.hold=bool_initial_conditions(self,init=init,hold=hold,steady=steady)
        elif init is None and hold is not None: #replace hold
            _,self.hold=bool_initial_conditions(self,init=init,hold=hold,steady=steady)
        elif init is not None and hold is None:
            self.init,_=bool_initial_conditions(self,init=init,hold=hold,steady=steady)
            #else save current vals if both none
        self.init,self.hold=self.init.astype(np.longdouble),self.hold.astype(np.int64)

    def vary(self,*args,scale=0.1):
        '''
        Perturb the attributes given as 'args' by 'scale'.
        e.g, vary('a',0.1) sets bn.a = perturb(bn.a,0.1)
        '''
        for a in args:
            if isinstance(a,(float,int)):
                scale=a
            else:
                setattr(self,a,perturb(getattr(self,a),scale=scale))

    def random_init(self): #random initial conditions
        '''
        Random initial condition compatible with 'parse_initial_condition'
        '''
        return rng.random(0,1,asint=True,shape=(len(self.dynamical_nodes)))

    def integrate(self,
                  init=None,
                  hold=None,
                  noise=None,
                  T=None,
                  dt=None,
                  steady=False,
                  override=False,
                  repeats=1,
                  decimation=1,
                  save=False #whether to save as self.x
                 ):
        '''
        Integrate the given dynamics.
            init: initial condition of the dynamical nodes.
                  can be a matrix, i.e multiple arrays of initial conditions
            hold: hold times of dynamical nodes
            noise: float value for max random noise at each timestep
            T,dt: integration time and timestep (T gets converted to T/dt internally)
            steady: whether to map the node boolean functions over the init first,
                    before starting the integration. For MPXs, this is ignored
            override: whether to ignore processing of init and hold and use
                      the exact values given, e.g non-standard inits for non-dynamical nodes
            repeats: how many times to repeat each init
            decimation: how many timesteps to ignore in the output, e.g x[::decimation]
        '''
        #update self attrs
        if dt is not None:
            self.dt=dt
        if decimation is None:
            decimation=int(1/dt)
        if T is None:
            T=self.T
        else:
            self.T=int(T/self.dt)
        if noise is not None:
            self.noise=noise
        #typecast
        if isinstance(hold,(int,float)):
            hold=[hold for n in self.dynamical_nodes]
        if init is not None and not isinstance(init,np.ndarray):
            init=np.array(init)
        if hold is not None and not isinstance(hold,np.ndarray):
            hold=np.array(hold)
        #if multiple inits given, parse each and store as matrix
        if isinstance(init,np.ndarray) and len(init.shape)>1:
            inits=np.zeros((init.shape[0],len(self))).astype(np.float64)
            for i in range(len(init)):
                self.parse_initial_condition(init=init[i],hold=hold,steady=steady,override=override)
                inits[i]=self.init
            self.init=inits
        else:#only one init, parse if given otherwise use currently saved init/hold
            if init is None and hold is None:
                pass
            else:
                self.parse_initial_condition(init=init,hold=hold,steady=steady,override=override)
        #final check that init and hold are array type
        self.init=np.array(self.init)
        self.hold=np.array(self.hold)
        #expand dims of init for jit integrator, which accepts a matrix
        self.init=self.init if len(self.init.shape)==2 else stack(self.init,times=1,axis=0)
        #integrate with appropriate type of each var for jit
        x=self.integral(iterator=self.iter.astype(np.int64),
                        time_delays=(self.delay/self.dt).astype(np.int64),
                        sigmoid_constants=self.a.astype(np.longdouble),
                        time_constants=self.tau.astype(np.longdouble),
                        predecessors=self.predar.astype(np.int64),
                        initial_condition_matrix=self.init.astype(np.longdouble),
                        hold_times=(self.hold/self.dt).astype(np.int64),
                        dt=self.dt,
                        T=self.T,
                        noise_scale=self.noise,
                        decimation=decimation,
                        repeats=repeats
                       )
        #remove any extra dimensions and return
        self.init=np.squeeze(self.init)
        x=np.squeeze(x)
        if not save:
            return x
        else:
            self.x=x

    def query(self,
              challenges=1,
              repeats=1,
              instances=1,
              scale=0.1,
              variables=('a','tau','hold'),
              noise=None,
              T=None,
              dt=None,
              steady=False,
              override=False,
              decimation=1,
              hold=None):
        '''
        Integrate the initial conditions 'challenges' 'repeats' number of times.
        Do this for each 'instances', which correspond to perturbed 'variables'
        mimicing the effects of process variation on e.g node time constants.
        `challenges` can be an array of inits, or an integer, which specifies
        the number of random inits to choose.
        '''
        #default decimation to inverse timestep if not given
        if decimation is None:
            decimation=int(1/self.dt)
        #if int, choose random inits
        if isinstance(challenges,int):
            inits=np.array([self.random_init() for c in range(challenges)])
            challenges=len(inits)
        else: #use given inits
            inits=challenges
            challenges=len(inits)
        #integration results
        responses=np.zeros((instances,challenges,repeats,int(self.T/decimation),len(self)))
        #vary properties of each instance and integrate
        for i in range(instances):
            self.vary(*variables,scale=scale)
            responses[i]=self.integrate(init=inits,hold=hold,noise=noise,dt=dt,T=T,steady=steady,
                                       override=override,repeats=repeats,decimation=decimation)
        return responses


    def view(self,view='in'):
        '''
        Change default edge view
        '''
        if view=='in':
            self.edge_view=nx.classes.reportviews.InMultiEdgeView

        elif view=='out':
            self.edge_view=nx.classes.reportviews.OutMultiEdgeView


    @nx.MultiDiGraph.edges.getter
    def edges(self):
        return self.edge_view(self)


    def sort(self,**kwargs):
        '''
        See `sort`
        '''
        sort_graph(self,**kwargs)


    def relabel(self,mapping=None):
        '''
        See `relabel`
        '''
        relabel_graph(self,mapping)


    def node_attrs(self):
        '''
        See `node_attrs`
        '''
        return node_attrs(self)


    def edge_attrs(self):
        '''
        See `edge_attrs`
        '''
        return edge_attrs(self)


    def give_nodes(self,data=None,nodes=None,**kwargs):
        '''
        See `give_nodes`
        '''
        give_nodes(self,data=data,nodes=nodes,**kwargs)


    def give_edges(self,data=None,edges=None,**kwargs):
        '''
        See `give_edges`
        '''
        give_edges(self,data=data,edges=edges,**kwargs)

    def give_self(self,data=None,**kwargs):
        '''
        Give any arg dict and kwargs to self as attrs
        '''
        try:
            sidis.give(self,**parse_kwargs(**kwargs))
        except:
            pass

        if data:
            for k,v in data.items():
                try:
                    self.__dict__.update(parse_kwargs(**v))
                except KeyError:
                    pass


    def clear_nodes(self,*args):
        '''
        Remove arg entries from node dict
        '''
        if not args:
            args=self.node_attrs()
        [[sidis.give(self.node[n],arg,None) for n in self.nodes] for arg in args]


    def clear_edges(self,*args):
        '''
        Remove arg entries from edge dict
        '''
        if not args:
            args=self.edge_attrs()
        [[sidis.give(self.edges[e],arg,None) for e in self.edges] for arg in args]


    def node_data(self,*args,save=False):
        '''
        Set node attributes as self attribute array.
        '''
        if not save:
            return node_data(self,*args)
        else:
            self.give_self(**node_data(self,*args))


    def edge_data(self,*args,save=False):
        '''
        Set edge attributes as self attribute array.
        '''
        if not save:
            return edge_data(self,*args)
        else:
            self.give_self(**edge_data(self,*args))


    def convert_edges(self,edge_replacements=None):
        '''
        see `convert_edges`; this function pipelines those
        arguments for BooleanNetwork attrs
        and sets defaults to multiplexing behavior
        '''
        if edge_replacements is not None:
            label=edge_replacements.get('label')
            edges=edge_replacements.get('edges')
            lengths=edge_replacements.get('lengths')
            a=edge_replacements.get('a')
            tau=edge_replacements.get('tau')
            f=edge_replacements.get('f')
            delay=edge_replacements.get('delay')
            if label is None:
                label = lambda g,node,iterable : len(g)+iterable
            if lengths is None:
                lengths=1
            if delay is None:
                delay = 0
            if a is None:
                a=np.inf
            if tau is None:
                tau=1
            if f is None:
                f=MPX
            convert_edges(self,edges=edges,lengths=lengths,delay=delay,label=label,
                         node_data=dict(a=a,f=f,tau=tau))


    def where(self,*args,**kwargs):
        '''
        See `where`
        '''
        return where(self,*args,**kwargs)

    def edgewhere(self,*args):
        '''
        `where` but args assume edges
        '''
        return np.array(self.edges)[self.where(*args)]

    def nodewhere(self,*args):
        '''
        `where` but args assume nodes
        '''
        return np.array(self.nodes)[self.where(*args)]

    def __str__(self):
        '''
        Yield string of the graph topology
        '''
        return print_graph(self,string=True)

    def __repr__(self):
        '''
        Truncate self string as representation
        '''
        s=str(self)
        S=super().__repr__()
        spl=s.split('\n')
        if len(spl)>10:
            spl=spl[:5]+['...']+spl[-5:]
        return S+'\n'+('\n').join(spl)

    def plot(self,x=None):
        '''
        Make separate plots of the node states `x` for each in-degree grouping.
        '''
        if x is None:
            try:
                x=self.x
            except:
                self.integrate(save=True)
                x=self.x
        for f,nodes in self.funcs.items():
            for i in nodes:
                plt.plot(np.arange(x.shape[0])*self.dt,x[:,i])
                title=f'{f.__name__} Nodes: {nodes[0]} to {nodes[-1]}'
                plt.title(title)
                plt.xlabel(r'Time $(\tau)$')
                plt.ylabel('Amplitude')
            plt.show()

    def plot_3d(self,
                x=None,
                Tstart=0,
                Tstop=None,
                nodes=None):
        '''
        Plot 3-D graph of three nodes over given timespan.
        Default to first 3 nodes and full timerange.
        '''
        if x is None:
            try:
                x=self.x
            except:
                self.integrate(save=True)
                x=self.x
        if nodes is None:
            nodes=list(self.nodes)[:3]
        if Tstop is None:
            Tstop=self.T
        fig = plt.figure()
        ax = fig.add_subplot(111, projection='3d')
        ax.scatter(x[Tstart:Tstop,nodes[0]],
                   x[Tstart:Tstop,nodes[1]],
                   x[Tstart:Tstop,nodes[2]],
                   c=np.arange(Tstop-Tstart), cmap='YlGn')


    def plot_graph(self,**kwargs):
        '''
        see `plot_graph`
        '''
        with warnings.catch_warnings(): #ignore warnings
            warnings.simplefilter("ignore")
            plot_graph(self,**kwargs)


    def plot_timeseries(self,nodes=None,xs=None,labels=None,colors=None,**kwargs):
        if nodes is None:
            nodes=self.nodes
        nodes=list(nodes)
        if xs is None:
            try:
                xs=np.squeeze(self.x)
            except:
                self.integrate(save=True)
                xs=np.squeeze(self.x)
        xs=xs[:,nodes]
        if labels is None:
            labels=np.array(differential_labels(self))[nodes]
        if colors is None:
            colors=[logic_colors[self.f[n]] for n in nodes]
        plot_timeseries(xs=xs.T,labels=labels,colors=colors,**kwargs)



# Cell
mpx_edges = dict(delay=0,a=np.inf,tau=0.5,f=MPX)
delay_edges = dict(f=COPY,lengths='delay',a=np.inf,tau=1,delay=0)

def ring_bn(N=3,
            left=True,
            right=False,
            loop=False,
            a=np.inf,
            tau=1,
            delay=0,
            f=NOT,
            edge_replacements=None,
            dt=0.01,
            T=15,
            noise=0,
            init=None,
            hold=None,
            steady=False,
            plot=False,
            view='out'):
    '''
    Factory function for producing a `ring` `BooleanNetwork`.
    '''
    g=ring(N=N,left=left,right=right,loop=loop)
    return BooleanNetwork(g=g,a=a,tau=tau,f=f,delay=delay,edge_replacements=edge_replacements,
               T=T,dt=dt,noise=noise,init=init,hold=hold,steady=steady,plot=plot,view=view)

def random_bn(N=256,
              k=3,
              a=np.inf,
              tau=1,
              delay=0,
              f=NOT,
              edge_replacements=None,
              dt=0.01,
              T=25,
              noise=0,
              init=None,
              hold=None,
              steady=False,
              plot=False,
              view='out',
              seed=None):
    '''
    Factory function for producing a `nx.random_regular_graph` `BooleanNetwork`.
    '''
    g=nx.random_regular_graph(k,N,seed=seed)
    return BooleanNetwork(g=g,a=a,tau=tau,f=f,delay=delay,edge_replacements=edge_replacements,
               T=T,dt=dt,noise=noise,init=init,hold=hold,steady=steady,plot=plot,view=view)

# Cell
def plot_comparison(x0,x,y0,y,i=0,t0=0,t1=-1,alpha=0.3,
                    loc='upper center',
                    bbox_to_anchor=(0.475, 1.2),
                    ncol=2,
                    set_size='x-small',
                    shrink=1,
                    fname=None,
                    figsize=(3.5,3)
                   ):


    #plt.rcParams['figure.figsize'] = [X,Y]
    z = lambda v,t0=t0,t1=t1,i=i: np.clip(v[t0:t1,i],0,1)
    base=z(x0)
    comparisons=[z(x),z(y0),z(y)]
    colors=['black','red','blue','purple']
    labels=['noise','heterogeneity','noise $&$ heterogeneity']
    I=len(comparisons)
    fig,ax=plt.subplots(I,sharex=True, sharey=True,figsize=figsize)
    t=np.around((np.arange(x0.shape[0]-1)*0.01),2)
    plots=[]
    for i in range(I):

        p,=ax[i].plot(t,comparisons[i],color=colors[i+1],ls='--',label=labels[i])
        plots+=[p]
        p_base,=ax[i].plot(t,base,color=colors[0],label=r'original node')
        #ax[i].legend(loc='upper right',bbox_to_anchor=(1.45, 1))
        X=np.abs(np.rint(base)-np.rint(comparisons[i]))
        p_bool,=ax[i].plot(t,X,color='green',alpha=alpha,label=r'bit difference')

        ax[i].spines['right'].set_visible(False)
        ax[i].spines['top'].set_visible(False)
        ax[i].spines['bottom'].set_visible(False if i!=I-1 else True)
        ax[i].xaxis.set_visible(False if i!=I-1 else True)

        box = ax[i].get_position()
        ax[i].set_position([box.x0, box.y0, box.width * shrink, box.height])

    plt.xlabel(r'Time ($\tau$)')
    ax[1].set_ylabel(r'Amplitude')

    plots+=[p_base,p_bool]
    from matplotlib.font_manager import FontProperties
    fontP = FontProperties()
    fontP.set_size(set_size)
    lgd=fig.legend(handles=plots,loc=loc, bbox_to_anchor=bbox_to_anchor,
          ncol=ncol, fancybox=True, shadow=True,prop=fontP)

    if fname:
        plt.savefig(fname=fname,bbox_extra_artists=(lgd,), bbox_inches='tight')
    plt.tight_layout()
    plt.show()
