import abc
import datetime
import inspect
import json
import pickle
import warnings
from dataclasses import dataclass, fields
from decimal import Decimal
from enum import Enum
from typing import Generic, Iterable, TypeVar, Union

import boto3
from boto3.dynamodb.conditions import Key


class ConstantString(Enum):
    """
    Holds constant string values. Use it as ConstantString.<STRING_NAME>.value

    To use:
    >>> ConstantString.INVERSE_INDEX_NAME.value
    gsi1
    """
    ITEM_ATTR_NAME = "item_attr_name"
    ITEM_ATTR_TYPE = "item_attr_type"
    ITEM2OBJ_CONV = "item2obj"
    OBJ2ITEM_CONV = "obj2item"
    INVERSE_INDEX_NAME = "inv_index_name"
    INVERSE_INDEX_PK = "inv_index_pk"
    INVERSE_INDEX_SK = "inv_index_sk"
    ENDPOINT_URL = "endpoint_url"
    REGION_NAME = "region_name"
    AWS_ACCESS_KEY_ID = "aws_access_key_id"
    AWS_SECRET_ACCESS_KEY = "aws_secret_access_key"




###### ---------- Global parameters ---------- ######


@dataclass
class DynamoDBGlobalConfiguration:
    """
    A singleton to define global parameters related to DynamoDB
    Do not use the __init(...)__ constructor, use the get_instance(...) method
    """

    _instance = None

    ddb: object
    use_aws_cli_credentials: bool
    access_params: dict
    single_table_name_env_var: str
    single_table_pk_attr_name: str
    single_table_sk_attr_name: str
    single_table_inverse_index_properties: dict

    def __post_init__(self):  # Connect to the database if ddb is None
        if self.ddb is None:
            try:
                if self.use_aws_cli_credentials:
                    self.ddb = boto3.resource("dynamodb")
                else:
                    self.ddb = boto3.resource(
                        "dynamodb",
                        endpoint_url=self.access_params["endpoint_url"],
                        region_name=self.access_params["region_name"],
                        aws_access_key_id=self.access_params["aws_access_key_id"],
                        aws_secret_access_key=self.access_params["aws_secret_access_key"],
                    )
            except Exception as ex:
                print(ex)
                raise ConnectionRefusedError("Not able to connect to DynamoDB")

    @classmethod
    def get_instance(
        cls,
        ddb: object = None,
        use_aws_cli_credentials: bool = True,
        access_params: dict = None,
        single_table_name_env_var: str = "DYNAMODB_TABLE_NAME",
        single_table_pk_attr_name: str = "pk",
        single_table_sk_attr_name: str = "sk",
        single_table_inverse_index_properties: dict = None,
        overwrite_existing_instance: bool = False
    ):
        """
        Get an instance of the singleton.

        Args:
            ddb(str, optional):
                The database connection if None, a connection will be made using the access_params or the credentials
                configured in the aws cli. Defaults to None.
            use_aws_cli_credentials(bool, optional):
                If the aws cli credentials should be used instead of the access_params. Defaults to True.
            access_params(dict, optional):
                A dict with parameters to access the DynamoDB.
                To use this you need to set use_aws_cli_credentials as False.
                The dict should have the following format:
                Defaults to
                    access_params = {
                        ConstantString.ENDPOINT_URL.value: "http://localhost:8000",
                        ConstantString.REGION_NAME.value: "dummy",
                        ConstantString.AWS_ACCESS_KEY_ID.value: "dummy",
                        ConstantString.AWS_SECRET_ACCESS_KEY.value: "dummy"
                    }
            single_table_name_env_var(str, optional):
                The name of the environment variable that holds the name of the single real table on the database.
                Defaults to 'DYNAMODB_TABLE_NAME'.
            single_table_pk_attr_name(str, optional):
                The name of the attribute on the table that holds tha partition key. Defaults to 'pk'.
            single_table_sk_attr_name(str, optional):
                The name of the attribute on the table that holds tha sort key. Defaults to 'sk'.
            single_table_inverse_index_properties(dict, optional):
                The properties of the inverse gsi for querying in the single table design.
                Defaults to
                single_table_inverse_index_properties = {
                    ConstantString.INVERSE_INDEX_NAME.value: 'gsi1',
                    ConstantString.INVERSE_INDEX_PK.value: 'gsi1pk',
                    ConstantString.INVERSE_INDEX_SK.value: 'gsi1sk'
                }
            overwrite_existing_instance(bool, optional):
                If true an instance exists, it will be overwritten. Defaults to False.
        """

        if cls._instance is None or overwrite_existing_instance:
            if single_table_inverse_index_properties is None:
                single_table_inverse_index_properties = {
                    ConstantString.INVERSE_INDEX_NAME.value: "gsi1",
                    ConstantString.INVERSE_INDEX_PK.value: "gsi1pk",
                    ConstantString.INVERSE_INDEX_SK.value: "gsi1sk"
                }
            if access_params is None:
                access_params = {
                    ConstantString.ENDPOINT_URL.value: "http://localhost:8000",
                    ConstantString.REGION_NAME.value: "dummy",
                    ConstantString.AWS_ACCESS_KEY_ID.value: "dummy",
                    ConstantString.AWS_SECRET_ACCESS_KEY.value: "dummy"
                }
            elif not {
                         ConstantString.ENDPOINT_URL.value,
                         ConstantString.REGION_NAME.value,
                         ConstantString.AWS_ACCESS_KEY_ID.value,
                         ConstantString.AWS_SECRET_ACCESS_KEY.value
                     } <= set(access_params):
                use_aws_cli_credentials = True
                warnings.warn("The provided access_params " + str(access_params) + " is missing required values, trying"
                                                                                   " to use the aws cli credentials")
            cls._instance = cls(
                ddb,
                use_aws_cli_credentials,
                access_params,
                single_table_name_env_var,
                single_table_pk_attr_name,
                single_table_sk_attr_name,
                single_table_inverse_index_properties,
            )
        return cls._instance

    @classmethod
    def is_instantiated(cls):
        return cls._instance is not None


###### ---------- Global parameters ---------- ######

###### ---------- Wrapper fo mappable classes ---------- ######


def _dict_if_none(a_dict):
    a_dict = a_dict if a_dict is not None else {}
    return a_dict


def _list_if_none(a_list):
    a_list = a_list if a_list is not None else []
    return a_list


def _wrap_class(  # noqa: C901
    cls=None,  # noqa: C901
    dynamo_table: str = None,  # noqa: C901
    pk: str = None,  # noqa: C901
    sk: str = None,  # noqa: C901
    table_name: str = None, # noqa: C901
    logical_table: str = None,  # noqa: C901
    unique_id: str = None,  # noqa: C901
    pk_name_on_table: str = None,  # noqa: C901
    sk_name_on_table: str = None,  # noqa: C901
    mapping_schema: dict = None,  # noqa: C901
    ignore_attributes: list = None,  # noqa: C901
):  # noqa: C901
    """
    Adds classmethods to the provided class, to be used by only the dynamo_entity decorator
    """

    if hasattr(cls, "dynamo_pk"):  # The class has already been decorated
        return cls
    mapping_schema = _dict_if_none(mapping_schema)
    ignore_attributes = _list_if_none(ignore_attributes)
    # May causes collisions if there are other entities with the same name
    table_name = table_name if table_name is not None else cls.__name__
    logical_table = logical_table if logical_table is not None else table_name
    if pk_name_on_table is None:
        if pk is not None:
            pk_name_on_table = pk
        else:
            pk_name_on_table = DynamoDBGlobalConfiguration.get_instance().single_table_pk_attr_name
    unique_id = unique_id if unique_id is not None else pk_name_on_table
    if sk_name_on_table is None:
        if sk is not None:
            sk_name_on_table = sk
        else:
            sk_name_on_table = DynamoDBGlobalConfiguration.get_instance().single_table_sk_attr_name

    # The methods to be added to the class
    @classmethod
    def dynamo_table_name(cls):
        return dynamo_table

    @classmethod
    def dynamo_logical_table_name(cls):
        return logical_table

    @classmethod
    def dynamo_unassigned_table_name(cls):
        return table_name

    @classmethod
    def dynamo_pk(cls):
        return pk

    @classmethod
    def dynamo_sk(cls):
        return sk

    @classmethod
    def dynamo_id(cls):
        return unique_id

    @classmethod
    def dynamo_pk_name_on_table(cls):
        return pk_name_on_table

    @classmethod
    def dynamo_sk_name_on_table(cls):
        return sk_name_on_table

    @classmethod
    def dynamo_map(cls):
        return mapping_schema

    @classmethod
    def dynamo_ignore(cls):
        return ignore_attributes

    cls.dynamo_unassigned_table_name = dynamo_unassigned_table_name

    cls.dynamo_logical_table_name = dynamo_logical_table_name

    # set the table name
    if dynamo_table is not None:
        cls.dynamo_table_name = dynamo_table_name

    cls.dynamo_pk_name_on_table = dynamo_pk_name_on_table
    # set the id
    cls.dynamo_id = dynamo_id

    # custom pk and sk
    if pk is not None:
        cls.dynamo_pk = dynamo_pk
        # set the sort key
        if sk is not None:
            cls.dynamo_sk = dynamo_sk


    # id and logical table to generate pk and sk
    else:
        # set the logical table name
        # set the name of the key attributes on the dynamo table
        cls.dynamo_sk_name_on_table = dynamo_sk_name_on_table

    # set the mapping of class attributes to table attributes
    cls.dynamo_map = dynamo_map

    # set the class attributes that will be ignored (not saved on the database)
    cls.dynamo_ignore = dynamo_ignore

    # To add non lambda functions, define the function first them assign it to cls.<function_name>

    return cls


def dynamo_entity(
    cls=None,
    dynamo_table: str = None,
    pk: str = None,
    sk: str = None,
    table_name: str = None,
    logical_table: str = None,
    unique_id: str = None,
    pk_name_on_table: str = None,
    sk_name_on_table: str = None,
    mapping_schema: dict = None,
    ignore_attributes: Union[list, tuple] = None,
):
    """
    Wraps a class so it is mappable to DynamoDB. Use it as a decorator.
    The entity class has to allow an empty constructor call.

    DynamoDB uses partition keys (pk) and sort keys (sk) to define a unique data entry,
    If you are not familiar with DynamoDB, this library can generate this keys if you provide just an id attribute
    However pay attention to this two rules:
    - If you provide a pk, the logical table and the id parameters will be ignored.
    - If you do not provide a pk, it is necessary to provide an id, the pk attribute's name on the database table as
    pk_name_on_table, and the sk attribute's name on the database table as sk_name_on_table.

    Args:
        dynamo_table(str, optional):
            The DynamoDB table name, if no name is provided, the class name will be used. Defaults to None
        pk(str, optional):
            The name of the attribute that will be used as the partition key on DynamoDB if the pk will be explicit in
            each object of the decorated class. Using this will ignore the id and logical_table. Defaults to None
        sk(str, optional):
            The name of the attribute that will be used as the sort key if the pk and sk will be explicit in each object
            of the decorated class. Can be also be None if there is no sort key. Defaults to None.
        table_name(str,optional):
            The name of the logical or real table on the database.
        logical_table(str, optional):
            A custom logical table name to store data on the database, to be used only if the pk parameter is not
            provided. If no pk and no logical_table parameters are provided, the logical table name will be the class
            name. That is not collision safe since other mapped classes may have the same name. Defaults to None
        unique_id(str, optional):
            The name of the attribute that will be used as the id, necessary if the pk parameter is not provided.
            The attribute will be cast to str. Defaults to None
        pk_name_on_table(str, optional):
            The name of the pk attribute on the DynamoDB table, necessary if the pk parameter is not provided.
            For compatibility with this library, this attribute should have the string (S) type on the database.
            Defaults to None
        sk_name_on_table(str, optional):
            The name of the sk attribute on the DynamoDB table, necessary if the pk parameter is not provided.
            For compatibility with this library, this attribute should have the string (S) type on the database.
            Defaults to None
        mapping_schema(dict, optional):
            A dict mapping the class attributes to the item attributes on DynamoDB.
            The map should have the following format:
            mapping_schema={
                <class_attribute_name>: {
                    ConstantString.ITEM_ATTR_NAME.value: <string with the attribute's name on the DynamoDB table>,
                    ConstantString.ITEM_ATTR_TYPE.value: <string with the attribute's type on the DynamoDB table>,
                    ConstantString.ITEM2OBJ_CONV.value: <convert function that receives the DynamoDB item's attribute
                                                        and returns the object attribute>,
                    ConstantString.OBJ2ITEM_CONV.value: <convert function that receives the object attribute and
                                                        returns DynamoDB item's attribute>
            }
            If no mapping is provided for a particular (or all) class attribute, the class attribute names and
            standard conversion functions will be used. Defaults to None
        ignore_attributes(list[str], optional):
            A list with the name of the class attributes that should not be saved to/loaded from the database.
            Defaults to None
    Returns:
        class: The decorated class.
    """

    def wrap(cls):
        return _wrap_class(
            cls,
            dynamo_table,
            pk,
            sk,
            table_name,
            logical_table,
            unique_id,
            pk_name_on_table,
            sk_name_on_table,
            mapping_schema,
            ignore_attributes,
        )

    if cls is None:
        return wrap

    return wrap(cls)


###### ---------- Wrapper fo mappable classes ---------- ######


###### ---------- Repository Interfaces and Implementation ---------- ######

T = TypeVar("T")  # A generic type var to hold Entity classes


class Repository(Generic[T], metaclass=abc.ABCMeta):
    """
    Just a simple "interface" inspired by the Java Spring Repository Interface
    """

    entity_type: T
    pass

    def check_provided_type(self):
        """Returns True if obj is a dynamo_entity class or an instance of a dynamo_entity class."""
        cls = T if isinstance(T, type) else type(T)
        return hasattr(cls, "_FIELDS")


class CrudRepository(Repository, metaclass=abc.ABCMeta):
    """
    Just a simple "interface" inspired by the Java Spring CrudRepository Interface
    """

    @abc.abstractmethod
    def count(self):
        """
        Counts the number of items in the table

        Returns:
            An int with the number of items in the table.
        """
        pass

    @abc.abstractmethod
    def remove(self, entity: T):
        """
        Removes an entity object from the database

        Args:
            entity: An object of the mapped entity class to be removed

        Returns:
            True if the object was removed or was not stored in the database
            False otherwise
        """
        pass

    @abc.abstractmethod
    def remove_all(self):
        """
        Removes every entity of the mapped class from the database

        Returns:
            True if there are no entities of the mapped class in the database anymore
            False otherwise
        """
        pass

    @abc.abstractmethod
    def remove_by_keys(self, keys: Union[dict, list]):
        """
        Removes every entity(ies) from the using the provided keys

        Args:
            keys(Union[dict, list]):
                A (pair of) key(s) that identify the entity(ies)

        Returns:
            True if there are no entities of the mapped class in the database anymore;
            False otherwise
        """
        pass

    @abc.abstractmethod
    def remove_all_by_keys(self, keys: Union[Iterable[dict], Iterable[list]]):
        """
        Removes every entity(ies) from the using the provided keys

        Args:
            keys(Union[Iterable[dict], Iterable[list]]):
                a (pair of) key(s) that identify the entity(ies)

        Returns:
            True if there are no entities of the mapped class in the database anymore;
            False otherwise
        """
        pass

    @abc.abstractmethod
    def exists_by_keys(self, keys: Union[dict, list]):
        """
        Checks if an entity identified by the provided keys exist in the database

        Args:
            keys(Union[dict, list]):
                a (pair of) key(s) that identify the entity

        Returns:
            True if a matching entry exist in the database;
            False otherwise
        """
        pass

    @abc.abstractmethod
    def find_all(self):
        """
        Gets all entities of the mapped class from the database

        Returns:
            A list with all the entities of the mapped class
        """
        pass

    @abc.abstractmethod
    def find_by_keys(self, keys: Union[dict, list]):
        """
        Gets all entities of the mapped class that match the provided keys from the database

        Args:
            keys(Union[dict, list]):
                A (pair of) key(s) that identify the entity

        Returns:
            An object of the mapped class if only one entity matches the keys;
            A list of objects of the mapped class if multiple entities match the keys;
            None if no entities match the keys
        """
        pass

    @abc.abstractmethod
    def save(self, entity: T):
        """
        Stores an entity in the database

        Args:
            entity(object): an object of the mapped class to save

        Returns:
            True if the object was stored in the database;
            False otherwise
        """
        pass

    @abc.abstractmethod
    def save_all(self, entities: Iterable[T]):
        """
        Stores a collection of entities in the database

        Args:
            entities(Iterable): objects of the mapped class to save

        Returns:
            True if the objects were stored in the database;
            False otherwise
        """
        pass


def _pk2id(pk):
    return pk.split("#", 1)[1]


class DynamoCrudRepository(CrudRepository):

    ddb = None
    table = None
    table_name: str = None
    map_dict: map = None
    map_filled: bool = False

    def __init__(
        self,
        entity_type: T,
        dynamo_table_name: str = None
    ):
        """
        Creates a new DynamoCrudRepository for a specific dynamo_entity class

        Args:
            entity_type(class):
                The class decorated with dynamo_entity that should be mapped to DynamoDB items.
            dynamo_table_name(str, optional):
                The name of the real table on DynamoDB
        """
        self.entity_type = entity_type

        self.map_dict = self.entity_type.dynamo_map()
        self._fill_map_dict(self.entity_type)

        global_values = DynamoDBGlobalConfiguration.get_instance()
        # For a single table design, define the table name as a global parameter,
        # if the table name is not set in the global parameters, will use the entity class table name
        self.ddb = global_values.ddb
        if dynamo_table_name is not None:
            self.table_name = dynamo_table_name
        elif hasattr(entity_type, "dynamo_table_name") and entity_type.dynamo_table_name() is not None:
            self.table_name = self.entity_type.dynamo_table_name()
        elif (hasattr(entity_type, "dynamo_pk_name_on_table") and hasattr(entity_type, "dynamo_unassigned_table_name")
              and entity_type.dynamo_unassigned_table_name() is not None):
            self.table_name = self.entity_type.dynamo_table_name()
        else:
            import os
            self.table_name = os.environ[DynamoDBGlobalConfiguration.get_instance().single_table_name_env_var]
        try:
            self.table = self.ddb.Table(self.table_name)
        except Exception as ex:
            print(ex)
            warning_str = (
                "Could not access table "
                + str(self.table_name)
                + " check if the table exists"
            )
            raise ResourceWarning(warning_str)

    def count(self):
        n_items = 0

        index_table_name = self.table_name if hasattr(self.entity_type, "dynamo_pk") else self.entity_type.dynamo_logical_table_name()

        try:
            inverse_index = DynamoDBGlobalConfiguration.get_instance().single_table_inverse_index_properties
            response = self.table.query(
                IndexName=inverse_index[ConstantString.INVERSE_INDEX_NAME.value],
                KeyConditionExpression=Key(inverse_index[ConstantString.INVERSE_INDEX_PK.value]).eq(
                    index_table_name
                ),
            )
        except Exception as ex:
            print(ex)
            raise ResourceWarning("Not able to query table", self.table_name)
        n_items = response["Count"]
        while "LastEvaluatedKey" in response:
            try:

                inverse_index = DynamoDBGlobalConfiguration.get_instance().single_table_inverse_index_properties
                response = self.table.query(
                    IndexName=inverse_index[ConstantString.INVERSE_INDEX_NAME.value],
                    KeyConditionExpression=Key(inverse_index[ConstantString.INVERSE_INDEX_PK.value]).eq(
                        index_table_name
                    ),
                    ExclusiveStartKey=response["LastEvaluatedKey"],
                )
            except Exception as ex:
                print(ex)
                warnings.warn("Not able to scan table" + str(self.table_name))
                return n_items
            n_items += response["Count"]
        return n_items

    def key_list2key_map(self, keys: Union[dict, list]):
        if isinstance(keys, list):  # Produce dict from list
            key_list = keys
            keys = {
                self.entity_type.dynamo_pk(): key_list[0]
            }
            if len(keys) > 1:
                keys[self.entity_type.dynamo_sk()] = key_list[1]
        return keys

    def _id2key_pair(self, unique_id):
        sk = self.entity_type.dynamo_logical_table_name()
        pk = self.entity_type.dynamo_logical_table_name() + "#" + str(unique_id)
        keys = {
            self.entity_type.dynamo_pk_name_on_table(): pk,
            self.entity_type.dynamo_sk_name_on_table(): sk,
        }
        return keys

    def remove_by_keys(self, keys: Union[dict, list]):
        """
        Deleted objects stored in the database using the given keys
        :param keys: dict, list a set of keys to search for the object.
         If a list if provided, assumes the pattern [pk, sk]
        :return: an o object of the mapped class or a list of objects of the mapped class
        """
        if isinstance(keys, list):  # Produce dict from list
            keys = self.key_list2key_map(keys)
        try:
            return bool(self.table.delete_item(Key=keys))
        except Exception as ex:
            print(ex)
            warnings.warn(
                "Not able to delete item with keys"
                + str(keys)
                + "from table"
                + str(self.table_name)
            )
            return False

    def remove_by_id(self, unique_id):
        """
        Remove an object of the mapped class from the database using a unique id

        Args:
            unique_id: a unique id that identify the object

        Returns:
            True if there are no entities of the mapped class in the database anymore;
            False otherwise
        """
        return self.remove_by_keys(self._id2key_pair(unique_id))

    def remove(self, to_delete: T):
        if hasattr(self.entity_type, "dynamo_pk"):
            pk_name = self.entity_type.dynamo_pk()
            pk_value = getattr(to_delete, pk_name)
            keys = {pk_name: pk_value}

            if hasattr(self.entity_type, "dynamo_sk"):
                sk_name = self.entity_type.dynamo_sk()
                sk_value = getattr(to_delete, sk_name)
                keys[sk_name] = sk_value

            return self.remove_by_keys(keys)
        else:
            return self.remove_by_id(getattr(to_delete, to_delete.dynamo_id()))



    def item2instance(self, item):
        entity_instance = self.entity_type()

        for fl in fields(self.entity_type):
            item_attr_name = self.map_dict[fl.name][ConstantString.ITEM_ATTR_NAME.value]
            if item_attr_name in item:
                if ConstantString.ITEM2OBJ_CONV.value in self.map_dict[fl.name]:
                    setattr(
                        entity_instance,
                        fl.name,
                        self.map_dict[fl.name][ConstantString.ITEM2OBJ_CONV.value](item[item_attr_name]),
                    )
                # convert to string then to the actual type to make sure the conversion will work
                elif issubclass(fl.type, (int, float, Decimal)):
                    setattr(entity_instance, fl.name, fl.type(str(item[item_attr_name])))

                elif issubclass(fl.type, (bytes, bool)):  # Perform a direct conversion
                    setattr(entity_instance, fl.name, fl.type(item[item_attr_name]))

                elif issubclass(fl.type, (dict, list)):  # json
                    setattr(entity_instance, fl.name, json.loads(item[item_attr_name]))

                elif issubclass(fl.type, Enum):  # Enum
                    setattr(entity_instance, fl.name, fl.type[item[item_attr_name]])

                elif issubclass(fl.type, str):
                    setattr(entity_instance, fl.name, str(item[item_attr_name]))

                # Use the iso format for storing datetime as strings
                elif issubclass(
                    fl.type, (datetime.date, datetime.time, datetime.datetime)
                ):
                    setattr(
                        entity_instance, fl.name, fl.type.fromisoformat(item[item_attr_name])
                    )

                elif issubclass(fl.type, object):  # objects in general are pickled
                    setattr(
                        entity_instance, fl.name, pickle.loads(bytes(item[item_attr_name]))
                    )

                else:  # No special case, use a simple cast, probably will never be reached
                    setattr(entity_instance, fl.name, fl.type(item[item_attr_name]))

        return entity_instance

    def find_by_keys(self, keys: Union[dict, list]):
        """
        Finds objects stored in the database using the given keys
        :param keys: dict, list a set of keys to search for the object.
         If a list if provided, assumes the pattern [pk, sk]
        :return: an object of the mapped class or a list of objects of the mapped class
        """
        if isinstance(keys, list):  # Produce dict from list
            keys = self.key_list2key_map(keys)
        try:
            response = self.table.get_item(Key=keys)

            if "Item" in response:
                item = response[
                    "Item"
                ]  # item is a dict {table_att_name: table_att_value}
                return self.item2instance(item)
            else:
                return None
        except Exception as ex:   # Check if the keys do not compose a unique key
            print(ex)
            key_cond_exp, exp_att_val = self.keys2KeyConditionExpression(keys)
            response = {}
            try:
                response = self.table.query(
                    KeyConditionExpression=key_cond_exp,
                    ExpressionAttributeValues=exp_att_val,
                )
            except Exception as ex:
                print(ex)
                warnings.warn(
                    "Not able to query" + str(self.table) + "with keys" + str(keys)
                )
                return None
            if "Items" in response:
                entity_list = []
                items = response["Items"]
                for item in items:
                    entity_list.append(self.item2instance(item))

                while "LastEvaluatedKey" in response:
                    try:
                        response = self.table.query(
                            KeyConditionExpression=self.keys2KeyConditionExpression(
                                keys
                            ),
                            ExpressionAttributeValues=exp_att_val,
                            ExclusiveStartKey=response["LastEvaluatedKey"],
                        )
                    except Exception:
                        warnings.warn(
                            "Not able to query"
                            + str(self.table)
                            + "with keys"
                            + str(keys)
                        )
                        return entity_list if len(entity_list) > 0 else None
                        return None

                    if "Items" in response:
                        entity_list = []
                        items = response["Items"]
                        for item in items:
                            entity_list.append(self.item2instance(item))

                if len(entity_list) == 0:
                    return None
                # If there is only one object, return the object, otherwise, return the list
                return entity_list if len(entity_list) != 1 else entity_list[0]
            return None

    def find_by_id(self, unique_id):
        return self.find_by_keys(self._id2key_pair(unique_id))

    def keys2KeyConditionExpression(self, keys: dict):
        buffer = ""
        exp_att_val = {}
        sortd = sorted(keys.keys())
        for key in sortd:
            buffer += str(key) + " = :" + str(key) + "val"
            if key != sortd[-1]:
                buffer += " AND "
            exp_att_val[":" + str(key) + "val"] = keys[key]
        return buffer, exp_att_val

    def find_all(self):
        entity_list = []

        try:
            inverse_index = DynamoDBGlobalConfiguration.get_instance().single_table_inverse_index_properties
            response = self.table.query(
                IndexName=inverse_index[ConstantString.INVERSE_INDEX_NAME.value],
                KeyConditionExpression=Key(inverse_index[ConstantString.INVERSE_INDEX_PK.value]).eq(
                    self.entity_type.dynamo_logical_table_name()
                ),
            )
        except Exception as ex:
            print(ex)
            warnings.warn("Not able to scan table" + str(self.table_name))
            return None

        if "Items" in response:
            items = response["Items"]
            for item in items:
                entity_list.append(self.item2instance(item))
        else:
            return entity_list

        while "LastEvaluatedKey" in response:
            try:
                inverse_index = DynamoDBGlobalConfiguration.get_instance().single_table_inverse_index_properties
                response = self.table.query(
                    IndexName=inverse_index[ConstantString.INVERSE_INDEX_NAME.value],
                    KeyConditionExpression=Key(inverse_index[ConstantString.INVERSE_INDEX_PK.value]).eq(
                        self.entity_type.dynamo_logical_table_name()
                    ),
                    ExclusiveStartKey=response["LastEvaluatedKey"],
                )
            except Exception as ex:
                print(ex)
                warnings.warn("Not able to scan table" + str(self.table_name))
                return entity_list
            if "Items" in response:
                items = response["Items"]
                for item in items:
                    entity_list.append(self.item2instance(item))

        return entity_list

    def instance2item_params_inject_keys(self, obj: T, item_params: dict = None):
        item_params = item_params if item_params is not None else {}
        if not hasattr(obj, "dynamo_pk"):  # If using id instead of pk and sk
            sk = obj.dynamo_logical_table_name()
            pk = (
                obj.dynamo_logical_table_name()
                + "#"
                + str(getattr(obj, obj.dynamo_id()))
            )
            item_params[obj.dynamo_pk_name_on_table()] = pk
            item_params[obj.dynamo_sk_name_on_table()] = sk
            item_params[ConstantString.INVERSE_INDEX_PK.value] = sk
            item_params[ConstantString.INVERSE_INDEX_SK.value] = pk

        else:
            item_params[ConstantString.INVERSE_INDEX_PK.value] = self.table_name
        return item_params

    def instance_attr2_item_attr(self, instance_attr_val, instance_attr_name):
        item_attr_name = self.map_dict[instance_attr_name][ConstantString.ITEM_ATTR_NAME.value]
        item_attr_val = None
        if ConstantString.OBJ2ITEM_CONV.value in self.map_dict[instance_attr_name]:
            item_attr_val = self.map_dict[instance_attr_name][ConstantString.OBJ2ITEM_CONV.value](
                instance_attr_val)
        # switch self.map_dict[<attribute_name>]
        elif self.map_dict[instance_attr_name][ConstantString.ITEM_ATTR_TYPE.value] == "N":  # case 'N' (number)
            item_attr_val = Decimal(
                str(instance_attr_val)
            )  # str cast to support numpy, pandas, etc

        elif self.map_dict[instance_attr_name][ConstantString.ITEM_ATTR_TYPE.value] == "B":  # case 'B' (bytes)
            if isinstance(instance_attr_val, bytes):
                item_attr_val = bytes(instance_attr_val)
            elif isinstance(
                    instance_attr_val, object
            ):  # objects in general are pickled
                item_attr_val = pickle.dumps(instance_attr_val)
            else:
                raise TypeError(
                    "Only bytes and objects should be stored as bytes"
                )
        elif self.map_dict[instance_attr_name][ConstantString.ITEM_ATTR_TYPE.value] == "BOOL":  # case 'BOOL' (boolean)
            item_attr_val = 1 if instance_attr_val else 0
        else:  # default (string)
            # Consider special cases and use specific string formats
            # datetime
            if isinstance(
                    instance_attr_val, (datetime.date, datetime.time, datetime.datetime)
            ):
                item_attr_val = instance_attr_val.isoformat()

            # enum
            elif isinstance(instance_attr_val, Enum):
                item_attr_val = instance_attr_val.name

            # maps and lists (converted to json)
            elif isinstance(instance_attr_val, (dict, list)):
                item_attr_val = json.dumps(instance_attr_val)

            # strings
            elif isinstance(instance_attr_val, str):
                item_attr_val = str(instance_attr_val)
            # No special case, use a simple str cast
            else:
                item_attr_val = str(instance_attr_val)
        return item_attr_val, item_attr_name

    def instance2item_params_inject_attributes(self, obj: T, item_params: dict = None):
        item_params = item_params if item_params is not None else {}
        # Get every attribute of obj, ignoring private members and methods
        for instance_attr in inspect.getmembers(obj):
            if (
                    (not instance_attr[0].startswith("_")) # ignore private attributes
                    and (not inspect.ismethod(instance_attr[1])) # ignore methods
                    and (not instance_attr[0] in obj.dynamo_ignore()) # ignore the attributes on the list to ignore
            ):
                instance_attr_name = instance_attr[0]
                instance_attr_val = instance_attr[1]
                item_attr_val, item_attr_name = self.instance_attr2_item_attr(instance_attr_val, instance_attr_name)
                item_params[item_attr_name] = item_attr_val
                # item_attr_name = self.map_dict[instance_attr_name][ConstantString.ITEM_ATTR_NAME.value]
                # if ConstantString.OBJ2ITEM_CONV.value in self.map_dict[instance_attr_name]:
                #     item_params[item_attr_name] = self.map_dict[instance_attr_name][ConstantString.OBJ2ITEM_CONV.value](instance_attr_val)
                # # switch self.map_dict[<attribute_name>]
                # elif self.map_dict[instance_attr_name][ConstantString.ITEM_ATTR_TYPE.value] == "N":  # case 'N' (number)
                #     item_params[item_attr_name] = Decimal(
                #         str(instance_attr_val)
                #     )  # str cast to support numpy, pandas, etc
                #
                # elif self.map_dict[instance_attr_name][ConstantString.ITEM_ATTR_TYPE.value] == "B":  # case 'B' (bytes)
                #     if isinstance(instance_attr_val, bytes):
                #         item_params[item_attr_name] = bytes(instance_attr_val)
                #     elif isinstance(
                #             instance_attr_val, object
                #     ):  # objects in general are pickled
                #         item_params[item_attr_name] = pickle.dumps(instance_attr_val)
                #     else:
                #         raise TypeError(
                #             "Only bytes and objects should be stored as bytes"
                #         )
                # elif self.map_dict[instance_attr_name][ConstantString.ITEM_ATTR_TYPE.value] == "BOOL":  # case 'BOOL' (boolean)
                #     item_params[item_attr_name] = 1 if instance_attr_val else 0
                # else:  # default (string)
                #     # Consider special cases and use specific string formats
                #     # datetime
                #     if isinstance(
                #             instance_attr_val, (datetime.date, datetime.time, datetime.datetime)
                #     ):
                #         item_params[item_attr_name] = instance_attr_val.isoformat()
                #
                #     # enum
                #     elif isinstance(instance_attr_val, Enum):
                #         item_params[item_attr_name] = instance_attr_val.name
                #
                #     # maps and lists (converted to json)
                #     elif isinstance(instance_attr_val, (dict, list)):
                #         item_params[item_attr_name] = json.dumps(instance_attr_val)
                #
                #     # strings
                #     elif isinstance(instance_attr_val, str):
                #         item_params[item_attr_name] = str(instance_attr_val)
                #     # No special case, use a simple str cast
                #     else:
                #         item_params[item_attr_name] = str(instance_attr_val)
        return item_params

    def instance2item_params(self, obj: T):
        item_params = {}
        self.instance2item_params_inject_keys(obj, item_params)
        self.instance2item_params_inject_attributes(obj, item_params)
        return item_params

    def save(self, obj: T):
        item_params = self.instance2item_params(obj)
        try:
            self.table.put_item(Item=item_params)
            return True
        except Exception as ex:
            print(ex)
            warnings.warn(
                "Not able to put item"
                + str(item_params)
                + " in table"
                + str(self.table_name)
            )
            return False

    def remove_entity_list(self, entity_list: Iterable):
        try:
            with self.table.batch_writer() as batch:
                if hasattr(self.entity_type, "dynamo_pk"):
                    pk_name = self.entity_type.dynamo_pk()
                    sk_name = self.entity_type.dynamo_sk()
                    for entity in entity_list:
                        batch.delete_item(
                            Key={
                                pk_name: entity.__getattribute__(pk_name),
                                sk_name: entity.__getattribute__(sk_name),
                            }
                        )
                else:
                    for entity in entity_list:
                        batch.delete_item(
                            Key=self._id2key_pair(
                                getattr(entity, entity.dynamo_id())
                            )
                        )
        except Exception as ex:
            print(ex)
            warnings.warn(
                "Not able to remove items from table " + str(self.table_name)
            )
            return False
        return True

    def remove_all(self):
        entity_list = self.find_all()
        return self.remove_entity_list(entity_list)

    def remove_all_by_keys(self, keys_list: Union[Iterable[dict], Iterable[list]]):
        try:
            with self.table.batch_writer() as batch:
                for keys in keys_list:
                    keys = self.key_list2key_map(keys)
                    batch.delete_item(Key=keys)
        except Exception as ex:
            print(ex)
            warnings.warn(
                "Not able to remove items with keys"
                + str(keys_list)
                + "from table"
                + str(self.table_name)
            )
            return False
        return True

    def remove_all_by_id(self, unique_id_list: list):
        keys_list = []
        for unique_id in unique_id_list:
            keys_list.append(self._id2key_pair(unique_id))
        self.remove_all_by_keys(keys_list)

    def exists_by_keys(self, keys: Union[dict, list]):
        if isinstance(keys, list):  # Produce dict from list
            keys = self.key_list2key_map(keys)
        try:
            response = self.table.get_item(Key=keys)
            if "Item" in response:
                return True
        except Exception as ex:  # Check if the keys do not compose a unique key
            print(ex)
            key_cond_exp, exp_att_val = self.keys2KeyConditionExpression(keys)
            try:
                response = self.table.query(
                    KeyConditionExpression=key_cond_exp,
                    ExpressionAttributeValues=exp_att_val,
                )
            except Exception:
                warnings.warn("Not able to query table" + str(self.table_name))
                return False
            if "Items" in response and len(response["Items"]) > 0:
                return True
        return False

    def exists_by_id(self, unique_id):
        return self.exists_by_keys(self._id2key_pair(unique_id))

    def save_all(self, entities: Iterable[T]):
        try:
            with self.table.batch_writer() as batch:
                for obj in entities:
                    item_params = self.instance2item_params(obj)
                    batch.put_item(Item=item_params)
        except Exception as err:
            print(err)
            raise ResourceWarning(
                "Not able to save item list", entities, "into table", self.table_name
            )

    @classmethod
    def dynamo_type_from_type(cls, python_type: type):
        # if using a specific library like numpy or pandas, the user should specify the "N" type himself
        if issubclass(python_type, (int, float, Decimal)):
            dynamo_type = "N"
        elif issubclass(
                python_type,
                (
                        str,
                        dict,
                        list,
                        datetime.date,
                        datetime.time,
                        datetime.datetime,
                        Enum,
                ),
        ):
            dynamo_type = "S"
        elif issubclass(
                python_type, (bytes, object)
        ):  # general objects will be pickled
            dynamo_type = "B"
        elif issubclass(python_type, bool):
            dynamo_type = "BOOL"
        else:  # this will probably never be reached since general objects are converted to bytes
            dynamo_type = "S"
        return dynamo_type

    def _fill_map_dict(self, cls):
        if not self.map_filled:
            fls = fields(cls)
            for fl in fls:
                attrib_type = str
                if fl.name not in self.map_dict:
                    self.map_dict[fl.name] = {}
                if fl.name not in self.entity_type.dynamo_ignore():
                    if ConstantString.ITEM_ATTR_TYPE.value not in self.map_dict[fl.name]:
                        # Try to infer the type from the class  attribute type
                        attrib_type = self.dynamo_type_from_type(fl.type)
                        self.map_dict[fl.name][ConstantString.ITEM_ATTR_TYPE.value] = attrib_type
                    if ConstantString.ITEM_ATTR_NAME.value not in self.map_dict[fl.name]:
                        self.map_dict[fl.name][ConstantString.ITEM_ATTR_NAME.value] = fl.name
            self.map_filled = True

    ###### ---------- Repository Interfaces and Implementation ---------- ######

class SingleTableDynamoCrudRepository(DynamoCrudRepository):
    def __init__(
            self,
            entity_type: T,
            dynamo_table_name: str = None
    ):
        """
        Creates a new SingleTableDynamoCrudRepository for a specific dynamo_entity class

        Args:
            entity_type(class):
                The class decorated with dynamo_entity that should be mapped to DynamoDB items.
            dynamo_table_name(str, optional):
                The name of the real table on DynamoDB
        """
        if dynamo_table_name is None:
            if hasattr(entity_type, "dynamo_table_name") and entity_type.dynamo_table_name() is not None:
                dynamo_table_name = entity_type.dynamo_table_name()
            else:
                import os
                dynamo_table_name = os.environ[DynamoDBGlobalConfiguration.get_instance().single_table_name_env_var]
        super().__init__(entity_type, dynamo_table_name)

    def instance2item_params_inject_keys(self, obj: T, item_params: dict = None):
        item_params = item_params if item_params is not None else {}
        sk = obj.dynamo_logical_table_name()
        pk = (
                obj.dynamo_logical_table_name()
                + "#"
                + str(getattr(obj, obj.dynamo_id()))
        )
        global_values = DynamoDBGlobalConfiguration.get_instance()
        item_params[obj.dynamo_pk_name_on_table()] = pk
        item_params[obj.dynamo_sk_name_on_table()] = sk
        item_params[global_values.single_table_inverse_index_properties[ConstantString.INVERSE_INDEX_PK.value]] = sk
        item_params[global_values.single_table_inverse_index_properties[ConstantString.INVERSE_INDEX_SK.value]] = pk
        return item_params

    def count(self):
        try:
            inverse_index = DynamoDBGlobalConfiguration.get_instance().single_table_inverse_index_properties
            response = self.table.query(
                IndexName=inverse_index[ConstantString.INVERSE_INDEX_NAME.value],
                KeyConditionExpression=Key(inverse_index[ConstantString.INVERSE_INDEX_PK.value]).eq(
                    self.entity_type.dynamo_logical_table_name()
                ),
            )
        except Exception as ex:
            print(ex)
            raise ResourceWarning("Not able to query table", self.table_name)
        return response["Count"] if "Count" in response else 0

    def remove_by_id(self, unique_id):
        return self.remove_by_keys(self._id2key_pair(unique_id))

    def remove(self, to_delete: T):
        return self.remove_by_id(getattr(to_delete, to_delete.dynamo_id()))

    def find_by_id(self, unique_id):
        return self.find_by_keys(self._id2key_pair(unique_id))

    def find_all(self):
        entity_list = []

        try:
            inverse_index = DynamoDBGlobalConfiguration.get_instance().single_table_inverse_index_properties
            response = self.table.query(
                IndexName=inverse_index[ConstantString.INVERSE_INDEX_NAME.value],
                KeyConditionExpression=Key(inverse_index[ConstantString.INVERSE_INDEX_PK.value]).eq(
                    self.entity_type.dynamo_logical_table_name()
                ),
            )
        except Exception as ex:
            print(ex)
            warnings.warn("Not able to scan table" + str(self.table_name))
            return None

        if "Items" in response:
            items = response["Items"]
            for item in items:
                entity_list.append(self.item2instance(item))

        return entity_list

    def remove_entity_list(self, entity_list: Iterable):
        try:
            with self.table.batch_writer() as batch:
                for entity in entity_list:
                    batch.delete_item(
                        Key=self._id2key_pair(
                            str(getattr(entity, entity.dynamo_id()))
                        )
                    )
        except Exception as ex:
            print(ex)
            warnings.warn(
                "Not able to remove items from table " + str(self.table_name)
            )
            return False
        return True

    def remove_all_by_id(self, unique_id_list: list):
        keys_list = []
        for unique_id in unique_id_list:
            keys_list.append(self._id2key_pair(unique_id))
        return self.remove_all_by_keys(keys_list)

    def exists_by_id(self, unique_id):
        return self.exists_by_keys(self._id2key_pair(unique_id))


class MultiTableDynamoCrudRepository(DynamoCrudRepository):
    def __init__(
            self,
            entity_type: T,
            dynamo_table_name: str = None
    ):
        """
        Creates a new MultiTableDynamoCrudRepository for a specific dynamo_entity class

        Args:
            entity_type(class):
                The class decorated with dynamo_entity that should be mapped to DynamoDB items.
            dynamo_table_name(str, optional):
                The name of the real table on DynamoDB
        """
        if dynamo_table_name is None:
            if hasattr(entity_type, "dynamo_table_name") and entity_type.dynamo_table_name() is not None:
                dynamo_table_name = entity_type.dynamo_table_name()
            elif hasattr(entity_type, "table_name") and entity_type.table_name() is not None:
                dynamo_table_name = entity_type.table_name()
            else:
                import os
                dynamo_table_name = os.environ[DynamoDBGlobalConfiguration.get_instance().single_table_name_env_var]
        if not hasattr(entity_type, "dynamo_pk"):  # Define dynamo_pk using the unique_id
            cls_map = entity_type.dynamo_map()
            if entity_type.dynamo_id() in cls_map and ConstantString.ITEM_ATTR_NAME.value in cls_map[entity_type.dynamo_id()]:
                pk_name = cls_map[entity_type.dynamo_id()][ConstantString.ITEM_ATTR_NAME.value]
            else:
                pk_name = entity_type.dynamo_id()

            class SubEntityType(entity_type):
                @classmethod
                def dynamo_pk(cls):
                    return pk_name

                @classmethod
                def dynamo_pk_name_on_table(cls):
                    return pk_name

            entity_type = SubEntityType

        super().__init__(entity_type, dynamo_table_name)

    def instance2item_params_inject_keys(self, obj: T, item_params: dict = None):
        item_params = item_params if item_params is not None else {}
        global_values = DynamoDBGlobalConfiguration.get_instance()
        gsi_pk = self.entity_type.dynamo_logical_table_name()
        gsi_sk = (
                self.entity_type.dynamo_logical_table_name()
                + "#"
                + str(getattr(obj, obj.dynamo_id(), ""))
        )
        item_params[global_values.single_table_inverse_index_properties[ConstantString.INVERSE_INDEX_PK.value]] = gsi_pk
        item_params[global_values.single_table_inverse_index_properties[ConstantString.INVERSE_INDEX_SK.value]] = gsi_sk

    def instance2item_params_inject_attributes(self, obj: T, item_params: dict = None):
        super().instance2item_params_inject_attributes(obj, item_params)
        if obj.dynamo_pk_name_on_table() not in item_params:
            item_params[obj.dynamo_pk_name_on_table()] = str(getattr(obj, obj.dynamo_id(), ""))

    def count(self):
        try:
            inverse_index = DynamoDBGlobalConfiguration.get_instance().single_table_inverse_index_properties
            response = self.table.query(
                IndexName=inverse_index[ConstantString.INVERSE_INDEX_NAME.value],
                KeyConditionExpression=Key(inverse_index[ConstantString.INVERSE_INDEX_PK.value]).eq(
                    self.entity_type.dynamo_logical_table_name()
                ),
            )
        except Exception as ex:
            print(ex)
            raise ResourceWarning("Not able to query table", self.table_name)
        return response["Count"] if "Count" in response else 0

    def _id2key_pair(self, unique_id):
        unique_id_attr_name = self.entity_type.dynamo_id()
        pk_val, pk_name = self.instance_attr2_item_attr(unique_id, unique_id_attr_name)
        return {pk_name: pk_val}


    def remove_by_id(self, unique_id):
        return self.remove_by_keys(self._id2key_pair(unique_id))

    def remove(self, to_delete: T):
        try:
            pk_name = self.entity_type.dynamo_pk()
            pk_value = getattr(to_delete, pk_name)
            keys = {pk_name: pk_value}

            if hasattr(self.entity_type, "dynamo_sk"):
                sk_name = self.entity_type.dynamo_sk()
                sk_value = getattr(to_delete, sk_name)
                keys[sk_name] = sk_value

            if self.remove_by_keys(keys):
                return True
            else:
                return self.remove_by_id(getattr(to_delete, to_delete.dynamo_id()))
        except Exception:
            return self.remove_by_id(getattr(to_delete, to_delete.dynamo_id()))

    def find_by_id(self, unique_id):
        return self.find_by_keys(self._id2key_pair(unique_id))

    def find_all(self):
        entity_list = []

        try:
            inverse_index = DynamoDBGlobalConfiguration.get_instance().single_table_inverse_index_properties
            response = self.table.query(
                IndexName=inverse_index[ConstantString.INVERSE_INDEX_NAME.value],
                KeyConditionExpression=Key(inverse_index[ConstantString.INVERSE_INDEX_PK.value]).eq(
                    self.entity_type.dynamo_logical_table_name()
                ),
            )
        except Exception as ex:
            print(ex)
            warnings.warn("Not able to scan table" + str(self.table_name))
            return None

        if "Items" in response:
            items = response["Items"]
            for item in items:
                entity_list.append(self.item2instance(item))

        return entity_list

    def remove_entity_list(self, entity_list: Iterable):
        try:
            with self.table.batch_writer() as batch:
                for entity in entity_list:
                    if(hasattr(self.entity_type, "dynamo_sk")):
                        batch.delete_item(
                            Key={
                                self.entity_type.dynamo_pk(): entity.__getattribute__(self.entity_type.dynamo_pk()),
                                self.entity_type.dynamo_sk(): entity.__getattribute__(self.entity_type.dynamo_sk()),
                            }
                        )
                    else:
                        batch.delete_item(
                            Key={
                                self.entity_type.dynamo_pk(): entity.__getattribute__(self.entity_type.dynamo_pk()),
                            }
                        )
        except Exception as ex:
            try:
                with self.table.batch_writer() as batch:
                    for entity in entity_list:
                        keys= self._id2key_pair(getattr(entity, entity.dynamo_id()))

                        batch.delete_item(
                            Key=keys
                        )
            except Exception as ex:
                print(ex)
                warnings.warn(
                    "Not able to remove items from table " + str(self.table_name)
                )
                return False
        return True

    def remove_all_by_id(self, unique_id_list: list):
        keys_list = []
        for unique_id in unique_id_list:
            keys_list.append(self._id2key_pair(unique_id))
        return self.remove_all_by_keys(keys_list)

    def exists_by_id(self, unique_id):
        return self.exists_by_keys(self._id2key_pair(unique_id))
