"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const codepipeline = require("../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const actions_1 = require("./actions");
const construct_internals_1 = require("./private/construct-internals");
const stage_1 = require("./stage");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const CODE_BUILD_LENGTH_LIMIT = 100;
/**
 * (experimental) A Pipeline to deploy CDK apps.
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 *
 * @experimental
 */
class CdkPipeline extends core_2.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        const pipelineStack = core_1.Stack.of(this);
        if (props.codePipeline) {
            if (props.pipelineName) {
                throw new Error('Cannot set \'pipelineName\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (props.crossAccountKeys !== undefined) {
                throw new Error('Cannot set \'crossAccountKeys\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            this._pipeline = props.codePipeline;
        }
        else {
            this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: props.pipelineName,
                crossAccountKeys: props.crossAccountKeys,
                restartExecutionOnUpdate: true,
            });
        }
        if (props.sourceAction && !props.synthAction) {
            // Because of ordering limitations, you can: bring your own Source, bring your own
            // Both, or bring your own Nothing. You cannot bring your own Build (which because of the
            // current CodePipeline API must go BEFORE what we're adding) and then having us add a
            // Source after it. That doesn't make any sense.
            throw new Error('When passing a \'sourceAction\' you must also pass a \'synthAction\' (or a \'codePipeline\' that already has both)');
        }
        if (!props.sourceAction && (!props.codePipeline || props.codePipeline.stages.length < 1)) {
            throw new Error('You must pass a \'sourceAction\' (or a \'codePipeline\' that already has a Source stage)');
        }
        if (props.sourceAction) {
            this._pipeline.addStage({
                stageName: 'Source',
                actions: [props.sourceAction],
            });
        }
        if (props.synthAction) {
            this._pipeline.addStage({
                stageName: 'Build',
                actions: [props.synthAction],
            });
        }
        if ((_b = props.selfMutating) !== null && _b !== void 0 ? _b : true) {
            this._pipeline.addStage({
                stageName: 'UpdatePipeline',
                actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                        cloudAssemblyInput: this._cloudAssemblyArtifact,
                        pipelineStackHierarchicalId: pipelineStack.node.path,
                        cdkCliVersion: props.cdkCliVersion,
                        projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                        privileged: props.supportDockerAssets,
                    })],
            });
        }
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: props.cdkCliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
        });
    }
    /**
     * (experimental) The underlying CodePipeline object.
     *
     * You can use this to add more Stages to the pipeline, or Actions
     * to Stages.
     *
     * @experimental
     */
    get codePipeline() {
        return this._pipeline;
    }
    /**
     * (experimental) Access one of the pipeline's stages by stage name.
     *
     * You can use this to add more Actions to a stage.
     *
     * @experimental
     */
    stage(stageName) {
        return this._pipeline.stage(stageName);
    }
    /**
     * (experimental) Add pipeline stage that will deploy the given application stage.
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     *
     * @experimental
     */
    addApplicationStage(appStage, options = {}) {
        const stage = this.addStage(appStage.stageName);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * (experimental) Add a new, empty stage to the pipeline.
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     *
     * @experimental
     */
    addStage(stageName) {
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * (experimental) Get the StackOutput object that holds this CfnOutput's value in this pipeline.
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     *
     * @experimental
     */
    stackOutput(cfnOutput) {
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            const artifactName = `${stack.artifactId}_Outputs`;
            const compactName = artifactName.slice(artifactName.length - Math.min(artifactName.length, CODE_BUILD_LENGTH_LIMIT));
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(compactName);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * (experimental) Validate that we don't have any stacks violating dependency order in the pipeline.
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     *
     * @experimental
     */
    validate() {
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    core_1.Annotations.of(this).addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
exports.CdkPipeline = CdkPipeline;
_a = JSII_RTTI_SYMBOL_1;
CdkPipeline[_a] = { fqn: "monocdk.pipelines.CdkPipeline", version: "1.106.1" };
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        // CodePipelines has a hard limit of 50 actions per stage. See https://github.com/aws/aws-cdk/issues/9353
        this.MAX_PUBLISHERS_PER_STAGE = 50;
        this.publishers = {};
        this.assetRoles = {};
        this.stages = [];
        this._fileAssetCtr = 0;
        this._dockerAssetCtr = 0;
        this.myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        this.pipeline = this.props.pipeline;
        // Hacks to get access to the innards of Pipeline
        const stages = this.props.pipeline._stages;
        // Any asset publishing stages will be added directly after the last stage that currently exists.
        this.lastStageBeforePublishing = stages.slice(-1)[0];
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        var _b;
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        // The path cannot be outside the asm root. I don't really understand how this could ever
        // come to pass, but apparently it has (see https://github.com/aws/aws-cdk/issues/9766).
        // Add a sanity check here so we can catch it more quickly next time.
        if (relativePath.startsWith(`..${path.sep}`)) {
            throw new Error(`The asset manifest (${command.assetManifestPath}) cannot be outside the Cloud Assembly directory (${this.myCxAsmRoot}). Please report this error at https://github.com/aws/aws-cdk/issues to help us debug why this is happening.`);
        }
        // Late-binding here (rather than in the constructor) to prevent creating the role in cases where no asset actions are created.
        if (!this.assetRoles[command.assetType]) {
            this.generateAssetRole(command.assetType);
        }
        let action = this.publishers[command.assetId];
        if (!action) {
            // Dynamically create new stages as needed, with `MAX_PUBLISHERS_PER_STAGE` assets per stage.
            const stageIndex = Math.floor((this._fileAssetCtr + this._dockerAssetCtr) / this.MAX_PUBLISHERS_PER_STAGE);
            if (stageIndex >= this.stages.length) {
                const previousStage = (_b = this.stages.slice(-1)[0]) !== null && _b !== void 0 ? _b : this.lastStageBeforePublishing;
                this.stages.push(this.pipeline.addStage({
                    stageName: `Assets${stageIndex > 0 ? stageIndex + 1 : ''}`,
                    placement: { justAfter: previousStage },
                }));
            }
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = command.assetType === actions_1.AssetType.FILE ? `FileAsset${++this._fileAssetCtr}` : `DockerAsset${++this._dockerAssetCtr}`;
            // NOTE: It's important that asset changes don't force a pipeline self-mutation.
            // This can cause an infinite loop of updates (see https://github.com/aws/aws-cdk/issues/9080).
            // For that reason, we use the id as the actionName below, rather than the asset hash.
            action = this.publishers[command.assetId] = new actions_1.PublishAssetsAction(this, id, {
                actionName: id,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
                role: this.assetRoles[command.assetType],
                vpc: this.props.vpc,
                subnetSelection: this.props.subnetSelection,
            });
            this.stages[stageIndex].addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    generateAssetRole(assetType) {
        if (this.assetRoles[assetType]) {
            return this.assetRoles[assetType];
        }
        const rolePrefix = assetType === actions_1.AssetType.DOCKER_IMAGE ? 'Docker' : 'File';
        const assetRole = new iam.Role(this, `${rolePrefix}Role`, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(core_1.Stack.of(this).account)),
        });
        // Logging permissions
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: '/aws/codebuild/*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: [logGroupArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        }));
        // CodeBuild report groups
        const codeBuildArn = core_1.Stack.of(this).formatArn({
            service: 'codebuild',
            resource: 'report-group',
            resourceName: '*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'codebuild:CreateReportGroup',
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
                'codebuild:BatchPutCodeCoverages',
            ],
            resources: [codeBuildArn],
        }));
        // CodeBuild start/stop
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // Publishing role access
        const rolePattern = assetType === actions_1.AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        // Artifact access
        this.pipeline.artifactBucket.grantRead(assetRole);
        // VPC permissions required for CodeBuild
        // Normally CodeBuild itself takes care of this but we're creating a singleton role so now
        // we need to do this.
        if (this.props.vpc) {
            assetRole.attachInlinePolicy(new iam.Policy(assetRole, 'VpcPolicy', {
                statements: [
                    new iam.PolicyStatement({
                        resources: [`arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:network-interface/*`],
                        actions: ['ec2:CreateNetworkInterfacePermission'],
                        conditions: {
                            StringEquals: {
                                'ec2:Subnet': this.props.vpc
                                    .selectSubnets(this.props.subnetSelection).subnetIds
                                    .map(si => `arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:subnet/${si}`),
                                'ec2:AuthorizedService': 'codebuild.amazonaws.com',
                            },
                        },
                    }),
                    new iam.PolicyStatement({
                        resources: ['*'],
                        actions: [
                            'ec2:CreateNetworkInterface',
                            'ec2:DescribeNetworkInterfaces',
                            'ec2:DeleteNetworkInterface',
                            'ec2:DescribeSubnets',
                            'ec2:DescribeSecurityGroups',
                            'ec2:DescribeDhcpOptions',
                            'ec2:DescribeVpcs',
                        ],
                    }),
                ],
            }));
        }
        this.assetRoles[assetType] = assetRole.withoutPolicyUpdates();
        return this.assetRoles[assetType];
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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