"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudAssemblyBuilder = exports.CloudAssembly = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const os = require("os");
const path = require("path");
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const cloudformation_artifact_1 = require("./artifacts/cloudformation-artifact");
const nested_cloud_assembly_artifact_1 = require("./artifacts/nested-cloud-assembly-artifact");
const tree_cloud_artifact_1 = require("./artifacts/tree-cloud-artifact");
const cloud_artifact_1 = require("./cloud-artifact");
const toposort_1 = require("./toposort");
/**
 * The name of the root manifest file of the assembly.
 */
const MANIFEST_FILE = 'manifest.json';
/**
 * (experimental) Represents a deployable cloud application.
 *
 * @experimental
 */
class CloudAssembly {
    /**
     * (experimental) Reads a cloud assembly from the specified directory.
     *
     * @param directory The root directory of the assembly.
     * @experimental
     */
    constructor(directory) {
        this.directory = directory;
        this.manifest = cxschema.Manifest.loadAssemblyManifest(path.join(directory, MANIFEST_FILE));
        this.version = this.manifest.version;
        this.artifacts = this.renderArtifacts();
        this.runtime = this.manifest.runtime || { libraries: {} };
        // force validation of deps by accessing 'depends' on all artifacts
        this.validateDeps();
    }
    /**
     * (experimental) Attempts to find an artifact with a specific identity.
     *
     * @param id The artifact ID.
     * @returns A `CloudArtifact` object or `undefined` if the artifact does not exist in this assembly.
     * @experimental
     */
    tryGetArtifact(id) {
        return this.artifacts.find(a => a.id === id);
    }
    /**
     * (experimental) Returns a CloudFormation stack artifact from this assembly.
     *
     * Will only search the current assembly.
     *
     * @param stackName the name of the CloudFormation stack.
     * @returns a `CloudFormationStackArtifact` object.
     * @experimental
     * @throws if there is more than one stack with the same stack name. You can
     * use `getStackArtifact(stack.artifactId)` instead.
     */
    getStackByName(stackName) {
        const artifacts = this.artifacts.filter(a => a instanceof cloudformation_artifact_1.CloudFormationStackArtifact && a.stackName === stackName);
        if (!artifacts || artifacts.length === 0) {
            throw new Error(`Unable to find stack with stack name "${stackName}"`);
        }
        if (artifacts.length > 1) {
            // eslint-disable-next-line max-len
            throw new Error(`There are multiple stacks with the stack name "${stackName}" (${artifacts.map(a => a.id).join(',')}). Use "getStackArtifact(id)" instead`);
        }
        return artifacts[0];
    }
    /**
     * (deprecated) Returns a CloudFormation stack artifact by name from this assembly.
     *
     * @deprecated renamed to `getStackByName` (or `getStackArtifact(id)`)
     */
    getStack(stackName) {
        return this.getStackByName(stackName);
    }
    /**
     * (experimental) Returns a CloudFormation stack artifact from this assembly.
     *
     * @param artifactId the artifact id of the stack (can be obtained through `stack.artifactId`).
     * @returns a `CloudFormationStackArtifact` object.
     * @experimental
     * @throws if there is no stack artifact with that id
     */
    getStackArtifact(artifactId) {
        const artifact = this.tryGetArtifactRecursively(artifactId);
        if (!artifact) {
            throw new Error(`Unable to find artifact with id "${artifactId}"`);
        }
        if (!(artifact instanceof cloudformation_artifact_1.CloudFormationStackArtifact)) {
            throw new Error(`Artifact ${artifactId} is not a CloudFormation stack`);
        }
        return artifact;
    }
    tryGetArtifactRecursively(artifactId) {
        return this.stacksRecursively.find(a => a.id === artifactId);
    }
    /**
     * (experimental) Returns all the stacks, including the ones in nested assemblies.
     *
     * @experimental
     */
    get stacksRecursively() {
        function search(stackArtifacts, assemblies) {
            if (assemblies.length === 0) {
                return stackArtifacts;
            }
            const [head, ...tail] = assemblies;
            const nestedAssemblies = head.nestedAssemblies.map(asm => asm.nestedAssembly);
            return search(stackArtifacts.concat(head.stacks), tail.concat(nestedAssemblies));
        }
        ;
        return search([], [this]);
    }
    /**
     * (experimental) Returns a nested assembly artifact.
     *
     * @param artifactId The artifact ID of the nested assembly.
     * @experimental
     */
    getNestedAssemblyArtifact(artifactId) {
        const artifact = this.tryGetArtifact(artifactId);
        if (!artifact) {
            throw new Error(`Unable to find artifact with id "${artifactId}"`);
        }
        if (!(artifact instanceof nested_cloud_assembly_artifact_1.NestedCloudAssemblyArtifact)) {
            throw new Error(`Found artifact '${artifactId}' but it's not a nested cloud assembly`);
        }
        return artifact;
    }
    /**
     * (experimental) Returns a nested assembly.
     *
     * @param artifactId The artifact ID of the nested assembly.
     * @experimental
     */
    getNestedAssembly(artifactId) {
        return this.getNestedAssemblyArtifact(artifactId).nestedAssembly;
    }
    /**
     * (experimental) Returns the tree metadata artifact from this assembly.
     *
     * @returns a `TreeCloudArtifact` object if there is one defined in the manifest, `undefined` otherwise.
     * @experimental
     * @throws if there is no metadata artifact by that name
     */
    tree() {
        const trees = this.artifacts.filter(a => a.manifest.type === cxschema.ArtifactType.CDK_TREE);
        if (trees.length === 0) {
            return undefined;
        }
        else if (trees.length > 1) {
            throw new Error(`Multiple artifacts of type ${cxschema.ArtifactType.CDK_TREE} found in manifest`);
        }
        const tree = trees[0];
        if (!(tree instanceof tree_cloud_artifact_1.TreeCloudArtifact)) {
            throw new Error('"Tree" artifact is not of expected type');
        }
        return tree;
    }
    /**
     * @returns all the CloudFormation stack artifacts that are included in this assembly.
     * @experimental
     */
    get stacks() {
        return this.artifacts.filter(isCloudFormationStackArtifact);
        function isCloudFormationStackArtifact(x) {
            return x instanceof cloudformation_artifact_1.CloudFormationStackArtifact;
        }
    }
    /**
     * (experimental) The nested assembly artifacts in this assembly.
     *
     * @experimental
     */
    get nestedAssemblies() {
        return this.artifacts.filter(isNestedCloudAssemblyArtifact);
        function isNestedCloudAssemblyArtifact(x) {
            return x instanceof nested_cloud_assembly_artifact_1.NestedCloudAssemblyArtifact;
        }
    }
    validateDeps() {
        for (const artifact of this.artifacts) {
            ignore(artifact.dependencies);
        }
    }
    renderArtifacts() {
        const result = new Array();
        for (const [name, artifact] of Object.entries(this.manifest.artifacts || {})) {
            const cloudartifact = cloud_artifact_1.CloudArtifact.fromManifest(this, name, artifact);
            if (cloudartifact) {
                result.push(cloudartifact);
            }
        }
        return toposort_1.topologicalSort(result, x => x.id, x => x._dependencyIDs);
    }
}
exports.CloudAssembly = CloudAssembly;
_a = JSII_RTTI_SYMBOL_1;
CloudAssembly[_a] = { fqn: "monocdk.cx_api.CloudAssembly", version: "1.106.1" };
/**
 * (experimental) Can be used to build a cloud assembly.
 *
 * @experimental
 */
class CloudAssemblyBuilder {
    /**
     * (experimental) Initializes a cloud assembly builder.
     *
     * @param outdir The output directory, uses temporary directory if undefined.
     * @experimental
     */
    constructor(outdir, props = {}) {
        var _c;
        this.artifacts = {};
        this.missing = new Array();
        this.outdir = determineOutputDirectory(outdir);
        this.assetOutdir = (_c = props.assetOutdir) !== null && _c !== void 0 ? _c : this.outdir;
        this.parentBuilder = props.parentBuilder;
        // we leverage the fact that outdir is long-lived to avoid staging assets into it
        // that were already staged (copying can be expensive). this is achieved by the fact
        // that assets use a source hash as their name. other artifacts, and the manifest itself,
        // will overwrite existing files as needed.
        ensureDirSync(this.outdir);
    }
    /**
     * (experimental) Adds an artifact into the cloud assembly.
     *
     * @param id The ID of the artifact.
     * @param manifest The artifact manifest.
     * @experimental
     */
    addArtifact(id, manifest) {
        this.artifacts[id] = filterUndefined(manifest);
    }
    /**
     * (experimental) Reports that some context is missing in order for this cloud assembly to be fully synthesized.
     *
     * @param missing Missing context information.
     * @experimental
     */
    addMissing(missing) {
        var _c;
        if (this.missing.every(m => m.key !== missing.key)) {
            this.missing.push(missing);
        }
        // Also report in parent
        (_c = this.parentBuilder) === null || _c === void 0 ? void 0 : _c.addMissing(missing);
    }
    /**
     * (experimental) Finalizes the cloud assembly into the output directory returns a `CloudAssembly` object that can be used to inspect the assembly.
     *
     * @experimental
     */
    buildAssembly(options = {}) {
        // explicitly initializing this type will help us detect
        // breaking changes. (For example adding a required property will break compilation).
        let manifest = {
            version: cxschema.Manifest.version(),
            artifacts: this.artifacts,
            runtime: options.runtimeInfo,
            missing: this.missing.length > 0 ? this.missing : undefined,
        };
        // now we can filter
        manifest = filterUndefined(manifest);
        const manifestFilePath = path.join(this.outdir, MANIFEST_FILE);
        cxschema.Manifest.saveAssemblyManifest(manifest, manifestFilePath);
        // "backwards compatibility": in order for the old CLI to tell the user they
        // need a new version, we'll emit the legacy manifest with only "version".
        // this will result in an error "CDK Toolkit >= CLOUD_ASSEMBLY_VERSION is required in order to interact with this program."
        fs.writeFileSync(path.join(this.outdir, 'cdk.out'), JSON.stringify({ version: manifest.version }));
        return new CloudAssembly(this.outdir);
    }
    /**
     * (experimental) Creates a nested cloud assembly.
     *
     * @experimental
     */
    createNestedAssembly(artifactId, displayName) {
        const directoryName = artifactId;
        const innerAsmDir = path.join(this.outdir, directoryName);
        this.addArtifact(artifactId, {
            type: cxschema.ArtifactType.NESTED_CLOUD_ASSEMBLY,
            properties: {
                directoryName,
                displayName,
            },
        });
        return new CloudAssemblyBuilder(innerAsmDir, {
            // Reuse the same asset output directory as the current Casm builder
            assetOutdir: this.assetOutdir,
            parentBuilder: this,
        });
    }
}
exports.CloudAssemblyBuilder = CloudAssemblyBuilder;
_b = JSII_RTTI_SYMBOL_1;
CloudAssemblyBuilder[_b] = { fqn: "monocdk.cx_api.CloudAssemblyBuilder", version: "1.106.1" };
/**
 * Returns a copy of `obj` without undefined values in maps or arrays.
 */
function filterUndefined(obj) {
    if (Array.isArray(obj)) {
        return obj.filter(x => x !== undefined).map(x => filterUndefined(x));
    }
    if (typeof (obj) === 'object') {
        const ret = {};
        for (const [key, value] of Object.entries(obj)) {
            if (value === undefined) {
                continue;
            }
            ret[key] = filterUndefined(value);
        }
        return ret;
    }
    return obj;
}
function ignore(_x) {
    return;
}
/**
 * Turn the given optional output directory into a fixed output directory
 */
function determineOutputDirectory(outdir) {
    return outdir !== null && outdir !== void 0 ? outdir : fs.mkdtempSync(path.join(fs.realpathSync(os.tmpdir()), 'cdk.out'));
}
function ensureDirSync(dir) {
    if (fs.existsSync(dir)) {
        if (!fs.statSync(dir).isDirectory()) {
            throw new Error(`${dir} must be a directory`);
        }
    }
    else {
        fs.mkdirSync(dir, { recursive: true });
    }
}
//# sourceMappingURL=data:application/json;base64,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