"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WaiterStateMachine = void 0;
const aws_iam_1 = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A very simple StateMachine construct highly customized to the provider framework.
 * This is so that this package does not need to depend on aws-stepfunctions module.
 *
 * The state machine continuously calls the isCompleteHandler, until it succeeds or times out.
 * The handler is called `maxAttempts` times with an `interval` duration and a `backoffRate` rate.
 */
class WaiterStateMachine extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const role = new aws_iam_1.Role(this, 'Role', {
            assumedBy: new aws_iam_1.ServicePrincipal('states.amazonaws.com'),
        });
        role.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [props.isCompleteHandler.functionArn],
        }));
        role.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [props.timeoutHandler.functionArn],
        }));
        const definition = core_1.Stack.of(this).toJsonString({
            StartAt: 'framework-isComplete-task',
            States: {
                'framework-isComplete-task': {
                    End: true,
                    Retry: [{
                            ErrorEquals: ['States.ALL'],
                            IntervalSeconds: props.interval.toSeconds(),
                            MaxAttempts: props.maxAttempts,
                            BackoffRate: props.backoffRate,
                        }],
                    Catch: [{
                            ErrorEquals: ['States.ALL'],
                            Next: 'framework-onTimeout-task',
                        }],
                    Type: 'Task',
                    Resource: props.isCompleteHandler.functionArn,
                },
                'framework-onTimeout-task': {
                    End: true,
                    Type: 'Task',
                    Resource: props.timeoutHandler.functionArn,
                },
            },
        });
        const resource = new core_1.CfnResource(this, 'Resource', {
            type: 'AWS::StepFunctions::StateMachine',
            properties: {
                DefinitionString: definition,
                RoleArn: role.roleArn,
            },
        });
        resource.node.addDependency(role);
        this.stateMachineArn = resource.ref;
    }
    grantStartExecution(identity) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee: identity,
            actions: ['states:StartExecution'],
            resourceArns: [this.stateMachineArn],
        });
    }
}
exports.WaiterStateMachine = WaiterStateMachine;
//# sourceMappingURL=data:application/json;base64,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