"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.captureStackTrace = void 0;
const debug_1 = require("./debug");
/**
 * Captures the current process' stack trace.
 *
 * Stack traces are often invaluable tools to help diagnose problems, however
 * their capture is a rather expensive operation, and the stack traces can be
 * large. Consequently, users are stronly advised to condition capturing stack
 * traces to specific user opt-in.
 *
 * Stack traces will only be captured if the `CDK_DEBUG` environment variable
 * is set to `'true'` or `1`.
 *
 * @param below an optional function starting from which stack frames will be
 *              ignored. Defaults to the `captureStackTrace` function itself.
 * @param limit and optional upper bound to the number of stack frames to be
 *              captured. If not provided, this defaults to
 *              `Number.MAX_SAFE_INTEGER`, effectively meaning "no limit".
 *
 * @returns the captured stack trace, as an array of stack frames.
 */
function captureStackTrace(below = captureStackTrace, limit = Number.MAX_SAFE_INTEGER) {
    if (!debug_1.debugModeEnabled()) {
        return ['stack traces disabled'];
    }
    const object = {};
    const previousLimit = Error.stackTraceLimit;
    try {
        Error.stackTraceLimit = limit;
        Error.captureStackTrace(object, below);
    }
    finally {
        Error.stackTraceLimit = previousLimit;
    }
    if (!object.stack) {
        return [];
    }
    return object.stack.split('\n').slice(1).map(s => s.replace(/^\s*at\s+/, ''));
}
exports.captureStackTrace = captureStackTrace;
//# sourceMappingURL=data:application/json;base64,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