"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TreeMetadata = void 0;
const fs = require("fs");
const path = require("path");
const cloud_assembly_schema_1 = require("../../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const annotations_1 = require("../annotations");
const construct_compat_1 = require("../construct-compat");
const stack_1 = require("../stack");
const tree_1 = require("../tree");
const runtime_info_1 = require("./runtime-info");
const FILE_PATH = 'tree.json';
/**
 * Construct that is automatically attached to the top-level `App`.
 * This generates, as part of synthesis, a file containing the construct tree and the metadata for each node in the tree.
 * The output is in a tree format so as to preserve the construct hierarchy.
 *
 */
class TreeMetadata extends construct_compat_1.Construct {
    constructor(scope) {
        super(scope, 'Tree');
    }
    /**
     * Create tree.json
     * @internal
     */
    _synthesizeTree(session) {
        const lookup = {};
        const visit = (construct) => {
            const children = construct.node.children.map((c) => {
                try {
                    return visit(c);
                }
                catch (e) {
                    annotations_1.Annotations.of(this).addWarning(`Failed to render tree metadata for node [${c.node.id}]. Reason: ${e}`);
                    return undefined;
                }
            });
            const childrenMap = children
                .filter((child) => child !== undefined)
                .reduce((map, child) => Object.assign(map, { [child.id]: child }), {});
            const node = {
                id: construct.node.id || 'App',
                path: construct.node.path,
                children: Object.keys(childrenMap).length === 0 ? undefined : childrenMap,
                attributes: this.synthAttributes(construct),
                constructInfo: runtime_info_1.constructInfoFromConstruct(construct),
            };
            lookup[node.path] = node;
            return node;
        };
        const tree = {
            version: 'tree-0.1',
            tree: visit(this.node.root),
        };
        const builder = session.assembly;
        fs.writeFileSync(path.join(builder.outdir, FILE_PATH), JSON.stringify(tree, undefined, 2), { encoding: 'utf-8' });
        builder.addArtifact('Tree', {
            type: cloud_assembly_schema_1.ArtifactType.CDK_TREE,
            properties: {
                file: FILE_PATH,
            },
        });
    }
    synthAttributes(construct) {
        // check if a construct implements IInspectable
        function canInspect(inspectable) {
            return inspectable.inspect !== undefined;
        }
        const inspector = new tree_1.TreeInspector();
        // get attributes from the inspector
        if (canInspect(construct)) {
            construct.inspect(inspector);
            return stack_1.Stack.of(construct).resolve(inspector.attributes);
        }
        return undefined;
    }
}
exports.TreeMetadata = TreeMetadata;
//# sourceMappingURL=data:application/json;base64,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