"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.synthesize = void 0;
const cxapi = require("../../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const constructs = require("constructs");
const annotations_1 = require("../annotations");
const aspect_1 = require("../aspect");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const metadata_resource_1 = require("./metadata-resource");
const prepare_app_1 = require("./prepare-app");
const tree_metadata_1 = require("./tree-metadata");
function synthesize(root, options = {}) {
    // we start by calling "synth" on all nested assemblies (which will take care of all their children)
    synthNestedAssemblies(root, options);
    invokeAspects(root);
    injectMetadataResources(root);
    // This is mostly here for legacy purposes as the framework itself does not use prepare anymore.
    prepareTree(root);
    // resolve references
    prepare_app_1.prepareApp(root);
    // give all children an opportunity to validate now that we've finished prepare
    if (!options.skipValidation) {
        validateTree(root);
    }
    // in unit tests, we support creating free-standing stacks, so we create the
    // assembly builder here.
    const builder = stage_1.Stage.isStage(root)
        ? root._assemblyBuilder
        : new cxapi.CloudAssemblyBuilder(options.outdir);
    // next, we invoke "onSynthesize" on all of our children. this will allow
    // stacks to add themselves to the synthesized cloud assembly.
    synthesizeTree(root, builder, options.validateOnSynthesis);
    return builder.buildAssembly();
}
exports.synthesize = synthesize;
/**
 * Find Assemblies inside the construct and call 'synth' on them
 *
 * (They will in turn recurse again)
 */
function synthNestedAssemblies(root, options) {
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            child.synth(options);
        }
        else {
            synthNestedAssemblies(child, options);
        }
    }
}
/**
 * Invoke aspects on the given construct tree.
 *
 * Aspects are not propagated across Assembly boundaries. The same Aspect will not be invoked
 * twice for the same construct.
 */
function invokeAspects(root) {
    const invokedByPath = {};
    let nestedAspectWarning = false;
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        const node = construct.node;
        const aspects = aspect_1.Aspects.of(construct);
        const allAspectsHere = [...inheritedAspects !== null && inheritedAspects !== void 0 ? inheritedAspects : [], ...aspects.aspects];
        const nodeAspectsCount = aspects.aspects.length;
        for (const aspect of allAspectsHere) {
            let invoked = invokedByPath[node.path];
            if (!invoked) {
                invoked = invokedByPath[node.path] = [];
            }
            if (invoked.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            // if an aspect was added to the node while invoking another aspect it will not be invoked, emit a warning
            // the `nestedAspectWarning` flag is used to prevent the warning from being emitted for every child
            if (!nestedAspectWarning && nodeAspectsCount !== aspects.aspects.length) {
                annotations_1.Annotations.of(construct).addWarning('We detected an Aspect was added via another Aspect, and will not be applied');
                nestedAspectWarning = true;
            }
            // mark as invoked for this node
            invoked.push(aspect);
        }
        for (const child of construct.node.children) {
            if (!stage_1.Stage.isStage(child)) {
                recurse(child, allAspectsHere);
            }
        }
    }
}
/**
 * Prepare all constructs in the given construct tree in post-order.
 *
 * Stop at Assembly boundaries.
 */
function prepareTree(root) {
    visit(root, 'post', construct => construct.onPrepare());
}
/**
 * Find all stacks and add Metadata Resources to all of them
 *
 * There is no good generic place to do this. Can't do it in the constructor
 * (because adding a child construct makes it impossible to set context on the
 * node), and the generic prepare phase is deprecated.
 *
 * Only do this on [parent] stacks (not nested stacks), don't do this when
 * disabled by the user.
 *
 * Also, only when running via the CLI. If we do it unconditionally,
 * all unit tests everywhere are going to break massively. I've spent a day
 * fixing our own, but downstream users would be affected just as badly.
 *
 * Stop at Assembly boundaries.
 */
function injectMetadataResources(root) {
    visit(root, 'post', construct => {
        if (!stack_1.Stack.isStack(construct) || !construct._versionReportingEnabled) {
            return;
        }
        // Because of https://github.com/aws/aws-cdk/blob/master/packages/assert-internal/lib/synth-utils.ts#L74
        // synthesize() may be called more than once on a stack in unit tests, and the below would break
        // if we execute it a second time. Guard against the constructs already existing.
        const CDKMetadata = 'CDKMetadata';
        if (construct.node.tryFindChild(CDKMetadata)) {
            return;
        }
        new metadata_resource_1.MetadataResource(construct, CDKMetadata);
    });
}
/**
 * Synthesize children in post-order into the given builder
 *
 * Stop at Assembly boundaries.
 */
function synthesizeTree(root, builder, validateOnSynth = false) {
    visit(root, 'post', construct => {
        const session = {
            outdir: builder.outdir,
            assembly: builder,
            validateOnSynth,
        };
        if (stack_1.Stack.isStack(construct)) {
            construct.synthesizer.synthesize(session);
        }
        else if (construct instanceof tree_metadata_1.TreeMetadata) {
            construct._synthesizeTree(session);
        }
        // this will soon be deprecated and removed in 2.x
        // see https://github.com/aws/aws-cdk-rfcs/issues/192
        construct.onSynthesize(session);
    });
}
/**
 * Validate all constructs in the given construct tree
 */
function validateTree(root) {
    const errors = new Array();
    // Validations added through `node.addValidation()`
    // This automatically also includes Ye Olde Method of validating, using
    // the `protected validate()` methods.
    errors.push(...constructs.Node.of(root).validate().map(e => ({
        message: e.message,
        source: e.source,
    })));
    if (errors.length > 0) {
        const errorList = errors.map(e => `[${e.source.node.path}] ${e.message}`).join('\n  ');
        throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
    }
}
/**
 * Visit the given construct tree in either pre or post order, stopping at Assemblies
 */
function visit(root, order, cb) {
    if (order === 'pre') {
        cb(root);
    }
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            continue;
        }
        visit(child, order, cb);
    }
    if (order === 'post') {
        cb(root);
    }
}
//# sourceMappingURL=data:application/json;base64,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