"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolvedTypeHint = exports.RememberingTokenResolver = exports.findTokens = exports.resolve = exports.ResolutionTypeHint = exports.INTRINSIC_KEY_PREFIX = void 0;
const resolvable_1 = require("../resolvable");
const encoding_1 = require("./encoding");
const token_map_1 = require("./token-map");
// This file should not be exported to consumers, resolving should happen through Construct.resolve()
const tokenMap = token_map_1.TokenMap.instance();
/**
 * Resolved complex values will have a type hint applied.
 *
 * The type hint will be based on the type of the input value that was resolved.
 *
 * If the value was encoded, the type hint will be the type of the encoded value. In case
 * of a plain `IResolvable`, a type hint of 'string' will be assumed.
 */
const RESOLUTION_TYPEHINT_SYM = Symbol.for('@aws-cdk/core.resolvedTypeHint');
/**
 * Prefix used for intrinsic keys
 *
 * If a key with this prefix is found in an object, the actual value of the
 * key doesn't matter. The value of this key will be an `[ actualKey, actualValue ]`
 * tuple, and the `actualKey` will be a value which otherwise couldn't be represented
 * in the types of `string | number | symbol`, which are the only possible JavaScript
 * object keys.
 */
exports.INTRINSIC_KEY_PREFIX = '$IntrinsicKey$';
/**
 * Type hints for resolved values
 */
var ResolutionTypeHint;
(function (ResolutionTypeHint) {
    ResolutionTypeHint["STRING"] = "string";
    ResolutionTypeHint["NUMBER"] = "number";
    ResolutionTypeHint["LIST"] = "list";
})(ResolutionTypeHint = exports.ResolutionTypeHint || (exports.ResolutionTypeHint = {}));
/**
 * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
 * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
 *
 * @param obj The object to resolve.
 * @param prefix Prefix key path components for diagnostics.
 */
function resolve(obj, options) {
    const prefix = options.prefix || [];
    const pathName = '/' + prefix.join('/');
    /**
     * Make a new resolution context
     */
    function makeContext(appendPath) {
        const newPrefix = appendPath !== undefined ? prefix.concat([appendPath]) : options.prefix;
        let postProcessor;
        const context = {
            preparing: options.preparing,
            scope: options.scope,
            registerPostProcessor(pp) { postProcessor = pp; },
            resolve(x, changeOptions) { return resolve(x, { ...options, ...changeOptions, prefix: newPrefix }); },
        };
        return [context, { postProcess(x) { return postProcessor ? postProcessor.postProcess(x, context) : x; } }];
    }
    // protect against cyclic references by limiting depth.
    if (prefix.length > 200) {
        throw new Error('Unable to resolve object tree with circular reference. Path: ' + pathName);
    }
    //
    // undefined
    //
    if (typeof (obj) === 'undefined') {
        return undefined;
    }
    //
    // null
    //
    if (obj === null) {
        return null;
    }
    //
    // functions - not supported (only tokens are supported)
    //
    if (typeof (obj) === 'function') {
        throw new Error(`Trying to resolve a non-data object. Only token are supported for lazy evaluation. Path: ${pathName}. Object: ${obj}`);
    }
    //
    // string - potentially replace all stringified Tokens
    //
    if (typeof (obj) === 'string') {
        // If this is a "list element" Token, it should never occur by itself in string context
        if (encoding_1.TokenString.forListToken(obj).test()) {
            throw new Error('Found an encoded list token string in a scalar string context. Use \'Fn.select(0, list)\' (not \'list[0]\') to extract elements from token lists.');
        }
        // Otherwise look for a stringified Token in this object
        const str = encoding_1.TokenString.forString(obj);
        if (str.test()) {
            const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
            return tagResolvedValue(options.resolver.resolveString(fragments, makeContext()[0]), ResolutionTypeHint.STRING);
        }
        return obj;
    }
    //
    // number - potentially decode Tokenized number
    //
    if (typeof (obj) === 'number') {
        return tagResolvedValue(resolveNumberToken(obj, makeContext()[0]), ResolutionTypeHint.NUMBER);
    }
    //
    // primitives - as-is
    //
    if (typeof (obj) !== 'object' || obj instanceof Date) {
        return obj;
    }
    //
    // arrays - resolve all values, remove undefined and remove empty arrays
    //
    if (Array.isArray(obj)) {
        if (encoding_1.containsListTokenElement(obj)) {
            return tagResolvedValue(options.resolver.resolveList(obj, makeContext()[0]), ResolutionTypeHint.LIST);
        }
        const arr = obj
            .map((x, i) => makeContext(`${i}`)[0].resolve(x))
            .filter(x => typeof (x) !== 'undefined');
        return arr;
    }
    //
    // tokens - invoke 'resolve' and continue to resolve recursively
    //
    if (encoding_1.unresolved(obj)) {
        const [context, postProcessor] = makeContext();
        const ret = tagResolvedValue(options.resolver.resolveToken(obj, context, postProcessor), ResolutionTypeHint.STRING);
        return ret;
    }
    //
    // objects - deep-resolve all values
    //
    // Must not be a Construct at this point, otherwise you probably made a typo
    // mistake somewhere and resolve will get into an infinite loop recursing into
    // child.parent <---> parent.children
    if (isConstruct(obj)) {
        throw new Error('Trying to resolve() a Construct at ' + pathName);
    }
    const result = {};
    let intrinsicKeyCtr = 0;
    for (const key of Object.keys(obj)) {
        const value = makeContext(String(key))[0].resolve(obj[key]);
        // skip undefined
        if (typeof (value) === 'undefined') {
            continue;
        }
        // Simple case -- not an unresolved key
        if (!encoding_1.unresolved(key)) {
            result[key] = value;
            continue;
        }
        const resolvedKey = makeContext()[0].resolve(key);
        if (typeof (resolvedKey) === 'string') {
            result[resolvedKey] = value;
        }
        else {
            if (!options.allowIntrinsicKeys) {
                // eslint-disable-next-line max-len
                throw new Error(`"${String(key)}" is used as the key in a map so must resolve to a string, but it resolves to: ${JSON.stringify(resolvedKey)}. Consider using "CfnJson" to delay resolution to deployment-time`);
            }
            // Can't represent this object in a JavaScript key position, but we can store it
            // in value position. Use a unique symbol as the key.
            result[`${exports.INTRINSIC_KEY_PREFIX}${intrinsicKeyCtr++}`] = [resolvedKey, value];
        }
    }
    // Because we may be called to recurse on already resolved values (that already have type hints applied)
    // and we just copied those values into a fresh object, be sure to retain any type hints.
    const previousTypeHint = resolvedTypeHint(obj);
    return previousTypeHint ? tagResolvedValue(result, previousTypeHint) : result;
}
exports.resolve = resolve;
/**
 * Find all Tokens that are used in the given structure
 */
function findTokens(scope, fn) {
    const resolver = new RememberingTokenResolver(new resolvable_1.StringConcat());
    resolve(fn(), { scope, prefix: [], resolver, preparing: true });
    return resolver.tokens;
}
exports.findTokens = findTokens;
/**
 * Remember all Tokens encountered while resolving
 */
class RememberingTokenResolver extends resolvable_1.DefaultTokenResolver {
    constructor() {
        super(...arguments);
        this.tokensSeen = new Set();
    }
    resolveToken(t, context, postProcessor) {
        this.tokensSeen.add(t);
        return super.resolveToken(t, context, postProcessor);
    }
    resolveString(s, context) {
        const ret = super.resolveString(s, context);
        return ret;
    }
    get tokens() {
        return Array.from(this.tokensSeen);
    }
}
exports.RememberingTokenResolver = RememberingTokenResolver;
/**
 * Determine whether an object is a Construct
 *
 * Not in 'construct.ts' because that would lead to a dependency cycle via 'uniqueid.ts',
 * and this is a best-effort protection against a common programming mistake anyway.
 */
function isConstruct(x) {
    return x._children !== undefined && x._metadata !== undefined;
}
function resolveNumberToken(x, context) {
    const token = token_map_1.TokenMap.instance().lookupNumberToken(x);
    if (token === undefined) {
        return x;
    }
    return context.resolve(token);
}
/**
 * Apply a type hint to a resolved value
 *
 * The type hint will only be applied to objects.
 *
 * These type hints are used for correct JSON-ification of intrinsic values.
 */
function tagResolvedValue(value, typeHint) {
    if (typeof value !== 'object' || value == null) {
        return value;
    }
    Object.defineProperty(value, RESOLUTION_TYPEHINT_SYM, {
        value: typeHint,
        configurable: true,
    });
    return value;
}
/**
 * Return the type hint from the given value
 *
 * If the value is not a resolved value (i.e, the result of resolving a token),
 * `undefined` will be returned.
 *
 * These type hints are used for correct JSON-ification of intrinsic values.
 */
function resolvedTypeHint(value) {
    if (typeof value !== 'object' || value == null) {
        return undefined;
    }
    return value[RESOLUTION_TYPEHINT_SYM];
}
exports.resolvedTypeHint = resolvedTypeHint;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb2x2ZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInJlc29sdmUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EsOENBQThKO0FBRTlKLHlDQUErRTtBQUMvRSwyQ0FBdUM7QUFJdkMscUdBQXFHO0FBQ3JHLE1BQU0sUUFBUSxHQUFHLG9CQUFRLENBQUMsUUFBUSxFQUFFLENBQUM7QUFDckM7Ozs7Ozs7R0FPRztBQUNILE1BQU0sdUJBQXVCLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0FBQzdFOzs7Ozs7OztHQVFHO0FBQ1UsUUFBQSxvQkFBb0IsR0FBRyxnQkFBZ0IsQ0FBQztBQUNyRDs7R0FFRztBQUNILElBQVksa0JBSVg7QUFKRCxXQUFZLGtCQUFrQjtJQUMxQix1Q0FBaUIsQ0FBQTtJQUNqQix1Q0FBaUIsQ0FBQTtJQUNqQixtQ0FBYSxDQUFBO0FBQ2pCLENBQUMsRUFKVyxrQkFBa0IsR0FBbEIsMEJBQWtCLEtBQWxCLDBCQUFrQixRQUk3QjtBQTRDRDs7Ozs7O0dBTUc7QUFDSCxTQUFnQixPQUFPLENBQUMsR0FBUSxFQUFFLE9BQXdCO0lBQ3RELE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxNQUFNLElBQUksRUFBRSxDQUFDO0lBQ3BDLE1BQU0sUUFBUSxHQUFHLEdBQUcsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3hDOztPQUVHO0lBQ0gsU0FBUyxXQUFXLENBQUMsVUFBbUI7UUFDcEMsTUFBTSxTQUFTLEdBQUcsVUFBVSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFDMUYsSUFBSSxhQUF5QyxDQUFDO1FBQzlDLE1BQU0sT0FBTyxHQUFvQjtZQUM3QixTQUFTLEVBQUUsT0FBTyxDQUFDLFNBQVM7WUFDNUIsS0FBSyxFQUFFLE9BQU8sQ0FBQyxLQUF1QjtZQUN0QyxxQkFBcUIsQ0FBQyxFQUFFLElBQUksYUFBYSxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDakQsT0FBTyxDQUFDLENBQU0sRUFBRSxhQUEyQyxJQUFJLE9BQU8sT0FBTyxDQUFDLENBQUMsRUFBRSxFQUFFLEdBQUcsT0FBTyxFQUFFLEdBQUcsYUFBYSxFQUFFLE1BQU0sRUFBRSxTQUFTLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUMzSSxDQUFDO1FBQ0YsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLFdBQVcsQ0FBQyxDQUFDLElBQUksT0FBTyxhQUFhLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsQ0FBQyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQy9HLENBQUM7SUFDRCx1REFBdUQ7SUFDdkQsSUFBSSxNQUFNLENBQUMsTUFBTSxHQUFHLEdBQUcsRUFBRTtRQUNyQixNQUFNLElBQUksS0FBSyxDQUFDLCtEQUErRCxHQUFHLFFBQVEsQ0FBQyxDQUFDO0tBQy9GO0lBQ0QsRUFBRTtJQUNGLFlBQVk7SUFDWixFQUFFO0lBQ0YsSUFBSSxPQUFPLENBQUMsR0FBRyxDQUFDLEtBQUssV0FBVyxFQUFFO1FBQzlCLE9BQU8sU0FBUyxDQUFDO0tBQ3BCO0lBQ0QsRUFBRTtJQUNGLE9BQU87SUFDUCxFQUFFO0lBQ0YsSUFBSSxHQUFHLEtBQUssSUFBSSxFQUFFO1FBQ2QsT0FBTyxJQUFJLENBQUM7S0FDZjtJQUNELEVBQUU7SUFDRix3REFBd0Q7SUFDeEQsRUFBRTtJQUNGLElBQUksT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLFVBQVUsRUFBRTtRQUM3QixNQUFNLElBQUksS0FBSyxDQUFDLDRGQUE0RixRQUFRLGFBQWEsR0FBRyxFQUFFLENBQUMsQ0FBQztLQUMzSTtJQUNELEVBQUU7SUFDRixzREFBc0Q7SUFDdEQsRUFBRTtJQUNGLElBQUksT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLFFBQVEsRUFBRTtRQUMzQix1RkFBdUY7UUFDdkYsSUFBSSxzQkFBVyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRTtZQUN0QyxNQUFNLElBQUksS0FBSyxDQUFDLG1KQUFtSixDQUFDLENBQUM7U0FDeEs7UUFDRCx3REFBd0Q7UUFDeEQsTUFBTSxHQUFHLEdBQUcsc0JBQVcsQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDdkMsSUFBSSxHQUFHLENBQUMsSUFBSSxFQUFFLEVBQUU7WUFDWixNQUFNLFNBQVMsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7WUFDakUsT0FBTyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQyxTQUFTLEVBQUUsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxrQkFBa0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUNuSDtRQUNELE9BQU8sR0FBRyxDQUFDO0tBQ2Q7SUFDRCxFQUFFO0lBQ0YsK0NBQStDO0lBQy9DLEVBQUU7SUFDRixJQUFJLE9BQU8sQ0FBQyxHQUFHLENBQUMsS0FBSyxRQUFRLEVBQUU7UUFDM0IsT0FBTyxnQkFBZ0IsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLEVBQUUsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxrQkFBa0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztLQUNqRztJQUNELEVBQUU7SUFDRixxQkFBcUI7SUFDckIsRUFBRTtJQUNGLElBQUksT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLFFBQVEsSUFBSSxHQUFHLFlBQVksSUFBSSxFQUFFO1FBQ2xELE9BQU8sR0FBRyxDQUFDO0tBQ2Q7SUFDRCxFQUFFO0lBQ0Ysd0VBQXdFO0lBQ3hFLEVBQUU7SUFDRixJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUU7UUFDcEIsSUFBSSxtQ0FBd0IsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUMvQixPQUFPLGdCQUFnQixDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLEdBQUcsRUFBRSxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLGtCQUFrQixDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ3pHO1FBQ0QsTUFBTSxHQUFHLEdBQUcsR0FBRzthQUNWLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQ2hELE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsS0FBSyxXQUFXLENBQUMsQ0FBQztRQUM3QyxPQUFPLEdBQUcsQ0FBQztLQUNkO0lBQ0QsRUFBRTtJQUNGLGdFQUFnRTtJQUNoRSxFQUFFO0lBQ0YsSUFBSSxxQkFBVSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQ2pCLE1BQU0sQ0FBQyxPQUFPLEVBQUUsYUFBYSxDQUFDLEdBQUcsV0FBVyxFQUFFLENBQUM7UUFDL0MsTUFBTSxHQUFHLEdBQUcsZ0JBQWdCLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsR0FBRyxFQUFFLE9BQU8sRUFBRSxhQUFhLENBQUMsRUFBRSxrQkFBa0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNwSCxPQUFPLEdBQUcsQ0FBQztLQUNkO0lBQ0QsRUFBRTtJQUNGLG9DQUFvQztJQUNwQyxFQUFFO0lBQ0YsNEVBQTRFO0lBQzVFLDhFQUE4RTtJQUM5RSxxQ0FBcUM7SUFDckMsSUFBSSxXQUFXLENBQUMsR0FBRyxDQUFDLEVBQUU7UUFDbEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQ0FBcUMsR0FBRyxRQUFRLENBQUMsQ0FBQztLQUNyRTtJQUNELE1BQU0sTUFBTSxHQUFRLEVBQUUsQ0FBQztJQUN2QixJQUFJLGVBQWUsR0FBRyxDQUFDLENBQUM7SUFDeEIsS0FBSyxNQUFNLEdBQUcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQ2hDLE1BQU0sS0FBSyxHQUFHLFdBQVcsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDNUQsaUJBQWlCO1FBQ2pCLElBQUksT0FBTyxDQUFDLEtBQUssQ0FBQyxLQUFLLFdBQVcsRUFBRTtZQUNoQyxTQUFTO1NBQ1o7UUFDRCx1Q0FBdUM7UUFDdkMsSUFBSSxDQUFDLHFCQUFVLENBQUMsR0FBRyxDQUFDLEVBQUU7WUFDbEIsTUFBTSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztZQUNwQixTQUFTO1NBQ1o7UUFDRCxNQUFNLFdBQVcsR0FBRyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDbEQsSUFBSSxPQUFPLENBQUMsV0FBVyxDQUFDLEtBQUssUUFBUSxFQUFFO1lBQ25DLE1BQU0sQ0FBQyxXQUFXLENBQUMsR0FBRyxLQUFLLENBQUM7U0FDL0I7YUFDSTtZQUNELElBQUksQ0FBQyxPQUFPLENBQUMsa0JBQWtCLEVBQUU7Z0JBQzdCLG1DQUFtQztnQkFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxHQUFHLENBQUMsa0ZBQWtGLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDLG1FQUFtRSxDQUFDLENBQUM7YUFDcE47WUFDRCxnRkFBZ0Y7WUFDaEYscURBQXFEO1lBQ3JELE1BQU0sQ0FBQyxHQUFHLDRCQUFvQixHQUFHLGVBQWUsRUFBRSxFQUFFLENBQUMsR0FBRyxDQUFDLFdBQVcsRUFBRSxLQUFLLENBQUMsQ0FBQztTQUNoRjtLQUNKO0lBQ0Qsd0dBQXdHO0lBQ3hHLHlGQUF5RjtJQUN6RixNQUFNLGdCQUFnQixHQUFHLGdCQUFnQixDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQy9DLE9BQU8sZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDLE1BQU0sRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUM7QUFDbEYsQ0FBQztBQS9IRCwwQkErSEM7QUFDRDs7R0FFRztBQUNILFNBQWdCLFVBQVUsQ0FBQyxLQUFpQixFQUFFLEVBQWE7SUFDdkQsTUFBTSxRQUFRLEdBQUcsSUFBSSx3QkFBd0IsQ0FBQyxJQUFJLHlCQUFZLEVBQUUsQ0FBQyxDQUFDO0lBQ2xFLE9BQU8sQ0FBQyxFQUFFLEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsRUFBRSxFQUFFLFFBQVEsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUNoRSxPQUFPLFFBQVEsQ0FBQyxNQUFNLENBQUM7QUFDM0IsQ0FBQztBQUpELGdDQUlDO0FBQ0Q7O0dBRUc7QUFDSCxNQUFhLHdCQUF5QixTQUFRLGlDQUFvQjtJQUFsRTs7UUFDcUIsZUFBVSxHQUFHLElBQUksR0FBRyxFQUFlLENBQUM7SUFZekQsQ0FBQztJQVhVLFlBQVksQ0FBQyxDQUFjLEVBQUUsT0FBd0IsRUFBRSxhQUE2QjtRQUN2RixJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN2QixPQUFPLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxhQUFhLENBQUMsQ0FBQztJQUN6RCxDQUFDO0lBQ00sYUFBYSxDQUFDLENBQTJCLEVBQUUsT0FBd0I7UUFDdEUsTUFBTSxHQUFHLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDNUMsT0FBTyxHQUFHLENBQUM7SUFDZixDQUFDO0lBQ0QsSUFBVyxNQUFNO1FBQ2IsT0FBTyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUN2QyxDQUFDO0NBQ0o7QUFiRCw0REFhQztBQUNEOzs7OztHQUtHO0FBQ0gsU0FBUyxXQUFXLENBQUMsQ0FBTTtJQUN2QixPQUFPLENBQUMsQ0FBQyxTQUFTLEtBQUssU0FBUyxJQUFJLENBQUMsQ0FBQyxTQUFTLEtBQUssU0FBUyxDQUFDO0FBQ2xFLENBQUM7QUFDRCxTQUFTLGtCQUFrQixDQUFDLENBQVMsRUFBRSxPQUF3QjtJQUMzRCxNQUFNLEtBQUssR0FBRyxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3ZELElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRTtRQUNyQixPQUFPLENBQUMsQ0FBQztLQUNaO0lBQ0QsT0FBTyxPQUFPLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQ2xDLENBQUM7QUFDRDs7Ozs7O0dBTUc7QUFDSCxTQUFTLGdCQUFnQixDQUFDLEtBQVUsRUFBRSxRQUE0QjtJQUM5RCxJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsSUFBSSxLQUFLLElBQUksSUFBSSxFQUFFO1FBQzVDLE9BQU8sS0FBSyxDQUFDO0tBQ2hCO0lBQ0QsTUFBTSxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsdUJBQXVCLEVBQUU7UUFDbEQsS0FBSyxFQUFFLFFBQVE7UUFDZixZQUFZLEVBQUUsSUFBSTtLQUNyQixDQUFDLENBQUM7SUFDSCxPQUFPLEtBQUssQ0FBQztBQUNqQixDQUFDO0FBQ0Q7Ozs7Ozs7R0FPRztBQUNILFNBQWdCLGdCQUFnQixDQUFDLEtBQVU7SUFDdkMsSUFBSSxPQUFPLEtBQUssS0FBSyxRQUFRLElBQUksS0FBSyxJQUFJLElBQUksRUFBRTtRQUM1QyxPQUFPLFNBQVMsQ0FBQztLQUNwQjtJQUNELE9BQU8sS0FBSyxDQUFDLHVCQUF1QixDQUFDLENBQUM7QUFDMUMsQ0FBQztBQUxELDRDQUtDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgSUNvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgRGVmYXVsdFRva2VuUmVzb2x2ZXIsIElQb3N0UHJvY2Vzc29yLCBJUmVzb2x2YWJsZSwgSVJlc29sdmVDb250ZXh0LCBJVG9rZW5SZXNvbHZlciwgUmVzb2x2ZUNoYW5nZUNvbnRleHRPcHRpb25zLCBTdHJpbmdDb25jYXQgfSBmcm9tICcuLi9yZXNvbHZhYmxlJztcbmltcG9ydCB7IFRva2VuaXplZFN0cmluZ0ZyYWdtZW50cyB9IGZyb20gJy4uL3N0cmluZy1mcmFnbWVudHMnO1xuaW1wb3J0IHsgY29udGFpbnNMaXN0VG9rZW5FbGVtZW50LCBUb2tlblN0cmluZywgdW5yZXNvbHZlZCB9IGZyb20gJy4vZW5jb2RpbmcnO1xuaW1wb3J0IHsgVG9rZW5NYXAgfSBmcm9tICcuL3Rva2VuLW1hcCc7XG4vLyB2MiAtIGxlYXZlIHRoaXMgYXMgYSBzZXBhcmF0ZSBzZWN0aW9uIHNvIGl0IHJlZHVjZXMgbWVyZ2UgY29uZmxpY3RzIHdoZW4gY29tcGF0IGlzIHJlbW92ZWRcbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBpbXBvcnQvb3JkZXJcbmltcG9ydCB7IElDb25zdHJ1Y3QgYXMgSUNvcmVDb25zdHJ1Y3QgfSBmcm9tICcuLi9jb25zdHJ1Y3QtY29tcGF0Jztcbi8vIFRoaXMgZmlsZSBzaG91bGQgbm90IGJlIGV4cG9ydGVkIHRvIGNvbnN1bWVycywgcmVzb2x2aW5nIHNob3VsZCBoYXBwZW4gdGhyb3VnaCBDb25zdHJ1Y3QucmVzb2x2ZSgpXG5jb25zdCB0b2tlbk1hcCA9IFRva2VuTWFwLmluc3RhbmNlKCk7XG4vKipcbiAqIFJlc29sdmVkIGNvbXBsZXggdmFsdWVzIHdpbGwgaGF2ZSBhIHR5cGUgaGludCBhcHBsaWVkLlxuICpcbiAqIFRoZSB0eXBlIGhpbnQgd2lsbCBiZSBiYXNlZCBvbiB0aGUgdHlwZSBvZiB0aGUgaW5wdXQgdmFsdWUgdGhhdCB3YXMgcmVzb2x2ZWQuXG4gKlxuICogSWYgdGhlIHZhbHVlIHdhcyBlbmNvZGVkLCB0aGUgdHlwZSBoaW50IHdpbGwgYmUgdGhlIHR5cGUgb2YgdGhlIGVuY29kZWQgdmFsdWUuIEluIGNhc2VcbiAqIG9mIGEgcGxhaW4gYElSZXNvbHZhYmxlYCwgYSB0eXBlIGhpbnQgb2YgJ3N0cmluZycgd2lsbCBiZSBhc3N1bWVkLlxuICovXG5jb25zdCBSRVNPTFVUSU9OX1RZUEVISU5UX1NZTSA9IFN5bWJvbC5mb3IoJ0Bhd3MtY2RrL2NvcmUucmVzb2x2ZWRUeXBlSGludCcpO1xuLyoqXG4gKiBQcmVmaXggdXNlZCBmb3IgaW50cmluc2ljIGtleXNcbiAqXG4gKiBJZiBhIGtleSB3aXRoIHRoaXMgcHJlZml4IGlzIGZvdW5kIGluIGFuIG9iamVjdCwgdGhlIGFjdHVhbCB2YWx1ZSBvZiB0aGVcbiAqIGtleSBkb2Vzbid0IG1hdHRlci4gVGhlIHZhbHVlIG9mIHRoaXMga2V5IHdpbGwgYmUgYW4gYFsgYWN0dWFsS2V5LCBhY3R1YWxWYWx1ZSBdYFxuICogdHVwbGUsIGFuZCB0aGUgYGFjdHVhbEtleWAgd2lsbCBiZSBhIHZhbHVlIHdoaWNoIG90aGVyd2lzZSBjb3VsZG4ndCBiZSByZXByZXNlbnRlZFxuICogaW4gdGhlIHR5cGVzIG9mIGBzdHJpbmcgfCBudW1iZXIgfCBzeW1ib2xgLCB3aGljaCBhcmUgdGhlIG9ubHkgcG9zc2libGUgSmF2YVNjcmlwdFxuICogb2JqZWN0IGtleXMuXG4gKi9cbmV4cG9ydCBjb25zdCBJTlRSSU5TSUNfS0VZX1BSRUZJWCA9ICckSW50cmluc2ljS2V5JCc7XG4vKipcbiAqIFR5cGUgaGludHMgZm9yIHJlc29sdmVkIHZhbHVlc1xuICovXG5leHBvcnQgZW51bSBSZXNvbHV0aW9uVHlwZUhpbnQge1xuICAgIFNUUklORyA9ICdzdHJpbmcnLFxuICAgIE5VTUJFUiA9ICdudW1iZXInLFxuICAgIExJU1QgPSAnbGlzdCdcbn1cbi8qKlxuICogT3B0aW9ucyB0byB0aGUgcmVzb2x2ZSgpIG9wZXJhdGlvblxuICpcbiAqIE5PVCB0aGUgc2FtZSBhcyB0aGUgUmVzb2x2ZUNvbnRleHQ7IFJlc29sdmVDb250ZXh0IGlzIGV4cG9zZWQgdG8gVG9rZW5cbiAqIGltcGxlbWVudG9ycyBhbmQgcmVzb2x1dGlvbiBob29rcywgd2hlcmVhcyB0aGlzIHN0cnVjdCBpcyBqdXN0IHRvIGJ1bmRsZVxuICogYSBudW1iZXIgb2YgdGhpbmdzIHRoYXQgd291bGQgb3RoZXJ3aXNlIGJlIGFyZ3VtZW50cyB0byByZXNvbHZlKCkgaW4gYVxuICogcmVhZGFibGUgd2F5LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElSZXNvbHZlT3B0aW9ucyB7XG4gICAgc2NvcGU6IElDb25zdHJ1Y3Q7XG4gICAgcHJlcGFyaW5nOiBib29sZWFuO1xuICAgIHJlc29sdmVyOiBJVG9rZW5SZXNvbHZlcjtcbiAgICBwcmVmaXg/OiBzdHJpbmdbXTtcbiAgICAvKipcbiAgICAgKiBXaGV0aGVyIG9yIG5vdCB0byBhbGxvdyBpbnRyaW5zaWNzIGluIGtleXMgb2YgYW4gb2JqZWN0XG4gICAgICpcbiAgICAgKiBCZWNhdXNlIGtleXMgb2YgYW4gb2JqZWN0IG11c3QgYmUgc3RyaW5ncywgYSAocmVzb2x2ZWQpIGludHJpbnNpYywgd2hpY2hcbiAgICAgKiBpcyBhbiBvYmplY3QsIGNhbm5vdCBiZSBzdG9yZWQgaW4gdGhhdCBwb3NpdGlvbi4gQnkgZGVmYXVsdCwgd2UgcmVqZWN0IHRoZXNlXG4gICAgICogaW50cmluc2ljcyBpZiB3ZSBlbmNvdW50ZXIgdGhlbS5cbiAgICAgKlxuICAgICAqIElmIHRoaXMgaXMgc2V0IHRvIGB0cnVlYCwgaW4gb3JkZXIgdG8gc3RvcmUgdGhlIGNvbXBsZXggdmFsdWUgaW4gYSBtYXAsXG4gICAgICoga2V5cyB0aGF0IGhhcHBlbiB0byBldmFsdWF0ZSB0byBpbnRyaW5zaWNzIHdpbGwgYmUgYWRkZWQgd2l0aCBhIHVuaXF1ZSBrZXlcbiAgICAgKiBpZGVudGlmaWVkIGJ5IGFuIHVuY29tbWluZyBwcmVmaXgsIG1hcHBlZCB0byBhIHR1cGxlIHRoYXQgcmVwcmVzZW50cyB0aGVcbiAgICAgKiBhY3R1YWwga2V5L3ZhbHVlLXBhaXIuIFRoZSBtYXAgd2lsbCBsb29rIGxpa2UgdGhpczpcbiAgICAgKlxuICAgICAqIHtcbiAgICAgKiAgICAnJEludHJpbnNpY0tleSQwJzogWyB7IFJlZjogLi4uIH0sICd2YWx1ZTEnIF0sXG4gICAgICogICAgJyRJbnRyaW5zaWNLZXkkMSc6IFsgeyBSZWY6IC4uLiB9LCAndmFsdWUyJyBdLFxuICAgICAqICAgICdyZWd1bGFyS2V5JzogJ3ZhbHVlMycsXG4gICAgICogICAgLi4uXG4gICAgICogfVxuICAgICAqXG4gICAgICogQ2FsbGVycyBzaG91bGQgb25seSBzZXQgdGhpcyBvcHRpb24gdG8gYHRydWVgIGlmIHRoZXkgYXJlIHByZXBhcmVkIHRvIGRlYWwgd2l0aFxuICAgICAqIHRoZSBvYmplY3QgaW4gdGhpcyB3ZWlyZCBzaGFwZSwgYW5kIG1hc3NhZ2UgaXQgYmFjayBpbnRvIGEgY29ycmVjdCBvYmplY3QgYWZ0ZXJ3YXJkcy5cbiAgICAgKlxuICAgICAqIChBIHJlZ3VsYXIgYnV0IHVuY29tbW9uIHN0cmluZyB3YXMgY2hvc2VuIG92ZXIgc29tZXRoaW5nIGxpa2Ugc3ltYm9scyBvclxuICAgICAqIG90aGVyIHdheXMgb2YgdGFnZ2luZyB0aGUgZXh0cmEgdmFsdWVzIGluIG9yZGVyIHRvIHNpbXBsaWZ5IHRoZSBpbXBsZW1lbnRhdGlvbiB3aGljaFxuICAgICAqIG1haW50YWlucyB0aGUgZGVzaXJlZCBiZWhhdmlvciBgcmVzb2x2ZShyZXNvbHZlKHgpKSA9PSByZXNvbHZlKHgpYCkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgICAqL1xuICAgIGFsbG93SW50cmluc2ljS2V5cz86IGJvb2xlYW47XG59XG4vKipcbiAqIFJlc29sdmVzIGFuIG9iamVjdCBieSBldmFsdWF0aW5nIGFsbCB0b2tlbnMgYW5kIHJlbW92aW5nIGFueSB1bmRlZmluZWQgb3IgZW1wdHkgb2JqZWN0cyBvciBhcnJheXMuXG4gKiBWYWx1ZXMgY2FuIG9ubHkgYmUgcHJpbWl0aXZlcywgYXJyYXlzIG9yIHRva2Vucy4gT3RoZXIgb2JqZWN0cyAoaS5lLiB3aXRoIG1ldGhvZHMpIHdpbGwgYmUgcmVqZWN0ZWQuXG4gKlxuICogQHBhcmFtIG9iaiBUaGUgb2JqZWN0IHRvIHJlc29sdmUuXG4gKiBAcGFyYW0gcHJlZml4IFByZWZpeCBrZXkgcGF0aCBjb21wb25lbnRzIGZvciBkaWFnbm9zdGljcy5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHJlc29sdmUob2JqOiBhbnksIG9wdGlvbnM6IElSZXNvbHZlT3B0aW9ucyk6IGFueSB7XG4gICAgY29uc3QgcHJlZml4ID0gb3B0aW9ucy5wcmVmaXggfHwgW107XG4gICAgY29uc3QgcGF0aE5hbWUgPSAnLycgKyBwcmVmaXguam9pbignLycpO1xuICAgIC8qKlxuICAgICAqIE1ha2UgYSBuZXcgcmVzb2x1dGlvbiBjb250ZXh0XG4gICAgICovXG4gICAgZnVuY3Rpb24gbWFrZUNvbnRleHQoYXBwZW5kUGF0aD86IHN0cmluZyk6IFtJUmVzb2x2ZUNvbnRleHQsIElQb3N0UHJvY2Vzc29yXSB7XG4gICAgICAgIGNvbnN0IG5ld1ByZWZpeCA9IGFwcGVuZFBhdGggIT09IHVuZGVmaW5lZCA/IHByZWZpeC5jb25jYXQoW2FwcGVuZFBhdGhdKSA6IG9wdGlvbnMucHJlZml4O1xuICAgICAgICBsZXQgcG9zdFByb2Nlc3NvcjogSVBvc3RQcm9jZXNzb3IgfCB1bmRlZmluZWQ7XG4gICAgICAgIGNvbnN0IGNvbnRleHQ6IElSZXNvbHZlQ29udGV4dCA9IHtcbiAgICAgICAgICAgIHByZXBhcmluZzogb3B0aW9ucy5wcmVwYXJpbmcsXG4gICAgICAgICAgICBzY29wZTogb3B0aW9ucy5zY29wZSBhcyBJQ29yZUNvbnN0cnVjdCxcbiAgICAgICAgICAgIHJlZ2lzdGVyUG9zdFByb2Nlc3NvcihwcCkgeyBwb3N0UHJvY2Vzc29yID0gcHA7IH0sXG4gICAgICAgICAgICByZXNvbHZlKHg6IGFueSwgY2hhbmdlT3B0aW9ucz86IFJlc29sdmVDaGFuZ2VDb250ZXh0T3B0aW9ucykgeyByZXR1cm4gcmVzb2x2ZSh4LCB7IC4uLm9wdGlvbnMsIC4uLmNoYW5nZU9wdGlvbnMsIHByZWZpeDogbmV3UHJlZml4IH0pOyB9LFxuICAgICAgICB9O1xuICAgICAgICByZXR1cm4gW2NvbnRleHQsIHsgcG9zdFByb2Nlc3MoeCkgeyByZXR1cm4gcG9zdFByb2Nlc3NvciA/IHBvc3RQcm9jZXNzb3IucG9zdFByb2Nlc3MoeCwgY29udGV4dCkgOiB4OyB9IH1dO1xuICAgIH1cbiAgICAvLyBwcm90ZWN0IGFnYWluc3QgY3ljbGljIHJlZmVyZW5jZXMgYnkgbGltaXRpbmcgZGVwdGguXG4gICAgaWYgKHByZWZpeC5sZW5ndGggPiAyMDApIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdVbmFibGUgdG8gcmVzb2x2ZSBvYmplY3QgdHJlZSB3aXRoIGNpcmN1bGFyIHJlZmVyZW5jZS4gUGF0aDogJyArIHBhdGhOYW1lKTtcbiAgICB9XG4gICAgLy9cbiAgICAvLyB1bmRlZmluZWRcbiAgICAvL1xuICAgIGlmICh0eXBlb2YgKG9iaikgPT09ICd1bmRlZmluZWQnKSB7XG4gICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuICAgIC8vXG4gICAgLy8gbnVsbFxuICAgIC8vXG4gICAgaWYgKG9iaiA9PT0gbnVsbCkge1xuICAgICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG4gICAgLy9cbiAgICAvLyBmdW5jdGlvbnMgLSBub3Qgc3VwcG9ydGVkIChvbmx5IHRva2VucyBhcmUgc3VwcG9ydGVkKVxuICAgIC8vXG4gICAgaWYgKHR5cGVvZiAob2JqKSA9PT0gJ2Z1bmN0aW9uJykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFRyeWluZyB0byByZXNvbHZlIGEgbm9uLWRhdGEgb2JqZWN0LiBPbmx5IHRva2VuIGFyZSBzdXBwb3J0ZWQgZm9yIGxhenkgZXZhbHVhdGlvbi4gUGF0aDogJHtwYXRoTmFtZX0uIE9iamVjdDogJHtvYmp9YCk7XG4gICAgfVxuICAgIC8vXG4gICAgLy8gc3RyaW5nIC0gcG90ZW50aWFsbHkgcmVwbGFjZSBhbGwgc3RyaW5naWZpZWQgVG9rZW5zXG4gICAgLy9cbiAgICBpZiAodHlwZW9mIChvYmopID09PSAnc3RyaW5nJykge1xuICAgICAgICAvLyBJZiB0aGlzIGlzIGEgXCJsaXN0IGVsZW1lbnRcIiBUb2tlbiwgaXQgc2hvdWxkIG5ldmVyIG9jY3VyIGJ5IGl0c2VsZiBpbiBzdHJpbmcgY29udGV4dFxuICAgICAgICBpZiAoVG9rZW5TdHJpbmcuZm9yTGlzdFRva2VuKG9iaikudGVzdCgpKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ZvdW5kIGFuIGVuY29kZWQgbGlzdCB0b2tlbiBzdHJpbmcgaW4gYSBzY2FsYXIgc3RyaW5nIGNvbnRleHQuIFVzZSBcXCdGbi5zZWxlY3QoMCwgbGlzdClcXCcgKG5vdCBcXCdsaXN0WzBdXFwnKSB0byBleHRyYWN0IGVsZW1lbnRzIGZyb20gdG9rZW4gbGlzdHMuJyk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gT3RoZXJ3aXNlIGxvb2sgZm9yIGEgc3RyaW5naWZpZWQgVG9rZW4gaW4gdGhpcyBvYmplY3RcbiAgICAgICAgY29uc3Qgc3RyID0gVG9rZW5TdHJpbmcuZm9yU3RyaW5nKG9iaik7XG4gICAgICAgIGlmIChzdHIudGVzdCgpKSB7XG4gICAgICAgICAgICBjb25zdCBmcmFnbWVudHMgPSBzdHIuc3BsaXQodG9rZW5NYXAubG9va3VwVG9rZW4uYmluZCh0b2tlbk1hcCkpO1xuICAgICAgICAgICAgcmV0dXJuIHRhZ1Jlc29sdmVkVmFsdWUob3B0aW9ucy5yZXNvbHZlci5yZXNvbHZlU3RyaW5nKGZyYWdtZW50cywgbWFrZUNvbnRleHQoKVswXSksIFJlc29sdXRpb25UeXBlSGludC5TVFJJTkcpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBvYmo7XG4gICAgfVxuICAgIC8vXG4gICAgLy8gbnVtYmVyIC0gcG90ZW50aWFsbHkgZGVjb2RlIFRva2VuaXplZCBudW1iZXJcbiAgICAvL1xuICAgIGlmICh0eXBlb2YgKG9iaikgPT09ICdudW1iZXInKSB7XG4gICAgICAgIHJldHVybiB0YWdSZXNvbHZlZFZhbHVlKHJlc29sdmVOdW1iZXJUb2tlbihvYmosIG1ha2VDb250ZXh0KClbMF0pLCBSZXNvbHV0aW9uVHlwZUhpbnQuTlVNQkVSKTtcbiAgICB9XG4gICAgLy9cbiAgICAvLyBwcmltaXRpdmVzIC0gYXMtaXNcbiAgICAvL1xuICAgIGlmICh0eXBlb2YgKG9iaikgIT09ICdvYmplY3QnIHx8IG9iaiBpbnN0YW5jZW9mIERhdGUpIHtcbiAgICAgICAgcmV0dXJuIG9iajtcbiAgICB9XG4gICAgLy9cbiAgICAvLyBhcnJheXMgLSByZXNvbHZlIGFsbCB2YWx1ZXMsIHJlbW92ZSB1bmRlZmluZWQgYW5kIHJlbW92ZSBlbXB0eSBhcnJheXNcbiAgICAvL1xuICAgIGlmIChBcnJheS5pc0FycmF5KG9iaikpIHtcbiAgICAgICAgaWYgKGNvbnRhaW5zTGlzdFRva2VuRWxlbWVudChvYmopKSB7XG4gICAgICAgICAgICByZXR1cm4gdGFnUmVzb2x2ZWRWYWx1ZShvcHRpb25zLnJlc29sdmVyLnJlc29sdmVMaXN0KG9iaiwgbWFrZUNvbnRleHQoKVswXSksIFJlc29sdXRpb25UeXBlSGludC5MSVNUKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBhcnIgPSBvYmpcbiAgICAgICAgICAgIC5tYXAoKHgsIGkpID0+IG1ha2VDb250ZXh0KGAke2l9YClbMF0ucmVzb2x2ZSh4KSlcbiAgICAgICAgICAgIC5maWx0ZXIoeCA9PiB0eXBlb2YgKHgpICE9PSAndW5kZWZpbmVkJyk7XG4gICAgICAgIHJldHVybiBhcnI7XG4gICAgfVxuICAgIC8vXG4gICAgLy8gdG9rZW5zIC0gaW52b2tlICdyZXNvbHZlJyBhbmQgY29udGludWUgdG8gcmVzb2x2ZSByZWN1cnNpdmVseVxuICAgIC8vXG4gICAgaWYgKHVucmVzb2x2ZWQob2JqKSkge1xuICAgICAgICBjb25zdCBbY29udGV4dCwgcG9zdFByb2Nlc3Nvcl0gPSBtYWtlQ29udGV4dCgpO1xuICAgICAgICBjb25zdCByZXQgPSB0YWdSZXNvbHZlZFZhbHVlKG9wdGlvbnMucmVzb2x2ZXIucmVzb2x2ZVRva2VuKG9iaiwgY29udGV4dCwgcG9zdFByb2Nlc3NvciksIFJlc29sdXRpb25UeXBlSGludC5TVFJJTkcpO1xuICAgICAgICByZXR1cm4gcmV0O1xuICAgIH1cbiAgICAvL1xuICAgIC8vIG9iamVjdHMgLSBkZWVwLXJlc29sdmUgYWxsIHZhbHVlc1xuICAgIC8vXG4gICAgLy8gTXVzdCBub3QgYmUgYSBDb25zdHJ1Y3QgYXQgdGhpcyBwb2ludCwgb3RoZXJ3aXNlIHlvdSBwcm9iYWJseSBtYWRlIGEgdHlwb1xuICAgIC8vIG1pc3Rha2Ugc29tZXdoZXJlIGFuZCByZXNvbHZlIHdpbGwgZ2V0IGludG8gYW4gaW5maW5pdGUgbG9vcCByZWN1cnNpbmcgaW50b1xuICAgIC8vIGNoaWxkLnBhcmVudCA8LS0tPiBwYXJlbnQuY2hpbGRyZW5cbiAgICBpZiAoaXNDb25zdHJ1Y3Qob2JqKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RyeWluZyB0byByZXNvbHZlKCkgYSBDb25zdHJ1Y3QgYXQgJyArIHBhdGhOYW1lKTtcbiAgICB9XG4gICAgY29uc3QgcmVzdWx0OiBhbnkgPSB7fTtcbiAgICBsZXQgaW50cmluc2ljS2V5Q3RyID0gMDtcbiAgICBmb3IgKGNvbnN0IGtleSBvZiBPYmplY3Qua2V5cyhvYmopKSB7XG4gICAgICAgIGNvbnN0IHZhbHVlID0gbWFrZUNvbnRleHQoU3RyaW5nKGtleSkpWzBdLnJlc29sdmUob2JqW2tleV0pO1xuICAgICAgICAvLyBza2lwIHVuZGVmaW5lZFxuICAgICAgICBpZiAodHlwZW9mICh2YWx1ZSkgPT09ICd1bmRlZmluZWQnKSB7XG4gICAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgfVxuICAgICAgICAvLyBTaW1wbGUgY2FzZSAtLSBub3QgYW4gdW5yZXNvbHZlZCBrZXlcbiAgICAgICAgaWYgKCF1bnJlc29sdmVkKGtleSkpIHtcbiAgICAgICAgICAgIHJlc3VsdFtrZXldID0gdmFsdWU7XG4gICAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCByZXNvbHZlZEtleSA9IG1ha2VDb250ZXh0KClbMF0ucmVzb2x2ZShrZXkpO1xuICAgICAgICBpZiAodHlwZW9mIChyZXNvbHZlZEtleSkgPT09ICdzdHJpbmcnKSB7XG4gICAgICAgICAgICByZXN1bHRbcmVzb2x2ZWRLZXldID0gdmFsdWU7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICBpZiAoIW9wdGlvbnMuYWxsb3dJbnRyaW5zaWNLZXlzKSB7XG4gICAgICAgICAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG1heC1sZW5cbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFwiJHtTdHJpbmcoa2V5KX1cIiBpcyB1c2VkIGFzIHRoZSBrZXkgaW4gYSBtYXAgc28gbXVzdCByZXNvbHZlIHRvIGEgc3RyaW5nLCBidXQgaXQgcmVzb2x2ZXMgdG86ICR7SlNPTi5zdHJpbmdpZnkocmVzb2x2ZWRLZXkpfS4gQ29uc2lkZXIgdXNpbmcgXCJDZm5Kc29uXCIgdG8gZGVsYXkgcmVzb2x1dGlvbiB0byBkZXBsb3ltZW50LXRpbWVgKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIC8vIENhbid0IHJlcHJlc2VudCB0aGlzIG9iamVjdCBpbiBhIEphdmFTY3JpcHQga2V5IHBvc2l0aW9uLCBidXQgd2UgY2FuIHN0b3JlIGl0XG4gICAgICAgICAgICAvLyBpbiB2YWx1ZSBwb3NpdGlvbi4gVXNlIGEgdW5pcXVlIHN5bWJvbCBhcyB0aGUga2V5LlxuICAgICAgICAgICAgcmVzdWx0W2Ake0lOVFJJTlNJQ19LRVlfUFJFRklYfSR7aW50cmluc2ljS2V5Q3RyKyt9YF0gPSBbcmVzb2x2ZWRLZXksIHZhbHVlXTtcbiAgICAgICAgfVxuICAgIH1cbiAgICAvLyBCZWNhdXNlIHdlIG1heSBiZSBjYWxsZWQgdG8gcmVjdXJzZSBvbiBhbHJlYWR5IHJlc29sdmVkIHZhbHVlcyAodGhhdCBhbHJlYWR5IGhhdmUgdHlwZSBoaW50cyBhcHBsaWVkKVxuICAgIC8vIGFuZCB3ZSBqdXN0IGNvcGllZCB0aG9zZSB2YWx1ZXMgaW50byBhIGZyZXNoIG9iamVjdCwgYmUgc3VyZSB0byByZXRhaW4gYW55IHR5cGUgaGludHMuXG4gICAgY29uc3QgcHJldmlvdXNUeXBlSGludCA9IHJlc29sdmVkVHlwZUhpbnQob2JqKTtcbiAgICByZXR1cm4gcHJldmlvdXNUeXBlSGludCA/IHRhZ1Jlc29sdmVkVmFsdWUocmVzdWx0LCBwcmV2aW91c1R5cGVIaW50KSA6IHJlc3VsdDtcbn1cbi8qKlxuICogRmluZCBhbGwgVG9rZW5zIHRoYXQgYXJlIHVzZWQgaW4gdGhlIGdpdmVuIHN0cnVjdHVyZVxuICovXG5leHBvcnQgZnVuY3Rpb24gZmluZFRva2VucyhzY29wZTogSUNvbnN0cnVjdCwgZm46ICgpID0+IGFueSk6IElSZXNvbHZhYmxlW10ge1xuICAgIGNvbnN0IHJlc29sdmVyID0gbmV3IFJlbWVtYmVyaW5nVG9rZW5SZXNvbHZlcihuZXcgU3RyaW5nQ29uY2F0KCkpO1xuICAgIHJlc29sdmUoZm4oKSwgeyBzY29wZSwgcHJlZml4OiBbXSwgcmVzb2x2ZXIsIHByZXBhcmluZzogdHJ1ZSB9KTtcbiAgICByZXR1cm4gcmVzb2x2ZXIudG9rZW5zO1xufVxuLyoqXG4gKiBSZW1lbWJlciBhbGwgVG9rZW5zIGVuY291bnRlcmVkIHdoaWxlIHJlc29sdmluZ1xuICovXG5leHBvcnQgY2xhc3MgUmVtZW1iZXJpbmdUb2tlblJlc29sdmVyIGV4dGVuZHMgRGVmYXVsdFRva2VuUmVzb2x2ZXIge1xuICAgIHByaXZhdGUgcmVhZG9ubHkgdG9rZW5zU2VlbiA9IG5ldyBTZXQ8SVJlc29sdmFibGU+KCk7XG4gICAgcHVibGljIHJlc29sdmVUb2tlbih0OiBJUmVzb2x2YWJsZSwgY29udGV4dDogSVJlc29sdmVDb250ZXh0LCBwb3N0UHJvY2Vzc29yOiBJUG9zdFByb2Nlc3Nvcikge1xuICAgICAgICB0aGlzLnRva2Vuc1NlZW4uYWRkKHQpO1xuICAgICAgICByZXR1cm4gc3VwZXIucmVzb2x2ZVRva2VuKHQsIGNvbnRleHQsIHBvc3RQcm9jZXNzb3IpO1xuICAgIH1cbiAgICBwdWJsaWMgcmVzb2x2ZVN0cmluZyhzOiBUb2tlbml6ZWRTdHJpbmdGcmFnbWVudHMsIGNvbnRleHQ6IElSZXNvbHZlQ29udGV4dCkge1xuICAgICAgICBjb25zdCByZXQgPSBzdXBlci5yZXNvbHZlU3RyaW5nKHMsIGNvbnRleHQpO1xuICAgICAgICByZXR1cm4gcmV0O1xuICAgIH1cbiAgICBwdWJsaWMgZ2V0IHRva2VucygpOiBJUmVzb2x2YWJsZVtdIHtcbiAgICAgICAgcmV0dXJuIEFycmF5LmZyb20odGhpcy50b2tlbnNTZWVuKTtcbiAgICB9XG59XG4vKipcbiAqIERldGVybWluZSB3aGV0aGVyIGFuIG9iamVjdCBpcyBhIENvbnN0cnVjdFxuICpcbiAqIE5vdCBpbiAnY29uc3RydWN0LnRzJyBiZWNhdXNlIHRoYXQgd291bGQgbGVhZCB0byBhIGRlcGVuZGVuY3kgY3ljbGUgdmlhICd1bmlxdWVpZC50cycsXG4gKiBhbmQgdGhpcyBpcyBhIGJlc3QtZWZmb3J0IHByb3RlY3Rpb24gYWdhaW5zdCBhIGNvbW1vbiBwcm9ncmFtbWluZyBtaXN0YWtlIGFueXdheS5cbiAqL1xuZnVuY3Rpb24gaXNDb25zdHJ1Y3QoeDogYW55KTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHguX2NoaWxkcmVuICE9PSB1bmRlZmluZWQgJiYgeC5fbWV0YWRhdGEgIT09IHVuZGVmaW5lZDtcbn1cbmZ1bmN0aW9uIHJlc29sdmVOdW1iZXJUb2tlbih4OiBudW1iZXIsIGNvbnRleHQ6IElSZXNvbHZlQ29udGV4dCk6IGFueSB7XG4gICAgY29uc3QgdG9rZW4gPSBUb2tlbk1hcC5pbnN0YW5jZSgpLmxvb2t1cE51bWJlclRva2VuKHgpO1xuICAgIGlmICh0b2tlbiA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIHJldHVybiB4O1xuICAgIH1cbiAgICByZXR1cm4gY29udGV4dC5yZXNvbHZlKHRva2VuKTtcbn1cbi8qKlxuICogQXBwbHkgYSB0eXBlIGhpbnQgdG8gYSByZXNvbHZlZCB2YWx1ZVxuICpcbiAqIFRoZSB0eXBlIGhpbnQgd2lsbCBvbmx5IGJlIGFwcGxpZWQgdG8gb2JqZWN0cy5cbiAqXG4gKiBUaGVzZSB0eXBlIGhpbnRzIGFyZSB1c2VkIGZvciBjb3JyZWN0IEpTT04taWZpY2F0aW9uIG9mIGludHJpbnNpYyB2YWx1ZXMuXG4gKi9cbmZ1bmN0aW9uIHRhZ1Jlc29sdmVkVmFsdWUodmFsdWU6IGFueSwgdHlwZUhpbnQ6IFJlc29sdXRpb25UeXBlSGludCk6IGFueSB7XG4gICAgaWYgKHR5cGVvZiB2YWx1ZSAhPT0gJ29iamVjdCcgfHwgdmFsdWUgPT0gbnVsbCkge1xuICAgICAgICByZXR1cm4gdmFsdWU7XG4gICAgfVxuICAgIE9iamVjdC5kZWZpbmVQcm9wZXJ0eSh2YWx1ZSwgUkVTT0xVVElPTl9UWVBFSElOVF9TWU0sIHtcbiAgICAgICAgdmFsdWU6IHR5cGVIaW50LFxuICAgICAgICBjb25maWd1cmFibGU6IHRydWUsXG4gICAgfSk7XG4gICAgcmV0dXJuIHZhbHVlO1xufVxuLyoqXG4gKiBSZXR1cm4gdGhlIHR5cGUgaGludCBmcm9tIHRoZSBnaXZlbiB2YWx1ZVxuICpcbiAqIElmIHRoZSB2YWx1ZSBpcyBub3QgYSByZXNvbHZlZCB2YWx1ZSAoaS5lLCB0aGUgcmVzdWx0IG9mIHJlc29sdmluZyBhIHRva2VuKSxcbiAqIGB1bmRlZmluZWRgIHdpbGwgYmUgcmV0dXJuZWQuXG4gKlxuICogVGhlc2UgdHlwZSBoaW50cyBhcmUgdXNlZCBmb3IgY29ycmVjdCBKU09OLWlmaWNhdGlvbiBvZiBpbnRyaW5zaWMgdmFsdWVzLlxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVzb2x2ZWRUeXBlSGludCh2YWx1ZTogYW55KTogUmVzb2x1dGlvblR5cGVIaW50IHwgdW5kZWZpbmVkIHtcbiAgICBpZiAodHlwZW9mIHZhbHVlICE9PSAnb2JqZWN0JyB8fCB2YWx1ZSA9PSBudWxsKSB7XG4gICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuICAgIHJldHVybiB2YWx1ZVtSRVNPTFVUSU9OX1RZUEVISU5UX1NZTV07XG59XG4iXX0=