"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.referenceNestedStackValueInParent = exports.resolveReferences = void 0;
// ----------------------------------------------------
// CROSS REFERENCES
// ----------------------------------------------------
const cfn_element_1 = require("../cfn-element");
const cfn_output_1 = require("../cfn-output");
const cfn_parameter_1 = require("../cfn-parameter");
const names_1 = require("../names");
const stack_1 = require("../stack");
const token_1 = require("../token");
const cfn_reference_1 = require("./cfn-reference");
const resolve_1 = require("./resolve");
/**
 * This is called from the App level to resolve all references defined. Each
 * reference is resolved based on it's consumption context.
 */
function resolveReferences(scope) {
    const edges = findAllReferences(scope);
    for (const { source, value } of edges) {
        const consumer = stack_1.Stack.of(source);
        // resolve the value in the context of the consumer
        if (!value.hasValueForStack(consumer)) {
            const resolved = resolveValue(consumer, value);
            value.assignValueForStack(consumer, resolved);
        }
    }
}
exports.resolveReferences = resolveReferences;
/**
 * Resolves the value for `reference` in the context of `consumer`.
 */
function resolveValue(consumer, reference) {
    const producer = stack_1.Stack.of(reference.target);
    // produce and consumer stacks are the same, we can just return the value itself.
    if (producer === consumer) {
        return reference;
    }
    // unsupported: stacks from different apps
    if (producer.node.root !== consumer.node.root) {
        throw new Error('Cannot reference across apps. Consuming and producing stacks must be defined within the same CDK app.');
    }
    // unsupported: stacks are not in the same environment
    if (producer.environment !== consumer.environment) {
        throw new Error(`Stack "${consumer.node.path}" cannot consume a cross reference from stack "${producer.node.path}". ` +
            'Cross stack references are only supported for stacks deployed to the same environment or between nested stacks and their parent stack');
    }
    // ----------------------------------------------------------------------
    // consumer is nested in the producer (directly or indirectly)
    // ----------------------------------------------------------------------
    // if the consumer is nested within the producer (directly or indirectly),
    // wire through a CloudFormation parameter and then resolve the reference with
    // the parent stack as the consumer.
    if (consumer.nestedStackParent && isNested(consumer, producer)) {
        const parameterValue = resolveValue(consumer.nestedStackParent, reference);
        return createNestedStackParameter(consumer, reference, parameterValue);
    }
    // ----------------------------------------------------------------------
    // producer is a nested stack
    // ----------------------------------------------------------------------
    // if the producer is nested, always publish the value through a
    // cloudformation output and resolve recursively with the Fn::GetAtt
    // of the output in the parent stack.
    // one might ask, if the consumer is not a parent of the producer,
    // why not just use export/import? the reason is that we cannot
    // generate an "export name" from a nested stack because the export
    // name must contain the stack name to ensure uniqueness, and we
    // don't know the stack name of a nested stack before we deploy it.
    // therefore, we can only export from a top-level stack.
    if (producer.nested) {
        const outputValue = createNestedStackOutput(producer, reference);
        return resolveValue(consumer, outputValue);
    }
    // ----------------------------------------------------------------------
    // export/import
    // ----------------------------------------------------------------------
    // export the value through a cloudformation "export name" and use an
    // Fn::ImportValue in the consumption site.
    // add a dependency between the producer and the consumer. dependency logic
    // will take care of applying the dependency at the right level (e.g. the
    // top-level stacks).
    consumer.addDependency(producer, `${consumer.node.path} -> ${reference.target.node.path}.${reference.displayName}`);
    return createImportValue(reference);
}
/**
 * Finds all the CloudFormation references in a construct tree.
 */
function findAllReferences(root) {
    const result = new Array();
    for (const consumer of root.node.findAll()) {
        // include only CfnElements (i.e. resources)
        if (!cfn_element_1.CfnElement.isCfnElement(consumer)) {
            continue;
        }
        try {
            const tokens = resolve_1.findTokens(consumer, () => consumer._toCloudFormation());
            // iterate over all the tokens (e.g. intrinsic functions, lazies, etc) that
            // were found in the cloudformation representation of this resource.
            for (const token of tokens) {
                // include only CfnReferences (i.e. "Ref" and "Fn::GetAtt")
                if (!cfn_reference_1.CfnReference.isCfnReference(token)) {
                    continue;
                }
                result.push({
                    source: consumer,
                    value: token,
                });
            }
        }
        catch (e) {
            // Note: it might be that the properties of the CFN object aren't valid.
            // This will usually be preventatively caught in a construct's validate()
            // and turned into a nicely descriptive error, but we're running prepare()
            // before validate(). Swallow errors that occur because the CFN layer
            // doesn't validate completely.
            //
            // This does make the assumption that the error will not be rectified,
            // but the error will be thrown later on anyway. If the error doesn't
            // get thrown down the line, we may miss references.
            if (e.type === 'CfnSynthesisError') {
                continue;
            }
            throw e;
        }
    }
    return result;
}
// ------------------------------------------------------------------------------------------------
// export/import
// ------------------------------------------------------------------------------------------------
/**
 * Imports a value from another stack by creating an "Output" with an "ExportName"
 * and returning an "Fn::ImportValue" token.
 */
function createImportValue(reference) {
    const exportingStack = stack_1.Stack.of(reference.target);
    const importExpr = exportingStack.exportValue(reference);
    // I happen to know this returns a Fn.importValue() which implements Intrinsic.
    return token_1.Tokenization.reverseCompleteString(importExpr);
}
// ------------------------------------------------------------------------------------------------
// nested stacks
// ------------------------------------------------------------------------------------------------
/**
 * Adds a CloudFormation parameter to a nested stack and assigns it with the
 * value of the reference.
 */
function createNestedStackParameter(nested, reference, value) {
    // we call "this.resolve" to ensure that tokens do not creep in (for example, if the reference display name includes tokens)
    const paramId = nested.resolve(`reference-to-${names_1.Names.nodeUniqueId(reference.target.node)}.${reference.displayName}`);
    let param = nested.node.tryFindChild(paramId);
    if (!param) {
        param = new cfn_parameter_1.CfnParameter(nested, paramId, { type: 'String' });
        // Ugly little hack until we move NestedStack to this module.
        if (!('setParameter' in nested)) {
            throw new Error('assertion failed: nested stack should have a "setParameter" method');
        }
        nested.setParameter(param.logicalId, token_1.Token.asString(value));
    }
    return param.value;
}
/**
 * Adds a CloudFormation output to a nested stack and returns an "Fn::GetAtt"
 * intrinsic that can be used to reference this output in the parent stack.
 */
function createNestedStackOutput(producer, reference) {
    const outputId = `${names_1.Names.nodeUniqueId(reference.target.node)}${reference.displayName}`;
    let output = producer.node.tryFindChild(outputId);
    if (!output) {
        output = new cfn_output_1.CfnOutput(producer, outputId, { value: token_1.Token.asString(reference) });
    }
    if (!producer.nestedStackResource) {
        throw new Error('assertion failed');
    }
    return producer.nestedStackResource.getAtt(`Outputs.${output.logicalId}`);
}
/**
 * Translate a Reference into a nested stack into a value in the parent stack
 *
 * Will create Outputs along the chain of Nested Stacks, and return the final `{ Fn::GetAtt }`.
 */
function referenceNestedStackValueInParent(reference, targetStack) {
    let currentStack = stack_1.Stack.of(reference.target);
    if (currentStack !== targetStack && !isNested(currentStack, targetStack)) {
        throw new Error(`Referenced resource must be in stack '${targetStack.node.path}', got '${reference.target.node.path}'`);
    }
    while (currentStack !== targetStack) {
        reference = createNestedStackOutput(stack_1.Stack.of(reference.target), reference);
        currentStack = stack_1.Stack.of(reference.target);
    }
    return reference;
}
exports.referenceNestedStackValueInParent = referenceNestedStackValueInParent;
/**
 * @returns true if this stack is a direct or indirect parent of the nested
 * stack `nested`.
 *
 * If `child` is not a nested stack, always returns `false` because it can't
 * have a parent, dah.
 */
function isNested(nested, parent) {
    // if the parent is a direct parent
    if (nested.nestedStackParent === parent) {
        return true;
    }
    // we reached a top-level (non-nested) stack without finding the parent
    if (!nested.nestedStackParent) {
        return false;
    }
    // recurse with the child's direct parent
    return isNested(nested.nestedStackParent, parent);
}
//# sourceMappingURL=data:application/json;base64,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