"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApp = void 0;
const constructs_1 = require("constructs");
const cfn_resource_1 = require("../cfn-resource");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const refs_1 = require("./refs");
/**
 * Prepares the app for synthesis. This function is called by the root `prepare`
 * (normally this the App, but if a Stack is a root, it is called by the stack),
 * which means it's the last 'prepare' that executes.
 *
 * It takes care of reifying cross-references between stacks (or nested stacks),
 * and of creating assets for nested stack templates.
 *
 * @param root The root of the construct tree.
 */
function prepareApp(root) {
    // apply dependencies between resources in depending subtrees
    for (const dependency of root.node.dependencies) {
        const targetCfnResources = findCfnResources(dependency.target);
        const sourceCfnResources = findCfnResources(dependency.source);
        for (const target of targetCfnResources) {
            for (const source of sourceCfnResources) {
                source.addDependsOn(target);
            }
        }
    }
    refs_1.resolveReferences(root);
    // depth-first (children first) queue of nested stacks. We will pop a stack
    // from the head of this queue to prepare its template asset.
    //
    // Depth-first since the a nested stack's template hash will be reflected in
    // its parent's template, which then changes the parent's hash, etc.
    const queue = findAllNestedStacks(root);
    if (queue.length > 0) {
        while (queue.length > 0) {
            const nested = queue.shift();
            defineNestedStackAsset(nested);
        }
        // ▷[ Given the legacy synthesizer and a 3-or-deeper nesting of nested stacks ]
        //
        // Adding nested stack assets may haved added CfnParameters to the top-level
        // stack which are referenced in a deeper-level stack. The values of these
        // parameters need to be carried through to the right location via Nested
        // Stack parameters, which `resolveReferences()` will do.
        //
        // Yes, this may add `Parameter` elements to a template whose hash has
        // already been calculated, but the invariant that if the functional part
        // of the template changes its hash will change is still upheld.
        refs_1.resolveReferences(root);
    }
}
exports.prepareApp = prepareApp;
/**
 * Prepares the assets for nested stacks in this app.
 * @returns `true` if assets were added to the parent of a nested stack, which
 * implies that another round of reference resolution is in order. If this
 * function returns `false`, we know we are done.
 */
function defineNestedStackAsset(nestedStack) {
    // this is needed temporarily until we move NestedStack to '@aws-cdk/core'.
    const nested = nestedStack;
    nested._prepareTemplateAsset();
}
function findAllNestedStacks(root) {
    const result = new Array();
    const includeStack = (stack) => {
        if (!stack_1.Stack.isStack(stack)) {
            return false;
        }
        if (!stack.nested) {
            return false;
        }
        // test: if we are not within a stage, then include it.
        if (!stage_1.Stage.of(stack)) {
            return true;
        }
        return stage_1.Stage.of(stack) === root;
    };
    // create a list of all nested stacks in depth-first post order this means
    // that we first prepare the leaves and then work our way up.
    for (const stack of root.node.findAll(constructs_1.ConstructOrder.POSTORDER /* <== important */)) {
        if (includeStack(stack)) {
            result.push(stack);
        }
    }
    return result;
}
/**
 * Find all resources in a set of constructs
 */
function findCfnResources(root) {
    return root.node.findAll().filter(cfn_resource_1.CfnResource.isCfnResource);
}
//# sourceMappingURL=data:application/json;base64,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