"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isGeneratedWhenNeededMarker = exports.GeneratedWhenNeededMarker = exports.generatePhysicalName = void 0;
const crypto = require("crypto");
const constructs_1 = require("constructs");
const names_1 = require("../names");
const stack_1 = require("../stack");
const token_1 = require("../token");
const token_map_1 = require("./token-map");
function generatePhysicalName(resource) {
    const stack = stack_1.Stack.of(resource);
    const stackPart = new PrefixNamePart(stack.stackName, 25);
    const idPart = new SuffixNamePart(names_1.Names.nodeUniqueId(resource.node), 24);
    const region = stack.region;
    if (token_1.Token.isUnresolved(region) || !region) {
        throw new Error(`Cannot generate a physical name for ${constructs_1.Node.of(resource).path}, because the region is un-resolved or missing`);
    }
    const account = stack.account;
    if (token_1.Token.isUnresolved(account) || !account) {
        throw new Error(`Cannot generate a physical name for ${constructs_1.Node.of(resource).path}, because the account is un-resolved or missing`);
    }
    const parts = [stackPart, idPart]
        .map(part => part.generate());
    const hashLength = 12;
    const sha256 = crypto.createHash('sha256')
        .update(stackPart.bareStr)
        .update(idPart.bareStr)
        .update(region)
        .update(account);
    const hash = sha256.digest('hex').slice(0, hashLength);
    const ret = [...parts, hash].join('');
    return ret.toLowerCase();
}
exports.generatePhysicalName = generatePhysicalName;
class NamePart {
    constructor(bareStr) {
        this.bareStr = bareStr;
    }
}
class PrefixNamePart extends NamePart {
    constructor(bareStr, prefixLength) {
        super(bareStr);
        this.prefixLength = prefixLength;
    }
    generate() {
        return this.bareStr.slice(0, this.prefixLength);
    }
}
class SuffixNamePart extends NamePart {
    constructor(str, suffixLength) {
        super(str);
        this.suffixLength = suffixLength;
    }
    generate() {
        const strLen = this.bareStr.length;
        const startIndex = Math.max(strLen - this.suffixLength, 0);
        return this.bareStr.slice(startIndex, strLen);
    }
}
const GENERATE_IF_NEEDED_SYMBOL = Symbol.for('@aws-cdk/core.<private>.GenerateIfNeeded');
/**
 * This marker token is used by PhysicalName.GENERATE_IF_NEEDED. When that token is passed to the
 * physicalName property of a Resource, it triggers different behavior in the Resource constructor
 * that will allow emission of a generated physical name (when the resource is used across
 * environments) or undefined (when the resource is not shared).
 *
 * This token throws an Error when it is resolved, as a way to prevent inadvertent mis-uses of it.
 */
class GeneratedWhenNeededMarker {
    constructor() {
        this.creationStack = [];
        Object.defineProperty(this, GENERATE_IF_NEEDED_SYMBOL, { value: true });
    }
    resolve(_ctx) {
        throw new Error('Invalid physical name passed to CloudFormation. Use "this.physicalName" instead');
    }
    toString() {
        return 'PhysicalName.GENERATE_IF_NEEDED';
    }
}
exports.GeneratedWhenNeededMarker = GeneratedWhenNeededMarker;
/**
 * Checks whether a stringified token resolves to a `GeneratedWhenNeededMarker`.
 */
function isGeneratedWhenNeededMarker(val) {
    const token = token_map_1.TokenMap.instance().lookupString(val);
    return !!token && GENERATE_IF_NEEDED_SYMBOL in token;
}
exports.isGeneratedWhenNeededMarker = isGeneratedWhenNeededMarker;
//# sourceMappingURL=data:application/json;base64,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