"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatAnalytics = exports.MetadataResource = void 0;
const zlib = require("zlib");
const region_info_1 = require("../../../region-info"); // Automatically re-written from '@aws-cdk/region-info'
const cfn_condition_1 = require("../cfn-condition");
const cfn_fn_1 = require("../cfn-fn");
const cfn_pseudo_1 = require("../cfn-pseudo");
const cfn_resource_1 = require("../cfn-resource");
const construct_compat_1 = require("../construct-compat");
const lazy_1 = require("../lazy");
const token_1 = require("../token");
const runtime_info_1 = require("./runtime-info");
/**
 * Construct that will render the metadata resource
 */
class MetadataResource extends construct_compat_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        const metadataServiceExists = token_1.Token.isUnresolved(scope.region) || region_info_1.RegionInfo.get(scope.region).cdkMetadataResourceAvailable;
        if (metadataServiceExists) {
            const resource = new cfn_resource_1.CfnResource(this, 'Default', {
                type: 'AWS::CDK::Metadata',
                properties: {
                    Analytics: lazy_1.Lazy.string({ produce: () => formatAnalytics(runtime_info_1.constructInfoFromStack(scope)) }),
                },
            });
            // In case we don't actually know the region, add a condition to determine it at deploy time
            if (token_1.Token.isUnresolved(scope.region)) {
                const condition = new cfn_condition_1.CfnCondition(this, 'Condition', {
                    expression: makeCdkMetadataAvailableCondition(),
                });
                // To not cause undue template changes
                condition.overrideLogicalId('CDKMetadataAvailable');
                resource.cfnOptions.condition = condition;
            }
        }
    }
}
exports.MetadataResource = MetadataResource;
function makeCdkMetadataAvailableCondition() {
    return cfn_fn_1.Fn.conditionOr(...region_info_1.RegionInfo.regions
        .filter(ri => ri.cdkMetadataResourceAvailable)
        .map(ri => cfn_fn_1.Fn.conditionEquals(cfn_pseudo_1.Aws.REGION, ri.name)));
}
/** Convenience type for arbitrarily-nested map */
class Trie extends Map {
}
/**
 * Formats a list of construct fully-qualified names (FQNs) and versions into a (possibly compressed) prefix-encoded string.
 *
 * The list of ConstructInfos is logically formatted into:
 * ${version}!${fqn} (e.g., "1.90.0!aws-cdk-lib.Stack")
 * and then all of the construct-versions are grouped with common prefixes together, grouping common parts in '{}' and separating items with ','.
 *
 * Example:
 * [1.90.0!aws-cdk-lib.Stack, 1.90.0!aws-cdk-lib.Construct, 1.90.0!aws-cdk-lib.service.Resource, 0.42.1!aws-cdk-lib-experiments.NewStuff]
 * Becomes:
 * 1.90.0!aws-cdk-lib.{Stack,Construct,service.Resource},0.42.1!aws-cdk-lib-experiments.NewStuff
 *
 * The whole thing is then either included directly as plaintext as:
 * v2:plaintext:{prefixEncodedList}
 * Or is compressed and base64-encoded, and then formatted as:
 * v2:deflate64:{prefixEncodedListCompressedAndEncoded}
 *
 * Exported/visible for ease of testing.
 */
function formatAnalytics(infos) {
    const trie = new Trie();
    infos.forEach(info => insertFqnInTrie(`${info.version}!${info.fqn}`, trie));
    const plaintextEncodedConstructs = prefixEncodeTrie(trie);
    const compressedConstructs = zlib.gzipSync(Buffer.from(plaintextEncodedConstructs)).toString('base64');
    return `v2:deflate64:${compressedConstructs}`;
}
exports.formatAnalytics = formatAnalytics;
/**
 * Splits after non-alphanumeric characters (e.g., '.', '/') in the FQN
 * and insert each piece of the FQN in nested map (i.e., simple trie).
 */
function insertFqnInTrie(fqn, trie) {
    var _a;
    for (const fqnPart of fqn.replace(/[^a-z0-9]/gi, '$& ').split(' ')) {
        const nextLevelTreeRef = (_a = trie.get(fqnPart)) !== null && _a !== void 0 ? _a : new Trie();
        trie.set(fqnPart, nextLevelTreeRef);
        trie = nextLevelTreeRef;
    }
    return trie;
}
/**
 * Prefix-encodes a "trie-ish" structure, using '{}' to group and ',' to separate siblings.
 *
 * Example input:
 * ABC,ABD,AEF
 *
 * Example trie:
 * A --> B --> C
 *  |     \--> D
 *  \--> E --> F
 *
 * Becomes:
 * A{B{C,D},EF}
 */
function prefixEncodeTrie(trie) {
    let prefixEncoded = '';
    let isFirstEntryAtLevel = true;
    [...trie.entries()].forEach(([key, value]) => {
        if (!isFirstEntryAtLevel) {
            prefixEncoded += ',';
        }
        isFirstEntryAtLevel = false;
        prefixEncoded += key;
        if (value.size > 1) {
            prefixEncoded += '{';
            prefixEncoded += prefixEncodeTrie(value);
            prefixEncoded += '}';
        }
        else {
            prefixEncoded += prefixEncodeTrie(value);
        }
    });
    return prefixEncoded;
}
//# sourceMappingURL=data:application/json;base64,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