"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isNameOfCloudFormationIntrinsic = exports.minimalCloudFormationJoin = exports.CLOUDFORMATION_TOKEN_RESOLVER = exports.CloudFormationLang = void 0;
const lazy_1 = require("../lazy");
const resolvable_1 = require("../resolvable");
const stack_1 = require("../stack");
const token_1 = require("../token");
const cfn_utils_provider_1 = require("./cfn-utils-provider");
const resolve_1 = require("./resolve");
/**
 * Routines that know how to do operations at the CloudFormation document language level
 */
class CloudFormationLang {
    /**
     * Turn an arbitrary structure potentially containing Tokens into a JSON string.
     *
     * Returns a Token which will evaluate to CloudFormation expression that
     * will be evaluated by CloudFormation to the JSON representation of the
     * input structure.
     *
     * All Tokens substituted in this way must return strings, or the evaluation
     * in CloudFormation will fail.
     *
     * @param obj The object to stringify
     * @param space Indentation to use (default: no pretty-printing)
     */
    static toJSON(obj, space) {
        return lazy_1.Lazy.uncachedString({
            // We used to do this by hooking into `JSON.stringify()` by adding in objects
            // with custom `toJSON()` functions, but it's ultimately simpler just to
            // reimplement the `stringify()` function from scratch.
            produce: (ctx) => tokenAwareStringify(obj, space !== null && space !== void 0 ? space : 0, ctx),
        });
    }
    /**
     * Produce a CloudFormation expression to concat two arbitrary expressions when resolving
     */
    static concat(left, right) {
        if (left === undefined && right === undefined) {
            return '';
        }
        const parts = new Array();
        if (left !== undefined) {
            parts.push(left);
        }
        if (right !== undefined) {
            parts.push(right);
        }
        // Some case analysis to produce minimal expressions
        if (parts.length === 1) {
            return parts[0];
        }
        if (parts.length === 2 && typeof parts[0] === 'string' && typeof parts[1] === 'string') {
            return parts[0] + parts[1];
        }
        // Otherwise return a Join intrinsic (already in the target document language to avoid taking
        // circular dependencies on FnJoin & friends)
        return fnJoinConcat(parts);
    }
}
exports.CloudFormationLang = CloudFormationLang;
/**
 * Return a CFN intrinsic mass concatting any number of CloudFormation expressions
 */
function fnJoinConcat(parts) {
    return { 'Fn::Join': ['', minimalCloudFormationJoin('', parts)] };
}
/**
 * Perform a JSON.stringify()-like operation, except aware of Tokens and CloudFormation intrincics
 *
 * Tokens will be resolved and if any resolve to CloudFormation intrinsics, the intrinsics
 * will be lifted to the top of a giant `{ Fn::Join }` expression.
 *
 * If Tokens resolve to primitive types (for example, by using Lazies), we'll
 * use the primitive type to determine how to encode the value into the JSON.
 *
 * If Tokens resolve to CloudFormation intrinsics, we'll use the type of the encoded
 * value as a type hint to determine how to encode the value into the JSON. The difference
 * is that we add quotes (") around strings, and don't add anything around non-strings.
 *
 * The following structure:
 *
 *    { SomeAttr: resource.someAttr }
 *
 * Will JSONify to either:
 *
 *    '{ "SomeAttr": "' ++ { Fn::GetAtt: [Resource, SomeAttr] } ++ '" }'
 * or '{ "SomeAttr": ' ++ { Fn::GetAtt: [Resource, SomeAttr] } ++ ' }'
 *
 * Depending on whether `someAttr` is type-hinted to be a string or not.
 *
 * (Where ++ is the CloudFormation string-concat operation (`{ Fn::Join }`).
 *
 * -----------------------
 *
 * This work requires 2 features from the `resolve()` function:
 *
 * - INTRINSICS TYPE HINTS: intrinsics are represented by values like
 *   `{ Ref: 'XYZ' }`. These values can reference either a string or a list/number at
 *   deploy time, and from the value alone there's no way to know which. We need
 *   to know the type to know whether to JSONify this reference to:
 *
 *      '{ "referencedValue": "' ++ { Ref: XYZ } ++ '"}'
 *   or '{ "referencedValue": ' ++ { Ref: XYZ } ++ '}'
 *
 *   I.e., whether or not we need to enclose the reference in quotes or not.
 *
 *   We COULD have done this by resolving one token at a time, and looking at the
 *   type of the encoded token we were resolving to obtain a type hint. However,
 *   the `resolve()` and Token system resist a level-at-a-time resolve
 *   operation: because of the existence of post-processors, we must have done a
 *   complete recursive resolution of a token before we can look at its result
 *   (after which any type information about the sources of nested resolved
 *   values is lost).
 *
 *   To fix this, "type hints" have been added to the `resolve()` function,
 *   giving an idea of the type of the source value for compplex result values.
 *   This only works for objects (not strings and numbers) but fortunately
 *   we only care about the types of intrinsics, which are always complex values.
 *
 *   Type hinting could have been added to the `IResolvable` protocol as well,
 *   but for now we just use the type of an encoded value as a type hint. That way
 *   we don't need to annotate anything more at the L1 level--we will use the type
 *   encodings added by construct authors at the L2 levels. L1 users can escape the
 *   default decision of "string" by using `Token.asList()`.
 *
 * - COMPLEX KEYS: since tokens can be string-encoded, we can use string-encoded tokens
 *   as the keys in JavaScript objects. However, after resolution, those string-encoded
 *   tokens could resolve to intrinsics (`{ Ref: ... }`), which CANNOT be stored in
 *   JavaScript objects anymore.
 *
 *   We therefore need a protocol to store the resolved values somewhere in the JavaScript
 *   type model,  which can be returned by `resolve()`, and interpreted by `tokenAwareStringify()`
 *   to produce the correct JSON.
 *
 *   And example will quickly show the point:
 *
 *    User writes:
 *       { [resource.resourceName]: 'SomeValue' }
 *    ------ string actually looks like ------>
 *       { '${Token[1234]}': 'SomeValue' }
 *    ------ resolve ------->
 *       { '$IntrinsicKey$0': [ {Ref: Resource}, 'SomeValue' ] }
 *    ------ tokenAwareStringify ------->
 *       '{ "' ++ { Ref: Resource } ++ '": "SomeValue" }'
 */
function tokenAwareStringify(root, space, ctx) {
    let indent = 0;
    const ret = new Array();
    // First completely resolve the tree, then encode to JSON while respecting the type
    // hints we got for the resolved intrinsics.
    recurse(ctx.resolve(root, { allowIntrinsicKeys: true }));
    switch (ret.length) {
        case 0: return undefined;
        case 1: return renderSegment(ret[0]);
        default:
            return fnJoinConcat(ret.map(renderSegment));
    }
    /**
     * Stringify a JSON element
     */
    function recurse(obj) {
        if (obj === undefined) {
            return;
        }
        if (token_1.Token.isUnresolved(obj)) {
            throw new Error('This shouldnt happen anymore');
        }
        if (Array.isArray(obj)) {
            return renderCollection('[', ']', obj, recurse);
        }
        if (typeof obj === 'object' && obj != null && !(obj instanceof Date)) {
            // Treat as an intrinsic if this LOOKS like a CFN intrinsic (`{ Ref: ... }`)
            // AND it's the result of a token resolution. Otherwise, we just treat this
            // value as a regular old JSON object (that happens to look a lot like an intrinsic).
            if (isIntrinsic(obj) && resolve_1.resolvedTypeHint(obj)) {
                renderIntrinsic(obj);
                return;
            }
            return renderCollection('{', '}', definedEntries(obj), ([key, value]) => {
                if (key.startsWith(resolve_1.INTRINSIC_KEY_PREFIX)) {
                    [key, value] = value;
                }
                recurse(key);
                pushLiteral(prettyPunctuation(':'));
                recurse(value);
            });
        }
        // Otherwise we have a scalar, defer to JSON.stringify()s serialization
        pushLiteral(JSON.stringify(obj));
    }
    /**
     * Render an object or list
     */
    function renderCollection(pre, post, xs, each) {
        pushLiteral(pre);
        indent += space;
        let atLeastOne = false;
        for (const [comma, item] of sepIter(xs)) {
            if (comma) {
                pushLiteral(',');
            }
            pushLineBreak();
            each(item);
            atLeastOne = true;
        }
        indent -= space;
        if (atLeastOne) {
            pushLineBreak();
        }
        pushLiteral(post);
    }
    function renderIntrinsic(intrinsic) {
        switch (resolve_1.resolvedTypeHint(intrinsic)) {
            case resolve_1.ResolutionTypeHint.STRING:
                pushLiteral('"');
                pushIntrinsic(deepQuoteStringLiterals(intrinsic));
                pushLiteral('"');
                return;
            case resolve_1.ResolutionTypeHint.LIST:
                // We need this to look like:
                //
                //    '{"listValue":' ++ STRINGIFY(CFN_EVAL({ Ref: MyList })) ++ '}'
                //
                // However, STRINGIFY would need to execute at CloudFormation deployment time, and that doesn't exist.
                //
                // We could *ALMOST* use:
                //
                //   '{"listValue":["' ++ JOIN('","', { Ref: MyList }) ++ '"]}'
                //
                // But that has the unfortunate side effect that if `CFN_EVAL({ Ref: MyList }) == []`, then it would
                // evaluate to `[""]`, which is a different value. Since CloudFormation does not have arbitrary
                // conditionals there's no way to deal with this case properly.
                //
                // Therefore, if we encounter lists we need to defer to a custom resource to handle
                // them properly at deploy time.
                const stack = stack_1.Stack.of(ctx.scope);
                // Because this will be called twice (once during `prepare`, once during `resolve`),
                // we need to make sure to be idempotent, so use a cache.
                const stringifyResponse = stringifyCache.obtain(stack, JSON.stringify(intrinsic), () => cfn_utils_provider_1.CfnUtils.stringify(stack, `CdkJsonStringify${stringifyCounter++}`, intrinsic));
                pushIntrinsic(stringifyResponse);
                return;
            case resolve_1.ResolutionTypeHint.NUMBER:
                pushIntrinsic(intrinsic);
                return;
        }
        throw new Error(`Unexpected type hint: ${resolve_1.resolvedTypeHint(intrinsic)}`);
    }
    /**
     * Push a literal onto the current segment if it's also a literal, otherwise open a new Segment
     */
    function pushLiteral(lit) {
        let last = ret[ret.length - 1];
        if ((last === null || last === void 0 ? void 0 : last.type) !== 'literal') {
            last = { type: 'literal', parts: [] };
            ret.push(last);
        }
        last.parts.push(lit);
    }
    /**
     * Add a new intrinsic segment
     */
    function pushIntrinsic(intrinsic) {
        ret.push({ type: 'intrinsic', intrinsic });
    }
    /**
     * Push a line break if we are pretty-printing, otherwise don't
     */
    function pushLineBreak() {
        if (space > 0) {
            pushLiteral(`\n${' '.repeat(indent)}`);
        }
    }
    /**
     * Add a space after the punctuation if we are pretty-printing, no space if not
     */
    function prettyPunctuation(punc) {
        return space > 0 ? `${punc} ` : punc;
    }
}
/**
 * Render a segment
 */
function renderSegment(s) {
    switch (s.type) {
        case 'literal': return s.parts.join('');
        case 'intrinsic': return s.intrinsic;
    }
}
const CLOUDFORMATION_CONCAT = {
    join(left, right) {
        return CloudFormationLang.concat(left, right);
    },
};
/**
 * Default Token resolver for CloudFormation templates
 */
exports.CLOUDFORMATION_TOKEN_RESOLVER = new resolvable_1.DefaultTokenResolver(CLOUDFORMATION_CONCAT);
/**
 * Do an intelligent CloudFormation join on the given values, producing a minimal expression
 */
function minimalCloudFormationJoin(delimiter, values) {
    let i = 0;
    while (i < values.length) {
        const el = values[i];
        if (isSplicableFnJoinIntrinsic(el)) {
            values.splice(i, 1, ...el['Fn::Join'][1]);
        }
        else if (i > 0 && isPlainString(values[i - 1]) && isPlainString(values[i])) {
            values[i - 1] += delimiter + values[i];
            values.splice(i, 1);
        }
        else {
            i += 1;
        }
    }
    return values;
    function isPlainString(obj) {
        return typeof obj === 'string' && !token_1.Token.isUnresolved(obj);
    }
    function isSplicableFnJoinIntrinsic(obj) {
        if (!isIntrinsic(obj)) {
            return false;
        }
        if (Object.keys(obj)[0] !== 'Fn::Join') {
            return false;
        }
        const [delim, list] = obj['Fn::Join'];
        if (delim !== delimiter) {
            return false;
        }
        if (token_1.Token.isUnresolved(list)) {
            return false;
        }
        if (!Array.isArray(list)) {
            return false;
        }
        return true;
    }
}
exports.minimalCloudFormationJoin = minimalCloudFormationJoin;
/**
 * Return whether the given value represents a CloudFormation intrinsic
 */
function isIntrinsic(x) {
    if (Array.isArray(x) || x === null || typeof x !== 'object') {
        return false;
    }
    const keys = Object.keys(x);
    if (keys.length !== 1) {
        return false;
    }
    return keys[0] === 'Ref' || isNameOfCloudFormationIntrinsic(keys[0]);
}
function isNameOfCloudFormationIntrinsic(name) {
    if (!name.startsWith('Fn::')) {
        return false;
    }
    // these are 'fake' intrinsics, only usable inside the parameter overrides of a CFN CodePipeline Action
    return name !== 'Fn::GetArtifactAtt' && name !== 'Fn::GetParam';
}
exports.isNameOfCloudFormationIntrinsic = isNameOfCloudFormationIntrinsic;
/**
 * Separated iterator
 */
function* sepIter(xs) {
    let comma = false;
    for (const item of xs) {
        yield [comma, item];
        comma = true;
    }
}
/**
 * Object.entries() but skipping undefined values
 */
function* definedEntries(xs) {
    for (const [key, value] of Object.entries(xs)) {
        if (value !== undefined) {
            yield [key, value];
        }
    }
}
/**
 * Quote string literals inside an intrinsic
 *
 * Formally, this should only match string literals that will be interpreted as
 * string literals. Fortunately, the strings that should NOT be quoted are
 * Logical IDs and attribute names, which cannot contain quotes anyway. Hence,
 * we can get away not caring about the distinction and just quoting everything.
 */
function deepQuoteStringLiterals(x) {
    if (Array.isArray(x)) {
        return x.map(deepQuoteStringLiterals);
    }
    if (typeof x === 'object' && x != null) {
        const ret = {};
        for (const [key, value] of Object.entries(x)) {
            ret[deepQuoteStringLiterals(key)] = deepQuoteStringLiterals(value);
        }
        return ret;
    }
    if (typeof x === 'string') {
        return quoteString(x);
    }
    return x;
}
/**
 * Quote the characters inside a string, for use inside toJSON
 */
function quoteString(s) {
    s = JSON.stringify(s);
    return s.substring(1, s.length - 1);
}
let stringifyCounter = 1;
/**
 * A cache scoped to object instances, that's maintained externally to the object instances
 */
class ScopedCache {
    constructor() {
        this.cache = new WeakMap();
    }
    obtain(object, key, init) {
        let kvMap = this.cache.get(object);
        if (!kvMap) {
            kvMap = new Map();
            this.cache.set(object, kvMap);
        }
        let ret = kvMap.get(key);
        if (ret === undefined) {
            ret = init();
            kvMap.set(key, ret);
        }
        return ret;
    }
}
const stringifyCache = new ScopedCache();
//# sourceMappingURL=data:application/json;base64,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