"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnReference = exports.ReferenceRendering = void 0;
const reference_1 = require("../reference");
const CFN_REFERENCE_SYMBOL = Symbol.for('@aws-cdk/core.CfnReference');
/**
 * An enum that allows controlling how will the created reference
 * be rendered in the resulting CloudFormation template.
 */
var ReferenceRendering;
(function (ReferenceRendering) {
    /**
     * Used for rendering a reference inside Fn::Sub expressions,
     * which mean these must resolve to "${Sth}" instead of { Ref: "Sth" }.
     */
    ReferenceRendering[ReferenceRendering["FN_SUB"] = 0] = "FN_SUB";
    /**
     * Used for rendering Fn::GetAtt with its arguments in string form
     * (as opposed to the more common arguments in array form, which we render by default).
     */
    ReferenceRendering[ReferenceRendering["GET_ATT_STRING"] = 1] = "GET_ATT_STRING";
})(ReferenceRendering = exports.ReferenceRendering || (exports.ReferenceRendering = {}));
/**
 * A Token that represents a CloudFormation reference to another resource
 *
 * If these references are used in a different stack from where they are
 * defined, appropriate CloudFormation `Export`s and `Fn::ImportValue`s will be
 * synthesized automatically instead of the regular CloudFormation references.
 *
 * Additionally, the dependency between the stacks will be recorded, and the toolkit
 * will make sure to deploy producing stack before the consuming stack.
 *
 * This magic happens in the prepare() phase, where consuming stacks will call
 * `consumeFromStack` on these Tokens and if they happen to be exported by a different
 * Stack, we'll register the dependency.
 */
class CfnReference extends reference_1.Reference {
    constructor(value, displayName, target) {
        // prepend scope path to display name
        super(value, target, displayName);
        this.replacementTokens = new Map();
        this.targetStack = stack_1.Stack.of(target);
        Object.defineProperty(this, CFN_REFERENCE_SYMBOL, { value: true });
    }
    /**
     * Check whether this is actually a Reference
     */
    static isCfnReference(x) {
        return CFN_REFERENCE_SYMBOL in x;
    }
    /**
     * Return the CfnReference for the indicated target
     *
     * Will make sure that multiple invocations for the same target and intrinsic
     * return the same CfnReference. Because CfnReferences accumulate state in
     * the prepare() phase (for the purpose of cross-stack references), it's
     * important that the state isn't lost if it's lazily created, like so:
     *
     *     Lazy.string({ produce: () => new CfnReference(...) })
     *
     */
    static for(target, attribute, refRender) {
        return CfnReference.singletonReference(target, attribute, refRender, () => {
            const cfnIntrinsic = refRender === ReferenceRendering.FN_SUB
                ? ('${' + target.logicalId + (attribute === 'Ref' ? '' : `.${attribute}`) + '}')
                : (attribute === 'Ref'
                    ? { Ref: target.logicalId }
                    : {
                        'Fn::GetAtt': refRender === ReferenceRendering.GET_ATT_STRING
                            ? `${target.logicalId}.${attribute}`
                            : [target.logicalId, attribute],
                    });
            return new CfnReference(cfnIntrinsic, attribute, target);
        });
    }
    /**
     * Return a CfnReference that references a pseudo referencd
     */
    static forPseudo(pseudoName, scope) {
        return CfnReference.singletonReference(scope, `Pseudo:${pseudoName}`, undefined, () => {
            const cfnIntrinsic = { Ref: pseudoName };
            return new CfnReference(cfnIntrinsic, pseudoName, scope);
        });
    }
    /**
     * Get or create the table.
     * Passing fnSub = true allows cloudformation-include to correctly handle Fn::Sub.
     */
    static singletonReference(target, attribKey, refRender, fresh) {
        let attribs = CfnReference.referenceTable.get(target);
        if (!attribs) {
            attribs = new Map();
            CfnReference.referenceTable.set(target, attribs);
        }
        let cacheKey = attribKey;
        switch (refRender) {
            case ReferenceRendering.FN_SUB:
                cacheKey += 'Fn::Sub';
                break;
            case ReferenceRendering.GET_ATT_STRING:
                cacheKey += 'Fn::GetAtt::String';
                break;
        }
        let ref = attribs.get(cacheKey);
        if (!ref) {
            ref = fresh();
            attribs.set(cacheKey, ref);
        }
        return ref;
    }
    resolve(context) {
        // If we have a special token for this consuming stack, resolve that. Otherwise resolve as if
        // we are in the same stack.
        const consumingStack = stack_1.Stack.of(context.scope);
        const token = this.replacementTokens.get(consumingStack);
        // if (!token && this.isCrossStackReference(consumingStack) && !context.preparing) {
        // eslint-disable-next-line max-len
        //   throw new Error(`Cross-stack reference (${context.scope.node.path} -> ${this.target.node.path}) has not been assigned a value--call prepare() first`);
        // }
        if (token) {
            return token.resolve(context);
        }
        else {
            return super.resolve(context);
        }
    }
    hasValueForStack(stack) {
        if (stack === this.targetStack) {
            return true;
        }
        return this.replacementTokens.has(stack);
    }
    assignValueForStack(stack, value) {
        if (stack === this.targetStack) {
            throw new Error('cannot assign a value for the same stack');
        }
        if (this.hasValueForStack(stack)) {
            throw new Error('Cannot assign a reference value twice to the same stack. Use hasValueForStack to check first');
        }
        this.replacementTokens.set(stack, value);
    }
    /**
     * Implementation of toString() that will use the display name
     */
    toString() {
        return token_1.Token.asString(this, {
            displayHint: `${this.target.node.id}.${this.displayName}`,
        });
    }
}
exports.CfnReference = CfnReference;
/**
 * Static table where we keep singleton CfnReference instances
 */
CfnReference.referenceTable = new Map();
const stack_1 = require("../stack");
const token_1 = require("../token");
//# sourceMappingURL=data:application/json;base64,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