"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParser = exports.CfnParsingContext = exports.FromCloudFormation = exports.FromCloudFormationPropertyObject = exports.FromCloudFormationResult = void 0;
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const lazy_1 = require("./lazy");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
const util_1 = require("./util");
/**
 * The class used as the intermediate result from the generated L1 methods
 * that convert from CloudFormation's UpperCase to CDK's lowerCase property names.
 * Saves any extra properties that were present in the argument object,
 * but that were not found in the CFN schema,
 * so that they're not lost from the final CDK-rendered template.
 */
class FromCloudFormationResult {
    constructor(value) {
        this.value = value;
        this.extraProperties = {};
    }
    appendExtraProperties(prefix, properties) {
        for (const [key, val] of Object.entries(properties !== null && properties !== void 0 ? properties : {})) {
            this.extraProperties[`${prefix}.${key}`] = val;
        }
    }
}
exports.FromCloudFormationResult = FromCloudFormationResult;
/**
 * A property object we will accumulate properties into
 */
class FromCloudFormationPropertyObject extends FromCloudFormationResult {
    constructor() {
        super({}); // We're still accumulating
        this.recognizedProperties = new Set();
    }
    /**
     * Add a parse result under a given key
     */
    addPropertyResult(cdkPropName, cfnPropName, result) {
        this.recognizedProperties.add(cfnPropName);
        if (!result) {
            return;
        }
        this.value[cdkPropName] = result.value;
        this.appendExtraProperties(cfnPropName, result.extraProperties);
    }
    addUnrecognizedPropertiesAsExtra(properties) {
        for (const [key, val] of Object.entries(properties)) {
            if (!this.recognizedProperties.has(key)) {
                this.extraProperties[key] = val;
            }
        }
    }
}
exports.FromCloudFormationPropertyObject = FromCloudFormationPropertyObject;
/**
 * This class contains static methods called when going from
 * translated values received from {@link CfnParser.parseValue}
 * to the actual L1 properties -
 * things like changing IResolvable to the appropriate type
 * (string, string array, or number), etc.
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code.
 *
 */
class FromCloudFormation {
    // nothing to for any but return it
    static getAny(value) {
        return new FromCloudFormationResult(value);
    }
    static getBoolean(value) {
        if (typeof value === 'string') {
            // CloudFormation allows passing strings as boolean
            switch (value) {
                case 'true': return new FromCloudFormationResult(true);
                case 'false': return new FromCloudFormationResult(false);
                default: throw new Error(`Expected 'true' or 'false' for boolean value, got: '${value}'`);
            }
        }
        // in all other cases, just return the value,
        // and let a validator handle if it's not a boolean
        return new FromCloudFormationResult(value);
    }
    static getDate(value) {
        // if the date is a deploy-time value, just return it
        if (token_1.isResolvableObject(value)) {
            return new FromCloudFormationResult(value);
        }
        // if the date has been given as a string, convert it
        if (typeof value === 'string') {
            return new FromCloudFormationResult(new Date(value));
        }
        // all other cases - just return the value,
        // if it's not a Date, a validator should catch it
        return new FromCloudFormationResult(value);
    }
    // won't always return a string; if the input can't be resolved to a string,
    // the input will be returned.
    static getString(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return new FromCloudFormationResult(value.toString());
        }
        // CloudFormation treats numbers and strings interchangeably;
        // so, if we get a number here, convert it to a string
        if (typeof value === 'number') {
            return new FromCloudFormationResult(value.toString());
        }
        // CloudFormation treats booleans and strings interchangeably;
        // so, if we get a boolean here, convert it to a string
        if (typeof value === 'boolean') {
            return new FromCloudFormationResult(value.toString());
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a string
        return new FromCloudFormationResult(value);
    }
    // won't always return a number; if the input can't be parsed to a number,
    // the input will be returned.
    static getNumber(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return new FromCloudFormationResult(token_1.Token.asNumber(value));
        }
        // return a number, if the input can be parsed as one
        if (typeof value === 'string') {
            const parsedValue = parseFloat(value);
            if (!isNaN(parsedValue)) {
                return new FromCloudFormationResult(parsedValue);
            }
        }
        // otherwise return the input,
        // and let a validator handle it if it's not a number
        return new FromCloudFormationResult(value);
    }
    static getStringArray(value) {
        // if the array is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return new FromCloudFormationResult(token_1.Token.asList(value));
        }
        // in all other cases, delegate to the standard mapping logic
        return this.getArray(this.getString)(value);
    }
    static getArray(mapper) {
        return (value) => {
            if (!Array.isArray(value)) {
                // break the type system, and just return the given value,
                // which hopefully will be reported as invalid by the validator
                // of the property we're transforming
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return new FromCloudFormationResult(value);
            }
            const values = new Array();
            const ret = new FromCloudFormationResult(values);
            for (let i = 0; i < value.length; i++) {
                const result = mapper(value[i]);
                values.push(result.value);
                ret.appendExtraProperties(`${i}`, result.extraProperties);
            }
            return ret;
        };
    }
    static getMap(mapper) {
        return (value) => {
            if (typeof value !== 'object') {
                // if the input is not a map (= object in JS land),
                // just return it, and let the validator of this property handle it
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return new FromCloudFormationResult(value);
            }
            const values = {};
            const ret = new FromCloudFormationResult(values);
            for (const [key, val] of Object.entries(value)) {
                const result = mapper(val);
                values[key] = result.value;
                ret.appendExtraProperties(key, result.extraProperties);
            }
            return ret;
        };
    }
    static getCfnTag(tag) {
        return tag == null
            ? new FromCloudFormationResult({}) // break the type system - this should be detected at runtime by a tag validator
            : new FromCloudFormationResult({
                key: tag.Key,
                value: tag.Value,
            });
    }
    /**
     * Return a function that, when applied to a value, will return the first validly deserialized one
     */
    static getTypeUnion(validators, mappers) {
        return (value) => {
            for (let i = 0; i < validators.length; i++) {
                const candidate = mappers[i](value);
                if (validators[i](candidate.value).isSuccess) {
                    return candidate;
                }
            }
            // if nothing matches, just return the input unchanged, and let validators catch it
            return new FromCloudFormationResult(value);
        };
    }
}
exports.FromCloudFormation = FromCloudFormation;
/**
 * The context in which the parsing is taking place.
 *
 * Some fragments of CloudFormation templates behave differently than others
 * (for example, the 'Conditions' sections treats { "Condition": "NameOfCond" }
 * differently than the 'Resources' section).
 * This enum can be used to change the created {@link CfnParser} behavior,
 * based on the template context.
 */
var CfnParsingContext;
(function (CfnParsingContext) {
    /** We're currently parsing the 'Conditions' section. */
    CfnParsingContext[CfnParsingContext["CONDITIONS"] = 0] = "CONDITIONS";
    /** We're currently parsing the 'Rules' section. */
    CfnParsingContext[CfnParsingContext["RULES"] = 1] = "RULES";
})(CfnParsingContext = exports.CfnParsingContext || (exports.CfnParsingContext = {}));
/**
 * This class contains methods for translating from a pure CFN value
 * (like a JS object { "Ref": "Bucket" })
 * to a form CDK understands
 * (like Fn.ref('Bucket')).
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code,
 * so any renames of it need to be reflected in cfn2ts/codegen.ts as well.
 *
 */
class CfnParser {
    constructor(options) {
        this.options = options;
    }
    handleAttributes(resource, resourceAttributes, logicalId) {
        var _a;
        const cfnOptions = resource.cfnOptions;
        cfnOptions.creationPolicy = this.parseCreationPolicy(resourceAttributes.CreationPolicy);
        cfnOptions.updatePolicy = this.parseUpdatePolicy(resourceAttributes.UpdatePolicy);
        cfnOptions.deletionPolicy = this.parseDeletionPolicy(resourceAttributes.DeletionPolicy);
        cfnOptions.updateReplacePolicy = this.parseDeletionPolicy(resourceAttributes.UpdateReplacePolicy);
        cfnOptions.version = this.parseValue(resourceAttributes.Version);
        cfnOptions.description = this.parseValue(resourceAttributes.Description);
        cfnOptions.metadata = this.parseValue(resourceAttributes.Metadata);
        // handle Condition
        if (resourceAttributes.Condition) {
            const condition = this.finder.findCondition(resourceAttributes.Condition);
            if (!condition) {
                throw new Error(`Resource '${logicalId}' uses Condition '${resourceAttributes.Condition}' that doesn't exist`);
            }
            cfnOptions.condition = condition;
        }
        // handle DependsOn
        resourceAttributes.DependsOn = (_a = resourceAttributes.DependsOn) !== null && _a !== void 0 ? _a : [];
        const dependencies = Array.isArray(resourceAttributes.DependsOn) ?
            resourceAttributes.DependsOn : [resourceAttributes.DependsOn];
        for (const dep of dependencies) {
            const depResource = this.finder.findResource(dep);
            if (!depResource) {
                throw new Error(`Resource '${logicalId}' depends on '${dep}' that doesn't exist`);
            }
            resource.node.addDependency(depResource);
        }
    }
    parseCreationPolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return util_1.undefinedIfAllValuesAreEmpty({
            autoScalingCreationPolicy: parseAutoScalingCreationPolicy(policy.AutoScalingCreationPolicy),
            resourceSignal: parseResourceSignal(policy.ResourceSignal),
        });
        function parseAutoScalingCreationPolicy(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent).value,
            });
        }
        function parseResourceSignal(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                count: FromCloudFormation.getNumber(p.Count).value,
                timeout: FromCloudFormation.getString(p.Timeout).value,
            });
        }
    }
    parseUpdatePolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return util_1.undefinedIfAllValuesAreEmpty({
            autoScalingReplacingUpdate: parseAutoScalingReplacingUpdate(policy.AutoScalingReplacingUpdate),
            autoScalingRollingUpdate: parseAutoScalingRollingUpdate(policy.AutoScalingRollingUpdate),
            autoScalingScheduledAction: parseAutoScalingScheduledAction(policy.AutoScalingScheduledAction),
            codeDeployLambdaAliasUpdate: parseCodeDeployLambdaAliasUpdate(policy.CodeDeployLambdaAliasUpdate),
            enableVersionUpgrade: FromCloudFormation.getBoolean(policy.EnableVersionUpgrade).value,
            useOnlineResharding: FromCloudFormation.getBoolean(policy.UseOnlineResharding).value,
        });
        function parseAutoScalingReplacingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                willReplace: p.WillReplace,
            });
        }
        function parseAutoScalingRollingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                maxBatchSize: FromCloudFormation.getNumber(p.MaxBatchSize).value,
                minInstancesInService: FromCloudFormation.getNumber(p.MinInstancesInService).value,
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent).value,
                pauseTime: FromCloudFormation.getString(p.PauseTime).value,
                suspendProcesses: FromCloudFormation.getStringArray(p.SuspendProcesses).value,
                waitOnResourceSignals: FromCloudFormation.getBoolean(p.WaitOnResourceSignals).value,
            });
        }
        function parseCodeDeployLambdaAliasUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return {
                beforeAllowTrafficHook: FromCloudFormation.getString(p.BeforeAllowTrafficHook).value,
                afterAllowTrafficHook: FromCloudFormation.getString(p.AfterAllowTrafficHook).value,
                applicationName: FromCloudFormation.getString(p.ApplicationName).value,
                deploymentGroupName: FromCloudFormation.getString(p.DeploymentGroupName).value,
            };
        }
        function parseAutoScalingScheduledAction(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                ignoreUnmodifiedGroupSizeProperties: FromCloudFormation.getBoolean(p.IgnoreUnmodifiedGroupSizeProperties).value,
            });
        }
    }
    parseDeletionPolicy(policy) {
        switch (policy) {
            case null: return undefined;
            case undefined: return undefined;
            case 'Delete': return cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
            case 'Retain': return cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
            case 'Snapshot': return cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
            default: throw new Error(`Unrecognized DeletionPolicy '${policy}'`);
        }
    }
    parseValue(cfnValue) {
        // == null captures undefined as well
        if (cfnValue == null) {
            return undefined;
        }
        // if we have any late-bound values,
        // just return them
        if (token_1.isResolvableObject(cfnValue)) {
            return cfnValue;
        }
        if (Array.isArray(cfnValue)) {
            return cfnValue.map(el => this.parseValue(el));
        }
        if (typeof cfnValue === 'object') {
            // an object can be either a CFN intrinsic, or an actual object
            const cfnIntrinsic = this.parseIfCfnIntrinsic(cfnValue);
            if (cfnIntrinsic !== undefined) {
                return cfnIntrinsic;
            }
            const ret = {};
            for (const [key, val] of Object.entries(cfnValue)) {
                ret[key] = this.parseValue(val);
            }
            return ret;
        }
        // in all other cases, just return the input
        return cfnValue;
    }
    get finder() {
        return this.options.finder;
    }
    parseIfCfnIntrinsic(object) {
        const key = this.looksLikeCfnIntrinsic(object);
        switch (key) {
            case undefined:
                return undefined;
            case 'Ref': {
                const refTarget = object[key];
                const specialRef = this.specialCaseRefs(refTarget);
                if (specialRef !== undefined) {
                    return specialRef;
                }
                else {
                    const refElement = this.finder.findRefTarget(refTarget);
                    if (!refElement) {
                        throw new Error(`Element used in Ref expression with logical ID: '${refTarget}' not found`);
                    }
                    return cfn_reference_1.CfnReference.for(refElement, 'Ref');
                }
            }
            case 'Fn::GetAtt': {
                const value = object[key];
                let logicalId, attributeName, stringForm;
                // Fn::GetAtt takes as arguments either a string...
                if (typeof value === 'string') {
                    // ...in which case the logical ID and the attribute name are separated with '.'
                    const dotIndex = value.indexOf('.');
                    if (dotIndex === -1) {
                        throw new Error(`Short-form Fn::GetAtt must contain a '.' in its string argument, got: '${value}'`);
                    }
                    logicalId = value.substr(0, dotIndex);
                    attributeName = value.substr(dotIndex + 1); // the +1 is to skip the actual '.'
                    stringForm = true;
                }
                else {
                    // ...or a 2-element list
                    logicalId = value[0];
                    attributeName = value[1];
                    stringForm = false;
                }
                const target = this.finder.findResource(logicalId);
                if (!target) {
                    throw new Error(`Resource used in GetAtt expression with logical ID: '${logicalId}' not found`);
                }
                return cfn_reference_1.CfnReference.for(target, attributeName, stringForm ? cfn_reference_1.ReferenceRendering.GET_ATT_STRING : undefined);
            }
            case 'Fn::Join': {
                // Fn::Join takes a 2-element list as its argument,
                // where the first element is the delimiter,
                // and the second is the list of elements to join
                const value = this.parseValue(object[key]);
                // wrap the array as a Token,
                // as otherwise Fn.join() will try to concatenate
                // the non-token parts,
                // causing a diff with the original template
                return cfn_fn_1.Fn.join(value[0], lazy_1.Lazy.list({ produce: () => value[1] }));
            }
            case 'Fn::Cidr': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.cidr(value[0], value[1], value[2]);
            }
            case 'Fn::FindInMap': {
                const value = this.parseValue(object[key]);
                // the first argument to FindInMap is the mapping name
                let mappingName;
                if (token_1.Token.isUnresolved(value[0])) {
                    // the first argument can be a dynamic expression like Ref: Param;
                    // if it is, we can't find the mapping in advance
                    mappingName = value[0];
                }
                else {
                    const mapping = this.finder.findMapping(value[0]);
                    if (!mapping) {
                        throw new Error(`Mapping used in FindInMap expression with name '${value[0]}' was not found in the template`);
                    }
                    mappingName = mapping.logicalId;
                }
                return cfn_fn_1.Fn._findInMap(mappingName, value[1], value[2]);
            }
            case 'Fn::Select': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.select(value[0], value[1]);
            }
            case 'Fn::GetAZs': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.getAzs(value);
            }
            case 'Fn::ImportValue': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.importValue(value);
            }
            case 'Fn::Split': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.split(value[0], value[1]);
            }
            case 'Fn::Transform': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.transform(value.Name, value.Parameters);
            }
            case 'Fn::Base64': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.base64(value);
            }
            case 'Fn::If': {
                // Fn::If takes a 3-element list as its argument,
                // where the first element is the name of a Condition
                const value = this.parseValue(object[key]);
                const condition = this.finder.findCondition(value[0]);
                if (!condition) {
                    throw new Error(`Condition '${value[0]}' used in an Fn::If expression does not exist in the template`);
                }
                return cfn_fn_1.Fn.conditionIf(condition.logicalId, value[1], value[2]);
            }
            case 'Fn::Equals': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionEquals(value[0], value[1]);
            }
            case 'Fn::And': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionAnd(...value);
            }
            case 'Fn::Not': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionNot(value[0]);
            }
            case 'Fn::Or': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionOr(...value);
            }
            case 'Fn::Sub': {
                const value = this.parseValue(object[key]);
                let fnSubString;
                let map;
                if (typeof value === 'string') {
                    fnSubString = value;
                    map = undefined;
                }
                else {
                    fnSubString = value[0];
                    map = value[1];
                }
                return this.parseFnSubString(fnSubString, map);
            }
            case 'Condition': {
                // a reference to a Condition from another Condition
                const condition = this.finder.findCondition(object[key]);
                if (!condition) {
                    throw new Error(`Referenced Condition with name '${object[key]}' was not found in the template`);
                }
                return { Condition: condition.logicalId };
            }
            default:
                if (this.options.context === CfnParsingContext.RULES) {
                    return this.handleRulesIntrinsic(key, object);
                }
                else {
                    throw new Error(`Unsupported CloudFormation function '${key}'`);
                }
        }
    }
    looksLikeCfnIntrinsic(object) {
        const objectKeys = Object.keys(object);
        // a CFN intrinsic is always an object with a single key
        if (objectKeys.length !== 1) {
            return undefined;
        }
        const key = objectKeys[0];
        return key === 'Ref' || key.startsWith('Fn::') ||
            // special intrinsic only available in the 'Conditions' section
            (this.options.context === CfnParsingContext.CONDITIONS && key === 'Condition')
            ? key
            : undefined;
    }
    parseFnSubString(templateString, expressionMap) {
        const map = expressionMap !== null && expressionMap !== void 0 ? expressionMap : {};
        const self = this;
        return cfn_fn_1.Fn.sub(go(templateString), Object.keys(map).length === 0 ? expressionMap : map);
        function go(value) {
            const leftBrace = value.indexOf('${');
            if (leftBrace === -1) {
                return value;
            }
            // search for the closing brace to the right of the opening '${'
            // (in theory, there could be other braces in the string,
            // for example if it represents a JSON object)
            const rightBrace = value.indexOf('}', leftBrace);
            if (rightBrace === -1) {
                return value;
            }
            const leftHalf = value.substring(0, leftBrace);
            const rightHalf = value.substring(rightBrace + 1);
            // don't include left and right braces when searching for the target of the reference
            const refTarget = value.substring(leftBrace + 2, rightBrace).trim();
            if (refTarget[0] === '!') {
                return value.substring(0, rightBrace + 1) + go(rightHalf);
            }
            // lookup in map
            if (refTarget in map) {
                return leftHalf + '${' + refTarget + '}' + go(rightHalf);
            }
            // since it's not in the map, check if it's a pseudo-parameter
            // (or a value to be substituted for a Parameter, provided by the customer)
            const specialRef = self.specialCaseSubRefs(refTarget);
            if (specialRef !== undefined) {
                if (token_1.Token.isUnresolved(specialRef)) {
                    // specialRef can only be a Token if the value passed by the customer
                    // for substituting a Parameter was a Token.
                    // This is actually bad here,
                    // because the Token can potentially be something that doesn't render
                    // well inside an Fn::Sub template string, like a { Ref } object.
                    // To handle this case,
                    // instead of substituting the Parameter directly with the token in the template string,
                    // add a new entry to the Fn::Sub map,
                    // with key refTarget, and the token as the value.
                    // This is safe, because this sort of shadowing is legal in CloudFormation,
                    // and also because we're certain the Fn::Sub map doesn't contain an entry for refTarget
                    // (as we check that condition in the code right above this).
                    map[refTarget] = specialRef;
                    return leftHalf + '${' + refTarget + '}' + go(rightHalf);
                }
                else {
                    return leftHalf + specialRef + go(rightHalf);
                }
            }
            const dotIndex = refTarget.indexOf('.');
            const isRef = dotIndex === -1;
            if (isRef) {
                const refElement = self.finder.findRefTarget(refTarget);
                if (!refElement) {
                    throw new Error(`Element referenced in Fn::Sub expression with logical ID: '${refTarget}' was not found in the template`);
                }
                return leftHalf + cfn_reference_1.CfnReference.for(refElement, 'Ref', cfn_reference_1.ReferenceRendering.FN_SUB).toString() + go(rightHalf);
            }
            else {
                const targetId = refTarget.substring(0, dotIndex);
                const refResource = self.finder.findResource(targetId);
                if (!refResource) {
                    throw new Error(`Resource referenced in Fn::Sub expression with logical ID: '${targetId}' was not found in the template`);
                }
                const attribute = refTarget.substring(dotIndex + 1);
                return leftHalf + cfn_reference_1.CfnReference.for(refResource, attribute, cfn_reference_1.ReferenceRendering.FN_SUB).toString() + go(rightHalf);
            }
        }
    }
    handleRulesIntrinsic(key, object) {
        // Rules have their own set of intrinsics:
        // https://docs.aws.amazon.com/servicecatalog/latest/adminguide/intrinsic-function-reference-rules.html
        switch (key) {
            case 'Fn::ValueOf': {
                // ValueOf is special,
                // as it takes the name of a Parameter as its first argument
                const value = this.parseValue(object[key]);
                const parameterName = value[0];
                if (parameterName in this.parameters) {
                    // since ValueOf returns the value of a specific attribute,
                    // fail here - this substitution is not allowed
                    throw new Error(`Cannot substitute parameter '${parameterName}' used in Fn::ValueOf expression with attribute '${value[1]}'`);
                }
                const param = this.finder.findRefTarget(parameterName);
                if (!param) {
                    throw new Error(`Rule references parameter '${parameterName}' which was not found in the template`);
                }
                // create an explicit IResolvable,
                // as Fn.valueOf() returns a string,
                // which is incorrect
                // (Fn::ValueOf can also return an array)
                return lazy_1.Lazy.any({ produce: () => ({ 'Fn::ValueOf': [param.logicalId, value[1]] }) });
            }
            default:
                // I don't want to hard-code the list of supported Rules-specific intrinsics in this function;
                // so, just return undefined here,
                // and they will be treated as a regular JSON object
                return undefined;
        }
    }
    specialCaseRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        switch (value) {
            case 'AWS::AccountId': return cfn_pseudo_1.Aws.ACCOUNT_ID;
            case 'AWS::Region': return cfn_pseudo_1.Aws.REGION;
            case 'AWS::Partition': return cfn_pseudo_1.Aws.PARTITION;
            case 'AWS::URLSuffix': return cfn_pseudo_1.Aws.URL_SUFFIX;
            case 'AWS::NotificationARNs': return cfn_pseudo_1.Aws.NOTIFICATION_ARNS;
            case 'AWS::StackId': return cfn_pseudo_1.Aws.STACK_ID;
            case 'AWS::StackName': return cfn_pseudo_1.Aws.STACK_NAME;
            case 'AWS::NoValue': return cfn_pseudo_1.Aws.NO_VALUE;
            default: return undefined;
        }
    }
    specialCaseSubRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        return value.indexOf('::') === -1 ? undefined : '${' + value + '}';
    }
    get parameters() {
        return this.options.parameters || {};
    }
}
exports.CfnParser = CfnParser;
//# sourceMappingURL=data:application/json;base64,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