import { Stack } from './stack';
/**
 * @experimental
 */
export interface ArnComponents {
    /**
     * (experimental) The partition that the resource is in.
     *
     * For standard AWS regions, the
     * partition is aws. If you have resources in other partitions, the
     * partition is aws-partitionname. For example, the partition for resources
     * in the China (Beijing) region is aws-cn.
     *
     * @default The AWS partition the stack is deployed to.
     * @experimental
     */
    readonly partition?: string;
    /**
     * (experimental) The service namespace that identifies the AWS product (for example, 's3', 'iam', 'codepipline').
     *
     * @experimental
     */
    readonly service: string;
    /**
     * (experimental) The region the resource resides in.
     *
     * Note that the ARNs for some resources
     * do not require a region, so this component might be omitted.
     *
     * @default The region the stack is deployed to.
     * @experimental
     */
    readonly region?: string;
    /**
     * (experimental) The ID of the AWS account that owns the resource, without the hyphens.
     *
     * For example, 123456789012. Note that the ARNs for some resources don't
     * require an account number, so this component might be omitted.
     *
     * @default The account the stack is deployed to.
     * @experimental
     */
    readonly account?: string;
    /**
     * (experimental) Resource type (e.g. "table", "autoScalingGroup", "certificate"). For some resource types, e.g. S3 buckets, this field defines the bucket name.
     *
     * @experimental
     */
    readonly resource: string;
    /**
     * (experimental) Separator between resource type and the resource.
     *
     * Can be either '/', ':' or an empty string. Will only be used if resourceName is defined.
     *
     * @default '/'
     * @experimental
     */
    readonly sep?: string;
    /**
     * (experimental) Resource name or path within the resource (i.e. S3 bucket object key) or a wildcard such as ``"*"``. This is service-dependent.
     *
     * @experimental
     */
    readonly resourceName?: string;
}
/**
 * @experimental
 */
export declare class Arn {
    /**
     * (experimental) Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *    arn:{partition}:{service}:{region}:{account}:{resource}{sep}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     *
     * @experimental
     */
    static format(components: ArnComponents, stack: Stack): string;
    /**
     * (experimental) Given an ARN, parses it and returns components.
     *
     * IF THE ARN IS A CONCRETE STRING...
     *
     * ...it will be parsed and validated. The separator (`sep`) will be set to '/'
     * if the 6th component includes a '/', in which case, `resource` will be set
     * to the value before the '/' and `resourceName` will be the rest. In case
     * there is no '/', `resource` will be set to the 6th components and
     * `resourceName` will be set to the rest of the string.
     *
     * IF THE ARN IS A TOKEN...
     *
     * ...it cannot be validated, since we don't have the actual value yet at the
     * time of this function call. You will have to supply `sepIfToken` and
     * whether or not ARNs of the expected format usually have resource names
     * in order to parse it properly. The resulting `ArnComponents` object will
     * contain tokens for the subexpressions of the ARN, not string literals.
     *
     * If the resource name could possibly contain the separator char, the actual
     * resource name cannot be properly parsed. This only occurs if the separator
     * char is '/', and happens for example for S3 object ARNs, IAM Role ARNs,
     * IAM OIDC Provider ARNs, etc. To properly extract the resource name from a
     * Tokenized ARN, you must know the resource type and call
     * `Arn.extractResourceName`.
     *
     * @param arn The ARN to parse.
     * @param sepIfToken The separator used to separate resource from resourceName.
     * @param hasName Whether there is a name component in the ARN at all.
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     * @experimental
     */
    static parse(arn: string, sepIfToken?: string, hasName?: boolean): ArnComponents;
    /**
     * (experimental) Extract the full resource name from an ARN.
     *
     * Necessary for resource names (paths) that may contain the separator, like
     * `arn:aws:iam::111111111111:role/path/to/role/name`.
     *
     * Only works if we statically know the expected `resourceType` beforehand, since we're going
     * to use that to split the string on ':<resourceType>/' (and take the right-hand side).
     *
     * We can't extract the 'resourceType' from the ARN at hand, because CloudFormation Expressions
     * only allow literals in the 'separator' argument to `{ Fn::Split }`, and so it can't be
     * `{ Fn::Select: [5, { Fn::Split: [':', ARN] }}`.
     *
     * Only necessary for ARN formats for which the type-name separator is `/`.
     *
     * @experimental
     */
    static extractResourceName(arn: string, resourceType: string): string;
    private constructor();
}
