"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deserialize = exports.serialize = void 0;
const yaml = require("yaml");
const yaml_types = require("yaml/types");
/**
 * Serializes the given data structure into valid YAML.
 *
 * @param obj the data structure to serialize
 * @returns a string containing the YAML representation of {@param obj}
 */
function serialize(obj) {
    const oldFold = yaml_types.strOptions.fold.lineWidth;
    try {
        yaml_types.strOptions.fold.lineWidth = 0;
        return yaml.stringify(obj, { schema: 'yaml-1.1' });
    }
    finally {
        yaml_types.strOptions.fold.lineWidth = oldFold;
    }
}
exports.serialize = serialize;
/**
 * Deserialize the YAML into the appropriate data structure.
 *
 * @param str the string containing YAML
 * @returns the data structure the YAML represents
 *   (most often in case of CloudFormation, an object)
 */
function deserialize(str) {
    return parseYamlStrWithCfnTags(str);
}
exports.deserialize = deserialize;
function makeTagForCfnIntrinsic(intrinsicName, addFnPrefix) {
    return {
        identify(value) { return typeof value === 'string'; },
        tag: `!${intrinsicName}`,
        resolve: (_doc, cstNode) => {
            const ret = {};
            ret[addFnPrefix ? `Fn::${intrinsicName}` : intrinsicName] =
                // the +1 is to account for the ! the short form begins with
                parseYamlStrWithCfnTags(cstNode.toString().substring(intrinsicName.length + 1));
            return ret;
        },
    };
}
const shortForms = [
    'Base64', 'Cidr', 'FindInMap', 'GetAZs', 'ImportValue', 'Join', 'Sub',
    'Select', 'Split', 'Transform', 'And', 'Equals', 'If', 'Not', 'Or', 'GetAtt',
].map(name => makeTagForCfnIntrinsic(name, true)).concat(makeTagForCfnIntrinsic('Ref', false), makeTagForCfnIntrinsic('Condition', false));
function parseYamlStrWithCfnTags(text) {
    return yaml.parse(text, {
        customTags: shortForms,
        schema: 'core',
    });
}
//# sourceMappingURL=data:application/json;base64,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