"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_parse = require("../../core/lib/cfn-parse"); // Automatically re-written from '@aws-cdk/core/lib/cfn-parse'
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * (experimental) Construct to import an existing CloudFormation template file into a CDK application.
 *
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 *
 * @experimental
 */
class CfnInclude extends core.CfnElement {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        this.parameters = {};
        this.mappings = {};
        this.rules = {};
        this.hooks = {};
        this.outputs = {};
        this.nestedStacks = {};
        this.parametersToReplace = props.parameters || {};
        // read the template into a JS object
        this.template = futils.readYamlSync(props.templateFile);
        this.preserveLogicalIds = (_b = props.preserveLogicalIds) !== null && _b !== void 0 ? _b : true;
        // check if all user specified parameter values exist in the template
        for (const logicalId of Object.keys(this.parametersToReplace)) {
            if (!(logicalId in (this.template.Parameters || {}))) {
                throw new Error(`Parameter with logical ID '${logicalId}' was not found in the template`);
            }
        }
        // instantiate the Mappings
        this.mappingsScope = new core_1.Construct(this, '$Mappings');
        for (const mappingName of Object.keys(this.template.Mappings || {})) {
            this.createMapping(mappingName);
        }
        // instantiate all parameters
        for (const logicalId of Object.keys(this.template.Parameters || {})) {
            this.createParameter(logicalId);
        }
        // instantiate the conditions
        this.conditionsScope = new core_1.Construct(this, '$Conditions');
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.getOrCreateCondition(conditionName);
        }
        // instantiate the rules
        this.rulesScope = new core_1.Construct(this, '$Rules');
        for (const ruleName of Object.keys(this.template.Rules || {})) {
            this.createRule(ruleName);
        }
        this.nestedStacksToInclude = props.loadNestedStacks || {};
        // instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
        // verify that all nestedStacks have been instantiated
        for (const nestedStackId of Object.keys(props.loadNestedStacks || {})) {
            if (!(nestedStackId in this.resources)) {
                throw new Error(`Nested Stack with logical ID '${nestedStackId}' was not found in the template`);
            }
        }
        // instantiate the Hooks
        this.hooksScope = new core_1.Construct(this, '$Hooks');
        for (const hookName of Object.keys(this.template.Hooks || {})) {
            this.createHook(hookName);
        }
        const outputScope = new core_1.Construct(this, '$Ouputs');
        for (const logicalId of Object.keys(this.template.Outputs || {})) {
            this.createOutput(logicalId, outputScope);
        }
    }
    /**
     * (experimental) Returns the low-level CfnResource from the template with the given logical ID.
     *
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *      // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *      const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *      // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file.
     * @experimental
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnCondition object from the 'Conditions' section of the CloudFormation template with the given name.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file.
     * @experimental
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnParameter object from the 'Parameters' section of the included template.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Parameter with the given name is not present in the template,
     * throws an exception.
     *
     * @param parameterName the name of the parameter to retrieve.
     * @experimental
     */
    getParameter(parameterName) {
        const ret = this.parameters[parameterName];
        if (!ret) {
            throw new Error(`Parameter with name '${parameterName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnMapping object from the 'Mappings' section of the included template.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Mapping with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param mappingName the name of the Mapping in the template to retrieve.
     * @experimental
     */
    getMapping(mappingName) {
        const ret = this.mappings[mappingName];
        if (!ret) {
            throw new Error(`Mapping with name '${mappingName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnOutput object from the 'Outputs' section of the included template.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If an Output with the given name is not present in the template,
     * throws an exception.
     *
     * @param logicalId the name of the output to retrieve.
     * @experimental
     */
    getOutput(logicalId) {
        const ret = this.outputs[logicalId];
        if (!ret) {
            throw new Error(`Output with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnRule object from the 'Rules' section of the CloudFormation template with the given name.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Rule with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param ruleName the name of the Rule in the CloudFormation template.
     * @experimental
     */
    getRule(ruleName) {
        const ret = this.rules[ruleName];
        if (!ret) {
            throw new Error(`Rule with name '${ruleName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnHook object from the 'Hooks' section of the included CloudFormation template with the given logical ID.
     *
     * Any modifications performed on the returned object will be reflected in the resulting CDK template.
     *
     * If a Hook with the given logical ID is not present in the template,
     * an exception will be thrown.
     *
     * @param hookLogicalId the logical ID of the Hook in the included CloudFormation template's 'Hooks' section.
     * @experimental
     */
    getHook(hookLogicalId) {
        const ret = this.hooks[hookLogicalId];
        if (!ret) {
            throw new Error(`Hook with logical ID '${hookLogicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns a loaded NestedStack with name logicalId.
     *
     * For a nested stack to be returned by this method,
     * it must be specified either in the {@link CfnIncludeProps.loadNestedStacks} property,
     * or through the {@link loadNestedStack} method.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template.
     * @experimental
     */
    getNestedStack(logicalId) {
        if (!this.nestedStacks[logicalId]) {
            if (!this.template.Resources[logicalId]) {
                throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
            }
            else if (this.template.Resources[logicalId].Type !== 'AWS::CloudFormation::Stack') {
                throw new Error(`Resource with logical ID '${logicalId}' is not a CloudFormation Stack`);
            }
            else {
                throw new Error(`Nested Stack '${logicalId}' was not included in the parent template. ` +
                    'To retrieve an included nested stack, it must be specified either in the `loadNestedStacks` property, or through the `loadNestedStack` method');
            }
        }
        return this.nestedStacks[logicalId];
    }
    /**
     * (experimental) Includes a template for a child stack inside of this parent template.
     *
     * A child with this logical ID must exist in the template,
     * and be of type AWS::CloudFormation::Stack.
     * This is equivalent to specifying the value in the {@link CfnIncludeProps.loadNestedStacks}
     * property on object construction.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template.
     * @param nestedStackProps the properties of the included child Stack.
     * @returns the same {@link IncludedNestedStack} object that {@link getNestedStack} returns for this logical ID
     * @experimental
     */
    loadNestedStack(logicalId, nestedStackProps) {
        if (logicalId in this.nestedStacks) {
            throw new Error(`Nested Stack '${logicalId}' was already included in its parent template`);
        }
        const cfnStack = this.resources[logicalId];
        if (!cfnStack) {
            throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
        }
        if (cfnStack instanceof core.CfnStack) {
            // delete the old CfnStack child - one will be created by the NestedStack object
            this.node.tryRemoveChild(logicalId);
            // remove the previously created CfnStack resource from the resources map
            delete this.resources[logicalId];
            // createNestedStack() (called by getOrCreateResource()) expects this to be filled
            this.nestedStacksToInclude[logicalId] = nestedStackProps;
            this.getOrCreateResource(logicalId);
            return this.nestedStacks[logicalId];
        }
        else {
            throw new Error(`Nested Stack with logical ID '${logicalId}' is not an AWS::CloudFormation::Stack resource`);
        }
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            const self = this;
            const finder = {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _b;
                    return (_b = self.resources[elementName]) !== null && _b !== void 0 ? _b : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            };
            const cfnParser = new cfn_parse.CfnParser({
                finder,
                parameters: this.parametersToReplace,
            });
            switch (section) {
                case 'Conditions':
                case 'Mappings':
                case 'Resources':
                case 'Parameters':
                case 'Rules':
                case 'Hooks':
                case 'Outputs':
                    // these are rendered as a side effect of instantiating the L1s
                    break;
                default:
                    ret[section] = cfnParser.parseValue(this.template[section]);
            }
        }
        return ret;
    }
    createMapping(mappingName) {
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findCondition() { throw new Error('Referring to Conditions in Mapping definitions is not allowed'); },
                findMapping() { throw new Error('Referring to other Mappings in Mapping definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Mapping definitions is not allowed'); },
                findResource() { throw new Error('Using GetAtt expressions in Mapping definitions is not allowed'); },
            },
            parameters: {},
        });
        const cfnMapping = new core.CfnMapping(this.mappingsScope, mappingName, {
            mapping: cfnParser.parseValue(this.template.Mappings[mappingName]),
        });
        this.mappings[mappingName] = cfnMapping;
        this.overrideLogicalIdIfNeeded(cfnMapping, mappingName);
    }
    createParameter(logicalId) {
        if (logicalId in this.parametersToReplace) {
            return;
        }
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt expressions in Parameter definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Parameter definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Parameter definitions is not allowed'); },
                findMapping() { throw new Error('Referring to Mappings in Parameter definitions is not allowed'); },
            },
            parameters: {},
        }).parseValue(this.template.Parameters[logicalId]);
        const cfnParameter = new core.CfnParameter(this, logicalId, {
            type: expression.Type,
            default: expression.Default,
            allowedPattern: expression.AllowedPattern,
            allowedValues: expression.AllowedValues,
            constraintDescription: expression.ConstraintDescription,
            description: expression.Description,
            maxLength: expression.MaxLength,
            maxValue: expression.MaxValue,
            minLength: expression.MinLength,
            minValue: expression.MinValue,
            noEcho: expression.NoEcho,
        });
        this.overrideLogicalIdIfNeeded(cfnParameter, logicalId);
        this.parameters[logicalId] = cfnParameter;
    }
    createRule(ruleName) {
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findRefTarget(refTarget) {
                    // only parameters can be referenced in Rules
                    return self.parameters[refTarget];
                },
                findResource() { throw new Error('Using GetAtt expressions in Rule definitions is not allowed'); },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
            context: cfn_parse.CfnParsingContext.RULES,
        });
        const ruleProperties = cfnParser.parseValue(this.template.Rules[ruleName]);
        const rule = new core.CfnRule(this.rulesScope, ruleName, {
            ruleCondition: ruleProperties.RuleCondition,
            assertions: ruleProperties.Assertions,
        });
        this.rules[ruleName] = rule;
        this.overrideLogicalIdIfNeeded(rule, ruleName);
    }
    createHook(hookName) {
        var _b;
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _b;
                    return (_b = self.resources[elementName]) !== null && _b !== void 0 ? _b : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        });
        const hookAttributes = this.template.Hooks[hookName];
        let hook;
        switch (hookAttributes.Type) {
            case 'AWS::CodeDeploy::BlueGreen':
                hook = core.CfnCodeDeployBlueGreenHook._fromCloudFormation(this.hooksScope, hookName, hookAttributes, {
                    parser: cfnParser,
                });
                break;
            default: {
                const hookProperties = (_b = cfnParser.parseValue(hookAttributes.Properties)) !== null && _b !== void 0 ? _b : {};
                hook = new core.CfnHook(this.hooksScope, hookName, {
                    type: hookAttributes.Type,
                    properties: hookProperties,
                });
            }
        }
        this.hooks[hookName] = hook;
        this.overrideLogicalIdIfNeeded(hook, hookName);
    }
    createOutput(logicalId, scope) {
        const self = this;
        const outputAttributes = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _b;
                    return (_b = self.resources[elementName]) !== null && _b !== void 0 ? _b : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        }).parseValue(this.template.Outputs[logicalId]);
        const cfnOutput = new core.CfnOutput(scope, logicalId, {
            value: outputAttributes.Value,
            description: outputAttributes.Description,
            exportName: outputAttributes.Export ? outputAttributes.Export.Name : undefined,
            condition: (() => {
                if (!outputAttributes.Condition) {
                    return undefined;
                }
                else if (this.conditions[outputAttributes.Condition]) {
                    return self.getCondition(outputAttributes.Condition);
                }
                throw new Error(`Output with name '${logicalId}' refers to a Condition with name ` +
                    `'${outputAttributes.Condition}' which was not found in this template`);
            })(),
        });
        this.overrideLogicalIdIfNeeded(cfnOutput, logicalId);
        this.outputs[logicalId] = cfnOutput;
    }
    getOrCreateCondition(conditionName) {
        if (conditionName in this.conditions) {
            return this.conditions[conditionName];
        }
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt in Condition definitions is not allowed'); },
                findRefTarget(elementName) {
                    // only Parameters can be referenced in the 'Conditions' section
                    return self.parameters[elementName];
                },
                findCondition(cName) {
                    return cName in (self.template.Conditions || {})
                        ? self.getOrCreateCondition(cName)
                        : undefined;
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            context: cfn_parse.CfnParsingContext.CONDITIONS,
            parameters: this.parametersToReplace,
        });
        const cfnCondition = new core.CfnCondition(this.conditionsScope, conditionName, {
            expression: cfnParser.parseValue(this.template.Conditions[conditionName]),
        });
        this.overrideLogicalIdIfNeeded(cfnCondition, conditionName);
        this.conditions[conditionName] = cfnCondition;
        return cfnCondition;
    }
    getOrCreateResource(logicalId) {
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        // fail early for resource attributes we don't support yet
        const knownAttributes = [
            'Condition', 'DependsOn', 'Description', 'Metadata', 'Properties', 'Type', 'Version',
            'CreationPolicy', 'DeletionPolicy', 'UpdatePolicy', 'UpdateReplacePolicy',
        ];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The '${attribute}' resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findMapping(mappingName) {
                return self.mappings[mappingName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId);
            },
            findRefTarget(elementName) {
                if (elementName in self.parameters) {
                    return self.parameters[elementName];
                }
                return this.findResource(elementName);
            },
        };
        const cfnParser = new cfn_parse.CfnParser({
            finder,
            parameters: this.parametersToReplace,
        });
        let l1Instance;
        if (this.nestedStacksToInclude[logicalId]) {
            l1Instance = this.createNestedStack(logicalId, cfnParser);
        }
        else {
            const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
            // The AWS::CloudFormation::CustomResource type corresponds to the CfnCustomResource class.
            // Unfortunately, it's quite useless; it only has a single property, ServiceToken.
            // For that reason, even the CustomResource class from @core doesn't use it!
            // So, special-case the handling of this one resource type
            if (l1ClassFqn && resourceAttributes.Type !== 'AWS::CloudFormation::CustomResource') {
                const options = {
                    parser: cfnParser,
                };
                const [moduleName, ...className] = l1ClassFqn.split('.');
                const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
                const jsClassFromModule = module[className.join('.')];
                l1Instance = jsClassFromModule._fromCloudFormation(this, logicalId, resourceAttributes, options);
            }
            else {
                l1Instance = new core.CfnResource(this, logicalId, {
                    type: resourceAttributes.Type,
                    properties: cfnParser.parseValue(resourceAttributes.Properties),
                });
                cfnParser.handleAttributes(l1Instance, resourceAttributes, logicalId);
            }
        }
        this.overrideLogicalIdIfNeeded(l1Instance, logicalId);
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
    createNestedStack(nestedStackId, cfnParser) {
        const templateResources = this.template.Resources || {};
        const nestedStackAttributes = templateResources[nestedStackId] || {};
        if (nestedStackAttributes.Type !== 'AWS::CloudFormation::Stack') {
            throw new Error(`Nested Stack with logical ID '${nestedStackId}' is not an AWS::CloudFormation::Stack resource`);
        }
        if (nestedStackAttributes.CreationPolicy) {
            throw new Error('CreationPolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        if (nestedStackAttributes.UpdatePolicy) {
            throw new Error('UpdatePolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        const nestedStackProps = cfnParser.parseValue(nestedStackAttributes.Properties);
        const nestedStack = new core.NestedStack(this, nestedStackId, {
            parameters: this.parametersForNestedStack(nestedStackProps.Parameters, nestedStackId),
            notificationArns: nestedStackProps.NotificationArns,
            timeout: nestedStackProps.Timeout,
        });
        const template = new CfnInclude(nestedStack, nestedStackId, this.nestedStacksToInclude[nestedStackId]);
        this.nestedStacks[nestedStackId] = { stack: nestedStack, includedTemplate: template };
        // we know this is never undefined for nested stacks
        const nestedStackResource = nestedStack.nestedStackResource;
        cfnParser.handleAttributes(nestedStackResource, nestedStackAttributes, nestedStackId);
        return nestedStackResource;
    }
    parametersForNestedStack(parameters, nestedStackId) {
        var _b;
        if (parameters == null) {
            return undefined;
        }
        const parametersToReplace = (_b = this.nestedStacksToInclude[nestedStackId].parameters) !== null && _b !== void 0 ? _b : {};
        const ret = {};
        for (const paramName of Object.keys(parameters)) {
            if (!(paramName in parametersToReplace)) {
                ret[paramName] = parameters[paramName];
            }
        }
        return ret;
    }
    overrideLogicalIdIfNeeded(element, id) {
        if (this.preserveLogicalIds) {
            element.overrideLogicalId(id);
        }
    }
}
exports.CfnInclude = CfnInclude;
_a = JSII_RTTI_SYMBOL_1;
CfnInclude[_a] = { fqn: "monocdk.cloudformation_include.CfnInclude", version: "1.106.1" };
//# sourceMappingURL=data:application/json;base64,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