"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Code = exports.InlineCode = exports.AssetCode = exports.Code = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
/**
 * (experimental) The code the canary should execute.
 *
 * @experimental
 */
class Code {
    /**
     * (experimental) Specify code inline.
     *
     * @param code The actual handler code (limited to 4KiB).
     * @returns `InlineCode` with inline code.
     * @experimental
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * (experimental) Specify code from a local path.
     *
     * Path must include the folder structure `nodejs/node_modules/myCanaryFilename.js`.
     *
     * @param assetPath Either a directory or a .zip file.
     * @returns `AssetCode` associated with the specified path.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     * @experimental
     */
    static fromAsset(assetPath, options) {
        return new AssetCode(assetPath, options);
    }
    /**
     * (experimental) Specify code from an s3 bucket.
     *
     * The object in the s3 bucket must be a .zip file that contains
     * the structure `nodejs/node_modules/myCanaryFilename.js`.
     *
     * @param bucket The S3 bucket.
     * @param key The object key.
     * @param objectVersion Optional S3 object version.
     * @returns `S3Code` associated with the specified S3 object.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     * @experimental
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
}
exports.Code = Code;
_a = JSII_RTTI_SYMBOL_1;
Code[_a] = { fqn: "monocdk.aws_synthetics.Code", version: "1.106.1" };
/**
 * (experimental) Canary code from an Asset.
 *
 * @experimental
 */
class AssetCode extends Code {
    /**
     * @param assetPath The path to the asset file or directory.
     * @experimental
     */
    constructor(assetPath, options) {
        super();
        this.assetPath = assetPath;
        this.options = options;
        if (!fs.existsSync(this.assetPath)) {
            throw new Error(`${this.assetPath} is not a valid path`);
        }
    }
    /**
     * (experimental) Called when the canary is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(scope, handler) {
        this.validateCanaryAsset(handler);
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.assetPath,
                ...this.options,
            });
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    /**
     * Validates requirements specified by the canary resource. For example, the canary code with handler `index.handler`
     * must be found in the file structure `nodejs/node_modules/index.js`.
     *
     * Requires path to be either zip file or directory.
     * Requires asset directory to have the structure 'nodejs/node_modules'.
     * Requires canary file to be directly inside node_modules folder.
     * Requires canary file name matches the handler name.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html
     *
     * @param handler the canary handler
     */
    validateCanaryAsset(handler) {
        if (path.extname(this.assetPath) !== '.zip') {
            if (!fs.lstatSync(this.assetPath).isDirectory()) {
                throw new Error(`Asset must be a .zip file or a directory (${this.assetPath})`);
            }
            const filename = `${handler.split('.')[0]}.js`;
            if (!fs.existsSync(path.join(this.assetPath, 'nodejs', 'node_modules', filename))) {
                throw new Error(`The canary resource requires that the handler is present at "nodejs/node_modules/${filename}" but not found at ${this.assetPath} (https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch)`);
            }
        }
    }
}
exports.AssetCode = AssetCode;
_b = JSII_RTTI_SYMBOL_1;
AssetCode[_b] = { fqn: "monocdk.aws_synthetics.AssetCode", version: "1.106.1" };
/**
 * (experimental) Canary code from an inline string.
 *
 * @experimental
 */
class InlineCode extends Code {
    /**
     * @experimental
     */
    constructor(code) {
        super();
        this.code = code;
        if (code.length === 0) {
            throw new Error('Canary inline code cannot be empty');
        }
    }
    /**
     * (experimental) Called when the canary is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(_scope, handler) {
        if (handler !== 'index.handler') {
            throw new Error(`The handler for inline code must be "index.handler" (got "${handler}")`);
        }
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
_c = JSII_RTTI_SYMBOL_1;
InlineCode[_c] = { fqn: "monocdk.aws_synthetics.InlineCode", version: "1.106.1" };
/**
 * (experimental) S3 bucket path to the code zip file.
 *
 * @experimental
 */
class S3Code extends Code {
    /**
     * @experimental
     */
    constructor(bucket, key, objectVersion) {
        super();
        this.bucket = bucket;
        this.key = key;
        this.objectVersion = objectVersion;
    }
    /**
     * (experimental) Called when the canary is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(_scope, _handler) {
        return {
            s3Location: {
                bucketName: this.bucket.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
_d = JSII_RTTI_SYMBOL_1;
S3Code[_d] = { fqn: "monocdk.aws_synthetics.S3Code", version: "1.106.1" };
//# sourceMappingURL=data:application/json;base64,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