"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateGraph = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * (experimental) A collection of connected states.
 *
 * A StateGraph is used to keep track of all states that are connected (have
 * transitions between them). It does not include the substatemachines in
 * a Parallel's branches: those are their own StateGraphs, but the graphs
 * themselves have a hierarchical relationship as well.
 *
 * By assigning states to a definitive StateGraph, we verify that no state
 * machines are constructed. In particular:
 *
 * - Every state object can only ever be in 1 StateGraph, and not inadvertently
 *    be used in two graphs.
 * - Every stateId must be unique across all states in the entire state
 *    machine.
 *
 * All policy statements in all states in all substatemachines are bubbled so
 * that the top-level StateMachine instantiation can read them all and add
 * them to the IAM Role.
 *
 * You do not need to instantiate this class; it is used internally.
 *
 * @experimental
 */
class StateGraph {
    /**
     * @param startState state that gets executed when the state machine is launched.
     * @param graphDescription description of the state machine.
     * @experimental
     */
    constructor(startState, graphDescription) {
        this.startState = startState;
        this.graphDescription = graphDescription;
        /**
         * (experimental) The accumulated policy statements.
         *
         * @experimental
         */
        this.policyStatements = new Array();
        /**
         * All states in this graph
         */
        this.allStates = new Set();
        /**
         * A mapping of stateId -> Graph for all states in this graph and subgraphs
         */
        this.allContainedStates = new Map();
        this.allStates.add(startState);
        startState.bindToGraph(this);
    }
    /**
     * (experimental) Register a state as part of this graph.
     *
     * Called by State.bindToGraph().
     *
     * @experimental
     */
    registerState(state) {
        this.registerContainedState(state.stateId, this);
        this.allStates.add(state);
    }
    /**
     * (experimental) Register a Policy Statement used by states in this graph.
     *
     * @experimental
     */
    registerPolicyStatement(statement) {
        if (this.superGraph) {
            this.superGraph.registerPolicyStatement(statement);
        }
        else {
            this.policyStatements.push(statement);
        }
    }
    /**
     * (experimental) Register this graph as a child of the given graph.
     *
     * Resource changes will be bubbled up to the given graph.
     *
     * @experimental
     */
    registerSuperGraph(graph) {
        if (this.superGraph === graph) {
            return;
        }
        if (this.superGraph) {
            throw new Error('Every StateGraph can only be registered into one other StateGraph');
        }
        this.superGraph = graph;
        this.pushContainedStatesUp(graph);
        this.pushPolicyStatementsUp(graph);
    }
    /**
     * (experimental) Return the Amazon States Language JSON for this graph.
     *
     * @experimental
     */
    toGraphJson() {
        const states = {};
        for (const state of this.allStates) {
            states[state.stateId] = state.toStateJson();
        }
        return {
            StartAt: this.startState.stateId,
            States: states,
            TimeoutSeconds: this.timeout && this.timeout.toSeconds(),
        };
    }
    /**
     * (experimental) Return a string description of this graph.
     *
     * @experimental
     */
    toString() {
        const someNodes = Array.from(this.allStates).slice(0, 3).map(x => x.stateId);
        if (this.allStates.size > 3) {
            someNodes.push('...');
        }
        return `${this.graphDescription} (${someNodes.join(', ')})`;
    }
    /**
     * Register a stateId and graph where it was registered
     */
    registerContainedState(stateId, graph) {
        if (this.superGraph) {
            this.superGraph.registerContainedState(stateId, graph);
        }
        else {
            const existingGraph = this.allContainedStates.get(stateId);
            if (existingGraph) {
                throw new Error(`State with name '${stateId}' occurs in both ${graph} and ${existingGraph}. All states must have unique names.`);
            }
            this.allContainedStates.set(stateId, graph);
        }
    }
    /**
     * Push all contained state info up to the given super graph
     */
    pushContainedStatesUp(superGraph) {
        for (const [stateId, graph] of this.allContainedStates) {
            superGraph.registerContainedState(stateId, graph);
        }
    }
    /**
     * Push all policy statements to into the given super graph
     */
    pushPolicyStatementsUp(superGraph) {
        for (const policyStatement of this.policyStatements) {
            superGraph.registerPolicyStatement(policyStatement);
        }
    }
}
exports.StateGraph = StateGraph;
_a = JSII_RTTI_SYMBOL_1;
StateGraph[_a] = { fqn: "monocdk.aws_stepfunctions.StateGraph", version: "1.106.1" };
//# sourceMappingURL=data:application/json;base64,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