"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.jsonPathString = exports.recurseObject = exports.findReferencedPaths = exports.renderObject = exports.JsonPathToken = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const JSON_PATH_TOKEN_SYMBOL = Symbol.for('@aws-cdk/aws-stepfunctions.JsonPathToken');
class JsonPathToken {
    constructor(path) {
        this.path = path;
        this.creationStack = core_1.captureStackTrace();
        this.displayHint = path.replace(/^[^a-zA-Z]+/, '');
        Object.defineProperty(this, JSON_PATH_TOKEN_SYMBOL, { value: true });
    }
    static isJsonPathToken(x) {
        return x[JSON_PATH_TOKEN_SYMBOL] === true;
    }
    resolve(_ctx) {
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.JsonPathToken = JsonPathToken;
/**
 * Deep render a JSON object to expand JSON path fields, updating the key to end in '.$'
 */
function renderObject(obj) {
    return recurseObject(obj, {
        handleString: renderString,
        handleList: renderStringList,
        handleNumber: renderNumber,
        handleBoolean: renderBoolean,
    });
}
exports.renderObject = renderObject;
/**
 * Return all JSON paths that are used in the given structure
 */
function findReferencedPaths(obj) {
    const found = new Set();
    recurseObject(obj, {
        handleString(_key, x) {
            const path = jsonPathString(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleList(_key, x) {
            const path = jsonPathStringList(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleNumber(_key, x) {
            const path = jsonPathNumber(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleBoolean(_key, _x) {
            return {};
        },
    });
    return found;
}
exports.findReferencedPaths = findReferencedPaths;
function recurseObject(obj, handlers, visited = []) {
    if (obj === undefined) {
        return undefined;
    }
    if (visited.includes(obj)) {
        return {};
    }
    else {
        visited.push(obj);
    }
    const ret = {};
    for (const [key, value] of Object.entries(obj)) {
        if (typeof value === 'string') {
            Object.assign(ret, handlers.handleString(key, value));
        }
        else if (typeof value === 'number') {
            Object.assign(ret, handlers.handleNumber(key, value));
        }
        else if (Array.isArray(value)) {
            Object.assign(ret, recurseArray(key, value, handlers, visited));
        }
        else if (typeof value === 'boolean') {
            Object.assign(ret, handlers.handleBoolean(key, value));
        }
        else if (value === null || value === undefined) {
            // Nothing
        }
        else if (typeof value === 'object') {
            ret[key] = recurseObject(value, handlers, visited);
        }
    }
    return ret;
}
exports.recurseObject = recurseObject;
/**
 * Render an array that may or may not contain a string list token
 */
function recurseArray(key, arr, handlers, visited = []) {
    if (isStringArray(arr)) {
        const path = jsonPathStringList(arr);
        if (path !== undefined) {
            return handlers.handleList(key, arr);
        }
        // Fall through to correctly reject encoded strings inside an array.
        // They cannot be represented because there is no key to append a '.$' to.
    }
    return {
        [key]: arr.map(value => {
            if ((typeof value === 'string' && jsonPathString(value) !== undefined)
                || (typeof value === 'number' && jsonPathNumber(value) !== undefined)
                || (isStringArray(value) && jsonPathStringList(value) !== undefined)) {
                throw new Error('Cannot use JsonPath fields in an array, they must be used in objects');
            }
            if (typeof value === 'object' && value !== null) {
                return recurseObject(value, handlers, visited);
            }
            return value;
        }),
    };
}
function isStringArray(x) {
    return Array.isArray(x) && x.every(el => typeof el === 'string');
}
/**
 * Render a parameter string
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderString(key, value) {
    const path = jsonPathString(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter string list
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderStringList(key, value) {
    const path = jsonPathStringList(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter number
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderNumber(key, value) {
    const path = jsonPathNumber(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter boolean
 */
function renderBoolean(key, value) {
    return { [key]: value };
}
/**
 * If the indicated string is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathString(x) {
    const fragments = core_1.Tokenization.reverseString(x);
    const jsonPathTokens = fragments.tokens.filter(JsonPathToken.isJsonPathToken);
    if (jsonPathTokens.length > 0 && fragments.length > 1) {
        throw new Error(`Field references must be the entire string, cannot concatenate them (found '${x}')`);
    }
    if (jsonPathTokens.length > 0) {
        return jsonPathTokens[0].path;
    }
    return undefined;
}
exports.jsonPathString = jsonPathString;
/**
 * If the indicated string list is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathStringList(x) {
    return pathFromToken(core_1.Tokenization.reverseList(x));
}
/**
 * If the indicated number is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathNumber(x) {
    return pathFromToken(core_1.Tokenization.reverseNumber(x));
}
function pathFromToken(token) {
    return token && (JsonPathToken.isJsonPathToken(token) ? token.path : undefined);
}
//# sourceMappingURL=data:application/json;base64,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