"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateModel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * (experimental) A Step Functions Task to create a SageMaker model.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-sagemaker.html
 * @experimental
 */
class SageMakerCreateModel extends sfn.TaskStateBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        /**
         * (experimental) Allows specify security group connections for instances of this fleet.
         *
         * @experimental
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateModel.SUPPORTED_INTEGRATION_PATTERNS);
        // add the security groups to the connections object
        if (props.vpc) {
            this.vpc = props.vpc;
            this.subnets = props.subnetSelection ? this.vpc.selectSubnets(props.subnetSelection).subnetIds : this.vpc.selectSubnets().subnetIds;
        }
        this.role = this.props.role || this.createSagemakerRole();
        this.grantPrincipal = this.role;
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * (experimental) Add the security group to all instances via the launch configuration security groups array.
     *
     * @param securityGroup : The security group to add.
     * @experimental
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createModel', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    renderParameters() {
        var _b, _c;
        return {
            EnableNetworkIsolation: this.props.enableNetworkIsolation,
            ExecutionRoleArn: this.role.roleArn,
            ModelName: this.props.modelName,
            Tags: (_b = this.props.tags) === null || _b === void 0 ? void 0 : _b.value,
            PrimaryContainer: this.props.primaryContainer.bind(this).parameters,
            Containers: (_c = this.props.containers) === null || _c === void 0 ? void 0 : _c.map(container => (container.bind(this))),
            ...this.renderVpcConfig(),
        };
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        return [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateModel'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'model',
                        // If the model name comes from input, we cannot target the policy to a particular ARN prefix reliably.
                        // SageMaker uses lowercase for resource name in the arn
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.modelName) ? '*' : `${this.props.modelName.toLowerCase()}*`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
    }
    createSagemakerRole() {
        // https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-roles.html
        const role = new iam.Role(this, 'SagemakerRole', {
            assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            inlinePolicies: {
                CreateModel: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatch.html
                                'cloudwatch:PutMetricData',
                                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatchlogs.html
                                'logs:CreateLogStream',
                                'logs:CreateLogGroup',
                                'logs:PutLogEvents',
                                'logs:DescribeLogStreams',
                                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerregistry.html
                                'ecr:GetAuthorizationToken',
                            ],
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        if (this.props.vpc) {
            role.addToPrincipalPolicy(
            // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
            new iam.PolicyStatement({
                actions: [
                    'ec2:CreateNetworkInterface',
                    'ec2:CreateNetworkInterfacePermission',
                    'ec2:DeleteNetworkInterface',
                    'ec2:DeleteNetworkInterfacePermission',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:DescribeVpcs',
                    'ec2:DescribeDhcpOptions',
                    'ec2:DescribeSubnets',
                    'ec2:DescribeSecurityGroups',
                ],
                resources: ['*'],
            }));
        }
        return role;
    }
    renderVpcConfig() {
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(this, 'ModelSecurityGroup', {
                vpc: this.vpc,
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        return this.vpc
            ? {
                VpcConfig: {
                    SecurityGroupIds: cdk.Lazy.list({ produce: () => this.securityGroups.map((sg) => sg.securityGroupId) }),
                    Subnets: this.subnets,
                },
            }
            : {};
    }
}
exports.SageMakerCreateModel = SageMakerCreateModel;
_a = JSII_RTTI_SYMBOL_1;
SageMakerCreateModel[_a] = { fqn: "monocdk.aws_stepfunctions_tasks.SageMakerCreateModel", version: "1.106.1" };
SageMakerCreateModel.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
];
//# sourceMappingURL=data:application/json;base64,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