"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BatchSubmitJob = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * (experimental) Task to submits an AWS Batch job from a job definition.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
 * @experimental
 */
class BatchSubmitJob extends sfn.TaskStateBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_b = props.integrationPattern) !== null && _b !== void 0 ? _b : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS);
        // validate arraySize limits
        core_1.withResolved(props.arraySize, (arraySize) => {
            if (arraySize !== undefined && (arraySize < 2 || arraySize > 10000)) {
                throw new Error(`arraySize must be between 2 and 10,000. Received ${arraySize}.`);
            }
        });
        // validate dependency size
        if (props.dependsOn && props.dependsOn.length > 20) {
            throw new Error(`dependencies must be 20 or less. Received ${props.dependsOn.length}.`);
        }
        // validate attempts
        core_1.withResolved(props.attempts, (attempts) => {
            if (attempts !== undefined && (attempts < 1 || attempts > 10)) {
                throw new Error(`attempts must be between 1 and 10. Received ${attempts}.`);
            }
        });
        // validate timeout
        props.timeout !== undefined && core_1.withResolved(props.timeout.toSeconds(), (timeout) => {
            if (timeout < 60) {
                throw new Error(`attempt duration must be greater than 60 seconds. Received ${timeout} seconds.`);
            }
        });
        // This is required since environment variables must not start with AWS_BATCH;
        // this naming convention is reserved for variables that are set by the AWS Batch service.
        if ((_c = props.containerOverrides) === null || _c === void 0 ? void 0 : _c.environment) {
            Object.keys(props.containerOverrides.environment).forEach(key => {
                if (key.match(/^AWS_BATCH/)) {
                    throw new Error(`Invalid environment variable name: ${key}. Environment variable names starting with 'AWS_BATCH' are reserved.`);
                }
            });
        }
        this.taskPolicies = this.configurePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        var _b;
        return {
            Resource: task_utils_1.integrationResourceArn('batch', 'submitJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                JobDefinition: this.props.jobDefinitionArn,
                JobName: this.props.jobName,
                JobQueue: this.props.jobQueueArn,
                Parameters: (_b = this.props.payload) === null || _b === void 0 ? void 0 : _b.value,
                ArrayProperties: this.props.arraySize !== undefined
                    ? { Size: this.props.arraySize }
                    : undefined,
                ContainerOverrides: this.props.containerOverrides
                    ? this.configureContainerOverrides(this.props.containerOverrides)
                    : undefined,
                DependsOn: this.props.dependsOn
                    ? this.props.dependsOn.map(jobDependency => ({
                        JobId: jobDependency.jobId,
                        Type: jobDependency.type,
                    }))
                    : undefined,
                RetryStrategy: this.props.attempts !== undefined
                    ? { Attempts: this.props.attempts }
                    : undefined,
                Timeout: this.props.timeout
                    ? { AttemptDurationSeconds: this.props.timeout.toSeconds() }
                    : undefined,
            }),
            TimeoutSeconds: undefined,
        };
    }
    configurePolicyStatements() {
        return [
            // Resource level access control for job-definition requires revision which batch does not support yet
            // Using the alternative permissions as mentioned here:
            // https://docs.aws.amazon.com/batch/latest/userguide/batch-supported-iam-actions-resources.html
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'batch',
                        resource: 'job-definition',
                        resourceName: '*',
                    }),
                    this.props.jobQueueArn,
                ],
                actions: ['batch:SubmitJob'],
            }),
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventsForBatchJobsRule',
                    }),
                ],
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            }),
        ];
    }
    configureContainerOverrides(containerOverrides) {
        var _b, _c;
        let environment;
        if (containerOverrides.environment) {
            environment = Object.entries(containerOverrides.environment).map(([key, value]) => ({
                Name: key,
                Value: value,
            }));
        }
        let resources;
        if (containerOverrides.gpuCount) {
            resources = [
                {
                    Type: 'GPU',
                    Value: `${containerOverrides.gpuCount}`,
                },
            ];
        }
        return {
            Command: containerOverrides.command,
            Environment: environment,
            InstanceType: (_b = containerOverrides.instanceType) === null || _b === void 0 ? void 0 : _b.toString(),
            Memory: (_c = containerOverrides.memory) === null || _c === void 0 ? void 0 : _c.toMebibytes(),
            ResourceRequirements: resources,
            Vcpus: containerOverrides.vcpus,
        };
    }
}
exports.BatchSubmitJob = BatchSubmitJob;
_a = JSII_RTTI_SYMBOL_1;
BatchSubmitJob[_a] = { fqn: "monocdk.aws_stepfunctions_tasks.BatchSubmitJob", version: "1.106.1" };
BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3VibWl0LWpvYi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInN1Ym1pdC1qb2IudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFDQSx3Q0FBd0MsQ0FBQyxtREFBbUQ7QUFDNUYsa0RBQWtELENBQUMsNkRBQTZEO0FBQ2hILHdDQUEwRCxDQUFDLGdEQUFnRDtBQUUzRyxzREFBeUY7Ozs7Ozs7QUF3SXpGLE1BQWEsY0FBZSxTQUFRLEdBQUcsQ0FBQyxhQUFhOzs7O0lBUWpELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQW1CLEtBQTBCOztRQUNqRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUQrQixVQUFLLEdBQUwsS0FBSyxDQUFxQjtRQUVqRixJQUFJLENBQUMsa0JBQWtCLFNBQUcsS0FBSyxDQUFDLGtCQUFrQixtQ0FBSSxHQUFHLENBQUMsa0JBQWtCLENBQUMsT0FBTyxDQUFDO1FBQ3JGLHFDQUF3QixDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxjQUFjLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUNqRyw0QkFBNEI7UUFDNUIsbUJBQVksQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLENBQUMsU0FBUyxFQUFFLEVBQUU7WUFDeEMsSUFBSSxTQUFTLEtBQUssU0FBUyxJQUFJLENBQUMsU0FBUyxHQUFHLENBQUMsSUFBSSxTQUFTLEdBQUcsS0FBSyxDQUFDLEVBQUU7Z0JBQ2pFLE1BQU0sSUFBSSxLQUFLLENBQUMsb0RBQW9ELFNBQVMsR0FBRyxDQUFDLENBQUM7YUFDckY7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUNILDJCQUEyQjtRQUMzQixJQUFJLEtBQUssQ0FBQyxTQUFTLElBQUksS0FBSyxDQUFDLFNBQVMsQ0FBQyxNQUFNLEdBQUcsRUFBRSxFQUFFO1lBQ2hELE1BQU0sSUFBSSxLQUFLLENBQUMsNkNBQTZDLEtBQUssQ0FBQyxTQUFTLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztTQUMzRjtRQUNELG9CQUFvQjtRQUNwQixtQkFBWSxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsQ0FBQyxRQUFRLEVBQUUsRUFBRTtZQUN0QyxJQUFJLFFBQVEsS0FBSyxTQUFTLElBQUksQ0FBQyxRQUFRLEdBQUcsQ0FBQyxJQUFJLFFBQVEsR0FBRyxFQUFFLENBQUMsRUFBRTtnQkFDM0QsTUFBTSxJQUFJLEtBQUssQ0FBQywrQ0FBK0MsUUFBUSxHQUFHLENBQUMsQ0FBQzthQUMvRTtRQUNMLENBQUMsQ0FBQyxDQUFDO1FBQ0gsbUJBQW1CO1FBQ25CLEtBQUssQ0FBQyxPQUFPLEtBQUssU0FBUyxJQUFJLG1CQUFZLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDLE9BQU8sRUFBRSxFQUFFO1lBQy9FLElBQUksT0FBTyxHQUFHLEVBQUUsRUFBRTtnQkFDZCxNQUFNLElBQUksS0FBSyxDQUFDLDhEQUE4RCxPQUFPLFdBQVcsQ0FBQyxDQUFDO2FBQ3JHO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFDSCw4RUFBOEU7UUFDOUUsMEZBQTBGO1FBQzFGLFVBQUksS0FBSyxDQUFDLGtCQUFrQiwwQ0FBRSxXQUFXLEVBQUU7WUFDdkMsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUMsV0FBVyxDQUFDLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFO2dCQUM1RCxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLEVBQUU7b0JBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQUMsc0NBQXNDLEdBQUcsc0VBQXNFLENBQUMsQ0FBQztpQkFDcEk7WUFDTCxDQUFDLENBQUMsQ0FBQztTQUNOO1FBQ0QsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMseUJBQXlCLEVBQUUsQ0FBQztJQUN6RCxDQUFDO0lBQ0Q7O09BRUc7SUFDTyxXQUFXOztRQUNqQixPQUFPO1lBQ0gsUUFBUSxFQUFFLG1DQUFzQixDQUFDLE9BQU8sRUFBRSxXQUFXLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDO1lBQy9FLFVBQVUsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQztnQkFDcEMsYUFBYSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCO2dCQUMxQyxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPO2dCQUMzQixRQUFRLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXO2dCQUNoQyxVQUFVLFFBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLDBDQUFFLEtBQUs7Z0JBQ3JDLGVBQWUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsS0FBSyxTQUFTO29CQUMvQyxDQUFDLENBQUMsRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUU7b0JBQ2hDLENBQUMsQ0FBQyxTQUFTO2dCQUNmLGtCQUFrQixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsa0JBQWtCO29CQUM3QyxDQUFDLENBQUMsSUFBSSxDQUFDLDJCQUEyQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUM7b0JBQ2pFLENBQUMsQ0FBQyxTQUFTO2dCQUNmLFNBQVMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVM7b0JBQzNCLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEVBQUUsQ0FBQyxDQUFDO3dCQUN6QyxLQUFLLEVBQUUsYUFBYSxDQUFDLEtBQUs7d0JBQzFCLElBQUksRUFBRSxhQUFhLENBQUMsSUFBSTtxQkFDM0IsQ0FBQyxDQUFDO29CQUNILENBQUMsQ0FBQyxTQUFTO2dCQUNmLGFBQWEsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsS0FBSyxTQUFTO29CQUM1QyxDQUFDLENBQUMsRUFBRSxRQUFRLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUU7b0JBQ25DLENBQUMsQ0FBQyxTQUFTO2dCQUNmLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU87b0JBQ3ZCLENBQUMsQ0FBQyxFQUFFLHNCQUFzQixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxFQUFFO29CQUM1RCxDQUFDLENBQUMsU0FBUzthQUNsQixDQUFDO1lBQ0YsY0FBYyxFQUFFLFNBQVM7U0FDNUIsQ0FBQztJQUNOLENBQUM7SUFDTyx5QkFBeUI7UUFDN0IsT0FBTztZQUNILHNHQUFzRztZQUN0Ryx1REFBdUQ7WUFDdkQsZ0dBQWdHO1lBQ2hHLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDcEIsU0FBUyxFQUFFO29CQUNQLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO3dCQUNyQixPQUFPLEVBQUUsT0FBTzt3QkFDaEIsUUFBUSxFQUFFLGdCQUFnQjt3QkFDMUIsWUFBWSxFQUFFLEdBQUc7cUJBQ3BCLENBQUM7b0JBQ0YsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXO2lCQUN6QjtnQkFDRCxPQUFPLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQzthQUMvQixDQUFDO1lBQ0YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUNwQixTQUFTLEVBQUU7b0JBQ1AsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7d0JBQ3JCLE9BQU8sRUFBRSxRQUFRO3dCQUNqQixRQUFRLEVBQUUsNkNBQTZDO3FCQUMxRCxDQUFDO2lCQUNMO2dCQUNELE9BQU8sRUFBRSxDQUFDLG1CQUFtQixFQUFFLGdCQUFnQixFQUFFLHFCQUFxQixDQUFDO2FBQzFFLENBQUM7U0FDTCxDQUFDO0lBQ04sQ0FBQztJQUNPLDJCQUEyQixDQUFDLGtCQUEyQzs7UUFDM0UsSUFBSSxXQUFXLENBQUM7UUFDaEIsSUFBSSxrQkFBa0IsQ0FBQyxXQUFXLEVBQUU7WUFDaEMsV0FBVyxHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUMsa0JBQWtCLENBQUMsV0FBVyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7Z0JBQ2hGLElBQUksRUFBRSxHQUFHO2dCQUNULEtBQUssRUFBRSxLQUFLO2FBQ2YsQ0FBQyxDQUFDLENBQUM7U0FDUDtRQUNELElBQUksU0FBUyxDQUFDO1FBQ2QsSUFBSSxrQkFBa0IsQ0FBQyxRQUFRLEVBQUU7WUFDN0IsU0FBUyxHQUFHO2dCQUNSO29CQUNJLElBQUksRUFBRSxLQUFLO29CQUNYLEtBQUssRUFBRSxHQUFHLGtCQUFrQixDQUFDLFFBQVEsRUFBRTtpQkFDMUM7YUFDSixDQUFDO1NBQ0w7UUFDRCxPQUFPO1lBQ0gsT0FBTyxFQUFFLGtCQUFrQixDQUFDLE9BQU87WUFDbkMsV0FBVyxFQUFFLFdBQVc7WUFDeEIsWUFBWSxRQUFFLGtCQUFrQixDQUFDLFlBQVksMENBQUUsUUFBUSxFQUFFO1lBQ3pELE1BQU0sUUFBRSxrQkFBa0IsQ0FBQyxNQUFNLDBDQUFFLFdBQVcsRUFBRTtZQUNoRCxvQkFBb0IsRUFBRSxTQUFTO1lBQy9CLEtBQUssRUFBRSxrQkFBa0IsQ0FBQyxLQUFLO1NBQ2xDLENBQUM7SUFDTixDQUFDOztBQWxJTCx3Q0FtSUM7OztBQWxJMkIsNkNBQThCLEdBQTZCO0lBQy9FLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxnQkFBZ0I7SUFDdkMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLE9BQU87Q0FDakMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjMiBmcm9tIFwiLi4vLi4vLi4vYXdzLWVjMlwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMidcbmltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCAqIGFzIHNmbiBmcm9tIFwiLi4vLi4vLi4vYXdzLXN0ZXBmdW5jdGlvbnNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJ1xuaW1wb3J0IHsgU2l6ZSwgU3RhY2ssIHdpdGhSZXNvbHZlZCB9IGZyb20gXCIuLi8uLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBpbnRlZ3JhdGlvblJlc291cmNlQXJuLCB2YWxpZGF0ZVBhdHRlcm5TdXBwb3J0ZWQgfSBmcm9tICcuLi9wcml2YXRlL3Rhc2stdXRpbHMnO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEJhdGNoQ29udGFpbmVyT3ZlcnJpZGVzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGNvbW1hbmQ/OiBzdHJpbmdbXTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBlbnZpcm9ubWVudD86IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogc3RyaW5nO1xuICAgIH07XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGluc3RhbmNlVHlwZT86IGVjMi5JbnN0YW5jZVR5cGU7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBtZW1vcnk/OiBTaXplO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGdwdUNvdW50PzogbnVtYmVyO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHZjcHVzPzogbnVtYmVyO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQmF0Y2hKb2JEZXBlbmRlbmN5IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBqb2JJZD86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHR5cGU/OiBzdHJpbmc7XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEJhdGNoU3VibWl0Sm9iUHJvcHMgZXh0ZW5kcyBzZm4uVGFza1N0YXRlQmFzZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBqb2JEZWZpbml0aW9uQXJuOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBqb2JOYW1lOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGpvYlF1ZXVlQXJuOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgYXJyYXlTaXplPzogbnVtYmVyO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgY29udGFpbmVyT3ZlcnJpZGVzPzogQmF0Y2hDb250YWluZXJPdmVycmlkZXM7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBkZXBlbmRzT24/OiBCYXRjaEpvYkRlcGVuZGVuY3lbXTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcGF5bG9hZD86IHNmbi5UYXNrSW5wdXQ7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGF0dGVtcHRzPzogbnVtYmVyO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBCYXRjaFN1Ym1pdEpvYiBleHRlbmRzIHNmbi5UYXNrU3RhdGVCYXNlIHtcbiAgICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBTVVBQT1JURURfSU5URUdSQVRJT05fUEFUVEVSTlM6IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm5bXSA9IFtcbiAgICAgICAgc2ZuLkludGVncmF0aW9uUGF0dGVybi5SRVFVRVNUX1JFU1BPTlNFLFxuICAgICAgICBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJVTl9KT0IsXG4gICAgXTtcbiAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgdGFza01ldHJpY3M/OiBzZm4uVGFza01ldHJpY3NDb25maWc7XG4gICAgcHJvdGVjdGVkIHJlYWRvbmx5IHRhc2tQb2xpY2llcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGludGVncmF0aW9uUGF0dGVybjogc2ZuLkludGVncmF0aW9uUGF0dGVybjtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBCYXRjaFN1Ym1pdEpvYlByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgICAgICB0aGlzLmludGVncmF0aW9uUGF0dGVybiA9IHByb3BzLmludGVncmF0aW9uUGF0dGVybiA/PyBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJVTl9KT0I7XG4gICAgICAgIHZhbGlkYXRlUGF0dGVyblN1cHBvcnRlZCh0aGlzLmludGVncmF0aW9uUGF0dGVybiwgQmF0Y2hTdWJtaXRKb2IuU1VQUE9SVEVEX0lOVEVHUkFUSU9OX1BBVFRFUk5TKTtcbiAgICAgICAgLy8gdmFsaWRhdGUgYXJyYXlTaXplIGxpbWl0c1xuICAgICAgICB3aXRoUmVzb2x2ZWQocHJvcHMuYXJyYXlTaXplLCAoYXJyYXlTaXplKSA9PiB7XG4gICAgICAgICAgICBpZiAoYXJyYXlTaXplICE9PSB1bmRlZmluZWQgJiYgKGFycmF5U2l6ZSA8IDIgfHwgYXJyYXlTaXplID4gMTAwMDApKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBhcnJheVNpemUgbXVzdCBiZSBiZXR3ZWVuIDIgYW5kIDEwLDAwMC4gUmVjZWl2ZWQgJHthcnJheVNpemV9LmApO1xuICAgICAgICAgICAgfVxuICAgICAgICB9KTtcbiAgICAgICAgLy8gdmFsaWRhdGUgZGVwZW5kZW5jeSBzaXplXG4gICAgICAgIGlmIChwcm9wcy5kZXBlbmRzT24gJiYgcHJvcHMuZGVwZW5kc09uLmxlbmd0aCA+IDIwKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYGRlcGVuZGVuY2llcyBtdXN0IGJlIDIwIG9yIGxlc3MuIFJlY2VpdmVkICR7cHJvcHMuZGVwZW5kc09uLmxlbmd0aH0uYCk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gdmFsaWRhdGUgYXR0ZW1wdHNcbiAgICAgICAgd2l0aFJlc29sdmVkKHByb3BzLmF0dGVtcHRzLCAoYXR0ZW1wdHMpID0+IHtcbiAgICAgICAgICAgIGlmIChhdHRlbXB0cyAhPT0gdW5kZWZpbmVkICYmIChhdHRlbXB0cyA8IDEgfHwgYXR0ZW1wdHMgPiAxMCkpIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYGF0dGVtcHRzIG11c3QgYmUgYmV0d2VlbiAxIGFuZCAxMC4gUmVjZWl2ZWQgJHthdHRlbXB0c30uYCk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH0pO1xuICAgICAgICAvLyB2YWxpZGF0ZSB0aW1lb3V0XG4gICAgICAgIHByb3BzLnRpbWVvdXQgIT09IHVuZGVmaW5lZCAmJiB3aXRoUmVzb2x2ZWQocHJvcHMudGltZW91dC50b1NlY29uZHMoKSwgKHRpbWVvdXQpID0+IHtcbiAgICAgICAgICAgIGlmICh0aW1lb3V0IDwgNjApIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYGF0dGVtcHQgZHVyYXRpb24gbXVzdCBiZSBncmVhdGVyIHRoYW4gNjAgc2Vjb25kcy4gUmVjZWl2ZWQgJHt0aW1lb3V0fSBzZWNvbmRzLmApO1xuICAgICAgICAgICAgfVxuICAgICAgICB9KTtcbiAgICAgICAgLy8gVGhpcyBpcyByZXF1aXJlZCBzaW5jZSBlbnZpcm9ubWVudCB2YXJpYWJsZXMgbXVzdCBub3Qgc3RhcnQgd2l0aCBBV1NfQkFUQ0g7XG4gICAgICAgIC8vIHRoaXMgbmFtaW5nIGNvbnZlbnRpb24gaXMgcmVzZXJ2ZWQgZm9yIHZhcmlhYmxlcyB0aGF0IGFyZSBzZXQgYnkgdGhlIEFXUyBCYXRjaCBzZXJ2aWNlLlxuICAgICAgICBpZiAocHJvcHMuY29udGFpbmVyT3ZlcnJpZGVzPy5lbnZpcm9ubWVudCkge1xuICAgICAgICAgICAgT2JqZWN0LmtleXMocHJvcHMuY29udGFpbmVyT3ZlcnJpZGVzLmVudmlyb25tZW50KS5mb3JFYWNoKGtleSA9PiB7XG4gICAgICAgICAgICAgICAgaWYgKGtleS5tYXRjaCgvXkFXU19CQVRDSC8pKSB7XG4gICAgICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBlbnZpcm9ubWVudCB2YXJpYWJsZSBuYW1lOiAke2tleX0uIEVudmlyb25tZW50IHZhcmlhYmxlIG5hbWVzIHN0YXJ0aW5nIHdpdGggJ0FXU19CQVRDSCcgYXJlIHJlc2VydmVkLmApO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMudGFza1BvbGljaWVzID0gdGhpcy5jb25maWd1cmVQb2xpY3lTdGF0ZW1lbnRzKCk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEBpbnRlcm5hbFxuICAgICAqL1xuICAgIHByb3RlY3RlZCBfcmVuZGVyVGFzaygpOiBhbnkge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgUmVzb3VyY2U6IGludGVncmF0aW9uUmVzb3VyY2VBcm4oJ2JhdGNoJywgJ3N1Ym1pdEpvYicsIHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuKSxcbiAgICAgICAgICAgIFBhcmFtZXRlcnM6IHNmbi5GaWVsZFV0aWxzLnJlbmRlck9iamVjdCh7XG4gICAgICAgICAgICAgICAgSm9iRGVmaW5pdGlvbjogdGhpcy5wcm9wcy5qb2JEZWZpbml0aW9uQXJuLFxuICAgICAgICAgICAgICAgIEpvYk5hbWU6IHRoaXMucHJvcHMuam9iTmFtZSxcbiAgICAgICAgICAgICAgICBKb2JRdWV1ZTogdGhpcy5wcm9wcy5qb2JRdWV1ZUFybixcbiAgICAgICAgICAgICAgICBQYXJhbWV0ZXJzOiB0aGlzLnByb3BzLnBheWxvYWQ/LnZhbHVlLFxuICAgICAgICAgICAgICAgIEFycmF5UHJvcGVydGllczogdGhpcy5wcm9wcy5hcnJheVNpemUgIT09IHVuZGVmaW5lZFxuICAgICAgICAgICAgICAgICAgICA/IHsgU2l6ZTogdGhpcy5wcm9wcy5hcnJheVNpemUgfVxuICAgICAgICAgICAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgICAgICBDb250YWluZXJPdmVycmlkZXM6IHRoaXMucHJvcHMuY29udGFpbmVyT3ZlcnJpZGVzXG4gICAgICAgICAgICAgICAgICAgID8gdGhpcy5jb25maWd1cmVDb250YWluZXJPdmVycmlkZXModGhpcy5wcm9wcy5jb250YWluZXJPdmVycmlkZXMpXG4gICAgICAgICAgICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgICAgIERlcGVuZHNPbjogdGhpcy5wcm9wcy5kZXBlbmRzT25cbiAgICAgICAgICAgICAgICAgICAgPyB0aGlzLnByb3BzLmRlcGVuZHNPbi5tYXAoam9iRGVwZW5kZW5jeSA9PiAoe1xuICAgICAgICAgICAgICAgICAgICAgICAgSm9iSWQ6IGpvYkRlcGVuZGVuY3kuam9iSWQsXG4gICAgICAgICAgICAgICAgICAgICAgICBUeXBlOiBqb2JEZXBlbmRlbmN5LnR5cGUsXG4gICAgICAgICAgICAgICAgICAgIH0pKVxuICAgICAgICAgICAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgICAgICBSZXRyeVN0cmF0ZWd5OiB0aGlzLnByb3BzLmF0dGVtcHRzICE9PSB1bmRlZmluZWRcbiAgICAgICAgICAgICAgICAgICAgPyB7IEF0dGVtcHRzOiB0aGlzLnByb3BzLmF0dGVtcHRzIH1cbiAgICAgICAgICAgICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgICAgICAgICAgVGltZW91dDogdGhpcy5wcm9wcy50aW1lb3V0XG4gICAgICAgICAgICAgICAgICAgID8geyBBdHRlbXB0RHVyYXRpb25TZWNvbmRzOiB0aGlzLnByb3BzLnRpbWVvdXQudG9TZWNvbmRzKCkgfVxuICAgICAgICAgICAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgVGltZW91dFNlY29uZHM6IHVuZGVmaW5lZCxcbiAgICAgICAgfTtcbiAgICB9XG4gICAgcHJpdmF0ZSBjb25maWd1cmVQb2xpY3lTdGF0ZW1lbnRzKCk6IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXSB7XG4gICAgICAgIHJldHVybiBbXG4gICAgICAgICAgICAvLyBSZXNvdXJjZSBsZXZlbCBhY2Nlc3MgY29udHJvbCBmb3Igam9iLWRlZmluaXRpb24gcmVxdWlyZXMgcmV2aXNpb24gd2hpY2ggYmF0Y2ggZG9lcyBub3Qgc3VwcG9ydCB5ZXRcbiAgICAgICAgICAgIC8vIFVzaW5nIHRoZSBhbHRlcm5hdGl2ZSBwZXJtaXNzaW9ucyBhcyBtZW50aW9uZWQgaGVyZTpcbiAgICAgICAgICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvdXNlcmd1aWRlL2JhdGNoLXN1cHBvcnRlZC1pYW0tYWN0aW9ucy1yZXNvdXJjZXMuaHRtbFxuICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICAgICAgICBTdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgICAgICAgICAgICAgICAgICAgc2VydmljZTogJ2JhdGNoJyxcbiAgICAgICAgICAgICAgICAgICAgICAgIHJlc291cmNlOiAnam9iLWRlZmluaXRpb24nLFxuICAgICAgICAgICAgICAgICAgICAgICAgcmVzb3VyY2VOYW1lOiAnKicsXG4gICAgICAgICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgICAgICAgICB0aGlzLnByb3BzLmpvYlF1ZXVlQXJuLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgYWN0aW9uczogWydiYXRjaDpTdWJtaXRKb2InXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICAgICAgICBTdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgICAgICAgICAgICAgICAgICAgc2VydmljZTogJ2V2ZW50cycsXG4gICAgICAgICAgICAgICAgICAgICAgICByZXNvdXJjZTogJ3J1bGUvU3RlcEZ1bmN0aW9uc0dldEV2ZW50c0ZvckJhdGNoSm9ic1J1bGUnLFxuICAgICAgICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgIGFjdGlvbnM6IFsnZXZlbnRzOlB1dFRhcmdldHMnLCAnZXZlbnRzOlB1dFJ1bGUnLCAnZXZlbnRzOkRlc2NyaWJlUnVsZSddLFxuICAgICAgICAgICAgfSksXG4gICAgICAgIF07XG4gICAgfVxuICAgIHByaXZhdGUgY29uZmlndXJlQ29udGFpbmVyT3ZlcnJpZGVzKGNvbnRhaW5lck92ZXJyaWRlczogQmF0Y2hDb250YWluZXJPdmVycmlkZXMpIHtcbiAgICAgICAgbGV0IGVudmlyb25tZW50O1xuICAgICAgICBpZiAoY29udGFpbmVyT3ZlcnJpZGVzLmVudmlyb25tZW50KSB7XG4gICAgICAgICAgICBlbnZpcm9ubWVudCA9IE9iamVjdC5lbnRyaWVzKGNvbnRhaW5lck92ZXJyaWRlcy5lbnZpcm9ubWVudCkubWFwKChba2V5LCB2YWx1ZV0pID0+ICh7XG4gICAgICAgICAgICAgICAgTmFtZToga2V5LFxuICAgICAgICAgICAgICAgIFZhbHVlOiB2YWx1ZSxcbiAgICAgICAgICAgIH0pKTtcbiAgICAgICAgfVxuICAgICAgICBsZXQgcmVzb3VyY2VzO1xuICAgICAgICBpZiAoY29udGFpbmVyT3ZlcnJpZGVzLmdwdUNvdW50KSB7XG4gICAgICAgICAgICByZXNvdXJjZXMgPSBbXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBUeXBlOiAnR1BVJyxcbiAgICAgICAgICAgICAgICAgICAgVmFsdWU6IGAke2NvbnRhaW5lck92ZXJyaWRlcy5ncHVDb3VudH1gLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBDb21tYW5kOiBjb250YWluZXJPdmVycmlkZXMuY29tbWFuZCxcbiAgICAgICAgICAgIEVudmlyb25tZW50OiBlbnZpcm9ubWVudCxcbiAgICAgICAgICAgIEluc3RhbmNlVHlwZTogY29udGFpbmVyT3ZlcnJpZGVzLmluc3RhbmNlVHlwZT8udG9TdHJpbmcoKSxcbiAgICAgICAgICAgIE1lbW9yeTogY29udGFpbmVyT3ZlcnJpZGVzLm1lbW9yeT8udG9NZWJpYnl0ZXMoKSxcbiAgICAgICAgICAgIFJlc291cmNlUmVxdWlyZW1lbnRzOiByZXNvdXJjZXMsXG4gICAgICAgICAgICBWY3B1czogY29udGFpbmVyT3ZlcnJpZGVzLnZjcHVzLFxuICAgICAgICB9O1xuICAgIH1cbn1cbiJdfQ==