"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BucketNotifications = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const destination_1 = require("../destination");
const notifications_resource_handler_1 = require("./notifications-resource-handler");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A custom CloudFormation resource that updates bucket notifications for a
 * bucket. The reason we need it is because the AWS::S3::Bucket notification
 * configuration is defined on the bucket itself, which makes it impossible to
 * provision notifications at the same time as the target (since
 * PutBucketNotifications validates the targets).
 *
 * Since only a single BucketNotifications resource is allowed for each Bucket,
 * this construct is not exported in the public API of this module. Instead, it
 * is created just-in-time by `s3.Bucket.onEvent`, so a 1:1 relationship is
 * ensured.
 *
 * @see
 * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-notificationconfig.html
 */
class BucketNotifications extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaNotifications = new Array();
        this.queueNotifications = new Array();
        this.topicNotifications = new Array();
        this.bucket = props.bucket;
    }
    /**
     * Adds a notification subscription for this bucket.
     * If this is the first notification, a BucketNotification resource is added to the stack.
     *
     * @param event The type of event
     * @param target The target construct
     * @param filters A set of S3 key filters
     */
    addNotification(event, target, ...filters) {
        const resource = this.createResourceOnce();
        // resolve target. this also provides an opportunity for the target to e.g. update
        // policies to allow this notification to happen.
        const targetProps = target.bind(this, this.bucket);
        const commonConfig = {
            Events: [event],
            Filter: renderFilters(filters),
        };
        // if the target specifies any dependencies, add them to the custom resource.
        // for example, the SNS topic policy must be created /before/ the notification resource.
        // otherwise, S3 won't be able to confirm the subscription.
        if (targetProps.dependencies) {
            resource.node.addDependency(...targetProps.dependencies);
        }
        // based on the target type, add the the correct configurations array
        switch (targetProps.type) {
            case destination_1.BucketNotificationDestinationType.LAMBDA:
                this.lambdaNotifications.push({ ...commonConfig, LambdaFunctionArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.QUEUE:
                this.queueNotifications.push({ ...commonConfig, QueueArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.TOPIC:
                this.topicNotifications.push({ ...commonConfig, TopicArn: targetProps.arn });
                break;
            default:
                throw new Error('Unsupported notification target type:' + destination_1.BucketNotificationDestinationType[targetProps.type]);
        }
    }
    renderNotificationConfiguration() {
        return {
            LambdaFunctionConfigurations: this.lambdaNotifications.length > 0 ? this.lambdaNotifications : undefined,
            QueueConfigurations: this.queueNotifications.length > 0 ? this.queueNotifications : undefined,
            TopicConfigurations: this.topicNotifications.length > 0 ? this.topicNotifications : undefined,
        };
    }
    /**
     * Defines the bucket notifications resources in the stack only once.
     * This is called lazily as we add notifications, so that if notifications are not added,
     * there is no notifications resource.
     */
    createResourceOnce() {
        if (!this.resource) {
            const handlerArn = notifications_resource_handler_1.NotificationsResourceHandler.singleton(this);
            this.resource = new cdk.CfnResource(this, 'Resource', {
                type: 'Custom::S3BucketNotifications',
                properties: {
                    ServiceToken: handlerArn,
                    BucketName: this.bucket.bucketName,
                    NotificationConfiguration: cdk.Lazy.any({ produce: () => this.renderNotificationConfiguration() }),
                },
            });
        }
        return this.resource;
    }
}
exports.BucketNotifications = BucketNotifications;
function renderFilters(filters) {
    if (!filters || filters.length === 0) {
        return undefined;
    }
    const renderedRules = new Array();
    let hasPrefix = false;
    let hasSuffix = false;
    for (const rule of filters) {
        if (!rule.suffix && !rule.prefix) {
            throw new Error('NotificationKeyFilter must specify `prefix` and/or `suffix`');
        }
        if (rule.suffix) {
            if (hasSuffix) {
                throw new Error('Cannot specify more than one suffix rule in a filter.');
            }
            renderedRules.push({ Name: 'suffix', Value: rule.suffix });
            hasSuffix = true;
        }
        if (rule.prefix) {
            if (hasPrefix) {
                throw new Error('Cannot specify more than one prefix rule in a filter.');
            }
            renderedRules.push({ Name: 'prefix', Value: rule.prefix });
            hasPrefix = true;
        }
    }
    return {
        Key: {
            FilterRules: renderedRules,
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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