import * as events from "../../aws-events";
import * as iam from "../../aws-iam";
import * as kms from "../../aws-kms";
import { IResource, RemovalPolicy, Resource } from "../../core";
import { Construct } from 'constructs';
import { BucketPolicy } from './bucket-policy';
import { IBucketNotificationDestination } from './destination';
import { LifecycleRule } from './rule';
/**
 * @experimental
 */
export interface IBucket extends IResource {
    /**
     * (experimental) The ARN of the bucket.
     *
     * @experimental
     * @attribute true
     */
    readonly bucketArn: string;
    /**
     * (experimental) The name of the bucket.
     *
     * @experimental
     * @attribute true
     */
    readonly bucketName: string;
    /**
     * (experimental) The URL of the static website.
     *
     * @experimental
     * @attribute true
     */
    readonly bucketWebsiteUrl: string;
    /**
     * (experimental) The Domain name of the static website.
     *
     * @experimental
     * @attribute true
     */
    readonly bucketWebsiteDomainName: string;
    /**
     * (experimental) The IPv4 DNS name of the specified bucket.
     *
     * @experimental
     * @attribute true
     */
    readonly bucketDomainName: string;
    /**
     * (experimental) The IPv6 DNS name of the specified bucket.
     *
     * @experimental
     * @attribute true
     */
    readonly bucketDualStackDomainName: string;
    /**
     * (experimental) The regional domain name of the specified bucket.
     *
     * @experimental
     * @attribute true
     */
    readonly bucketRegionalDomainName: string;
    /**
     * (experimental) If this bucket has been configured for static website hosting.
     *
     * @experimental
     */
    readonly isWebsite?: boolean;
    /**
     * (experimental) Optional KMS encryption key associated with this bucket.
     *
     * @experimental
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * (experimental) The resource policy associated with this bucket.
     *
     * If `autoCreatePolicy` is true, a `BucketPolicy` will be created upon the
     * first call to addToResourcePolicy(s).
     *
     * @experimental
     */
    policy?: BucketPolicy;
    /**
     * (experimental) Adds a statement to the resource policy for a principal (i.e. account/role/service) to perform actions on this bucket and/or it's contents. Use `bucketArn` and `arnForObjects(keys)` to obtain ARNs for this bucket or objects.
     *
     * @experimental
     */
    addToResourcePolicy(permission: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * (experimental) The https URL of an S3 object.
     *
     * For example:
     *
     * @param key The S3 key of the object.
     * @returns an ObjectS3Url token
     * @experimental
     * @example
     *
     * https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey
     */
    urlForObject(key?: string): string;
    /**
     * (experimental) The virtual hosted-style URL of an S3 object.
     *
     * Specify `regional: false` at
     * the options for non-regional URL. For example:
     *
     * @param key The S3 key of the object.
     * @param options Options for generating URL.
     * @returns an ObjectS3Url token
     * @experimental
     * @example
     *
     * https://china-bucket.s3.cn-north-1.amazonaws.com.cn/mykey
     */
    virtualHostedUrlForObject(key?: string, options?: VirtualHostedStyleUrlOptions): string;
    /**
     * (experimental) The S3 URL of an S3 object.
     *
     * For example:
     *
     * @param key The S3 key of the object.
     * @returns an ObjectS3Url token
     * @experimental
     * @example
     *
     * s3://bucket/key
     */
    s3UrlForObject(key?: string): string;
    /**
     * (experimental) Returns an ARN that represents all objects within the bucket that match the key pattern specified.
     *
     * To represent all keys, specify ``"*"``.
     *
     * @experimental
     */
    arnForObjects(keyPattern: string): string;
    /**
     * (experimental) Grant read permissions for this bucket and it's contents to an IAM principal (Role/Group/User).
     *
     * If encryption is used, permission to use the key to decrypt the contents
     * of the bucket will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @experimental
     */
    grantRead(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * (experimental) Grant write permissions to this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * Before CDK version 1.85.0, this method granted the `s3:PutObject*` permission that included `s3:PutObjectAcl`,
     * which could be used to grant read/write object access to IAM principals in other accounts.
     * If you want to get rid of that behavior, update your CDK version to 1.85.0 or later,
     * and make sure the `@aws-cdk/aws-s3:grantWriteWithoutAcl` feature flag is set to `true`
     * in the `context` key of your cdk.json file.
     * If you've already updated, but still need the principal to have permissions to modify the ACLs,
     * use the {@link grantPutAcl} method.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @experimental
     */
    grantWrite(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * (experimental) Grants s3:PutObject* and s3:Abort* permissions for this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @experimental
     */
    grantPut(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * (experimental) Grant the given IAM identity permissions to modify the ACLs of objects in the given Bucket.
     *
     * If your application has the '@aws-cdk/aws-s3:grantWriteWithoutAcl' feature flag set,
     * calling {@link grantWrite} or {@link grantReadWrite} no longer grants permissions to modify the ACLs of the objects;
     * in this case, if you need to modify object ACLs, call this method explicitly.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @experimental
     */
    grantPutAcl(identity: iam.IGrantable, objectsKeyPattern?: string): iam.Grant;
    /**
     * (experimental) Grants s3:DeleteObject* permission to an IAM principal for objects in this bucket.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @experimental
     */
    grantDelete(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * (experimental) Grants read/write permissions for this bucket and it's contents to an IAM principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     *
     * Before CDK version 1.85.0, this method granted the `s3:PutObject*` permission that included `s3:PutObjectAcl`,
     * which could be used to grant read/write object access to IAM principals in other accounts.
     * If you want to get rid of that behavior, update your CDK version to 1.85.0 or later,
     * and make sure the `@aws-cdk/aws-s3:grantWriteWithoutAcl` feature flag is set to `true`
     * in the `context` key of your cdk.json file.
     * If you've already updated, but still need the principal to have permissions to modify the ACLs,
     * use the {@link grantPutAcl} method.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @experimental
     */
    grantReadWrite(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * (experimental) Allows unrestricted access to objects from this bucket.
     *
     * IMPORTANT: This permission allows anyone to perform actions on S3 objects
     * in this bucket, which is useful for when you configure your bucket as a
     * website and want everyone to be able to read objects in the bucket without
     * needing to authenticate.
     *
     * Without arguments, this method will grant read ("s3:GetObject") access to
     * all objects ("*") in the bucket.
     *
     * The method returns the `iam.Grant` object, which can then be modified
     * as needed. For example, you can add a condition that will restrict access only
     * to an IPv4 range like this:
     *
     *      const grant = bucket.grantPublicAccess();
     *      grant.resourceStatement!.addCondition(‘IpAddress’, { “aws:SourceIp”: “54.240.143.0/24” });
     *
     * @param keyPrefix the prefix of S3 object keys (e.g. `home/*`). Default is "*".
     * @param allowedActions the set of S3 actions to allow.
     * @returns The `iam.PolicyStatement` object, which can be used to apply e.g. conditions.
     * @experimental
     */
    grantPublicAccess(keyPrefix?: string, ...allowedActions: string[]): iam.Grant;
    /**
     * (experimental) Defines a CloudWatch event that triggers when something happens to this bucket.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @experimental
     */
    onCloudTrailEvent(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * (experimental) Defines an AWS CloudWatch event that triggers when an object is uploaded to the specified paths (keys) in this bucket using the PutObject API call.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using `onCloudTrailWriteObject` may be preferable.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @experimental
     */
    onCloudTrailPutObject(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * (experimental) Defines an AWS CloudWatch event that triggers when an object at the specified paths (keys) in this bucket are written to.
     *
     * This includes
     * the events PutObject, CopyObject, and CompleteMultipartUpload.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using this method may be preferable to `onCloudTrailPutObject`.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @experimental
     */
    onCloudTrailWriteObject(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
}
/**
 * (experimental) A reference to a bucket.
 *
 * The easiest way to instantiate is to call
 * `bucket.export()`. Then, the consumer can use `Bucket.import(this, ref)` and
 * get a `Bucket`.
 *
 * @experimental
 */
export interface BucketAttributes {
    /**
     * (experimental) The ARN of the bucket.
     *
     * At least one of bucketArn or bucketName must be
     * defined in order to initialize a bucket ref.
     *
     * @experimental
     */
    readonly bucketArn?: string;
    /**
     * (experimental) The name of the bucket.
     *
     * If the underlying value of ARN is a string, the
     * name will be parsed from the ARN. Otherwise, the name is optional, but
     * some features that require the bucket name such as auto-creating a bucket
     * policy, won't work.
     *
     * @experimental
     */
    readonly bucketName?: string;
    /**
     * (experimental) The domain name of the bucket.
     *
     * @default Inferred from bucket name
     * @experimental
     */
    readonly bucketDomainName?: string;
    /**
     * (experimental) The website URL of the bucket (if static web hosting is enabled).
     *
     * @default Inferred from bucket name
     * @experimental
     */
    readonly bucketWebsiteUrl?: string;
    /**
     * (experimental) The regional domain name of the specified bucket.
     *
     * @experimental
     */
    readonly bucketRegionalDomainName?: string;
    /**
     * (experimental) The IPv6 DNS name of the specified bucket.
     *
     * @experimental
     */
    readonly bucketDualStackDomainName?: string;
    /**
     * (experimental) The format of the website URL of the bucket.
     *
     * This should be true for
     * regions launched since 2014.
     *
     * @default false
     * @experimental
     */
    readonly bucketWebsiteNewUrlFormat?: boolean;
    /**
     * @experimental
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * (experimental) If this bucket has been configured for static website hosting.
     *
     * @default false
     * @experimental
     */
    readonly isWebsite?: boolean;
    /**
     * (experimental) The account this existing bucket belongs to.
     *
     * @default - it's assumed the bucket belongs to the same account as the scope it's being imported into
     * @experimental
     */
    readonly account?: string;
    /**
     * (experimental) The region this existing bucket is in.
     *
     * @default - it's assumed the bucket is in the same region as the scope it's being imported into
     * @experimental
     */
    readonly region?: string;
}
/**
 * Represents an S3 Bucket.
 *
 * Buckets can be either defined within this stack:
 *
 *   new Bucket(this, 'MyBucket', { props });
 *
 * Or imported from an existing bucket:
 *
 *   Bucket.import(this, 'MyImportedBucket', { bucketArn: ... });
 *
 * You can also export a bucket and import it into another stack:
 *
 *   const ref = myBucket.export();
 *   Bucket.import(this, 'MyImportedBucket', ref);
 *
 */
declare abstract class BucketBase extends Resource implements IBucket {
    abstract readonly bucketArn: string;
    abstract readonly bucketName: string;
    abstract readonly bucketDomainName: string;
    abstract readonly bucketWebsiteUrl: string;
    abstract readonly bucketWebsiteDomainName: string;
    abstract readonly bucketRegionalDomainName: string;
    abstract readonly bucketDualStackDomainName: string;
    /**
     * Optional KMS encryption key associated with this bucket.
     */
    abstract readonly encryptionKey?: kms.IKey;
    /**
     * If this bucket has been configured for static website hosting.
     */
    abstract readonly isWebsite?: boolean;
    /**
     * The resource policy associated with this bucket.
     *
     * If `autoCreatePolicy` is true, a `BucketPolicy` will be created upon the
     * first call to addToResourcePolicy(s).
     */
    abstract policy?: BucketPolicy;
    /**
     * Indicates if a bucket resource policy should automatically created upon
     * the first call to `addToResourcePolicy`.
     */
    protected abstract autoCreatePolicy: boolean;
    /**
     * Whether to disallow public access
     */
    protected abstract disallowPublicAccess?: boolean;
    /**
     * (experimental) Define a CloudWatch event that triggers when something happens to this repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @experimental
     */
    onCloudTrailEvent(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * (experimental) Defines an AWS CloudWatch event that triggers when an object is uploaded to the specified paths (keys) in this bucket using the PutObject API call.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using `onCloudTrailWriteObject` may be preferable.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @experimental
     */
    onCloudTrailPutObject(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * (experimental) Defines an AWS CloudWatch event that triggers when an object at the specified paths (keys) in this bucket are written to.
     *
     * This includes
     * the events PutObject, CopyObject, and CompleteMultipartUpload.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using this method may be preferable to `onCloudTrailPutObject`.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @experimental
     */
    onCloudTrailWriteObject(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * (experimental) Adds a statement to the resource policy for a principal (i.e. account/role/service) to perform actions on this bucket and/or it's contents. Use `bucketArn` and `arnForObjects(keys)` to obtain ARNs for this bucket or objects.
     *
     * @experimental
     */
    addToResourcePolicy(permission: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * (experimental) Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @experimental
     */
    protected validate(): string[];
    /**
     * (experimental) The https URL of an S3 object.
     *
     * Specify `regional: false` at the options
     * for non-regional URLs. For example:
     *
     * @param key The S3 key of the object.
     * @returns an ObjectS3Url token
     * @experimental
     * @example
     *
     * https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey
     */
    urlForObject(key?: string): string;
    /**
     * (experimental) The virtual hosted-style URL of an S3 object.
     *
     * Specify `regional: false` at
     * the options for non-regional URL. For example:
     *
     * @param key The S3 key of the object.
     * @param options Options for generating URL.
     * @returns an ObjectS3Url token
     * @experimental
     * @example
     *
     * https://china-bucket.s3.cn-north-1.amazonaws.com.cn/mykey
     */
    virtualHostedUrlForObject(key?: string, options?: VirtualHostedStyleUrlOptions): string;
    /**
     * (experimental) The S3 URL of an S3 object.
     *
     * For example:
     *
     * @param key The S3 key of the object.
     * @returns an ObjectS3Url token
     * @experimental
     * @example
     *
     * s3://bucket/key
     */
    s3UrlForObject(key?: string): string;
    /**
     * (experimental) Returns an ARN that represents all objects within the bucket that match the key pattern specified.
     *
     * To represent all keys, specify ``"*"``.
     *
     * If you need to specify a keyPattern with multiple components, concatenate them into a single string, e.g.:
     *
     *    arnForObjects(`home/${team}/${user}/*`)
     *
     * @experimental
     */
    arnForObjects(keyPattern: string): string;
    /**
     * (experimental) Grant read permissions for this bucket and it's contents to an IAM principal (Role/Group/User).
     *
     * If encryption is used, permission to use the key to decrypt the contents
     * of the bucket will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @experimental
     */
    grantRead(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * (experimental) Grant write permissions to this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * Before CDK version 1.85.0, this method granted the `s3:PutObject*` permission that included `s3:PutObjectAcl`,
     * which could be used to grant read/write object access to IAM principals in other accounts.
     * If you want to get rid of that behavior, update your CDK version to 1.85.0 or later,
     * and make sure the `@aws-cdk/aws-s3:grantWriteWithoutAcl` feature flag is set to `true`
     * in the `context` key of your cdk.json file.
     * If you've already updated, but still need the principal to have permissions to modify the ACLs,
     * use the {@link grantPutAcl} method.
     *
     * @experimental
     */
    grantWrite(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * (experimental) Grants s3:PutObject* and s3:Abort* permissions for this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @experimental
     */
    grantPut(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * (experimental) Grant the given IAM identity permissions to modify the ACLs of objects in the given Bucket.
     *
     * If your application has the '@aws-cdk/aws-s3:grantWriteWithoutAcl' feature flag set,
     * calling {@link grantWrite} or {@link grantReadWrite} no longer grants permissions to modify the ACLs of the objects;
     * in this case, if you need to modify object ACLs, call this method explicitly.
     *
     * @experimental
     */
    grantPutAcl(identity: iam.IGrantable, objectsKeyPattern?: string): iam.Grant;
    /**
     * (experimental) Grants s3:DeleteObject* permission to an IAM principal for objects in this bucket.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @experimental
     */
    grantDelete(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * (experimental) Grants read/write permissions for this bucket and it's contents to an IAM principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     *
     * Before CDK version 1.85.0, this method granted the `s3:PutObject*` permission that included `s3:PutObjectAcl`,
     * which could be used to grant read/write object access to IAM principals in other accounts.
     * If you want to get rid of that behavior, update your CDK version to 1.85.0 or later,
     * and make sure the `@aws-cdk/aws-s3:grantWriteWithoutAcl` feature flag is set to `true`
     * in the `context` key of your cdk.json file.
     * If you've already updated, but still need the principal to have permissions to modify the ACLs,
     * use the {@link grantPutAcl} method.
     *
     * @experimental
     */
    grantReadWrite(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * (experimental) Allows unrestricted access to objects from this bucket.
     *
     * IMPORTANT: This permission allows anyone to perform actions on S3 objects
     * in this bucket, which is useful for when you configure your bucket as a
     * website and want everyone to be able to read objects in the bucket without
     * needing to authenticate.
     *
     * Without arguments, this method will grant read ("s3:GetObject") access to
     * all objects ("*") in the bucket.
     *
     * The method returns the `iam.Grant` object, which can then be modified
     * as needed. For example, you can add a condition that will restrict access only
     * to an IPv4 range like this:
     *
     *      const grant = bucket.grantPublicAccess();
     *      grant.resourceStatement!.addCondition(‘IpAddress’, { “aws:SourceIp”: “54.240.143.0/24” });
     *
     * @param keyPrefix the prefix of S3 object keys (e.g. `home/*`). Default is "*".
     * @param allowedActions the set of S3 actions to allow.
     * @experimental
     */
    grantPublicAccess(keyPrefix?: string, ...allowedActions: string[]): iam.Grant;
    private get writeActions();
    private get putActions();
    private urlJoin;
    private grant;
}
/**
 * @experimental
 */
export interface BlockPublicAccessOptions {
    /**
     * (experimental) Whether to block public ACLs.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html#access-control-block-public-access-options
     * @experimental
     */
    readonly blockPublicAcls?: boolean;
    /**
     * (experimental) Whether to block public policy.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html#access-control-block-public-access-options
     * @experimental
     */
    readonly blockPublicPolicy?: boolean;
    /**
     * (experimental) Whether to ignore public ACLs.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html#access-control-block-public-access-options
     * @experimental
     */
    readonly ignorePublicAcls?: boolean;
    /**
     * (experimental) Whether to restrict public access.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html#access-control-block-public-access-options
     * @experimental
     */
    readonly restrictPublicBuckets?: boolean;
}
/**
 * @experimental
 */
export declare class BlockPublicAccess {
    /**
     * @experimental
     */
    static readonly BLOCK_ALL: BlockPublicAccess;
    /**
     * @experimental
     */
    static readonly BLOCK_ACLS: BlockPublicAccess;
    /**
     * @experimental
     */
    blockPublicAcls: boolean | undefined;
    /**
     * @experimental
     */
    blockPublicPolicy: boolean | undefined;
    /**
     * @experimental
     */
    ignorePublicAcls: boolean | undefined;
    /**
     * @experimental
     */
    restrictPublicBuckets: boolean | undefined;
    /**
     * @experimental
     */
    constructor(options: BlockPublicAccessOptions);
}
/**
 * (experimental) Specifies a metrics configuration for the CloudWatch request metrics from an Amazon S3 bucket.
 *
 * @experimental
 */
export interface BucketMetrics {
    /**
     * (experimental) The ID used to identify the metrics configuration.
     *
     * @experimental
     */
    readonly id: string;
    /**
     * (experimental) The prefix that an object must have to be included in the metrics results.
     *
     * @experimental
     */
    readonly prefix?: string;
    /**
     * (experimental) Specifies a list of tag filters to use as a metrics configuration filter.
     *
     * The metrics configuration includes only objects that meet the filter's criteria.
     *
     * @experimental
     */
    readonly tagFilters?: {
        [tag: string]: any;
    };
}
/**
 * (experimental) All http request methods.
 *
 * @experimental
 */
export declare enum HttpMethods {
    /**
     * (experimental) The GET method requests a representation of the specified resource.
     *
     * @experimental
     */
    GET = "GET",
    /**
     * (experimental) The PUT method replaces all current representations of the target resource with the request payload.
     *
     * @experimental
     */
    PUT = "PUT",
    /**
     * (experimental) The HEAD method asks for a response identical to that of a GET request, but without the response body.
     *
     * @experimental
     */
    HEAD = "HEAD",
    /**
     * (experimental) The POST method is used to submit an entity to the specified resource, often causing a change in state or side effects on the server.
     *
     * @experimental
     */
    POST = "POST",
    /**
     * (experimental) The DELETE method deletes the specified resource.
     *
     * @experimental
     */
    DELETE = "DELETE"
}
/**
 * (experimental) Specifies a cross-origin access rule for an Amazon S3 bucket.
 *
 * @experimental
 */
export interface CorsRule {
    /**
     * (experimental) A unique identifier for this rule.
     *
     * @default - No id specified.
     * @experimental
     */
    readonly id?: string;
    /**
     * (experimental) The time in seconds that your browser is to cache the preflight response for the specified resource.
     *
     * @default - No caching.
     * @experimental
     */
    readonly maxAge?: number;
    /**
     * (experimental) Headers that are specified in the Access-Control-Request-Headers header.
     *
     * @default - No headers allowed.
     * @experimental
     */
    readonly allowedHeaders?: string[];
    /**
     * (experimental) An HTTP method that you allow the origin to execute.
     *
     * @experimental
     */
    readonly allowedMethods: HttpMethods[];
    /**
     * (experimental) One or more origins you want customers to be able to access the bucket from.
     *
     * @experimental
     */
    readonly allowedOrigins: string[];
    /**
     * (experimental) One or more headers in the response that you want customers to be able to access from their applications.
     *
     * @default - No headers exposed.
     * @experimental
     */
    readonly exposedHeaders?: string[];
}
/**
 * (experimental) All http request methods.
 *
 * @experimental
 */
export declare enum RedirectProtocol {
    /**
     * @experimental
     */
    HTTP = "http",
    /**
     * @experimental
     */
    HTTPS = "https"
}
/**
 * (experimental) Specifies a redirect behavior of all requests to a website endpoint of a bucket.
 *
 * @experimental
 */
export interface RedirectTarget {
    /**
     * (experimental) Name of the host where requests are redirected.
     *
     * @experimental
     */
    readonly hostName: string;
    /**
     * (experimental) Protocol to use when redirecting requests.
     *
     * @default - The protocol used in the original request.
     * @experimental
     */
    readonly protocol?: RedirectProtocol;
}
/**
 * (experimental) All supported inventory list formats.
 *
 * @experimental
 */
export declare enum InventoryFormat {
    /**
     * (experimental) Generate the inventory list as CSV.
     *
     * @experimental
     */
    CSV = "CSV",
    /**
     * (experimental) Generate the inventory list as Parquet.
     *
     * @experimental
     */
    PARQUET = "Parquet",
    /**
     * (experimental) Generate the inventory list as Parquet.
     *
     * @experimental
     */
    ORC = "ORC"
}
/**
 * (experimental) All supported inventory frequencies.
 *
 * @experimental
 */
export declare enum InventoryFrequency {
    /**
     * (experimental) A report is generated every day.
     *
     * @experimental
     */
    DAILY = "Daily",
    /**
     * (experimental) A report is generated every Sunday (UTC timezone) after the initial report.
     *
     * @experimental
     */
    WEEKLY = "Weekly"
}
/**
 * (experimental) Inventory version support.
 *
 * @experimental
 */
export declare enum InventoryObjectVersion {
    /**
     * (experimental) Includes all versions of each object in the report.
     *
     * @experimental
     */
    ALL = "All",
    /**
     * (experimental) Includes only the current version of each object in the report.
     *
     * @experimental
     */
    CURRENT = "Current"
}
/**
 * (experimental) The destination of the inventory.
 *
 * @experimental
 */
export interface InventoryDestination {
    /**
     * (experimental) Bucket where all inventories will be saved in.
     *
     * @experimental
     */
    readonly bucket: IBucket;
    /**
     * (experimental) The prefix to be used when saving the inventory.
     *
     * @default - No prefix.
     * @experimental
     */
    readonly prefix?: string;
    /**
     * (experimental) The account ID that owns the destination S3 bucket.
     *
     * If no account ID is provided, the owner is not validated before exporting data.
     * It's recommended to set an account ID to prevent problems if the destination bucket ownership changes.
     *
     * @default - No account ID.
     * @experimental
     */
    readonly bucketOwner?: string;
}
/**
 * (experimental) Specifies the inventory configuration of an S3 Bucket.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/storage-inventory.html
 * @experimental
 */
export interface Inventory {
    /**
     * (experimental) The destination of the inventory.
     *
     * @experimental
     */
    readonly destination: InventoryDestination;
    /**
     * (experimental) The inventory will only include objects that meet the prefix filter criteria.
     *
     * @default - No objects prefix
     * @experimental
     */
    readonly objectsPrefix?: string;
    /**
     * (experimental) The format of the inventory.
     *
     * @default InventoryFormat.CSV
     * @experimental
     */
    readonly format?: InventoryFormat;
    /**
     * (experimental) Whether the inventory is enabled or disabled.
     *
     * @default true
     * @experimental
     */
    readonly enabled?: boolean;
    /**
     * (experimental) The inventory configuration ID.
     *
     * @default - generated ID.
     * @experimental
     */
    readonly inventoryId?: string;
    /**
     * (experimental) Frequency at which the inventory should be generated.
     *
     * @default InventoryFrequency.WEEKLY
     * @experimental
     */
    readonly frequency?: InventoryFrequency;
    /**
     * (experimental) If the inventory should contain all the object versions or only the current one.
     *
     * @default InventoryObjectVersion.ALL
     * @experimental
     */
    readonly includeObjectVersions?: InventoryObjectVersion;
    /**
     * (experimental) A list of optional fields to be included in the inventory result.
     *
     * @default - No optional fields.
     * @experimental
     */
    readonly optionalFields?: string[];
}
/**
 * (experimental) The ObjectOwnership of the bucket.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/about-object-ownership.html
 * @experimental
 */
export declare enum ObjectOwnership {
    /**
     * (experimental) Objects uploaded to the bucket change ownership to the bucket owner .
     *
     * @experimental
     */
    BUCKET_OWNER_PREFERRED = "BucketOwnerPreferred",
    /**
     * (experimental) The uploading account will own the object.
     *
     * @experimental
     */
    OBJECT_WRITER = "ObjectWriter"
}
/**
 * @experimental
 */
export interface BucketProps {
    /**
     * (experimental) The kind of server-side encryption to apply to this bucket.
     *
     * If you choose KMS, you can specify a KMS key via `encryptionKey`. If
     * encryption key is not specified, a key will automatically be created.
     *
     * @default - `Kms` if `encryptionKey` is specified, or `Unencrypted` otherwise.
     * @experimental
     */
    readonly encryption?: BucketEncryption;
    /**
     * (experimental) External KMS key to use for bucket encryption.
     *
     * The 'encryption' property must be either not specified or set to "Kms".
     * An error will be emitted if encryption is set to "Unencrypted" or
     * "Managed".
     *
     * @default - If encryption is set to "Kms" and this property is undefined,
     * a new KMS key will be created and associated with this bucket.
     * @experimental
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * (experimental) Enforces SSL for requests.
     *
     * S3.5 of the AWS Foundational Security Best Practices Regarding S3.
     *
     * @default false
     * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-ssl-requests-only.html
     * @experimental
     */
    readonly enforceSSL?: boolean;
    /**
     * (experimental) Specifies whether Amazon S3 should use an S3 Bucket Key with server-side encryption using KMS (SSE-KMS) for new objects in the bucket.
     *
     * Only relevant, when Encryption is set to {@link BucketEncryption.KMS}
     *
     * @default - false
     * @experimental
     */
    readonly bucketKeyEnabled?: boolean;
    /**
     * (experimental) Physical name of this bucket.
     *
     * @default - Assigned by CloudFormation (recommended).
     * @experimental
     */
    readonly bucketName?: string;
    /**
     * (experimental) Policy to apply when the bucket is removed from this stack.
     *
     * @default - The bucket will be orphaned.
     * @experimental
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * (experimental) Whether all objects should be automatically deleted when the bucket is removed from the stack or when the stack is deleted.
     *
     * Requires the `removalPolicy` to be set to `RemovalPolicy.DESTROY`.
     *
     * @default false
     * @experimental
     */
    readonly autoDeleteObjects?: boolean;
    /**
     * (experimental) Whether this bucket should have versioning turned on or not.
     *
     * @default false
     * @experimental
     */
    readonly versioned?: boolean;
    /**
     * (experimental) Rules that define how Amazon S3 manages objects during their lifetime.
     *
     * @default - No lifecycle rules.
     * @experimental
     */
    readonly lifecycleRules?: LifecycleRule[];
    /**
     * (experimental) The name of the index document (e.g. "index.html") for the website. Enables static website hosting for this bucket.
     *
     * @default - No index document.
     * @experimental
     */
    readonly websiteIndexDocument?: string;
    /**
     * (experimental) The name of the error document (e.g. "404.html") for the website. `websiteIndexDocument` must also be set if this is set.
     *
     * @default - No error document.
     * @experimental
     */
    readonly websiteErrorDocument?: string;
    /**
     * (experimental) Specifies the redirect behavior of all requests to a website endpoint of a bucket.
     *
     * If you specify this property, you can't specify "websiteIndexDocument", "websiteErrorDocument" nor , "websiteRoutingRules".
     *
     * @default - No redirection.
     * @experimental
     */
    readonly websiteRedirect?: RedirectTarget;
    /**
     * (experimental) Rules that define when a redirect is applied and the redirect behavior.
     *
     * @default - No redirection rules.
     * @experimental
     */
    readonly websiteRoutingRules?: RoutingRule[];
    /**
     * (experimental) Specifies a canned ACL that grants predefined permissions to the bucket.
     *
     * @default BucketAccessControl.PRIVATE
     * @experimental
     */
    readonly accessControl?: BucketAccessControl;
    /**
     * (experimental) Grants public read access to all objects in the bucket.
     *
     * Similar to calling `bucket.grantPublicAccess()`
     *
     * @default false
     * @experimental
     */
    readonly publicReadAccess?: boolean;
    /**
     * (experimental) The block public access configuration of this bucket.
     *
     * @default - CloudFormation defaults will apply. New buckets and objects don't allow public access, but users can modify bucket policies or object permissions to allow public access
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html
     * @experimental
     */
    readonly blockPublicAccess?: BlockPublicAccess;
    /**
     * (experimental) The metrics configuration of this bucket.
     *
     * @default - No metrics configuration.
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-metricsconfiguration.html
     * @experimental
     */
    readonly metrics?: BucketMetrics[];
    /**
     * (experimental) The CORS configuration of this bucket.
     *
     * @default - No CORS configuration.
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-cors.html
     * @experimental
     */
    readonly cors?: CorsRule[];
    /**
     * (experimental) Destination bucket for the server access logs.
     *
     * @default - If "serverAccessLogsPrefix" undefined - access logs disabled, otherwise - log to current bucket.
     * @experimental
     */
    readonly serverAccessLogsBucket?: IBucket;
    /**
     * (experimental) Optional log file prefix to use for the bucket's access logs.
     *
     * If defined without "serverAccessLogsBucket", enables access logs to current bucket with this prefix.
     *
     * @default - No log file prefix
     * @experimental
     */
    readonly serverAccessLogsPrefix?: string;
    /**
     * (experimental) The inventory configuration of the bucket.
     *
     * @default - No inventory configuration
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/storage-inventory.html
     * @experimental
     */
    readonly inventories?: Inventory[];
    /**
     * (experimental) The objectOwnership of the bucket.
     *
     * @default - No ObjectOwnership configuration, uploading account will own the object.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/about-object-ownership.html
     * @experimental
     */
    readonly objectOwnership?: ObjectOwnership;
}
/**
 * (experimental) An S3 bucket with associated policy objects.
 *
 * This bucket does not yet have all features that exposed by the underlying
 * BucketResource.
 *
 * @experimental
 */
export declare class Bucket extends BucketBase {
    /**
     * @experimental
     */
    static fromBucketArn(scope: Construct, id: string, bucketArn: string): IBucket;
    /**
     * @experimental
     */
    static fromBucketName(scope: Construct, id: string, bucketName: string): IBucket;
    /**
     * (experimental) Creates a Bucket construct that represents an external bucket.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `BucketAttributes` object.
     * @experimental
     */
    static fromBucketAttributes(scope: Construct, id: string, attrs: BucketAttributes): IBucket;
    /**
     * (experimental) The ARN of the bucket.
     *
     * @experimental
     */
    readonly bucketArn: string;
    /**
     * (experimental) The name of the bucket.
     *
     * @experimental
     */
    readonly bucketName: string;
    /**
     * (experimental) The IPv4 DNS name of the specified bucket.
     *
     * @experimental
     */
    readonly bucketDomainName: string;
    /**
     * (experimental) The URL of the static website.
     *
     * @experimental
     */
    readonly bucketWebsiteUrl: string;
    /**
     * (experimental) The Domain name of the static website.
     *
     * @experimental
     */
    readonly bucketWebsiteDomainName: string;
    /**
     * (experimental) The IPv6 DNS name of the specified bucket.
     *
     * @experimental
     */
    readonly bucketDualStackDomainName: string;
    /**
     * (experimental) The regional domain name of the specified bucket.
     *
     * @experimental
     */
    readonly bucketRegionalDomainName: string;
    /**
     * (experimental) Optional KMS encryption key associated with this bucket.
     *
     * @experimental
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * (experimental) If this bucket has been configured for static website hosting.
     *
     * @experimental
     */
    readonly isWebsite?: boolean;
    /**
     * (experimental) The resource policy associated with this bucket.
     *
     * If `autoCreatePolicy` is true, a `BucketPolicy` will be created upon the
     * first call to addToResourcePolicy(s).
     *
     * @experimental
     */
    policy?: BucketPolicy;
    /**
     * (experimental) Indicates if a bucket resource policy should automatically created upon the first call to `addToResourcePolicy`.
     *
     * @experimental
     */
    protected autoCreatePolicy: boolean;
    /**
     * (experimental) Whether to disallow public access.
     *
     * @experimental
     */
    protected disallowPublicAccess?: boolean;
    private accessControl?;
    private readonly lifecycleRules;
    private readonly versioned?;
    private readonly notifications;
    private readonly metrics;
    private readonly cors;
    private readonly inventories;
    /**
     * @experimental
     */
    constructor(scope: Construct, id: string, props?: BucketProps);
    /**
     * (experimental) Add a lifecycle rule to the bucket.
     *
     * @param rule The rule to add.
     * @experimental
     */
    addLifecycleRule(rule: LifecycleRule): void;
    /**
     * (experimental) Adds a metrics configuration for the CloudWatch request metrics from the bucket.
     *
     * @param metric The metric configuration to add.
     * @experimental
     */
    addMetric(metric: BucketMetrics): void;
    /**
     * (experimental) Adds a cross-origin access configuration for objects in an Amazon S3 bucket.
     *
     * @param rule The CORS configuration rule to add.
     * @experimental
     */
    addCorsRule(rule: CorsRule): void;
    /**
     * (experimental) Adds a bucket notification event destination.
     *
     * @param event The event to trigger the notification.
     * @param dest The notification destination (Lambda, SNS Topic or SQS Queue).
     * @param filters S3 object key filter rules to determine which objects trigger this event.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html
     * @experimental
     * @example
     *
     *    bucket.addEventNotification(EventType.OnObjectCreated, myLambda, 'home/myusername/*')
     */
    addEventNotification(event: EventType, dest: IBucketNotificationDestination, ...filters: NotificationKeyFilter[]): void;
    /**
     * (experimental) Subscribes a destination to receive notifications when an object is created in the bucket.
     *
     * This is identical to calling
     * `onEvent(EventType.ObjectCreated)`.
     *
     * @param dest The notification destination (see onEvent).
     * @param filters Filters (see onEvent).
     * @experimental
     */
    addObjectCreatedNotification(dest: IBucketNotificationDestination, ...filters: NotificationKeyFilter[]): void;
    /**
     * (experimental) Subscribes a destination to receive notifications when an object is removed from the bucket.
     *
     * This is identical to calling
     * `onEvent(EventType.ObjectRemoved)`.
     *
     * @param dest The notification destination (see onEvent).
     * @param filters Filters (see onEvent).
     * @experimental
     */
    addObjectRemovedNotification(dest: IBucketNotificationDestination, ...filters: NotificationKeyFilter[]): void;
    /**
     * (experimental) Add an inventory configuration.
     *
     * @param inventory configuration to add.
     * @experimental
     */
    addInventory(inventory: Inventory): void;
    /**
     * Adds an iam statement to enforce SSL requests only.
     */
    private enforceSSLStatement;
    private validateBucketName;
    /**
     * Set up key properties and return the Bucket encryption property from the
     * user's configuration.
     */
    private parseEncryption;
    /**
     * Parse the lifecycle configuration out of the bucket props
     * @param props Par
     */
    private parseLifecycleConfiguration;
    private parseServerAccessLogs;
    private parseMetricConfiguration;
    private parseCorsConfiguration;
    private parseTagFilters;
    private parseOwnershipControls;
    private renderWebsiteConfiguration;
    /**
     * Allows the LogDelivery group to write, fails if ACL was set differently.
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl
     */
    private allowLogDelivery;
    private parseInventoryConfiguration;
    private enableAutoDeleteObjects;
}
/**
 * (experimental) What kind of server-side encryption to apply to this bucket.
 *
 * @experimental
 */
export declare enum BucketEncryption {
    /**
     * (experimental) Objects in the bucket are not encrypted.
     *
     * @experimental
     */
    UNENCRYPTED = "NONE",
    /**
     * (experimental) Server-side KMS encryption with a master key managed by KMS.
     *
     * @experimental
     */
    KMS_MANAGED = "MANAGED",
    /**
     * (experimental) Server-side encryption with a master key managed by S3.
     *
     * @experimental
     */
    S3_MANAGED = "S3MANAGED",
    /**
     * (experimental) Server-side encryption with a KMS key managed by the user.
     *
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     *
     * @experimental
     */
    KMS = "KMS"
}
/**
 * (experimental) Notification event types.
 *
 * @experimental
 */
export declare enum EventType {
    /**
     * (experimental) Amazon S3 APIs such as PUT, POST, and COPY can create an object.
     *
     * Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     *
     * @experimental
     */
    OBJECT_CREATED = "s3:ObjectCreated:*",
    /**
     * (experimental) Amazon S3 APIs such as PUT, POST, and COPY can create an object.
     *
     * Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     *
     * @experimental
     */
    OBJECT_CREATED_PUT = "s3:ObjectCreated:Put",
    /**
     * (experimental) Amazon S3 APIs such as PUT, POST, and COPY can create an object.
     *
     * Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     *
     * @experimental
     */
    OBJECT_CREATED_POST = "s3:ObjectCreated:Post",
    /**
     * (experimental) Amazon S3 APIs such as PUT, POST, and COPY can create an object.
     *
     * Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     *
     * @experimental
     */
    OBJECT_CREATED_COPY = "s3:ObjectCreated:Copy",
    /**
     * (experimental) Amazon S3 APIs such as PUT, POST, and COPY can create an object.
     *
     * Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     *
     * @experimental
     */
    OBJECT_CREATED_COMPLETE_MULTIPART_UPLOAD = "s3:ObjectCreated:CompleteMultipartUpload",
    /**
     * (experimental) By using the ObjectRemoved event types, you can enable notification when an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     *
     * @experimental
     */
    OBJECT_REMOVED = "s3:ObjectRemoved:*",
    /**
     * (experimental) By using the ObjectRemoved event types, you can enable notification when an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     *
     * @experimental
     */
    OBJECT_REMOVED_DELETE = "s3:ObjectRemoved:Delete",
    /**
     * (experimental) By using the ObjectRemoved event types, you can enable notification when an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     *
     * @experimental
     */
    OBJECT_REMOVED_DELETE_MARKER_CREATED = "s3:ObjectRemoved:DeleteMarkerCreated",
    /**
     * (experimental) Using restore object event types you can receive notifications for initiation and completion when restoring objects from the S3 Glacier storage class.
     *
     * You use s3:ObjectRestore:Post to request notification of object restoration
     * initiation.
     *
     * @experimental
     */
    OBJECT_RESTORE_POST = "s3:ObjectRestore:Post",
    /**
     * (experimental) Using restore object event types you can receive notifications for initiation and completion when restoring objects from the S3 Glacier storage class.
     *
     * You use s3:ObjectRestore:Completed to request notification of
     * restoration completion.
     *
     * @experimental
     */
    OBJECT_RESTORE_COMPLETED = "s3:ObjectRestore:Completed",
    /**
     * (experimental) You can use this event type to request Amazon S3 to send a notification message when Amazon S3 detects that an object of the RRS storage class is lost.
     *
     * @experimental
     */
    REDUCED_REDUNDANCY_LOST_OBJECT = "s3:ReducedRedundancyLostObject",
    /**
     * (experimental) You receive this notification event when an object that was eligible for replication using Amazon S3 Replication Time Control failed to replicate.
     *
     * @experimental
     */
    REPLICATION_OPERATION_FAILED_REPLICATION = "s3:Replication:OperationFailedReplication",
    /**
     * (experimental) You receive this notification event when an object that was eligible for replication using Amazon S3 Replication Time Control exceeded the 15-minute threshold for replication.
     *
     * @experimental
     */
    REPLICATION_OPERATION_MISSED_THRESHOLD = "s3:Replication:OperationMissedThreshold",
    /**
     * (experimental) You receive this notification event for an object that was eligible for replication using the Amazon S3 Replication Time Control feature replicated after the 15-minute threshold.
     *
     * @experimental
     */
    REPLICATION_OPERATION_REPLICATED_AFTER_THRESHOLD = "s3:Replication:OperationReplicatedAfterThreshold",
    /**
     * (experimental) You receive this notification event for an object that was eligible for replication using Amazon S3 Replication Time Control but is no longer tracked by replication metrics.
     *
     * @experimental
     */
    REPLICATION_OPERATION_NOT_TRACKED = "s3:Replication:OperationNotTracked"
}
/**
 * @experimental
 */
export interface NotificationKeyFilter {
    /**
     * (experimental) S3 keys must have the specified prefix.
     *
     * @experimental
     */
    readonly prefix?: string;
    /**
     * (experimental) S3 keys must have the specified suffix.
     *
     * @experimental
     */
    readonly suffix?: string;
}
/**
 * (experimental) Options for the onCloudTrailPutObject method.
 *
 * @experimental
 */
export interface OnCloudTrailBucketEventOptions extends events.OnEventOptions {
    /**
     * (experimental) Only watch changes to these object paths.
     *
     * @default - Watch changes to all objects
     * @experimental
     */
    readonly paths?: string[];
}
/**
 * (experimental) Default bucket access control types.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html
 * @experimental
 */
export declare enum BucketAccessControl {
    /**
     * (experimental) Owner gets FULL_CONTROL.
     *
     * No one else has access rights.
     *
     * @experimental
     */
    PRIVATE = "Private",
    /**
     * (experimental) Owner gets FULL_CONTROL.
     *
     * The AllUsers group gets READ access.
     *
     * @experimental
     */
    PUBLIC_READ = "PublicRead",
    /**
     * (experimental) Owner gets FULL_CONTROL.
     *
     * The AllUsers group gets READ and WRITE access.
     * Granting this on a bucket is generally not recommended.
     *
     * @experimental
     */
    PUBLIC_READ_WRITE = "PublicReadWrite",
    /**
     * (experimental) Owner gets FULL_CONTROL.
     *
     * The AuthenticatedUsers group gets READ access.
     *
     * @experimental
     */
    AUTHENTICATED_READ = "AuthenticatedRead",
    /**
     * (experimental) The LogDelivery group gets WRITE and READ_ACP permissions on the bucket.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/ServerLogs.html
     * @experimental
     */
    LOG_DELIVERY_WRITE = "LogDeliveryWrite",
    /**
     * (experimental) Object owner gets FULL_CONTROL.
     *
     * Bucket owner gets READ access.
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     *
     * @experimental
     */
    BUCKET_OWNER_READ = "BucketOwnerRead",
    /**
     * (experimental) Both the object owner and the bucket owner get FULL_CONTROL over the object.
     *
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     *
     * @experimental
     */
    BUCKET_OWNER_FULL_CONTROL = "BucketOwnerFullControl",
    /**
     * (experimental) Owner gets FULL_CONTROL.
     *
     * Amazon EC2 gets READ access to GET an Amazon Machine Image (AMI) bundle from Amazon S3.
     *
     * @experimental
     */
    AWS_EXEC_READ = "AwsExecRead"
}
/**
 * @experimental
 */
export interface RoutingRuleCondition {
    /**
     * (experimental) The HTTP error code when the redirect is applied.
     *
     * In the event of an error, if the error code equals this value, then the specified redirect is applied.
     *
     * If both condition properties are specified, both must be true for the redirect to be applied.
     *
     * @default - The HTTP error code will not be verified
     * @experimental
     */
    readonly httpErrorCodeReturnedEquals?: string;
    /**
     * (experimental) The object key name prefix when the redirect is applied.
     *
     * If both condition properties are specified, both must be true for the redirect to be applied.
     *
     * @default - The object key name will not be verified
     * @experimental
     */
    readonly keyPrefixEquals?: string;
}
/**
 * @experimental
 */
export declare class ReplaceKey {
    readonly withKey?: string | undefined;
    readonly prefixWithKey?: string | undefined;
    /**
     * (experimental) The specific object key to use in the redirect request.
     *
     * @experimental
     */
    static with(keyReplacement: string): ReplaceKey;
    /**
     * (experimental) The object key prefix to use in the redirect request.
     *
     * @experimental
     */
    static prefixWith(keyReplacement: string): ReplaceKey;
    private constructor();
}
/**
 * (experimental) Rule that define when a redirect is applied and the redirect behavior.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html
 * @experimental
 */
export interface RoutingRule {
    /**
     * (experimental) The host name to use in the redirect request.
     *
     * @default - The host name used in the original request.
     * @experimental
     */
    readonly hostName?: string;
    /**
     * (experimental) The HTTP redirect code to use on the response.
     *
     * @default "301" - Moved Permanently
     * @experimental
     */
    readonly httpRedirectCode?: string;
    /**
     * (experimental) Protocol to use when redirecting requests.
     *
     * @default - The protocol used in the original request.
     * @experimental
     */
    readonly protocol?: RedirectProtocol;
    /**
     * (experimental) Specifies the object key prefix to use in the redirect request.
     *
     * @default - The key will not be replaced
     * @experimental
     */
    readonly replaceKey?: ReplaceKey;
    /**
     * (experimental) Specifies a condition that must be met for the specified redirect to apply.
     *
     * @default - No condition
     * @experimental
     */
    readonly condition?: RoutingRuleCondition;
}
/**
 * (experimental) Options for creating Virtual-Hosted style URL.
 *
 * @experimental
 */
export interface VirtualHostedStyleUrlOptions {
    /**
     * (experimental) Specifies the URL includes the region.
     *
     * @default - true
     * @experimental
     */
    readonly regional?: boolean;
}
export {};
