"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcEndpointServiceDomainName = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const custom_resources_1 = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
const lib_1 = require("../lib");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * (experimental) A Private DNS configuration for a VPC endpoint service.
 *
 * @experimental
 */
class VpcEndpointServiceDomainName extends core_2.Construct {
    // The way this class works is by using three custom resources and a TxtRecord in conjunction
    // The first custom resource tells the VPC endpoint service to use the given DNS name
    // The VPC endpoint service will then say:
    // "ok, create a TXT record using these two values to prove you own the domain"
    // The second custom resource retrieves these two values from the service
    // The TxtRecord is created from these two values
    // The third custom resource tells the VPC Endpoint Service to verify the domain ownership
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        const serviceUniqueId = core_1.Names.nodeUniqueId(props.endpointService.node);
        const serviceId = props.endpointService.vpcEndpointServiceId;
        const privateDnsName = props.domainName;
        // Make sure a user doesn't accidentally add multiple domains
        this.validateProps(props);
        VpcEndpointServiceDomainName.endpointServicesMap[serviceUniqueId] = privateDnsName;
        VpcEndpointServiceDomainName.endpointServices.push(props.endpointService);
        // Enable Private DNS on the endpoint service and retrieve the AWS-generated configuration
        const privateDnsConfiguration = this.getPrivateDnsConfiguration(serviceUniqueId, serviceId, privateDnsName);
        // Tell AWS to verify that this account owns the domain attached to the service
        this.verifyPrivateDnsConfiguration(privateDnsConfiguration, props.publicHostedZone);
        // Finally, don't do any of the above before the endpoint service is created
        this.node.addDependency(props.endpointService);
    }
    validateProps(props) {
        const serviceUniqueId = core_1.Names.nodeUniqueId(props.endpointService.node);
        if (serviceUniqueId in VpcEndpointServiceDomainName.endpointServicesMap) {
            const endpoint = VpcEndpointServiceDomainName.endpointServicesMap[serviceUniqueId];
            throw new Error(`Cannot create a VpcEndpointServiceDomainName for service ${serviceUniqueId}, another VpcEndpointServiceDomainName (${endpoint}) is already associated with it`);
        }
    }
    /**
     * Sets up Custom Resources to make AWS calls to set up Private DNS on an endpoint service,
     * returning the values to use in a TxtRecord, which AWS uses to verify domain ownership.
     */
    getPrivateDnsConfiguration(serviceUniqueId, serviceId, privateDnsName) {
        // The custom resource which tells AWS to enable Private DNS on the given service, using the given domain name
        // AWS will generate a name/value pair for use in a TxtRecord, which is used to verify domain ownership.
        const enablePrivateDnsAction = {
            service: 'EC2',
            action: 'modifyVpcEndpointServiceConfiguration',
            parameters: {
                ServiceId: serviceId,
                PrivateDnsName: privateDnsName,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(serviceUniqueId),
        };
        const removePrivateDnsAction = {
            service: 'EC2',
            action: 'modifyVpcEndpointServiceConfiguration',
            parameters: {
                ServiceId: serviceId,
                RemovePrivateDnsName: true,
            },
        };
        const enable = new custom_resources_1.AwsCustomResource(this, 'EnableDns', {
            onCreate: enablePrivateDnsAction,
            onUpdate: enablePrivateDnsAction,
            onDelete: removePrivateDnsAction,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [
                    core_1.Fn.join(':', [
                        'arn',
                        core_1.Stack.of(this).partition,
                        'ec2',
                        core_1.Stack.of(this).region,
                        core_1.Stack.of(this).account,
                        core_1.Fn.join('/', [
                            'vpc-endpoint-service',
                            serviceId,
                        ]),
                    ]),
                ],
            }),
        });
        // Look up the name/value pair if the domain changes, or the service changes,
        // which would cause the values to be different. If the unique ID changes,
        // the resource may be entirely recreated, so we will need to look it up again.
        const lookup = hashcode(core_1.Names.uniqueId(this) + serviceUniqueId + privateDnsName);
        // Create the custom resource to look up the name/value pair generated by AWS
        // after the previous API call
        const retrieveNameValuePairAction = {
            service: 'EC2',
            action: 'describeVpcEndpointServiceConfigurations',
            parameters: {
                ServiceIds: [serviceId],
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(lookup),
        };
        const getNames = new custom_resources_1.AwsCustomResource(this, 'GetNames', {
            onCreate: retrieveNameValuePairAction,
            onUpdate: retrieveNameValuePairAction,
            // describeVpcEndpointServiceConfigurations can't take an ARN for granular permissions
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        // We only want to call and get the name/value pair after we've told AWS to enable Private DNS
        // If we call before then, we'll get an empty pair of values.
        getNames.node.addDependency(enable);
        // Get the references to the name/value pair associated with the endpoint service
        const name = getNames.getResponseField('ServiceConfigurations.0.PrivateDnsNameConfiguration.Name');
        const value = getNames.getResponseField('ServiceConfigurations.0.PrivateDnsNameConfiguration.Value');
        return { name, value, serviceId };
    }
    /**
     * Creates a Route53 entry and a Custom Resource which explicitly tells AWS to verify ownership
     * of the domain name attached to an endpoint service.
     */
    verifyPrivateDnsConfiguration(config, publicHostedZone) {
        // Create the TXT record in the provided hosted zone
        const verificationRecord = new lib_1.TxtRecord(this, 'DnsVerificationRecord', {
            recordName: config.name,
            values: [config.value],
            zone: publicHostedZone,
        });
        // Tell the endpoint service to verify the domain ownership
        const startVerificationAction = {
            service: 'EC2',
            action: 'startVpcEndpointServicePrivateDnsVerification',
            parameters: {
                ServiceId: config.serviceId,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(core_1.Fn.join(':', [config.name, config.value])),
        };
        const startVerification = new custom_resources_1.AwsCustomResource(this, 'StartVerification', {
            onCreate: startVerificationAction,
            onUpdate: startVerificationAction,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [
                    core_1.Fn.join(':', [
                        'arn',
                        core_1.Stack.of(this).partition,
                        'ec2',
                        core_1.Stack.of(this).region,
                        core_1.Stack.of(this).account,
                        core_1.Fn.join('/', [
                            'vpc-endpoint-service',
                            config.serviceId,
                        ]),
                    ]),
                ],
            }),
        });
        // Only verify after the record has been created
        startVerification.node.addDependency(verificationRecord);
    }
}
exports.VpcEndpointServiceDomainName = VpcEndpointServiceDomainName;
_a = JSII_RTTI_SYMBOL_1;
VpcEndpointServiceDomainName[_a] = { fqn: "monocdk.aws_route53.VpcEndpointServiceDomainName", version: "1.106.1" };
// Track all domain names created, so someone doesn't accidentally associate two domains with a single service
VpcEndpointServiceDomainName.endpointServices = [];
// Track all domain names created, so someone doesn't accidentally associate two domains with a single service
VpcEndpointServiceDomainName.endpointServicesMap = {};
/**
 * Hash a string
 */
function hashcode(s) {
    const hash = crypto.createHash('md5');
    hash.update(s);
    return hash.digest('hex');
}
;
//# sourceMappingURL=data:application/json;base64,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