"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeHostedZoneArn = exports.determineFullyQualifiedDomainName = exports.validateZoneName = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Validates a zone name is valid by Route53 specifc naming rules,
 * and that there is no trailing dot in the name.
 *
 * @param zoneName the zone name to be validated.
 * @returns +zoneName+
 * @throws ValidationError if the name is not valid.
 */
function validateZoneName(zoneName) {
    if (zoneName.endsWith('.')) {
        throw new ValidationError('zone name must not end with a trailing dot');
    }
    if (zoneName.length > 255) {
        throw new ValidationError('zone name cannot be more than 255 bytes long');
    }
    if (zoneName.split('.').find(label => label.length > 63)) {
        throw new ValidationError('zone name labels cannot be more than 63 bytes long');
    }
    if (!zoneName.match(/^[a-z0-9!"#$%&'()*+,/:;<=>?@[\\\]^_`{|}~.-]+$/i)) {
        throw new ValidationError('zone names can only contain a-z, 0-9, -, ! " # $ % & \' ( ) * + , - / : ; < = > ? @ [ \ ] ^ _ ` { | } ~ .');
    }
}
exports.validateZoneName = validateZoneName;
class ValidationError extends Error {
    constructor(message) {
        super(message);
    }
}
/**
 * Route53 requires the record names are specified as fully qualified names, but this
 * forces lots of redundant work on the user (repeating the zone name over and over).
 * This function allows the user to be lazier and offers a nicer experience, by
 * qualifying relative names appropriately:
 *
 * @param providedName the user-specified name of the record.
 * @param zoneName   the fully-qualified name of the zone the record will be created in.
 *
 * @returns <ul>
 *        <li>If +providedName+ ends with a +.+, use it as-is</li>
 *        <li>If +providedName+ ends with or equals +zoneName+, append a trailing +.+</li>
 *        <li>Otherwise, append +.+, +zoneName+ and a trailing +.+</li>
 *      </ul>
 */
function determineFullyQualifiedDomainName(providedName, hostedZone) {
    if (providedName.endsWith('.')) {
        return providedName;
    }
    const hostedZoneName = hostedZone.zoneName.endsWith('.')
        ? hostedZone.zoneName.substring(0, hostedZone.zoneName.length - 1)
        : hostedZone.zoneName;
    const suffix = `.${hostedZoneName}`;
    if (providedName.endsWith(suffix) || providedName === hostedZoneName) {
        return `${providedName}.`;
    }
    return `${providedName}${suffix}.`;
}
exports.determineFullyQualifiedDomainName = determineFullyQualifiedDomainName;
function makeHostedZoneArn(construct, hostedZoneId) {
    return core_1.Stack.of(construct).formatArn({
        account: '',
        region: '',
        service: 'route53',
        resource: 'hostedzone',
        resourceName: hostedZoneId,
    });
}
exports.makeHostedZoneArn = makeHostedZoneArn;
//# sourceMappingURL=data:application/json;base64,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